/* GNOME-DB Components
 * Copyright (C) 2000-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <libgda/gda-client.h>
#include <libgda/gda-config.h>
#include <libgda/gda-data-model-array.h>
#include <bonobo/bonobo-i18n.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktable.h>
#include <gtk/gtkvbbox.h>
#include <libgnomedb/gnome-db-config.h>
#include <libgnomedb/gnome-db-dsn-config-druid.h>
#include <libgnomedb/gnome-db-dsn-config.h>
#include <libgnomedb/gnome-db-grid.h>
#include <libgnomedb/gnome-db-login.h>
#include <libgnomedb/gnome-db-stock.h>
#include <libgnomedb/gnome-db-util.h>
#include <libgnomeui/gnome-window-icon.h>
#include "dsn-config.h"

#define DSN_CONFIG_DATA "DSN_ConfigData"

typedef struct {
	GtkWidget *dsn_list;
	GtkWidget *dialog;
} DsnConfigPrivate;

static void
free_private_data (gpointer data)
{
	DsnConfigPrivate *priv = (DsnConfigPrivate *) data;

	g_free (priv);
}

static void
show_data_sources (DsnConfigPrivate *priv)
{
	GList *dsn_list;
	GList *l;
	GdaDataModel *model;

	/* create the data model */
	model = gda_data_model_array_new (2);
	gda_data_model_set_column_title (model, 0, _("Data source"));
	gda_data_model_set_column_title (model, 1, _("Description"));

	dsn_list = gnome_db_config_get_data_source_list ();
	for (l = dsn_list; l != NULL; l = l->next) {
		GList *value_list;
		GdaDataSourceInfo *dsn_info = l->data;

		value_list = g_list_append (NULL, gda_value_new_string (dsn_info->name));
		value_list = g_list_append (value_list, gda_value_new_string (dsn_info->description));

		gda_data_model_append_row (model, (const GList *) value_list);

		g_list_foreach (value_list, (GFunc) gda_value_free, NULL);
		g_list_free (value_list);
	}

	gnome_db_config_free_data_source_list (dsn_list);

	/* display the new data model */
	gnome_db_grid_set_model (GNOME_DB_GRID (priv->dsn_list), model);
	g_object_unref (G_OBJECT (model));
}

/*
 * Callbacks
 */

static void
druid_finished_cb (GnomeDbDsnConfigDruid *druid, gboolean error, gpointer user_data)
{
	const GdaDataSourceInfo *dsn_info;
	DsnConfigPrivate *priv = (DsnConfigPrivate *) user_data;

	if (!error) {
		dsn_info = gnome_db_dsn_config_druid_get_dsn (druid);
		if (dsn_info) {
			gnome_db_config_save_data_source (dsn_info->name,
						     dsn_info->provider,
						     dsn_info->cnc_string,
						     dsn_info->description,
						     dsn_info->username,
						     dsn_info->password);
			show_data_sources (priv);
		}
		else
			gnome_db_show_error (_("No valid data source info was created"));
	}

	gtk_widget_destroy (priv->dialog);
}

static void
foreach_edit_cb (GnomeDbGrid *grid, gint row, gpointer user_data)
{
	GdaDataModel *model;
	gchar *str;
	GdaDataSourceInfo *dsn_info;
	GtkWidget *dialog;
	GtkWidget *props;
	DsnConfigPrivate *priv = (DsnConfigPrivate *) user_data;

	model = gnome_db_grid_get_model (GNOME_DB_GRID (priv->dsn_list));
	if (!GDA_IS_DATA_MODEL (model))
		return;

	str = gda_value_stringify (
		(GdaValue *) gda_data_model_get_value_at (model, 0, row));
	if (!str)
		return;

	/* display the dialog for editing the data source */
	dsn_info = gnome_db_config_find_data_source (str);
	g_free (str);
	if (!dsn_info)
		return;

	dialog = gtk_dialog_new_with_buttons (
		_("Data source properties"),
		GTK_WINDOW (gtk_widget_get_toplevel (priv->dsn_list)), 0,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		GTK_STOCK_SAVE, GTK_RESPONSE_OK,
		NULL);
	gnome_window_icon_set_from_file (GTK_WINDOW (dialog),
					 LIBGNOMEDB_ICONSDIR "/gnome-db.png");

	props = gnome_db_dsn_config_new ();
	gnome_db_dsn_config_set_info (GNOME_DB_DSN_CONFIG (props), dsn_info);
	gtk_widget_show (props);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), props, TRUE, TRUE, 0);

	gnome_db_config_free_data_source_info (dsn_info);

	/* run the dialog */
	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
		dsn_info = gnome_db_dsn_config_get_info (GNOME_DB_DSN_CONFIG (props));
		if (dsn_info) {
			gnome_db_config_save_data_source (dsn_info->name,
						     dsn_info->provider,
						     dsn_info->cnc_string,
						     dsn_info->description,
						     dsn_info->username,
						     dsn_info->password);
			gnome_db_config_free_data_source_info (dsn_info);

			show_data_sources (priv);
		}
	}

	gtk_widget_destroy (dialog);
}

static void
edit_dsn_cb (GtkButton *button, gpointer user_data)
{
	DsnConfigPrivate *priv = (DsnConfigPrivate *) user_data;
	gnome_db_grid_foreach_selected (GNOME_DB_GRID (priv->dsn_list),
					(GnomeDbGridForeachFunc) foreach_edit_cb,
					priv);
}

static void
new_dsn_cb (GtkButton *button, gpointer user_data)
{
	GtkWidget *druid;
	DsnConfigPrivate *priv = (DsnConfigPrivate *) user_data;

	/* create the window to contain the druid */
	priv->dialog = gtk_dialog_new_with_buttons (
		_("New data source"),
		GTK_WINDOW (gtk_widget_get_toplevel (priv->dsn_list)),
		0, NULL);
	gnome_window_icon_set_from_file (GTK_WINDOW (priv->dialog),
					 LIBGNOMEDB_ICONSDIR "/gnome-db.png");
	gtk_dialog_set_has_separator (GTK_DIALOG (priv->dialog), FALSE);

	druid = gnome_db_dsn_config_druid_new ();
	g_signal_connect (G_OBJECT (druid), "finished",
			  G_CALLBACK (druid_finished_cb), priv);
	gtk_widget_show (druid);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (priv->dialog)->vbox), druid, TRUE, TRUE, 0);

	/* run the dialog */
	gtk_widget_show (priv->dialog);
}

static void
refresh_list_cb (GtkButton *button, gpointer user_data)
{
	DsnConfigPrivate *priv = (DsnConfigPrivate *) user_data;
	show_data_sources (priv);
}

static void
test_dsn_cb (GtkButton *button, gpointer user_data)
{
	GList *selection;
	GdaDataModel *model;
	GtkWidget *dialog;
	GtkWidget *login;
	gchar *str;
	DsnConfigPrivate *priv = (DsnConfigPrivate *) user_data;

	selection = gnome_db_grid_get_selection (GNOME_DB_GRID (priv->dsn_list));
	if (!selection)
		return;

	model = gnome_db_grid_get_model (GNOME_DB_GRID (priv->dsn_list));

	/* we're only interested in the first selected row */
	str = gda_value_stringify (
		(GdaValue *) gda_data_model_get_value_at (
			model, 0, GPOINTER_TO_INT (selection->data)));
	if (!str)
		return;

	/* create the dialog box */
	dialog = gtk_dialog_new_with_buttons (_("Test Data Source"),
					      GTK_WINDOW (gtk_widget_get_toplevel (priv->dsn_list)), 0,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GNOME_DB_STOCK_CONNECT, GTK_RESPONSE_OK,
					      NULL);
	gnome_window_icon_set_from_file (GTK_WINDOW (dialog),
					 LIBGNOMEDB_ICONSDIR "/gnome-db.png");
	login = gnome_db_login_new (str);
	gtk_widget_show (login);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), login, TRUE, TRUE, 0);

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
		GdaClient *client;
		GdaConnection *cnc;

		client = gda_client_new ();
		cnc = gda_client_open_connection (
			client,
			gnome_db_login_get_dsn (GNOME_DB_LOGIN (login)),
			gnome_db_login_get_username (GNOME_DB_LOGIN (login)),
			gnome_db_login_get_password (GNOME_DB_LOGIN (login)));
		if (GDA_IS_CONNECTION (cnc) && gda_connection_is_open (cnc)) {
			gnome_db_show_message (_("Connection to %s successful!"), str);
			gda_connection_close (cnc);
		}
		else
			gnome_db_show_error (_("Could not open connection to %s"), str);

		g_object_unref (G_OBJECT (client));
	}

	gtk_widget_destroy (dialog);
	
	/* free memory */
	g_free (str);
	g_list_free (selection);
}

static void
foreach_remove_cb (GnomeDbGrid *grid, gint row, gpointer user_data)
{
	GdaDataModel *model;
	gchar *str;
	DsnConfigPrivate *priv = (DsnConfigPrivate *) user_data;

	model = gnome_db_grid_get_model (GNOME_DB_GRID (priv->dsn_list));
	if (!GDA_IS_DATA_MODEL (model))
		return;

	str = gda_value_stringify (
		(GdaValue *) gda_data_model_get_value_at (model, 0, row));
	if (!str)
		return;

	if (gnome_db_yes_no_dialog (
		    gtk_widget_get_toplevel (priv->dsn_list),
		    _("Do you want to remove '%s' data source?"),
		    str)) {
		gnome_db_config_remove_data_source (str);
		show_data_sources (priv);
	}

	g_free (str);
}

static void
remove_dsn_cb (GtkButton *button, gpointer user_data)
{
	DsnConfigPrivate *priv = (DsnConfigPrivate *) user_data;

	gnome_db_grid_foreach_selected (GNOME_DB_GRID (priv->dsn_list),
					(GnomeDbGridForeachFunc) foreach_remove_cb,
					priv);
}

static void
select_all_cb (GtkButton *button, gpointer user_data)
{
	DsnConfigPrivate *priv = (DsnConfigPrivate *) user_data;
	gnome_db_grid_select_all (GNOME_DB_GRID (priv->dsn_list));
}

static void
unselect_all_cb (GtkButton *button, gpointer user_data)
{
	DsnConfigPrivate *priv = (DsnConfigPrivate *) user_data;
	gnome_db_grid_unselect_all (GNOME_DB_GRID (priv->dsn_list));
}

/*
 * Public functions
 */

GtkWidget *
dsn_config_new (void)
{
	DsnConfigPrivate *priv;
	GtkWidget *dsn;
	GtkWidget *table;
	GtkWidget *box;
	GtkWidget *button;
	GtkWidget *label;

	priv = g_new0 (DsnConfigPrivate, 1);
	dsn = gnome_db_new_vbox_widget (FALSE, 0);
	g_object_set_data_full (G_OBJECT (dsn), DSN_CONFIG_DATA, priv,
				(GDestroyNotify) free_private_data);

	table = gnome_db_new_table_widget (2, 2, FALSE);
	gtk_box_pack_start (GTK_BOX (dsn), table, TRUE, TRUE, 0);

	/* create the data source list */
	priv->dsn_list = gnome_db_grid_new ();
	gnome_db_grid_set_title (GNOME_DB_GRID (priv->dsn_list), _("Data sources"));
	gnome_db_grid_set_show_title (GNOME_DB_GRID (priv->dsn_list), TRUE);
	gtk_widget_show (priv->dsn_list);
	gtk_table_attach (GTK_TABLE (table), priv->dsn_list, 0, 1, 0, 1,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  2, 2);
	show_data_sources (priv);

	/* add buttons */
	box = gtk_vbutton_box_new ();
	gtk_box_set_spacing (GTK_BOX (box), 4);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (box), GTK_BUTTONBOX_START);
	gtk_widget_show (box);
	gtk_table_attach (GTK_TABLE (table), box, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 2, 2);

	button = gnome_db_new_button_widget_from_stock (GTK_STOCK_NEW);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (new_dsn_cb), priv);
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 2);

	button = gnome_db_new_button_widget_from_stock (GTK_STOCK_PROPERTIES);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (edit_dsn_cb), priv);
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 2);

	button = gnome_db_new_button_widget (_("Test"));
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (test_dsn_cb), priv);
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 2);

	button = gnome_db_new_button_widget_from_stock (GTK_STOCK_DELETE);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (remove_dsn_cb), priv);
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 2);

	button = gnome_db_new_button_widget (_("Select all"));
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (select_all_cb), priv);
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 2);

	button = gnome_db_new_button_widget (_("Unselect all"));
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (unselect_all_cb), priv);
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 2);

	button = gnome_db_new_button_widget_from_stock (GTK_STOCK_REFRESH);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (refresh_list_cb), priv);
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 2);

	/* add tip */
	box = gnome_db_new_hbox_widget (FALSE, 3);
	gtk_table_attach (GTK_TABLE (table), box, 0, 2, 1, 2,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK, 5, 5);

	button = gtk_image_new_from_stock (GTK_STOCK_DIALOG_INFO, GTK_ICON_SIZE_DIALOG);
	gtk_widget_show (button);
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 3);

	label = gnome_db_new_label_widget (
		_("Data sources are the means by which database "
		  "connections are identified in libgda. All "
		  "information needed to open a connection to "
		  "a specific provider/database combo is stored using "
		  "a unique name. It is by use of this unique name "
		  "you identify the connections in the applications "
		  "that make use of libgda for database access."));
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_box_pack_start (GTK_BOX (box), label, TRUE, TRUE, 3);
	
	return dsn;
}
