/* GNOME DB components libary
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 * 	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtkwidget.h>
#include <bonobo/bonobo-control.h>
#include <bonobo/bonobo-ui-component.h>
#include <bonobo/bonobo-ui-container.h>
#include <bonobo/bonobo-ui-util.h>
#include <libgnomedb/gnome-db-util.h>
#include <libgnomedb/gnome-db-window.h>

#define PARENT_TYPE BONOBO_TYPE_WINDOW

struct _GnomeDbWindowPrivate {
	BonoboUIComponent *ui_component;
	GtkWidget *status_bar;
	GtkWidget *progress_bar;
};

static void gnome_db_window_class_init   (GnomeDbWindowClass *klass);
static void gnome_db_window_init         (GnomeDbWindow *window, GnomeDbWindowClass *klass);
static void gnome_db_window_set_property (GObject *object,
					  guint paramid,
					  const GValue *value,
					  GParamSpec *pspec);
static void gnome_db_window_get_property (GObject *object,
					  guint param_id,
					  GValue *value,
					  GParamSpec *pspec);
static void gnome_db_window_finalize     (GObject *object);

enum {
	CLOSE,
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_CONTENTS,
	PROP_STATUS
};

static gint db_window_signals[LAST_SIGNAL];
static GObjectClass *parent_class = NULL;

/*
 * Callbacks
 */

static gboolean
delete_event_cb (GtkWidget *widget, GdkEventAny *event, gpointer user_data)
{
	return FALSE;
}

static void
window_destroyed_cb (GtkWidget *widget, gpointer user_data)
{
	GnomeDbWindow *window = (GnomeDbWindow *) widget;

	g_return_if_fail (GNOME_DB_IS_WINDOW (window));
	g_signal_emit (G_OBJECT(window), db_window_signals[CLOSE], 0);
}

/*
 * GnomeDbWindow class implementation
 */

static void
gnome_db_window_class_init (GnomeDbWindowClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->set_property = gnome_db_window_set_property;
	object_class->get_property = gnome_db_window_get_property;
	object_class->finalize = gnome_db_window_finalize;
	klass->close = NULL; // FIXME: add default close handler

	/* add class properties */
	g_object_class_install_property (
		object_class, PROP_CONTENTS,
		g_param_spec_object ("contents", NULL, NULL,
				     GTK_TYPE_WIDGET,
				     (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (
		object_class, PROP_STATUS,
		g_param_spec_string ("status", NULL, NULL, NULL, G_PARAM_WRITABLE));

	/* add class signals */
	db_window_signals[CLOSE] =
		g_signal_new ("close",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbWindowClass, close),
		              NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
}

static void
gnome_db_window_init (GnomeDbWindow *window, GnomeDbWindowClass *klass)
{
	window->priv = g_new0(GnomeDbWindowPrivate, 1);
	window->priv->progress_bar = NULL;
}

static void
gnome_db_window_set_property (GObject *object,
			      guint param_id,
			      const GValue *value,
			      GParamSpec *pspec)
{
	GnomeDbWindow *window = (GnomeDbWindow *) object;

	g_return_if_fail (GNOME_DB_IS_WINDOW (window));

	switch (param_id) {
	case PROP_CONTENTS :
		gnome_db_window_set_contents (
			window,
			GTK_WIDGET (g_value_get_object (value)));
		break;
	case PROP_STATUS :
		gnome_db_window_set_status (window, g_value_get_string (value));
		break;
	default :
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_window_get_property (GObject *object,
			      guint param_id,
			      GValue *value,
			      GParamSpec *pspec)
{
	GnomeDbWindow *window = (GnomeDbWindow *) object;

	g_return_if_fail (GNOME_DB_IS_WINDOW (window));

	switch (param_id) {
	case PROP_CONTENTS :
		g_value_set_object (value,
				    G_OBJECT (gnome_db_window_get_contents (window)));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_window_finalize (GObject *object)
{
	GnomeDbWindow *window = (GnomeDbWindow *) object;

	g_return_if_fail (GNOME_DB_IS_WINDOW (window));

	/* free memory */
	g_free ((gpointer) window->priv);
	window->priv = NULL;

	/* chain to parent handler */
	parent_class->finalize (object);
}

GType
gnome_db_window_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbWindowClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_window_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbWindow),
			0,
			(GInstanceInitFunc) gnome_db_window_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbWindow", &info, 0);
	}
	return type;
}

/**
 * gnome_db_window_new
 * @name: name of the window 
 * @app_prefix: prefix used for the UI XML file (/usr/share, /usr/local/share...)
 * @ui_xml_file: XML file that describes the UI for this window
 * @verbs: UI verbs handlers
 * @data: data to be passed to callbacks
 * 
 * Create a new GnomeDbWindow widget, which is a convenience widget which
 * manages a typical application window, with its menus and toolbar. Apart
 * from this, it implements a full-features Bonobo container, thus letting
 * you easily add Bonobo support to your applications.
 */
GtkWidget *
gnome_db_window_new (const gchar *name,
		     const gchar *app_prefix,
		     const gchar *ui_xml_file,
		     BonoboUIVerb *verbs,
		     gpointer data)
{
	GnomeDbWindow *window;
	BonoboUIContainer *ui_container;
	Bonobo_UIContainer corba_container;

	/* create the window */
	window = g_object_new (GNOME_DB_TYPE_WINDOW, "win_name", name, "title", name, NULL);
	g_signal_connect (G_OBJECT (window), "delete_event",
			  G_CALLBACK (delete_event_cb), NULL);
	g_signal_connect (G_OBJECT (window), "destroy",
			  G_CALLBACK (window_destroyed_cb), NULL);

	ui_container = bonobo_window_get_ui_container ( BONOBO_WINDOW (window));
	corba_container = BONOBO_OBJREF (ui_container);
	window->priv->ui_component = bonobo_ui_component_new (name);
	bonobo_ui_component_set_container (window->priv->ui_component, corba_container, NULL);

	/* set UI for window */
	if (verbs != NULL && ui_xml_file != NULL) {
		bonobo_ui_component_freeze (window->priv->ui_component, NULL);
		bonobo_ui_util_set_ui (window->priv->ui_component,
				       app_prefix, ui_xml_file, name, NULL);
		bonobo_ui_component_add_verb_list_with_data (window->priv->ui_component,
							     verbs, data);
		bonobo_ui_component_thaw (window->priv->ui_component, NULL);
	}

	return GTK_WIDGET (window);
}

/**
 * gnome_db_window_close
 */
void
gnome_db_window_close (GnomeDbWindow *window)
{
	g_return_if_fail (GNOME_DB_IS_WINDOW (window));
	gtk_widget_destroy (GTK_WIDGET (window));
}

/**
 * gnome_db_window_show
 */
void
gnome_db_window_show (GnomeDbWindow *window)
{
	g_return_if_fail (GNOME_DB_IS_WINDOW (window));
	gtk_widget_show_all (GTK_WIDGET (window));
}

/**
 * gnome_db_window_get_contents
 */
GtkWidget *
gnome_db_window_get_contents (GnomeDbWindow *window)
{
	g_return_val_if_fail (GNOME_DB_IS_WINDOW (window), NULL);
	return bonobo_window_get_contents (BONOBO_WINDOW (window));
}

/**
 * gnome_db_window_set_contents
 */
void
gnome_db_window_set_contents (GnomeDbWindow *window, GtkWidget *contents)
{
	g_return_if_fail (GNOME_DB_IS_WINDOW (window));
	g_return_if_fail (GTK_IS_WIDGET (contents));
	
	bonobo_window_set_contents (BONOBO_WINDOW (window), contents);
}

/**
 * gnome_db_window_set_status
 */
void
gnome_db_window_set_status (GnomeDbWindow *window, const gchar *status)
{
	g_return_if_fail (GNOME_DB_IS_WINDOW (window));

	bonobo_ui_component_set_status (window->priv->ui_component, status, NULL);
}

/**
 * gnome_db_window_get_ui_component
 * @window: a #GnomeDbWindow widget.
 *
 * Return the #BonoboUIComponent associated with the given window.
 */
BonoboUIComponent *
gnome_db_window_get_ui_component (GnomeDbWindow *window)
{
	g_return_val_if_fail (GNOME_DB_IS_WINDOW (window), NULL);
	return window->priv->ui_component;
}

/**
 * gnome_db_window_get_ui_container
 */
Bonobo_UIContainer
gnome_db_window_get_ui_container (GnomeDbWindow *window)
{
	g_return_val_if_fail (GNOME_DB_IS_WINDOW (window), NULL);
	return bonobo_ui_component_get_container (window->priv->ui_component);
}

