/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <libgda/gda-util.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktable.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtktreeview.h>
#include <gobject/gmarshal.h>
#include <libgnomedb/gnome-db-form.h>
#include <libgnomedb/gnome-db-gray-bar.h>
#include <libgnomedb/gnome-db-grid.h>
#include <libgnomedb/gnome-db-util.h>
#include <libgnomedb/gnome-db-model.h>
#include <libgnomedb/gnome-db-stock.h>
#include <libgnomedb/gnome-db-table-editor.h>
#include <libgnomeui/gnome-file-entry.h>
#include <libgnomeui/gnome-window-icon.h>
#include <bonobo/bonobo-i18n.h>
#include "libgnomedb-private.h"

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbGridPrivate {
	GdaDataModel *model;

	/* widgets */
	GtkWidget *title;
	GtkWidget *scroll;
	GtkWidget *tree_view;
	GtkWidget *popup_menu;

	gchar *title_string;
	gboolean show_title;
	GtkSelectionMode selection_mode;
};

static void gnome_db_grid_class_init   (GnomeDbGridClass *klass);
static void gnome_db_grid_init         (GnomeDbGrid *grid, GnomeDbGridClass *klass);
static void gnome_db_grid_set_property (GObject *object,
					guint paramid,
					const GValue *value,
					GParamSpec *pspec);
static void gnome_db_grid_get_property (GObject *object,
					guint param_id,
					GValue *value,
					GParamSpec *pspec);
static void gnome_db_grid_finalize   (GObject *object);

enum {
	ROW_SELECTED,
	SELECTION_CLEARED,
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_MODEL
};

static gint gnome_db_grid_signals[LAST_SIGNAL] = { 0 };
static GdkAtom clipboard_atom = GDK_NONE;
static GObjectClass *parent_class = NULL;

/*
 * Callbacks
 */

#if 0
static void
menu_describe_cb (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *dialog;
	GtkWidget *description;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	/* create the dialog */
	dialog = gtk_dialog_new_with_buttons (
		_("Recordset description"),
		GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (grid))), 0,
		GTK_STOCK_CLOSE, GTK_RESPONSE_CANCEL,
		NULL);
	gnome_window_icon_set_from_file (GTK_WINDOW (dialog),
					 LIBGNOMEDB_ICONSDIR "/gnome-db.png");

	description = gnome_db_table_editor_new ();
	gtk_widget_show (description);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), description, TRUE, TRUE, 0);

	/* run the dialog */
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}
#endif

static void
menu_hide_columns_cb (GtkWidget *widget, gpointer user_data)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	gnome_db_grid_hide_column_titles (grid);
}

static void
menu_save_as_cb (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *dialog;
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *filename;
	GtkWidget *to_tab;
	GtkWidget *to_comma;
	GtkWidget *to_xml;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	g_return_if_fail (GDA_IS_DATA_MODEL (grid->priv->model));

	/* create dialog box */
	dialog = gtk_dialog_new_with_buttons (
		_("Save as"),
		GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (grid))), 0,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		GTK_STOCK_OK, GTK_RESPONSE_OK,
		NULL);
	gnome_window_icon_set_from_file (GTK_WINDOW (dialog),
					 LIBGNOMEDB_ICONSDIR "/gnome-db.png");
	table = gnome_db_new_table_widget (6, 2, FALSE);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, TRUE, TRUE, 0);

	label = gnome_db_new_label_widget (_("Save to"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
	filename = gnome_db_new_file_entry_widget ("gnome-db-grid-save-as", _("Select File"));
	gtk_table_attach (GTK_TABLE (table), filename, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 2, 2);

	to_tab = gnome_db_new_radio_button_widget (NULL, _("Tab-delimited file"));
	gtk_table_attach (GTK_TABLE (table), to_tab, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
	to_comma = gnome_db_new_radio_button_widget (
		gtk_radio_button_get_group (GTK_RADIO_BUTTON (to_tab)),
		_("Comma-delimited file"));
	gtk_table_attach (GTK_TABLE (table), to_comma, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 2, 2);
	to_xml = gnome_db_new_radio_button_widget (
		gtk_radio_button_get_group (GTK_RADIO_BUTTON (to_tab)),
		_("XML file"));
	gtk_table_attach (GTK_TABLE (table), to_xml, 1, 2, 3, 4, GTK_FILL, GTK_FILL, 2, 2);

	/* run the dialog */
	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
		gchar *body;
		gchar *path;

		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (to_tab)))
			body = gda_data_model_to_tab_separated (grid->priv->model);
		else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (to_comma)))
			body = gda_data_model_to_comma_separated (grid->priv->model);
		else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (to_xml)))
			body = gda_data_model_to_xml (grid->priv->model, FALSE);
		else
			body = NULL;

		if (body) {
			path = gnome_file_entry_get_full_path (GNOME_FILE_ENTRY (filename), FALSE);
			if (path) {
				if (!gda_file_save (path, body, strlen (body)))
					gnome_db_show_error (_("Could not save file %s"), path);
				g_free (path);
			}
			else
				gnome_db_show_error (_("You must specify a file name"));
			g_free (body);
		}
	}

	gtk_widget_destroy (dialog);
}

static void
menu_select_all_cb (GtkWidget *widget, gpointer user_data)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	gnome_db_grid_select_all (grid);
}

static void
menu_show_columns_cb (GtkWidget *widget, gpointer user_data)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	gnome_db_grid_show_column_titles (grid);
}

static void
menu_unselect_all_cb (GtkWidget *widget, gpointer user_data)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	gnome_db_grid_unselect_all (grid);
}

static void
menu_view_detail_cb (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *dialog;
	GtkWidget *form;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	/* create the dialog */
	dialog = gtk_dialog_new_with_buttons (
		_("View detail"),
		GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (grid))), 0,
		GTK_STOCK_CLOSE, GTK_RESPONSE_CANCEL,
		NULL);
	gnome_window_icon_set_from_file (GTK_WINDOW (dialog),
					 LIBGNOMEDB_ICONSDIR "/gnome-db.png");

	form = gnome_db_form_new ();
	gtk_widget_show (form);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), form, TRUE, TRUE, 0);
	gnome_db_form_set_form_type (GNOME_DB_FORM (form), GNOME_DB_FORM_TYPE_NAVIGATOR);
	gnome_db_form_set_model (GNOME_DB_FORM (form), grid->priv->model);

	/* run the dialog */
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

static void
selection_foreach (GtkTreeModel *tree_model,
		   GtkTreePath *path,
		   GtkTreeIter *iter,
		   gpointer user_data)
{
	DataModelInfo *iter_info;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	g_return_if_fail (iter != NULL);

	gtk_tree_model_get (tree_model, iter, 0, &iter_info, -1);
	if (iter_info && iter_info->row == 0) {
		g_signal_emit (G_OBJECT (grid), gnome_db_grid_signals[ROW_SELECTED],
			       0, iter_info->row);
	}
}

static void
selection_changed_cb (GtkTreeSelection *selection, gpointer user_data)
{
	GtkTreeModel *tree_model;
	GtkTreeIter iter;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	switch (gtk_tree_selection_get_mode (selection)) {
	case GTK_SELECTION_MULTIPLE :
		gtk_tree_selection_selected_foreach (
			selection,
			(GtkTreeSelectionForeachFunc) selection_foreach,
			grid);
		break;
	default :
		if (gtk_tree_selection_get_selected (selection, &tree_model, &iter)) {
			DataModelInfo *iter_info;

			gtk_tree_model_get (tree_model, &iter, 0, &iter_info, -1);
			g_signal_emit (G_OBJECT (grid), gnome_db_grid_signals[ROW_SELECTED],
				       0, iter_info->row);
		}
		else
			g_signal_emit (G_OBJECT (grid),
				       gnome_db_grid_signals[SELECTION_CLEARED], 0);
		break;
	}
}

/*
 * Private functions
 */

static GnomeUIInfo grid_popup_menu[] = {
	{ GNOME_APP_UI_ITEM, N_("Select All"), N_("Select all items in list"),
	  menu_select_all_cb, NULL, NULL, GNOME_APP_PIXMAP_NONE,
	  NULL, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Unselect All"), N_("Unselect all selected items in list"),
	  menu_unselect_all_cb, NULL, NULL, GNOME_APP_PIXMAP_NONE,
	  NULL, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Show columns"), N_("Display column headers"),
	  menu_show_columns_cb, NULL, NULL, GNOME_APP_PIXMAP_NONE,
	  NULL, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Hide columns"), N_("Hide column headers"),
	  menu_hide_columns_cb, NULL, NULL, GNOME_APP_PIXMAP_NONE,
	  NULL, 0, 0, NULL },
	GNOMEUIINFO_SEPARATOR,
	{ GNOME_APP_UI_ITEM, N_("Save as..."), N_("Export data to different formats"),
          menu_save_as_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
          GTK_STOCK_SAVE_AS, 0, 0, NULL },
	GNOMEUIINFO_SEPARATOR,
#if 0
	{ GNOME_APP_UI_ITEM, N_("Describe..."), N_("Describe current data set"),
          menu_describe_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
          GNOME_DB_STOCK_TABLES, 0, 0, NULL },
#endif
	{ GNOME_APP_UI_ITEM, N_("View detail..."), N_("View current data in more detail"),
          menu_view_detail_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
          GTK_STOCK_PROPERTIES, 0, 0, NULL },
	GNOMEUIINFO_END
};

static void
setup_grid (GnomeDbGrid *grid)
{
	GtkTreeSelection *selection;

	/* setup selection stuff */
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->tree_view));
	gtk_tree_selection_set_mode (selection, grid->priv->selection_mode);
	g_signal_connect (G_OBJECT (selection), "changed",
			  G_CALLBACK (selection_changed_cb), grid);

	grid->priv->popup_menu = gnome_db_new_popup_menu (grid->priv->tree_view,
							  grid_popup_menu, grid);
}

/*
 * GnomeDbGrid class implementation
 */

static void
gnome_db_grid_class_init (GnomeDbGridClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->set_property = gnome_db_grid_set_property;
	object_class->get_property = gnome_db_grid_get_property;
	object_class->finalize = gnome_db_grid_finalize;
	klass->row_selected = NULL;
	klass->selection_cleared = NULL;

	/* add class properties */
	g_object_class_install_property (
		object_class, PROP_MODEL,
		g_param_spec_object ("model", NULL, NULL,
				     GDA_TYPE_DATA_MODEL,
				     (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* add class signals */
	gnome_db_grid_signals[ROW_SELECTED] =
		g_signal_new ("row_selected",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbGridClass, row_selected),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__INT,
			      G_TYPE_NONE, 1, G_TYPE_INT);
	gnome_db_grid_signals[SELECTION_CLEARED] =
		g_signal_new ("selection_cleared",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbGridClass, selection_cleared),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	/* clipboard atom */
	if (!clipboard_atom)
		clipboard_atom = gdk_atom_intern ("CLIPBOARD", FALSE);
}

static void
gnome_db_grid_init (GnomeDbGrid *grid, GnomeDbGridClass *klass)
{
	/* allocate the internal structure */
	grid->priv = g_new0 (GnomeDbGridPrivate, 1);
	grid->priv->show_title = FALSE;
	grid->priv->title_string = g_strdup (_("Title"));

	/* set up widgets */
	grid->priv->title = gnome_db_gray_bar_new (grid->priv->title_string);
	gtk_box_pack_start (GTK_BOX (grid), grid->priv->title, 0, 1, 2);

	grid->priv->scroll = gnome_db_new_scrolled_window_widget ();
	gtk_box_pack_start (GTK_BOX (grid), grid->priv->scroll, 1, 1, 0);

	grid->priv->tree_view = gnome_db_new_tree_view_widget (NULL);
	gtk_container_add (GTK_CONTAINER (grid->priv->scroll), grid->priv->tree_view);
	setup_grid (grid);

	gnome_db_grid_set_selection_mode (grid, GTK_SELECTION_MULTIPLE);
}

static void
gnome_db_grid_set_property (GObject *object,
			    guint param_id,
			    const GValue *value,
			    GParamSpec *pspec)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) object;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	switch (param_id) {
	case PROP_MODEL :
		gnome_db_grid_set_model (grid, GDA_DATA_MODEL (g_value_get_object (value)));
		break;
	default :
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_grid_get_property (GObject *object,
			    guint param_id,
			    GValue *value,
			    GParamSpec *pspec)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) object;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	switch (param_id) {
	case PROP_MODEL :
		g_value_set_object (value, G_OBJECT (grid->priv->model));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_grid_finalize (GObject *object)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) object;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	/* free memory */
	if (grid->priv->title_string) {
		g_free (grid->priv->title_string);
		grid->priv->title_string = NULL;
	}

	g_free (grid->priv);
	grid->priv = NULL;

	parent_class->finalize (object);
}

GType
gnome_db_grid_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGridClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_grid_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGrid),
			0,
			(GInstanceInitFunc) gnome_db_grid_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbGrid", &info, 0);
	}
	return type;
}

/**
 * gnome_db_grid_new
 */
GtkWidget *
gnome_db_grid_new (void)
{
	GnomeDbGrid *grid;

	grid = g_object_new (GNOME_DB_TYPE_GRID, NULL);
	return GTK_WIDGET (grid);
}

/**
 * gnome_db_grid_new_with_model
 */
GtkWidget *
gnome_db_grid_new_with_model (GdaDataModel *model)
{
	GnomeDbGrid *grid;

	grid = g_object_new (GNOME_DB_TYPE_GRID, NULL);
	gnome_db_grid_set_model (grid, model);
	return GTK_WIDGET (grid);
}

/**
 * gnome_db_grid_get_title
 * @grid: a #GnomeDbGrid widget.
 *
 * Get the title for the given grid.
 *
 * Returns: the title associated with the grid.
 */
const gchar *
gnome_db_grid_get_title (GnomeDbGrid *grid)
{
	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), NULL);
	return (const gchar *) grid->priv->title_string;
}

/**
 * gnome_db_grid_set_title
 * @grid: a #GnomeDbGrid widget.
 * @title: new title.
 *
 * Set the title for the given grid.
 */
void
gnome_db_grid_set_title (GnomeDbGrid *grid, const gchar *title)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	if (grid->priv->title_string)
		g_free (grid->priv->title_string);

	grid->priv->title_string = g_strdup (title);
	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (grid->priv->title), title);
}

/**
 * gnome_db_grid_get_show_title
 * @grid: a #GnomeDbGrid widget.
 *
 * Get whether the title is being shown for the given grid.
 *
 * Returns: TRUE if the title is shown, FALSE if not.
 */
gboolean
gnome_db_grid_get_show_title (GnomeDbGrid *grid)
{
	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), FALSE);
	return grid->priv->show_title;
}

/**
 * gnome_db_grid_set_show_title
 * @grid: a #GnomeDbGrid widget.
 * @show: whether to show the title or not.
 *
 * Set the title displating mode for the given grid.
 */
void
gnome_db_grid_set_show_title (GnomeDbGrid *grid, gboolean show)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	grid->priv->show_title = show;
	if (show)
		gtk_widget_show (grid->priv->title);
	else
		gtk_widget_hide (grid->priv->title);
}

/**
 * gnome_db_grid_get_model
 */
GdaDataModel *
gnome_db_grid_get_model (GnomeDbGrid *grid)
{
	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), NULL);
	return grid->priv->model;
}

/**
 * gnome_db_grid_set_model
 */
void
gnome_db_grid_set_model (GnomeDbGrid *grid, GdaDataModel *model)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	if (GDA_IS_DATA_MODEL (grid->priv->model))
		g_object_unref (G_OBJECT (grid->priv->model));

	grid->priv->model = model;
	if (GDA_IS_DATA_MODEL (model))
		g_object_ref (G_OBJECT (model));
	else {
		/* just empty the grid and return */
		gtk_tree_view_set_model (GTK_TREE_VIEW (grid->priv->tree_view), NULL);
		return;
	}

	/* redisplay the GtkTreeView */
	gtk_widget_destroy (grid->priv->tree_view);
	grid->priv->tree_view = GTK_WIDGET (gnome_db_model_to_gtk_tree_view (grid->priv->model));
	gtk_container_add (GTK_CONTAINER (grid->priv->scroll), grid->priv->tree_view);
	gtk_widget_show (grid->priv->tree_view);
	setup_grid (grid);
}

static void
foreach_selected_cb (GtkTreeModel *tree_model,
		     GtkTreePath *path,
		     GtkTreeIter *iter,
		     gpointer user_data)
{
	DataModelInfo *iter_info;
	GList **list = (GList **) user_data;

	gtk_tree_model_get (tree_model, iter, 0, &iter_info, -1);
	if (iter_info)
		*list = g_list_append (*list, GINT_TO_POINTER (iter_info->row));
}

/**
 * gnome_db_grid_get_selection
 * @grid: a #GnomeDbGrid widget.
 *
 * Returns the list of the currently selected rows in a
 * #GnomeDbGrid widget. The returned value is a list of integers,
 * which represent each of the selected rows.
 *
 * Returns: a GList of integers. This list should be freed (by calling
 * #g_list_free) when no longer needed.
 */
GList *
gnome_db_grid_get_selection (GnomeDbGrid *grid)
{	
	GtkTreeSelection *selection;
	GList *list = NULL;

	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), NULL);

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->tree_view));
	gtk_tree_selection_selected_foreach (
		selection,
		(GtkTreeSelectionForeachFunc) foreach_selected_cb,
		&list);

	return list;
}

/**
 * gnome_db_grid_foreach_selected
 * @grid: a #GnomeDbGrid widget.
 * @foreach_func: function to be called for each selected row.
 * @user_data: data to pass over to @foreach_func.
 *
 * Call the given function for each selected row in the #GnomeDbGrid
 * widget.
 */
void
gnome_db_grid_foreach_selected (GnomeDbGrid *grid,
				GnomeDbGridForeachFunc foreach_func,
				gpointer user_data)
{
	GList *selection;
	GList *l;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	g_return_if_fail (foreach_func != NULL);

	selection = gnome_db_grid_get_selection (grid);
	for (l = selection; l != NULL; l = l->next)
		foreach_func (grid, GPOINTER_TO_INT (l->data), user_data);
}

/**
 * gnome_db_grid_select_all
 */
void
gnome_db_grid_select_all (GnomeDbGrid *grid)
{
	GtkTreeSelection *selection;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->tree_view));
	gnome_db_grid_set_selection_mode (grid, GTK_SELECTION_MULTIPLE);
	gtk_tree_selection_select_all (selection);
}

/**
 * gnome_db_grid_unselect_all
 */
void
gnome_db_grid_unselect_all (GnomeDbGrid *grid)
{
	GtkTreeSelection *selection;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->tree_view));
	gtk_tree_selection_unselect_all (selection);
}

/**
 * gnome_db_grid_get_selection_mode
 */
GtkSelectionMode
gnome_db_grid_get_selection_mode (GnomeDbGrid *grid)
{
	GtkTreeSelection *selection;

	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), GTK_SELECTION_NONE);

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->tree_view));
	return gtk_tree_selection_get_mode (selection);
}

/**
 * gnome_db_set_selection_mode
 */
void
gnome_db_grid_set_selection_mode (GnomeDbGrid *grid, GtkSelectionMode mode)
{
	GtkTreeSelection *selection;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->tree_view));
	gtk_tree_selection_set_mode (selection, mode);

	grid->priv->selection_mode = mode;
}

/**
 * gnome_db_grid_hide_column_titles
 */
void
gnome_db_grid_hide_column_titles (GnomeDbGrid *grid)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (grid->priv->tree_view), FALSE);
}

/**
 * gnome_db_grid_show_column_titles
 */
void
gnome_db_grid_show_column_titles (GnomeDbGrid *grid)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (grid->priv->tree_view), TRUE);
}
