/* GNOME-DB Components
 * Copyright (C) 2000-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <libgda/gda-client.h>
#include <libgda/gda-config.h>
#include <libgda/gda-data-model-array.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktable.h>
#include <gtk/gtkvbbox.h>
#include <libgnomedb/gnome-db-config.h>
#include <libgnomedb/gnome-db-dsn-config-druid.h>
#include <libgnomedb/gnome-db-dsn-config.h>
#include <libgnomedb/gnome-db-grid.h>
#include <libgnomedb/gnome-db-i18n.h>
#include <libgnomedb/gnome-db-login-dialog.h>
#include <libgnomedb/gnome-db-login.h>
#include <libgnomedb/gnome-db-stock.h>
#include <libgnomedb/gnome-db-util.h>
#include <libgnomeui/gnome-window-icon.h>
#include "dsn-config.h"
#include "dsn-properties-dialog.h"

#define DSN_CONFIG_DATA "DSN_ConfigData"

typedef struct {
	GtkWidget *dsn_list;
	GtkWidget *dialog;
	gint config_listener;
} DsnConfigPrivate;

static void
free_private_data (gpointer data)
{
	DsnConfigPrivate *priv = (DsnConfigPrivate *) data;

	if (priv->config_listener != -1) {
		gnome_db_config_remove_listener (priv->config_listener);
		priv->config_listener = -1;
	}

	g_free (priv);
}

static void
show_data_sources (DsnConfigPrivate *priv)
{
	GList *dsn_list;
	GList *l;
	GdaDataModel *model;

	/* create the data model */
	model = gda_data_model_array_new (2);
	gda_data_model_set_column_title (model, 0, _("Name"));
	gda_data_model_set_column_title (model, 1, _("Description"));

	dsn_list = gnome_db_config_get_data_source_list ();
	for (l = dsn_list; l != NULL; l = l->next) {
		GList *value_list;
		GdaDataSourceInfo *dsn_info = l->data;

		value_list = g_list_append (NULL, gda_value_new_string (dsn_info->name));
		value_list = g_list_append (value_list, gda_value_new_string (dsn_info->description));

		gda_data_model_append_row (model, (const GList *) value_list);

		g_list_foreach (value_list, (GFunc) gda_value_free, NULL);
		g_list_free (value_list);
	}

	gnome_db_config_free_data_source_list (dsn_list);

	/* display the new data model */
	gnome_db_grid_set_model (GNOME_DB_GRID (priv->dsn_list), model);
	g_object_unref (G_OBJECT (model));
}

/*
 * Callbacks
 */

static void
config_changed_cb (GConfClient *conf_client, guint cnxn_id, GConfEntry *entry, gpointer user_data)
{
	show_data_sources (user_data);
}

static void
list_double_clicked_cb (GnomeDbGrid *grid, gint row, gpointer user_data)
{
	dsn_config_edit_properties (GTK_WIDGET (user_data));
}

/*
 * Public functions
 */

GtkWidget *
dsn_config_new (void)
{
	DsnConfigPrivate *priv;
	GtkWidget *dsn;
	GtkWidget *table;
	GtkWidget *box;
	GtkWidget *button;
	GtkWidget *label;
	gchar *title;

	priv = g_new0 (DsnConfigPrivate, 1);
	dsn = gnome_db_new_vbox_widget (FALSE, 6);
        gtk_container_set_border_width (GTK_CONTAINER (dsn), 6);
	g_object_set_data_full (G_OBJECT (dsn), DSN_CONFIG_DATA, priv,
				(GDestroyNotify) free_private_data);

	/* set up config listener */
	priv->config_listener = gnome_db_config_add_listener ("/apps/libgda/Datasources",
							      (GConfClientNotifyFunc) config_changed_cb,
							       priv);

	/* create the main table */
	table = gnome_db_new_table_widget (2, 1, FALSE);
	gtk_box_pack_start (GTK_BOX (dsn), table, TRUE, TRUE, 0);

	/* create the data source list */
	priv->dsn_list = gnome_db_grid_new ();

	title = g_strdup_printf ("<b>%s</b>\n%s", _("Data Sources"),
				 _("Configured data sources in the system"));
	gnome_db_grid_set_title (GNOME_DB_GRID (priv->dsn_list), title);
	g_free (title);

	gnome_db_grid_set_show_title (GNOME_DB_GRID (priv->dsn_list), TRUE);
	gnome_db_grid_set_show_title_icon (GNOME_DB_GRID (priv->dsn_list), TRUE);
	gnome_db_grid_set_title_icon_from_stock (GNOME_DB_GRID (priv->dsn_list), GNOME_DB_STOCK_DATABASE, GTK_ICON_SIZE_BUTTON);
	gtk_widget_show (priv->dsn_list);
	g_signal_connect (G_OBJECT (priv->dsn_list), "double_clicked",
			  G_CALLBACK (list_double_clicked_cb), dsn);
	gtk_table_attach (GTK_TABLE (table), priv->dsn_list, 0, 1, 0, 1,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  0, 0);
	show_data_sources (priv);

	/* add tip */
	box = gnome_db_new_hbox_widget (FALSE, 6);
        gtk_container_set_border_width (GTK_CONTAINER (box), 6);
	gtk_table_attach (GTK_TABLE (table), box, 0, 1, 1, 2,
			  GTK_FILL,
			  GTK_FILL,
                          0, 0);

	button = gtk_image_new_from_stock (GTK_STOCK_DIALOG_INFO, GTK_ICON_SIZE_DIALOG);
        gtk_misc_set_alignment (GTK_MISC (button), 0.5, 0.0);
	gtk_widget_show (button);
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 0);

	label = gnome_db_new_label_widget (
		_("Data sources are the means by which database "
		  "connections are identified in GNOME-DB. All "
		  "information needed to open a connection to "
		  "a specific provider/database combo is stored using "
		  "a unique name. It is by use of this unique name "
		  "you identify the connections in the applications "
		  "that make use of GNOME-DB for database access."));
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
        gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
        gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start (GTK_BOX (box), label, TRUE, TRUE, 0);
	
	return dsn;
}

void
dsn_config_edit_properties (GtkWidget *dsn)
{
	DsnConfigPrivate *priv;
	GdaDataModel *model;
	GList *selection;
	gchar *str;

	priv = g_object_get_data (G_OBJECT (dsn), DSN_CONFIG_DATA);

	selection = gnome_db_grid_get_selection (GNOME_DB_GRID (priv->dsn_list));
	if (!selection)
		return;

	model = gnome_db_grid_get_model (GNOME_DB_GRID (priv->dsn_list));
	if (!GDA_IS_DATA_MODEL (model))
		return;

	str = gda_value_stringify (
		(GdaValue *) gda_data_model_get_value_at (model, 0, GPOINTER_TO_INT (selection->data)));

	dsn_properties_dialog (GTK_WINDOW (gtk_widget_get_toplevel (dsn)), str);

	g_list_free (selection);
	g_free (str);
}

static void
foreach_remove_cb (GnomeDbGrid *grid, gint row, gpointer user_data)
{
	GdaDataModel *model;
	gchar *str;
	DsnConfigPrivate *priv = (DsnConfigPrivate *) user_data;

	model = gnome_db_grid_get_model (GNOME_DB_GRID (priv->dsn_list));
	if (!GDA_IS_DATA_MODEL (model))
		return;

	str = gda_value_stringify (
		(GdaValue *) gda_data_model_get_value_at (model, 0, row));
	if (!str)
		return;

	if (gnome_db_yes_no_dialog (
		    gtk_widget_get_toplevel (priv->dsn_list),
		    _("Do you want to remove '%s' data source?"),
		    str)) {
		gnome_db_config_remove_data_source (str);
	}

	g_free (str);
}

void
dsn_config_delete (GtkWidget *dsn)
{
	DsnConfigPrivate *priv;

	priv = g_object_get_data (G_OBJECT (dsn), DSN_CONFIG_DATA);
	gnome_db_grid_foreach_selected (GNOME_DB_GRID (priv->dsn_list),
					(GnomeDbGridForeachFunc) foreach_remove_cb,
					priv);
}
