/* GNOME-DB Library
 * Copyright (C) 2000-2003 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkstock.h>
#include <libgnomedb/gnome-db-error-dialog.h>
#include <libgnomedb/gnome-db-i18n.h>
#include <libgnomedb/gnome-db-login-dialog.h>
#include <libgnomedb/gnome-db-login.h>
#include <libgnomedb/gnome-db-stock.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include "utils.h"

static GdaClient *db_client = NULL;

static void
client_notification_cb (GdaClient *client,
			GdaConnection *cnc,
			GdaClientEvent event,
			GdaParameterList *params)
{
	GtkWidget *dialog;

	switch (event) {
	case GDA_CLIENT_EVENT_ERROR :
		dialog = gnome_db_error_dialog_new (_("Error"));
		gnome_db_error_dialog_show_errors (GNOME_DB_ERROR_DIALOG (dialog),
						   gda_connection_get_errors (cnc));
	default :
	}
}

GdaConnection *
db_connect_dialog_for_dsn (const gchar *dsn)
{
	GtkWidget *dialog, *login;
	gchar *title;
	GdaConnection *cnc = NULL;

	title = g_strdup_printf (_("Login for %s"), dsn);
	dialog = gnome_db_login_dialog_new (title);
	login = gnome_db_login_dialog_get_login_widget (GNOME_DB_LOGIN_DIALOG (dialog));
	gnome_db_login_set_dsn (GNOME_DB_LOGIN (login), dsn);

	if (gnome_db_login_dialog_run (GNOME_DB_LOGIN_DIALOG (dialog))) {
		if (!db_client) {
			db_client = gda_client_new ();
			g_signal_connect (db_client, "event_notification",
					  G_CALLBACK (client_notification_cb), NULL);
		}

		cnc = gda_client_open_connection (db_client,
						  gnome_db_login_get_dsn (GNOME_DB_LOGIN (login)),
						  gnome_db_login_get_username (GNOME_DB_LOGIN (login)),
						  gnome_db_login_get_password (GNOME_DB_LOGIN (login)),
						  0);
	}

	gtk_widget_destroy (dialog);

	return cnc;
}

GdaConnection *
db_connect_dialog (GdaXmlConnection *xmlcnc)
{
	GtkWidget *dialog, *login;
	gchar *title;
	GdaConnection *cnc = NULL;

	if (GDA_IS_XML_CONNECTION (xmlcnc))
		title = g_strdup_printf (_("Login for %s"), gda_xml_connection_get_dsn (xmlcnc));
	else
		title = g_strdup (_("Login to database"));

	dialog = gnome_db_login_dialog_new (title);
	login = gnome_db_login_dialog_get_login_widget (GNOME_DB_LOGIN_DIALOG (dialog));
	if (xmlcnc) {
		gnome_db_login_set_dsn (GNOME_DB_LOGIN (login), gda_xml_connection_get_dsn (xmlcnc));
		gnome_db_login_set_username (GNOME_DB_LOGIN (login), gda_xml_connection_get_username (xmlcnc));
		gnome_db_login_set_password (GNOME_DB_LOGIN (login), gda_xml_connection_get_password (xmlcnc));
	}

	if (gnome_db_login_dialog_run (GNOME_DB_LOGIN_DIALOG (dialog))) {
		if (!db_client) {
			db_client = gda_client_new ();
			g_signal_connect (db_client, "event_notification",
					  G_CALLBACK (client_notification_cb), NULL);
		}

		cnc = gda_client_open_connection (db_client,
						  gnome_db_login_get_dsn (GNOME_DB_LOGIN (login)),
						  gnome_db_login_get_username (GNOME_DB_LOGIN (login)),
						  gnome_db_login_get_password (GNOME_DB_LOGIN (login)),
						  0);
	}

	gtk_widget_destroy (dialog);

	return cnc;
}

GdaConnection *
db_connect (const gchar *uri)
{
	GnomeVFSHandle *handle;
	GString *str = NULL;
	GdaConnection *cnc = NULL;

	g_return_val_if_fail (uri != NULL, NULL);

	if (!uri)
		return db_connect_dialog (NULL);

	if (gnome_vfs_open (&handle, uri, GNOME_VFS_OPEN_READ) == GNOME_VFS_OK) {
		GnomeVFSResult vfsres;
		gchar buffer[8193];

		do {
			GnomeVFSFileSize bytes_read;

			memset (buffer, 0, sizeof (buffer));
			vfsres = gnome_vfs_read (handle, buffer, sizeof (buffer) - 1, &bytes_read);
			if (vfsres == GNOME_VFS_OK) {
				if (!str)
					str = g_string_new (buffer);
				else
					str = g_string_append (str, buffer);
			} else if (vfsres == GNOME_VFS_ERROR_EOF && str) {
				GdaXmlConnection *xmlcnc;

				xmlcnc = gda_xml_connection_new_from_string (str->str);
				if (xmlcnc) {
					cnc = db_connect_dialog (xmlcnc);
					g_object_unref (xmlcnc);
				}
			}
		} while (vfsres == GNOME_VFS_OK);

		gnome_vfs_close (handle);
		if (str)
			g_string_free (str, TRUE);
	}
	
	return cnc;
}
