/* GNOME-DB Components
 * Copyright (C) 2000 - 2006 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *      Vivien Malerba <malerba@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib/gi18n-lib.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtklabel.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktable.h>
#include <libgnomedb/libgnomedb.h>
#include "dsn-properties-dialog.h"

enum {
	REVERT_BUTTON,
	TEST_BUTTON,
	BROWSE_BUTTON
};

static GdaClient *db_client = NULL;

static void
save_dsn (GnomeDbDsnConfig *props)
{
	GdaDataSourceInfo *dsn_info;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG (props));
	dsn_info = gnome_db_dsn_config_get_info (GNOME_DB_DSN_CONFIG (props));
	if (dsn_info) {
		gda_config_save_data_source_info (dsn_info);
		gda_data_source_info_free (dsn_info);
	}
}

static void
dsn_changed_cb (GnomeDbDsnConfig *config, gpointer user_data)
{
	gboolean changed;
	GdaDataSourceInfo *dsn_info , *old_dsn_info;

	dsn_info = gnome_db_dsn_config_get_info (config);
	old_dsn_info = g_object_get_data (G_OBJECT (user_data), "old_dsn_info");

	changed = !gda_data_source_info_equal (dsn_info, old_dsn_info);
	gtk_dialog_set_response_sensitive (GTK_DIALOG (user_data), REVERT_BUTTON, changed);

	save_dsn (config);
	if (dsn_info)
		gda_data_source_info_free (dsn_info);
}

GdaConnection *
do_connect_dialog_for_dsn (const gchar *dsn)
{
	GtkWidget *dialog, *login;
	gchar *title;
	GdaConnection *cnc = NULL;

	title = g_strdup_printf (_("Login for %s"), dsn);
	dialog = gnome_db_login_dialog_new (title);
	login = gnome_db_login_dialog_get_login_widget (GNOME_DB_LOGIN_DIALOG (dialog));
	gnome_db_login_set_dsn (GNOME_DB_LOGIN (login), dsn);

	if (gnome_db_login_dialog_run (GNOME_DB_LOGIN_DIALOG (dialog))) {
		if (!db_client) 
			db_client = gda_client_new ();

		cnc = gda_client_open_connection (db_client,
						  gnome_db_login_get_dsn (GNOME_DB_LOGIN (login)),
						  gnome_db_login_get_username (GNOME_DB_LOGIN (login)),
						  gnome_db_login_get_password (GNOME_DB_LOGIN (login)),
						  0, NULL);
	}

	gtk_widget_destroy (dialog);

	return cnc;
}

void
dsn_properties_dialog (GtkWindow *parent, const gchar *dsn)
{
	GdaDataSourceInfo *old_dsn_info;
	GtkWidget *dialog, *props, *label, *hbox;
	GdkPixbuf *icon;
	gchar *str;
	gint result;

	old_dsn_info = gda_config_find_data_source (dsn);
	if (!old_dsn_info)
		return;

	/* create the dialog */
	dialog = gtk_dialog_new_with_buttons (_("Data Source Properties"),
					      parent, GTK_DIALOG_DESTROY_WITH_PARENT,
					      _("Test"), TEST_BUTTON,
					      _("Browse"), BROWSE_BUTTON,
					      GTK_STOCK_REVERT_TO_SAVED, REVERT_BUTTON,
					      GTK_STOCK_CLOSE, GTK_RESPONSE_OK,
					      NULL);
	gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), REVERT_BUTTON, FALSE);
	g_object_set_data (G_OBJECT (dialog), "old_dsn_info", old_dsn_info);

	icon = gdk_pixbuf_new_from_file (LIBGNOMEDB_ICONSDIR "/gnome-db.png", NULL);
	if (icon) {
		gtk_window_set_icon (GTK_WINDOW (dialog), icon);
		g_object_unref (icon);
	}

	/* textual explanation */
	if (!old_dsn_info->is_global || gda_config_can_modify_global_config ())
		str = g_strdup_printf ("<b>%s:</b>\n<small>%s</small>",
				       _("Data Source Properties"),
				       _("Change the data source properties (the name can't be modified)."));
	else
		str = g_strdup_printf ("<b>%s:</b>\n<small>%s</small>",
				       _("Data Source Properties"),
				       _("For information only, this data source is a system wide data source\n"
					 "and you don't have the permission change it."));

	label = gtk_label_new ("");
        gtk_label_set_markup (GTK_LABEL (label), str);
        gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
        g_free (str);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, FALSE, FALSE, 10);
        gtk_widget_show (hbox);
        label = gtk_label_new ("    ");
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
        gtk_widget_show (label);

	/* create data source settings page */
	props = gnome_db_dsn_config_new ();
	gnome_db_dsn_config_set_info (GNOME_DB_DSN_CONFIG (props), old_dsn_info);
	gtk_widget_show (props);
	g_signal_connect (G_OBJECT (props), "changed", G_CALLBACK (dsn_changed_cb), dialog);	
	gtk_box_pack_start (GTK_BOX (hbox), props, TRUE, TRUE, 0);
	gtk_widget_set_size_request (props, 500, -1);

	/* run the dialog */
	do {
		result = gtk_dialog_run (GTK_DIALOG (dialog));
		switch (result) {
		case REVERT_BUTTON:
			/* reverting changes... */
			gnome_db_dsn_config_set_info (GNOME_DB_DSN_CONFIG (props), old_dsn_info);
			gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), REVERT_BUTTON, FALSE);
			break;
		case TEST_BUTTON:
		{
			GtkWidget *dialog, *login;
			gchar *title;
			GdaConnection *cnc = NULL;

			title = g_strdup_printf (_("Login for %s"), dsn);
			dialog = gnome_db_login_dialog_new (title);
			g_free (title);
			login = gnome_db_login_dialog_get_login_widget (GNOME_DB_LOGIN_DIALOG (dialog));
			gnome_db_login_set_dsn (GNOME_DB_LOGIN (login), dsn);
			gnome_db_login_set_show_dsn_selector (GNOME_DB_LOGIN (login), FALSE);

			if (gnome_db_login_dialog_run (GNOME_DB_LOGIN_DIALOG (dialog))) {
				GtkWidget *msgdialog;
				GError *error = NULL;

				if (!db_client) 
					db_client = gda_client_new ();
				
				cnc = gda_client_open_connection (db_client,
								  gnome_db_login_get_dsn (GNOME_DB_LOGIN (login)),
								  gnome_db_login_get_username (GNOME_DB_LOGIN (login)),
								  gnome_db_login_get_password (GNOME_DB_LOGIN (login)),
								  0, &error);
				if (cnc && GDA_IS_CONNECTION (cnc)) {
					str = g_strdup_printf ("<b>%s</b>", _("Connection successfully opened!"));
					msgdialog = gtk_message_dialog_new_with_markup (GTK_WINDOW (dialog), GTK_DIALOG_MODAL,
											GTK_MESSAGE_INFO, GTK_BUTTONS_OK, str);
					g_free (str);
					gda_connection_close (cnc);
				}
				else {
					if (error) {
						str = g_strdup_printf ("<b>%s:</b>\n%s", _("Could not open connection"), 
								       error->message ? error->message : _("Unknown error"));
						g_error_free (error);
					}
					else
						str = g_strdup_printf ("<b>%s</b>", _("Could not open connection"));
					msgdialog = gtk_message_dialog_new_with_markup (GTK_WINDOW (dialog), GTK_DIALOG_MODAL,
											GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, str);
					g_free (str);
				}
				
				gtk_dialog_run (GTK_DIALOG (msgdialog));
				gtk_widget_destroy (msgdialog);
			}
			gtk_widget_destroy (dialog);
		}
		break;
		case BROWSE_BUTTON:
		{
			char *argv[3];
			gboolean sresult;
			
			/* run gnome-database-properties dictig tool */
			argv[0] = (char *) "gnome-db-browser";
			argv[1] = (char *) dsn;
			argv[2] = NULL;
			
			sresult = g_spawn_async (NULL, argv, NULL, G_SPAWN_SEARCH_PATH,
						NULL, NULL, NULL, NULL);
			if (!sresult) {
				GtkWidget *msgdialog;
				str = g_strdup_printf ("<b>%s</b>", _("Could not execute browser program (gnome-db-browser)."));
				msgdialog = gtk_message_dialog_new_with_markup (GTK_WINDOW (dialog), GTK_DIALOG_MODAL,
										GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, str);
				g_free (str);
				gtk_dialog_run (GTK_DIALOG (msgdialog));
				gtk_widget_destroy (msgdialog);
			}
			else
				result = GTK_RESPONSE_OK; /* force closing of this dialog */
		}
		break;
		default:
			break;
		}
	}
	while (result != GTK_RESPONSE_OK);
	gtk_widget_destroy (dialog);

	/* free memory */
	gda_data_source_info_free (old_dsn_info);
}
