/* gnome-db-goo-join.c
 *
 * Copyright (C) 2004 - 2007 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gtk/gtk.h>
#include <libgda/libgda.h>
#include <glib/gi18n-lib.h>
#include "gnome-db-goo.h"
#include "gnome-db-goo-join.h"
#include "gnome-db-goo-entity.h"
#include "gnome-db-goo-text.h"
#include "gnome-db-goo-utility.h"

static void gnome_db_goo_join_class_init (GnomeDbGooJoinClass * class);
static void gnome_db_goo_join_init       (GnomeDbGooJoin * cc);
static void gnome_db_goo_join_dispose    (GObject   * object);
static void gnome_db_goo_join_finalize   (GObject   * object);

static void gnome_db_goo_join_set_property    (GObject *object,
					    guint param_id,
					    const GValue *value,
					    GParamSpec *pspec);
static void gnome_db_goo_join_get_property    (GObject *object,
					    guint param_id,
					    GValue *value,
					    GParamSpec *pspec);

static void gnome_db_goo_join_set_join (GnomeDbGooJoin *cc, GdaQueryJoin *join);
static void clean_items (GnomeDbGooJoin *cc);
static void create_items (GnomeDbGooJoin *cc);
static void update_items (GnomeDbGooJoin *cc);

enum
{
	PROP_0,
	PROP_JOIN
};

struct _GnomeDbGooJoinPrivate
{
	GdaQueryJoin       *join;
	GnomeDbGooEntity   *target1_entity;
	GnomeDbGooEntity   *target2_entity;
	GSList             *shapes; /* list of GnomeDbGooCanvasShape structures */
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

GType
gnome_db_goo_join_get_type (void)
{
	static GType type = 0;

        if (G_UNLIKELY (type == 0)) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGooJoinClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_goo_join_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGooJoin),
			0,
			(GInstanceInitFunc) gnome_db_goo_join_init
		};		

		type = g_type_register_static (GNOME_DB_TYPE_GOO_ITEM, "GnomeDbGooJoin", &info, 0);
	}

	return type;
}	

static void
gnome_db_goo_join_class_init (GnomeDbGooJoinClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_goo_join_dispose;
	object_class->finalize = gnome_db_goo_join_finalize;

	/* Properties */
	object_class->set_property = gnome_db_goo_join_set_property;
	object_class->get_property = gnome_db_goo_join_get_property;
	g_object_class_install_property (object_class, PROP_JOIN,
					 g_param_spec_pointer ("join", "Query's join to represent", 
							       NULL, G_PARAM_WRITABLE));
       
}

static void
gnome_db_goo_join_init (GnomeDbGooJoin *cc)
{
	cc->priv = g_new0 (GnomeDbGooJoinPrivate, 1);
	cc->priv->join = NULL;
	cc->priv->target1_entity = NULL;
	cc->priv->target2_entity = NULL;
}


static void entity_destroy_cb (GnomeDbGooEntity *entity, GnomeDbGooJoin *cc);
static void join_destroyed_cb (GdaQueryJoin *join, GnomeDbGooJoin *cc);
static void join_changed_cb (GdaQueryJoin *join, GnomeDbGooJoin *cc);

static void
gnome_db_goo_join_dispose (GObject *object)
{
	GnomeDbGooJoin *cc;
	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_GOO_JOIN (object));
	
	cc = GNOME_DB_GOO_JOIN (object);

	clean_items (cc);
	if (cc->priv->join) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->join),
						      G_CALLBACK (join_destroyed_cb), cc);
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->join),
						      G_CALLBACK (join_changed_cb), cc);
		cc->priv->join = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}


static void
gnome_db_goo_join_finalize (GObject *object)
{
	GnomeDbGooJoin *cc;
	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_GOO_JOIN (object));

	cc = GNOME_DB_GOO_JOIN (object);
	if (cc->priv) {
		g_free (cc->priv);
		cc->priv = NULL;
	}

	/* for the parent class */
	parent_class->finalize (object);
}

static void 
gnome_db_goo_join_set_property    (GObject *object,
				guint param_id,
				const GValue *value,
				GParamSpec *pspec)
{
	GnomeDbGooJoin *cc;

	cc = GNOME_DB_GOO_JOIN (object);

	switch (param_id) {
	case PROP_JOIN:
		gnome_db_goo_join_set_join (cc, (GdaQueryJoin *) g_value_get_pointer (value));
		break;
	}
}

static void 
gnome_db_goo_join_get_property    (GObject *object,
				guint param_id,
				GValue *value,
				GParamSpec *pspec)
{
	GnomeDbGooJoin *cc;

	cc = GNOME_DB_GOO_JOIN (object);

	switch (param_id) {
	default:
		g_warning ("No such property!");
		break;
	}
}

static void
join_destroyed_cb (GdaQueryJoin *join, GnomeDbGooJoin *cc)
{
	goo_canvas_item_remove (GOO_CANVAS_ITEM (cc));	
}

static void
join_changed_cb (GdaQueryJoin *join, GnomeDbGooJoin *cc)
{
	clean_items (cc);
	create_items (cc);
}

static void
entity_destroy_cb (GnomeDbGooEntity *entity, GnomeDbGooJoin *cc)
{
	goo_canvas_item_remove (GOO_CANVAS_ITEM (cc));
}

static gboolean single_item_enter_notify_event_cb (GooCanvasItem *ci, GooCanvasItem *target_item,
                                                   GdkEventCrossing *event, GnomeDbGooJoin *cc);
static gboolean single_item_leave_notify_event_cb (GooCanvasItem *ci, GooCanvasItem *target_item,
                                                   GdkEventCrossing *event, GnomeDbGooJoin *cc);
static gboolean single_item_button_press_event_cb (GooCanvasItem *ci, GooCanvasItem *target_item,
                                                   GdkEventButton *event, GnomeDbGooJoin *cc);
static void entity_item_moved_cb (GooCanvasItem *entity, GnomeDbGooJoin *cc);

typedef struct {
	GnomeDbGooJoin    *cc;
	GdaQueryCondition *scond;
} ForeachData;

static void
shapes_foreach_connect_signals (GnomeDbGooCanvasShape *shape, ForeachData *data)
{
	if (shape->is_new) {
		g_signal_connect (G_OBJECT (shape->item), "enter-notify-event",
				  G_CALLBACK (single_item_enter_notify_event_cb), data->cc);
		g_signal_connect (G_OBJECT (shape->item), "leave-notify-event",
				  G_CALLBACK (single_item_leave_notify_event_cb), data->cc);
		g_signal_connect (G_OBJECT (shape->item), "button-press-event",
				  G_CALLBACK (single_item_button_press_event_cb), data->cc);
		g_object_set_data (G_OBJECT (shape->item), "cond", data->scond);
		shape->is_new = FALSE;
		/*g_print ("Shape item %p => cond=%p (%s)\n", shape->item, data->scond, G_OBJECT_TYPE_NAME (shape->item));*/
	}
}


/* 
 * destroy any existing GooCanvasItem objects 
 */
static void 
clean_items (GnomeDbGooJoin *cc)
{
	if (cc->priv->target1_entity) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->target1_entity),
						      G_CALLBACK (entity_item_moved_cb), cc);
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->target1_entity),
						      G_CALLBACK (entity_destroy_cb), cc);
		cc->priv->target1_entity = NULL;
	}

	if (cc->priv->target2_entity) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->target2_entity),
						      G_CALLBACK (entity_item_moved_cb), cc);
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->target2_entity),
						      G_CALLBACK (entity_destroy_cb), cc);
		cc->priv->target2_entity = NULL;
	}

	/* remove all the GooCanvasItem objects */
	gnome_db_goo_canvas_shapes_remove_all (cc->priv->shapes);
        cc->priv->shapes = NULL;
}

static void
create_or_update_shapes (GnomeDbGooJoin *cc)
{
	GdaQueryCondition *cond;
	gboolean cond_is_strict = FALSE;
	GdaQueryTarget *target1;
	guint extra_mode = 0;
	GdaQueryJoinType join_type;

	/* Analyse Join */
	target1 = gda_query_join_get_target_1 (cc->priv->join);

	cond = gda_query_join_get_condition (cc->priv->join);
	if (cond) 
		cond_is_strict = gda_query_condition_represents_join_strict (cond, NULL, NULL);

	/* Analyse Join */
	target1 = gda_query_join_get_target_1 (cc->priv->join);
	cond = gda_query_join_get_condition (cc->priv->join);
	if (cond) 
		cond_is_strict = gda_query_condition_represents_join_strict (cond, NULL, NULL);
	join_type = gda_query_join_get_join_type (cc->priv->join);
	switch (join_type) {
	case GDA_QUERY_JOIN_TYPE_LEFT_OUTER:
		extra_mode = CANVAS_SHAPE_EXT_JOIN_OUTER_1;
		break;
	case GDA_QUERY_JOIN_TYPE_RIGHT_OUTER:
		extra_mode = CANVAS_SHAPE_EXT_JOIN_OUTER_2;
		break;
	case GDA_QUERY_JOIN_TYPE_FULL_OUTER:
		extra_mode = CANVAS_SHAPE_EXT_JOIN_OUTER_1 | CANVAS_SHAPE_EXT_JOIN_OUTER_2;
	default:
		break;
	}

	/* actual line(s) */
	if (cond_is_strict) {
		/* lines from field to field for each sub-condition */
		gint nb_scond, i;
		GSList *condlist, *canvas_shapes, *clist;

		condlist = gda_query_condition_get_main_conditions (cond);
		nb_scond = g_slist_length (condlist);
		for (clist = condlist, i = 0; clist; clist = clist->next, i++) {
			GdaQueryCondition *scond = GDA_QUERY_CONDITION (clist->data);
			GdaQueryTarget *t;
			GdaEntityField *f1, *f2;
			GdaQueryField *tmp;
			ForeachData fhdata;

			g_assert (gda_query_condition_get_cond_type (scond) == GDA_QUERY_CONDITION_LEAF_EQUAL);
			tmp =  gda_query_condition_leaf_get_operator (scond, GDA_QUERY_CONDITION_OP_LEFT);
			f1 = gda_query_field_field_get_ref_field (GDA_QUERY_FIELD_FIELD (tmp));
			t = gda_query_field_field_get_target (GDA_QUERY_FIELD_FIELD (tmp));
			tmp =  gda_query_condition_leaf_get_operator (scond, GDA_QUERY_CONDITION_OP_RIGHT);
			f2 = gda_query_field_field_get_ref_field (GDA_QUERY_FIELD_FIELD (tmp));

			if (t == target1)
				canvas_shapes = gnome_db_goo_util_compute_connect_shapes (GOO_CANVAS_ITEM (cc), 
											  cc->priv->shapes,
											  cc->priv->target1_entity, f1,
											  cc->priv->target2_entity, f2, 
											  i, extra_mode);
			else {
				guint em2 = extra_mode;
				if (em2 == CANVAS_SHAPE_EXT_JOIN_OUTER_1)
					em2 = CANVAS_SHAPE_EXT_JOIN_OUTER_2;
				else {
					if (em2 == CANVAS_SHAPE_EXT_JOIN_OUTER_2)
						em2 = CANVAS_SHAPE_EXT_JOIN_OUTER_1;
				}
				canvas_shapes = gnome_db_goo_util_compute_connect_shapes (GOO_CANVAS_ITEM (cc), 
											  cc->priv->shapes,
											  cc->priv->target2_entity, f1,
											  cc->priv->target1_entity, f2, 
											  i, em2);
			}
			fhdata.cc = cc;
			fhdata.scond = nb_scond > 1 ? scond : NULL;
			cc->priv->shapes = canvas_shapes;
			g_slist_foreach (cc->priv->shapes, (GFunc) shapes_foreach_connect_signals, &fhdata);
		}
		g_slist_free (condlist);
	}
	else {
		/* one global shape from one entity to the other */
		GSList *canvas_shapes;
		ForeachData fhdata;
		canvas_shapes = gnome_db_goo_util_compute_anchor_shapes (GOO_CANVAS_ITEM (cc), cc->priv->shapes,
									 cc->priv->target1_entity, 
									 cc->priv->target2_entity, 1, extra_mode, FALSE);
		cc->priv->shapes = canvas_shapes;
		fhdata.cc = cc;
		fhdata.scond = NULL;
		g_slist_foreach (canvas_shapes, (GFunc) shapes_foreach_connect_signals, &fhdata);
	}
}

/*
 * create new GooCanvasItem objects
 */
static void 
create_items (GnomeDbGooJoin *cc)
{
	GooCanvasItem *item;
	GdaQueryCondition *cond;
	GnomeDbGooItem *entity_item;
	GnomeDbGoo *canvas = gnome_db_goo_item_get_canvas (GNOME_DB_GOO_ITEM (cc));
	GdaQueryTarget *target1, *target2;
	
	if (!cc->priv->join)
		return;
	g_assert (! cc->priv->shapes);

	cond = gda_query_join_get_condition (cc->priv->join);
	
	/* Analyse Join */
	item = GOO_CANVAS_ITEM (cc);
	target1 = gda_query_join_get_target_1 (cc->priv->join);
	entity_item = gnome_db_goo_get_item_for_object (canvas, GDA_OBJECT (target1));
	if (!entity_item) {
		g_warning ("Can't find GnomeDbGooItem object for a join target '%s'; perhaps you forgot to "
			   "call gda_graph_query_sync_targets()?", gda_object_get_name (GDA_OBJECT (target1)));
		return;
	}
	cc->priv->target1_entity = GNOME_DB_GOO_ENTITY (entity_item);
	g_signal_connect (G_OBJECT (entity_item), "moving",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "moved",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "shifted",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "destroy",
			  G_CALLBACK (entity_destroy_cb), cc);

	target2 = gda_query_join_get_target_2 (cc->priv->join);
	entity_item = gnome_db_goo_get_item_for_object (canvas, GDA_OBJECT (target2));
	if (!entity_item) {
		g_warning ("Can't find GnomeDbGooItem object for a join target '%s'; perhaps you forgot to "
			   "call gda_graph_query_sync_targets()?", gda_object_get_name (GDA_OBJECT (target2)));
		return;
	}
	cc->priv->target2_entity = GNOME_DB_GOO_ENTITY (entity_item);
	g_signal_connect (G_OBJECT (entity_item), "moving",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "moved",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "shifted",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "destroy",
			  G_CALLBACK (entity_destroy_cb), cc);

	create_or_update_shapes (cc);
}


/*
 * update GooCanvasItem objects
 */
static void 
update_items (GnomeDbGooJoin *cc)
{
	if (!cc->priv->join)
		return;

	create_or_update_shapes (cc);
	cc->priv->shapes = gnome_db_goo_canvas_shapes_remove_obsolete_shapes (cc->priv->shapes);
}

static void popup_delete_cb (GtkMenuItem *mitem, GnomeDbGooJoin *cc);
static void popup_join_properties_cb (GtkMenuItem *mitem, GnomeDbGooJoin *cc);

static gboolean
single_item_enter_notify_event_cb (GooCanvasItem *ci, GooCanvasItem *target_item,
                                   GdkEventCrossing *event, GnomeDbGooJoin *cc)
{
	GSList *list;
	GdaQueryCondition *cond = g_object_get_data (G_OBJECT (ci), "cond");

	for (list = cc->priv->shapes; list; list = list->next) {
		GooCanvasItem *item = GNOME_DB_GOO_CANVAS_SHAPE (list->data)->item;
		if ((item == (gpointer) ci) || 
		    (g_object_get_data (G_OBJECT (item), "cond") == cond)) {
			if (GOO_IS_CANVAS_TEXT (item)) 
				g_object_set (G_OBJECT (item),
					      "stroke-color", "red", NULL);
			else 
				g_object_set (G_OBJECT (item),
					      "stroke-color", "red", 
					      "line-width", 4., NULL);
		}
	}

	return FALSE;
}

static gboolean
single_item_leave_notify_event_cb (GooCanvasItem *ci, GooCanvasItem *target_item,
                                   GdkEventCrossing *event, GnomeDbGooJoin *cc)
{
	GSList *list;
	GdaQueryCondition *cond = g_object_get_data (G_OBJECT (ci), "cond");

	for (list = cc->priv->shapes; list; list = list->next) {
		GooCanvasItem *item = GNOME_DB_GOO_CANVAS_SHAPE (list->data)->item;
		if ((item == (gpointer) ci) || 
		    (g_object_get_data (G_OBJECT (item), "cond") == cond)) {
			if (GOO_IS_CANVAS_TEXT (item)) 
				g_object_set (G_OBJECT (item),
					      "stroke-color", "black", NULL);
			else 
				g_object_set (G_OBJECT (item),
					      "stroke-color", "black", 
					      "line-width", 2., NULL);
		}
	}

	return FALSE;
}

static gboolean
single_item_button_press_event_cb (GooCanvasItem *ci, GooCanvasItem *target_item,
                                   GdkEventButton *event, GnomeDbGooJoin *cc)
{
	GdaQueryCondition *cond = g_object_get_data (G_OBJECT (ci), "cond");
	GtkWidget *menu, *entry;

	menu = gtk_menu_new ();
	
	entry = gtk_menu_item_new_with_label (_("Remove join condition"));
	gtk_menu_append (GTK_MENU (menu), entry);
	g_signal_connect (G_OBJECT (entry), "activate", G_CALLBACK (popup_delete_cb), cc);
	g_object_set_data (G_OBJECT (entry), "cond", cond);
	gtk_widget_show (entry);
	gtk_widget_set_sensitive (entry, cond ? TRUE : FALSE);
	
	entry = gtk_menu_item_new_with_label (_("Remove complete join"));
	gtk_menu_append (GTK_MENU (menu), entry);
	g_signal_connect (G_OBJECT (entry), "activate", G_CALLBACK (popup_delete_cb), cc);
	gtk_widget_show (entry);
	
	entry = gtk_menu_item_new_with_label (_("Join properties"));
	gtk_menu_append (GTK_MENU (menu), entry);
	g_signal_connect (G_OBJECT (entry), "activate", G_CALLBACK (popup_join_properties_cb), cc);
	gtk_widget_show (entry);
	
	gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
			NULL, NULL, ((GdkEventButton *)event)->button,
			((GdkEventButton *)event)->time);

	return TRUE;
}

static const gchar *join_type_to_char (GdaQueryJoinType type);
static void dlg_join_destroyed_cb (GdaQueryJoin *join, GtkDialog *dlg) ;
static void change_join_type_cb (GtkToggleButton *rb, GdaQueryJoin *join);
static void change_join_cond_cb (GtkButton *button, GdaQueryJoin *join);
static void change_join_cond_fkcons_cb (GtkButton *button, GdaQueryJoin *join);
static void join_properties_type_changed_cb (GdaQueryJoin *join, GtkDialog *dlg);
static void join_properties_condition_changed_cb (GdaQueryJoin *join, GtkDialog *dlg);
static void
popup_join_properties_cb (GtkMenuItem *mitem, GnomeDbGooJoin *cc)
{
	GtkWidget *top = NULL; /*gtk_widget_get_toplevel ();*/
	GtkWidget *dlg, *vbox, *label, *hbox, *table, *bbox;
	GtkWidget *rb, *rb2, *entry, *button;
	gchar *str;
	
	dlg = gtk_dialog_new_with_buttons (_("Join properties"), (GtkWindow *) top /*GTK_WINDOW (top)*/, 0,
					   GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
					   NULL);
	vbox = GTK_DIALOG (dlg)->vbox;
	gtk_box_set_spacing (GTK_BOX (vbox), 6);

	/*
	 * join type
	 */
	str = g_strdup_printf ("<b>%s</b>", _("Type of join:"));
	label = gtk_label_new ("");
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, TRUE, 6);
	gtk_widget_show (label);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	table = gtk_table_new (5, 2, FALSE);
	gtk_table_set_col_spacings (GTK_TABLE (table), 6);
	gtk_box_pack_start (GTK_BOX (hbox), table, TRUE, TRUE, 0);
	gtk_widget_show (table);

	rb = gtk_radio_button_new_with_label (NULL, _("Inner join"));
	gtk_table_attach_defaults (GTK_TABLE (table), rb, 0, 1, 0, 1);
	gtk_widget_show (rb);
	g_object_set_data (G_OBJECT (dlg), join_type_to_char (GDA_QUERY_JOIN_TYPE_INNER), rb);
	g_object_set_data (G_OBJECT (rb), "type", GINT_TO_POINTER (GDA_QUERY_JOIN_TYPE_INNER));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (change_join_type_cb), cc->priv->join);

	rb2 = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (rb), _("Left outer join"));
	gtk_table_attach_defaults (GTK_TABLE (table), rb2, 0, 1, 1, 2);
	gtk_widget_show (rb2);
	g_object_set_data (G_OBJECT (dlg), join_type_to_char (GDA_QUERY_JOIN_TYPE_LEFT_OUTER), rb2);
	g_object_set_data (G_OBJECT (rb2), "type", GINT_TO_POINTER (GDA_QUERY_JOIN_TYPE_LEFT_OUTER));
	g_signal_connect (G_OBJECT (rb2), "toggled", G_CALLBACK (change_join_type_cb), cc->priv->join);

	rb2 = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (rb), _("Right outer join"));
	gtk_table_attach_defaults (GTK_TABLE (table), rb2, 0, 1, 2, 3);
	gtk_widget_show (rb2);
	g_object_set_data (G_OBJECT (dlg), join_type_to_char (GDA_QUERY_JOIN_TYPE_RIGHT_OUTER), rb2);
	g_object_set_data (G_OBJECT (rb2), "type", GINT_TO_POINTER (GDA_QUERY_JOIN_TYPE_RIGHT_OUTER));
	g_signal_connect (G_OBJECT (rb2), "toggled", G_CALLBACK (change_join_type_cb), cc->priv->join);

	rb2 = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (rb), _("Full outer join"));
	gtk_table_attach_defaults (GTK_TABLE (table), rb2, 0, 1, 3, 4);
	gtk_widget_show (rb2);
	g_object_set_data (G_OBJECT (dlg), join_type_to_char (GDA_QUERY_JOIN_TYPE_FULL_OUTER), rb2);
	g_object_set_data (G_OBJECT (rb2), "type", GINT_TO_POINTER (GDA_QUERY_JOIN_TYPE_FULL_OUTER));
	g_signal_connect (G_OBJECT (rb2), "toggled", G_CALLBACK (change_join_type_cb), cc->priv->join);

	rb2 = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (rb), _("Cross join"));
	gtk_table_attach_defaults (GTK_TABLE (table), rb2, 0, 1, 4, 5);
	gtk_widget_show (rb2);
	g_object_set_data (G_OBJECT (dlg), join_type_to_char (GDA_QUERY_JOIN_TYPE_CROSS), rb2);
	g_object_set_data (G_OBJECT (rb2), "type", GINT_TO_POINTER (GDA_QUERY_JOIN_TYPE_CROSS));
	g_signal_connect (G_OBJECT (rb2), "toggled", G_CALLBACK (change_join_type_cb), cc->priv->join);

	label = gtk_label_new ("");
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label), 0., 0.);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 0, 5);
	
	gtk_widget_show (label);
	g_object_set_data (G_OBJECT (dlg), "join_expl", label);

	/*
	 * join condition
	 */
	str = g_strdup_printf ("<b>%s</b>", _("Associated join condition:"));
	label = gtk_label_new ("");
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, TRUE, 6);
	gtk_widget_show (label);
	g_object_set_data (G_OBJECT (dlg), "join_cond_label", label);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	g_object_set_data (G_OBJECT (dlg), "join_cond_box", hbox);

	entry = gtk_text_view_new ();
	gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
	gtk_widget_show (entry);
	g_object_set_data (G_OBJECT (dlg), "join_cond_entry", entry);

	bbox = gtk_vbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_SPREAD);
	gtk_box_pack_start (GTK_BOX (hbox), bbox, FALSE, TRUE, 0);
	gtk_widget_show (bbox);

	/* "apply" button*/
	button = gtk_button_new_from_stock (GTK_STOCK_APPLY);
	gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 0);
	gtk_widget_show (button);
	g_object_set_data (G_OBJECT (dlg), "join_cond_apply", button);
	g_object_set_data (G_OBJECT (button), "join_cond_entry", entry);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (change_join_cond_cb), cc->priv->join);

	/* "propose a condition" button */
	button = gtk_button_new_with_label (_("Get proposal"));
	gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 0);
	gtk_widget_show (button);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (change_join_cond_fkcons_cb), cc->priv->join);
		
	/*
	 * Final init
	 */
	join_properties_type_changed_cb (cc->priv->join, GTK_DIALOG (dlg));
	join_properties_condition_changed_cb (cc->priv->join, GTK_DIALOG (dlg));

	gda_object_connect_destroy (cc->priv->join, G_CALLBACK (dlg_join_destroyed_cb), dlg);
	g_signal_connect (G_OBJECT (cc->priv->join), "type_changed",
			  G_CALLBACK (join_properties_type_changed_cb), dlg);
	g_signal_connect (G_OBJECT (cc->priv->join), "condition_changed",
			  G_CALLBACK (join_properties_condition_changed_cb), dlg);

	gtk_widget_show (dlg);
	gtk_dialog_run (GTK_DIALOG (dlg));
	gtk_widget_destroy (dlg);

	g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->join),
					      G_CALLBACK (dlg_join_destroyed_cb), dlg);
	g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->join),
					      G_CALLBACK (join_properties_type_changed_cb), dlg);
	g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->join),
					      G_CALLBACK (join_properties_condition_changed_cb), dlg);
}

static void
dlg_join_destroyed_cb (GdaQueryJoin *join, GtkDialog *dlg) 
{
	gtk_dialog_response (dlg, 0);
}

static void
change_join_type_cb (GtkToggleButton *rb, GdaQueryJoin *join)
{
	/* only modify the model (join type), no widget modification */
	GdaQueryJoinType type = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (rb), "type"));

	if (gtk_toggle_button_get_active (rb))
		gda_query_join_set_join_type (join, type);
}

static void
change_join_cond_fkcons_cb (GtkButton *button, GdaQueryJoin *join)
{
	if (!gda_query_join_set_condition_from_fkcons (join)) {
		GtkWidget *dlg;
		gchar *msg = g_strdup_printf ("<big>%s</big>\n\n%s",
					      _("Could not propose a join condition:"),
					      _("The database does not contain any foreign key constraint "
						"which could be used to propose a suitable join condition.\n\n"
						"The database model may need to be updated"));
		dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_INFO,
							  GTK_BUTTONS_CLOSE, msg);
		g_free (msg);
		gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
	}

#ifdef debug
	gda_object_dump (GDA_OBJECT (gda_query_join_get_query (join)), 0);
#endif
}

static void
change_join_cond_cb (GtkButton *button, GdaQueryJoin *join)
{
	/* only modify the model (join condition), no widget modification */
	GtkWidget *wid;
	GtkTextBuffer *buffer;
	gchar *str;
	GError *error = NULL;
	GtkTextIter start, end;

	wid = g_object_get_data (G_OBJECT (button), "join_cond_entry");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wid));
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	str = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
	if (str && *str) {
		/* set join condition from SQL */
		if (!gda_query_join_set_condition_from_sql (join, str, &error)) {
			GtkWidget *dlg;
			gchar *msg = g_strdup_printf ("<big>%s</big>\n\n%s",
						      _("Could not set the new join condition:"),
						      error->message);
			dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR,
								  GTK_BUTTONS_CLOSE, msg);
			g_free (msg);
			gtk_dialog_run (GTK_DIALOG (dlg));
			gtk_widget_destroy (dlg);
		}
	}
	else 
		/* propose a join condition */
		change_join_cond_fkcons_cb (NULL, join);

	g_free (str);

#ifdef debug
	gda_object_dump (GDA_OBJECT (gda_query_join_get_query (join)), 0);
#endif
}

static void
join_properties_type_changed_cb (GdaQueryJoin *join, GtkDialog *dlg)
{
	gchar *str = NULL;
	gchar *t1name, *t2name;
	GtkWidget *label, *wid;
	gboolean sensitive;
	GdaQueryJoinType type = gda_query_join_get_join_type (join);

	/* selected radio button */
	wid = g_object_get_data (G_OBJECT (dlg), join_type_to_char (type));
	g_signal_handlers_block_by_func (G_OBJECT (wid), G_CALLBACK (change_join_type_cb), join);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wid), TRUE);
	g_signal_handlers_unblock_by_func (G_OBJECT (wid), G_CALLBACK (change_join_type_cb), join);

	/* explanation text */
	t1name = gda_query_target_get_complete_name (gda_query_join_get_target_1 (join));
	t2name = gda_query_target_get_complete_name (gda_query_join_get_target_2 (join));

	switch (type) {
	case GDA_QUERY_JOIN_TYPE_INNER:
		str = g_strdup_printf (_("Select only the records of the two targets "
                                         "('%s' and '%s') where the fields respect the "
					 "join condition."),
                                       t1name, t2name);
		break;
	case GDA_QUERY_JOIN_TYPE_LEFT_OUTER:
		str = g_strdup_printf (_("Select all the records of '%s' "
                                         "and those of '%s' where the fields "
                                         "respect the join condition."), t1name, t2name);
		break;
	case GDA_QUERY_JOIN_TYPE_RIGHT_OUTER:
		str = g_strdup_printf (_("Select all the records of '%s' "
                                         "and those of '%s' where the fields "
                                         "respect the join condition."), t2name, t1name);
		break;
	case GDA_QUERY_JOIN_TYPE_FULL_OUTER:
		str = g_strdup_printf (_("Select all the records of '%s' "
                                         "and those of '%s' linking the two "
                                         "where the fields respect the join condition."), 
				       t1name, t2name);
		break;
	case GDA_QUERY_JOIN_TYPE_CROSS:
		str = g_strdup_printf (_("Select all the combinations of the records of '%s' "
                                         "and of '%s' without any attempt at linking the two "
                                         "(no join condition applies)."), 
				       t1name, t2name);
		break;
	default:
		g_assert_not_reached ();
		break;
	}

	g_free (t1name);
	g_free (t2name);

	label = g_object_get_data (G_OBJECT (dlg), "join_expl");
	gtk_label_set_text (GTK_LABEL (label), str);
	g_free (str);

	sensitive = type != GDA_QUERY_JOIN_TYPE_CROSS ? TRUE : FALSE;
	wid = g_object_get_data (G_OBJECT (dlg), "join_cond_label");
	gtk_widget_set_sensitive (wid, sensitive);
	wid = g_object_get_data (G_OBJECT (dlg), "join_cond_box");
	gtk_widget_set_sensitive (wid, sensitive);
}

static void
join_properties_condition_changed_cb (GdaQueryJoin *join, GtkDialog *dlg)
{
	GtkWidget *wid;
	GtkTextBuffer *buffer;
	GError *error = NULL;
	GdaQueryCondition *cond = gda_query_join_get_condition (join);
	wid = g_object_get_data (G_OBJECT (dlg), "join_cond_entry");

	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wid));
	if (cond) {
		gchar *str;

		str = gda_renderer_render_as_sql (GDA_RENDERER (cond), NULL, NULL,
						  GDA_RENDERER_EXTRA_PRETTY_SQL | GDA_RENDERER_PARAMS_AS_DETAILED, &error);
		if (str) {
			gtk_text_buffer_set_text (buffer, str, -1);
			g_free (str);
		}
		else
			gtk_text_buffer_set_text (buffer, error->message ? error->message : _("Error"), -1);
	}
	else {
		gtk_text_buffer_set_text (buffer, _("No defined join condition"), -1);
	}
}


static const gchar *
join_type_to_char (GdaQueryJoinType type)
{
	switch (type) {
	case GDA_QUERY_JOIN_TYPE_INNER:
		return "join_I";
	case GDA_QUERY_JOIN_TYPE_LEFT_OUTER:
		return "join_L";
	case GDA_QUERY_JOIN_TYPE_RIGHT_OUTER:
		return "join_R";
	case GDA_QUERY_JOIN_TYPE_FULL_OUTER:
		return "join_F";
	case GDA_QUERY_JOIN_TYPE_CROSS:
		return "join_C";
	default:
		g_assert_not_reached ();
		break;	
	}

	return NULL;
}

static void
popup_delete_cb (GtkMenuItem *mitem, GnomeDbGooJoin *cc)
{
	GdaQueryCondition *cond;
#ifdef debug
	GdaQuery *query;

	query = gda_query_join_get_query (cc->priv->join);
#endif
	cond = g_object_get_data (G_OBJECT (mitem), "cond");
	if (cond)
		/* remove the condition only */
		gda_object_destroy (GDA_OBJECT (cond));
	else 
		/* remove the whole join */
		gda_object_destroy (GDA_OBJECT (cc->priv->join));
#ifdef debug
	gda_object_dump (GDA_OBJECT (query), 0);
#endif
}

static void
entity_item_moved_cb (GooCanvasItem *entity, GnomeDbGooJoin *cc)
{
	update_items (cc);
}



static void
gnome_db_goo_join_set_join (GnomeDbGooJoin *cc, GdaQueryJoin *join)
{
	g_return_if_fail (cc && GNOME_DB_IS_GOO_JOIN (cc));
	g_return_if_fail (cc->priv);
	if (join)
		g_return_if_fail (GDA_IS_QUERY_JOIN (join ));
	
	if (cc->priv->join == join)
		return;
	
	if (cc->priv->join) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (join),
						      G_CALLBACK (join_destroyed_cb), cc);
		g_signal_handlers_disconnect_by_func (G_OBJECT (join),
						      G_CALLBACK (join_changed_cb), cc);
		cc->priv->join = NULL;
	}

	if (join) {
		cc->priv->join = join;
		gda_object_connect_destroy (join, G_CALLBACK (join_destroyed_cb), cc);
		g_signal_connect (G_OBJECT (join), "condition_changed",
				  G_CALLBACK (join_changed_cb), cc);
		g_signal_connect (G_OBJECT (join), "type_changed",
				  G_CALLBACK (join_changed_cb), cc);
	}

	clean_items (cc);
	create_items (cc);
}

/**
 * gnome_db_goo_join_new
 * @parent: the parent item, or NULL. 
 * @join: the #GdaQueryJoin to represent
 * @...: optional pairs of property names and values, and a terminating NULL.
 *
 * Creates a new canvas item to represent the @join join
 *
 * Returns: a new #GooCanvasItem object
 */
GooCanvasItem *
gnome_db_goo_join_new (GooCanvasItem *parent, GdaQueryJoin *join, ...)
{
	GooCanvasItem *item;
        const char *first_property;
        va_list var_args;

        item = g_object_new (GNOME_DB_TYPE_GOO_JOIN, NULL);

        if (parent) {
                goo_canvas_item_add_child (parent, item, -1);
                g_object_unref (item);
        }

        g_object_set (item, "join", join, NULL);

        va_start (var_args, join);
        first_property = va_arg (var_args, char*);
        if (first_property)
                g_object_set_valist ((GObject*) item, first_property, var_args);
        va_end (var_args);

        return item;
}
