/* GNOME DB library
 * Copyright (C) 1999 - 2006 The GNOME Foundation.
 *
 * AUTHORS:
 * 	Rodrigo Moya <rodrigo@gnome-db.org>
 *      Vivien Malerba <malerba@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n-lib.h>
#ifdef HAVE_LIBGLADE
#include <glade/glade-init.h>
#endif
#include <libgda/libgda.h>
#ifdef HAVE_GCONF
#include <gconf/gconf-client.h>
#endif
#include <libgda/graph/gda-dict-reg-graphs.h>
#include <libgnomedb/binreloc/gnome-db-binreloc.h>

/* plugins list */
GHashTable *gnome_db_plugins_hash = NULL; /* key = plugin name, value = GnomeDbPlugin structure pointer */

extern void gnome_db_stock_init (void);

/**
 * gnome_db_init
 * @app_id:
 * @version:
 * @nargs:
 * @args:
 *
 * Initialization of the libgnomedb library
 */
void
gnome_db_init (const gchar *app_id, const gchar *version, gint nargs, gchar *args[])
{
	static gboolean initialized = FALSE;
	gchar *str;

	if (initialized) {
		gda_log_error (_("Attempt to initialize an already initialized client"));
		return;
	}

	gnome_db_gbr_init ();
	str = gnome_db_gbr_get_locale_dir_path ();
	bindtextdomain (GETTEXT_PACKAGE, str);
	g_free (str);

	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");

	gda_init (app_id, version, nargs, args);
	gtk_init (&nargs, &args);

#ifdef HAVE_LIBGLADE
	glade_init ();
#endif
	gnome_db_stock_init ();

#ifdef HAVE_GCONF
	/* init GConf */
	if (!gconf_is_initialized ())
		gconf_init (0, NULL, NULL);
#endif

	/* make sure all the GdaDict object register GdaGraph */
	gda_dict_class_always_register (gda_graphs_get_register);

	initialized = TRUE;
}

typedef struct {
	GdaInitFunc init_func;
	gpointer user_data;
} InitCbData;

static gboolean
idle_cb (gpointer user_data)
{
	InitCbData *cb_data = (InitCbData *) user_data;

	g_return_val_if_fail (cb_data != NULL, FALSE);

	if (cb_data->init_func)
		cb_data->init_func (cb_data->user_data);

	g_free (cb_data);

	return FALSE;
}

/**
 * gnome_db_main_run
 * @init_func: function to be called when everything has been initialized
 * @user_data: data to be passed to the init function
 *
 * This function is a convenience function to specify a function to be called 
 * after everything has been correctly
 * initialized (that is, for initializing your own stuff), in an idle callback.
 *
 * Using it is not mandatory, and usually gtk_main() can be used.
 */
void
gnome_db_main_run (GdaInitFunc init_func, gpointer user_data)
{
	if (init_func) {
		InitCbData *cb_data;

		cb_data = g_new (InitCbData, 1);
		cb_data->init_func = init_func;
		cb_data->user_data = user_data;

		g_idle_add ((GSourceFunc) idle_cb, cb_data);
	}

	gtk_main ();
}

/**
 * gnome_db_main_quit
 *
 * Convenience function, simply calls gtk_main_quit()
 */
void
gnome_db_main_quit (void)
{
	gtk_main_quit ();
}

/**
 * gnome_db_get_application_exec_path
 * @app_name: the name of the application to find
 *
 * Find the path to the application identified by @app_name.
 *
 * Returns: the path as a new string, or %NULL if the application cannot be found
 */
gchar *
gnome_db_get_application_exec_path (const gchar *app_name)
{
	gchar *str;
	gchar *fname;
	g_return_val_if_fail (app_name, NULL);

	gnome_db_gbr_init ();
	fname = g_strdup_printf ("%s-%s", app_name, ABI_VERSION);
	str = gnome_db_gbr_get_file_path (GNOME_DB_BIN_DIR, fname, NULL);
	g_free (fname);
	if (!g_file_test (str, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_EXECUTABLE)) {
		g_free (str);
		str = NULL;
	}

	return str;
}
