/* gnome-db-goo-db-relations.c
 *
 * Copyright (C) 2002 - 2007 Vivien Malerba
 * Copyright (C) 2002 Fernando Martins
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gtk/gtk.h>
#include <libgnomedb/marshal.h>
#include <libgda/libgda.h>
#include <glib/gi18n-lib.h>
#include "gnome-db-goo-db-relations.h"
#include "gnome-db-goo-entity.h"
#include "gnome-db-goo-field.h"
#include "gnome-db-goo-fkconstraint.h"

static void gnome_db_goo_db_relations_class_init (GnomeDbGooDbRelationsClass * class);
static void gnome_db_goo_db_relations_init       (GnomeDbGooDbRelations * canvas);
static void gnome_db_goo_db_relations_dispose   (GObject   * object);

/* virtual functions */
static void       create_canvas_items (GnomeDbGoo *canvas);
static void       clean_canvas_items  (GnomeDbGoo *canvas);
static void       graph_item_added    (GnomeDbGoo *canvas, GdaGraphItem *item);
static void       graph_item_dropped  (GnomeDbGoo *canvas, GdaGraphItem *item);
static GtkWidget *build_context_menu  (GnomeDbGoo *canvas);

static void       db_destroyed_cb     (GdaDictDatabase *db, GnomeDbGoo *canvas);
static void       db_constraint_added_cb (GdaDictDatabase *db, GdaDictConstraint *cstr, GnomeDbGoo *canvas);

static void       drag_action_dcb     (GnomeDbGoo *canvas, GnomeDbGooItem *from_item, GnomeDbGooItem *to_item);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

/* HASH functions for the key of type FKTables */
typedef struct {
	GdaDictTable *fk_table;
	GdaDictTable *ref_pk_table;
} FKTables;

static guint
g_fktables_hash (gconstpointer key)
{
	FKTables *thekey = (FKTables *) key;
	gint h;

	h = (gint) (thekey->fk_table);
	h = (h << 5) - h + (gint) (thekey->ref_pk_table);
	
	/*g_print ("HASH: %p,%p -> %ld\n", thekey->fk_table, thekey->ref_pk_table, h);*/
	return h;
}

static gboolean
g_fktables_equal (gconstpointer a, gconstpointer b)
{
	FKTables *akey = (FKTables *) a;
	FKTables *bkey = (FKTables *) b;

	if (!a && !b)
		return TRUE;

	if (!a || !b)
		return FALSE;

	if ((akey->fk_table == bkey->fk_table) &&
	    (akey->ref_pk_table == bkey->ref_pk_table))
		return TRUE;
	else
		return FALSE;
}


struct _GnomeDbGooDbRelationsPrivate
{
	GHashTable       *hash_tables; /* key = GdaDictTable, value = GnomeDbGooEntity */
	GHashTable       *hash_constraints; /* key = FKTables, value = GnomeDbGooFkconstrains */

	GdaDictDatabase  *db;
};

GType
gnome_db_goo_db_relations_get_type (void)
{
	static GType type = 0;

	if (G_UNLIKELY (type == 0)) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGooDbRelationsClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_goo_db_relations_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGooDbRelations),
			0,
			(GInstanceInitFunc) gnome_db_goo_db_relations_init
		};		

		type = g_type_register_static (GNOME_DB_TYPE_GOO, "GnomeDbGooDbRelations", &info, 0);
	}
	return type;
}

static void canvas_fkconstraint_destroy_cb (GnomeDbGooItem *canvas_item, GnomeDbGooDbRelations *canvas);
static void
fk_constraint_hash_remove_func (GooCanvasItem *item)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (item),
					      G_CALLBACK (canvas_fkconstraint_destroy_cb), 
					      goo_canvas_item_get_canvas (item));
}

static void
gnome_db_goo_db_relations_init (GnomeDbGooDbRelations * canvas)
{
	canvas->priv = g_new0 (GnomeDbGooDbRelationsPrivate, 1);
	canvas->priv->hash_tables = g_hash_table_new (NULL, NULL);
	canvas->priv->hash_constraints = g_hash_table_new_full (g_fktables_hash, g_fktables_equal, 
								g_free, (GDestroyNotify) fk_constraint_hash_remove_func);
	canvas->priv->db = NULL;
}

static void
gnome_db_goo_db_relations_class_init (GnomeDbGooDbRelationsClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	parent_class = g_type_class_peek_parent (class);

	/* GnomeDbGoo virtual functions */
	GNOME_DB_GOO_CLASS (class)->create_canvas_items = create_canvas_items;
	GNOME_DB_GOO_CLASS (class)->clean_canvas_items = clean_canvas_items;
	GNOME_DB_GOO_CLASS (class)->graph_item_added = graph_item_added;
	GNOME_DB_GOO_CLASS (class)->graph_item_dropped = graph_item_dropped;
	GNOME_DB_GOO_CLASS (class)->build_context_menu = build_context_menu;

	GNOME_DB_GOO_CLASS (class)->drag_action = drag_action_dcb;
	
	object_class->dispose = gnome_db_goo_db_relations_dispose;
}

static void
gnome_db_goo_db_relations_dispose (GObject *object)
{
	GnomeDbGooDbRelations *canvas;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_GOO_DB_RELATIONS (object));

	canvas = GNOME_DB_GOO_DB_RELATIONS (object);

	if (canvas->priv) {
		clean_canvas_items (GNOME_DB_GOO (canvas));
		if (canvas->priv->db) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (canvas->priv->db),
							      G_CALLBACK (db_destroyed_cb), canvas);
			g_signal_handlers_disconnect_by_func (G_OBJECT (canvas->priv->db),
							      G_CALLBACK (db_constraint_added_cb), canvas);
		}

		g_hash_table_destroy (canvas->priv->hash_tables);
		g_hash_table_destroy (canvas->priv->hash_constraints);

		g_free (canvas->priv);
		canvas->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static GtkWidget *
drag_action_question_dlg (GnomeDbGoo *canvas, GSList *cstr_list)
{
	GtkWidget *dlg, *label, *hbox, *box, *button, *bgroup;
	GSList *list = cstr_list;

	dlg = gtk_dialog_new_with_buttons (_("Constraint information request"), NULL,
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					   GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
					   NULL);
	box = GTK_DIALOG (dlg)->vbox;
	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<big><b>New tables relations:</b></big>\n"
				 "You have selected a pair of table fields to be related to each other "
				 "and must decide to either create a new constraint or modify an "
				 "already existing constraint."));
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	
	gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 5);
	gtk_widget_show (label);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	box = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), box, TRUE, TRUE, 0);
	gtk_widget_show (box);

	button = gtk_radio_button_new_with_label (NULL,_("Create a new constraint"));
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, TRUE, 0);
	gtk_widget_show (button);
	g_object_set_data (G_OBJECT (dlg), "rb", button);
	bgroup = button;

	for (list = cstr_list; list; list = list->next) {
		GtkWidget *table;
		GSList *pairs, *plist;
		gint nb_pairs, i;

		pairs = gda_dict_constraint_fkey_get_fields (GDA_DICT_CONSTRAINT (list->data));
		nb_pairs = g_slist_length (pairs);

		button = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (bgroup), 
								      _("Add to constraint:"));
		gtk_box_pack_start (GTK_BOX (box), button, FALSE, TRUE, 5);
		gtk_widget_show (button);
		g_object_set_data (G_OBJECT (button), "constraint", list->data);

		hbox = gtk_hbox_new (FALSE, 0); /* HIG */
                gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, FALSE, 0);
                gtk_widget_show (hbox);
                label = gtk_label_new ("      ");
                gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
                gtk_widget_show (label);

		table = gtk_table_new (nb_pairs, 3, FALSE);
		gtk_box_pack_start (GTK_BOX (hbox), table, FALSE, FALSE, 0);
		gtk_widget_show (table);
		plist = pairs;
		i = 0;
		while (plist) {
			GdaDictConstraintFkeyPair *pair = GDA_DICT_CONSTRAINT_FK_PAIR (pairs->data);
			gchar *str;

			str = g_strdup_printf ("%s.%s", 
					       gda_object_get_name (GDA_OBJECT (gda_entity_field_get_entity (GDA_ENTITY_FIELD (pair->fkey)))),
					       gda_object_get_name (GDA_OBJECT (pair->fkey)));
			label = gtk_label_new (str);
			g_free (str);
			gtk_table_attach (GTK_TABLE (table), label, 0, 1, i, i+1, 0, 0, 0, 0);
			gtk_widget_show (label);

			label = gtk_label_new ("--");
			gtk_table_attach (GTK_TABLE (table), label, 1, 2, i, i+1, 0, 0, 5, 0);
			gtk_widget_show (label);
			
			str = g_strdup_printf ("%s.%s", 
					       gda_object_get_name (GDA_OBJECT (gda_entity_field_get_entity (GDA_ENTITY_FIELD (pair->ref_pkey)))),
					       gda_object_get_name (GDA_OBJECT (pair->ref_pkey)));
			label = gtk_label_new (str);
			g_free (str);
			gtk_table_attach (GTK_TABLE (table), label, 2, 3, i, i+1, 0, 0, 0, 0);
			gtk_widget_show (label);

			g_free (plist->data);
			plist = g_slist_next (plist);
			i++;
		}
		g_slist_free (pairs);
	}

	return dlg;
}

static void
drag_action_dcb (GnomeDbGoo *canvas, GnomeDbGooItem *from_item, GnomeDbGooItem *to_item)
{
	GdaEntityField *fk_field = NULL, *ref_pk_field = NULL;
	GdaDictTable *fk_table, *ref_pk_table;
	GSList *cstr_list;
	gboolean create_constraint = FALSE;

	if (GNOME_DB_IS_GOO_FIELD (from_item))
		fk_field = gnome_db_goo_field_get_field (GNOME_DB_GOO_FIELD (from_item));
	if (GNOME_DB_IS_GOO_FIELD (to_item))
		ref_pk_field = gnome_db_goo_field_get_field (GNOME_DB_GOO_FIELD (to_item));

	/* dismiss if we don't have two fields */
	if (!fk_field || !GDA_IS_DICT_FIELD (fk_field) || 
	    !ref_pk_field || !GDA_IS_DICT_FIELD (ref_pk_field))
		return;

	fk_table = GDA_DICT_TABLE (gda_entity_field_get_entity (fk_field));
	ref_pk_table = GDA_DICT_TABLE (gda_entity_field_get_entity (ref_pk_field));

	cstr_list = gda_dict_database_get_tables_fk_constraints (GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->db,
							   fk_table, ref_pk_table, FALSE);
	if (cstr_list) {
		/* already some existing constraints */
		GSList *clist;
		GdaDictConstraint *found = NULL;

		/* see if a constraint already has that pair of fields */
		clist = cstr_list;
		while (clist && !found) {
			GSList *pairs, *plist;

			pairs = gda_dict_constraint_fkey_get_fields (GDA_DICT_CONSTRAINT (clist->data));
			plist = pairs;
			while (plist) {
				if (!found) {
					GdaDictConstraintFkeyPair *pair = GDA_DICT_CONSTRAINT_FK_PAIR (pairs->data);
					if ((((GdaEntityField*) pair->fkey == fk_field) && ((GdaEntityField*) pair->ref_pkey == ref_pk_field)) ||
					    (((GdaEntityField*) pair->fkey == ref_pk_field) && ((GdaEntityField*) pair->ref_pkey == fk_field)))
						found = GDA_DICT_CONSTRAINT (clist->data);
				}
				g_free (plist->data);
				plist = g_slist_next (plist);
			}
			g_slist_free (pairs);

			clist = g_slist_next (clist);
		}
		
		if (found) {
			/* simply display a message */
			GtkWidget *dlg;

			if (gda_dict_constraint_get_table (found) == fk_table)
				dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_INFO,
									  GTK_BUTTONS_CLOSE,
									  _("<big>A constraint already exists</big>\n"
									    "for the selected couple of fields."));
			else
				dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_INFO,
									  GTK_BUTTONS_CLOSE,
									  _("<big>A constraint already exists</big>\n"
									    "for the selected couple of fields, but "
									    "not in the same order of tables, please "
									    "remove that other constraint before adding "
									    "a new one."));
			gtk_dialog_run (GTK_DIALOG (dlg));
			gtk_widget_destroy (dlg);
		}
		else {
			/* further information required */
			GSList *clist = NULL, *list;
			GtkWidget *dlg = NULL;
			gint response;

			/* keep only the constraints where the table is the FK_table */
			list = cstr_list;
			while (list) {
				if (gda_dict_constraint_get_table (GDA_DICT_CONSTRAINT (list->data)) == fk_table)
					clist = g_slist_append (clist, list->data);
				list = g_slist_next (list);
			}
			g_slist_free (cstr_list);
			cstr_list = clist;

			dlg = drag_action_question_dlg (canvas, cstr_list);
			if (!cstr_list)
				response == GTK_RESPONSE_ACCEPT;
			else
				response = gtk_dialog_run (GTK_DIALOG (dlg));
			if (response == GTK_RESPONSE_ACCEPT) {
				GtkWidget *rb = g_object_get_data (G_OBJECT (dlg), "rb");
				GSList *group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
				GdaDictConstraint *add_to = NULL;

				while (group && !add_to) {
					if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (group->data))) {
						add_to = g_object_get_data (G_OBJECT (group->data), "constraint");
					}
					group = g_slist_next (group);
				}
				
				if (add_to) {
					GSList *nlist;
					GdaDictConstraintFkeyPair *pair;

					nlist = gda_dict_constraint_fkey_get_fields (add_to);
					pair = g_new0 (GdaDictConstraintFkeyPair, 1);
					pair->fkey = GDA_DICT_FIELD (fk_field);
					pair->ref_pkey = GDA_DICT_FIELD (ref_pk_field);
					pair->ref_pkey_repl = NULL;
					nlist = g_slist_append (nlist, pair);
					gda_dict_constraint_fkey_set_fields (add_to, nlist);
					
					/* memory libreation */
					list = nlist;
					while (list) {
						g_free (list->data);
						list = g_slist_next (list);
					}
					g_slist_free (nlist);
				}
				else 
					create_constraint = TRUE;
			}
			gtk_widget_destroy (dlg);
		}
		
		g_slist_free (cstr_list);
	}
	else
		/* no existing constraint yet: create one */
		create_constraint = TRUE;


	/* actually create the constraint */
	if (create_constraint) {
		GSList *nlist, *list;
		GdaDictConstraint *cstr;
		GdaDictConstraintFkeyPair *pair;

		cstr = GDA_DICT_CONSTRAINT (gda_dict_constraint_new (fk_table, CONSTRAINT_FOREIGN_KEY));
		g_object_set (G_OBJECT (cstr), "user_constraint", TRUE, NULL);
		
		/* add FK pair */
		pair = g_new0 (GdaDictConstraintFkeyPair, 1);
		pair->fkey = GDA_DICT_FIELD (fk_field);
		pair->ref_pkey = GDA_DICT_FIELD (ref_pk_field);
		pair->ref_pkey_repl = NULL;

		nlist = g_slist_append (NULL, pair);
		gda_dict_constraint_fkey_set_fields (cstr, nlist);
		
		/* memory libreation */
		list = nlist;
		while (list) {
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (nlist);

		gda_dict_database_add_constraint (GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->db, cstr);
		g_object_unref (G_OBJECT (cstr));
	}
}

/**
 * gnome_db_goo_db_relations_new
 * @dict: a #GdaDict object
 *
 * Creates a new canvas widget to display the relations between the database's tables.
 * The database is the one managed by the #GdaDict object which @graph refers.
 *
 * After the #GnomeDbGooDbRelations has been created, it is possible to display
 * the tables as listed in a #GdaGdaph by setting the "graph" property (using g_object_set()).
 *
 * Returns: a new #GnomeDbGooDbRelations widget
 */
GtkWidget *
gnome_db_goo_db_relations_new (GdaDict *dict)
{
	GObject *obj;
	GdaDictDatabase *db;
	GnomeDbGooDbRelations *canvas;

	g_return_val_if_fail (!dict || GDA_IS_DICT (dict), NULL);
        obj = g_object_new (GNOME_DB_TYPE_GOO_DB_RELATIONS, NULL);
	canvas = GNOME_DB_GOO_DB_RELATIONS (obj);

	/* connecting to the GdaDictDatabase */
	db = gda_dict_get_database (ASSERT_DICT (dict));
	canvas->priv->db = db;
	gda_object_connect_destroy (db, G_CALLBACK (db_destroyed_cb), obj);
	g_signal_connect (G_OBJECT (db), "constraint_added",
			  G_CALLBACK (db_constraint_added_cb), obj);

	return GTK_WIDGET (obj);
}


static void
db_destroyed_cb (GdaDictDatabase *db, GnomeDbGoo *canvas)
{
	/* disconnecting signals */
	GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->db = NULL;
	g_signal_handlers_disconnect_by_func (G_OBJECT (db),
					      G_CALLBACK (db_destroyed_cb), canvas);
	g_signal_handlers_disconnect_by_func (G_OBJECT (db),
					      G_CALLBACK (db_constraint_added_cb), canvas);

	/* clean the canvas */
	clean_canvas_items (canvas);
}

static void
db_constraint_added_cb (GdaDictDatabase *db, GdaDictConstraint *cstr, GnomeDbGoo *canvas)
{
	FKTables key;
	GnomeDbGooItem *canvas_fk;
	GooCanvasItem *canvas_item;

	/* ignore non FK constraints */
	if (gda_dict_constraint_get_constraint_type (cstr) != CONSTRAINT_FOREIGN_KEY)
		return;

	key.fk_table = gda_dict_constraint_get_table (cstr);
	key.ref_pk_table = gda_dict_constraint_fkey_get_ref_table (cstr);
	canvas_fk = g_hash_table_lookup (GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->hash_constraints, &key);
	
	if (canvas_fk) 
		gnome_db_goo_fkconstraint_add_constraint (GNOME_DB_GOO_FKCONSTRAINT (canvas_fk), cstr);
	else {
		if (g_hash_table_lookup (GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->hash_tables, key.fk_table) &&
		    g_hash_table_lookup (GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->hash_tables, key.ref_pk_table)) {
			FKTables *newkey = g_new0 (FKTables, 1);
			
			canvas_item = gnome_db_goo_fkconstraint_new (goo_canvas_get_root_item (GOO_CANVAS (canvas)), 
								     cstr, NULL);
			
			newkey->fk_table = key.fk_table;
			newkey->ref_pk_table = key.ref_pk_table;
			g_hash_table_insert (GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->hash_constraints, 
					     newkey, canvas_item);
			g_object_set_data (G_OBJECT (canvas_item), "key", newkey);
			g_signal_connect (G_OBJECT (canvas_item), "destroy",
					  G_CALLBACK (canvas_fkconstraint_destroy_cb), canvas);
		}
	}
}

static void
canvas_fkconstraint_destroy_cb (GnomeDbGooItem *canvas_item, GnomeDbGooDbRelations *canvas)
{
	FKTables *key = g_object_get_data (G_OBJECT (canvas_item), "key");
	g_assert (key);
	g_hash_table_remove (canvas->priv->hash_constraints, key);
	g_object_set_data (G_OBJECT (canvas_item), "key", NULL);
}

/*
 * Add all the required GooCanvasItem objects for the associated #GdaGraph object 
 */
static void
create_canvas_items (GnomeDbGoo *canvas)
{
	GSList *list, *graph_items;
	GdaGraph *graph = gnome_db_goo_get_graph (canvas);

	if (!graph && ((GnomeDbGooDbRelations*) canvas)->priv->db) {
		graph = (GdaGraph*) 
			gda_graph_new (gda_dict_database_get_dict (((GnomeDbGooDbRelations*) canvas)->priv->db),
				       GDA_GRAPH_DB_RELATIONS);
		g_object_set (G_OBJECT (canvas), "graph", graph, NULL);
		g_object_unref (graph);
	}

	graph_items = gda_graph_get_items (graph);
	list = graph_items;
	while (list) {
		graph_item_added (canvas, GDA_GRAPH_ITEM (list->data));
		list = g_slist_next (list);
	}
	g_slist_free (graph_items);
}

static void
clean_canvas_items (GnomeDbGoo *canvas)
{
	GnomeDbGooDbRelations *dbrel = GNOME_DB_GOO_DB_RELATIONS (canvas);

	/* clean memory */
	g_hash_table_destroy (dbrel->priv->hash_tables);
	g_hash_table_destroy (dbrel->priv->hash_constraints);
	dbrel->priv->hash_tables = g_hash_table_new (NULL, NULL);
	dbrel->priv->hash_constraints = 
		g_hash_table_new_full (g_fktables_hash, g_fktables_equal, 
				       g_free, (GDestroyNotify) fk_constraint_hash_remove_func);
}

static GtkWidget *canvas_entity_popup_func (GnomeDbGooEntity *ce);

/*
 * Add the GnomeDbGooEntity corresponding to the graph item
 */
static void
graph_item_added (GnomeDbGoo *canvas, GdaGraphItem *item)
{
	GooCanvasItem *canvas_item;
	GdaObject *ref_obj = gda_graph_item_get_ref_object (item);

	if (GDA_IS_DICT_TABLE (ref_obj)) {
		GSList *fklist, *list;

		/* GnomeDbGooEntity for the table */
		canvas_item = gnome_db_goo_entity_new (goo_canvas_get_root_item (GOO_CANVAS (canvas)),
						       GDA_ENTITY (ref_obj), 50., 50.,
						       "popup_menu_func", canvas_entity_popup_func,
						       "graph_item", item,
						       NULL);
		g_hash_table_insert (GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->hash_tables, ref_obj, canvas_item);
		gnome_db_goo_declare_item (canvas, GNOME_DB_GOO_ITEM (canvas_item));

		/* GnomeDbGooFkconstraint object(s) for the FK constraints using that table */
		fklist = gda_dict_database_get_tables_fk_constraints (GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->db,
								GDA_DICT_TABLE (ref_obj), NULL, FALSE);
		list = fklist;
		while (list) {
			FKTables key;
			GdaDictConstraint *fkcons = GDA_DICT_CONSTRAINT (list->data);
			GnomeDbGooItem *canvas_fk;

			key.fk_table = gda_dict_constraint_get_table (fkcons);
			key.ref_pk_table = gda_dict_constraint_fkey_get_ref_table (fkcons);
			canvas_fk = g_hash_table_lookup (GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->hash_constraints, &key);
			
			if (canvas_fk)
				gnome_db_goo_fkconstraint_add_constraint (GNOME_DB_GOO_FKCONSTRAINT (canvas_fk), fkcons);
			else {
				if (g_hash_table_lookup (GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->hash_tables, key.fk_table) &&
				    g_hash_table_lookup (GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->hash_tables, key.ref_pk_table)) {
					FKTables *newkey = g_new0 (FKTables, 1);
					
					canvas_item = gnome_db_goo_fkconstraint_new (goo_canvas_get_root_item (GOO_CANVAS (canvas)), 
										     fkcons, NULL);
					
					newkey->fk_table = key.fk_table;
					newkey->ref_pk_table = key.ref_pk_table;
					gnome_db_goo_declare_item (canvas, GNOME_DB_GOO_ITEM (canvas_item));
					g_hash_table_insert (GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->hash_constraints, 
							     newkey, canvas_item);
					g_object_set_data (G_OBJECT (canvas_item), "key", newkey);
					g_signal_connect (G_OBJECT (canvas_item), "destroy",
							  G_CALLBACK (canvas_fkconstraint_destroy_cb), canvas);
				}
			}

			list = g_slist_next (list);
		}
		g_slist_free (fklist);
	}
}

static void popup_func_delete_cb (GtkMenuItem *mitem, GnomeDbGooEntity *ce);
static GtkWidget *
canvas_entity_popup_func (GnomeDbGooEntity *ce)
{
	GtkWidget *menu, *entry;

	menu = gtk_menu_new ();
	entry = gtk_menu_item_new_with_label (_("Remove"));
	g_signal_connect (G_OBJECT (entry), "activate", G_CALLBACK (popup_func_delete_cb), ce);
	gtk_menu_append (GTK_MENU (menu), entry);
	gtk_widget_show (entry);

	return menu;
}

static void
popup_func_delete_cb (GtkMenuItem *mitem, GnomeDbGooEntity *ce)
{
	GdaGraphItem *gitem;

	gitem = gnome_db_goo_item_get_graph_item (GNOME_DB_GOO_ITEM (ce));
	gda_object_destroy (GDA_OBJECT (gitem));
}


/*
 * Remove the GnomeDbGooEntity corresponding to the graph item
 */
static void
graph_item_dropped (GnomeDbGoo *canvas, GdaGraphItem *item)
{
	GdaObject *ref_obj = gda_graph_item_get_ref_object (item);

	if (GDA_IS_DICT_TABLE (ref_obj)) {
		GooCanvasItem *canvas_item = 
			g_hash_table_lookup (GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->hash_tables,
					     ref_obj);
		if (canvas_item) {
			g_hash_table_remove (GNOME_DB_GOO_DB_RELATIONS (canvas)->priv->hash_tables, ref_obj);
			goo_canvas_item_remove (canvas_item);
		}
	}
}

static void popup_add_table_cb (GtkMenuItem *mitem, GnomeDbGooDbRelations *canvas);
static GtkWidget *
build_context_menu (GnomeDbGoo *canvas)
{
	GtkWidget *menu, *mitem, *submenu, *submitem;
	GSList *tables, *list;
	GnomeDbGooDbRelations *dbrels = GNOME_DB_GOO_DB_RELATIONS (canvas);
	GdaObjectRef *refbase;
	GdaDict *dict = gda_object_get_dict (GDA_OBJECT (dbrels->priv->db));
	gboolean other_tables = FALSE;

	tables = gda_dict_database_get_tables (dbrels->priv->db);

	menu = gtk_menu_new ();
	submitem = gtk_menu_item_new_with_label (_("Add table"));
	gtk_widget_show (submitem);
	gtk_menu_append (menu, submitem);

	if (tables) {
		submenu = gtk_menu_new ();
		gtk_menu_item_set_submenu (GTK_MENU_ITEM (submitem), submenu);
		gtk_widget_show (submenu);
		
		/* add a menu item for each table not currently shown displayed.
		 * WARNING: if a GdaDictTable is detroyed while that menu is displayed, it can still be selected
		 * from within the menu even though it will not do anything.
		 */
		list = tables;
		while (list) {
			if (! g_hash_table_lookup (dbrels->priv->hash_tables, list->data)) {
				mitem = gtk_menu_item_new_with_label (gda_object_get_name (GDA_OBJECT (list->data)));
				gtk_widget_show (mitem);
				gtk_menu_append (submenu, mitem);
				
				refbase = GDA_OBJECT_REF (gda_object_ref_new (dict));
				gda_object_ref_set_ref_object (refbase, GDA_OBJECT (list->data));
				g_object_set_data_full (G_OBJECT (mitem), "table", refbase, g_object_unref);
				
				g_signal_connect (G_OBJECT (mitem), "activate", G_CALLBACK (popup_add_table_cb), canvas);
				other_tables = TRUE;
			}
			list = g_slist_next (list);
		}
		g_slist_free (tables);
	}

	/* sub menu is incensitive if there are no more tables left to add */
	gtk_widget_set_sensitive (submitem, other_tables);

	return menu;
}

static void
popup_add_table_cb (GtkMenuItem *mitem, GnomeDbGooDbRelations *canvas)
{
	GdaObjectRef *refbase;
	GdaDictTable *table;

	refbase = g_object_get_data (G_OBJECT (mitem), "table");
	table = (GdaDictTable *) gda_object_ref_get_ref_object (refbase);

	if (table) {
		GdaGraphItem *gitem;
		GdaGraph *graph;

		gitem = GDA_GRAPH_ITEM (gda_graph_item_new (gda_object_get_dict (GDA_OBJECT (canvas->priv->db)), GDA_OBJECT (table)));
		gda_graph_item_set_position (gitem, GNOME_DB_GOO (canvas)->xmouse, GNOME_DB_GOO (canvas)->ymouse);

		graph = gnome_db_goo_get_graph (GNOME_DB_GOO (canvas));
		if (!graph && canvas->priv->db) {
			graph = (GdaGraph*) gda_graph_new (gda_dict_database_get_dict (canvas->priv->db),
							   GDA_GRAPH_DB_RELATIONS);
			g_object_set (G_OBJECT (canvas), "graph", graph, NULL);
			g_object_unref (graph);
		}
		gda_graph_add_item (graph, gitem);
		g_object_unref (G_OBJECT (gitem));
	}
}
