'''
Defines the base class for all L{AEOutput} devices.

@author: Larry Weiss
@author: Peter Parente
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import UIElement, Style

class AEOutput(UIElement.UIE):
  '''
  Defines the base class for all output devices. All output devices used by
  LSR should derive from this class and should implement the methods defined
  here.

  All methods defined here raise L{NotImplementedError} to ensure that
  derived classes create appropriate implementions.
  
  @cvar DEFAULT_STYLE: Style class to use when constructing a default style
    object for this device. A subclass with configurable device settings should
    override the default value (L{AEOutput.StyleDefault}) with a reference to
    its own default class defining default values for all device properties.
  @type DEFAULT_STYLE: L{AEOutput.StyleDefault} class
  @cvar FLYWEIGHT_STYLE: Style class to use when constructing a flyweight style
    object for this device. A subclass with configurable device settings on a
    per information type basis should override the default value 
    (L{AEOutput.StyleFlyweight}) with a reference to its own flyweight class
    possibly defining property values relative to the default.
  @type FLYWEIGHT_STYLE: L{AEOutput.StyleFlyweight} class
  @cvar USE_THREAD: Should this device use a separate thread to queue output?
    Devices that block while doing output (e.g. serial Braille device) should 
    use a thread to keep the main thread unlocked and responsive. Defaults to 
    None so a subclass must override it with an explicit boolean value.
  @type USE_THREAD: boolean
  @cvar COMMAND_CHARS: String of characters that are treated as commands on the
    device implementing this interface. These characters are replaced with
    blanks in any string sent to the device.
  @type COMMAND_CHARS: string
  @ivar listeners: List of callables that should be notified when a marker 
    inserted with L{sendIndex} is encountered
  @type listeners: list
  @ivar styles: Mapping from arbitrary, immutable keys to style objects
  @type styles: dictionary
  '''
  USE_THREAD = None
  COMMAND_CHARS = ''
  DEFAULT_STYLE = Style.StyleDefault
  FLYWEIGHT_STYLE = Style.StyleFlyweight

  def __init__(self):
    '''
    Initializes the empty listeners list.
    '''
    self.listeners = []
    self.styles = {}
    # instantiate an instance of the DEFAULT_STYLE class; it may be filled with 
    # data when preInit is called
    self.default_style = self.DEFAULT_STYLE()
    
  def postInitOutput(self, sett_man):
    '''
    Loads persisted style objects from disk. Called after the L{init} method
    by the L{DeviceManager} to ensure that the device is functioning before
    time is spent unserializing its style data.
    
    @param sett_man: Instance of the settings manager
    @type sett_man: L{SettingsManager}
    @raise KeyError: When styles have not previously been persisted for this
      device
    @raise OSError: When the profile file cannot be opened or read
    '''
    # use the settings manager to try to load persisted styles
    self.styles = sett_man.loadStyles(self.getClassName(), self.default_style,
                                      self.FLYWEIGHT_STYLE)
    
  def postCloseOutput(self, sett_man):
    '''
    Persists styles to disk. Called after the L{close} method by the 
    L{DeviceManager} to ensure the device is properly shutdown before 
    serializing its data.
    
    @param sett_man: Instance of the settings manager
    @type sett_man: L{SettingsManager}
    @raise KeyError: When styles have not previously been persisted for this
      device
    @raise OSError: When the profile file cannot be opened or read
    '''
    # use the settings manager to try to persist styles
    sett_man.saveStyles(self.getClassName(), self.default_style, self.styles)

  def init(self):
    '''
    Called after the instance is created to initialize the device.

    If called when already initialized, this will restore the device to it's
    initialized state. May also be called to re-initialize the device after
    a call to L{close}.

    @raise NotImplementedError: When not overridden in a subclass
    @raise Error.InitError: When a communication or state problem exists for 
      the specific device
    '''
    raise NotImplementedError
  
  def createDistinctStyles(self, num_groups, num_layers):
    '''
    Creates a default set of distinct styles for this device. The device should
    instantiate L{AEOutput.Style} objects having properties that reflect the
    capabilities of this device, leaving any unsupported fields set to the 
    value of None. A total number of num_groups + num_layers style objects
    should be returned.
    
    The properties of the style object should be set so as to distinguish 
    content presented using the style. For audio devices, the suggested way
    to distinguish styles is the following:
    
      - Create a total of num_groups styles with different voices and pitches.
      - If the device supports multiple channels,
        - Duplicate the styles num_layers times assigning each set of 
          duplicates to the same channel.
      - If the device does not support multiple channels,
        - Duplicate the styles num_layers times.

    For Braille devices, just create num_groups * num_layers copies of a single
    style object.
    
    The total number of requested styles (num_groups * num_layers) must be
    returned. If the device cannot honor the request for the number of distinct
    styles it is asked to generate, it may duplicate styles it has already
    created using L{AEState.AEState.copy} to fulfill the quota.
    
    The device is B{not} expected to create distinct styles across invocations 
    of this method. This method should only be called once by L{DeviceManager}
    to create a default set of styles for the device.
    
    @param num_groups: Number of sematic groups the requestor would like to
      represent using distinct styles
    @type num_groups: integer
    @param num_layers: Number of content origins (e.g. output originating from
      a background task versus the focus) the requestor would like to represent
      using distinct styles
    @type num_layers: integer
    @return: New styles
    @rtype: list of L{AEOutput.Style}
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError
  
  def setStyle(self, key, style):
    '''
    Stores the style object under the given key. The style object should be
    one previously generated by this device (e.g. using 
    L{createDistinctStyles}) but it is not an enforced requirement. Always
    makes the style clean before storing it.
    
    @param key: Any immutable object
    @type key: immutable
    @param style: L{AEOutput} subclass of L{AEState}
    @type style: L{AEState}
    '''
    style.makeClean()
    self.styles[key] = style
    
  def getStyle(self, key):
    '''
    Gets the style object stored under the given key. If the key is unknown,
    returns an empty flyweight backed by the default style and stores the
    new style in L{styles}.
    
    @param key: Any immutable object
    @type key: immutable
    @return: L{AEOutput} subclass of L{AEState}
    @rtype: L{AEState}
    '''
    try:
      return self.styles[key]
    except KeyError:
      st = self.FLYWEIGHT_STYLE(self.default_style)
      self.styles[key] = st
      return st
  
  def getDefaultStyle(self):
    '''
    Gets the default style of the device.
    
    @return: Default style of the device
    @rtype: L{AEOutput.Style}
    '''
    return self.default_style

  def close(self):
    '''
    Closes an initialized output device.

    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError
  
  def getProxy(self):
    '''
    Gets the object that the L{DeviceManager} will use to communicate with this
    device. The returned object may either be a proxy (e.g. a thread) or this
    device itself.
    
    @return: An output object that implements this class
    @rtype: L{AEOutput}
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError

  def sendString(self, text, style):
    '''
    Sends a string of one or more characters to the device.

    Depending on the device and it's features, the character may first be
    buffered until either "enough" characters and commands have been
    received, or L{sendTalk} has been called. To guard against sending
    unintended commands, the class implementing this method should remove all
    L{COMMAND_CHARS} from the text.

    @param text: Text to send to the device
    @type text: string
    @param style: Style with which this text should be output
    @type style: L{AEOutput.Style}
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError
  
  def sendFilename(self, fn, style):
    '''
    Sends a string filename to the device, the contents of which should be 
    output.
    
    Typically, this method should be implemented by an audio device that 
    supports playback of waveform or sequencer files. It might also be used
    by devices as a way of synthesizing the entire contents of a file without
    having to pass all of the contents through the rest of the system.
    
    @param fn: Absolute filename
    @type fn: string
    @param style: Style with which this text should be output
    @type style: L{AEOutput.Style}
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError

  def sendStop(self, style=None):
    '''
    Purges buffered text and styles, and interrupts current output.

    @param style: Style indicating which channel on which the stop should be 
      performed; None indicates stop on all channels
    @type style: L{AEOutput.Style}
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError

  def sendTalk(self, style=None):
    '''
    Indicates all text buffered by L{sendString} should now be output. 
    For devices that do the buffering in the driver, this action may mean simply 
    sending the command. For devices that do not buffer, this action means 
    sending text and styles buffered in the LSR device definition.

    @param style: Style indicating which channel on which the talk should be 
      performed; None indicates talk on all channels
    @type style: L{AEOutput.Style}
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError
    
  def sendIndex(self, style):
    '''
    Inserts a marker in the output stream. The device should notify all 
    listeners when the marker is reached. The marker is typically a 
    monotonically increase integer number mod the maximum integer value 
    supported by the device.
    
    @param style: Style indicating which channel on which the marker should be 
      inserted
    @type style: L{AEOutput.Style}
    @return: Unique marker identifying the index inserted
    @rtype: integer
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError

  def sendStringSync(self, text, style=None, stop=True):
    '''
    Sends a complete string to a device followed by a talk command to output it
    immediately. Blocks until the text has finished being spoken.
    
    Do B{not} use this method in place of L{sendString} for normal device 
    operation. It is intended as a method for doing device output pre-runtime
    (e.g. when debugging).
    
    @param text: String to send to the device
    @type text: string
    @param style: Style on which this string should be output; None implies some
      reasonable default should be used
    @type style: L{AEOutput.Style}
    @param stop: Stop current output before doing this output?
    @type stop: boolean
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError
    
  def addIndexListener(self, listener):
    '''
    Adds a listener that should be notified when speech has progressed to the 
    point where a marker was inserted with L{sendIndex}.
    
    @param listener: The method that should be called when markers received.
    @type listener: callable
    '''
    self.listeners.append(listener)

  def removeIndexListener(self, listener):
    '''
    Removes the specified listener.
    
    @param listener: The method that should no longer be called when markers 
      received.
    @type listener: callable
    @raise ValueError: When the given listener is not already registered
    '''
    self.listeners.remove(listener)
  
  def isActive(self):
    '''
    Indicates whether the device is active (giving output) or not.

    @return: True when content is buffered or the device is outputing
    @rtype: boolean
    @raise NotImplementedError: When not overriden in a subclass
    '''
    raise NotImplementedError
  
  def parseString(self, text, style, por):
    '''
    Parses a string to be output based on the provided style. The goal of the 
    parsing is to:
    
      - break the string into words which can be indexed
      - manipulate the string to affect its presentation
      - manipulate the style to affect the presentation of one or more words
    
    This method returns the list of parsed words, their associated L{POR}s, and
    their associated styles. When L{POR}s are not available or the style has
    not be changed for a particular word, None values are inserted into the 
    respective lists so that all returned lists are of the same length.
    
    @param text: Text to be parsed
    @type text: string
    @param style: Style object defining how the text should be parsed
    @type style: L{AEOutput.Style}
    @param por: Point of regard for the first character in the text, or None if
      the text is not associated with a POR
    @type por: L{POR}
    @return: Parsed words
    @rtype: 3-tuple of lists of string, L{POR}, L{AEOutput.Style}
    '''
    raise NotImplementedError
