'''
Defines global constants. All SEM_* constants represent information types that
a user may wish to style differently for output. All HELP_* strings are shown
in response to a --help request at the command prompt. All STYLE_GROUP_* 
constants represent the default groupings for semantic constants and are used
to provide a default distinction of information type using output styles.

@var PROG_NAME: Name of this program
@type PROG_NAME: string
@var PROG_VERSION: Version of this program
@type PROG_VERSION: string
@var PROG_DATE: Date of this program's last install or packaging
@type PROG_DATE: string
@var NAME: Human readable name of this program
@type NAME: string
@var COPYRIGHT: Copyright notice for this program
@type COPYRIGHT: string
@var LICENSE: License summary for this program
@type LICENSE: string
@var COLOR_MAP: Mapping from a six bit palette of colors to names describing
  them
@type COLOR_MAP: dictionary
@var PREFIX: Install prefix for this program
@type PREFIX: string
@var HOME_USER: Configuration directory for the Unix user currently running
  this program
@type HOME_USER: string
@var HOME_DIR: Home directory for this program install (e.g. 
  /usr/lib/python2.4/site-packages/lsr)
@type HOME_DIR: string
@var DATA_DIR: System data directory for this program install (e.g. 
  /usr/share/lsr)
@type DATA_DIR: string
@var TEMPLATE_DIR: Template data directory for this program install (e.g. 
  /usr/share/lsr/templates)
@type TEMPLATE_DIR: string
@var MSG_CONFIRM_REMOVE: Message displayed to confirm a removal at the command
  line
@type MSG_CONFIRM_REMOVE: string
@var STYLE_GROUP_ALL: List of all style group constants
@type STYLE_GROUP_ALL: list
@var SEMANTIC_STYLES: Mapping from semantic to its default group
@type SEMANTIC_STYLES: dictionary
@var SEMANTIC_NAMES: Mapping from semantic constant to its human readable name,
  currently the constant variable name
@type SEMANTIC_NAMES: dictionary
@var DEFAULT_PROFILE: Name of the default profile to use when none is specified
@type DEFAULT_PROFILE: string
@var BUILTIN_PROFILES: Names of built-in, protected profiles
@type BUILTIN_PROFILES: list of string
@var DEFAULT_ASSOCIATIONS: Mapping from built-in profile name to L{UIElement}s
  that ship with LSR that should be part of the profile by default
@type DEFAULT_ASSOCIATIONS: dictionary

@author: Pete Brunet
@author: Peter Parente
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import os
from i18n import _

# package name
PROG_NAME = 'lsr'

# version numbers
PROG_VERSION = '0.3.1'
PROG_DATE = 'Wed Nov  1 15:40:25 UTC 2006'

# name, version, copyright and license notice to print on startup
NAME = 'Linux Screen Reader'
COPYRIGHT = 'Copyright (c) 2005, 2006 IBM Corporation'
LICENSE = 'All rights reserved. This program and the accompanying ' \
        'materials are made available under the terms of the Common Public ' \
        'License v1.0 which accompanies this distribution, and is available ' \
        'at http://www.opensource.org/licenses/cpl1.0.php'

# home directories and data directories
PREFIX = '/usr/local'
HOME_USER = os.path.join(os.environ['HOME'], '.'+PROG_NAME)
HOME_DIR = os.path.dirname(__file__)
# make sure the user home directory is initialized
try:
  os.makedirs(HOME_USER)
except OSError:
  pass
DATA_DIR = os.path.join(PREFIX, 'share', PROG_NAME)
TEMPLATE_DIR = os.path.join(DATA_DIR, 'templates')

# command line help strings
HELP_GENERATE = _('generate a template for a new UIE')
HELP_INSTALL = _('install a UIE')
HELP_UNINSTALL = _('uninstall a UIE')
HELP_ASSOCIATE = _('associate a UIE with a profile')
HELP_DISASSOCIATE = _('disassociate a UIE from a profile')
HELP_CREATE_PROFILE = _('create a new user profile')
HELP_REMOVE_PROFILE = _('remove a user profile')
HELP_DUPLICATE_PROFILE = _('duplicate a user profile')
HELP_SHOW = _('show installed UIEs')
HELP_PROFILE = _('profile name')
HELP_SAY = _('output a string using the default device')
HELP_INDEX = _('load order index for the UIE')
HELP_TIER = _('process name with which a Perk should be associated')
HELP_ALL_TIERS = _('flag indicating a Perk should be associated with all apps')
HELP_GLOBAL = _('flag indicating the command should be performed globally')
HELP_LOG_LEVEL = _('level of log messages: %s')
HELP_LOG_CHANNEL = _('channel of log messages, any of: %s')
HELP_LOG_FILENAME = _('filename for simple log output')
HELP_LOG_CONFIG = _('filename for Python logging module configuration')
HELP_NO_INTRO = _('suppress the welcome announcement')
HELP_INIT_PROFILES = _('initialize built-in profiles')
HELP_INIT_GLOBAL = _('initialize installed UIEs with packaged UIEs')
HELP_A11Y = _('Desktop accessibility disabled. Enabling...' \
              '\nPlease logout and run LSR again.')

# command line messages
MSG_CONFIRM_REMOVE = _('Are you sure you want to remove %s? ') + '(y/n) '

COLOR_MAP = {0:_('black'),
             1:_('dark purple'),
             2:_('dark blue'),
             3:_('blue'), 
             4:_('dark green'),
             5:_('dark cyan'),
             6:_('aqua'), 
             7:_('medium blue'),
             8:_('dark green'),
             9:_('lime green'), 
             10:_('dark cyan'),
             11:_('aqua'),
             12:_('green'), 
             13:_('bright green'),
             14:_('blue green'),
             15:_('cyan'),
             16:_('dark red'),
             17:_('magenta'),
             18:_('purple'),
             19:_('purple'), 
             20:_('olive green'),
             21:_('medium grey'),
             22:_('steel blue'),
             23:_('blue grey'),
             24:_('green'),
             25:_('pale green'), 
             26:_('blue grey'),
             27:_('light blue'),
             28:_('light green'), 
             29:_('light green'),
             30:_('cyan'),
             31:_('light blue'), 
             32:_('dark red'),
             33:_('magenta'),
             34:_('dark pink'),
             35:_('purple'),
             36:_('light brown'),
             37:_('flesh'),
             38:_('pink'), 
             39:_('light purple'),
             40:_('olive green'),
             41:_('light grey'),
             42:_('dark grey'),
             43:_('light blue'),
             44:_('light green'), 
             45:_('pale green'),
             46:_('pale green'),
             47:_('light blue'), 
             48:_('red'), 
             49:_('pink'), 
             50:_('pink'), 
             51:_('pink'), 
             52:_('orange'), 
             53:_('puce'), 
             54:_('pink'), 
             55:_('pink'), 
             56:_('dark yellow'), 
             57:_('tan'), 
             58:_('light pink'), 
             59:_('light pink'),
             60:_('yellow'),
             61:_('pale yellow'), 
             62:_('off white'),
             63:_('white') }

# default style groupings
STYLE_GROUP_CONTENT = 0
STYLE_GROUP_PROPERTIES = 1
STYLE_GROUP_ECHO = 2
STYLE_GROUP_CONTEXT = 3
STYLE_GROUP_ALL = [STYLE_GROUP_CONTENT, STYLE_GROUP_PROPERTIES,
                   STYLE_GROUP_ECHO, STYLE_GROUP_CONTEXT]

# enumeration of all default semantics
# CONTENT 10-29
SEM_CHAR = 1
SEM_WORD = 2
SEM_ITEM = 3
SEM_HARDLINE = 4
SEM_NAME = 5
SEM_LABEL = 6
SEM_DESC = 7
SEM_VALUE = 8
# PROPERTIES 30-59
SEM_ROLE = 30
SEM_FONT = 31
SEM_TEXT_ATTR = 32
SEM_COLOR = 33
SEM_STATE = 34
SEM_LENGTH = 35
SEM_INDEX = 36
SEM_LEVEL = 37
SEM_STATE = 38
SEM_STATUS = 39
SEM_HOTKEY = 40
SEM_EXTENTS = 41
# ECHO 60-79
SEM_INFO = 60
SEM_ERROR = 61
SEM_CONFIRM = 62
# CONTEXT 80-99
SEM_APP = 80
SEM_WINDOW = 81
SEM_SECTION = 82

# mapping of semantics to default styles
SEMANTIC_STYLES = {
  # CONTENT
  SEM_CHAR :      STYLE_GROUP_CONTENT,
  SEM_WORD :      STYLE_GROUP_CONTENT,
  SEM_ITEM :      STYLE_GROUP_CONTENT,
  SEM_HARDLINE :  STYLE_GROUP_CONTENT,
  SEM_LABEL :     STYLE_GROUP_CONTENT,
  SEM_DESC :      STYLE_GROUP_CONTENT,
  SEM_VALUE:      STYLE_GROUP_CONTENT,
  # PROPERTIES
  SEM_ROLE :      STYLE_GROUP_PROPERTIES,
  SEM_FONT :      STYLE_GROUP_PROPERTIES,
  SEM_TEXT_ATTR : STYLE_GROUP_PROPERTIES,
  SEM_COLOR :     STYLE_GROUP_PROPERTIES,
  SEM_STATE :     STYLE_GROUP_PROPERTIES,
  SEM_LENGTH :    STYLE_GROUP_PROPERTIES,
  SEM_INDEX :     STYLE_GROUP_PROPERTIES,
  SEM_LEVEL :     STYLE_GROUP_PROPERTIES,
  SEM_STATE :     STYLE_GROUP_PROPERTIES,
  SEM_STATUS :    STYLE_GROUP_PROPERTIES,
  SEM_HOTKEY :    STYLE_GROUP_PROPERTIES,
  SEM_EXTENTS :   STYLE_GROUP_PROPERTIES,
  # CONTEXT
  SEM_APP :       STYLE_GROUP_CONTEXT,
  SEM_WINDOW :    STYLE_GROUP_CONTEXT,
  SEM_SECTION :   STYLE_GROUP_CONTEXT,
  # ECHO
  SEM_INFO :      STYLE_GROUP_ECHO,
  SEM_ERROR :     STYLE_GROUP_ECHO,
  SEM_CONFIRM :   STYLE_GROUP_ECHO
}

# provide a reverse mapping from semantic value to semantic variable name
SEMANTIC_NAMES = dict([(value, name.lower()[4:]) for name, value in 
                       locals().items() if name.startswith('SEM_')])

# associations for default profiles that ship with LSR
DEFAULT_ASSOCIATIONS = {
  'user': ['Keyboard', 'CliqueAudio', 'IBMSpeech', 'DECtalkSpeech',
           'SpeechDispatcher', 'FestivalSpeech', 'DefaultPerk',
           'DefaultDialogPerk', 'AltShiftPerk', 'MetacityPerk', 'GaimPerk',
           'GTerminalPerk', 'FirefoxPerk'],
  'developer' :
          ['RawEventMonitor','TaskMonitor', 'IOMonitor', 'Keyboard',
           'CliqueAudio', 'IBMSpeech', 'DECtalkSpeech', 'SpeechDispatcher',
           'FestivalSpeech', 'DefaultPerk', 'DefaultDialogPerk', 
           'AltShiftPerk', 'DeveloperPerk', 'MetacityPerk', 'GaimPerk', 
           'GTerminalPerk', 'FirefoxPerk'],
  'monitors' :
          ['RawEventMonitor','TaskMonitor', 'IOMonitor', 'Keyboard',
           'DefaultPerk', 'DefaultDialogPerk', 
           'AltShiftPerk', 'DeveloperPerk', 'MetacityPerk', 'GaimPerk', 
           'GTerminalPerk', 'FirefoxPerk'],
  'autotest' :
          ['TestLogMonitor','Keyboard', 'DefaultPerk', 'DefaultDialogPerk',
           'AltShiftPerk', 'MetacityPerk', 'GaimPerk', 'GTerminalPerk'],
  'login' :
          ['Keyboard', 'CliqueAudio', 'IBMSpeech', 'DECtalkSpeech',
           'SpeechDispatcher', 'FestivalSpeech', 'DefaultPerk',
           'AltShiftPerk', 'GdmPerk']
        }
# names of built-in and default LSR profiles
BUILTIN_PROFILES = DEFAULT_ASSOCIATIONS.keys()
DEFAULT_PROFILE = 'user'
