'''
Defines L{Tools} for getting input.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made 
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''

import Base
from Error import *

class Input(Base.TaskTools):
  '''
  Provides methods for getting input on any L{AEInput} device.
  '''
  def getInputDevice(self, name):
    '''
    Gets an L{AEInput} device from the L{DeviceManager} given its name.
    
    @param name: UIE name of the L{AEInput} device to get
    @type name: string
    @return: The named input device
    @rtype: L{AEInput}
    @raise InvalidDeviceError: When a L{AEInput} device with the given name is
      not registered
    '''
    device = self.device_man.getInputByName(name)
    if device is None:
      # let the system report the device isn't available
      raise InvalidDeviceError
    return device
  
  def addInputModifiers(self, dev, *codes):
    '''
    Adds the given device code as a modifier on the named L{AEInput} device.
    If the device doesn't support modifiers, ignores the error since it's a 
    non-critical operation.
    
    @param dev: Reference to an input device
    @type dev: L{AEInput}
    @param codes: Action codes to add as modifiers
    @type codes: integer
    '''
    try:
      func = dev.addModifier 
    except AttributeError:
      return
    map(func, codes)

  def registerCommand(self, dev, task_ident, propogate, *codes):
    '''
    Registers a L{Task} in this L{Perk} to be executed in response to an 
    L{AEEvent} indicating that the given action codes were input on the given 
    L{AEInput} device. 
    
    @param dev: Device to monitor for input
    @type dev: L{AEInput}
    @param task_ident: Name of the L{Task} registered via 
      L{Task.Tools.System.System.registerTask} to execute when the input 
      gesture is detected on the device
    @type task_ident: string
    @param propogate: Should the input gesture be allowed to propogate to the 
      OS after we receive it?
    @type propogate: boolean
    @param codes: List of lists of action codes forming the L{AEInput.Gesture} 
      that will trigger the execution of the named L{Task}. For example, 
      codes=[[Keyboard.AEK_CTRL, Keyboard.AEK_TILDE]] indicates the single
      gesture of simultaneously pressing Ctrl and ~ on the keyboard device.
    @type codes: list of list of integer
    @raise ValueError: When a L{Task} with the given name is not registered
    @raise InvalidDeviceError: When a L{AEInput} device with the given name is
      not registered
    '''
    self.perk.registerCommandTask(dev, codes, task_ident, propogate)
    
  def unregisterCommand(self, dev, *codes):
    '''
    Unregisters a L{Task} set to execute in response to the given action codes 
    on the given device B{from this L{Perk} only}.
    
    @param dev: Device to monitor for input
    @type dev: L{AEInput}
    @param codes: List of lists of action codes forming the L{AEInput.Gesture} 
      that will trigger the execution of the named L{Task}. For example, 
      codes=[[Keyboard.AEK_CTRL, Keyboard.AEK_TILDE]] indicates the single
      gesture of simultaneously pressing Ctrl and ~ on the keyboard device.
    @type codes: list of list of integer
    @raise KeyError: When a L{AEInput.GestureList} is not registered
    @raise InvalidDeviceError: When a L{AEInput} device with the given name is
      not registered
    '''
    self.perk.unregisterCommandTask(dev, codes)