'''
Defines a threaded proxy for L{AEOutput.Audio} devices.

@author: Brett Clippingdale
@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''

import threading, Queue, Base

class AudioThreadProxy(threading.Thread, Base.AEOutput): 
  '''  
  Buffers calls to methods on a L{AEOutput.Audio} device in a secondary thread
  and executes them some time later when that thread runs.
  
  @ivar device: Device on which to invoke send* methods
  @type device: L{AEOutput.Audio}
  @ivar want_stop: Flag indicating a stop is requested
  @type want_stop: boolean
  @ivar just_stopped: Flag indicating that the last written command was a stop.
    Used to avoid unnecessary stops
  @type just_stopped: boolean
  @ivar lock: Semaphore used to ensure no commands are buffered while the 
    buffer is being reset after L{sendStop} is called
  @type lock: threading.Sempahore
  @ivar init_event: Event used to block non-threaded calls to the device until
    the device has been initialized in the context of the running thread.
  @type init_event: threading.Event
  @ivar data_buffer: Buffer of commands to be sent to the output device
  @type data_buffer: Queue.Queue
  @ivar alive: Is the thread running or not?
  @type alive: boolean
  '''
  def __init__(self, device):
    '''
    Initializes the parent class and stores the device reference. Creates a 
    queue that will buffer commands to the speech device. Creates flags used
    for indicating whether a stop is requested or has been requested recently.
    Creates a semaphore used to ensure that no commands can be added to the 
    buffer while it is being reset by a stop command.

    @param device: The device reference to use for writing commands
    @type device: L{AEOutput.Audio}
    '''
    threading.Thread.__init__(self)
    Base.AEOutput.__init__(self)
    self.device = device
    self.want_stop = False
    self.just_stopped = False
    self.lock = threading.Semaphore()
    self.init_event = threading.Event()
    self.data_buffer = Queue.Queue()
    self.alive = False
    
  def init(self):
    '''
    Called after the instance is created to start the device running. The
    device's init method is called in the context of the running thread's
    L{run} method before the thread enters its loop.
    '''
    self.alive = True
    self.start()
    
  def getCapabilities(self):
    '''    
    Gets the capabilities of the proxied device. This method is called in the
    context of the caller, not the running thread.

    @return: List of capability names
    @rtype: list of string
    '''
    return self.device.getCapabilities()

  def createDistinctStyles(self, num_groups, num_layers):
    '''
    Creates up to the given number of styles for this device. This method is 
    called in the context of the caller, not the running thread. It blocks
    until the L{init_event} is set indicating the device has been initialized
    in the second thread.
    
    @param num_groups: Number of sematic groups the requestor would like to
      represent using distinct styles
    @type num_groups: integer
    @param num_layers: Number of content origins (e.g. output originating from
      a background task versus the focus) the requestor would like to represent
      using distinct styles
    @type num_layers: integer
    @return: New styles
    @rtype: list of L{AEOutput.Style}
    '''
    self.init_event.wait()
    return self.device.createDistinctStyles(num_groups, num_layers)
  
  def getDefaultStyle(self):
    '''
    Creates up to the given number of styles for this device. This method is 
    called in the context of the caller, not the running thread. It blocks
    until the L{init_event} is set indicating the device has been initialized
    in the second thread.
    
    @return: Default style
    @rtype: L{AEOutput.Style}
    '''
    self.init_event.wait()
    return self.device.getDefaultStyle()

  def close(self):
    '''
    Stops the running thread. Puts a null callable in the queue to wake the
    thread if it is sleeping.
    '''
    self.alive = False
    self._put(lambda : None)
  
  def getProxy(self):
    '''
    Returns this object as the proxy for itself because a thread proxying for
    another thread proxying for a device is not supported.
    
    @return: self
    @rtype: L{AudioThreadProxy}
    '''
    return self

  def getName(self):
    '''
    Gives the user displayable (localized) name for this output device.
    Relevant version and device status should be included. This method is 
    called in the context of the caller, not the running thread.

    @return: The localized name for the device
    @rtype: string
    '''
    return self.device.getName()
  
  def send(self, name, value, style=None):
    '''    
    Buffers methods to call for known commands in the context of the thread.
    
    @param name: Descriptor of the data value sent
    @type name: object
    @param value: Content value
    @type value: object
    @param style: Style with which this value should be output
    @type style: L{AEOutput.Style}
    @return: Return value specific to the given command
    @rtype: object
    @raise NotImplementedError: When not overridden in a subclass
    '''
    if name == Constants.CMD_STOP:
      self._put(self.device.sendStop, style)
    elif name == Constants.CMD_TALK:
       self._put(self.device.sendTalk, style)
    elif name == Constants.CMD_STRING:
       self._put(self.device.sendString, value, style)
    elif name == Constants.CMD_STRING_SYNC:
      self.device.sendStringSync(value, style)
    elif name == Constants.CMD_FILENAME:
      try:
        # may not be implemented
        self._put(self.device.sendFilename, value, style)
      except NotImplementedError:
        pass
    elif name == Constants.CMD_INDEX:
      try:
        # may not be implemented
        return self._put(self.device.sendIndex, style)
      except NotImplementedError:
        pass
  
  def isActive(self):
    '''
    Indicates whether the device is active (giving output) or not.

    @return: True when content is buffered or the device is outputing
    @rtype: boolean
    @raise NotImplementedError: When not overriden in a subclass
    '''
    return (self.data_buffer.qsize() != 0 or self.device.isActive())
  
  def parseString(self, text, style, por):
    '''
    Parses the string using the implementation provided by the proxied device.
    This method is called in the context of the caller, not the running thread.
    
    @param text: Text to be parsed
    @type text: string
    @param style: Style object defining how the text should be parsed
    @type style: L{AEOutput.Style}
    @param por: Point of regard for the first character in the text, or None if
      the text is not associated with a POR
    @type por: L{POR}
    @return: Parsed words
    @rtype: 3-tuple of lists of string, L{POR}, L{AEOutput.Style}
    '''
    return self.device.parseString(text, style, por)
    
  def _put(self, mtd, *args):
    '''
    Buffers any non-stop command in the queue and returns immediately. Sets the
    L{want_stop} flag for a stop command if neither L{want_stop} nor 
    L{just_stopped} is set. Blocks on entry if the thread is busy clearing out
    the buffer in response to a previous stop command. Leaves the lock set if
    a new stop command is buffered. It will be unset when the command is
    processed.
    
    @param mtd: Device method to call at a later time
    @type mtd: callable
    @param args: Additional positional arguments to be passed to the device
      when the method is invoked
    @type args: list
    '''
    self.lock.acquire()
    if (mtd is self.device.sendStop):
      if not self.want_stop and not self.just_stopped:
        # only queue the stop and set the flag if it's not already set
        self.want_stop = True
        # add the stop command to the buffer in case the thread is sleeping
        self.data_buffer.put_nowait((mtd, args))
        # IMPORTANT: do not release the semaphore, the thread will do it when
        # it processes the stop command; it must be held so that further 
        # invocations of this method block until the buffer has been emptied
        return
    else:
      self.data_buffer.put_nowait((mtd, args))
      # as soon as something is buffered, stop is no longer last.
      self.just_stopped = False
    self.lock.release()
    
  def run(self):
    '''
    Runs the thread until alive is not longer True. Sleeps until methods and
    arguments to be applied are put in the L{data_buffer}. Wakes up and 
    invokes all buffered methods and arguments. Initializes the device before
    entering the loop and closes it after leaving the loop.
    '''
    self.device.init()
    self.init_event.set()
    while self.alive:
      # Queue.get() blocks for data
      mtd, args = self.data_buffer.get()

      # if want to stop, it doesn't matter what the buffer element was
      if self.want_stop:
        # reset the buffer immediately
        self.data_buffer = Queue.Queue()
        # send the device specific stop command
        self.device.sendStop()
        # reset the stop flag
        self.want_stop = False
        # indicate we've just stopped
        self.just_stopped = True
        # IMPORTANT: release the semaphore so the put method can continue
        self.lock.release()
      else:
        # handle the command
        try:
          mtd(*args)
        except NotImplementedError:
          pass
    self.device.close()