'''
Defines an abstract base class for all persistent and/or configurable setting
classes.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''

import Setting

class AEState(object):
  '''
  Abstract base class for objects containing data that will have their data 
  serialized to disk or configured by the user. Any instance variables not 
  prefixed with at least one underscore are candidates for persistence. The 
  L{getSettings} method must be overridden to support user configuration. The 
  L{initNonPersistent} method can be overridden to initialize non-persistent
  variables when they are loaded from disk, though defining default values
  in class variables may be a better solution for forward compatibility.
  
  One attribute name is reserved to optimize persistence of L{AEState} objects.
  The L{_dirty} set stores the names of attributes that have been written to in
  an instance. The list is automatically reset when the instance is persisted. 
  The L{isDirty}, L{makeClean}, L{iterDirty} methods may be used to check if
  any part of the state is dirty, clear the list of dirty attributes, and 
  iterate over the names and values of the dirty attributes to enable 
  optimization at higher levels.
  
  Note that the L{AEState} object itself is not designed to be persisted to 
  disk, but rather to act as a delegate for settings to be persisted. The 
  reason for this is extension may use the L{AEState} class as a base for 
  defining their own settings, those extensions may be reloaded at runtime, and
  Pickle refuses to persist instances of classes that were dynamically 
  reloaded.
  
  @ivar _dirty: Set of dirty attribute names in this state object
  @type _dirty: set
  '''
  def __init__(self):
    '''Initialize the dirty set.'''
    self._dirty = set()
    
  def __eq__(self, other):
    '''
    Compares two state objects for equality based on their attributes.
    
    @param other: Another state object
    @type other: L{AEState}
    '''
    if other is None:
      return False
    return vars(self) == vars(other)
  
  def __ne__(self, other):
    '''
    Compares two state objects for inequality based on their attributes.
    
    @param other: Another state object
    @type other: L{AEState}
    '''
    if other is None:
      return True
    return vars(self) != vars(other)
    
  def __setattr__(self, name, value):
    '''
    Updates the dirty list with the name to be set to the given value. Ignores
    names that start with _.
    
    @param name: Name of the attribute to set
    @type name: string
    @param value: Arbitrary value
    @type value: object
    '''
    if not name.startswith('_'):
      self._dirty.add(name)
    object.__setattr__(self, name, value)
    
  def copy(self):
    '''
    Makes a fast copy of this object's __dict__.
    
    @return: New state object
    @rtype: L{AEState}
    '''
    other = self.__class__()
    other.__dict__.update(self.__dict__)
    return other
  
  def getState(self):
    '''
    Removes all non-persistent instance variables from a copy of the instance
    dictionary before persisting the instance.

    @return: Values to persist
    @rtype: dictionary
    '''
    # reset the dirty set in the live object
    self.makeClean()
    d = self.__dict__.copy()
    # remove all non-persistent attributes and the dirty set
    names = [name for name in d if name.startswith('_')]
    map(d.pop, names)
    return d
  
  def setState(self, dict):
    '''
    Restores the state of this object using the given dictionary of values. 
    Calls L{initNonPersistent} to allow the object to reinitialize all 
    non-persistent instance variables.
    
    @param dict: Values to restore
    @type dict: dictionary
    '''
    # update this instance with the variables persisted on disk
    self.__dict__.update(dict)
    try:
      # reinitialize any non-persistent data
      self.initNonPersistent()
    except NotImplementedError:
      pass
    # reset the dirty state when loading
    self.makeClean()
    
  def isDirty(self):
    '''
    @return: Is there anything in the L{_dirty} set?
    @rtype: boolean
    '''
    return len(self._dirty) != 0
  
  def makeClean(self):
    '''Resets the L{_dirty} set.'''
    self._dirty = set()
    
  def iterDirty(self):
    '''
    Acts as an iterator over all of the dirty name/value pairs in this object.
    
    @return: One dirty name/value pair on each iteration
    @rtype: 2-tuple of string, object
    '''
    for name in self._dirty:
      value = getattr(self, name)
      yield name, value
  
  def _newGroup(self, name=None):
    '''
    Creates a new L{AEState.Setting.Group} object with the given name. 
    Associates the group with this instance so that it may modify variables in
    this instance.
    
    @param name: Name of the group
    @type name: string
    @return: New group object
    @rtype: L{AEState.Setting.Group}
    '''
    return Setting.Group(self, name)
  
  def initNonPersistent(self):
    '''
    Re-initializes non-persistent instance variables when this object is 
    loaded from persistent storage. Not required if this class will not be
    persisted to disk or has no non-persistent instance variables to 
    reinitialize.
    '''
    raise NotImplementedError
    
  def getSettings(self):
    '''    
    Gets a L{AEState.Setting.Group} of L{Setting} objects representing the 
    values that should be configurable by the user. L{Setting} objects are used
    because they provide additional metadata useful in constructing
    configuration dialogs.
    
    The L{_newGroup} method can be used to instantiate the root group. Methods
    in that group can then be used to instantiate L{Setting}s.
    
    Any changes made to the settings objects will be immediately reflected in
    the instance variables in this object. Use L{AEState.Setting.Group.save}
    and L{Setting.Group.restore} to take a snapshot of the current values and
    later restore them.
    
    @return: Collection of L{Setting} objects
    @rtype: L{AEState.Setting.Group}
    @raise NotImplementedError: When not implemented in a subclass
    '''
    raise NotImplementedError