'''
Defines a L{Perk} for the Firefox web browser.

Keys registered

CapsLock-A: Read the current web page address.
CapsLock-H: Move the pointer to the next heading.
CapsLock-Home: Move the pointer to the beginning of the active document.

@var DOCUMENT_ROLE: Role name of the top of a document view.
@type DOCUMENT_ROLE: string

@author: Peter Parente
@author: Scott Haeger
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2007 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made 
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''
# import useful modules for Perks
import Perk, Task
from POR import POR
from i18n import _
from AEConstants import *
import AEConstants 
import time

__uie__ = dict(kind='perk', tier='Minefield', all_tiers=False)

DOCUMENT_ROLE = 'document frame'

class FirefoxPerkState(Perk.PerkState):
  '''
  Defines settings specific to Firefox.
  
  BoundReview (boolean): Keep review functions within the Firefox document when
  reviewing begins inside the document, or let them walk across document
  boundaries?
  
  AutoManipulable(boolean): When True, sets focus on controls when reviewing.
  
  RegConvReview(boolean): When True, automatically registers convenience 
  review keys.
  '''
  def init(self):
    self.newBool('BoundReview', True, _('Bound review to document?'),
                 _('When set, the review keys will not navigate outside '
                   'the current web page when reviewing starts in the page. '
                   'Otherwise, review keys can walk out of the document.'))           
    self.newBool('AutoManipulable', False, 
              _('Automatically move focus?'),
              _('When set, focus will be given automatically to any '
                'interactive control while reviewing the document. '
                'Otherwise, you must manually move focus to interactive '
                'widgets.'))
    self.newBool('RegConvReview', True, 
              _('Review with arrows?'),
              _('When set, the arrow keys, in conjuction with Caps-Lock, '
                'will serve as a convenient review tool providing item, word, '
                'and character review.  When unset, the arrow keys will never '
                'be registered.  These keys are unregistered outside the '
                'document and within interactive controls.  Use Caps-Lock/down'
                ' arrow to toggle this setting.'))

  def getGroups(self):
    '''
    Gets configurable settings for this L{Perk}.
    
    @return: Group of all configurable settings
    @rtype: L{AEState.Setting.Group}
    '''
    root = self.newGroup()
    root.append('BoundReview')
    root.append('AutoManipulable')
    root.append('RegConvReview')
    return root
    
class FirefoxPerk(Perk.Perk):
  '''
  Defines hotkeys to improve document browsing and chrome access for the 
  Firefox web browser.
  
  Flowcharts:
  Review task chaining: keyboard event -> BoundReview(All Review) -> 
  HandleAutofocus -> ConvReviewReg.
  Convenience review chaining: keyboard event -> ConvReviewKeys -> 
  BoundReview(All Review) -> HandleAutofocus -> ConvReviewReg.
  Review skip: ShouldSkip -> FFShouldSkip.
  Read role: ReadReviewRole(ReadNewRole)
  
  @cvar STATE: L{AEState} object that holds user setting values.
  @type STATE: L{AEState}
  @cvar SKIPPED_ITEMS: Additional items to be skipped when reviewing.
  @type SKIPPED_ITEMS: tuple
  @cvar SKIPPED_ROLES: Additional roles to be skipped when reviewing.
  @type SKIPPED_ROLES: tuple
  @cvar SKIPPED_CONVREG_ROLES: A dictionary where the keys are roles which will
  prevent the convenience review keys from being registered. When a value is
  not None, it is a state that must be contained in this accessible's state set
  before convenience review keys are registered.
  @type SKIPPED_CONVREG_ROLES: dictionary
  
  @ivar convkeyregistered: Are convenience review keys registered?
  @type convkeyregistered: boolean
  @ivar doc_pors: Dictionary containing saved pors.  Key is doc frame of web
  page to be restored.
  @type doc_pors: dictionary
  @ivar cached_pointer: L{POR} saved during caret event.  Used to restore POR 
  during certain restoration scenerios involving chrome.
  @type cached_pointer: L{POR}
  @ivar last_focus_por: L{POR} saved during restoration routine to help
  differentiate tab browsing from Tab/Tab-Shift events.
  @type last_focus_por: L{POR}
  '''
  SKIPPED_CONVREG_ROLES = {'menu item' : 'focusable',
                           'list item' : 'focusable',
                           'entry': 'focusable',
                           'password text': 'focusable',
                           'text box': 'focusable', 
                           'combo box': 'focusable', 
                           'list': 'focusable'}
  STATE = FirefoxPerkState
  SKIPPED_ITEMS = ()
  SKIPPED_ROLES = ()
  
  def init(self):
    self.convkeyregistered = False
    self.doc_pors = {}
    self.cached_pointer = None
    self.lastporinfocus = None
    self.lastdocinfocus = None
    self.viewlost = False
    
    # set an audio device as the default output
    self.setPerkIdealOutput('audio')
    
    # Register for events
    self.registerTask(ExtraCarets(None))
    self.registerTask(PointerCache(None))
    
    # Register general review related tasks
    self.registerTask(FirefoxReview('firefox review'))
    self.registerTask(ReadReviewRole('firefox review role'))
    self.registerTask(ReviewShouldSkip('firefox review should skip'))
    
    # POR restoration related (Firefox tabbing, app switching, menu selection)
    self.registerTask(HandleDocumentPORView(None, all=True))
    self.registerTask(HandleDocumentPORFocus(None, all=True))
    
    # Register convenience review key related.  Registration order matters
    # ConvReviewViewReg and ConvReviewFocusReg.  They must come after
    # HandleDocumentPORView and HandleDocumentPORFocus, respectively.
    self.registerTask(ConvReviewKeys('firefox convenience review'))
    self.registerTask(ConvReviewViewReg('firefox conv keys view reg', 
                                        all=True))
    self.registerTask(ConvReviewFocusReg('firefox conv keys focus reg', 
                                         all=True))  
    self.registerTask(ConvReviewUserReg('firefox conv keys user reg'))
    
    # Register user triggered move/announcement events
    self.registerTask(MoveToNextHeader('firefox move next header'))
    self.registerTask(MoveToActiveDoc('firefox move active doc'))
    self.registerTask(ReadDocumentAddress('firefox read document address'))
    
    # Register state watching tasks
    self.registerTask(HandleAutocomplete('firefox say autocomplete', 
                                         tier=True))
    
    # Add keyboard modifiers
    kbd = self.getInputDevice(None, 'keyboard')
    self.addInputModifiers(kbd, kbd.AEK_CAPS_LOCK)
    self.addInputModifiers(kbd, kbd.AEK_CONTROL_R)
    self.addInputModifiers(kbd, kbd.AEK_CONTROL_L)
    
    # Register keyboard commands for reading document address, move to next
    # header, and move to active doc frame.
    self.registerCommand(kbd, 'firefox read document address', False, 
                         [kbd.AEK_CAPS_LOCK, kbd.AEK_A])
    self.registerCommand(kbd, 'firefox move next header', False, 
                         [kbd.AEK_CAPS_LOCK, kbd.AEK_H])
    self.registerCommand(kbd, 'firefox move active doc', False, 
                         [kbd.AEK_CAPS_LOCK, kbd.AEK_HOME])
    self.registerCommand(kbd, 'firefox conv keys user reg', False, 
                         [kbd.AEK_CAPS_LOCK, kbd.AEK_DOWN])
    
    # chain to other tasks.  See Perk docstring for task flowcharts.
    self.chainTask('firefox review role', CHAIN_AROUND, 'read new role')
    self.chainTask('firefox review should skip', CHAIN_AROUND, 
                   'review should skip')

    # register the firefox review task around all the basic review tasks that
    # move the pointer so we can account for specific cases and features
    names = self.getPerkTaskNames('ReviewPerk')
    for name in names:
      if (((name.startswith('review previous') or # reviewing, not peeking
            name.startswith('review next')) and
            not name.endswith('peek'))):
        self.chainTask('firefox review', CHAIN_AROUND, name)
        

  def getName(self):
    return _('Firefox')
  
  def getDescription(self):
    return _('Defines keyboard commands and settings to customize Firefox '
             'access.')
             
  def registerConvKeys(self):
    '''
    Registers convenience review keys.  Does not attempt to register if
    already registered
    '''
    if not self.convkeyregistered and self.perk_state.RegConvReview:
      kbd = self.getInputDevice(None, 'keyboard')
      self.registerCommand(kbd, 'firefox convenience review', False, 
                            [kbd.AEK_LEFT])
      self.registerCommand(kbd, 'firefox convenience review', False, 
                            [kbd.AEK_RIGHT])
      self.registerCommand(kbd, 'firefox convenience review', False, 
                            [kbd.AEK_UP])
      self.registerCommand(kbd, 'firefox convenience review', False, 
                            [kbd.AEK_DOWN])
                            
      modifiers = [[kbd.AEK_CONTROL_R], [kbd.AEK_CONTROL_L]]
      keys = [kbd.AEK_LEFT, kbd.AEK_RIGHT]        
      for modifier in modifiers:
        for key in keys:
          self.registerCommand(kbd, 'firefox convenience review', False,
                                modifier+[key])
      self.convkeyregistered = True
  
  def unregisterConvKeys(self):
    '''
    Unregisters convenience review keys.  Does not attempt to unregister if
    already unregistered
    '''
    if self.convkeyregistered:
      kbd = self.getInputDevice(None, 'keyboard')
      self.unregisterCommand(kbd, [kbd.AEK_LEFT])
      self.unregisterCommand(kbd, [kbd.AEK_RIGHT])
      self.unregisterCommand(kbd, [kbd.AEK_UP])
      self.unregisterCommand(kbd, [kbd.AEK_DOWN])
                          
      modifiers = [[kbd.AEK_CONTROL_R], [kbd.AEK_CONTROL_L]]
      keys = [kbd.AEK_LEFT, kbd.AEK_RIGHT]        
      for modifier in modifiers:
        for key in keys:
          self.unregisterCommand(kbd, modifier+[key])
      self.convkeyregistered = False 
      
  def getDocFrame(self, por):
    '''
    Determines if a L{POR} is within or equal to the doc frame.
    
    @param por: L{POR} in question
    @type por: L{POR}
    @return: Is por within doc frame?
    @rtype: tuple
    '''
    if por == None:
      return None
    if self.hasAccRole(DOCUMENT_ROLE, por):
      return por
    return self.findAccByRole(DOCUMENT_ROLE, por=por, ancestor=True)

  def needsArrows(self, por):
    '''
    Determines if the given L{POR} needs to receive arrow key events when it 
    has focus. Useful when deciding whether to register or unregister the
    convenience review keys.
    
    @param por: Some point of regard
    @type por: L{POR}
    @return: Will the control need the arrow keys?
    @rtype: boolean
    '''
    try:
      # check if the role matches one of ours
      state = self.SKIPPED_CONVREG_ROLES[self.getAccRole(por)]
    except KeyError:
      # if not, it doesn't need arrows
      return False
    # now check if the state value is correct or if we don't care about state
    if state is None or self.hasAccState(state, por):
      return True
    # if not, it doesn't need arrows
    return False

  def getActiveDocFrame(self):
    '''
    Returns the active document frame as a L{POR}.
    
    @return: Point of regard to the document frame accessible
    @rtype: L{POR}
    '''
    frame = self.getViewRootAcc()
    relations = self.getAccRelations('embeds', por=frame)
    # Good enough for now. Might need to change this in future if more than one
    # relation is in the relation set.
    if relations:
      return relations[0]
    else:
      return None
  
      
  def isLayoutTable(self, por, role):
    '''
    Determines if a table or cell is for layout only using Firefox's 
    suggestion in the layout-guess attribute.

    @param por: Point of regard to a potential layout table or cell
    @type por: L{POR}
    @param role: Pre-determined role of the L{POR}
    @type role: string    
    '''
    if role == 'table cell':
      table = self.getParentAcc(por=por)
      if table is None:
        return False
      attrs = self.getAccAttrs(por=table) 
      return (attrs.has_key('layout-guess') and attrs['layout-guess'] ==
              'true')
    elif role == 'table':
      attrs = self.getAccAttrs(por=por) 
      return (attrs.has_key('layout-guess') and attrs['layout-guess'] ==
              'true')
    else:
      return False
    
class HandleAutocomplete(Task.StateTask):
  '''
  Announces autocomplete menu opening
  '''
  def execute(self, por, name, value, **kwargs):
    if name == 'visible' and value == 1:
      role = self.getAccRole(por)
      if role == 'tree' or role == 'tree table':
        self.sayState(text=_('autocomplete menu opened'))
  
class ConvReviewViewReg(Task.ViewTask):
  '''
  Registers/Unregisters convenience review keys on a view event.
  '''
  def execute(self, por, **kwargs):
    if (self.perk.getDocFrame(por) and not self.perk.needsArrows(por)):
      self.perk.registerConvKeys()
    else:
      self.perk.unregisterConvKeys()
      

class ConvReviewFocusReg(Task.FocusTask):
  '''
  Registers/Unregisters convenience review keys on a focus event if pointer
  is within document frame. 
  '''
  def execute(self, por, **kwargs): 
    # checking if active doc frame is valid helps situation where a control
    # is in focus and user app switches.  A late focus on control happens 
    # after view event but active doc frame is no longer valid. 
    if (self.perk.getDocFrame(por) and not self.perk.needsArrows(por)) and \
              self.perk.getActiveDocFrame():
      self.perk.registerConvKeys()
    else:
      self.perk.unregisterConvKeys()

class ConvReviewKeys(Task.InputTask):
  '''
  Provides convenience key functionality by calling the appropriate 
  review task. 
  
  Review keys are registered/unregistered with the focus and view event 
  handlers, L{ConvReviewFocusReg} and L{ConvReviewViewReg} respectively.  They
  force the review keys to be active only within a document frame. 
  '''
  def execute(self, gesture=None, **kwargs):    
    kbd = self.getInputDevice(None, 'keyboard')

    # handle item first, most common
    if kbd.AEK_UP in gesture:
      self.doTask('review previous item')
    elif kbd.AEK_DOWN in gesture:
      self.doTask('review next item')
    # check for word before character, since it's the more specific case
    elif kbd.AEK_LEFT in gesture and \
          (kbd.AEK_CONTROL_R in gesture or kbd.AEK_CONTROL_L in gesture):
      self.doTask('review previous word')
    elif kbd.AEK_RIGHT in gesture and \
          (kbd.AEK_CONTROL_R in gesture or kbd.AEK_CONTROL_L in gesture):
      self.doTask('review next word')
    # now check character
    elif kbd.AEK_LEFT in gesture:
      self.doTask('review previous char')
    elif kbd.AEK_RIGHT in gesture:
      self.doTask('review next char')
      
class HandleDocumentPORView(Task.ViewTask):
  '''
  Returns to the last user point of regard in a document after the Firefox 
  window is reactivated.
  '''
  def executeLost(self, por, **kwargs):
    '''
    Store the pointer using it's doc frame as key.
    '''
    self.perk.viewlost = True
    doc = self.findAccByRole(DOCUMENT_ROLE, por=self.getPointerPOR(), 
                             ancestor=True)
    if doc is not None:
      self.perk.doc_pors[doc] = self.getPointerPOR()
    # clear cached pointer that is set during caret event
    self.perk.cached_pointer = None
           
class HandleDocumentPORFocus(Task.FocusTask):
  '''
  Tracks the last position reviewed in a document so that it may become the
  user's L{POR} again after focus is restored to the document.
  '''  
  def executeGained(self, por, **kwargs):
    '''
    Try to restore the pointer.  Use the current doc frame as index into
    perk.doc_pors.
    '''
    # clear cached pointer that is set during caret event and only used in 
    # executeLost.
    self.perk.cached_pointer = None
    
    # get a local copy of viewlost.  viewlost is set in view executeLost and 
    # is used for restoration after an application switch
    viewlost = self.perk.viewlost
    self.perk.viewlost = False
    
    # get a local copy of the last por that was in focus.  This is used to
    # help differentiate menu selections from tab/tab-shift events.
    lastporinfocus = self.perk.lastporinfocus
    self.perk.lastporinfocus = por
    
    # is the user tabbing or tab browsing
    currentdocinfocus = self.perk.getActiveDocFrame()
    if self.perk.lastdocinfocus is None or \
       self.perk.lastdocinfocus != currentdocinfocus:
      tabbrowsing = True
    else:
      tabbrowsing = False
    self.perk.lastdocinfocus = currentdocinfocus
    
    # try to restore the saved POR for all tab browsing events
    if tabbrowsing:
      # try to restore pointer for this doc frame
      try:
        # only restore if we are within doc frame
        if self.perk.getDocFrame(self.task_por):
          self.doTask('pointer to por',
                              por=self.perk.doc_pors[currentdocinfocus])
          # consume the next selector to block selector announcements
          self.blockNTasks(1, Task.SelectorTask)
          # register/unregister convenience keys after pointer move
          self._handleConvReviewReg(self.task_por)
          # don't propagate task chain
          return False
        else:
          return True
      except KeyError:
        # let propagated events make announcements
        return True
    # user is hitting tab/tab-shift.    
    else:
      lastrole = self.getAccRole(lastporinfocus)    
   
      # Try to restore only for certain cases that are misinterpreted
      # as a tab/tab-shift event.  This includes when a view lost event
      # occured
      if viewlost or ((lastrole == 'menu' or lastrole == 'menu item') \
                       and self.perk.getDocFrame(por)): 
        try:
          # only restore if we are within doc frame
          if self.perk.getDocFrame(self.task_por):
            self.doTask('pointer to por',
                                por=self.perk.doc_pors[currentdocinfocus])
            # consume the next selector to block selector announcements
            self.blockNTasks(1, Task.SelectorTask)
            # register/unregister convenience keys after pointer move
            self._handleConvReviewReg(self.task_por)
            # don't propagate task chain
            return False
          else:
            return True
        except KeyError:
          # let propagated events make announcements
          return True
        
  def _handleConvReviewReg(self, por):
    if self.perk.needsArrows(por):
      self.perk.unregisterConvKeys()
    else:
      self.perk.registerConvKeys()
    
  def executeLost(self, por, **kwargs):
    '''
    Store the pointer using it's container doc frame as index if both task_por and
    pointer are with doc frame.
    '''
    # get the cached pointer from caret event or the current pointer
    pointer = self.perk.cached_pointer or self.getPointerPOR()
    # If the current por is the doc frame, make sure pointer is also within
    # the doc frame.  Store it if it is.
    if self.hasAccRole(DOCUMENT_ROLE, por):
      doc = self.findAccByRole(DOCUMENT_ROLE, por=pointer, ancestor=True)
      if doc is not None:
        self.perk.doc_pors[por] = pointer
    else:
      # find doc frame of current pointer.  Store pointer if found.
      doc = self.findAccByRole(DOCUMENT_ROLE, por=pointer, ancestor=True)
      if doc is not None:
        self.perk.doc_pors[doc] = pointer 
    # clear cached pointer that is set during caret event
    self.perk.cached_pointer = None
   
    
class PointerCache(Task.CaretTask):
  '''  
  Saves pointer for document, fixes case where tabbing from page with focus in
  document frame to page with focus in URL bar. Without this fix a caret event
  switches the pointer before the focus event. Must be the
  first caret event handler.
  '''
  def execute(self, por, **kwargs):
    if self.perk.getDocFrame(self.getPointerPOR()) and \
                         not self.perk.getDocFrame(por):
      self.perk.cached_pointer = self.getPointerPOR()
      
    
class ExtraCarets(Task.CaretTask):
  '''
  Ignores extra caret events sent after selection of in-page content that
  this is not editable.
  '''
  def execute(self, por, **kwargs):
    if not self.hasAccState('editable'):
      return False

class ReadDocumentAddress(Task.InputTask):
  '''
  Reads the address of the current document.
  '''
  def execute(self, **kwargs):
    '''
    Does a search of the accessible hierarchy for the location bar, then finds
    the entry field in the bar to announce its text.
    '''
    self.stopNow()
    root = self.getViewRootAcc()
    toolbar = self.findAccByRole('tool bar', por=root, depth_first=True)
    urlbar = self.findAccByObjId('urlbar', por=toolbar)
    urlentry = self.findAccByRole('entry', por=urlbar, depth_first=True)
    self.sayItem(urlentry)
 
class FirefoxReview(Task.InputTask):
  '''
  This L{Task} is chained around all previous/next review tasks so that it can
  handle cases specific to Firefox. It provides the following features.
  
  1. Registration and unregistration of convenience review keys when reviewing
     over unfocusable and focusable elements in the document frame.
  2. Giving focus to focusable elements encountered during review while 
     respecting user settings.
  3. Bounding the review to the active document while respecting user settings.
  '''
  def _isInDoc(self, por=None):
    '''
    @return: Is the given L{POR} inside the document frame or is it the 
      document frame itself?
    @rtype: boolean
    '''
    return (self.findAccByRole(DOCUMENT_ROLE, por, ancestor=True) is not None 
            or self.hasAccRole(DOCUMENT_ROLE, por))

  def _needsFocus(self, por):
    '''
    @return: Does the given L{POR} need to be given focus according to the 
      user settings and the states of the accessible?
    @rtype: boolean
    '''
    return (self.getPerkSettingVal('AutoManipulable') and 
            self.hasAccState('focusable', por) and 
            not self.hasAccState('focused', por) and 
            not self.hasAccRole(DOCUMENT_ROLE, por))
  
  def _needsBounding(self):
    '''
    @return: Should the review be bounded to the current document frame 
      according to the user settings?
    @rtype: boolean
    '''
    return self.getPerkSettingVal('BoundReview')
  
  def execute(self, **kwargs):
    # start off assuming we're going to call the arounded review task
    original = True
    # start off assuming we're going to let other tasks execute in the chain
    propagate = True
    # check if we're inside or outside the document to start
    in_doc = self._isInDoc()
    # check whether the next/previous chunk we're reviewing is inside or 
    # outside the document
    self.doTask('%s peek' % self.getAnchorTaskId(), chain=False)
    next_por = self.getTempVal('peek')
    next_in_doc = self._isInDoc(next_por)
    
    # now determine whether we're 1) in doc to in doc, 2) out of doc to in doc,
    # 3) in doc to out of doc, or 4) out of doc to out of doc
    
    # navigating within document or into the document (1 and 2)
    if next_in_doc:
      # check if the item should get the focus
      # edge case where we just entered doc frame from chrome
      if not in_doc:
        # set the focus to doc frame to make later events easier to handle
        self.setAccFocus(self.perk.getActiveDocFrame())
      elif self._needsFocus(next_por):
        # set focus to a manipulable
        self.setAccPOR(next_por)
        # inhibit the focus announcement to follow
        self.inhibitMayStop()
      # register/unregister convenience review keys 
      if self.perk.needsArrows(next_por):
        self.perk.unregisterConvKeys()
      else:
        self.perk.registerConvKeys()
    elif in_doc and not next_in_doc:
      # navigating out of the document (3)
      # check if the user wants the review to hit a wall here
      if self._needsBounding():
        # we're at the edge of a document, set the review return value properly
        anchor = self.getAnchorTaskId()
        if anchor.startswith('review next'):
          # at the end
          self.setTempVal('review', AEConstants.REVIEW_NO_NEXT_ITEM)
        elif anchor.startswith('review previous'):
          # at the beginning
          self.setTempVal('review', AEConstants.REVIEW_NO_PREV_ITEM)
        # do not call the arounded task as we're setting the bound flags 
        # ourselves
        original = False
      else:
        # unregister review keys
        self.perk.unregisterConvKeys()
    # navigating in the chrome (4)
    # do nothing, let the review procede as normal
    if original:
      # execute the anchor, but do not let its immediate chains run too since
      # that would end up re-executing this task ad infinitum
      self.doTask(self.getAnchorTaskId(), chain=False)
      # moving caret programmatically scrolls screen and brings por into view.
      self.setAccCaret()
    return propagate

class ReadReviewRole(Task.Task):
  '''
  Task is chained around (overrides) 'read new role'. Speaks the role 
  of the given L{POR} or substitutes an output based on the role type.
  Speaks the role of the given L{POR} or the L{task_por} if no L{POR} is 
  given only if it is different than the last role spoken by this method or
  if the RepeatRole setting is True.
    
  @param por: Point of regard to the accessible whose role should be said, 
    defaults to L{task_por} if None
  @type por: L{POR}
  '''
  def execute(self, por=None, **kwargs):
    rolename = self.getAccRoleName(por) or _('link') 
    # get untranslated role name
    role = self.getAccRole(por) 
    if role == 'heading':
      attrs = self.getAccAttrs()
      if attrs.has_key('level'):
        # i18n: string is translated word 'heading', int is value 1 through 6
        outtext = _("%s level %s") % (rolename, attrs['level'])
      else:
        outtext = rolename
    else: # default case
      outtext = rolename
        
    if (self.getPerkSettingVal("RepeatRole", 'BasicSpeechPerk') or 
        outtext != self.getPerkVar("BasicSpeechPerk", "last_role")):
      if not ((role in self.perk.SKIPPED_ROLES) or
              (role == 'table cell' and self.perk.isLayoutTable(por, role))):
        self.sayRole(text=outtext)
        # only store new last role if we actually announce it
        self.setPerkVar("BasicSpeechPerk", "last_role", outtext)
        
        
class ReviewShouldSkip(Task.InputTask):
  '''
  Task is chained after 'review should skip' and provides additional rules to
  determines if the current L{POR} should be skipped or not based on the
  role of the accessible.
    
  @param por: Point of regard to test
  @type por: L{POR}
  '''
  def execute(self, peek, por=None, **kwargs):
    text = self.getItemText(por)
    if text.strip() or self.hasOneAccState(por, 'focusable', 'editable'):
      # has content or can have content from the user, don't skip
      self.setTempVal('should skip', False)
      return 
    
    skipping = self.getPerkSettingVal('Skipping', perk_name='ReviewPerk')
    if skipping == AEConstants.SKIP_ALL:
      self.setTempVal('should skip', True)
    elif skipping == AEConstants.SKIP_NONE:
      self.setTempVal('should skip', False)
    elif skipping == AEConstants.SKIP_REPORT:
      if not peek:
        role = self.getAccRole(por=por)
        # then invoke the review report task
        if role == 'table' and not self.perk.isLayoutTable(por, role):
          # update the Perk object's task_por
          self.moveToPOR(por)
          self.doTask('review skip report')
        else:
          self.moveToPOR(por)
        # indicate something has been skipped and reported
        self.setTempVal('has skipped', True)        
      else:
        # indicate something will be skipped and reported
        self.setTempVal('will skip', True)
      self.setTempVal('should skip', True)

    
class MoveToNextHeader(Task.InputTask):
  '''
  Moves pointer to the next header.
    
  @param por: Point of regard to test
  @type por: L{POR}
  '''
  def execute(self, por=None, **kwargs):
    # Search from active doc frame  if user is in chrome
    if not self.perk.getDocFrame(self.task_por):
      subtreeroot = self.perk.getActiveDocFrame()
    else: # otherwise search from current position
      subtreeroot = self.task_por
      
    self.stopNow()
    for nextpor in self.iterNextItems(wrap=True, por=subtreeroot):
      try:
        attrs = self.getAccAttrs(nextpor)
        # is it a header?  If so, move pointer, announcement is automatic.
        if attrs.has_key('tag') and len(attrs['tag']) == 2 and attrs['tag'][0] == 'H':
          # header may be an empty container so look for next item
          if self.getItemText(nextpor) == '':
            startpor = nextpor
            nextpor = self.getNextItem(por=nextpor, wrap=True)
            # if the next item is empty just revert back to header
            if self.getItemText(nextpor) == '':
              nextpor = startpor
          self.doTask('pointer to por', por=nextpor)
          # Register convenience review keys
          self.perk.registerConvKeys()
          # consider this a review step so we must update restoration por.
          docframe = self.perk.getActiveDocFrame()
          self.perk.doc_pors[docframe] = docframe
          return
      except ValueError:
        pass
    self.sayInfo(text=_('header not found'))
    
class MoveToActiveDoc(Task.InputTask):
  '''
  Moves pointer to the active document frame.
    
  @param por: Point of regard to test
  @type por: L{POR}
  '''
  def execute(self, por=None, **kwargs):
    self.stopNow()
    docframe = self.perk.getActiveDocFrame()
    # consider this a review step so we must update restoration por.
    self.perk.doc_pors[docframe] = docframe
    self.doTask('pointer to por', por=docframe)
    # Register convenience review keys
    self.perk.registerConvKeys()

class ConvReviewUserReg(Task.InputTask):
  '''
  Registers/Unregisters convenience review keys on user command.
    
  @param por: Point of regard to test
  @type por: L{POR}
  '''
  def execute(self, **kwargs):
    if self.perk_state.RegConvReview:
      self.setPerkSettingVal('RegConvReview', False)
      self.perk.unregisterConvKeys()
      self.doTask('read message', 
                  text=_('arrows review off'))
    else:
      self.setPerkSettingVal('RegConvReview', True)
      if (self.perk.getDocFrame(self.task_por)):
        self.perk.registerConvKeys()
        self.doTask('read message', 
                    text=_('arrows review on'))
      else:
        self.perk.unregisterConvKeys()
        self.doTask('read message', 
                     text=_('arrows review off'))
