/* query-field-private.h
 *
 * Copyright (C) 1999 - 2002 Vivien Malerba
 * Copyright (C) 2001 Fernando Martins
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifndef __QUERY_FIELD_PRIVATE__
#define __QUERY_FIELD_PRIVATE__

#include <gtk/gtkobject.h>
#include "conf-manager.h"
#include "database.h"

G_BEGIN_DECLS

typedef struct _QueryFieldIface QueryFieldIface;


/*
 * Description of the common interface of all the
 * QueryFields:
 *
 * - init():
 *   will initialise any private data, connect signals, etc
 *   for the QueryField type; the QueryField object has already 
 *   been initialised by the GTK object system
 *
 * - destroy():
 *   will only free any memory, signal handlers, etc from the
 *   object, not free the object itself
 *
 * - get_edit_widget():
 *   returns a widget to edit the contents of a QueryField
 *
 * - get_sel_widget():
 *   returns a widget to enable the selection of the QueryField but also
 *   of any QueryField which is being used by this one.
 *   This function MUST connect a selection event to the "func" callback and
 *   MUST set the "qf" property to the widget emitting that signal (as well as
 *   optionnaly the "pqf" or "ref" ones).
 *
 * - get_monitored_objects():
 *   returns a list (to be freed by the caller)
 *   with all the objects the QueryField receives signals from.
 *
 * - render_as_sql ():
 *   returns the SQL version of the QF, or NULL if it is NULL
 *
 * - render_as_string():
 *   will give a string describing the query field
 *   the missing_values argument can always be NULL
 *
 * - get_xml_id():
 *   will return the ID part of the XML node where the QueryField
 *   saves itself, so that other objects can reference it in their XML element.
 *
 * - replace_comp():
 *   replace a QueryField with another (it is possible to give ref=-1 or old=NULL
 *   but not the two at the same time)
 *
 */
struct _QueryFieldIface {
	enum _QueryFieldType field_type;
	gchar               *name; /* the name of the 'type' attribute in the XML file */
	gchar               *pretty_name; /* A name understandable by the user */
	void               (*init)            (QueryField *qf);
	void               (*destroy)         (QueryField *qf);
	void               (*deactivate)      (QueryField *qf);
	void               (*activate)        (QueryField *qf);
	GtkWidget *        (*get_edit_widget) (QueryField *qf);
	GtkWidget *        (*get_sel_widget)  (QueryField *qf, QueryField *sel_qf, 
					GCallback callback, gpointer data);
	gchar     *        (*render_as_sql)   (QueryField *qf, GSList *missing_values);
	xmlNodePtr         (*render_as_xml)   (QueryField *qf, GSList *missing_values);
	gchar     *        (*render_as_string)(QueryField *qf, GSList *missing_values);
	void               (*save_to_xml)     (QueryField *qf, xmlNodePtr node);
	void               (*load_from_xml)   (QueryField *qf, xmlNodePtr node);
	void               (*copy_other_field)(QueryField *qf, QueryField *other);
	gboolean           (*is_equal_to)     (QueryField *qf, QueryField *other);
	GSList    *        (*get_monitored_objects) (QueryField *qf);
	void               (*replace_comp)    (QueryField *qf, gint ref, GObject   *old, GObject   *new);
};

struct _QueryFieldPrivate {
	Query         *query;
	gboolean       is_printed;   /* TRUE if it appears in the resulting columns */
	gchar         *name;
	gchar         *alias;
        gboolean       is_hidden;    /* TRUE if it is printed but has been added by Mergeant itself */
	QueryFieldType field_type;
		
	guint          id; /* Id unique only inside the Query */
		
	/* TRUE if the object has references to all the objects
	   it wants to receive events from */
	gboolean       activated;
		
	/* private data which can be used by the different query field types */
	gpointer       private_data;
	guint          ref_counter;
};



/* Private stuff for the QueryField object */
GObject          *query_field_new                 (Query *q, gchar * name, QueryFieldType field_type);

QueryFieldIface  *query_field_field_get_iface     ();
QueryFieldIface  *query_field_allfields_get_iface ();
QueryFieldIface  *query_field_value_get_iface     ();
QueryFieldIface  *query_field_function_get_iface  ();
QueryFieldIface  *query_field_query_get_iface     ();

void              query_field_set_activated       (QueryField *qf, gboolean activated);

G_END_DECLS

#endif
