/* serveraccess.h
 *
 * Copyright (C) 1999 - 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


/*
 * Implementation of the objects here is in two files:
 *    serveraccess.c
 *    sqldata.c
 */

#ifndef __SERVER_ACCESS__
#define __SERVER_ACCESS__

#include <config.h>
#include <gnome.h>
#include "server-rs.h"
#include <libgda/gda-connection.h>
#include <libgda/gda-command.h>
#include "data-handler.h"
#include "mergeant_xml.h"

G_BEGIN_DECLS

typedef struct _ServerAccess ServerAccess;
typedef struct _ServerAccessClass ServerAccessClass;
typedef struct _ServerAccessItemType ServerAccessItemType;

typedef struct _ServerDataType ServerDataType;
typedef struct _ServerDataTypeClass ServerDataTypeClass;

typedef struct _ServerFunction ServerFunction;
typedef struct _ServerFunctionClass ServerFunctionClass;

typedef struct _SqlDataOperator SqlDataOperator;
typedef struct _SqlDataOperatorClass SqlDataOperatorClass;

typedef struct _ServerAggregate ServerAggregate;
typedef struct _ServerAggregateClass ServerAggregateClass;

/*
 * 
 * ServerAccess object
 *
 */
#define SERVER_ACCESS_TYPE          (server_access_get_type())
#define SERVER_ACCESS(obj)          G_TYPE_CHECK_INSTANCE_CAST (obj, server_access_get_type(), ServerAccess)
#define SERVER_ACCESS_CLASS(klass)  G_TYPE_CHECK_CLASS_CAST (klass, server_access_get_type (), ServerAccessClass)
#define IS_SERVER_ACCESS(obj)       G_TYPE_CHECK_INSTANCE_TYPE (obj, server_access_get_type ())

typedef enum {
	SERVER_ACCESS_QUERY_SQL,
	SERVER_ACCESS_QUERY_XML
}
ServerAccessQueryType;

typedef struct
{
	gboolean sequences;
	gboolean procs;
	gboolean inheritance;
	gboolean xml_queries;
}
GdaConnection_Supports;

/* struct for the object's data */
struct _ServerAccess
{
	GdaClient               object;
	
	/* server description */
	gchar                  *description;
	GdaConnection_Supports  features;
	GdaConnection          *cnc;
	
	/* SQL connectivity */
	GString                *gda_datasource;
	GString                *user_name;
	GString                *password;
	
	/* Data types and stuff */
	GSList                 *data_types;
	GSList                 *data_functions;
	GSList                 *data_aggregates;
	GSList                 *data_types_handlers;	/* list of DataHandler objects */
	GHashTable             *types_objects_hash;	/* hash to store the bindings of
							   DataHandler objects to several objects */
	GSList                 *weak_ref_objects;
	GSList                 *bindable_objects;	/* list the functions called by 
							   server_access_get_object_data_handler() */
};

/* struct for the object's class */
struct _ServerAccessClass
{
	GdaClientClass                    parent_class;
	
	void (*conn_opened)              (ServerAccess * srv);
	void (*conn_to_close)            (ServerAccess * srv);	/* connexion about to be closed */
	void (*conn_closed)              (ServerAccess * srv);
	/* data types, functions, operators and aggregates signals */
	void (*data_type_added)          (ServerAccess * srv, ServerDataType *dtype);
	void (*data_type_removed)        (ServerAccess * srv, ServerDataType *dtype);
	void (*data_types_updated)       (ServerAccess * srv);
	void (*data_function_added)      (ServerAccess * srv, ServerFunction * func);
	void (*data_function_removed)    (ServerAccess * srv, ServerFunction * func);
	void (*data_functions_updated)   (ServerAccess * srv);
	void (*data_aggregate_added)     (ServerAccess * srv, ServerAggregate * func);
	void (*data_aggregate_removed)   (ServerAccess * srv, ServerAggregate * func);
	void (*data_aggregates_updated)  (ServerAccess * srv);
	void (*progress)                 (ServerAccess * srv, gchar * msg, guint now, guint total);
	/* for the objects to plugins bindings */
	void (*objects_bindings_updated) (ServerAccess * srv);
};

/*
 * generic object's functions
 */
GType              server_access_get_type                 (void);
ServerAccess      *server_access_new                      (void);

/*
 * open/close and query functions
 */
void               server_access_open_connect             (ServerAccess * srv);
gboolean           server_access_is_open                  (ServerAccess * srv);
void               server_access_close_connect            (ServerAccess * srv);
void               server_access_close_connect_no_warning (ServerAccess * srv);
ServerResultset   *server_access_do_query                 (ServerAccess * srv, const gchar * query, 
							   ServerAccessQueryType type);

/*
 * data type, etc lookup and management
 */
void               server_access_refresh_datas            (ServerAccess * srv);
const GList       *server_access_get_data_type_list       (ServerAccess * srv);	/* to free */
ServerDataType    *server_access_get_type_from_name       (ServerAccess * srv, const gchar * name);

/*
 * XML functions
 */
void               server_access_build_xml_tree           (ServerAccess * srv, xmlDocPtr doc);
gboolean           server_access_build_from_xml_tree      (ServerAccess * srv, xmlNodePtr tree);

/*
 * plugins management
 */
DataHandler       *server_access_get_handler_from_name    (ServerAccess * srv, const gchar *plugin_name);
DataHandler       *server_access_get_handler_from_type    (ServerAccess * srv, ServerDataType * dt);
DataHandler       *server_access_get_handler_from_gda     (ServerAccess * srv, GdaValueType gda_type);
/* returns the display functions associated with an object 
   (ServerDataType objects get their onw default data functions in case 
   they are non overloaded) Cannot return NULL! */
DataHandler       *server_access_get_object_handler       (ServerAccess * srv, GObject * obj);
void               server_access_bind_object_handler      (ServerAccess * srv, GObject * obj,
							   DataHandler *dh);
gboolean           server_access_is_object_bound          (ServerAccess * srv, GObject * obj);
void               server_access_unbind_object_handler    (ServerAccess * srv, GObject * obj);
void               server_access_rescan_display_plugins   (ServerAccess * srv, gchar * path);
/* this function allows modules to give another object to look for in the 
   hash table, given an object. For example if the object is a 
   DbTableField, then the 'func' function will return the 
   data type of the field (this is set by the Database module) */
void               server_access_declare_object_bindable  (ServerAccess * srv, gpointer (*func) (GObject *));











/*
 *
 * ServerDataType object
 *
 */

#define SERVER_DATA_TYPE_TYPE          (server_data_type_get_type())
#define SERVER_DATA_TYPE(obj)          G_TYPE_CHECK_INSTANCE_CAST (obj, server_data_type_get_type(), ServerDataType)
#define SERVER_DATA_TYPE_CLASS(klass)  G_TYPE_CHECK_CLASS_CAST (klass, server_data_type_get_type (), ServerDataTypeClass)
#define IS_SERVER_DATA_TYPE(obj)       G_TYPE_CHECK_INSTANCE_TYPE (obj, server_data_type_get_type ())


/* struct for the object's data */
struct _ServerDataType
{
	GObject            object;

	gchar             *descr;
	gchar             *sqlname;
	guint              numparams;
	GdaValueType       gda_type;

	/* Types display implementation */
	DataHandler       *data_handler;
	gboolean           updated;
};

/* struct for the object's class */
struct _ServerDataTypeClass
{
	GObjectClass       parent_class;
};

/*
 * generic object's functions
 */
GType           server_data_type_get_type             (void);
GObject        *server_data_type_new                  (void);

/*
 * data types management
 */
void            server_data_type_set_sqlname          (ServerDataType * dt, gchar * name);
void            server_data_type_set_descr            (ServerDataType * dt, gchar * name);
void            server_data_type_update_list          (struct _ServerAccess *srv);
#ifdef debug
void            server_data_type_show_types           (GSList * dtl);
#endif

/* 
 * data types lookup
 */
ServerDataType *server_data_type_get_from_name        (GSList * dtl, const gchar * name);
GList          *server_data_type_get_name_list        (GSList * dtl);


/*
 * XML manipulations
 */
gchar          *server_data_type_get_xml_id           (ServerDataType * dt);
ServerDataType *server_data_type_get_from_xml_id      (ServerAccess *srv, const gchar * id);



/*
 *
 * ServerFunction object
 *
 */
#define SERVER_FUNCTION_TYPE          (server_function_get_type())
#define SERVER_FUNCTION(obj)          G_TYPE_CHECK_INSTANCE_CAST (obj, server_function_get_type(), ServerFunction)
#define SERVER_FUNCTION_CLASS(klass)  G_TYPE_CHECK_CLASS_CAST (klass, server_function_get_type (), ServerFunctionClass)
#define IS_SERVER_FUNCTION(obj)       G_TYPE_CHECK_INSTANCE_TYPE (obj, server_function_get_type ())

/* struct for the object's data */
struct _ServerFunction
{
	GObject         object;

	gchar          *descr;
	gchar          *sqlname;
	gchar          *objectid;	/* unique id for the function */
	ServerDataType *result_type;
	GSList         *args;	/* ServerDataType list */
	gboolean        is_user;	/* TRUE  if user defined function */
	gboolean        updated;
};

/* struct for the object's class */
struct _ServerFunctionClass
{
	GObjectClass    parent_class;
};

/*
 * generic object's functions 
 */
GType           server_function_get_type (void);
GObject        *server_function_new (void);

/*
 * function management
 */
void            server_function_set_sqlname        (ServerFunction * df, gchar * name);
void            server_function_set_descr          (ServerFunction * df, gchar * name);
void            server_function_update_list        (struct _ServerAccess *srv);
#ifdef debug
void            server_function_show_functions     (GSList * dfl);
#endif

/* 
 * functions lookup
 */
GSList         *server_function_get_list_from_name (GSList * dfl, gchar * name);
/* argtypes is a list of ServerDataTypes in the right order */
ServerFunction *server_function_get_from_name      (GSList * dfl, gchar * name,
						    GSList * argtypes);
ServerFunction *server_function_get_from_objid     (GSList * dfl, gchar * id);


/*
 * XML manipulations
 */
gchar          *server_function_get_xml_id         (ServerFunction * df);
ServerFunction *server_function_get_from_xml_id    (ServerAccess *srv, const gchar * id);

/* function to bind for data types and plugins usage */
gpointer        server_function_binding_func       (GObject * obj);








/*
 *
 * ServerAggregate object
 *
 */
#define SERVER_AGGREGATE_TYPE          (server_aggregate_get_type())
#define SERVER_AGGREGATE(obj)          G_TYPE_CHECK_INSTANCE_CAST (obj, server_aggregate_get_type(), ServerAggregate)
#define SERVER_AGGREGATE_CLASS(klass)  G_TYPE_CHECK_CLASS_CAST (klass, server_aggregate_get_type (), ServerAggregateClass)
#define IS_SERVER_AGGREGATE(obj)       G_TYPE_CHECK_INSTANCE_TYPE (obj, server_aggregate_get_type ())


/* struct for the object's data */
struct _ServerAggregate
{
	GObject         object;

	gchar          *descr;
	gchar          *sqlname;
	gchar          *objectid;
	ServerDataType *arg_type;
	gboolean        updated;
};

/* struct for the object's class */
struct _ServerAggregateClass
{
	GObjectClass    parent_class;
};

/*
 * generic widget's aggregates 
 */
GType            server_aggregate_get_type        (void);
GObject         *server_aggregate_new             (void);

/*
 * aggregate management
 */
void             server_aggregate_set_sqlname     (ServerAggregate * da, gchar * name);
void             server_aggregate_set_descr       (ServerAggregate * da, gchar * name);
void             server_aggregate_update_list     (ServerAccess *srv);
#ifdef debug
void             server_aggregate_show_aggregates (GSList * dal);
#endif


/*
 * aggregates lookup
 */
ServerAggregate *server_aggregate_get_from_name   (GSList * dal, gchar * name,
						   ServerDataType * arg_type);
ServerAggregate *server_aggregate_get_from_objid  (GSList * dal, gchar * oid);

/*
 * XML manipulations
 */
gchar           *server_aggregate_get_xml_id      (ServerAggregate * da);
ServerAggregate *server_aggregate_get_from_xml_id (ServerAccess *srv, const gchar * id);

G_END_DECLS

#ifdef debug
#define D_COL_NOR "\033[0m"
#define D_COL_H0 "\033[;34;7m"
#define D_COL_H1 "\033[;36;7m"
#define D_COL_H2 "\033[;36;4m"
#define D_COL_OK "\033[;32m"
#define D_COL_ERR "\033[;31;1m"
#define AAA(X) g_print (D_COL_H1 "DEBUG MARK %d\n" D_COL_NOR, X)
#endif

#endif
