/*
 *
 * mtm-config-gui:
 *
 * Copyright (C) 2001 Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Richard Hestilow <tvgm@ximian.com>
 *
 */

#include <mtm-config-gui.h>
#include <gtk/gtksignal.h>

static GtkObjectClass *mtm_config_gui_parent_class;

struct _MtmConfigGuiPrivate
{
	gint ext_changed;
};

enum
{
	ARG_0,
	ARG_EXT,
	ARG_FILE_LIST,
	ARG_CONFIG_AREA
};

enum
{
	SET_EXT,
	SET_CONFIG_AREA,
	SET_FILE_LIST,
	EXT_MODIFIED,
	LAST_SIGNAL
};

guint mtm_config_gui_signals[LAST_SIGNAL] = {0 };

static void mtm_config_gui_set_arg (GtkObject *object, GtkArg *arg,
				    guint arg_id);
static void mtm_config_gui_get_arg (GtkObject *object, GtkArg *arg,
				    guint arg_id);

static gchar* mtm_config_gui_browse (MtmConfigGui *gui);
static void mtm_config_gui_ext_changed (MtmConfigGui *gui, MtmExt *ext);

static gchar** mtm_strdupv (const gchar** list);

/* Destroy handler for MtmConfigGui */
static void
mtm_config_gui_destroy (GtkObject *object)
{
	MtmConfigGui *gui = MTM_CONFIG_GUI (object);
	
	if (gui->file_list)
		g_strfreev (gui->file_list);
	
	if (gui->ext)
	{
		if (gui->priv->ext_changed != -1)
			gtk_signal_disconnect (GTK_OBJECT (gui->ext),
					       gui->priv->ext_changed);
		gtk_object_unref (GTK_OBJECT (gui->ext));
	}

	g_free (gui->priv);

	if (GTK_OBJECT_CLASS (mtm_config_gui_parent_class)->destroy)
		(*GTK_OBJECT_CLASS (mtm_config_gui_parent_class)->destroy)(object);
}

/* Class initialization function for MtmConfigGui */
static void
mtm_config_gui_class_init (GtkObjectClass *object_class)
{
	mtm_config_gui_parent_class = gtk_type_class (gtk_object_get_type ());

	MTM_CONFIG_GUI_CLASS (object_class)->browse = mtm_config_gui_browse;

	mtm_config_gui_signals[SET_EXT] =
		gtk_signal_new ("set_ext",
			GTK_RUN_LAST, object_class->type,
			GTK_SIGNAL_OFFSET (MtmConfigGuiClass, set_ext),
			gtk_marshal_NONE__OBJECT, GTK_TYPE_NONE, 1,
			MTM_EXT_TYPE);

	mtm_config_gui_signals[SET_CONFIG_AREA] =
		gtk_signal_new ("set_config_area",
			GTK_RUN_LAST, object_class->type,
			GTK_SIGNAL_OFFSET (MtmConfigGuiClass, set_config_area),
			gtk_marshal_NONE__OBJECT, GTK_TYPE_NONE, 1,
			GTK_TYPE_WIDGET);

	mtm_config_gui_signals[SET_FILE_LIST] =
		gtk_signal_new ("set_file_list",
			GTK_RUN_LAST, object_class->type,
			GTK_SIGNAL_OFFSET (MtmConfigGuiClass, set_file_list),
			gtk_marshal_NONE__OBJECT, GTK_TYPE_NONE, 1,
			GTK_TYPE_POINTER);

	mtm_config_gui_signals[EXT_MODIFIED] =
		gtk_signal_new ("ext_modified",
			GTK_RUN_LAST, object_class->type,
			GTK_SIGNAL_OFFSET (MtmConfigGuiClass, ext_modified),
			gtk_marshal_NONE__OBJECT, GTK_TYPE_NONE, 1,
			MTM_EXT_TYPE);

	gtk_object_class_add_signals (object_class, mtm_config_gui_signals,
				      LAST_SIGNAL);

	object_class->set_arg = mtm_config_gui_set_arg;
	object_class->get_arg = mtm_config_gui_get_arg;

	gtk_object_add_arg_type ("MtmConfigGui::ext", MTM_EXT_TYPE,
		GTK_ARG_READWRITE, ARG_EXT);
	gtk_object_add_arg_type ("MtmConfigGui::file_list", GTK_TYPE_POINTER,
		GTK_ARG_READWRITE, ARG_FILE_LIST);
	gtk_object_add_arg_type ("MtmConfigGui::config_area", GTK_TYPE_WIDGET,
		GTK_ARG_READWRITE, ARG_CONFIG_AREA);
	
	object_class->destroy = mtm_config_gui_destroy;
}

static void
mtm_config_gui_init (GtkObject *object)
{
	MtmConfigGui *gui = MTM_CONFIG_GUI (object);
	
	gui->priv = g_new0 (MtmConfigGuiPrivate, 1);

	gui->config_area = NULL;	
	gui->file_list = NULL;
	gui->priv->ext_changed = -1;
}


/**
 * mtm_config_gui_get_type:
 * @void:
 *
 * Registers the #MtmConfigGui class if necessary, and returns the type ID
 * associated to it.
 *
 * Return value: The type ID of the #MtmConfigGui class.
 */
GtkType
mtm_config_gui_get_type (void)
{
	static GtkType type = 0;

	if (!type)
	{
		GtkTypeInfo info =
		{
			"MtmConfigGui",
			sizeof (MtmConfigGui),
			sizeof (MtmConfigGuiClass),
			(GtkClassInitFunc) mtm_config_gui_class_init,
			(GtkObjectInitFunc) mtm_config_gui_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (gtk_object_get_type (), &info);
	}

	return type;
}

/**
 * mtm_config_gui_new:
 * @void:
 * 
 * Creates a new ext configuration GUI.
 *
 * Return value: A newly created ext configuration GUI
 */
MtmConfigGui*
mtm_config_gui_new (void)
{
	MtmConfigGui *gui;

	gui = gtk_type_new (mtm_config_gui_get_type ());
	return gui;
}


/* g_strdup over a NULL-terminated list...counterpart to g_strfreev */
static gchar**
mtm_strdupv (const gchar** list)
{
	gchar **ret;
	int i, len;
	
	g_return_val_if_fail (list != NULL, NULL);
	
	for (len = 0; list[len] != NULL; len++)
	{
	}

	ret = g_new0 (gchar*, len + 1);
	
	for (i = 0; i < len; i++)
		ret[i] = g_strdup (list[i]);

	return ret;
}

/**
 * mtm_config_gui_get_file_list:
 * @gui: The configuration gui
 *
 * Queries the associated list of possible files.
 *
 * Return value: The null-terminated list of files.
 **/
gchar**
mtm_config_gui_get_file_list (MtmConfigGui *gui)
{
	g_return_val_if_fail (MTM_IS_CONFIG_GUI (gui), NULL);

	return gui->file_list;
}

/**
 * mtm_config_gui_set_file_list:
 * @gui: The configuration list
 * @file_list: The new list of possible files (null-terminated)
 *
 * Sets the associated list of possible files to the one provided.
 **/
void
mtm_config_gui_set_file_list (MtmConfigGui *gui, const gchar **file_list)
{
	gchar **new_file_list = NULL;

	g_return_if_fail (MTM_IS_CONFIG_GUI (gui));
	
	if (gui->file_list)
		g_strfreev (gui->file_list);
	
	if (file_list)
		new_file_list = mtm_strdupv (file_list);
	
	gtk_signal_emit (GTK_OBJECT (gui), mtm_config_gui_signals[SET_FILE_LIST], new_file_list);
	
	gui->file_list = new_file_list;
}

void
mtm_config_gui_set_ext (MtmConfigGui *gui, MtmExt *ext)
{
	g_return_if_fail (MTM_IS_CONFIG_GUI (gui));
	g_return_if_fail (MTM_IS_EXT (ext));
	
	/* We disconnect before emitting so as to prevent confusion in
	 * set_ext signal handlers */
	if (gui->ext && gui->priv->ext_changed != -1)
			gtk_signal_disconnect (GTK_OBJECT (gui->ext),
					       gui->priv->ext_changed);

	/* We emit first in case any objects need to know the old object */
	gtk_signal_emit (GTK_OBJECT (gui),
		mtm_config_gui_signals[SET_EXT],
		ext);

	if (gui->ext)
		gtk_object_unref (GTK_OBJECT (gui->ext));

	gui->ext = ext;
	gtk_object_ref (GTK_OBJECT (ext));
	gtk_object_sink (GTK_OBJECT (ext));
	gui->priv->ext_changed = gtk_signal_connect_object (
					GTK_OBJECT (gui->ext),
					"changed",
					mtm_config_gui_ext_changed,
					GTK_OBJECT (gui));
}

const MtmExt*
mtm_config_gui_get_ext (MtmConfigGui *gui)
{
	g_return_val_if_fail (MTM_IS_CONFIG_GUI (gui), NULL);

	return gui->ext;
}

static void
mtm_config_gui_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	MtmConfigGui *gui = MTM_CONFIG_GUI (object);

	switch (arg_id)
	{
		case ARG_EXT:
			mtm_config_gui_set_ext (gui, MTM_EXT (GTK_VALUE_OBJECT (*arg)));
			break;
		case ARG_FILE_LIST:
			mtm_config_gui_set_file_list (gui, GTK_VALUE_POINTER (*arg));
			break;
		case ARG_CONFIG_AREA:
			mtm_config_gui_set_config_area (gui, GTK_WIDGET (GTK_VALUE_OBJECT (*arg)));
	}
}

static void
mtm_config_gui_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	MtmConfigGui *gui = MTM_CONFIG_GUI (object);

	switch (arg_id)
	{
		case ARG_EXT:
			GTK_VALUE_OBJECT (*arg) = GTK_OBJECT (gui->ext);
			break;
		case ARG_FILE_LIST:
			GTK_VALUE_POINTER (*arg) = gui->file_list;
			break;
		case ARG_CONFIG_AREA:
			GTK_VALUE_POINTER (*arg) = GTK_OBJECT (gui->config_area);
			break;
	}
}

static void
mtm_config_gui_ext_changed (MtmConfigGui *gui, MtmExt *ext)
{
	g_return_if_fail (MTM_IS_CONFIG_GUI (gui));
	
	gtk_signal_emit (GTK_OBJECT (gui),
		mtm_config_gui_signals[EXT_MODIFIED],
		ext);
}

static gchar* 
mtm_config_gui_browse (MtmConfigGui *gui)
{
	g_return_val_if_fail (MTM_IS_CONFIG_GUI (gui), NULL);
	
	return NULL;
}

GtkWidget*
mtm_config_gui_get_config_area (MtmConfigGui *gui)
{
	g_return_val_if_fail (MTM_IS_CONFIG_GUI (gui), NULL);
	
	return gui->config_area;
}

void
mtm_config_gui_set_config_area (MtmConfigGui *gui, GtkWidget *config_area)
{
	g_return_if_fail (MTM_IS_CONFIG_GUI (gui));
	g_return_if_fail (GTK_IS_WIDGET (config_area));

	gtk_signal_emit (GTK_OBJECT (gui), mtm_config_gui_signals[SET_CONFIG_AREA], config_area);
	if (gui->config_area)
		gtk_object_unref (GTK_OBJECT (gui->config_area));

	gui->config_area = config_area;
	gtk_object_ref (GTK_OBJECT (config_area));
	gtk_object_sink (GTK_OBJECT (config_area));
}

