/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <gnome.h>

#include <pan/base/acache.h>
#include <pan/base/article-thread.h>
#include <pan/base/article.h>
#include <pan/base/debug.h>
#include <pan/base/file-headers.h>
#include <pan/base/fnmatch.h>
#include <pan/base/group.h>
#include <pan/base/log.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/status-item.h>
#include <pan/base/util-file.h>

#include <pan/article-actions.h>
#include <pan/article-filter-ui.h>
#include <pan/articlelist.h>
#include <pan/article-toolbar.h>
#include <pan/dialogs/dialogs.h>
#include <pan/grouplist.h>
#include <pan/globals.h>
#include <pan/gui.h>
#include <pan/gui-paned.h>
#include <pan/message-window.h>
#include <pan/prefs.h>
#include <pan/queue.h>
#include <pan/save.h>
#include <pan/task-body.h>
#include <pan/task-bodies.h>
#include <pan/task-decode.h>
#include <pan/task-headers.h>
#include <pan/thread-watch.h>
#include <pan/text.h>
#include <pan/util.h>

#include <pan/xpm/old_read.xpm>
#include <pan/xpm/new_read.xpm>
#include <pan/xpm/old_unread.xpm>
#include <pan/xpm/new_unread.xpm>
#include <pan/xpm/bluecheck.xpm>
#include <pan/xpm/disk.xpm>
#include <pan/xpm/flagged.xpm>
#include <pan/xpm/x.xpm>
#include <pan/xpm/binary.xpm>
#include <pan/xpm/binary_complete.xpm>
#include <pan/xpm/binary_incomplete.xpm>
#include <pan/xpm/old_binary_complete.xpm>
#include <pan/xpm/old_binary_incomplete.xpm>
#include <pan/xpm/copy_message.xpm>
#include <pan/xpm/move_message.xpm>
#include <pan/xpm/watched.xpm>

/*ccc*/
#include <pan/filters/filter-xml.h>
#include <pan/filters/filter-top.h>

/***
****
****
****  Abandon hope, all ye who enter here
****
****
***/



/**********************************************************
 * Private Variables
 **********************************************************/

#define COLUMNS 6

/* state */
static Group * my_group                          = NULL;
static Filter * my_filter                        = NULL;
static FilterShow my_filter_show                 = FILTER_SHOW_MATCHES_AND_REFERENCES;
static GHashTable * messageid_to_node            = NULL;
static gboolean is_threaded                      = TRUE;
static int sort_type                             = ARTICLE_SORT_SUBJECT;
static pthread_mutex_t article_ctree_lock        = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t group_mutex               = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t hash_mutex                = PTHREAD_MUTEX_INITIALIZER;

/* article list */
static GnomePixmap * old_read_pixmap = NULL;
static GnomePixmap * old_unread_pixmap = NULL;
static GnomePixmap * new_read_pixmap = NULL;
static GnomePixmap * new_unread_pixmap = NULL;

static GnomePixmap * article_cached_pixmap       = NULL;
static GnomePixmap * articlelist_closed_pixmap   = NULL;
static GnomePixmap * articlelist_open_pixmap     = NULL;
static GnomePixmap * binary_decoded_pixmap       = NULL;
static GnomePixmap * decode_queue_pixmap         = NULL;
static GnomePixmap * error_pixmap                = NULL;
static GnomePixmap * flagged_for_download_pixmap = NULL;
static GnomePixmap * mp_complete_pixmap          = NULL;
static GnomePixmap * mp_incomplete_pixmap        = NULL;
static GnomePixmap * old_mp_complete_pixmap      = NULL;
static GnomePixmap * old_mp_incomplete_pixmap    = NULL;

static void tree_select_row_cb (GtkCTree*, GtkCTreeNode*, gint, gpointer);

PanCallback * articlelist_sort_changed           = NULL;
PanCallback * articlelist_thread_changed         = NULL;
PanCallback * articlelist_selection_changed      = NULL;

static GtkStyle * normal_style[2] = { NULL, NULL };
static GtkStyle * read_style[2] = { NULL, NULL };
static GtkStyle * watched_style[2] = { NULL, NULL };
static GtkStyle * killfile_style[2] = { NULL, NULL };

GdkColor killfile_color;
GdkColor watched_color;
GdkColor thread_normal_color;
GdkColor thread_read_color;

GtkWidget * article_ctree_menu;


/**********************************************************
 * Private Functions
 **********************************************************/

static void
fire_group_changed (Group * g);

static void
articlelist_menu_popup (GdkEventButton* bevent);
static void
articlelist_unread_inc_nolock (GtkCTreeNode * child, int inc);

static void
articlelist_repopulate (const Group* group,
                        Article** article_buf,
                       	int article_qty,
                        StatusItem *item,
                        GPtrArray * sel_articles);

static GtkCTreeNode* articlelist_get_node_from_message_id (
	const char* message_id);

static GtkCTreeNode*
add_article_to_ctree (GtkCTree      * tree,
		      GtkCTreeNode  * parent,
		      GtkCTreeNode  * sibling,
		      Article       * article,
		      StatusItem    * item,
		      gboolean        expanded);

static void articlelist_update_node_impl (const Article* article,
                                          gboolean need_lock);

static void articlelist_click_column (GtkCList*, int col_clicked);

static void article_ctree_destroy_cb (void);

static void articlelist_update_node_fast_nolock (GtkCTreeNode*, const Article*);

static void articlelist_selected_delete_attachment (void);

static void articlelist_save_selected_article_text (void);

static void articlelist_spamize (void);

/***
****
****
****  POPUP MENUS
****
****
***/


static GnomeUIInfo save_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("_Save Article Text..."),
		N_("Save the article headers and body to a text file."),
		articlelist_save_selected_article_text, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
		'S', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Save _Binary Attachment"),
		N_("Save Binary Attachment"),
		articlelist_selected_decode, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
		'S', GDK_SHIFT_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Save Binary Attachment _As..."),
		N_("Save Binary Attachment As..."),
		articlelist_selected_decode_as, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
		'S', GDK_CONTROL_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("_Open Binary Attachment"),
		N_("Open Binary Attachment"),
		articlelist_selected_open, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN,
		'O', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("_Delete Decoded Binary Attachment"),
		N_("Delete Decoded Binary Attachment"),
		articlelist_selected_delete_attachment, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
		'\0', 0, NULL
	},
	GNOMEUIINFO_END
};


static GnomeUIInfo delete_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Selected _Articles"),
		N_("Selected Articles"),
		articlelist_selected_delete, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
		0, 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Selected Articles and _Replies"),
		N_("Selected Articles and Replies"),
		articlelist_selected_delete_thread, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
		0, 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("All Articles in _Group"),
		N_("All Articles in Group"),
		articlelist_all_delete, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
		0, 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("_Delete Decoded Binary Attachment"),
		N_("Delete Decoded Binary Attachment"),
		articlelist_selected_delete_attachment, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
		0, 0, NULL
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo folder_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("_Edit Selected Article"),
		N_("Edit Selected Article"),
		articlelist_selected_edit
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("_Copy Selected Articles"),
		N_("Copy Selected Articles"),
		articlelist_selected_copy_to_folder, NULL,
		NULL,
		GNOME_APP_PIXMAP_DATA, copy_message_xpm,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Copy Selected Articles and _Replies"),
		N_("Copy Selected Articles and Replies"),
		articlelist_selected_copy_to_folder_deep, NULL,
		NULL,
		GNOME_APP_PIXMAP_DATA, copy_message_xpm,
		'\0', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("_Move Selected Articles"),
		N_("Move Selected Articles"),
		articlelist_selected_move_to_folder, NULL,
		NULL,
		GNOME_APP_PIXMAP_DATA, move_message_xpm,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Move Selected Articles and Re_plies"),
		N_("Move Selected Articles and Replies"),
		articlelist_selected_move_to_folder_deep, NULL,
		NULL,
		GNOME_APP_PIXMAP_DATA, move_message_xpm,
		'\0', 0, NULL
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo mark_read_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Selected _Articles"),
		N_("Selected Articles"),
		articlelist_selected_mark_read, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'M', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Selected Articles and _Replies"),
		N_("Selected Articles and Replies"),
		articlelist_selected_mark_read_deep, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'M', GDK_SHIFT_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Selected _Threads"),
		N_("Selected Threads"),
		articlelist_selected_mark_read_thread, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'M', GDK_MOD1_MASK, NULL
	},
        {
		GNOME_APP_UI_ITEM,
		N_("Entire _Group"),
		N_("Entire Group"),
		articlelist_all_mark_read, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'M', GDK_SHIFT_MASK|GDK_CONTROL_MASK, NULL,
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo mark_unread_menu[] =
{

	{
		GNOME_APP_UI_ITEM,
		N_("Selected _Articles"),
		N_("Selected Articles"),
		articlelist_selected_mark_unread
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Selected Articles and _Replies"),
		N_("Selected Articles and Replies"),
		articlelist_selected_mark_unread_deep
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Selected _Threads"),
		N_("Selected Threads"),
		articlelist_selected_mark_unread_thread,
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Entire _Group"),
		N_("Entire Group"),
		articlelist_all_mark_unread,
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo purge_bodies_menu[] =
{
        {
		GNOME_APP_UI_ITEM,
		N_("Selected _Articles"),
		N_("Selected Articles"),
		articlelist_selected_purge_bodies
	},                                                                      
	{
		GNOME_APP_UI_ITEM,
		N_("Selected Articles and _Replies"),
		N_("Selected Articles and Replies"),
		articlelist_selected_purge_bodies_deep
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Entire _Group"),
		N_("Entire Group"),
		articlelist_all_purge_bodies,
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo bodies_menu[] =
{
        {
		GNOME_APP_UI_ITEM,
		N_("Selected _Articles"),
		N_("Selected Articles"),
		articlelist_selected_download, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_RCV,
		'\0', 0, NULL
	},                                                                      
	{
		GNOME_APP_UI_ITEM,
		N_("Selected Articles and _Replies"),
		N_("Selected Articles and Replies"),
		articlelist_selected_thread_download, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_RCV,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Entire _Group"),
		N_("Entire Group"),
		articlelist_all_download, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_RCV,
		'\0', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("_Flag Selected Articles"),
		N_("Flag Selected Articles"),
		articlelist_selected_flag_for_dl, NULL,
		NULL,
		GNOME_APP_PIXMAP_DATA, flagged_xpm,
		'J', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Fla_g Selected Articles and Replies"),
		N_("Flag Selected Articles and Replies"),
		articlelist_selected_flag_for_dl_deep, NULL,
		NULL,
		GNOME_APP_PIXMAP_DATA, flagged_xpm,
		'J', GDK_SHIFT_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("_Unflag Selected Articles"),
		N_("Unflag Selected Articles"),
		articlelist_selected_unflag_for_dl, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'K', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("U_nflag Selected Articles and Replies"),
		N_("Unflag Selected Articles and Replies"),
		articlelist_selected_unflag_for_dl_deep, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'K', GDK_SHIFT_MASK, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("_Download Flagged Articles"),
		N_("Download Flagged Articles"),
		articlelist_download_flagged_bodies, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_RCV,
		'J', GDK_SHIFT_MASK|GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_END
};

GnomeUIInfo
articlelist_menu [] =
{
        {
		GNOME_APP_UI_ITEM, 
		N_("_Read Selected Message"),
		N_("Read Selected Message"),
		articlelist_selected_read_body, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'V', 0, NULL
	},
        {
		GNOME_APP_UI_ITEM, 
		N_("Read Selected Message in New _Window"),
		N_("Read Selected Message in New Window"),
		articlelist_selected_read_body_in_new_window, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'V', GDK_SHIFT_MASK, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_SUBTREE,
		N_("_Save"),
		N_("Save"),
		&save_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Wa_tch Thread"),
		N_("Watch Thread"),
		articlelist_selected_thread_watch, NULL,
		NULL,
		GNOME_APP_PIXMAP_DATA, watched_xpm,
		'W', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("_Ignore Thread"),
		N_("Ignore Thread"),
		articlelist_selected_thread_ignore, NULL,
		NULL,
		GNOME_APP_PIXMAP_NONE, NULL,
		'I', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Add Article to Bozo or Spam _Filter"),
		N_("Add Article to Bozo or Spam Filter"),
		articlelist_spamize, NULL,
		NULL,
		GNOME_APP_PIXMAP_NONE, NULL,
		'Z', GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_SUBTREE,
		N_("_Mark as Read"),
		N_("Mark as Read"),
		&mark_read_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_SUBTREE,
		N_("Mark as _Unread"),
		N_("Mark as Unread"),
		&mark_unread_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_SUBTREE,
		N_("Download _Bodies"),
		N_("Download Bodies"),
		&bodies_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_RCV,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_SUBTREE,
		N_("_Purge Downloaded Bodies"),
		N_("Purge Downloaded Bodies"),
		&purge_bodies_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
		'\0', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_SUBTREE,
		N_("Fol_der Tools"),
		N_("Folder Tools"),
		&folder_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_SUBTREE,
		N_("_Delete"),
		N_("Delete"),
		&delete_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
		'\0', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Select _All Articles"),
		N_("Select All Articles"),
		articlelist_select_all
	},
	GNOMEUIINFO_END
};



/***
****
****
****  INTERNAL UTILITIES:  SELECTIONS, ARTICLES, and NODES
****
****
***/


void
articlelist_select_all (void)
{
	pan_lock ();
	gtk_ctree_select_recursive (GTK_CTREE(Pan.article_ctree), NULL);
	pan_unlock ();
}

static Article*
articlelist_get_article_from_node (GtkCTreeNode* node)
{
	Article* article = NULL;

	if (node != NULL)
		article = ARTICLE(
			  gtk_ctree_node_get_row_data (
				  GTK_CTREE(Pan.article_ctree), node));

	return article;
}

static GPtrArray*
articlelist_get_selected_nodes (void)
{
	GPtrArray * a;
	const GList * l;
	debug_enter ("articlelist_get_selected_nodes");

       	a = g_ptr_array_new ();
	pan_lock ();
	for (l=GTK_CLIST(Pan.article_ctree)->selection; l!=NULL; l=l->next)
		g_ptr_array_add (a, GTK_CTREE_NODE(l->data));
	pan_unlock ();

	debug_exit ("articlelist_get_selected_nodes");
	return a;
}

GtkCTreeNode*
articlelist_get_selected_node (void)
{
	GtkCTreeNode* node = NULL;

	const GList* list =  GTK_CLIST(Pan.article_ctree)->selection;
	if (list != NULL)
		node = GTK_CTREE_NODE(list->data);

	return node;
}


GPtrArray*
articlelist_get_selected_articles (void)
{
	GPtrArray * retval;
	GList * list;
	debug_enter ("articlelist_get_selected_articles");

	retval = g_ptr_array_new ();

	list = GTK_CLIST(Pan.article_ctree)->selection;
	pan_g_ptr_array_reserve (retval, 64);
	for (; list!=NULL; list=list->next)
	{
		GtkCTreeNode * n = GTK_CTREE_NODE(list->data);
		Article * a = articlelist_get_article_from_node(n);
		if (a != NULL)
			g_ptr_array_add (retval, a);
	}

	debug_exit ("articlelist_get_selected_articles");
	return retval;
}

Article*
articlelist_get_selected_article (void)
{
	return articlelist_get_article_from_node (
		articlelist_get_selected_node());
}

void
articlelist_set_selected_nodes (GtkCTreeNode **nodes, gint node_qty)
{
	gint i;
	gfloat hadj;
	GtkCTree * tree;
	GtkCList * list;
	GtkCTreeNode * first = NULL;

	if (nodes==NULL || node_qty<=0)
		return;

	pan_lock();
	tree = GTK_CTREE(Pan.article_ctree);
	list = GTK_CLIST(Pan.article_ctree);
	gtk_clist_freeze (list);
	gtk_ctree_unselect_recursive (tree, gtk_ctree_node_nth(tree, 0));

	for (i=0; i<node_qty; ++i)
	{
		GtkCTreeNode * n = NULL;
		GtkCTreeNode * node = nodes[i];

		if (node != NULL)
		{
			/* remember the first good node */
			if (first == NULL)
				first = node;

			/* select the node */
			gtk_ctree_select (tree, node);

			/* show all the children */
			for (n=GTK_CTREE_ROW(node)->parent; n; n=GTK_CTREE_ROW(n)->parent)
				gtk_ctree_expand(tree, n);
		}
	}

	/* hack to sync the focus row on new selection -
	 * thanks to Julien Plissonneau Duquene. */
	if (list->selection != NULL)
		list->focus_row = g_list_position (list->row_list,
		                                   (GList*)(list->selection->data));

	/*
	 * Due to a bug in GtkCList this won't work right if the window
	 * isn't drawable.  In that case we can just set a flag and
	 * handle it later.
	 */
	if (!GTK_WIDGET_DRAWABLE(GTK_WIDGET(list))) {
		gtk_object_set_data (GTK_OBJECT(list), "pan_moveto_hack", GINT_TO_POINTER(1));
	} else if (first!=NULL) {
		hadj = gtk_clist_get_hadjustment(list)->value;
		gtk_ctree_node_moveto (tree, nodes[0], 0, (gfloat)0.5, (gfloat)0.0);
		gtk_adjustment_set_value (gtk_clist_get_hadjustment(list), hadj);
	}

	gtk_clist_thaw (list);
	pan_unlock();
}

void
articlelist_expand_selected_threads (void)
{
	guint i;
	GPtrArray * a;
	GtkCTree * tree;
	debug_enter ("articlelist_expand_selected_threads");

	/* expand */
	a = articlelist_get_selected_nodes ();
	pan_lock ();
	tree = GTK_CTREE(Pan.article_ctree);
	for (i=0; i<a->len; ++i)
		gtk_ctree_expand (tree, (GtkCTreeNode*)g_ptr_array_index(a,i));
	pan_unlock ();

	/* cleanup */
	g_ptr_array_free (a, TRUE);
	debug_exit ("articlelist_expand_selected_threads");
}
void
articlelist_collapse_selected_threads (void)
{
	guint i;
	GPtrArray * a;
	GtkCTree * tree;
	debug_enter ("articlelist_collapse_selected_threads");

	/* collapse */
	a = articlelist_get_selected_nodes ();
	pan_lock ();
	tree = GTK_CTREE(Pan.article_ctree);
	for (i=0; i<a->len; ++i)
		gtk_ctree_collapse (tree, (GtkCTreeNode*)g_ptr_array_index(a,i));
	pan_unlock ();

	/* cleanup */
	g_ptr_array_free (a, TRUE);
	debug_exit ("articlelist_collapse_selected_threads");
}


/***
****
****
****  INTERNAL UTILITIES:   GETTING THREADS
****
****
***/

typedef enum
{
	GET_WHOLE_THREAD,
	GET_SUBTHREAD
}
ThreadGet;

static GPtrArray*
articlelist_get_unique_threads (const GPtrArray    * articles,
                                ThreadGet            thread_get)
{
	int i;
	GPtrArray * thread = g_ptr_array_new ();
	GHashTable * all = g_hash_table_new (g_str_hash, g_str_equal);
	GPtrArray * retval;
	debug_enter ("articlelist_get_unique_threads");

	/* get the thread for each article passed in */
	for (i=0; i<articles->len; ++i)
	{
		int j;
		Article * a = ARTICLE(g_ptr_array_index(articles,i));

		/* if we already have the article, then we have its thread, so skip */
		if (g_hash_table_lookup (all, article_get_message_id(a)) != NULL)
			continue;

		/* get the thread associated with sel... */
		g_ptr_array_set_size (thread, 0);
		if (thread_get == GET_SUBTHREAD)
			article_get_subthread (a, thread);
		else
			article_get_entire_thread (a, thread);

		/* add the articles into "all" hash */
		for (j=0; j<thread->len; ++j) {
			Article * b = ARTICLE(g_ptr_array_index(thread,j));
			g_hash_table_insert (all, (gpointer)article_get_message_id(b), (gpointer)b);
		}
	}

	/* now massage the hashtable into an array */
	retval = g_ptr_array_new ();
	pan_g_ptr_array_reserve (retval, g_hash_table_size(all));
	pan_hash_to_ptr_array (all, retval);

	/* cleanup */
	g_hash_table_destroy (all);
	g_ptr_array_free (thread, TRUE);
	debug_exit ("articlelist_get_unique_threads");
	return retval;
}

/***
****
****
****  NAVIGATION
****
****
***/

GtkCTreeNode*
articlelist_node_next (GtkCTreeNode * node, gboolean sibling_only)
{
	GtkCTreeNode* n;

	if (!node)
		return NULL;

	if (sibling_only)
		return GTK_CTREE_ROW(node)->sibling;

	if (GTK_CTREE_ROW(node)->children)
		return GTK_CTREE_ROW(node)->children;

	for (n=node; n!=NULL; n=GTK_CTREE_ROW(n)->parent)
		if (GTK_CTREE_ROW(n)->sibling)
			return GTK_CTREE_ROW(n)->sibling;

	return NULL;
}

GtkCTreeNode*
articlelist_node_prev (GtkCTreeNode * node, gboolean sibling_only)
{
	GtkCTreeNode * parent;
	GtkCTreeNode * sibling;
	
	if (!node)
		return NULL;

	/* get parent */	
	parent = GTK_CTREE_ROW(node)->parent;
	if (!parent)
		return NULL;

	/* parent's first child */
	sibling=GTK_CTREE_ROW(parent)->children;
	if (sibling==node) /* this is the first child */
		return parent;

	/* previous sibling of node */
	while (GTK_CTREE_ROW(sibling)->sibling != node) {
		sibling = GTK_CTREE_ROW(sibling)->sibling;
		pan_warn_if_fail (sibling != NULL);
	}

	if (sibling_only)
		return sibling;

	/* find the absolutely last child of the older sibling */
	for (;;) {
		GtkCTreeNode* tmp = GTK_CTREE_ROW(sibling)->children;
		if (!tmp)
			return sibling;

		/* last child of sibling */
		while (GTK_CTREE_ROW(tmp)->sibling != NULL)
			tmp = GTK_CTREE_ROW(tmp)->sibling;

		sibling = tmp;
	}

	pan_warn_if_reached ();
}

static GtkCTreeNode*
find_prev_thread (GtkCTreeNode * node)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode * top = gtk_ctree_node_nth (tree, 0);
	const int row_qty = GTK_CLIST(tree)->rows;
	gboolean wrap = FALSE;

	/* nothing to do */
	if (row_qty < 2)
		return NULL;

	/* get starting point */	
	if (node == NULL)
		node = articlelist_get_selected_node ();

	/* find previous */
	if (node != NULL)
		while ( node!=top && GTK_CTREE_ROW(node)->parent!=top )
			node = GTK_CTREE_ROW(node)->parent;

	/* either no node selected or we're at the first thread; wrap */
	if (!node || articlelist_node_prev(node,TRUE)==top) {
		wrap = TRUE;
		node = gtk_ctree_node_nth ( tree, row_qty-1 );
		while ( node!=top && GTK_CTREE_ROW(node)->parent!=top )
			node = GTK_CTREE_ROW(node)->parent;
	}

	/* if we didn't wrap around, we need to back up one. */
	if (!wrap)
		node = articlelist_node_prev (node, TRUE);

	return node;
}

static GtkCTreeNode*
find_next_thread (GtkCTreeNode * node)
{
	GtkCTree* tree;
	GtkCTreeNode * top;

	tree = GTK_CTREE(Pan.article_ctree);
	top = gtk_ctree_node_nth (tree, 0);

	/* if nothing to read... */
	if (GTK_CLIST(tree)->rows<2)
		return NULL;

	/* find the start node */
	if (node == NULL)
		node = articlelist_get_selected_node ();

	/* walk backwards */
	if (node != NULL)
	{
		while (node!=top && GTK_CTREE_ROW(node)->parent!=top)
			node = GTK_CTREE_ROW(node)->parent;
		if (node==top)
			return NULL;
		node = GTK_CTREE_ROW(node)->sibling;
	}
	else if (GTK_CLIST(tree)->rows > 1)
	{
		node = gtk_ctree_node_nth (tree, 1);
	}

	return node;
}

static GtkCTreeNode*
find_prev_article (GtkCTreeNode * node)
{
	GtkCTree * tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode * top = gtk_ctree_node_nth (tree, 0);
	const int row_qty = GTK_CLIST(tree)->rows;
	gboolean wrap = FALSE;

	/* if nothing to read... */
	if (row_qty<2)
		return NULL;

	/* get a starting node... */
	if (node==NULL) {
		node = articlelist_get_selected_node (); 
		if ( !node || articlelist_node_prev(node,TRUE)==top ) {
			wrap = TRUE;
			node = gtk_ctree_node_nth (tree, row_qty-1 );
		}
	}

	/* if we didn't wrap, we need to move backwards by hand */
	if (!wrap)
		node = articlelist_node_prev (node, FALSE);

	return node;
}

static GtkCTreeNode*
find_next_article (GtkCTreeNode * node)
{
	GtkCTree * tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode * retval = NULL;

	/* if nothing to read... */
	if (GTK_CLIST(tree)->rows <2)
		return NULL;

	/* get starting point */
	if (node == NULL)
		node = articlelist_get_selected_node ();
	if (!node)
		node = gtk_ctree_node_nth (tree, 0);

	/* move to next node */
	retval = articlelist_node_next (node, FALSE);

	return retval;
}

static GtkCTreeNode*
find_next_unread (GtkCTreeNode * node)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);

	/* find the first node */
	if (node == NULL) {
		node = articlelist_get_selected_node ();
		if (!node)
			node = gtk_ctree_node_nth (tree, 0);
	}

	/* march through all the next nodes untili we find an unread one */
	for (;;) {
		const Article* article;
		node = articlelist_node_next (node, FALSE);
		if (node == NULL) /* end of articles */
			break;
		article = articlelist_get_article_from_node (node);
		if (article && !article_is_read (article))
			break;
	}

	return node;
}
static GtkCTreeNode*
find_next_unread_cached (GtkCTreeNode * node)
{
	node = find_next_unread (node);
	while (node!=NULL && !article_has_body(articlelist_get_article_from_node(node)))
		node = find_next_unread (node);
	return node;
}
void
articlelist_skip_to_next_thread (void)
{
	GtkCTreeNode * node = find_next_thread (NULL);
	if (node != NULL)
		articlelist_set_selected_nodes (&node, 1);
}
 
void
articlelist_view_next_thread (void)
{
	articlelist_skip_to_next_thread ();
	articlelist_selected_read_body ();
}

void
articlelist_skip_to_prev_thread (void)
{
	GtkCTreeNode * node = find_prev_thread (NULL);
	if (node != NULL)
		articlelist_set_selected_nodes (&node, 1);
}

void
articlelist_view_prev_thread (void)
{
	articlelist_skip_to_prev_thread ();
	articlelist_selected_read_body ();
}

void
articlelist_skip_to_prev (void)
{
	GtkCTreeNode * node = find_prev_article (NULL);
	if (node != NULL)
		articlelist_set_selected_nodes (&node, 1);
}
void
articlelist_view_prev (void)
{
	articlelist_skip_to_prev ();
	articlelist_selected_read_body ();
}

void
articlelist_skip_to_message_id (const gchar * message_id)
{
	GtkCTreeNode * node = articlelist_get_node_from_message_id (message_id);
	if (node != NULL)
		articlelist_set_selected_nodes (&node, 1);
}
void
articlelist_view_message_id (const gchar * message_id)
{
	articlelist_skip_to_message_id (message_id);
	articlelist_selected_read_body ();
}
   

void
articlelist_skip_to_next (void)
{
	GtkCTreeNode * node = find_next_article (NULL);
	if (node != NULL)
		articlelist_set_selected_nodes (&node, 1);
}
void
articlelist_view_next (void)
{
	articlelist_skip_to_next ();
	articlelist_selected_read_body ();
}
void
articlelist_skip_to_next_unread (void)
{
	GtkCTreeNode * node = find_next_unread (NULL);
	if (node != NULL)
		articlelist_set_selected_nodes (&node, 1);
}
void
articlelist_view_next_unread (void)
{
	articlelist_skip_to_next_unread ();
	articlelist_selected_read_body ();
}
void
articlelist_skip_to_next_unread_cached (void)
{
	GtkCTreeNode * node = find_next_unread_cached (NULL);
	if (node != NULL)
		articlelist_set_selected_nodes (&node, 1);
}
void
articlelist_view_next_unread_cached (void)
{
	articlelist_skip_to_next_unread_cached ();
	articlelist_selected_read_body ();
}

void
articlelist_selected_read_body (void)
{
	Article * article = articlelist_get_selected_article ();
	if (article != NULL)
		queue_add(TASK(task_body_new_defaults(article)));
}

void
articlelist_selected_read_body_in_new_window (void)
{
	Article * article = articlelist_get_selected_article ();
	if (article != NULL)
		message_read_window (article);
}

/***
****
****
****  CLEARING OUT THE ARTICLELIST
****
****
***/

static void
clear_hash_table (void)
{
	pthread_mutex_lock (&hash_mutex);
	if (messageid_to_node != NULL)
		g_hash_table_destroy (messageid_to_node);
	messageid_to_node = g_hash_table_new (g_str_hash, g_str_equal);
	pthread_mutex_unlock (&hash_mutex);
}

static void
clear_clist (void)
{
	GtkCList *list = GTK_CLIST(Pan.article_ctree);
	if (list->rows)
	{
		pan_lock();
		gtk_clist_clear (GTK_CLIST(Pan.article_ctree));
		pan_unlock();
	}
}

static void
clear_article_buffer (StatusItem * status)
{
	clear_hash_table ();
	clear_clist ();

	if (my_group != NULL)
		group_unref_articles (my_group, status);
}

/***
****
****
****   SORTING
****
****
***/

static const gchar*
column_to_title (int col)
{
	switch (col) {
		case 0: return " ";
		case 1: return " ";
		case 2: return _("Subject");
		case 3: return _("Lines");
		case 4: return _("Author");
		case 5: return _("Date");
		default: break;
	}
	
	pan_warn_if_reached();
	return _("BUG!!");
}

static int
sort_type_to_column (int type)
{
	int col = 0;
	switch (abs(type)) {
		case ARTICLE_SORT_SUBJECT: col=2; break;
		case ARTICLE_SORT_LINES: col=3; break;
		case ARTICLE_SORT_AUTHOR: col=4; break;
		case ARTICLE_SORT_DATE: col=5; break;
		default: pan_warn_if_reached(); break;
	}

	return col;
}

static void
articlelist_set_sort_type_nosort (int type, gboolean force)
{
	char * pch;
	int col = 0;

	if (!force && (type==sort_type))
		return;

	/* update old column */
	if (abs(sort_type) != abs(type)) {
		col = sort_type_to_column (sort_type);
		pan_lock();
		gtk_clist_set_column_title (GTK_CLIST(Pan.article_ctree), col, column_to_title(col));
		pan_unlock();
	}

	/* update sort type, fire callbacks */
	sort_type = type;
	pan_callback_call (articlelist_sort_changed, Pan.article_ctree, GINT_TO_POINTER(type));

	/* update new column */
	col = sort_type_to_column (sort_type);
	pch = g_strdup_printf ("%c%s", (type>0?'+':'-'), column_to_title(col));
	pan_lock();
	gtk_clist_set_column_title (GTK_CLIST(Pan.article_ctree), col, pch);
	pan_unlock();
	g_free (pch);
}

static gchar*
sort_selected_describe (const StatusItem* item)
{
	return g_strdup (_("Sorting Articles"));
}

static void
sort_in_thread (void* data)
{
	const int sort = GPOINTER_TO_INT(data);
        Group * group = articlelist_get_group ();
        StatusItem * item = NULL;
	debug_enter ("sort_in_thread");

        if (!group) /* articlelist is empty */
                return;

	pthread_mutex_lock (&article_ctree_lock);

	if (sort != sort_type)
	{
		GPtrArray * selected;
		GPtrArray * all = group_get_article_array (group);

		/* create a new status item to get sort/thread messages */
		item = STATUS_ITEM(status_item_new(sort_selected_describe));
		status_item_set_active (item, TRUE);

		/* tell the world what we're doing */
		status_item_emit_init_steps (item, all->len*2);
		status_item_emit_status (item, _("Sorting Articles"));

		/* update the sort type */
		articlelist_set_sort_type_nosort (sort, FALSE);
		group_set_sort_style (my_group, sort);
		sort_articles ((Article**)all->pdata, all->len, abs(sort), sort>0);

		/* update articles */
		selected = articlelist_get_selected_articles ();
		articlelist_repopulate (my_group, (Article**)all->pdata, all->len, item, selected);

		/* clean out the status item */
		status_item_set_active (item, FALSE);
		pan_object_unref(PAN_OBJECT(item));

		/* cleanup */
		g_ptr_array_free (selected, TRUE);
		g_ptr_array_free (all, TRUE);
	}

	pthread_mutex_unlock (&article_ctree_lock);
	debug_exit ("sort_in_thread");
}

void
articlelist_set_sort_type (int sort)
{
       	pthread_t thread;
       	pthread_create (&thread, NULL, (void*)sort_in_thread, GINT_TO_POINTER(sort));
       	pthread_detach (thread);
}

/***
****
****
****   ccc
****
****
***/

Group*
articlelist_get_group (void)
{
	return my_group;
}


/**
***
**/

typedef struct
{
	Article * a;
	gint depth;
}
SortDepthStruct;

static int
compare_pSDS_to_pSDS_by_depth (const void * a, const void * b)
{
	return ((SortDepthStruct*)a)->depth -
	       ((SortDepthStruct*)b)->depth;
}

static void
sort_articles_by_depth (GPtrArray * articles, gboolean parents_first)
{
	guint i;
	SortDepthStruct * buf;
	debug_enter ("sort_articles_by_depth");

	/* make an array of article,depth pairs */
	buf = g_new0 (SortDepthStruct, articles->len);
	for (i=0; i!=articles->len; ++i)
	{
		SortDepthStruct * item = buf + i;
		Article * tmp;
		item->a = ARTICLE(g_ptr_array_index(articles,i));
		item->depth = 0;
		for (tmp=item->a; tmp->parent!=NULL; tmp=tmp->parent)
			++item->depth;
	}

	/* sort the array */
	qsort (buf,
	       articles->len,
	       sizeof(SortDepthStruct),
	       compare_pSDS_to_pSDS_by_depth);


	/* put back into the original GPtrArray */
	if (parents_first)
		for (i=0; i!=articles->len; ++i)
			g_ptr_array_index(articles,i)= buf[i].a;
	else
		for (i=0; i!=articles->len; ++i)
			g_ptr_array_index(articles,i)= buf[articles->len-1-i].a;

	/* cleanup */
	g_free (buf);
	debug_exit ("sort_articles_by_depth");
}

static int
group_articles_added_cb (gpointer call_obj,
                         gpointer call_arg,
                         gpointer client_arg)
{
	Group * group = GROUP(call_obj);
	GPtrArray * added = (GPtrArray*) call_arg;

	/* sanity checks */
	g_return_val_if_fail (group!=NULL, 0);
	g_return_val_if_fail (added!=NULL, 0);
	g_return_val_if_fail (added->len>0, 0);

	/* if it's our group, reload */
	if (group == my_group)
		articlelist_reload();

	return 0;
}


static void
remove_article_nolock (Article * article)
{
	const gchar * message_id;
	GtkCTreeNode * node = NULL;
	GtkCTree * ctree = GTK_CTREE(Pan.article_ctree);

	/* sanity checks */
	g_return_if_fail (article_is_valid(article));

	/* if this article currently has a node, remove the node */
	message_id = article_get_message_id (article);
	node = articlelist_get_node_from_message_id (message_id);
	if (node!=NULL)
	{
		GSList *l=NULL, *l2;
		GtkCTreeNode * child;

		/* make a list of children */
		child = GTK_CTREE_ROW(node)->children;
		while (child != NULL) {
			l = g_slist_append (l, child);
			child = GTK_CTREE_ROW(child)->sibling;
		}

		/* for each child, reparent to the dying node's parent */
		for (l2=l; l2!=NULL; l2=l2->next) {
			GtkCTreeNode * child = (GtkCTreeNode*) l2->data;
			gtk_ctree_move (ctree, child,
			                GTK_CTREE_ROW(node)->parent,
			                GTK_CTREE_ROW(node)->sibling);
		}

		g_slist_free (l);

		/* remove from hashtable */
		pthread_mutex_lock (&hash_mutex);
		g_hash_table_remove (messageid_to_node, message_id);
		pthread_mutex_unlock (&hash_mutex);

		/* don't remove the node yet; we need it
		   for article_unread_inc () below */
	}

	/* if the article's unread, update the parents' unread children count */
	if (!article_is_read (article))
	{
		GtkCTreeNode *n = node;
		if (n == NULL)
		{
			Article* a;
			for (a=article->parent; a!=NULL && !n; a=a->parent)
				n = articlelist_get_node_from_message_id (article_get_message_id(article));
		}
		if (n != NULL) 
		{
			articlelist_unread_inc_nolock (n, -1);
		}
	}

	/* remove this node from the tree */
	if (node != NULL)
		gtk_ctree_remove_node (ctree, node);
}

static int
group_articles_removed_cb (gpointer call_obj,
                           gpointer call_arg,
                           gpointer client_data)
{
	Group * group = GROUP(call_obj);
	GPtrArray * removed = (GPtrArray*) call_arg;
	GPtrArray * tmp;
	debug_enter ("group_articles_removed_cb");

	/* sanity checks */
	g_return_val_if_fail (group!=NULL, 0);
	g_return_val_if_fail (removed!=NULL, 0);
	g_return_val_if_fail (removed->len>0, 0);

	/* if it's not important to us, don't bother */
	if (group != my_group) {
		debug_exit ("group_articles_removed_cb");
		return 0;
	}

	/* make sure we do the children first--this reduces the number
	   of cnodes we have to reparent and keeps ctree from falling
	   over. */
	tmp = pan_g_ptr_array_dup (removed);
	sort_articles_by_depth (tmp, FALSE);

	/* we now know that articles have been removed
	   from the group the articlelist is currently
	   looking at.  Try to remove all traces of them
	   from this list. */
	pan_lock ();
	gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));
	pan_g_ptr_array_foreach (tmp, (GFunc)remove_article_nolock, NULL);
	gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));
	pan_unlock ();

	/* cleanup */
	g_ptr_array_free (tmp, TRUE);

	debug_exit ("group_articles_removed_cb");
	return 0;
}


/****
*****
*****  MARKING ARTICLES READ/UNREAD
*****
****/

static void
articlelist_mark_articles (const GPtrArray * articles, gboolean read)
{
	g_return_if_fail (articles!=NULL);
	g_return_if_fail (articles->len!=0);
	articles_set_read ((Article**)articles->pdata, articles->len, read);
}

void
articlelist_all_mark_read (void)
{
	if (my_group != NULL)
	{
		guint i;
		GPtrArray * a;

		/* set unread_children to zero for all. */
		a = group_get_article_array (my_group);
		for (i=0; i!=a->len; ++i)
			ARTICLE(g_ptr_array_index(a,i))->unread_children = 0;
		g_ptr_array_free (a, TRUE);

		/* mark all read */
		group_mark_all_read (my_group, TRUE);
	}
}
void
articlelist_all_mark_unread (void)
{
	/* FIXME: update unread_children */

	group_mark_all_read (my_group, FALSE);
}

void
articlelist_selected_mark_unread (void)
{
	GPtrArray * sel;
	debug_enter ("articlelist_selected_mark_unread");

	sel = articlelist_get_selected_articles ();
	if (sel->len != 0)
		articlelist_mark_articles (sel, FALSE);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("articlelist_selected_mark_unread");
}
void
articlelist_selected_mark_read (void)
{
	GPtrArray * sel;
	debug_enter ("articlelist_selected_mark_read");

       	sel = articlelist_get_selected_articles ();
	if (sel->len != 0)
		articlelist_mark_articles (sel, TRUE);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("articlelist_selected_mark_read");
}

static void
articlelist_selected_mark_threads_impl (int threads, gboolean read)
{
	GPtrArray * sel;
	GPtrArray * all;
	debug_enter ("articlelist_selected_mark_threads_impl");

       	sel = articlelist_get_selected_articles ();
       	all = articlelist_get_unique_threads (sel, threads);
	if (all->len != 0)
		articlelist_mark_articles (all, read);
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (all, TRUE);

	debug_exit ("articlelist_selected_mark_threads_impl");
}
void
articlelist_selected_mark_unread_deep (void)
{
	articlelist_selected_mark_threads_impl (GET_SUBTHREAD, FALSE);
}
void
articlelist_selected_mark_unread_thread (void)
{
	articlelist_selected_mark_threads_impl (GET_WHOLE_THREAD, FALSE);
}
void
articlelist_selected_mark_read_deep (void)
{
	articlelist_selected_mark_threads_impl (GET_SUBTHREAD, TRUE);
}
void
articlelist_selected_mark_read_thread (void)
{
	articlelist_selected_mark_threads_impl (GET_WHOLE_THREAD, TRUE);
}


void
articlelist_selected_unflag_for_dl (void)
{
	GPtrArray * s;
	debug_enter ("articlelist_selected_unflag_for_dl");

       	s = articlelist_get_selected_articles ();
	articles_remove_flag ((Article**)s->pdata, s->len, STATE_DOWNLOAD_FLAGGED);
	g_ptr_array_free (s, TRUE);

	debug_exit ("articlelist_selected_unflag_for_dl");
}

void
articlelist_selected_unflag_for_dl_deep (void)
{
	GPtrArray * s;
	GPtrArray * a;
	debug_enter ("articlelist_selected_unflag_for_dl_deep");

	s = articlelist_get_selected_articles ();
	a = articlelist_get_unique_threads (s, GET_SUBTHREAD);
	articles_remove_flag ((Article**)a->pdata, a->len, STATE_DOWNLOAD_FLAGGED);
	g_ptr_array_free (s, TRUE);
	g_ptr_array_free (a, TRUE);

	debug_exit ("articlelist_selected_unflag_for_dl_deep");
}

void
articlelist_selected_flag_for_dl (void)
{
	GPtrArray * s;
	debug_enter ("articlelist_selected_flag_for_dl");

       	s = articlelist_get_selected_articles ();
	articles_add_flag ((Article**)s->pdata, s->len, STATE_DOWNLOAD_FLAGGED);
	g_ptr_array_free (s, TRUE);

	debug_exit ("articlelist_selected_flag_for_dl");
}

void
articlelist_selected_flag_for_dl_deep (void)
{
	GPtrArray * s;
	GPtrArray * a;
	debug_enter ("articlelist_selected_flag_for_dl_deep");

       	s = articlelist_get_selected_articles ();
       	a = articlelist_get_unique_threads (s, GET_SUBTHREAD);
	articles_add_flag ((Article**)a->pdata, a->len, STATE_DOWNLOAD_FLAGGED);
	g_ptr_array_free (s, TRUE);
	g_ptr_array_free (a, TRUE);

	debug_exit ("articlelist_selected_flag_for_dl_deep");
}

void
articlelist_download_flagged_bodies (void)
{
	debug_enter ("articlelist_download_flagged_bodies");

	if (my_group != NULL)
	{
		guint i;
		GPtrArray * all = group_get_article_array (my_group);
		GPtrArray * flagged = g_ptr_array_new ();

		/* build an array of flagged articles */
		pan_g_ptr_array_reserve (flagged, all->len);
		for (i=0; i!=all->len; ++i) {
			Article * a = ARTICLE(g_ptr_array_index(all,i));
			if (article_flag_on (a, STATE_DOWNLOAD_FLAGGED))
				g_ptr_array_add (flagged, a);
		}

		/* queue a task */
		if (flagged->len != 0) {
			Task * task = TASK(task_bodies_new (my_group, flagged));
			if (task != NULL)
				queue_add (task);
		}

		/* cleanup */
		g_ptr_array_free (all, TRUE);
		g_ptr_array_free (flagged, TRUE);
	}

	debug_exit ("articlelist_download_flagged_bodies");
}

/****
*****
*****  COPYING TO FOLDERS
*****
****/


static void
articlelist_article_to_folder (const GPtrArray * articles, gboolean keep_original)
{
	Group * folder;
	gboolean success;
	debug_enter ("articlelist_article_to_folder");

	/* make sure the user's selected a folder. */
       	folder = grouplist_get_selected_group ();
	if (folder==NULL || !group_is_folder(folder)) {
		pan_error_dialog (_("Please Select a Folder from the Group list."));
		return;
	}

	success = article_copy_articles_to_folder (folder,
	                                           (Article**)articles->pdata,
	                                           articles->len);

	if (success && !keep_original)
		group_remove_articles (my_group, articles);

	debug_exit ("articlelist_article_to_folder");
}
void
articlelist_selected_copy_to_folder (void)
{
	GPtrArray * sel;
	debug_enter ("articlelist_selected_copy_to_folder");

       	sel = articlelist_get_selected_articles ();
	articlelist_article_to_folder (sel, TRUE);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("articlelist_selected_copy_to_folder");
}
void
articlelist_selected_copy_to_folder_deep (void)
{
	GPtrArray * sel;
	GPtrArray * all;
	debug_enter ("articlelist_selected_copy_to_folder_deep");

	sel = articlelist_get_selected_articles ();
	all = articlelist_get_unique_threads (sel, GET_SUBTHREAD);
	articlelist_article_to_folder (all, TRUE);
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (all, TRUE);

	debug_exit ("articlelist_selected_copy_to_folder_deep");
}
void
articlelist_selected_move_to_folder (void)
{
	GPtrArray * sel;
	debug_enter ("articlelist_selected_move_to_folder");

       	sel = articlelist_get_selected_articles ();
	articlelist_article_to_folder (sel, FALSE);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("articlelist_selected_move_to_folder");
}
void
articlelist_selected_move_to_folder_deep (void)
{
	GPtrArray * sel;
	GPtrArray * all;
	debug_enter ("articlelist_selected_move_to_folder_deep");

       	sel = articlelist_get_selected_articles ();
       	all = articlelist_get_unique_threads (sel, GET_SUBTHREAD);
	articlelist_article_to_folder (all, FALSE);
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (all, TRUE);

	debug_exit ("articlelist_selected_move_to_folder_deep");
}
void
articlelist_selected_edit (void)
{
	Article * a;
	debug_enter ("articlelist_selected_edit");

       	a = articlelist_get_selected_article ();
	if (a != NULL)
		message_edit_window (a);

	debug_exit ("articlelist_selected_edit");
}

void
articlelist_selected_cancel (void)
{
	Article * a;
	debug_enter ("articlelist_selected_cancel");

       	a = articlelist_get_selected_article ();
	if (a != NULL)
		article_cancel (a);

	debug_exit ("articlelist_selected_cancel");
}

void
articlelist_selected_supersede (void)
{
	Article * a;
	debug_enter ("articlelist_selected_supersede");

       	a = articlelist_get_selected_article ();
	if (a != NULL)
		article_supersede (a);

	debug_exit ("articlelist_selected_supersede");
}

/****
*****
*****  DOWNLOADING ARTICLE BODIES
*****
****/

void
articlelist_all_download (void)
{
	GPtrArray * articles;
	debug_enter ("articlelist_all_download");

	/* sanity check */
	g_return_if_fail (my_group!=NULL);

	/* queue all the articles for download */
	articles = group_get_article_array (my_group);
	if (articles->len != 0) {
		Task * task = TASK(task_bodies_new (my_group, articles));
		if (task != NULL)
			queue_add (task);
	}

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	debug_exit ("articlelist_all_download");
}

void
articlelist_selected_download (void)
{
	GPtrArray * sel;
	Task * task;
	debug_enter ("articlelist_selected_download");

	g_return_if_fail (my_group != NULL);

	/* add the task */
       	sel = articlelist_get_selected_articles ();
	task = TASK(task_bodies_new(my_group, sel));
	if (task != NULL)
		queue_add (task);

	/* cleanup */
	g_ptr_array_free (sel, TRUE);
	debug_exit ("articlelist_selected_download");
}



void
articlelist_selected_thread_download (void)
{
	GPtrArray * sel;
	GPtrArray * all;
	Task * task;
	debug_enter ("articlelist_selected_thread_download");

	/* add the task */
	sel = articlelist_get_selected_articles ();
	all = articlelist_get_unique_threads (sel, GET_SUBTHREAD);
	task = TASK(task_bodies_new(my_group, all));
	if (task != NULL)
		queue_add (task);

	/* cleanup */
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (all, TRUE);

	debug_exit ("articlelist_selected_thread_download");
}

static void
purge_articles (const GPtrArray * articles)
{
	gint i;
	const gchar ** message_ids;

	g_return_if_fail (articles!=NULL);
	g_return_if_fail (articles->len!=0);

	message_ids = g_new (const gchar*, articles->len);
	for (i=0; i<articles->len; ++i)
		message_ids[i] = article_get_message_id (ARTICLE(g_ptr_array_index(articles,i)));
	acache_expire_messages (message_ids, articles->len);
	g_free (message_ids);
}

void
articlelist_selected_purge_bodies (void)
{
	GPtrArray * sel = articlelist_get_selected_articles ();
	purge_articles (sel);
	g_ptr_array_free (sel, TRUE);
}

void
articlelist_selected_purge_bodies_deep (void)
{
	GPtrArray * sel = articlelist_get_selected_articles ();
	GPtrArray * all = articlelist_get_unique_threads (sel, GET_SUBTHREAD);
	purge_articles (all);
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (all, TRUE);
}

void
articlelist_all_purge_bodies (void)
{
	if (my_group!=NULL)
	{
		GPtrArray * all = group_get_article_array (my_group);
		purge_articles (all);
		g_ptr_array_free (all, TRUE);
	}
}

/***
****
****
****   IGNORING / WATCHING THREADS
****
****
***/

static void
rekill_thread (Article * article)
{
	guint i;
	GPtrArray * articles;
	gint state;
	debug_enter ("rekill_thread");

	g_return_if_fail (article != NULL);

	/* get the articles and their state */
	state = thread_get_state (article);
       	articles = g_ptr_array_new ();
	article_get_entire_thread (article, articles);

	/* update the tree */
	pan_lock ();
	gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));
	for (i=0; i!=articles->len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(articles,i));
		articlelist_update_node_impl (a, FALSE);
	}
	gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));
	pan_unlock();

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	debug_exit ("rekill_thread");
}

static void
articlelist_spamize (void)
{
	guint i;
	GPtrArray * articles;
	debug_enter ("articlelist_spamize");

	articles = articlelist_get_selected_articles ();

	pan_lock ();
	for (i=0; i<articles->len; ++i)
	{
		GtkWidget * w = article_filter_dialog_new (ARTICLE(g_ptr_array_index(articles,i)));
		gnome_dialog_set_parent (GNOME_DIALOG(w), GTK_WINDOW(Pan.window));
	        gtk_widget_show_all (w);
	}
	pan_unlock ();

	g_ptr_array_free (articles, TRUE);
	debug_exit ("articlelist_spamize");
}

void
articlelist_selected_thread_watch (void)
{
	guint i;
	GPtrArray * sel;
	debug_enter ("articlelist_selected_thread_watch");

	sel = articlelist_get_selected_articles ();
	for (i=0; i<sel->len; ++i)
	{
		Article * a = ARTICLE(g_ptr_array_index(sel,i));
		gint o = thread_get_state (a);
		gint n = o==THREAD_WATCHED ? THREAD_NOTHING : THREAD_WATCHED;
		thread_set_state (a, n);
		rekill_thread (a);
	}

	g_ptr_array_free (sel, TRUE);
	debug_exit ("articlelist_selected_thread_watch");
}

void
articlelist_selected_thread_ignore (void)
{
	guint i;
	GPtrArray * sel;
	debug_enter ("articlelist_selected_thread_ignore");

	sel = articlelist_get_selected_articles ();
	for (i=0; i<sel->len; ++i)
	{
		Article * a = ARTICLE(g_ptr_array_index(sel,i));
		gint o = thread_get_state (a);
		gint n = o==THREAD_IGNORED ? THREAD_NOTHING : THREAD_IGNORED;
		thread_set_state (a, n);
		rekill_thread (a);
	}

	g_ptr_array_free (sel, TRUE);
	debug_exit ("articlelist_selected_thread_ignore");
}

/***
****
****
****   DECODING ARTICLES
****
****
***/

static void
open_article (const Article * a)
{
	gchar * file;
	const gchar * files;

	g_return_if_fail (a!=NULL);

	files = article_get_header (a, PAN_HEADER_FILENAME);
	while ((file = get_next_token_str (files, '\n', &files))) {
		open_outside_file (file);
		g_free (file);
	}

}

static gint
open_decoded_cb (gpointer call_obj,
                 gpointer call_arg,
                 gpointer user_data)
{
	gint status = GPOINTER_TO_INT(call_arg);

	if (status == TASK_SUCCESS) {
		TaskDecode * task = TASK_DECODE(call_obj);
		if (task!=NULL && task->articles!=NULL) {
			const Article * a = ARTICLE(task->articles->data);
			if (a!=NULL)
				open_article (a);
		}
	}

	return 0;
}

static void
articlelist_decode_impl (Article * a, gboolean open)
{
	Task * task;
	GSList * articles;
	const gchar * files;
	debug_enter ("articlelist_decode_impl");

	g_return_if_fail (a!=NULL);

	/* if user is opening the files & we already have them... */
	files = article_get_header (a, PAN_HEADER_FILENAME);
	if (open && is_nonempty_string(files)) {
		open_article (a);
		debug_exit ("articlelist_decode_impl");
		return;
	}

	/* build an array of this article and all its children. */
	articles = g_slist_copy (a->threads);
	articles = g_slist_prepend (articles, a);

	/* queue for decode */
	task = TASK(task_decode_new (my_group, article_get_subject(a), NULL, NULL, articles));
	if (open)
		pan_callback_add (task->task_ran_callback, open_decoded_cb, NULL);
	task_decode_validate_and_queue_or_self_destruct (TASK_DECODE(task), -1);

	/* cleanup this loop */
	g_slist_free (articles);
	debug_exit ("articlelist_decode_impl");
}

static void
articlelist_selected_decode_impl (gboolean open)
{
	guint i;
	GPtrArray * sel;
	debug_enter ("articlelist_selected_decode_impl");

	/* sanity checks... */
	g_return_if_fail (my_group!=NULL);

	/* decode each selected article */
	sel = articlelist_get_selected_articles ();
	for (i=0; i<sel->len; ++i)
		articlelist_decode_impl (ARTICLE(g_ptr_array_index(sel,i)), open);
	g_ptr_array_free (sel, TRUE);

	/* unselect selection */
	if (1) {
		GtkWidget * w = Pan.article_ctree;
		gtk_clist_freeze (GTK_CLIST(w));
		gtk_ctree_unselect_recursive (GTK_CTREE(w), NULL);
		gtk_clist_thaw (GTK_CLIST(w));
	}

	debug_exit ("articlelist_selected_decode_impl");
}

void
articlelist_selected_decode_as (void)
{
	guint i;
	GPtrArray * sel;
	GtkWidget * w = Pan.article_ctree;
	GSList * gslists_of_articles = NULL;
	debug_enter ("articlelist_selected_decode_as");

	/* sanity checks... */
	g_return_if_fail (my_group!=NULL);

	/* loop through all the selected articles */
	sel = articlelist_get_selected_articles ();
	for (i=0; i!=sel->len; ++i)
	{
		GSList * l;
		Article * article = ARTICLE(g_ptr_array_index(sel,i));

		/* build an array of this node and all its children. */
		l = g_slist_copy (article->threads);
		l = g_slist_prepend (l, article);

		/* add this to the gslist of articles */
		gslists_of_articles = g_slist_prepend (gslists_of_articles, l);
	}

	/* save_attach_as owns gslists_of_articles now */
	gslists_of_articles = g_slist_reverse (gslists_of_articles);
	save_attachment_as (my_group, gslists_of_articles);

	/* cleanup */
	gtk_clist_freeze (GTK_CLIST(w));
	gtk_ctree_unselect_recursive (GTK_CTREE(w), NULL);
	gtk_clist_thaw (GTK_CLIST(w));

	g_ptr_array_free (sel, TRUE);
	debug_exit ("articlelist_selected_decode_as");
}

void
articlelist_selected_decode (void)
{		
	articlelist_selected_decode_impl (FALSE);
}

void
articlelist_selected_open (void)
{
	articlelist_selected_decode_impl (TRUE);
}


static void
articlelist_selected_delete_attachment (void)
{
	guint i;
	GPtrArray * sel;
	GtkWidget * w = Pan.article_ctree;
	debug_enter ("articlelist_selected_delete_attachment");

	/* sanity checks... */
	g_return_if_fail (my_group!=NULL);

	sel = articlelist_get_selected_articles ();

	/* loop through all the selected articles */
	articles_remove_flag ((Article**)sel->pdata, sel->len, STATE_DECODED);
	articles_set_read ((Article**)sel->pdata, sel->len, FALSE);
	for (i=0; i!=sel->len; ++i)
	{
		Article * article = ARTICLE(g_ptr_array_index(sel,i));
		gchar * file;
		const gchar * files = article_get_header (article, PAN_HEADER_FILENAME);
		if (files == NULL)
			continue;

		while ((file = get_next_token_str (files, '\n', &files))) {
			unlink (file);
			g_free (file);
		}

		article_remove_header (article, PAN_HEADER_FILENAME);
	}


	/* cleanup */
	gtk_ctree_unselect_recursive (GTK_CTREE(w), NULL);
	g_ptr_array_free (sel, TRUE);
	debug_exit ("articlelist_selected_delete_attachment");
}

static void
articlelist_save_selected_article_text (void)
{
	Article * a = articlelist_get_selected_article ();
	if (a != NULL)
		save_article (a);
}

/***
****
****
****  POPULATING THE TREE:  UTILITY FUNCTIONS
****
****
***/

static GnomePixmap*
get_column_1_pixmap (const Article * a)
{
	GnomePixmap * retval = NULL;
	const gboolean is_new = article_is_new (a);
	const gboolean is_read = article_is_read (a);

	g_return_val_if_fail (a!=NULL, NULL);

	if (article_flag_on (a, STATE_ERROR))
		retval = error_pixmap;
	else if (article_flag_on (a, STATE_MULTIPART_ALL))
		retval = is_new ? mp_complete_pixmap : old_mp_complete_pixmap;
	else if (article_flag_on (a, STATE_MULTIPART_SOME))
		retval = is_new ? mp_incomplete_pixmap : old_mp_incomplete_pixmap;
	else if (is_new && is_read)
		retval = new_read_pixmap;
	else if (is_new && !is_read)
		retval = new_unread_pixmap;
	else if (!is_new && !is_read)
		retval = old_unread_pixmap;
	else
		retval = old_read_pixmap;

	return retval;
}

static GnomePixmap*
get_column_2_pixmap (const Article * a)
{
	GnomePixmap * retval = NULL;

	g_return_val_if_fail (article_is_valid(a), NULL);

	if (article_flag_on (a, STATE_DOWNLOAD_FLAGGED))
		retval = flagged_for_download_pixmap;
	else if (article_flag_on (a, STATE_DECODED))
		retval = binary_decoded_pixmap;
	else if (article_flag_on (a, STATE_DECODE_QUEUED))
		retval = decode_queue_pixmap;
	else if (acache_has_message (article_get_message_id(a)))
		retval = article_cached_pixmap;

	return retval;
}

static GdkFont * normalfont = NULL;
static GdkFont * newrepliesfont = NULL;

static GdkFont*
pan_load_font (const gchar * name)
{
	return use_gdk_fontset_load
		? gdk_fontset_load (name)
		: gdk_font_load (name);
}

static gboolean rebuild_style = TRUE;

static void
articlelist_set_node_style_nolock (GtkCTreeNode   * node,
			           const Article  * article)
{
	GtkCTree* tree = GTK_CTREE (Pan.article_ctree);

	/* initialize the styles/fonts the first time through */
	if (rebuild_style)
	{
		GtkStyle * s = gtk_widget_get_style (GTK_WIDGET(Pan.window));
		gint       i;

		rebuild_style = FALSE;

		normalfont = pan_load_font (thread_normal_font);
		newrepliesfont = pan_load_font (thread_new_replies_font);
		pan_widget_set_font (Pan.article_ctree, thread_new_replies_font);

		for (i=0; i<2; i++)
		{
			normal_style[i] = gtk_style_copy (s);
			normal_style[i]->fg[0] = thread_normal_color;
			normal_style[i]->font = i ? newrepliesfont : normalfont;

			read_style[i] = gtk_style_copy (s);
			read_style[i]->fg[0] = thread_read_color;
			read_style[i]->font = i ? newrepliesfont : normalfont;

			killfile_style[i] = gtk_style_copy (s);
			killfile_style[i]->fg[0] = killfile_color;
			killfile_style[i]->font = i ? newrepliesfont : normalfont;

			watched_style[i] = gtk_style_copy (s);
			watched_style[i]->fg[0] = watched_color;
			watched_style[i]->font = i ? newrepliesfont : normalfont;
		}
	}

	if (1)
	{
		GtkStyle ** style = NULL;
		const gboolean is_read = article ? article_is_read (article) : FALSE;
		const gboolean is_new = article ? article_is_new (article) : FALSE;
		const gboolean new_children = article ? article->new_children!=0 : FALSE;
		const gboolean unread_children = article ? article->unread_children!=0 : FALSE;
		const gint thread_state = article ? thread_get_state(article) : THREAD_NOTHING;

		if (thread_state == THREAD_IGNORED)
			style = killfile_style;
		else if (thread_state == THREAD_WATCHED)
			style = watched_style;
		else if (is_read && !unread_children)
			style = read_style;
		else
			style = normal_style;

		gtk_ctree_node_set_row_style (tree, node, style[0]);

		if (is_new || new_children)
			gtk_ctree_node_set_cell_style (tree, node, 2, 
				style[1]);
	}
}

static void
apply_filter_tests (GPtrArray * headers)
{
	register int i;
	register Article * a;
	GPtrArray * tmp;
	GPtrArray * family;

	g_return_if_fail (headers!=NULL);

	/* make a working copy of the headers */
	tmp = pan_g_ptr_array_dup (headers);

	/* clear the filter state */
	for (i=0; i!=tmp->len; ++i) {
		a = ARTICLE(g_ptr_array_index(tmp,i));
		a->self_passes_filter = FALSE;
		a->tree_passes_filter = FALSE;
	}

	/* remove the articles that don't pass */
	if (my_filter != NULL)
		filter_remove_failures (my_filter, tmp);

	/* mark up the successes */
	family = g_ptr_array_new ();
	for (i=0; i<tmp->len; ++i)
	{
		gint j;
		a = ARTICLE(g_ptr_array_index(tmp,i));

		/* this article passes the filter */
		a->self_passes_filter = TRUE;
		a->tree_passes_filter = TRUE;

		/* process the family */
		g_ptr_array_set_size (family, 0);
		switch (my_filter_show) {
			case FILTER_SHOW_MATCHES:
				break;
			case FILTER_SHOW_MATCHES_AND_REPLIES:
				article_get_subthread (a, family);
				break;
			case FILTER_SHOW_MATCHES_AND_REFERENCES:
				article_get_references (a, family);
				break;
			case FILTER_SHOW_THREAD:
				article_get_entire_thread (a, family);
				break;
		}
		for (j=0; j<family->len; ++j) {
			Article * relative = ARTICLE(g_ptr_array_index(family,j));
			relative->tree_passes_filter = TRUE;
		}
	}

	/* cleanup */
	g_ptr_array_free (tmp, TRUE);
	g_ptr_array_free (family, TRUE);
}

/***
****
****
****  POPULATING THE TREE:  THE DIRTY WORK
****
****
***/

static GtkCTreeNode*
add_article_to_ctree (GtkCTree      * tree,
                      GtkCTreeNode  * parent,
                      GtkCTreeNode  * sibling,
                      Article       * article,
                      StatusItem    * status,
                      gboolean        expanded)
{
	GnomePixmap * col_1;
	GnomePixmap * col_2;
	gchar line_count_buf[32];
	GtkCTreeNode * node;
	gulong lcountsum;
	gchar * text[COLUMNS] = { NULL, NULL, NULL, NULL, NULL, NULL };
	gchar * author;
	gchar * date;
	gint i;

	/* author */
	author = article_get_short_author_str (article);

	/* date */
	date = get_date_display_string (article->date, thread_date_format);

	/* line count */
	lcountsum = article->linecount;
	if (article_flag_on (article, STATE_MULTIPART_ALL)) {
		GSList *l;
		for (l=article->threads; l; l=l->next)
			lcountsum += ARTICLE(l->data)->linecount;
	}
	commatize_ulong (lcountsum, line_count_buf);

	/* icons */
	col_1 = get_column_1_pixmap (article);
	col_2 = get_column_2_pixmap (article);
       
	if (status != NULL)
		status_item_emit_next_step (status);

	/* populate the line */
	i = 0;
	i++; /* skip icon column 1 */
	i++; /* skip icon column 2 */
	text[i++] = (gchar*) article_get_subject (article);
	text[i++] = line_count_buf;
	text[i++] = (gchar*)author;
	text[i++] = date;
	g_assert (i==COLUMNS);

	/* insert the node */
	pan_lock ();
	node = gtk_ctree_insert_node (tree, parent, sibling,
	                              (char**)text, GNOME_PAD_SMALL,
	                              NULL, NULL, NULL, NULL,
	                              FALSE, expanded);
	gtk_ctree_node_set_pixmap (tree, node, 0, col_1->pixmap, col_1->mask);
	if (col_2 != NULL)
		gtk_ctree_node_set_pixmap (tree, node, 1, col_2->pixmap, col_2->mask);
	gtk_ctree_node_set_row_data (tree, node, article);
	articlelist_set_node_style_nolock (node, article);
	pan_unlock ();

	/* update hash table */
	pthread_mutex_lock (&hash_mutex);
	g_hash_table_insert (messageid_to_node, (gpointer)article_get_message_id(article), node);
	pthread_mutex_unlock (&hash_mutex);

	g_free (author);
	g_free (date);
	return node;
}

static void
build_article_ctree_recursive (GtkCTree      * tree,
                               GtkCTreeNode  * parent,
                               GSList        * list,
                               StatusItem    * status,
                               gboolean        expanded)
{
	GSList * tmp;
	GSList * pos;
	GtkCTreeNode * sibling = NULL;

	/* sanity check */
	g_return_if_fail (parent!=NULL);

	/* this item gets inserted before the 'sibling' node,
	 * so we must add in reverse order */
	tmp = g_slist_reverse(g_slist_copy(list));
        for (pos=tmp; pos!=NULL; pos=pos->next)
        {
		Article* article = ARTICLE(pos->data);
		GtkCTreeNode* node = NULL;

		/* possibly don't show subtrees that don't measure up to filter */
		if (is_threaded && !article->tree_passes_filter)
			continue;

		/* possibly don't show 2...n of multipart nodes */
		if (hide_mpart_child_nodes
			&& (article->parent!=NULL)
			&& (article->parent->state&STATE_MULTIPART_ALL)
			&& (article->part>1))
			continue;

		node = add_article_to_ctree (tree,
		                             parent,
		                             sibling,
		                             article,
		                             status,
		                             expanded);
		if (article->threads!=NULL)
			build_article_ctree_recursive (tree,
			                               node,
			                               article->threads,
			                               status,
			                               TRUE);

		sibling = node;
        }

	/* cleanup */
	g_slist_free (tmp);
}



typedef struct
{
	const Group *group;
	Article **article_buf;
	int article_qty;
	StatusItem *item;
	int index;
	GtkCTreeNode *root;
}
BuildArticleListStruct;

static void
add_nodes (gpointer data)
{
	BuildArticleListStruct *build = (BuildArticleListStruct*) data;
	GtkCTree * tree = GTK_CTREE (Pan.article_ctree);
	GtkCTreeNode * sibling = NULL;
	gint i;
	debug_enter ("add_nodes");

	for (i=build->article_qty-1; i>=0; --i)
	{
		Article * article = build->article_buf[i];
		GtkCTreeNode * node = NULL;
		gboolean has_parent;

		/* make sure we pass the state filter */
		if (is_threaded && !article->tree_passes_filter)
			continue;
		if (!is_threaded && !article->self_passes_filter)
			continue;

		has_parent = article->parent != NULL;

		if ((!is_threaded && article->part<2) ||
			(is_threaded && (!has_parent || !article->parent->tree_passes_filter)))
		{
			node = add_article_to_ctree (tree,
			                             build->root,
			                             sibling,
			                             article,
			                             build->item,
			                             FALSE);
			sibling = node;
		}

		/* if threads are turned on, and there are children of this thread... */
		if (is_threaded && node!=NULL && article->threads!=NULL)
		{
			build_article_ctree_recursive (tree,
			                               node,
			                               article->threads,
			                               build->item,
			                               TRUE);
		}
	}

	debug_exit ("add_nodes");
}

/* fires next_step article_qty times */
static void
articlelist_repopulate (const Group   * group,
                        Article      ** article_buf,
                        gint            article_qty,
                        StatusItem    * item,
                        GPtrArray     * sel_articles)
{
	GtkCTree * tree = GTK_CTREE (Pan.article_ctree);
	gchar * text[COLUMNS] = { NULL, NULL, NULL, NULL, NULL, NULL };
	BuildArticleListStruct build;
	debug_enter ("articlelist_repopulate");

	build.group = group;
	build.article_buf = article_buf;
	build.article_qty = article_qty;
	build.item = item;
	build.index = 0;
	build.root = NULL;

	status_item_emit_status (item, _("Updating article display"));

	/* clear out old */
	clear_hash_table ();
	pan_lock();
	gtk_clist_freeze (GTK_CLIST(tree));
	gtk_clist_clear (GTK_CLIST(tree));
	
	/* add root node */
	text[2] = (char*) group_get_readable_name (group);
	build.root = gtk_ctree_insert_node (tree, NULL, NULL,
	                                    text, GNOME_PAD_SMALL,
	                                    articlelist_closed_pixmap->pixmap,
	                                    articlelist_closed_pixmap->mask,
	                                    articlelist_open_pixmap->pixmap,
	                                    articlelist_open_pixmap->mask,
	                                    FALSE, FALSE);
	gtk_ctree_node_set_row_data (tree, build.root, NULL);
	articlelist_set_node_style_nolock (build.root, NULL);
	pan_unlock ();

	add_nodes (&build);

	pan_lock ();
	gtk_ctree_expand (tree, build.root);
	gtk_clist_thaw (GTK_CLIST(tree));
	gtk_widget_queue_resize (Pan.article_ctree);
	pan_unlock ();

	/* set the selection */
	if (sel_articles!=NULL && sel_articles->len)
	{
		gint i;
		GPtrArray * tmp;

		tmp = g_ptr_array_new ();
		pan_g_ptr_array_reserve (tmp, sel_articles->len);
		for (i=0; i<sel_articles->len; ++i) {
			Article * a = ARTICLE(g_ptr_array_index(sel_articles,i));
			GtkCTreeNode * node = articlelist_get_node_from_message_id (article_get_message_id(a));
			if (node != NULL)
				g_ptr_array_add (tmp, node);
		}

		if (tmp->len)
			articlelist_set_selected_nodes ((GtkCTreeNode**)tmp->pdata, tmp->len);

		/* cleanup */
		g_ptr_array_free (tmp, TRUE);
	}

	debug_exit ("articlelist_repopulate");
}

/***
****
****
****  POPULATING THE TREE:  THE CONTROLLERS
****
****
***/

void
articlelist_refresh (StatusItem* item)
{
	debug_enter ("articlelist_refresh");

	if (my_group != NULL)
	{
		GPtrArray * sel;
		GPtrArray * all;

		pthread_mutex_lock (&article_ctree_lock);

		all = group_get_article_array (my_group);
		status_item_emit_init_steps (item, all->len*2);
		sel = articlelist_get_selected_articles ();
		sort_articles ((Article**)all->pdata, all->len, abs(sort_type), sort_type>0);
		articlelist_repopulate (my_group, (Article**)all->pdata, all->len, item, sel);

		pthread_mutex_unlock (&article_ctree_lock);
	
		g_ptr_array_free (all, TRUE);
		g_ptr_array_free (sel, TRUE);
	}

	debug_exit ("articlelist_refresh");
}

static void
articlelist_set_contents (Group        * group,
			  StatusItem   * status)
{
	int i;
	GPtrArray * all;
	debug_enter ("articlelist_set_contents");

	g_return_if_fail (group != NULL);

	pthread_mutex_lock (&article_ctree_lock);

	/* get sorting information */
	i = group->sort_style;
	if (!i || abs(i)>ARTICLE_SORT_TYPES_QTY)
		i = -ARTICLE_SORT_DATE;
	articlelist_set_sort_type_nosort (i, TRUE);

	/* let the user know what the heck we're doing */
	if (status != NULL) {
		int steps = 0;
		if (my_group!=NULL)
			steps += my_group->article_qty; /* for saving */
		if (group!=NULL)
			steps += group->article_qty * 2; /* thread, node */
		if (group!=NULL)
			steps += group->article_qty; /* load */
		status_item_emit_init_steps (status, steps);
	}

	/* we ref the new articles before unreffing the old, in case this
	   is a forced reload where new==old.  In this instance putting the
	   ref first keeps us from having to free, then reload the articles */
	group_ref_articles (group, status);
	clear_article_buffer (status);
	my_group = group;
	group_thread_if_needed (my_group, status);

	all = group_get_article_array (my_group);
	fire_group_changed (my_group);

	/* filter */
	if (status != NULL)
		status_item_emit_status_va (status, _("Filtering %u articles"), all->len);
	apply_filter_tests (all);

	/* sort */
	sort_articles ((Article**)all->pdata, all->len, abs(sort_type), sort_type>0);

	/* add to tree */
	status_item_emit_status (status, _("Displaying article list"));
	articlelist_repopulate (group, (Article**)all->pdata, all->len, status, NULL);

	pthread_mutex_unlock (&article_ctree_lock);

	/* cleanup */
	g_ptr_array_free (all, TRUE);
	debug_exit ("articlelist_set_contents");
}

/***
****
****
****  SETTING THE CURRENT GROUP
****
****
***/

static gchar*
set_current_group_describe (const StatusItem* item)
{
	return g_strdup (_("Loading Group Headers"));
}

/**
 * This is an ugly little function that's scary to touch.
 */
static void
articlelist_set_group_maybe (Group * g, gboolean force_reload)
{
	StatusItem * status;
	debug_enter ("articlelist_set_group_maybe");

	/* if nothing to do, then quit */
	if (g==my_group && !force_reload)
		return;

	pthread_mutex_lock (&group_mutex);

	/* set up a status item so that the users will see what's happening.. */
	status = STATUS_ITEM(status_item_new(set_current_group_describe));
	status_item_set_active (status, TRUE);

	if (g!=NULL)
		group_ref_articles (g, status);

	if (g!=NULL && g->article_qty!=0) /* group & cached headers exist */
	{
		Group * prev_g = my_group;
		articlelist_set_contents (g, status);
		if (prev_g!=my_group && fetch_new_on_group_load && !group_is_folder(g))
			queue_add (TASK(task_headers_new (g, HEADERS_NEW)));
	}
	else /* probably have to hit the news server for articles */
	{
		if (g!=NULL)
		{
			/* make sure the ref persists after we leave this func */
			group_ref_articles (g, status);

			/* try & download headers for the group */
			if (!group_is_folder (g))
				dialog_download_headers (g);
		}

		clear_article_buffer (NULL);
		my_group = g;
		fire_group_changed (my_group);
	}

	if (g!=NULL)
		group_unref_articles (g, status);

	pthread_mutex_unlock (&group_mutex);

	status_item_set_active (status, FALSE);
	pan_object_unref(PAN_OBJECT(status));
	debug_exit ("articlelist_set_group_maybe");
}

void
articlelist_set_group_thread (Group * group)
{
	pthread_t thread;
	pthread_create (&thread, NULL, (void*)articlelist_set_group, group);
	pthread_detach (thread);
}
void
articlelist_set_group (Group * group)
{
	articlelist_set_group_maybe (group, FALSE);
}
void
articlelist_reload (void)
{
	articlelist_set_group_maybe (my_group, TRUE);
}

/***
****
****
****  UPDATING EXISTING NODES: THE DIRTY WORK
****
****
***/

static void 
articlelist_update_node_fast_nolock (GtkCTreeNode   * node,
                                     const Article  * article)
{
	GtkCTree * ctree = GTK_CTREE(Pan.article_ctree);
	GnomePixmap * col_1 = get_column_1_pixmap (article);
	GnomePixmap * col_2 = get_column_2_pixmap (article);

	gtk_ctree_set_node_info (ctree, node,
	                         article_get_subject(article),
	                         GNOME_PAD_SMALL,
	                         NULL, NULL, NULL, NULL,
	                         GTK_CTREE_ROW(node)->is_leaf,
	                         GTK_CTREE_ROW(node)->expanded );

	gtk_ctree_node_set_pixmap (ctree, node, 0, col_1->pixmap, col_1->mask);
	if (col_2 == NULL)
		gtk_ctree_node_set_text (ctree, node, 1, "");
	else
		gtk_ctree_node_set_pixmap (ctree, node, 1, col_2->pixmap, col_2->mask);

	articlelist_set_node_style_nolock (node, article);
}
static GtkCTreeNode*
articlelist_get_node_from_message_id (const char* message_id)
{
	GtkCTreeNode * retval = NULL;

	if (is_nonempty_string(message_id))
	{
		pthread_mutex_lock (&hash_mutex);
		retval = g_hash_table_lookup (messageid_to_node, message_id);
		pthread_mutex_unlock (&hash_mutex);
	}

	return retval;
}
static void
articlelist_update_node_impl (const Article* article, gboolean need_lock)
{
	GtkCTreeNode* node =
		articlelist_get_node_from_message_id (
			article_get_message_id (article));

        if (node != NULL)
	{
		/* update the article state */
		Article* real_article =
			articlelist_get_article_from_node (node);
		real_article->state = article->state;

		/* repaint */
		if (need_lock)
			pan_lock ();
                articlelist_update_node_fast_nolock (node, real_article);
		if (need_lock)
			pan_unlock ();
	}
}

static gint
articlelist_update_articles_idle (gpointer data)
{
	gint i;
	GPtrArray * a = (GPtrArray*) data;

	/* update the nodes */
	pan_lock_unconditional ();
	for (i=0; i<a->len; ++i)
	{
		const Article * article = ARTICLE(g_ptr_array_index(a,i));
		if (article->group == my_group)
			articlelist_update_node_impl (article, FALSE);
	}
	pan_unlock_unconditional ();

	/* cleanup */
	g_ptr_array_free (a, TRUE);
	return 0;
}

static void
articlelist_update_articles (const Article** a, gint qty)
{
	gint i;
	GPtrArray * tmp;

	/* make a list of all the articles that affect the ctree */
	tmp = g_ptr_array_new ();
	for (i=0; i<qty; ++i)
		if (articlelist_get_node_from_message_id (article_get_message_id(a[i])))
			g_ptr_array_add (tmp, (gpointer)a[i]);

	/* if the ctree needs updating, queue an update */
	if (!tmp->len) {
		g_ptr_array_free (tmp, TRUE);
	} else {
		pan_lock ();
		gtk_idle_add (articlelist_update_articles_idle, (gpointer)tmp);
		pan_unlock ();
	}
}

/***
****
****
****  MANAGING UNREAD CHILDREN COUNTS
****
****
***/

static void
articlelist_unread_inc_nolock (GtkCTreeNode* node, int inc)
{
	GtkCTreeNode * child = node;
	debug_enter ("articlelist_unread_inc_nolock");

	/* update the node & parents */
	while (node != NULL)
	{
		Article * a = articlelist_get_article_from_node (node);
		if (a == NULL)
			break;

		/* all the parents have one less/more unread child */
		if (child != node)
			a->unread_children = MAX(a->unread_children+inc, 0);

		articlelist_update_node_fast_nolock (node, a);

		node = GTK_CTREE_ROW(node)->parent;
	}

	debug_exit ("articlelist_unread_inc_nolock");
}

static gint
articlelist_articles_read_changed_idle (gpointer data)
{
	gint i;
	GPtrArray * articles;
	GtkCList * clist;
	debug_enter ("articlelist_articles_read_changed_idle");

 	articles = (GPtrArray *) data;
	clist = GTK_CLIST(Pan.article_ctree);

	pan_lock_unconditional ();
	gtk_clist_freeze (clist);
	for (i=0; i!=articles->len; ++i)
	{
		const Article * article = ARTICLE(g_ptr_array_index(articles,i));
		const Article * a = NULL;
		GtkCTreeNode* node = NULL;

		/* find the matching node & article */
		node = articlelist_get_node_from_message_id (article_get_message_id(article));
		if (node != NULL)
			a = articlelist_get_article_from_node (node);

		/* if we've got a match, update the node */
		if (a != NULL) {
			int inc = article_is_read (article) ? -1 : 1;
			articlelist_unread_inc_nolock (node, inc);
		}
	}
	gtk_clist_thaw (clist);
	pan_unlock_unconditional ();

	debug_exit ("articlelist_articles_read_changed_idle");
	return 0;
}

static void
articlelist_articles_read_changed (const Article ** articles, gint len)
{
	debug_enter ("articlelist_articles_read_changed");

	g_return_if_fail (articles!=NULL);
	g_return_if_fail (len>=1);

	if (my_group == articles[0]->group)
	{
		GPtrArray * a = g_ptr_array_new ();
		pan_g_ptr_array_assign  (a, (gpointer*)articles, len);
		pan_lock ();
		gtk_idle_add (articlelist_articles_read_changed_idle, a);
		pan_unlock ();
	}

	debug_exit ("articlelist_articles_read_changed");
}

/***
****
****
****  DELETING ARTICLES
****
****
***/

static void
remove_duplicate_articles (GPtrArray * articles)
{
	gint i;
	GHashTable * hash;

	/* sanity claus */
	g_return_if_fail (articles!=NULL);
	g_return_if_fail (articles->len>0);
	if (articles->len==1) return;

	/* remove duplicates */
	hash = g_hash_table_new (g_str_hash, g_str_equal);
	for (i=0; i<articles->len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(articles,i));
		g_hash_table_insert (hash, (gpointer)article_get_message_id(a), a);
	}
	pan_hash_to_ptr_array  (hash, articles);
	g_hash_table_destroy (hash);
}

/**
 * This is for selecting the next non-deleted article.
 * Given the list of articles (about to be deleted), we return the next
 * article in the articlelist which is not in the articles array.
 */
static Article*
articlelist_get_next_article_not_in_list (GPtrArray * articles)
{
	guint i = 0;
	Article * a = NULL;
	Article * retval = NULL;
	Article * first = NULL;
	GHashTable * h = NULL;
	GtkCTreeNode * node = NULL;

	/* sanity clause */
	g_return_val_if_fail (articles!=NULL, NULL);
	g_return_val_if_fail (articles->len!=0, NULL);

	/* build a hashtable of doomed articles for quick lookup */
	h = g_hash_table_new (g_direct_hash, g_direct_equal);
	for (i=0; i<articles->len; ++i) {
		gpointer g = g_ptr_array_index (articles, i);
		if (g != NULL)
			g_hash_table_insert (h, g, g);
	}

	/* now walk through articles, looking for a doomed one */
	node = articlelist_get_selected_node ();
	first = articlelist_get_article_from_node (node);
	while (retval==NULL)
	{
		node = find_next_article (node);
		a = articlelist_get_article_from_node (node);
		if (a==NULL || a==first) /* end of list */
			break;
		if (g_hash_table_lookup (h, a) == NULL)
			retval = a;
	}

	/* cleanup */
	g_hash_table_destroy (h);
	return retval;
}


/**
 * Pairs of Group* and GArrays of gulong article numbers to remove from
 * the group.  We pass these arrays on to group_remove_crossposts(),
 * then free the array because we're done with it.
 * @see articlelist_delete_articles
 */
static void
articlelist_remove_xrefs_ghfunc (gpointer key,
                                 gpointer value,
                                 gpointer user_data)
{
	Group * g = GROUP(key);
	GArray * numbers = (GArray*) value;
	group_remove_crossposts (g, (gulong*)numbers->data, numbers->len);
	g_array_free (numbers, TRUE);
}

/**
 * data is a GHashTable of (key=group,val=GArray) where we fill those
 * GArrays with the article numbers of this crosspost.
 */
static void
articlelist_remove_xrefs_xreffunc (Group   * group,
                                   gulong    number,
                                   gpointer  data)
{
	GHashTable * hash = (GHashTable*) data;

	if (group_is_subscribed (group))
	{
		GArray * a = g_hash_table_lookup (hash, group);
		if (a == NULL)
			g_hash_table_insert (hash,
			                     group,
			                     a = g_array_new (FALSE, FALSE, sizeof(gulong)));
		g_array_append_val (a, number);
	}
}

/**
 * @see remove_duplicate_articles
 * @see articlelist_remove_xrefs_xreffunc
 * @see articlelist_remove_xrefs_ghfunc
 */
static void
articlelist_delete_articles (GPtrArray * articles)
{
	debug_enter ("articlelist_delete_articles");

	if (articles->len)
	{
		guint i = 0;
		Article * a = NULL;
		GtkCTreeNode * n = NULL;
		GHashTable * hash = NULL;

		/* uniqueness check */
		remove_duplicate_articles (articles);

		/* if these articles are posted elsewhere, delete those crossposts too */
		hash = g_hash_table_new (g_direct_hash, g_direct_equal);
		for (i=0; i<articles->len; ++i) {
			Article * a = ARTICLE(g_ptr_array_index(articles,i));
			article_xref_foreach (a, articlelist_remove_xrefs_xreffunc, hash, TRUE);
		}
		g_hash_table_foreach (hash, articlelist_remove_xrefs_ghfunc, NULL);
		g_hash_table_destroy (hash);

		/* find the next nondoomed article */
		a = articlelist_get_next_article_not_in_list (articles);

		/* delete the articles in this group */
		group_remove_articles (my_group, articles);

		/* select the next article */
		if (a != NULL) {
			n = articlelist_get_node_from_message_id (article_get_message_id(a));
			articlelist_set_selected_nodes (&n, 1);
		}

		/* log */
		log_add_va (LOG_INFO, _("Deleted %u articles from `%s'"),
		            articles->len,
		            group_get_readable_name(my_group));
	}

	debug_exit ("articlelist_delete_articles");
}

void
articlelist_all_delete (void)
{
	GPtrArray * all = group_get_article_array (my_group);
	articlelist_delete_articles (all);
	g_ptr_array_free (all, TRUE);
}

void
articlelist_selected_delete_thread (void)
{
	GPtrArray * sel;
	GPtrArray * all;
	debug_enter ("articlelist_selected_delete_thread");

       	sel = articlelist_get_selected_articles ();
       	all = articlelist_get_unique_threads (sel, GET_SUBTHREAD);
	articlelist_delete_articles (all);
	g_ptr_array_free (all, TRUE);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("articlelist_selected_delete_thread");
}

void
articlelist_selected_delete (void)
{
	GPtrArray * articles;
	int i, len;
	debug_enter ("articlelist_selected_delete");

	/* if it's a multipart, delete the children too */
       	articles = articlelist_get_selected_articles ();
	for (i=0, len=articles->len; i!=len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(articles,i));
		if (a->part==1 && a->parts>1) {
			GSList * l;
			for (l=a->threads; l!=NULL; l=l->next)
				if (ARTICLE(l->data)->part > 1)
					g_ptr_array_add (articles, l->data);
		}
	}

	/* delete the articles */
	articlelist_delete_articles (articles);

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	debug_exit ("articlelist_selected_delete");
}

/***
****
****  LIVING WITH ACACHE
****
***/

static gint
articlelist_acache_changed_idle (gpointer data)
{
	gchar * message_id;
	GtkCTreeNode * node;
	debug_enter ("articlelist_acache_changed_idle");

 	message_id = (gchar*) data;
	node = articlelist_get_node_from_message_id (message_id);
        if (node != NULL)
	{
		Article * a = articlelist_get_article_from_node (node);
		pan_lock_unconditional ();
                articlelist_update_node_fast_nolock (node, a);
		pan_unlock_unconditional ();
	}

	g_free (message_id);
	debug_exit ("articlelist_acache_changed_idle");
	return 0;
}
static gint
articlelist_acache_changed_cb (gpointer call_object,
                               gpointer call_arg,
                               gpointer user_data)
{
	const gchar * message_id;
	debug_enter ("articlelist_acache_changed_cb");

 	message_id = (const gchar*) call_object;
	if (articlelist_get_node_from_message_id (message_id) != NULL)
	{
		pan_lock ();
		gtk_idle_add (articlelist_acache_changed_idle, g_strdup(message_id));
		pan_unlock ();
	}

	debug_exit ("articlelist_acache_changed_cb");
	return 0;
}

/***
****
****
****  SELECTION / KEYPRESS / BUTTONPRESS HANDLING
****
****
***/

/**
 * The following buttonpress, keypress, and selection callbacks are all here
 * to interact with each other to do one thing: force this damn GtkCTree to
 * have the UI behavior that we want:
 * 
 * (1) Never activate an article during a group selection
 * 
 * (2) If only a single article is selected by single mouse click or
 *     by up/down arrow, and the 'single click loads' prefs is on,
 *     or if it's selected by a double click, then load the article.
 *
 * The problem with (1) is that the GtkCTree fires a selection event for each
 * row selected, rather than once for the whole range, so there's no way to
 * tell if the first selection event is a single or the beginning of a range.
 * The workaround is to use an idle func that gets invoked after all the
 * selections are done.  It asks the tree what's selected and uses that in
 * the "article selection changed" pan callback.  (2) is handled by the
 * button_click_count variable, which keeps click information from the
 * kepress/buttonpress callbacks and uses it in the selection callback.
 */
static gint button_click_count = -1;
static gint mb = -1;



static gint
articlelist_selection_changed_cb (gpointer call_object,
                                  gpointer call_arg,
                                  gpointer user_data)
{
	GPtrArray * articles;
	Article * single_article = NULL;
	debug_enter ("articlelist_selection_changed_cb");

	/* load the specified article */
       	articles = pan_g_ptr_array_dup ((GPtrArray*)call_arg);
	if (articles!=NULL && articles->len==1)
		single_article = ARTICLE(g_ptr_array_index(articles,0));
        if ((single_article!=NULL)
		&& ((button_click_count==2) /* double click */
			|| (button_click_count==1 /*&& Pan.viewmode==GUI_PANED*/
				&& (  (mb==2 && articlelist_mb2_action==CLICK_ACTION_LOAD)
				    ||(mb==1 && articlelist_mb1_action==CLICK_ACTION_LOAD)))))
	{
		pan_warn_if_fail (single_article->group != NULL);
		pan_warn_if_fail (single_article->group->server != NULL);
		gui_page_set (MESSAGE_PAGE, Pan.text);
		queue_add (TASK(task_body_new_defaults(single_article)));
	}

	button_click_count = -1;
	mb = -1;

	debug_exit ("articlelist_selection_changed_cb");
	return 0;
}

static void
articlelist_key_press (GtkWidget      * widget,
                       GdkEventKey    * event,
                       gpointer         data)
{
	switch (event->keyval)
	{
		case GDK_Up: case GDK_Down:
		case GDK_Left: case GDK_Right:
		case GDK_Page_Up: case GDK_Page_Down:
			button_click_count = 1;
			break;
		default:
			button_click_count = -1;
	}
}

static void
articlelist_button_press (GtkWidget* widget, GdkEventButton* bevent)
{
	switch (bevent->button)
	{
		case 1:
		case 2:
			mb = bevent->button;
			if (bevent->type == GDK_2BUTTON_PRESS)
				button_click_count = 2;
			else
				button_click_count = 1;
			break;
		case 3:
			articlelist_menu_popup (bevent);
			break;
	}
}

static gboolean select_callback_pending = FALSE;
 
static gint
tree_select_row_idle (gpointer data)
{
	GPtrArray * articles;
	debug_enter ("tree_select_row_idle");

	/* let everyone know that the selection has changed */
       	articles = articlelist_get_selected_articles ();
	pan_callback_call (articlelist_selection_changed,
			   Pan.article_ctree,
			   articles);

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	select_callback_pending = FALSE;
	debug_exit ("tree_select_row_idle");
	return FALSE;
}
static void
tree_select_row_cb (GtkCTree     *tree,
		    GtkCTreeNode *node,
		    gint          column,
		    gpointer      data)
{
	if (!select_callback_pending)
	{
		select_callback_pending = TRUE;
		gtk_idle_add (tree_select_row_idle, NULL);
	}
}

/***
****
****
****  ccc
****
****
***/

static void
articlelist_click_column (GtkCList* clist, int n)
{
	int type;
	switch (n)
	{
		case 0:
		case 1:
			return;
			break;
		case 2:
			type = ARTICLE_SORT_SUBJECT;
			break;
		case 3:
			type = ARTICLE_SORT_LINES;
			break;
		case 4:
			type = ARTICLE_SORT_AUTHOR;
			break;
		case 5:
			type = ARTICLE_SORT_DATE;
			break;
		default:
			pan_warn_if_reached();
			type = -ARTICLE_SORT_DATE;
			break;
	}

	if (type == sort_type)
		type = -type;

	articlelist_set_sort_type (type);
}

/***
****  Callbacks
***/

static int
articlelist_group_marked_read_cb (gpointer group, gpointer unused1, gpointer unused2)
{
	if (my_group!=NULL && my_group==GROUP(group))
		articlelist_reload ();

	return 0;
}

static int
articlelist_group_emptied_cb (gpointer group_data, gpointer foo, gpointer bar)
{
	const Group * group = GROUP(group_data);

	if (my_group!=NULL && my_group==group)
		articlelist_set_group (NULL);
	
	return 0;
}


static int
articlelist_groups_removed_cb (gpointer server_data,
                               gpointer groups_array,
                               gpointer unused)
{
	guint i;
	gboolean my_group_removed = FALSE;
	const GPtrArray * groups = (const GPtrArray*) groups_array;

	for (i=0; !my_group_removed && i<groups->len; ++i)
		if (GROUP(g_ptr_array_index(groups,i)) == my_group)
			my_group_removed = TRUE;

	if (my_group_removed)
		articlelist_set_group (NULL);

	return 0;
}

static gint
articlelist_articles_changed_cb (gpointer event, gpointer foo, gpointer bar)
{
	ArticleChangeEvent * e = (ArticleChangeEvent*) event;
	switch (e->type)
	{
		case ARTICLE_CHANGED_READ:
			articlelist_articles_read_changed ((const Article**)e->articles, e->article_qty);
			break;
		case ARTICLE_CHANGED_DIRTY:
			articlelist_update_articles ((const Article**)e->articles, e->article_qty);
			break;
		default:
			pan_warn_if_reached ();
			break;
	}
	return 0;
}                                                                                                                               

/***
****
***/

static void
article_ctree_destroy_cb (void)
{
	/* stop listening to events */
	pan_callback_remove (article_get_articles_changed_callback(),
	                     articlelist_articles_changed_cb, NULL);
	pan_callback_remove (server_get_groups_removed_callback(),
	                     articlelist_groups_removed_cb, NULL);
	pan_callback_remove (acache_bodies_added_callback,
	                     articlelist_acache_changed_cb, NULL);
	pan_callback_remove (acache_bodies_removed_callback,
	                     articlelist_acache_changed_cb, NULL);
	pan_callback_remove (articlelist_selection_changed,
	                     articlelist_selection_changed_cb, NULL);
	pan_callback_remove (group_get_group_marked_read_callback(),
	                     articlelist_group_marked_read_cb, NULL);
	pan_callback_remove (group_get_group_emptied_callback(),
	                     articlelist_group_emptied_cb, NULL);

	gtk_widget_destroy (GTK_WIDGET(articlelist_closed_pixmap));
	gtk_widget_destroy (GTK_WIDGET(articlelist_open_pixmap));
	gtk_widget_destroy (GTK_WIDGET(binary_decoded_pixmap));
	gtk_widget_destroy (GTK_WIDGET(decode_queue_pixmap));
	gtk_widget_destroy (GTK_WIDGET(error_pixmap));
	gtk_widget_destroy (GTK_WIDGET(flagged_for_download_pixmap));
	gtk_widget_destroy (GTK_WIDGET(mp_complete_pixmap));
	gtk_widget_destroy (GTK_WIDGET(mp_incomplete_pixmap));
	gtk_widget_destroy (GTK_WIDGET(old_mp_complete_pixmap));
	gtk_widget_destroy (GTK_WIDGET(old_mp_incomplete_pixmap));

	pan_callback_free (articlelist_sort_changed);
	pan_callback_free (articlelist_thread_changed);
	pan_callback_free (articlelist_selection_changed);
}

/*
 * Due to a bug in GtkCList (as of 1.2) we can't moveto a window that
 * isn't drawable.  A object data flag is set if we wanted to move that
 * when we map (right here) we can take care of the move
 */
static void
map_articlelist_cb (GtkWidget * w, gpointer user_data)
{
	if (gtk_object_get_data(GTK_OBJECT(w), "pan_moveto_hack") != NULL)
	{
		const GList * list = GTK_CLIST(w)->selection;
		if (list != NULL)
		{
			pan_lock ();
			gtk_ctree_node_moveto (GTK_CTREE(w),
			                       GTK_CTREE_NODE(list->data),
			                       0, (gfloat)0.5, (gfloat)0.0);
			pan_unlock ();
		}

		gtk_object_remove_data (GTK_OBJECT(w), "pan_moveto_hack");
	}
}


static void
tree_toggle_focus_row_cb (GtkCList   * clist,
                          gpointer     user_data)
{
	/* This is just a hack to allow space-reading when the article ctree
	   has focus.  The space key is used in clists to toggle the row
	   focus, so if we don't stop the signal here ctree calls
	   resync_selection, which reselects the focused node, and makes
	   space reading loop back on the same article each time. */
	gtk_signal_emit_stop_by_name (GTK_OBJECT(clist), "toggle_focus_row");
}


/*--------------------------------------------------------------------
 * generate the listing of articles, for the "Articles" tab
 *--------------------------------------------------------------------*/
GtkWidget*
create_articlelist_ctree (void)
{
	int i;
	GtkWidget * w;
	GtkCList * list;
	GtkCTree * tree;
	GtkWidget * s_window;
	GtkWidget * vbox;
	const char * titles[COLUMNS];

	/* callbacks */
	articlelist_sort_changed = pan_callback_new ();
	articlelist_thread_changed = pan_callback_new ();
	articlelist_selection_changed = pan_callback_new ();

	pan_callback_add (article_get_articles_changed_callback(),
	                  articlelist_articles_changed_cb, NULL);
	pan_callback_add (server_get_groups_removed_callback(),
	                  articlelist_groups_removed_cb, NULL);
	pan_callback_add (acache_bodies_added_callback,
	                  articlelist_acache_changed_cb, NULL);
	pan_callback_add (acache_bodies_removed_callback,
	                  articlelist_acache_changed_cb, NULL);
	pan_callback_add (articlelist_selection_changed,
	                  articlelist_selection_changed_cb, NULL);
	pan_callback_add (group_get_group_marked_read_callback(),
	                  articlelist_group_marked_read_cb, NULL);
	pan_callback_add (group_get_group_emptied_callback(),
	                  articlelist_group_emptied_cb, NULL);

	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GNOME_PAD_SMALL);

	/* filter */
	gtk_box_pack_start (GTK_BOX(vbox), article_toolbar_new(), FALSE, FALSE, 0);

	/* get the titles */
	for (i=0; i<COLUMNS; ++i)
		titles[i] = column_to_title (i);

	/* load the pixmaps */
	old_read_pixmap = GNOME_PIXMAP(gnome_pixmap_new_from_xpm_d(old_read_xpm));
	old_unread_pixmap = GNOME_PIXMAP(gnome_pixmap_new_from_xpm_d(old_unread_xpm));
	new_read_pixmap =  GNOME_PIXMAP(gnome_pixmap_new_from_xpm_d(new_read_xpm));
	new_unread_pixmap =  GNOME_PIXMAP(gnome_pixmap_new_from_xpm_d(new_unread_xpm));

	flagged_for_download_pixmap = GNOME_PIXMAP(gnome_pixmap_new_from_xpm_d (flagged_xpm));
	articlelist_closed_pixmap = GNOME_PIXMAP (gnome_stock_pixmap_widget_new (Pan.window, GNOME_STOCK_MENU_BOOK_RED));
	articlelist_open_pixmap = GNOME_PIXMAP (gnome_stock_pixmap_widget_new (Pan.window, GNOME_STOCK_MENU_BOOK_OPEN));
        article_cached_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (disk_xpm));
	decode_queue_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (bluecheck_xpm));
	error_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (x_xpm));
	binary_decoded_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (binary_xpm));
	mp_complete_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (binary_complete_xpm));
	mp_incomplete_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (binary_incomplete_xpm));
	old_mp_complete_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (old_binary_complete_xpm));
	old_mp_incomplete_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (old_binary_incomplete_xpm));

	clear_hash_table ();

	/* create the widget */
	Pan.article_ctree = w = gtk_ctree_new_with_titles (COLUMNS, 2, (gchar**)titles);
	tree = GTK_CTREE(w);
	list = GTK_CLIST(w);
	list->button_actions[1] = list->button_actions[0];
	gtk_signal_connect (GTK_OBJECT(w), "tree-select-row", tree_select_row_cb, NULL);
	gtk_signal_connect (GTK_OBJECT(w), "tree-unselect-row", tree_select_row_cb, NULL);
	gtk_signal_connect (GTK_OBJECT(w), "toggle_focus_row", tree_toggle_focus_row_cb, NULL);

	/* wrap it in a scrolled window */
	s_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (
		GTK_SCROLLED_WINDOW(s_window),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER(s_window), Pan.article_ctree);

	/* ui settings */
	gtk_ctree_set_line_style (tree, GTK_CTREE_LINES_DOTTED);
	gtk_clist_set_selection_mode (list, GTK_SELECTION_EXTENDED);
	gtk_clist_set_column_justification (list, 3, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_width (list, 0, 16);
	gtk_clist_set_column_width (list, 1, 16);
	gtk_clist_set_column_width (list, 2, 220);
	gtk_clist_set_column_width (list, 3, 30);
	gtk_clist_set_column_width (list, 4, 120);
	gtk_clist_set_column_width (list, 5, 40);

	/* create the right click popup menu */
	article_ctree_menu = gnome_popup_menu_new (articlelist_menu);

	/* connect signals */
	gtk_signal_connect (GTK_OBJECT(tree), "button_press_event",
	                    GTK_SIGNAL_FUNC(articlelist_button_press), NULL);
	gtk_signal_connect (GTK_OBJECT(tree), "key_press_event",
	                    GTK_SIGNAL_FUNC(articlelist_key_press), NULL);
	gtk_signal_connect (GTK_OBJECT(tree), "click_column",
	                    GTK_SIGNAL_FUNC(articlelist_click_column), NULL);
	gtk_signal_connect (GTK_OBJECT(tree), "destroy",
	                    GTK_SIGNAL_FUNC(article_ctree_destroy_cb), NULL);

	pan_callback_add (group_get_articles_removed_callback(),
			  group_articles_removed_cb,
			  NULL);

	pan_callback_add (group_get_articles_added_callback(),
			  group_articles_added_cb,
			  NULL);

	/* due to a bug in GtkCList (as of v1.2) we can't moveto an
	 * undrawable window -- so we remember to moveto upon mapping */
	gtk_signal_connect_after (GTK_OBJECT(tree), "map",
	                          GTK_SIGNAL_FUNC(map_articlelist_cb), NULL);

	gtk_box_pack_start (GTK_BOX(vbox), s_window, TRUE, TRUE, 0);

	return vbox;
}
void
articlelist_reset_style (void)
{
	rebuild_style = TRUE;
	articlelist_reload ();
}


/**
***
**/

void
articlelist_expand_all_threads (void)
{
	GtkCTree * tree;
	GtkCTreeNode * node;
	GtkCTreeNode * sel;
	debug_enter ("articlelist_expand_all_threads");

 	tree = GTK_CTREE(Pan.article_ctree);
	pan_lock ();
	sel = articlelist_get_selected_node ();
	node = gtk_ctree_node_nth (tree, 0);
	if (node != NULL) {
		gtk_ctree_expand_recursive (tree, node);
		if (sel != NULL)
			articlelist_set_selected_nodes (&sel, 1);
	}
	pan_unlock ();
	debug_exit ("articlelist_expand_all_threads");
}

void
articlelist_collapse_all_threads (void)
{
	GtkCTree * tree;
	GtkCTreeNode * root;
	debug_enter ("articlelist_collapse_all_threads");

	tree = GTK_CTREE(Pan.article_ctree);

	/* get the root of the tree */
	pan_lock ();
	root = gtk_ctree_node_nth (tree, 0);
	pan_unlock ();

	/* if the tree's not empty, collapse the threads */
	if (root != NULL)
	{
		GtkCTreeNode * sel; 

		/* find the parent node that will be selected when we're done */
		sel = articlelist_get_selected_node ();
		while (sel!=NULL && GTK_CTREE_ROW(sel)->parent!=root)
			sel = GTK_CTREE_ROW(sel)->parent;

		/* collapse the threads */
		pan_lock ();
		gtk_ctree_collapse_recursive (tree, root);
		gtk_ctree_expand (tree, root);
		pan_unlock ();

		/* select the topmost parent of the old selection */
		if (sel != NULL)
			articlelist_set_selected_nodes (&sel, 1);
	}

	debug_exit ("articlelist_collapse_all_threads");
}

/**
***
***  ARTICLE FILTERING
***
**/

static gchar*
filter_selected_describe (const StatusItem* item)
{
	const gchar * name;

	/* find the current filter's name */
	name = NULL;
	if (IS_FILTER_TOP(my_filter))
		name = FILTER_TOP(my_filter)->name;

	/* return the description string */
	if (is_nonempty_string(name))
		return g_strdup_printf (_("Applying Filter `%s' to Group"), name);
	else
		return g_strdup (_("Filtering Articles"));
}
static void
filter_changed_thread (void *data)
{
	debug_enter ("filter_changed_thread");

	if (my_group != NULL)
	{
		GPtrArray * all;
		StatusItem * status = NULL;

		/* create a new status item to get sort/thread messages */
		status = STATUS_ITEM(status_item_new(filter_selected_describe));
		status_item_set_active (status, TRUE);

		/* filter */
		all = group_get_article_array (my_group);
		if (status != NULL)
			status_item_emit_status_va (status, _("Filtering %u articles"), all->len);
		apply_filter_tests (all);
		g_ptr_array_free (all, TRUE);

		/* repopulate */
		articlelist_refresh (status);

		/* clean out the status item */
		status_item_set_active (status, FALSE);
		pan_object_unref(PAN_OBJECT(status));
	}

	debug_exit ("filter_changed_thread");
}

void
articlelist_set_filter (const Filter * f, FilterShow show, gboolean rebuild)
{
	debug_enter ("articlelist_set_filter");

	if (my_group!=NULL && f!=NULL)
	{
		Filter * old;

		/* update filter variable */
		my_filter_show = show;
		old = my_filter;
		my_filter = filter_dup (f);
		if (old != NULL)
			pan_object_unref (PAN_OBJECT(old));

odebug1 ("new articlelist filter: \n\n[%s]\n\n", filter_xml_to_string (&my_filter,1));

		/* set a thread to rebuild the articlelist */
		if (rebuild) {
			pthread_t thread;
			pthread_create (&thread, 0, (void*)filter_changed_thread, 0);
			pthread_detach (thread);
		}
	}

	debug_exit ("articlelist_set_filter");
}


/**
***
***  ARTICLE THREADING
***
**/

static gchar*
thread_selected_describe (const StatusItem* item)
{
	return g_strdup (_("Threading Articles"));
}

static void
articlelist_set_threaded_thread (void* data)
{
	const gboolean threaded_on = GPOINTER_TO_INT(data);
	StatusItem *item = NULL;
	debug_enter ("articlelist_set_threaded_thread");

	/* create a new status item to get thread messages */
	item = STATUS_ITEM(status_item_new(thread_selected_describe));
	status_item_set_active (item, TRUE);

	/* update the articlelist */
	is_threaded = threaded_on;
	pan_callback_call (articlelist_thread_changed,
			   Pan.article_ctree, GINT_TO_POINTER(threaded_on));
	if (my_group!=NULL)
		articlelist_refresh (item);

        /* clean out the status item */
	status_item_set_active (item, FALSE);
	pan_object_unref(PAN_OBJECT(item));
	debug_exit ("articlelist_set_threaded_thread");
}

void
articlelist_set_threaded (gboolean threaded_on)
{
	if (threaded_on != is_threaded)
	{
		pthread_t thread;
		pthread_create (&thread,
				NULL, (void*)articlelist_set_threaded_thread,
				GINT_TO_POINTER(threaded_on));
		pthread_detach (thread);
	}
}

/**
***
***  POPUP MENU 
***
**/

static void
articlelist_menu_popup (GdkEventButton* bevent)
{
	const Article *article = articlelist_get_selected_article();
	const gboolean have_article = article!=NULL;
	const gboolean group_has_articles = GTK_CLIST(Pan.article_ctree)->rows;
	const gboolean is_folder = my_group!=NULL && group_is_folder (my_group);
	gint i;

odebug1 ("group_has_articles [%d]", (int)group_has_articles);

	pan_lock();

	/* MAIN MENU */
	i = -1;
	++i; /* read selected message */
	gtk_widget_set_sensitive (articlelist_menu[i].widget, have_article);
	++i; /* read selected message in new window */
	gtk_widget_set_sensitive (articlelist_menu[i].widget, have_article);
	++i; /* separator */
	++i; /* save */
	gtk_widget_set_sensitive (articlelist_menu[i].widget, group_has_articles);
	++i; /* watch thread */
	gtk_widget_set_sensitive (articlelist_menu[i].widget, group_has_articles);
	++i; /* ignore thread */
	gtk_widget_set_sensitive (articlelist_menu[i].widget, group_has_articles);
	++i; /* spamize */
	gtk_widget_set_sensitive (articlelist_menu[i].widget, group_has_articles);
	++i; /* separator */
	++i; /* mark read */
	gtk_widget_set_sensitive (articlelist_menu[i].widget, group_has_articles);
	++i; /* mark unread */
	gtk_widget_set_sensitive (articlelist_menu[i].widget, group_has_articles);
	++i; /* separator */
	++i; /* download bodies */
	gtk_widget_set_sensitive (articlelist_menu[i].widget, group_has_articles);
	++i; /* purge bodies */
	gtk_widget_set_sensitive (articlelist_menu[i].widget, group_has_articles);
	++i; /* separator */
	++i; /* folders */
	gtk_widget_set_sensitive (articlelist_menu[i].widget, group_has_articles);
	++i; /* delete */
	gtk_widget_set_sensitive (articlelist_menu[i].widget, group_has_articles);
	++i; /* separator */
	++i; /* select all */
	gtk_widget_set_sensitive (articlelist_menu[i].widget, group_has_articles);

	/* DELETE SUBMENU */
	i = -1;
	++i; /* i is delete selected article */
	gtk_widget_set_sensitive (delete_menu[i].widget, have_article);
	++i; /* i is delete selected articles + replies */
	gtk_widget_set_sensitive (delete_menu[i].widget, have_article);
	++i; /* delete all */
	gtk_widget_set_sensitive (delete_menu[i].widget, group_has_articles);
	++i; /* separator */
	++i; /* i is delete decoded binary attachment */
	gtk_widget_set_sensitive (delete_menu[i].widget, have_article);

	/* MARK READ SUBMENU */
	i = -1;
	++i; /* i is mark selected read */
	gtk_widget_set_sensitive (mark_read_menu[i].widget, have_article);
	++i; /* i is mark selected + replies read */
	gtk_widget_set_sensitive (mark_read_menu[i].widget, have_article);
	++i; /* i is mark selected threads */
	gtk_widget_set_sensitive (mark_read_menu[i].widget, have_article);
	++i; /* i is mark group */
	gtk_widget_set_sensitive (mark_read_menu[i].widget, group_has_articles);

	/* MARK UNREAD SUBMENU */
	i = -1;
	++i; /* i is mark selected */
	gtk_widget_set_sensitive (mark_unread_menu[i].widget, have_article);
	++i; /* i is mark selected + replies */
	gtk_widget_set_sensitive (mark_unread_menu[i].widget, have_article);
	++i; /* i is mark selected threads */
	gtk_widget_set_sensitive (mark_unread_menu[i].widget, have_article);
	++i; /* i is mark group */
	gtk_widget_set_sensitive (mark_unread_menu[i].widget, group_has_articles);

	/* DOWNLOAD BODIES SUBMENU */
	i = -1;
	++i; /* i is mark selected */
	gtk_widget_set_sensitive (bodies_menu[i].widget, have_article);
	++i; /* i is mark selected + replies */
	gtk_widget_set_sensitive (bodies_menu[i].widget, have_article);
	++i; /* i is mark group */
	gtk_widget_set_sensitive (bodies_menu[i].widget, group_has_articles);

	/* PURGE DOWNLOADED BODIES SUBMENU */
	i = -1;
	++i; /* i is purge selected */
	gtk_widget_set_sensitive (purge_bodies_menu[i].widget, have_article);
	++i; /* i is purge selected + replies */
	gtk_widget_set_sensitive (purge_bodies_menu[i].widget, have_article);
	++i; /* i is purge group */
	gtk_widget_set_sensitive (purge_bodies_menu[i].widget, group_has_articles);

	/* FOLDER SUBMENU */
	i = -1;
	++i; /* i is edit */
	gtk_widget_set_sensitive (folder_menu[i].widget, is_folder);

	/* show the popup */
	gtk_menu_popup (GTK_MENU (article_ctree_menu),
			NULL, NULL, NULL, NULL,
			bevent->button, bevent->time);
	pan_unlock();
}

/***
****  Events
***/

PanCallback*
articlelist_get_group_changed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_group_changed (Group * g)
{
	pan_callback_call (articlelist_get_group_changed_callback(),
	                   Pan.article_ctree,
	                   g);
}

