/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>
#include <pthread.h>

#include <pan/base/acache.h>
#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-file.h>
#include <pan/base/log.h>

PanCallback* acache_bodies_added_callback = NULL;
PanCallback* acache_bodies_removed_callback = NULL;

static gint acache_size = 0;
static pthread_mutex_t entries_mutex = PTHREAD_MUTEX_INITIALIZER;
static GMemChunk * acache_entry_chunk = NULL;
static GHashTable * filename_to_entry = NULL;

typedef struct
{
	gint refcount;
	gchar * filename;
	gchar * message_id;
	size_t size;
	time_t date;
}
AcacheEntry;

/***
****
***/

static gchar*
get_file_message_id (const gchar * filename)
{
	gchar buf [2048];
	gchar * retval;
	FILE * fp;

	/* try to find the line */
	*buf = '\0';
	fp = fopen (filename, "r");
	if (fp != NULL) {
		const gchar * key = "Message-I";
		const size_t key_len = strlen (key);
		gboolean found;
		for (found=FALSE; !found && fgets(buf,sizeof(buf),fp); ) {
			if (strlen(buf) < 3)
				break;
			if (!memcmp (key, buf, key_len))
				found = TRUE;
		}
		if (!found)
			*buf = '\0';
		fclose (fp);
	}

	/* parse the line */
	retval = NULL;
	if (*buf != '\0') {
		gchar * pch = strchr (buf, ':');
		if (pch != NULL) {
			++pch;
			while (*pch && isspace((int)*pch))
				++pch;
			retval = pch;
			while (*pch && !isspace((int)*pch))
				++pch;
			*pch = '\0';
		}
	}

	return g_strdup (retval);
}

/***
****
****  PATHS & FILENAMES
****
***/

static gchar*
acache_get_path (void)
{
	return g_strdup_printf ("%s/cache/", data_dir);
}

static gchar*
acache_get_full_path (const gchar * filename)
{
	gchar * retval = g_strdup_printf ("%s/cache/%s", data_dir, filename);
	pan_normalize_filename_inplace (retval);
	return retval;
}

static gchar*
acache_get_filename (const gchar * message_id)
{
	gchar * pch;
	gchar * retval;
	const gchar * cpch;

	/* sanity clause */
	g_return_val_if_fail (is_nonempty_string(message_id), NULL);

	/* build acache string */
       	pch = retval = g_new0 (gchar, strlen(data_dir) + strlen(message_id) + 64);
	pch += sprintf (pch, "%s/cache/", data_dir);
	for (cpch=message_id; *cpch!='\0'; ++cpch)
		if (*cpch!='%' && *cpch!='$' && *cpch!='<' && *cpch!='>' && *cpch!='/' && *cpch!='\\')
			*pch++ = *cpch;
	pch += sprintf (pch, ".msg");
	pan_normalize_filename_inplace (retval);

	return retval;
}

/***
****
****  INIT / SHUTDOWN
****
***/

void
acache_init (void)
{
	gchar * pch;
	debug_enter ("acache_init");

	/* ensure the cache directory exists */
	pch = acache_get_path ();
	directory_check (pch);
	g_free (pch);

	/* init the callbacks */
	acache_bodies_added_callback = pan_callback_new ();
	acache_bodies_removed_callback = pan_callback_new ();

	/* init the message id hash  */
	acache_entry_chunk = g_mem_chunk_create (AcacheEntry, 256, G_ALLOC_AND_FREE);
	filename_to_entry = g_hash_table_new (g_str_hash, g_str_equal);
	acache_size = 0;
	if (1)
	{
		struct dirent * dirent_p = NULL;
		gchar * dirpath = acache_get_path ();
		DIR * dir_p = opendir (dirpath);
		while (dir_p!=NULL && (dirent_p = readdir(dir_p)))
		{
			struct stat stat_p;
			gchar * path;

			if (!string_ends_with(dirent_p->d_name, ".msg"))
				continue;

			path = g_strdup_printf ("%s/%s", dirpath, dirent_p->d_name);
			if (stat(path, &stat_p) == 0)
			{
				gchar * message_id = get_file_message_id (path);

				if (is_nonempty_string (message_id))
				{
					AcacheEntry * entry = g_chunk_new (AcacheEntry, acache_entry_chunk);
					entry->filename = g_strdup (dirent_p->d_name);
					entry->message_id = g_strdup (message_id);
					entry->size = stat_p.st_size;
					entry->date = stat_p.st_mtime;
					entry->refcount = 0;
					g_hash_table_insert (filename_to_entry, entry->filename, entry);
					acache_size += entry->size;
				}

				g_free (message_id);
			}

			g_free (path);
		}

		if (dir_p!=NULL)
			closedir (dir_p);

		g_free (dirpath);

		log_add_va (LOG_INFO, _("Article cache contains %.1f MB in %d files"),
			(double)acache_size/(1024*1024),
			(int)g_hash_table_size(filename_to_entry));
	}

	debug_exit ("acache_init");
}

static void
acache_shutdown_ghfunc (gpointer key, gpointer val, gpointer data)
{
	AcacheEntry * entry = (AcacheEntry*)val;
	g_free (entry->filename);
	g_free (entry->message_id);
}
void
acache_shutdown (void)
{
	if (acache_flush_on_exit)
		acache_expire_all ();

	g_hash_table_foreach (filename_to_entry, acache_shutdown_ghfunc, NULL);
	g_hash_table_destroy (filename_to_entry);
	filename_to_entry = NULL;

	g_mem_chunk_destroy (acache_entry_chunk);
	acache_entry_chunk = NULL;
}

/***
****
****  CHECKIN / CHECKOUT
****
***/

static void
acache_file_update_refcount_nolock (const gchar ** message_ids,
                                    gint           message_id_qty,
                                    gint           inc)
{
	gint i;

	g_return_if_fail (message_ids != NULL);
	g_return_if_fail (message_id_qty >= 1);

	for (i=0; i<message_id_qty; ++i)
	{
		const gchar * message_id = message_ids[i];
		AcacheEntry * entry;
		gchar * filename;

		if (!is_nonempty_string (message_id))
			continue;

		filename = acache_get_filename (message_id);
		if (filename == NULL)
			continue;

		entry = g_hash_table_lookup (filename_to_entry, g_basename(filename));
		if (entry == NULL)
		{
			entry = g_chunk_new (AcacheEntry, acache_entry_chunk);
			entry->refcount = 0;
			entry->filename = g_strdup(g_basename(filename));
			entry->message_id = g_strdup (message_id);
			entry->size = 0;
			entry->date = time (NULL);
			g_hash_table_insert (filename_to_entry, entry->filename, entry);
			debug1 (DEBUG_ACACHE, "Added new entry %d", entry->message_id);
		}

		/* If we're checking it out, then move it to the safe end of the
		   least-recently-used kill heuristic */
		if (inc > 0)
			entry->date = time (NULL);

		entry->refcount += inc;
		debug3 (DEBUG_ACACHE, "%s refcount - inc by %d to %d", entry->message_id, inc, entry->refcount);

		g_free (filename);
	}
}

void
acache_file_checkout (const gchar ** message_ids,
                      gint           message_id_qty)
{
	g_return_if_fail (message_ids!=NULL);
	g_return_if_fail (message_id_qty >= 1);

	pthread_mutex_lock (&entries_mutex);
	acache_file_update_refcount_nolock (message_ids, message_id_qty, 1);
	pthread_mutex_unlock (&entries_mutex);
}

void
acache_file_checkin (const gchar ** message_ids,
                     gint           message_id_qty)
{
	g_return_if_fail (message_ids != NULL);
	g_return_if_fail (message_id_qty >= 1);

	pthread_mutex_lock (&entries_mutex);
	acache_file_update_refcount_nolock (message_ids, message_id_qty, -1);
	pthread_mutex_unlock (&entries_mutex);

	if (acache_size > (acache_max_megs*1024*1024))
		acache_expire ();
}

/***
****
****  ADD / REMOVE FILES
****
***/

void
acache_set_message (const gchar    * message_id,
                    const gchar    * message,
                    gboolean         checkout)
{
       	FILE * fp;
	gchar * filename;
	AcacheEntry * entry;

	g_return_if_fail (is_nonempty_string(message));
	g_return_if_fail (is_nonempty_string(message_id));

	/* find out where to write the message */
	filename = acache_get_filename (message_id);

	fp = fopen (filename, "w+");
	if (fp != NULL)
	{
		/* write the message */
		fwrite (message, sizeof(char), strlen(message), fp);
		fclose (fp);

		/* update our tables */
		replace_gstr (&filename, g_strdup(g_basename(filename)));
		pthread_mutex_lock (&entries_mutex);
		acache_file_update_refcount_nolock (&message_id, 1, checkout?1:0);
		entry = g_hash_table_lookup (filename_to_entry, filename);
		g_assert (entry!=NULL);
		entry->filename = filename;
		entry->message_id = g_strdup (message_id);
		entry->size = strlen (message);
		entry->date = time (NULL);
		pthread_mutex_unlock (&entries_mutex);

		/* notify everyone that this message has been added */
		pan_callback_call (acache_bodies_added_callback, (gpointer)message_id, NULL);

		/* if the acache is too big, purge the old stuff */
		if (acache_size > (acache_max_megs*1024*1024))
			acache_expire ();
	}
}

/**
 * This must be called from inside an entries_mutex lock.
 */
static void
acache_remove_entry (AcacheEntry * entry)
{
	gchar * path;

	/* sanity clause */
	g_return_if_fail (entry!=NULL);

	/* let everyone know */
	debug2 (DEBUG_ACACHE, "Expiring %s with refcount %d", entry->message_id, entry->refcount);
	pan_callback_call (acache_bodies_removed_callback, (gpointer)entry->message_id, NULL);

	/* update the acache size */
	acache_size -= entry->size;

	/* remove the file */
	path = acache_get_full_path (entry->filename);
	unlink (path);
	g_free (path);

	/* update acache tables */
	g_hash_table_remove (filename_to_entry, entry->filename);
	g_free (entry->filename);
	g_free (entry->message_id);
	g_mem_chunk_free (acache_entry_chunk, entry);
}

gchar*
acache_get_message (const gchar * message_id)
{
	gchar * buf = NULL;

	gchar * filename = acache_get_filename (message_id);
	if (is_nonempty_string (filename))
	{
		GArray* file = read_file (filename);
		if (file != NULL)
		{
			buf = file->data;
			g_array_free (file, FALSE);
		}
		g_free (filename);
	}

	return buf;
}



gboolean
acache_has_message (const gchar * message_id)
{
	gchar * filename = NULL;
	gboolean retval = FALSE;
	AcacheEntry * entry = NULL;

	if (is_nonempty_string(message_id))
		filename = acache_get_filename (message_id);
	if (is_nonempty_string(filename))
		entry = g_hash_table_lookup (filename_to_entry, g_basename(filename));
	if (entry != NULL)
		retval = entry->size != 0;

	g_free (filename);
	return retval;
}

/***
****
****  EXPIRE
****
***/

static gint
compare_ppEntry_ppEntry_by_youth (const void * a, const void *b)
{
	const time_t aval = (**(AcacheEntry**)a).date;
	const time_t bval = (**(AcacheEntry**)b).date;
	return (gint) -difftime (aval, bval);
}


void
acache_expire_messages (const gchar   ** message_ids,
	                gint             message_id_qty)
{
	gint i;

	g_return_if_fail (message_ids!=NULL);
	g_return_if_fail (message_id_qty>0);

	for (i=0; i<message_id_qty; ++i)
	{
		gchar * filename = NULL;
		AcacheEntry * entry = NULL;
		const gchar * message_id = NULL;

		/* find the entry */
		message_id = message_ids[i];
		if (is_nonempty_string(message_id))
			filename = acache_get_filename (message_id);
		if (is_nonempty_string(filename))
			entry = (AcacheEntry*) g_hash_table_lookup (
				filename_to_entry, g_basename(filename));

		/* if we have such an entry, remove it */
		if (entry != NULL)
			acache_remove_entry (entry);

		/* cleanup */
		g_free (filename);
	}
}


static gint
acache_expire_to_size (gdouble max_megs)
{
	gint files_removed = 0;
	const glong cache_max = (glong)(max_megs * 1024 * 1024);
	debug_enter ("acache_expire_to_size");

	/* lock the entries mutex so that our array doesn't wind up holding
	   pointers to free'd entries */
	pthread_mutex_lock (&entries_mutex);

	if (cache_max < acache_size)
	{
		gint i;
		GPtrArray * files;

		/* get an array of files sorted by age */
		files = g_ptr_array_new ();
		pan_hash_to_ptr_array (filename_to_entry, files);
		qsort (files->pdata, files->len, sizeof(gpointer), compare_ppEntry_ppEntry_by_youth);

		/* Start blowing away files */
		for (i=files->len; i>0 && cache_max<acache_size; --i) {
			AcacheEntry * entry = (AcacheEntry*) g_ptr_array_index (files, i-1);
			if (entry->refcount <= 0) {
				acache_remove_entry (entry);
				entry = NULL;
				++files_removed;
			}
		}

		/*  cleanup */
		g_ptr_array_free (files, TRUE);
	}

	/* log */
	if (files_removed != 0) {
		log_add_va (LOG_INFO, _("Removed %d articles from local cache"), files_removed);
		log_add_va (LOG_INFO, _("Article cache now has %.1f MB in %d articles"),
			(double)(acache_size/(1024.0*1024.0)),
			 (gint)g_hash_table_size(filename_to_entry));
	}

	/* entries mutex lock */
	pthread_mutex_unlock (&entries_mutex);

	/* done */
	debug_exit ("acache_expire_to_size");
	return files_removed;
}

gint 
acache_expire (void)
{
	return acache_expire_to_size (acache_max_megs * 0.8);
}

gint
acache_expire_all (void)
{
	gint retval;
	printf ("Pan is flushing article cache... ");
	retval = acache_expire_to_size(0);
	printf ("%d files erased.\n", retval);
	fflush (NULL);
	return retval;
}
