/*
 * Pan - A Newsreader for X
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <string.h>
#include <stdlib.h>

#include <gnome.h>

#include <pan/base/debug.h>
#include <pan/base/group.h>
#include <pan/base/serverlist.h>

#include <pan/globals.h>
#include <pan/group-ui.h>
#include <pan/util.h>

static void
group_empty_cb (GnomeDialog    * dialog,
		int              button_number,
		gpointer         group)
{
	switch (button_number)
	{
		case 0: /* 'yes' */
		{
			GtkWidget * w = gtk_object_get_data (GTK_OBJECT(dialog), "reset");
			gboolean clear_counts = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(w));
			group_empty (group, clear_counts);
			break;
		}
		case 1: /* 'no' */
			break;

		default: /* cancel */
			break;
	};

	gtk_widget_destroy (GTK_WIDGET(dialog));
}

void
group_empty_dialog (Group *group)
{
	GString * str = g_string_new (NULL);
	GtkWidget * dialog = NULL;
	GtkWidget * w = NULL;

	/* sanity check */
	g_return_if_fail (group!=NULL);

	pan_lock();

	/* create dialog */
	g_string_sprintf (str,
	                  _("Clean group \"%s\""),
	                  group_get_readable_name(group));
	dialog = gnome_dialog_new (str->str,
				   GNOME_STOCK_BUTTON_YES,
				   GNOME_STOCK_BUTTON_NO, NULL);

	/* body */
	g_string_sprintf (str,
	                  _("Remove all messages from group: \n%s ?"),
	                  group_get_readable_name(group));
	w = gtk_label_new (str->str);
	gtk_container_add (GTK_CONTAINER(GNOME_DIALOG(dialog)->vbox), w);

	/* "clear count?" check button */
	w = gtk_check_button_new_with_label (_("Reset new message counter"));
	gtk_container_add (GTK_CONTAINER(GNOME_DIALOG(dialog)->vbox), w);
	gtk_object_set_data (GTK_OBJECT(dialog), "reset", w);

	/* listen for clicks */
	gtk_signal_connect (GTK_OBJECT(dialog), "clicked",
			    GTK_SIGNAL_FUNC (group_empty_cb), group);

        gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(Pan.window));
	gtk_widget_show_all (dialog);
	pan_unlock();

	g_string_free (str, TRUE);
}

/****
*****
*****
****/

static void
new_folder_cb (gchar* string, gpointer data)
{
	Group * group = NULL;
	Server * server = serverlist_get_named_server (INTERNAL_SERVER_NAME);
	gchar * tmp;
	debug_enter ("new_folder_cb");

	/* user hit cancel */
	if (string == NULL)
		return;

	/* make sure the user entered something */
	tmp = g_strdup (string);
	g_strstrip (tmp);
	if (!is_nonempty_string (tmp)) {
		pan_error_dialog (_("Invalid folder name: \"%s\""), string);
		g_free (tmp);
		return;
	}

	/* make sure there's not already a folder with that name */
	g_strdown (tmp);
	group = serverlist_get_named_folder (tmp);
	if (group != NULL) {
		pan_error_dialog (_("Folder \"%s\" already exists."), string);
		g_free (tmp);
		return;
	}

	/* make sure it's not in the reserved namespace */
	if (!strncmp(tmp, "pan.", 4)) {
		pan_error_dialog (_("Please don't begin folders with \"pan.\"; it confuses me terribly."));
		g_free (tmp);
		return;
	}

	/* create new folder */
	group = group_new (tmp);
	group_set_is_folder (group, TRUE);
	group_set_dirty (group);
	server_add_groups (server, &group, 1, NULL, NULL);

	/* cleanup */
	g_free (tmp);
	debug_exit ("new_folder_cb");
}

void
group_new_folder_dialog (void)
{
	gchar * prompt = g_strdup (_("New Folder Name:"));
	gnome_request_dialog (FALSE, prompt, NULL, 128,
			      new_folder_cb,
			      NULL, GTK_WINDOW(Pan.window));
	g_free (prompt);
}
