#include <stdio.h>
#include <stdlib.h>

#include <pan/base/article.h>
#include <pan/base/base-prefs.h>
#include <pan/base/gnksa.h>
#include <pan/base/message-check.h>
#include <pan/base/pan-glib-extensions.h>

#define PRINT_ERRORS \
	if (1) { \
		gint i; \
		g_message ("[%d] errors", errors->len); \
		for (i=0; i<errors->len; ++i) \
			g_message ("[%d][%s]", i, (gchar*)g_ptr_array_index(errors,i)); \
	}

int main (void)
{
	gint test = 0;
	Server * s;
	Group * g;
	Article * a;
	gchar * pch;
	GPtrArray * errors = g_ptr_array_new ();
	GoodnessLevel goodness = OKAY;

	/* init panbase */
	g_thread_init (NULL);
	if (1) {
		gchar * data_dir = g_strdup_printf ("%s/data/", getenv("PWD"));
		gchar * temp_dir = g_strdup_printf ("%s/data/tmp/", getenv("PWD"));
		gchar * download_dir = g_strdup_printf ("%s/data/save/", getenv("PWD"));
		base_prefs_init (data_dir, temp_dir, download_dir, 1000, FALSE, FALSE);
		g_free (data_dir);
		g_free (temp_dir);
		g_free (download_dir);
	}

	/* create an article */
	s = server_new ();
	s->name = g_strdup ("Server");
	g = group_new ("alt.test");
	server_add_groups (s, &g, 1, NULL, NULL);
	a = article_new (g);

	/* populate a simple article */
	++test;
	a->number = 1;
	article_set_header (a, HEADER_FROM, "\"Charles Kerr\" <charles@rebelbase.com>", DO_CHUNK);
	pch = gnksa_generate_message_id ("rebelbase.com");
	article_set_header (a, HEADER_MESSAGE_ID, pch, DO_CHUNK);
	article_set_header (a, HEADER_SUBJECT, "MAKE MONEY FAST", DO_CHUNK);
	article_set_header (a, HEADER_ORGANIZATION, "Lazars Android Works", DO_CHUNK);
	article_set_header (a, HEADER_NEWSGROUPS, "alt.test", DO_CHUNK);
	article_set_header (a, PAN_ATTRIBUTION, "Someone wrote", DO_CHUNK);
	article_set_header (a, PAN_BODY, "Hello World!", DO_CHUNK);

	/* this should pass the tests */
	check_article (a, s, errors, &goodness);
	if (errors->len!=0) return test;
	if (goodness!=OKAY) return test;
	g_message ("PASS simple message");

	/* all quoted */
	++test;
	goodness = OKAY;
	g_ptr_array_set_size (errors, 0);
	article_set_header (a, PAN_BODY, "> Hello World!\n> All quoted text", DO_CHUNK);
	check_article (a, s, errors, &goodness);
	if (errors->len!=2) return test;
	if (goodness!=REFUSE) return test;
	if (pan_strcmp((gchar*)g_ptr_array_index(errors,0), "WARNING: The message is entirely quoted text!")) return test;
	if (pan_strcmp((gchar*)g_ptr_array_index(errors,1), "ERROR: Message appears to have no new content.")) return test;
	g_message ("PASS all quoted text");

	/* mostly quoted: less than 20% of message is new content */
	++test;
	goodness = OKAY;
	g_ptr_array_set_size (errors, 0);
	article_set_header (a, PAN_BODY, "> Hello World!\n> quoted\n> text\n> foo\n> bar\nnew text", DO_CHUNK);
	check_article (a, s, errors, &goodness);
	if (errors->len!=1) return test;
	if (goodness!=WARN) return test;
	if (pan_strcmp((gchar*)g_ptr_array_index(errors,0), "WARNING: The message is mostly quoted text.")) return test;
	g_message ("PASS < 20%% quoted text");

	/* mostly quoted border condition: 20% of message is new content (should pass) */
	++test;
	goodness = OKAY;
	g_ptr_array_set_size (errors, 0);
	article_set_header (a, PAN_BODY, "> Hello World!\n> quoted\n> text\n> foo\nnew text", DO_CHUNK);
	check_article (a, s, errors, &goodness);
PRINT_ERRORS
	if (errors->len!=0) return test;
	if (goodness!=OKAY) return test;
	g_message ("PASS 20%% quoted text");

	/* sig check: too long */
	++test;
	goodness = OKAY;
	g_ptr_array_set_size (errors, 0);
	article_set_header (a, PAN_BODY, "Hello!\n\n-- \nThis\nSig\nIs\nToo\nLong\n", DO_CHUNK);
	check_article (a, s, errors, &goodness);
	if (errors->len!=1) return test;
	if (goodness!=WARN) return test;
	if (pan_strcmp((gchar*)g_ptr_array_index(errors,0), "WARNING: signature is more than 4 lines long")) return test;
	g_message ("PASS sig too long");

	/* sig check: too wide */
	++test;
	goodness = OKAY;
	g_ptr_array_set_size (errors, 0);
	article_set_header (a, PAN_BODY,
"Hello!\n"
"\n"
"-- \n"
"This sig line is exactly 80 characters wide.  I'll keep typing until I reach 80.\n"
"This sig line is greater than 80 characters wide.  In fact, it's 84 characters wide.\n"
"This sig line is greater than 80 characters wide.  In fact, it measures 95 characters in width!\n"
"This sig line is less than 80 characters wide.", DO_CHUNK);
	check_article (a, s, errors, &goodness);
	if (errors->len!=1) return test;
	if (goodness!=WARN) return test;
	if (pan_strcmp((gchar*)g_ptr_array_index(errors,0), "WARNING: 2 lines too wide in signature")) return test;
	g_message ("PASS sig too wide");


	/* sig check: sig marker, no sig */
	++test;
	goodness = OKAY;
	g_ptr_array_set_size (errors, 0);
	article_set_header (a, PAN_BODY, "Hello!\n\n-- \n", DO_CHUNK);
	check_article (a, s, errors, &goodness);
	if (errors->len!=1) return test;
	if (goodness!=WARN) return test;
	if (pan_strcmp((gchar*)g_ptr_array_index(errors,0),
		"WARNING: Signature prefix with no signature.")) return test;
	g_message ("PASS sig marker, no sig");


	/* sig check: okay sig */
	++test;
	goodness = OKAY;
	g_ptr_array_set_size (errors, 0);
	article_set_header (a, PAN_BODY,
		"Hello!\n\n-- \nThis is a short, narrow sig.\nIt should pass.\n", DO_CHUNK);
	check_article (a, s, errors, &goodness);
	if (errors->len!=0) return test;
	if (goodness!=OKAY) return test;
	g_message ("PASS good sig");


	/* adrian's fake followup */
	++test;
	goodness = OKAY;
	g_ptr_array_set_size (errors, 0);
	article_set_header (a, PAN_BODY,
		">>>>>>>>>>>> I think A\n"
		">>>>>>>>>>> No, it's not\n"
		">>>>>>>>>> But B => C\n"
		">>>>>>>>> What's that got to do with A?\n"
		">>>>>>>> I still think B => C\n"
		">>>>>>> It's not even B => C. But Still waiting for proof for A\n"
		">>>>>> You don't prove !A, either.\n"
		">>>>> There's the FAQ: X => !A and Y => !A\n"
		">>>> But there in the text it sais T' => A\n"
		">>> But T' is only a subset of T. T => !A.\n"
		">> Moron\n"
		"> Jackass.\n"
		"\n"
		"I don't know wether I am amused or annoyed. Apparently the funny side\n"
		"prevailed so far, as I'm still reading.\n"
		"\n"
		"-- vbi", DO_CHUNK);
	check_article (a, s, errors, &goodness);
	if (errors->len!=0) return test;
	if (goodness!=OKAY) return test;
	g_message ("PASS Adrian's Fake Followup");



	/* body too wide */
	++test;
	goodness = OKAY;
	g_ptr_array_set_size (errors, 0);
	article_set_header (a, PAN_BODY,
	"Hello!\n"
	"This sig line is exactly 80 characters wide.  I'll keep typing until I reach 80.\n"
	"This sig line is greater than 80 characters wide.  In fact, it's 84 characters wide.\n"
	"This sig line is greater than 80 characters wide.  In fact, it measures 95 characters in width!\n"
	"This sig line is less than 80 characters wide.", DO_CHUNK);
	check_article (a, s, errors, &goodness);
	if (errors->len!=1) return test;
	if (goodness!=WARN) return test;
	if (pan_strcmp((gchar*)g_ptr_array_index(errors,0), "WARNING: 2 lines >80 characters in the message.")) return test;
	g_message ("PASS body too wide");

	/* body empty */
	++test;
	goodness = OKAY;
	g_ptr_array_set_size (errors, 0);
	article_set_header (a, PAN_BODY, "\n\t\n   \n-- \nThis is the sig.", DO_CHUNK);
	check_article (a, s, errors, &goodness);
	if (errors->len!=2) return test;
	if (goodness!=REFUSE) return test;
	if (pan_strcmp((gchar*)g_ptr_array_index(errors,0), "ERROR: Message is empty.")) return test;
	if (pan_strcmp((gchar*)g_ptr_array_index(errors,1), "ERROR: Message appears to have no new content.")) return test;
	g_message ("PASS empty body");
	article_set_header (a, PAN_BODY, "Innocent body.", DO_CHUNK);

	/* empty subject */
	++test;
	goodness = OKAY;
	g_ptr_array_set_size (errors, 0);
	article_set_header (a, HEADER_SUBJECT, "", DO_CHUNK);
	check_article (a, s, errors, &goodness);
	if (errors->len<1) return test;
	if (goodness!=REFUSE) return test;
	g_message ("PASS refuse to post with no subject");
	article_set_header (a, HEADER_SUBJECT, "Happy Lucky Feeling", DO_CHUNK);

	/* newsgroups */
	++test;
	goodness = OKAY;
	g_ptr_array_set_size (errors, 0);
	article_set_header (a, HEADER_NEWSGROUPS, "alt.test,unknown.group", DO_CHUNK);
	check_article (a, s, errors, &goodness);
	if (errors->len!=1) return test;
	if (goodness!=WARN) return test;
	if (pan_strcmp((gchar*)g_ptr_array_index(errors,0), "WARNING: Unknown group: \"unknown.group\".")) return test;
	g_message ("PASS unknown group");
	article_set_header (a, HEADER_NEWSGROUPS, "alt.test", DO_CHUNK);

	/* newsgroups */
	++test;
	goodness = OKAY;
	g_ptr_array_set_size (errors, 0);
	article_set_header (a, HEADER_FOLLOWUP_TO, "alt.test,unknown.group", DO_CHUNK);
	check_article (a, s, errors, &goodness);
	if (errors->len!=1) return test;
	if (goodness!=WARN) return test;
	if (pan_strcmp((gchar*)g_ptr_array_index(errors,0), "WARNING: Unknown group: \"unknown.group\".")) return test;
	g_message ("PASS unknown followup-to group");


	/* success */
	g_message ("Passed all message-check tests");
	return 0;
}
