/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <glib.h>

#include <string.h>

#include <pan/base/acache.h>
#include <pan/base/article.h>
#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/gnksa.h>
#include <pan/base/group.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/serverlist.h>
#include <pan/base/status-item.h>

#include <pan/article-actions.h>
#include <pan/articlelist.h>
#include <pan/message-window.h>
#include <pan/nntp.h>
#include <pan/queue.h>
#include <pan/save-ui.h>
#include <pan/task.h>
#include <pan/task-bodies.h>
#include <pan/task-func-ptr.h>
#include <pan/task-save.h>
#include <pan/util.h>

/**
 * Find a matching article without comparing by message-id.
 * This is because many news servers alter the message-id of
 * an article being posted, so we can't rely on searching by
 * the message-id that Pan generated.
 *
 * If a match is found, the refcount of Pan.sent is incremented
 * by one and must be decremented by the caller.
 */
static Article*
find_matching_sent_article (const Article * article)
{
	Group * sent;
	Article * retval = NULL;
	GPtrArray * articles;
	guint i;
	debug_enter ("find_matching_sent_article");

       	sent = serverlist_get_named_folder (PAN_SENT);

	/* sanity clause */
	g_return_val_if_fail (article_is_valid(article), NULL);
	g_return_val_if_fail (sent!=NULL, NULL);

	group_ref_articles (sent, NULL);
	articles = group_get_article_array (sent);

	for (i=0; retval==NULL && i!=articles->len; ++i)
	{
		gchar * body_1;
		gchar * body_2;
		Article * compare = ARTICLE(g_ptr_array_index(articles,i));

		/* comparison #1: subject must match */
		if (pan_strcmp (article_get_subject(compare), article_get_subject(article)))
			continue;

		/* comparison #2: author address must match */
		if (pan_strcmp (compare->author_addr, article->author_addr))
			continue;

		/* comparison #3: body address must match */
		body_1 = article_get_body (article);
		body_2 = article_get_body (compare);
		if (body_1!=NULL && body_2!=NULL)
		{
			g_strstrip (body_1);
			g_strstrip (body_2);
			if (!strcmp (body_1, body_2))
				retval = compare;
		}
		g_free (body_1);
		g_free (body_2);
	}

	if (retval == NULL)
		group_unref_articles (sent, NULL);

	debug_exit ("find_matching_sent_article");
	g_ptr_array_free (articles, TRUE);
	return retval;
}

static gint
article_cancel_run (TaskFuncPtr* task, gpointer user_data)
{
        int status = nntp_cancel (STATUS_ITEM(task),
	                          ARTICLE(user_data),
	                          TASK(task)->sock);

	if (status == TASK_SUCCESS)
	{
		/* FIXME: add something to the article to denote that it's been cancelled */
	}

	return 0;
}
static gchar*
article_cancel_describe (const StatusItem * status)
{
	return g_strdup (_("Cancelling article"));
}
static void
article_cancel_dtor (gpointer data)
{
	debug_enter ("article_cancel_dtor");

	group_unref_articles (serverlist_get_named_folder(PAN_SENT), NULL);

	debug_exit ("article_cancel_dtor");
}
void
article_cancel (Article * article)
{
	Server * server;
	Article * ours;
	const gchar * newsgroups;

	/* sanity clause */
	g_return_if_fail (article != NULL);
	server = article->group==NULL ? NULL : article->group->server;
	g_return_if_fail (server != NULL);
	newsgroups = article_get_extra_header (article, HEADER_NEWSGROUPS);
	g_return_if_fail (is_nonempty_string (newsgroups));

	ours = find_matching_sent_article (article);
	if (ours == NULL)
	{
		log_add_va (LOG_ERROR|LOG_URGENT,
		            _("Unable to cancel article: Couldn't find matching article in folder `pan.sent'!"));
	}
	else
	{
		queue_add (TASK(task_func_ptr_new (server,
			article_cancel_describe,
			article_cancel_run,
			article_cancel_dtor, 
			ours, FALSE)));
	}
}

/****
*****
****/

void
article_supersede (const Article * article)
{
	const Article * ours;
	Group * sendlater;
	debug_enter ("article_supersede");

       	sendlater = serverlist_get_named_folder (PAN_SENDLATER);

	/* sanity clause */
	g_return_if_fail (article_is_valid(article));
	g_return_if_fail (sendlater != NULL);

	/* find the original copy so that the headers will match */
	ours = find_matching_sent_article (article);
	if (ours == NULL)
	{
		log_add_va (LOG_ERROR|LOG_URGENT,
		            _("Unable to supersede article: Couldn't find matching article in folder `pan.sent'!"));
	}
	else
	{
		gchar * message_id = gnksa_generate_message_id_from_email_addr (ours->author_addr);
		Article * super = article_dup (sendlater, ours);
		article_set_header (super, HEADER_MESSAGE_ID, message_id, DO_CHUNK);
		article_set_header (super, HEADER_SUPERSEDES, article_get_message_id(article), DO_CHUNK);

		/* edit the copy */
		message_edit_window (super);

		/* cleanup */
		g_free (message_id);
		group_unref_articles (ours->group, NULL);
		/* FIXME: super dangles */
	}

	debug_exit ("article_supersede");
}

/****
*****
****/

static void
article_copy_articles_to_folder_cb (gpointer obj, gpointer arg, gpointer data)
{
	gint status;
	TaskBodies * task;
	GPtrArray * articles;
	Group * folder;
	debug_enter ("article_copy_articles_to_folder_cb");

	/* sanity clause */
	g_return_if_fail (obj!=NULL);
	g_return_if_fail (data!=NULL);
	g_return_if_fail (group_is_folder(GROUP(data)));

	status = GPOINTER_TO_INT(arg);
	task = TASK_BODIES(obj);
	articles = task->articles;
	folder = GROUP(data);

	if (status == TASK_SUCCESS)
	{
		gint i;
		GPtrArray * addme;

		/* init */
		group_ref_articles (folder, NULL);

		/* create an array of articles to add. */
		addme = g_ptr_array_new ();
		pan_g_ptr_array_reserve (addme, articles->len);
		for (i=0; i!=articles->len; ++i)
		{
			const Article * a_old;
			Article * a_new;
			gchar * body;

			a_old = ARTICLE(g_ptr_array_index(articles,i));
			body = article_get_body (a_old);
			a_new = article_dup (folder, a_old);
			article_set_header (a_new, PAN_BODY, body, DO_CHUNK);
			g_ptr_array_add (addme, a_new);

			g_free (body);
		}

		/* smoke 'em if you've got 'em */
		if (addme->len != 0)
			group_add_articles (folder, addme, NULL, NULL, NULL);

		/* clean up */
		group_unref_articles (folder, NULL);
		g_ptr_array_free (addme, TRUE);
	}

	debug_exit ("articlelist_article_to_folder_cb");
}

gboolean
article_copy_articles_to_folder (Group * folder, Article ** articles, gint qty)
{
	gint i;
	Task * task;
	GPtrArray * a;
	debug_enter ("article_copy_articles_to_folder");

	/* sanity clause */
	g_return_val_if_fail (folder!=NULL, FALSE);
	g_return_val_if_fail (group_is_folder(folder), FALSE);
	g_return_val_if_fail (articles!=NULL, FALSE);
	g_return_val_if_fail (qty>0, FALSE);
	for (i=0; i<qty; ++i)
		g_return_val_if_fail (article_is_valid(articles[i]), FALSE);

	/* make a task to download the bodies if we don't already have them */
	a = g_ptr_array_new ();
	pan_g_ptr_array_assign (a, (gpointer*)articles, qty);
	task = TASK(task_bodies_new (articles[0]->group, a));
	g_ptr_array_free (a, TRUE);
	if (task != NULL) {
		pan_callback_add (task->task_ran_callback, article_copy_articles_to_folder_cb, folder);
		queue_add (task);
	}

	debug_exit ("article_copy_articles_to_folder");
	return TRUE;
}

/***
****
****
****   SAVING ARTICLES / OPENING ATTACHMENTS
****
****
***/

static void
open_article (const Article * a)
{
	gchar * file;
	const gchar * files;

	g_return_if_fail (article_is_valid(a));

	files = article_get_extra_header (a, PAN_HEADER_FILENAME);
	while ((file = get_next_token_str (files, '\n', &files))) {
		open_outside_file (file);
		g_free (file);
	}

}

static void
open_decoded_cb (gpointer call_obj, gpointer call_arg, gpointer user_data)
{
	gint status = GPOINTER_TO_INT(call_arg);

	if (status == TASK_SUCCESS) {
		TaskSave * task = TASK_SAVE(call_obj);
		if (task!=NULL && task->articles!=NULL) {
			const Article * a = ARTICLE(task->articles->data);
			if (article_is_valid(a))
				open_article (a);
		}
	}
}

static void
decode_impl (Article * a,
             gboolean save_bodies,
             gboolean save_attachments,
             gboolean open_attachments)
{
	Task * task;
	GSList * articles;
	const gchar * files;
	debug_enter ("decode_impl");

	g_return_if_fail (article_is_valid(a));
	g_return_if_fail (save_bodies || save_attachments || open_attachments);

	/* got to save before you can open */
	if (open_attachments)
		save_attachments = TRUE;

	/* if user is opening the files & we already have them... */
	files = article_get_extra_header (a, PAN_HEADER_FILENAME);
	if (open_attachments && is_nonempty_string(files)) {
		open_article (a);
		debug_exit ("decode_impl");
		return;
	}

	/* build an array of this article and all its children. */
	articles = g_slist_copy (a->threads);
	articles = g_slist_prepend (articles, a);

	/* queue for decode */
	task = TASK(task_save_new (a->group, article_get_subject(a), articles));
	if (save_bodies)
		task_save_set_bodies (TASK_SAVE(task), NULL, NULL);
	if (save_attachments)
		task_save_set_attachments (TASK_SAVE(task), NULL, NULL);
	if (open_attachments)
		pan_callback_add (task->task_ran_callback, open_decoded_cb, NULL);
	task_save_validate_and_queue_or_self_destruct (TASK_SAVE(task), -1);

	/* cleanup this loop */
	g_slist_free (articles);
	debug_exit ("decode_impl");
}

static void
seleted_decode_impl (gboolean save_bodies,
                     gboolean save_attachments,
                     gboolean open_attachments)
{
	guint i;
	GPtrArray * sel;
	debug_enter ("seleted_decode_impl");

	/* decode each selected article */
	sel = articlelist_get_selected_articles_nolock ();
	for (i=0; i<sel->len; ++i)
		decode_impl (ARTICLE(g_ptr_array_index(sel,i)),
		             save_bodies,
		             save_attachments,
		             open_attachments);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("seleted_decode_impl");
}

void
article_action_selected_save (void)
{
	seleted_decode_impl (TRUE, TRUE, FALSE);
}

void
article_action_selected_save_attachments (void)
{		
	seleted_decode_impl (FALSE, TRUE, FALSE);
}

void
article_action_selected_open_attachments (void)
{
	seleted_decode_impl (FALSE, TRUE, TRUE);
}


void
article_action_selected_save_as (void)
{
	guint i;
	GPtrArray * sel;
	Group * group = articlelist_get_group ();
	GSList * gslists_of_articles = NULL;
	debug_enter ("articlelist_selected_decode_as");

	g_return_if_fail (group != NULL);

	/* loop through all the selected articles */
	sel = articlelist_get_selected_articles_nolock ();
	for (i=0; i!=sel->len; ++i)
	{
		GSList * l;
		Article * article = ARTICLE(g_ptr_array_index(sel,i));

		/* build an array of this node and all its children. */
		l = g_slist_copy (article->threads);
		l = g_slist_prepend (l, article);

		/* add this to the gslist of articles */
		gslists_of_articles = g_slist_prepend (gslists_of_articles, l);
	}

	/* save_attach_as owns gslists_of_articles now */
	gslists_of_articles = g_slist_reverse (gslists_of_articles);
	save_attachment_as (group, gslists_of_articles);

	g_ptr_array_free (sel, TRUE);
	debug_exit ("articlelist_selected_decode_as");
}

/***
****
****
****   DELETING ARTICLES
****
****
***/

static void
remove_duplicate_articles (GPtrArray * articles)
{
	gint i;
	GHashTable * hash;

	/* sanity claus */
	g_return_if_fail (articles!=NULL);
	g_return_if_fail (articles->len>0);
	if (articles->len==1) return;

	/* remove duplicates */
	hash = g_hash_table_new (g_str_hash, g_str_equal);
	for (i=0; i<articles->len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(articles,i));
		g_hash_table_insert (hash, (gpointer)article_get_message_id(a), a);
	}
	pan_hash_to_ptr_array  (hash, articles);

	/* cleanup */
	g_hash_table_destroy (hash);
}


/**
 * Pairs of Group* and GArrays of gulong article numbers to remove from
 * the group.  We pass these arrays on to group_remove_crossposts(),
 * then free the array because we're done with it.
 * @see articlelist_delete_articles
 */
static void
articlelist_remove_xrefs_ghfunc (gpointer key,
                                 gpointer value,
                                 gpointer user_data)
{
	Group * g = GROUP(key);
	GArray * numbers = (GArray*) value;
	group_remove_crossposts (g, (gulong*)numbers->data, numbers->len);
	g_array_free (numbers, TRUE);
}

/**
 * data is a GHashTable of (key=group,val=GArray) where we fill those
 * GArrays with the article numbers of this crosspost.
 */
static void
articlelist_remove_xrefs_xreffunc (Group   * group,
                                   gulong    number,
                                   gpointer  data)
{
	GHashTable * hash = (GHashTable*) data;

	if (group_is_subscribed (group))
	{
		GArray * a = g_hash_table_lookup (hash, group);
		if (a == NULL)
			g_hash_table_insert (hash,
			                     group,
			                     a = g_array_new (FALSE, FALSE, sizeof(gulong)));
		g_array_append_val (a, number);
	}
}

/**
 * @see remove_duplicate_articles
 * @see articlelist_remove_xrefs_xreffunc
 * @see articlelist_remove_xrefs_ghfunc
 */
static void
delete_articles (GPtrArray * articles)
{
	debug_enter ("article_action_delete_articles");

	g_return_if_fail (articles!=NULL);

	if (articles->len)
	{
		Group * group = ARTICLE(g_ptr_array_index(articles,0))->group;

		/* uniqueness check */
		remove_duplicate_articles (articles);

		/* remove these articles from the cache */
		if (1) {
			guint i;
			GPtrArray * ids = g_ptr_array_new ();
			for (i=0; i<articles->len; ++i) {
				Article * a = ARTICLE(g_ptr_array_index(articles,i));
				g_ptr_array_add (ids, (gpointer)article_get_message_id(a));
			}
			acache_expire_messages ((const gchar **)ids->pdata, ids->len);
			g_ptr_array_free (ids, TRUE);
		}

		/* if these articles are posted elsewhere, delete those crossposts too */
		if (1) {
			guint i;
			GHashTable * hash = g_hash_table_new (g_direct_hash, g_direct_equal);
			for (i=0; i<articles->len; ++i) {
				Article * a = ARTICLE(g_ptr_array_index(articles,i));
				article_xref_foreach (a, articlelist_remove_xrefs_xreffunc, hash, SERVER_GROUPS_SUBSCRIBED, TRUE);
			}
			g_hash_table_foreach (hash, articlelist_remove_xrefs_ghfunc, NULL);
			g_hash_table_destroy (hash);
		}

		/* delete the articles in this group */
		group_remove_articles (group, articles);

		/* log */
		log_add_va (LOG_INFO, _("Deleted %u articles from `%s'"),
		            articles->len,
		            group_get_readable_name(group));
	}

	debug_exit ("article_action_delete_articles");
}

void
article_action_delete_selected_threads (void)
{
	GPtrArray * sel;
	GPtrArray * all;
	debug_enter ("article_action_delete_selected_threads");

       	sel = articlelist_get_selected_articles_nolock ();
       	all = article_get_unique_threads (sel, GET_WHOLE_THREAD);
	delete_articles (all);
	g_ptr_array_free (all, TRUE);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("article_action_delete_selected_threads");
}

void
article_action_delete_selected_articles (void)
{
	GPtrArray * articles;
	int i, len;
	debug_enter ("article_action_delete_selected_articles");

	/* if it's a multipart, delete the children too */
       	articles = articlelist_get_selected_articles_nolock ();
	for (i=0, len=articles->len; i!=len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(articles,i));
		if (a->part==1 && a->parts>1) {
			GSList * l;
			for (l=a->threads; l!=NULL; l=l->next)
				if (ARTICLE(l->data)->part > 1)
					g_ptr_array_add (articles, l->data);
		}
	}

	/* delete the articles */
	delete_articles (articles);

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	debug_exit ("article_action_delete_selected_articles");
}

void
article_action_edit_selected (void)
{
        Article * a;
	debug_enter ("article_action_edit_selected");

	a = articlelist_get_selected_article_nolock ();
	if (a != NULL)
		message_edit_window (a);

	debug_exit ("article_action_edit_selected");
}
