/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@rebelbase.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <stdio.h>
#include <sys/stat.h>
#include <errno.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-util.h>

#include <pan/base/base-prefs.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-file.h>

/***
****
****  TEMP FILES
****
***/


gchar *
pan_make_temp (FILE ** setme_fp)
{
	guint pid = (guint) getpid();
	char * retval = NULL;
	char * fname;
	int fd;

	/* try the Pan tmp directory */
	fname = g_strdup_printf ("%s%cpan%uXXXXXX", g_get_tmp_dir(), G_DIR_SEPARATOR, pid);
	fd = mkstemp (fname);

	/* open the file */
	if (fd != -1) {
		retval = g_strdup (fname);
		*setme_fp = fdopen (fd, "w+");
	}

	g_free (fname);
	return retval;
}

void
pan_clean_temp_dir (void)
{
	const gchar * tmp_dir = g_get_tmp_dir ();
	GString *filename = g_string_new (NULL);
	char *prefix = g_strdup_printf ("pan%u", (unsigned int)getpid ());
	gint prefix_l = strlen (prefix);
	DIR *dir_p = opendir(tmp_dir);
	struct dirent *dirent_p;

	while ((dirent_p = readdir(dir_p)))
	{
		if (!strncmp (prefix, dirent_p->d_name, prefix_l))
		{
			g_string_sprintf (filename,
				  "%s%c%s", tmp_dir, G_DIR_SEPARATOR, dirent_p->d_name);
			unlink (filename->str);
		}	
	}
	closedir (dir_p);
	g_free (prefix);
	g_string_free (filename, TRUE);
}


/***
****
****  FILES
****
***/


gchar*
pan_normalize_filename_inplace (gchar * filename)
{
	gchar * in;
	gchar * out;

	for (in=out=filename; *in; )
		if (in[0]==G_DIR_SEPARATOR && in[1]==G_DIR_SEPARATOR)
			++in;
		else
			*out++ = *in++;
	*out = '\0';

	return filename;
}


gboolean
file_exists (const char* filename)
{
	return is_nonempty_string(filename) && g_file_exists (filename);
}

GArray*
read_file (const char* filename)
{
	FILE * fp;
	GArray * array;

	/* open file */
	size_t len = get_filesize(filename);
	fp = fopen (filename, "r");
	if (fp == NULL)
		return NULL;

	array = g_array_new (TRUE, FALSE, 1);

	if (len>0) /* got the size, so alloc a buffer */
	{
		/* read the file... */
		g_array_set_size (array, len);
		array->len = fread (array->data, 1, len, fp);
	}
	else /* possibly from a FIFO, so read & grow */
	{
		gchar buf[2048];
		size_t read;
		while ((read = fread(buf, 1, sizeof(buf), fp)))
			g_array_append_vals (array, buf, read);
	}

	fclose (fp);
	return array;
}

size_t
get_filesize (const char* filename)
{
	struct stat buf;
	if ( !stat(filename, &buf) )
		return buf.st_size;

	return -1;
}

void
directory_check (const char *pathname)
{
	g_return_if_fail (is_nonempty_string(pathname));

	if (!g_file_test (pathname, G_FILE_TEST_ISDIR))
	{
		const gchar * in = pathname;
		gchar * buf = g_malloc (strlen(pathname) + 1);
		gchar * out = buf;

		for (;;)
		{
			*out++ = *in;

			if (*in=='\0' || *in==G_DIR_SEPARATOR)
			{
				if (*in==G_DIR_SEPARATOR)
					*out = '\0';

				if (!g_file_test (buf, G_FILE_TEST_ISDIR))
				{
					if (mkdir (buf, S_IRUSR | S_IWUSR | S_IXUSR))
					{
						log_add_va (LOG_ERROR, _("Couldn't create directory `%s': %s"),
						                       buf, g_strerror (errno));
						break;
					}
				}
			}

			if (!*in)
				break;

			++in;
		}

		g_free (buf);
	}
}

/***
****
****  MBOX
****
***/

const gchar*
mbox_get_next_message (const gchar * text,
                       gchar ** setme_from,
                       gchar ** setme_msg)
{
	GString * buf;
	const gchar * march;
	gboolean have_from = FALSE;

	/* find the end of the message */
       	buf = g_string_sized_new (2048);
	march = text;
	for (;;)
	{
		gboolean is_from;
		gint line_len = 0;
		const gchar * line_begin = NULL;
		const gchar * prev_march = march;

		if (!get_next_token_range (march, '\n', &march, &line_begin, &line_len))
			break;

		/* Is this a From line? */
		is_from = line_len>5
			&& line_begin[0]=='F'
			&& !strncmp (line_begin, "From ", 5);

		if (is_from && have_from) /* beginning of next message */
		{
			march = prev_march;
			break;
		}
		else if (is_from) /* beginning of our message */
		{
			have_from = TRUE;
			if (*setme_from != NULL)
				*setme_from = g_strndup (line_begin, line_len);
		}
		else if (have_from) /* a line in our message */
		{
			/* is this a nested From_ line? */
			if (*line_begin=='>') {
				const gchar * pch = line_begin;
				while (*pch=='>')
					++pch;
				if (*pch=='F' && !strncmp(pch,"From ", 5)) {
					gint diff = pch - line_begin;
					line_begin += diff;
					line_len -= diff;
				}
			}

			/* append the line */
			pan_g_string_append_len (buf, line_begin, line_len);
			g_string_append_c (buf, '\n');
		}
	}

	/* retval & cleanup */
	*setme_msg = buf->len ? buf->str : NULL;
	g_string_free (buf, FALSE);

	return is_nonempty_string(march) ? march : NULL;
}

gchar*
mbox_format_message (const gchar * message,
                     const gchar * author_address,
                     time_t date)
{
	gchar * retval;
	GString * tmp;
	const gchar * march;

	/* default author address */
	if (!is_nonempty_string(author_address)) {
		pan_warn_if_reached ();
		author_address = _("unknown@spammesenseless.com");
	}

	/* sanity clause */
	g_return_val_if_fail (message!=NULL, NULL);
	g_return_val_if_fail (author_address!=NULL, NULL);

	/* add the From_ line.  note ctime() adds the '\n' */
       	tmp = g_string_new (NULL);
	g_string_sprintf (tmp, "From %s %s", author_address, ctime(&date));

	/* walk through the message, adding each line, prepending
	   From_ lines with '>' as specified by the mbox format */
	march = message;
	for (;;)
	{
		const gchar * line_begin;
		const gchar * line_end;
		gboolean is_from = FALSE;
		gboolean is_nested_from = FALSE;

		/* ran out of text */
		if (march == NULL)
			break;

		/* delimit the line */
		line_begin = march;
		line_end = strchr (march, '\n');
		if (line_end != NULL)
			march = line_end + 1;
		else { /* this new line is the last one */
			line_end = line_begin + strlen(line_begin);
			march = NULL;
		}

		/* check the line for delimiter */
		is_from = !strncmp (line_begin, "From ", 5);

		/* is this a nested From_ line? */
		if (*line_begin) {
			const gchar * pch = line_begin;
			while (*pch=='>')
				++pch;
			if (!strncmp(pch,"From ", 5))
				is_nested_from = TRUE;
		}

		/* append the line to the output */
		g_string_sprintfa (tmp, "%s%*.*s\n",
			(is_from||is_nested_from) ? ">" : "",
			line_end-line_begin,
			line_end-line_begin,
			line_begin);
	}

	/* mbox message ends with a blank line */
	if (tmp->len != 0)
		g_string_append_c (tmp, '\n');

	/* cleanup */
	retval = tmp->len ? tmp->str : NULL;
	g_string_free (tmp, retval==NULL);

	return retval;
}

static gint
pan_copy (const gchar * old_filename,
          const gchar * new_filename)
{
	FILE * in;
	FILE * out;
	gchar buf [4096];
	size_t size_read;

	g_return_val_if_fail (is_nonempty_string(old_filename), -1);
	g_return_val_if_fail (is_nonempty_string(new_filename), -1);

	in = fopen (old_filename, "r");
	if (in == NULL) {
		log_add_va (LOG_ERROR, _("Can't open file `%s'"), old_filename);
		return -1;
	}
	out = fopen (new_filename, "w+");
	if (out == NULL) {
		log_add_va (LOG_ERROR, _("Can't open file `%s'"), new_filename);
		fclose (in);
		return -1;
	}

	while ((size_read = fread (buf, sizeof(char), sizeof(buf), in)))
		fwrite (buf, sizeof(char), size_read, out);

	fclose (in);
	fclose (out);
	return 0;
}

void
pan_rename (const gchar * old_filename,
            const gchar * new_filename)
{
	g_return_if_fail (is_nonempty_string(old_filename));
	g_return_if_fail (is_nonempty_string(new_filename));

	if (rename (old_filename, new_filename)) {
		if (!pan_copy (old_filename, new_filename))
			unlink (old_filename);
		else
			log_add_va (LOG_ERROR,
				_("Error renaming `%s' as `%s': %s."),
				old_filename,
				new_filename,
				strerror(errno));
	}

}

gint
change_file_mode_rw (FILE * fp, const gchar * file)
{
#if HAVE_FCHMOD
	return fchmod(fileno(fp), S_IRUSR|S_IWUSR);
#else
	return chmod(file, S_IRUSR|S_IWUSR);
#endif
}
