/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <libgnomeui/libgnomeui.h>

#include <pan/base/acache.h>
#include <pan/base/argset.h>
#include <pan/base/article-thread.h>
#include <pan/base/article.h>
#include <pan/base/debug.h>
#include <pan/base/group.h>
#include <pan/base/log.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/status-item.h>

#include <pan/article-actions.h>
#include <pan/article-list-toolbar-mediator.h> /* just for util */
#include <pan/article-filter-ui.h>
#include <pan/articlelist.h>
#include <pan/article-toolbar.h>
#include <pan/grouplist.h>
#include <pan/globals.h>
#include <pan/gui.h>
#include <pan/prefs.h>
#include <pan/queue.h>
#include <pan/task-body.h>
#include <pan/task-bodies.h>
#include <pan/task-headers.h>
#include <pan/thread-watch.h>
#include <pan/text.h>
#include <pan/util.h>

#include <pan/xpm/bozo.xpm>
#include <pan/xpm/old_read.xpm>
#include <pan/xpm/new_read.xpm>
#include <pan/xpm/old_unread.xpm>
#include <pan/xpm/new_unread.xpm>
#include <pan/xpm/bluecheck.xpm>
#include <pan/xpm/disk.xpm>
#include <pan/xpm/flagged.xpm>
#include <pan/xpm/x.xpm>
#include <pan/xpm/binary.xpm>
#include <pan/xpm/binary_complete.xpm>
#include <pan/xpm/binary_incomplete.xpm>
#include <pan/xpm/old_binary_complete.xpm>
#include <pan/xpm/old_binary_incomplete.xpm>
#include <pan/xpm/copy_message.xpm>
#include <pan/xpm/move_message.xpm>
#include <pan/xpm/watched.xpm>

#include <pan/filters/filter-top.h>

/***
****
****  Abandon hope, all ye who enter here
****
***/

/**********************************************************
 * Private Variables
 **********************************************************/

typedef enum
{
	REFRESH_SORT            = (1<<0),
	REFRESH_THREAD          = (1<<1),
	REFRESH_FILTER          = (1<<2),
	REFRESH_RELOAD          = (1<<3)
}
RefreshActions;

#define COLUMNS 6

/* state */
static GPtrArray * my_articles                   = NULL;
static Group * my_group                          = NULL;
static Filter * my_filter                        = NULL;
static FilterShow my_filter_show                 = FILTER_SHOW_MATCHES_AND_REFERENCES;
static GHashTable * messageid_to_node            = NULL;
static gboolean is_threaded                      = TRUE;
static int sort_type                             = ARTICLE_SORT_SUBJECT;
static GStaticMutex article_ctree_lock           = G_STATIC_MUTEX_INIT;
static GStaticMutex hash_mutex                   = G_STATIC_MUTEX_INIT;

/* article list */
static GnomePixmap * old_read_pixmap             = NULL;
static GnomePixmap * old_unread_pixmap           = NULL;
static GnomePixmap * new_read_pixmap             = NULL;
static GnomePixmap * new_unread_pixmap           = NULL;

static GnomePixmap * article_cached_pixmap       = NULL;
static GnomePixmap * articlelist_closed_pixmap   = NULL;
static GnomePixmap * articlelist_open_pixmap     = NULL;
static GnomePixmap * binary_decoded_pixmap       = NULL;
static GnomePixmap * decode_queue_pixmap         = NULL;
static GnomePixmap * error_pixmap                = NULL;
static GnomePixmap * flagged_for_download_pixmap = NULL;
static GnomePixmap * mp_complete_pixmap          = NULL;
static GnomePixmap * mp_incomplete_pixmap        = NULL;
static GnomePixmap * old_mp_complete_pixmap      = NULL;
static GnomePixmap * old_mp_incomplete_pixmap    = NULL;

static void tree_select_row_cb (GtkCTree*, GtkCTreeNode*, gint, gpointer);

PanCallback * articlelist_thread_changed         = NULL;
PanCallback * articlelist_selection_changed      = NULL;

static GtkStyle * normal_style[2]                = { NULL, NULL };
static GtkStyle * read_style[2]                  = { NULL, NULL };
static GtkStyle * watched_style[2]               = { NULL, NULL };
static GtkStyle * killfile_style[2]              = { NULL, NULL };

GdkColor killfile_color;
GdkColor watched_color;
GdkColor thread_normal_color;
GdkColor thread_read_color;

static GnomeUIInfo header_pane_popup_uiinfo [];
GtkWidget * header_pane_popup_menu;
GnomeUIInfo articlelist_menu [];


/**********************************************************
 * Private Functions
 **********************************************************/

static void
refresh_nolock_impl (RefreshActions actions);

static void
apply_filter_tests (Filter * filter, GPtrArray * headers);

static void
fire_group_changed (Group * g);

static void
articlelist_menu_popup_nolock (GdkEventButton* bevent);
static void
articlelist_unread_inc_nolock (GtkCTreeNode * child, int inc);

static void
articlelist_repopulate_nolock (const Group* group,
                               Article** article_buf,
                       	       int article_qty,
                               GPtrArray * sel_articles);

static GtkCTreeNode* articlelist_get_node_from_message_id (
	const char* message_id);

static void articlelist_update_node_impl (const Article* article,
                                          gboolean need_lock);

static void articlelist_click_column (GtkCList*, int col_clicked);

static void article_ctree_destroy_cb (void);

static void articlelist_update_node_fast_nolock (GtkCTreeNode*, const Article*);

/***
****
****
****  INTERNAL UTILITIES:  SELECTIONS, ARTICLES, and NODES
****
****
***/


void
articlelist_select_all_nolock (void)
{
	gtk_ctree_select_recursive (GTK_CTREE(Pan.article_ctree), NULL);
}

void
articlelist_deselect_all_nolock (void)
{
	gtk_ctree_unselect_recursive (GTK_CTREE(Pan.article_ctree), NULL);
}

static Article*
articlelist_get_article_from_node (GtkCTreeNode* node)
{
	Article* article = NULL;

	if (node != NULL)
		article = ARTICLE(
			  gtk_ctree_node_get_row_data (
				  GTK_CTREE(Pan.article_ctree), node));

	return article;
}

static GPtrArray*
articlelist_get_selected_nodes_nolock (void)
{
	GPtrArray * a;
	const GList * l;
	debug_enter ("articlelist_get_selected_nodes_nolock");

       	a = g_ptr_array_new ();
	pan_g_ptr_array_reserve (a, 64);
	for (l=GTK_CLIST(Pan.article_ctree)->selection; l!=NULL; l=l->next)
		g_ptr_array_add (a, GTK_CTREE_NODE(l->data));

	debug_exit ("articlelist_get_selected_nodes_nolock");
	return a;
}

GtkCTreeNode*
articlelist_get_selected_node (void)
{
	GtkCTreeNode* node = NULL;

	const GList* list =  GTK_CLIST(Pan.article_ctree)->selection;
	if (list != NULL)
		node = GTK_CTREE_NODE(list->data);

	return node;
}


GPtrArray*
articlelist_get_selected_articles_nolock (void)
{
	GPtrArray * retval;
	GList * list;
	debug_enter ("articlelist_get_selected_articles");

	retval = g_ptr_array_new ();

	list = GTK_CLIST(Pan.article_ctree)->selection;
	pan_g_ptr_array_reserve (retval, 64);
	for (; list!=NULL; list=list->next)
	{
		GtkCTreeNode * n = GTK_CTREE_NODE(list->data);
		Article * a = articlelist_get_article_from_node(n);
		if (a != NULL)
			g_ptr_array_add (retval, a);
	}

	debug_exit ("articlelist_get_selected_articles");
	return retval;
}

Article*
articlelist_get_selected_article_nolock (void)
{
	return articlelist_get_article_from_node (
		articlelist_get_selected_node());
}

void
articlelist_set_selected_nodes_nolock (GtkCTreeNode **nodes, gint node_qty)
{
	gint i;
	gfloat hadj;
	GtkCTree * tree;
	GtkCList * list;
	GtkCTreeNode * first = NULL;

	if (nodes==NULL || node_qty<=0)
		return;

	tree = GTK_CTREE(Pan.article_ctree);
	list = GTK_CLIST(Pan.article_ctree);
	gtk_clist_freeze (list);
	gtk_ctree_unselect_recursive (tree, gtk_ctree_node_nth(tree, 0));

	for (i=0; i<node_qty; ++i)
	{
		GtkCTreeNode * n = NULL;
		GtkCTreeNode * node = nodes[i];

		if (node != NULL)
		{
			/* remember the first good node */
			if (first == NULL)
				first = node;

			/* select the node */
			gtk_ctree_select (tree, node);

			/* show all the children */
			for (n=GTK_CTREE_ROW(node)->parent; n; n=GTK_CTREE_ROW(n)->parent)
				gtk_ctree_expand(tree, n);
		}
	}

	/* hack to sync the focus row on new selection -
	 * thanks to Julien Plissonneau Duquene. */
	if (list->selection != NULL)
		list->focus_row = g_list_position (list->row_list,
		                                   (GList*)(list->selection->data));

	/*
	 * Due to a bug in GtkCList this won't work right if the window
	 * isn't drawable.  In that case we can just set a flag and
	 * handle it later.  FIXME (gtk 1.2.11 || gtk 2.0): fixed in gtk cvs;
	 * remove this when we raise the gtk requirement.
	 */ 
	if (!GTK_WIDGET_DRAWABLE(GTK_WIDGET(list))) {
		gtk_object_set_data (GTK_OBJECT(list), "pan_moveto_hack", GINT_TO_POINTER(1));
	} else if (first!=NULL) {
		hadj = gtk_clist_get_hadjustment(list)->value;
		gtk_ctree_node_moveto (tree, nodes[0], 0, (gfloat)0.5, (gfloat)0.0);
		gtk_adjustment_set_value (gtk_clist_get_hadjustment(list), hadj);
	}

	gtk_clist_thaw (list);
}

void
articlelist_expand_selected_threads (void)
{
	guint i;
	GPtrArray * a;
	GtkCTree * tree;
	debug_enter ("articlelist_expand_selected_threads");

	/* expand */
	pan_lock ();
	a = articlelist_get_selected_nodes_nolock ();
	tree = GTK_CTREE(Pan.article_ctree);
	for (i=0; i<a->len; ++i)
		gtk_ctree_expand (tree, (GtkCTreeNode*)g_ptr_array_index(a,i));
	pan_unlock ();

	/* cleanup */
	g_ptr_array_free (a, TRUE);
	debug_exit ("articlelist_expand_selected_threads");
}
void
articlelist_collapse_selected_threads (void)
{
	guint i;
	GPtrArray * a;
	GtkCTree * tree;
	debug_enter ("articlelist_collapse_selected_threads");

	/* collapse */
	pan_lock ();
	a = articlelist_get_selected_nodes_nolock ();
	gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));
	tree = GTK_CTREE(Pan.article_ctree);
	for (i=0; i<a->len; ++i)
		gtk_ctree_collapse (tree, (GtkCTreeNode*)g_ptr_array_index(a,i));
	gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));
	pan_unlock ();

	/* cleanup */
	g_ptr_array_free (a, TRUE);
	debug_exit ("articlelist_collapse_selected_threads");
}


/***
****
****
****  INTERNAL UTILITIES:   GETTING THREADS
****
****
***/

static void
add_to_selection_nolock (ThreadGet get)
{
	gint i;
       	GPtrArray * sel = articlelist_get_selected_articles_nolock ();
       	GPtrArray * all = article_get_unique_threads (sel, get);
	GPtrArray * nodes = g_ptr_array_new ();

	for (i=0; i<all->len; ++i) {
		const Article * a = ARTICLE(g_ptr_array_index(all,i));
		const gchar * message_id = article_get_message_id (a);
		GtkCTreeNode * node = articlelist_get_node_from_message_id (message_id);
		if (node != NULL)
			g_ptr_array_add (nodes, node);
	}

	articlelist_set_selected_nodes_nolock ((GtkCTreeNode**)nodes->pdata, nodes->len);

	g_ptr_array_free (nodes, TRUE);
	g_ptr_array_free (all, TRUE);
	g_ptr_array_free (sel, TRUE);
}

void
articlelist_add_replies_to_selection_nolock (void)
{
	add_to_selection_nolock (GET_SUBTHREAD);
}
void
articlelist_add_thread_to_selection_nolock (void)
{
	add_to_selection_nolock (GET_WHOLE_THREAD);
}


/***
****
****
****  NAVIGATION
****
****
***/

GtkCTreeNode*
articlelist_node_next (GtkCTreeNode * node)
{
	GtkCTreeNode* n;

	if (!node)
		return NULL;

	if (GTK_CTREE_ROW(node)->children)
		return GTK_CTREE_ROW(node)->children;

	for (n=node; n!=NULL; n=GTK_CTREE_ROW(n)->parent)
		if (GTK_CTREE_ROW(n)->sibling)
			return GTK_CTREE_ROW(n)->sibling;

	return NULL;
}

GtkCTreeNode*
articlelist_node_prev (GtkCTreeNode * node, gboolean sibling_only)
{
	GtkCTreeNode * parent;
	GtkCTreeNode * sibling;
	
	if (!node)
		return NULL;

	/* get parent */	
	parent = GTK_CTREE_ROW(node)->parent;
	if (!parent)
		return NULL;

	/* parent's first child */
	sibling=GTK_CTREE_ROW(parent)->children;
	if (sibling==node) /* this is the first child */
		return parent;

	/* previous sibling of node */
	while (GTK_CTREE_ROW(sibling)->sibling != node) {
		sibling = GTK_CTREE_ROW(sibling)->sibling;
		pan_warn_if_fail (sibling != NULL);
	}

	if (sibling_only)
		return sibling;

	/* find the absolutely last child of the older sibling */
	for (;;) {
		GtkCTreeNode* tmp = GTK_CTREE_ROW(sibling)->children;
		if (!tmp)
			return sibling;

		/* last child of sibling */
		while (GTK_CTREE_ROW(tmp)->sibling != NULL)
			tmp = GTK_CTREE_ROW(tmp)->sibling;

		sibling = tmp;
	}

	pan_warn_if_reached ();
}

static GtkCTreeNode*
find_prev_thread (GtkCTreeNode * node)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode * top = gtk_ctree_node_nth (tree, 0);
	const int row_qty = GTK_CLIST(tree)->rows;
	gboolean wrap = FALSE;

	/* nothing to do */
	if (row_qty < 2)
		return NULL;

	/* get starting point */	
	if (node == NULL)
		node = articlelist_get_selected_node ();

	/* find previous */
	if (node != NULL)
		while ( node!=top && GTK_CTREE_ROW(node)->parent!=top )
			node = GTK_CTREE_ROW(node)->parent;

	/* either no node selected or we're at the first thread; wrap */
	if (!node || articlelist_node_prev(node,TRUE)==top) {
		wrap = TRUE;
		node = gtk_ctree_node_nth (tree, row_qty-1);
		while (node!=top && GTK_CTREE_ROW(node)->parent!=top)
			node = GTK_CTREE_ROW(node)->parent;
	}

	/* if we didn't wrap around, we need to back up one. */
	if (!wrap)
		node = articlelist_node_prev (node, TRUE);

	return node;
}

static GtkCTreeNode*
find_next_thread (GtkCTreeNode * node)
{
	GtkCTree* tree;
	GtkCTreeNode * top;
	GtkCTreeNode * in = node;
	const int row_qty = GTK_CLIST(Pan.article_ctree)->rows;

	tree = GTK_CTREE(Pan.article_ctree);
	top = gtk_ctree_node_nth (tree, 0);

	/* if nothing to read... */
	if (GTK_CLIST(tree)->rows<2)
		return NULL;

	/* find the start node */
	if (node == NULL)
		node = articlelist_get_selected_node ();

	/* walk backwards */
	if (node != NULL)
	{
		while (node!=top && GTK_CTREE_ROW(node)->parent!=top)
			node = GTK_CTREE_ROW(node)->parent;
		if (node==top)
			return NULL;
		node = GTK_CTREE_ROW(node)->sibling;
		if (node== NULL && row_qty>=1) /* end of list -- loop around */
			node = gtk_ctree_node_nth (tree, 1);
		if (node == in) /* detect loop */
			node = NULL;

	}
	else if (row_qty > 1)
	{
		node = gtk_ctree_node_nth (tree, 1);
	}

	return node;
}

static GtkCTreeNode*
find_prev_article (GtkCTreeNode * node)
{
	GtkCTree * tree = GTK_CTREE(Pan.article_ctree);
	const int row_qty = GTK_CLIST(tree)->rows;
	GtkCTreeNode * top = gtk_ctree_node_nth (tree, 0);
	gboolean wrap = FALSE;
	debug_enter ("find_prev_article");

	/* if nothing to read... */
	if (row_qty<2)
		return NULL;

	/* get a starting node... */
	if (node==NULL) {
		node = articlelist_get_selected_node (); 
		if ( !node || articlelist_node_prev(node,TRUE)==top ) {
			wrap = TRUE;
			node = gtk_ctree_node_nth (tree, row_qty-1 );
		}
	}

	/* if we didn't wrap, we need to move backwards by hand */
	if (!wrap)
		node = articlelist_node_prev (node, FALSE);

	debug_exit ("find_prev_article");
	return node;
}

static GtkCTreeNode*
find_next_article (GtkCTreeNode * node)
{
	GtkCTreeNode * retval = NULL;
	GtkCTree * tree = GTK_CTREE(Pan.article_ctree);

	/* if nothing to read... */
	if (GTK_CLIST(tree)->rows <2)
		return NULL;

	/* get starting point */
	if (!node) {
		node = articlelist_get_selected_node ();
		if (!node)
			node = gtk_ctree_node_nth (tree, 0);
	}

	/* move to next node */
	retval = articlelist_node_next (node);
	if (!retval)
		retval = gtk_ctree_node_nth (tree, 1);

	return retval;
}

static void
select_and_maybe_read_nolock (GtkCTreeNode * node, gboolean do_read)
{
	GtkCTreeNode * sel = articlelist_get_selected_node ();

	if (node!=NULL && node!=sel)
	{
		articlelist_set_selected_nodes_nolock (&node, 1);
		if (do_read)
		{
			Article * a = articlelist_get_article_from_node (node);
			if (a != NULL)
				queue_add(TASK(task_body_new(a)));
		}
	}
}

void
articlelist_read_next (gboolean cached_only, gboolean do_read)
{
	Article * a = NULL;
	GtkCTreeNode * in;
	GtkCTreeNode * node;

	/* find the first "next" article */
	if ((node = articlelist_get_selected_node ()) == NULL)
		node = gtk_ctree_node_nth (GTK_CTREE(Pan.article_ctree), 1);
	else
		node = find_next_article (node);

	/* test until we find one that matches our criteria */
	in = node;
	do {
		a = articlelist_get_article_from_node (node);
		if (!a)
			break;
		if (!cached_only)
			break;
		if (article_has_body (a))
			break;
		if ((node = find_next_article (node)) == in) {
			node = NULL;
			break;
		}
	} while (a!=NULL);

	select_and_maybe_read_nolock (node, do_read);
}

void
articlelist_read_next_unread (void)
{
	Article * a = NULL;
	GtkCTreeNode * node;
	GtkCTreeNode * in;

	if ((node = articlelist_get_selected_node ()) == NULL)
		node = gtk_ctree_node_nth (GTK_CTREE(Pan.article_ctree), 1);
	else
		node = find_next_article (node);
	in = node;

	do {
		if ((a = articlelist_get_article_from_node (node)) == NULL)
			break;
		if (!article_is_read (a)) {
			if (!navigate_cached_only)
				break;
			if (article_has_body(a))
				break;
		}
		if ((node = find_next_article (node)) == in) {
			node = NULL;
			break;
		}
	} while (a!=NULL);

	select_and_maybe_read_nolock (node, navigate_read_on_select);
}

void
articlelist_read_next_thread (void)
{
	GtkCTreeNode * node;
	
	if ((node = articlelist_get_selected_node ()) == NULL)
		node = gtk_ctree_node_nth (GTK_CTREE(Pan.article_ctree), 1);
	else for (;;) {
		Article * a;
		if ((node = find_next_thread (node)) == NULL)
			break;
		if (!navigate_cached_only)
			break;
		a = articlelist_get_article_from_node (node);
		if (a!=NULL && article_has_body(a))
			break;
	}

	select_and_maybe_read_nolock (node, navigate_read_on_select);
}

void
articlelist_read_next_unread_thread (void)
{
	Article * a = NULL;
	GtkCTreeNode * in;
	GtkCTreeNode * node;
	
	if ((node = articlelist_get_selected_node ()) == NULL)
		node = gtk_ctree_node_nth (GTK_CTREE(Pan.article_ctree), 1);
	else
		node = find_next_thread (node);
	in = node;

	do {
		a = articlelist_get_article_from_node (node);
		if ((a = articlelist_get_article_from_node (node)) == NULL)
			break;
		if (!article_is_read (a)) {
			if (!navigate_cached_only)
				break;
			if (article_has_body (a))
				break;
		}
		if ((node = find_next_article (node)) == in) {
			node = NULL;
			break;
		}
	} while (a!=NULL);

	select_and_maybe_read_nolock (node, navigate_read_on_select);
}

void
articlelist_read_prev (void)
{
	Article * a = NULL;
	GtkCTreeNode * node = NULL;

	do {
		node = find_prev_article (node);
		a = articlelist_get_article_from_node (node);
		if (!navigate_cached_only || (a!=NULL && article_has_body(a)))
			break;
	} while (a!=NULL);

	select_and_maybe_read_nolock (node, navigate_read_on_select);
}

void
articlelist_read_prev_thread (void)
{
	GtkCTreeNode * node = NULL;

	for (;;)
	{
		Article * a;
		if ((node = find_prev_thread (node)) == NULL)
			break;
		if (!navigate_cached_only)
			break;
		a = articlelist_get_article_from_node (node);
		if (a!=NULL && article_has_body(a))
			break;
	}

	select_and_maybe_read_nolock (node, navigate_read_on_select);
}

void
articlelist_read_top_of_thread (void)
{
	GtkCTreeNode * n;
	GtkCTreeNode * node = articlelist_get_selected_node ();

	if ((node = articlelist_get_selected_node ()) == NULL)
		return;
	for (n=GTK_CTREE_ROW(node)->parent; n; n=GTK_CTREE_ROW(n)->parent) {
		if (articlelist_get_article_from_node (n) != NULL)
			node = n;
	}

	select_and_maybe_read_nolock (node, navigate_read_on_select);
}

void
articlelist_read_parent (void)
{
	GtkCTreeNode * node = articlelist_get_selected_node ();

	if ((node = articlelist_get_selected_node ()) == NULL)
		return;

	node = GTK_CTREE_ROW(node)->parent;
	if (articlelist_get_article_from_node (node) == NULL)
		node = NULL;

	select_and_maybe_read_nolock (node, navigate_read_on_select);
}

void
articlelist_read_message_id (const gchar * message_id)
{
	GtkCTreeNode * node = articlelist_get_node_from_message_id (message_id);
	if (node != NULL)
		select_and_maybe_read_nolock (node, TRUE);
}



/***
****
****
****  CLEARING OUT THE ARTICLELIST
****
****
***/

static void
clear_hash_table (void)
{
	g_static_mutex_lock (&hash_mutex);
	if (messageid_to_node != NULL)
		g_hash_table_destroy (messageid_to_node);
	messageid_to_node = g_hash_table_new (g_str_hash, g_str_equal);
	g_static_mutex_unlock (&hash_mutex);
}


/***
****
****
****   SORTING
****
****
***/

static const gchar*
column_to_title (int col)
{
	switch (col) {
		case 0: return " ";
		case 1: return " ";
		case 2: return _("Subject");
		case 3: return _("Lines");
		case 4: return _("Author");
		case 5: return _("Date");
		default: break;
	}
	
	pan_warn_if_reached();
	return _("BUG!!");
}

static int
sort_type_to_column (int type)
{
	int col = 0;
	switch (abs(type)) {
		case ARTICLE_SORT_READ_STATE:   col=0; break;
		case ARTICLE_SORT_ACTION_STATE: col=1; break;
		case ARTICLE_SORT_SUBJECT:      col=2; break;
		case ARTICLE_SORT_LINES:        col=3; break;
		case ARTICLE_SORT_AUTHOR:       col=4; break;
		case ARTICLE_SORT_DATE:         col=5; break;
		default: pan_warn_if_reached(); break;
	}

	return col;
}

static void
articlelist_set_sort_bits_nolock (int type)
{
	if (type != sort_type)
	{
		gchar * pch;
		int col = 0;

		/* update old column */
		if (abs(sort_type) != abs(type)) {
			col = sort_type_to_column (sort_type);
			gtk_clist_set_column_title (GTK_CLIST(Pan.article_ctree), col, column_to_title(col));
		}

		/* update sort type, fire callbacks */
		sort_type = type;

		/* update new column */
		col = sort_type_to_column (sort_type);
		pch = g_strdup_printf ("%c%s", (type>0?'+':'-'), column_to_title(col));
		gtk_clist_set_column_title (GTK_CLIST(Pan.article_ctree), col, pch);
		g_free (pch);
	}
}

static void
articlelist_set_sort_type (int sort)
{
	articlelist_set_sort_bits_nolock (sort);

	if (my_group != NULL) {
		my_group->sort_style = sort;
		group_set_dirty (my_group);
	}

	refresh_nolock_impl (REFRESH_SORT);
}

static void
articlelist_set_filter_bits_nolock (const Filter * f, FilterShow show)
{
	debug_enter ("articlelist_set_filter_bits_nolock");

	if (my_group!=NULL && f!=NULL)
	{
		Filter * old_filter;

		/* update filter variable */
		my_filter_show = show;
		old_filter = my_filter;
		my_filter = filter_dup (f);
		if (old_filter != NULL)
			pan_object_unref (PAN_OBJECT(old_filter));
	}

	debug_exit ("articlelist_set_filter_bits_nolock");
}

void
articlelist_set_filter (const Filter * f, FilterShow show)
{
	debug_enter ("articlelist_set_filter_bits_nolock");

	articlelist_set_filter_bits_nolock (f, show);
	refresh_nolock_impl (REFRESH_FILTER);

	debug_exit ("articlelist_set_filter_bits_nolock");
}


/***
****
****
****
***/

Group*
articlelist_get_group (void)
{
	return my_group;
}


/***
****
****  ARTICLES ADDED
****
***/

static void
group_articles_added_cb (gpointer call_obj, gpointer call_arg, gpointer client_arg)
{
	Group * group = GROUP(call_obj);
	GPtrArray * added = (GPtrArray*) call_arg;
	debug_enter ("group_articles_added_cb");

	/* sanity checks */
	g_return_if_fail (group_is_valid(group));
	g_return_if_fail (added!=NULL);
	g_return_if_fail (added->len>0);

	if (group == my_group) {
		pan_lock ();
		pan_g_ptr_array_append (my_articles, added->pdata, added->len);
		refresh_nolock_impl (~0);
		pan_unlock ();
	}

	debug_exit ("group_articles_added_cb");
}

/***
****
****  ARTICLES REMOVED
****
***/

typedef struct
{
	Article * a;
	gint depth;
}
SortDepthStruct;

static int
compare_pSDS_to_pSDS_by_depth (const void * a, const void * b)
{
	return ((SortDepthStruct*)a)->depth -
	       ((SortDepthStruct*)b)->depth;
}

static void
sort_articles_by_depth (GPtrArray * articles, gboolean parents_first)
{
	guint i;
	SortDepthStruct * buf;
	debug_enter ("sort_articles_by_depth");

	/* make an array of article,depth pairs */
	buf = g_new0 (SortDepthStruct, articles->len);
	for (i=0; i!=articles->len; ++i)
	{
		SortDepthStruct * item = buf + i;
		Article * tmp;
		item->a = ARTICLE(g_ptr_array_index(articles,i));
		item->depth = 0;
		for (tmp=item->a; tmp->parent!=NULL; tmp=tmp->parent)
			++item->depth;
	}

	/* sort the array */
	qsort (buf,
	       articles->len,
	       sizeof(SortDepthStruct),
	       compare_pSDS_to_pSDS_by_depth);


	/* put back into the original GPtrArray */
	for (i=0; i!=articles->len; ++i)
		g_ptr_array_index(articles,i)= buf[parents_first ? i : articles->len-1-i].a;

	/* cleanup */
	g_free (buf);
	debug_exit ("sort_articles_by_depth");
}

/**
 * This is for selecting the next non-deleted article.
 * Given the list of articles (about to be deleted), we return the next
 * article in the articlelist which is not in the articles array.
 */
static Article*
articlelist_get_next_article_not_in_list (GPtrArray * articles)
{
	guint i = 0;
	Article * a = NULL;
	Article * retval = NULL;
	Article * first = NULL;
	GHashTable * h = NULL;
	GtkCTreeNode * node = NULL;

	/* sanity clause */
	g_return_val_if_fail (articles!=NULL, NULL);
	g_return_val_if_fail (articles->len!=0, NULL);

	/* build a hashtable of doomed articles for quick lookup */
	h = g_hash_table_new (g_direct_hash, g_direct_equal);
	for (i=0; i<articles->len; ++i) {
		gpointer g = g_ptr_array_index (articles, i);
		if (g != NULL)
			g_hash_table_insert (h, g, g);
	}

	/* now walk through articles, looking for a doomed one */
	node = articlelist_get_selected_node ();
	first = articlelist_get_article_from_node (node);
	while (retval==NULL)
	{
		node = find_next_article (node);
		a = articlelist_get_article_from_node (node);
		if (a==NULL || a==first) /* end of list */
			break;
		if (g_hash_table_lookup (h, a) == NULL)
			retval = a;
	}

	/* cleanup */
	g_hash_table_destroy (h);
	return retval;
}

static void
remove_article_nolock (Article * article)
{
	const gchar * message_id;
	GtkCTreeNode * node = NULL;
	GtkCTree * ctree = GTK_CTREE(Pan.article_ctree);

	/* sanity checks */
	g_return_if_fail (article_is_valid(article));

	/* if this article currently has a node, remove the node */
	message_id = article_get_message_id (article);
	node = articlelist_get_node_from_message_id (message_id);
	if (node!=NULL)
	{
		GSList *l=NULL, *l2;
		GtkCTreeNode * child;

		/* make a list of children */
		child = GTK_CTREE_ROW(node)->children;
		while (child != NULL) {
			l = g_slist_append (l, child);
			child = GTK_CTREE_ROW(child)->sibling;
		}

		/* for each child, reparent to the granparent */
		for (l2=l; l2!=NULL; l2=l2->next) {
			GtkCTreeNode * child = (GtkCTreeNode*) l2->data;
			gtk_ctree_move (ctree, child,
			                GTK_CTREE_ROW(node)->parent,
			                GTK_CTREE_ROW(node)->sibling);
		}

		g_slist_free (l);

		/* remove from hashtable */
		g_static_mutex_lock (&hash_mutex);
		g_hash_table_remove (messageid_to_node, message_id);
		g_static_mutex_unlock (&hash_mutex);

		/* don't remove the node yet; we need it
		   for article_unread_inc () below */
	}

	/* if the article was unread, update the parents' unread children count */
	if (!article_is_read (article))
	{
		GtkCTreeNode *n = node;
		if (n == NULL)
		{
			Article* a;
			for (a=article->parent; a!=NULL && !n; a=a->parent)
				n = articlelist_get_node_from_message_id (article_get_message_id(article));
		}
		if (n != NULL) 
		{
			articlelist_unread_inc_nolock (n, -1);
		}
	}

	/* remove this node from the tree */
	if (node != NULL)
		gtk_ctree_remove_node (ctree, node);
}


static gint
articles_removed_mainthread (gpointer p)
{
	ArgSet * argset;
	Group * group;
	GPtrArray * articles;
	gint i;
	Article * next = NULL;
	const gboolean has_selection = GTK_CLIST(Pan.article_ctree)->selection != NULL;
	debug_enter ("articles_removed_mainthread");

	/* pump out the arguments */
	argset = (ArgSet*) p;
	group = (Group*) argset_get (argset, 0);
	articles = (GPtrArray*) argset_get (argset, 1);

	/* sort the articles so that we remove the children first --
	   this reduces the number of cnodes we have to reparent */
	sort_articles_by_depth (articles, FALSE);

	if (has_selection)
		next = articlelist_get_next_article_not_in_list (articles);

	/* lock */
	g_static_mutex_lock (&article_ctree_lock);
	pan_lock_unconditional ();
	gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));

	/* remove the articles from the ctree */
	pan_g_ptr_array_foreach (articles, (GFunc)remove_article_nolock, NULL);
	/* try to select the next article that wasn't deleted */
	if (next != NULL) {
		GtkCTreeNode * n = articlelist_get_node_from_message_id (article_get_message_id(next));
		if (n != NULL)
			select_and_maybe_read_nolock (n, FALSE);
	}

	/* unlock */
	gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));
	pan_unlock_unconditional ();
	g_static_mutex_unlock (&article_ctree_lock);

	/* remove them from our array */
	for (i=0; i<articles->len; ++i) {
		Article * needle = ARTICLE(g_ptr_array_index(articles,i));
		g_ptr_array_remove (my_articles, needle);
	}

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	group_unref_articles (group, NULL);
	argset_free (argset);

	debug_exit ("articles_removed_mainthread");
	return 0;
}

static void
group_articles_removed_cb (gpointer call_obj, gpointer call_arg, gpointer client_data)
{
	Group * group = GROUP(call_obj);
	GPtrArray * removed = (GPtrArray*) call_arg;

	/* if it's not important to us, don't bother */
	if (group == my_group)
	{
		ArgSet * argset;

		pan_lock ();
		group_ref_articles (group, NULL);
	       	argset = argset_new2 (group, pan_g_ptr_array_dup(removed));
		run_in_main_thread_nolock (articles_removed_mainthread, argset);
		pan_unlock ();
	}
}


/****
*****
*****  MARKING ARTICLES READ/UNREAD
*****
****/

static void
articlelist_mark_articles_nolock (const GPtrArray * articles, gboolean read)
{
	g_return_if_fail (articles!=NULL);
	g_return_if_fail (articles->len!=0);

	articles_set_read ((Article**)articles->pdata, articles->len, read);
}

static void
articlelist_all_mark_read (void)
{
	if (my_group != NULL)
	{
		guint i;

		/* set unread_children to zero for all. */
		for (i=0; i!=my_articles->len; ++i)
			ARTICLE(g_ptr_array_index(my_articles,i))->unread_children = 0;
	}
}

static void
articlelist_selected_mark_unread_nolock (void)
{
	GPtrArray * sel;
	debug_enter ("articlelist_selected_mark_unread_nolock");

	sel = articlelist_get_selected_articles_nolock ();
	if (sel->len != 0)
		articlelist_mark_articles_nolock (sel, FALSE);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("articlelist_selected_mark_unread_nolock");
}
static void
articlelist_selected_mark_read_nolock (void)
{
	GPtrArray * sel;
	debug_enter ("articlelist_selected_mark_read_nolock");

       	sel = articlelist_get_selected_articles_nolock ();
	if (sel->len != 0)
		articlelist_mark_articles_nolock (sel, TRUE);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("articlelist_selected_mark_read_nolock");
}

static void
articlelist_selected_mark_threads_nolock_impl (int threads, gboolean read)
{
	GPtrArray * sel;
	GPtrArray * all;
	debug_enter ("articlelist_selected_mark_threads_impl");

       	sel = articlelist_get_selected_articles_nolock ();
       	all = article_get_unique_threads (sel, threads);
	if (all->len != 0)
		articlelist_mark_articles_nolock (all, read);
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (all, TRUE);

	debug_exit ("articlelist_selected_mark_threads_impl");
}

static void
articlelist_selected_mark_unread_thread_nolock (void)
{
	articlelist_selected_mark_threads_nolock_impl (GET_WHOLE_THREAD, FALSE);
}
static void
articlelist_selected_mark_read_thread_nolock (void)
{
	articlelist_selected_mark_threads_nolock_impl (GET_WHOLE_THREAD, TRUE);
}


static void
articlelist_selected_unflag_for_dl_nolock (void)
{
	GPtrArray * s;
	debug_enter ("articlelist_selected_unflag_for_dl_nolock");

       	s = articlelist_get_selected_articles_nolock ();
	articles_remove_flag ((Article**)s->pdata, s->len, STATE_DOWNLOAD_FLAGGED);
	g_ptr_array_free (s, TRUE);

	debug_exit ("articlelist_selected_unflag_for_dl_nolock");
}

static void
articlelist_selected_flag_for_dl_nolock (void)
{
	GPtrArray * s;
	debug_enter ("articlelist_selected_flag_for_dl_nolock");

       	s = articlelist_get_selected_articles_nolock ();
	articles_add_flag ((Article**)s->pdata, s->len, STATE_DOWNLOAD_FLAGGED);
	g_ptr_array_free (s, TRUE);

	debug_exit ("articlelist_selected_flag_for_dl_nolock");
}


static void
articlelist_download_flagged_bodies (void)
{
	debug_enter ("articlelist_download_flagged_bodies");

	if (my_group != NULL)
	{
		guint i;
		GPtrArray * flagged = g_ptr_array_new ();

		/* build an array of flagged articles */
		pan_g_ptr_array_reserve (flagged, my_articles->len);
		for (i=0; i!=my_articles->len; ++i) {
			Article * a = ARTICLE(g_ptr_array_index(my_articles,i));
			if (article_flag_on (a, STATE_DOWNLOAD_FLAGGED))
				g_ptr_array_add (flagged, a);
		}

		/* queue a task */
		if (flagged->len != 0) {
			Task * task = TASK(task_bodies_new (my_group, flagged));
			if (task != NULL)
				queue_add (task);
		}

		/* cleanup */
		g_ptr_array_free (flagged, TRUE);
	}

	debug_exit ("articlelist_download_flagged_bodies");
}

/****
*****
*****  COPYING TO FOLDERS
*****
****/


static void
articlelist_article_to_folder (const GPtrArray * articles, gboolean keep_original)
{
	Group * folder;
	gboolean success;
	debug_enter ("articlelist_article_to_folder");

	/* make sure the user's selected a folder. */
       	folder = grouplist_get_selected_group ();
	if (folder==NULL || !group_is_folder(folder)) {
		pan_error_dialog (_("Please Select a Folder from the Group list."));
		return;
	}

	success = article_copy_articles_to_folder (folder,
	                                           (Article**)articles->pdata,
	                                           articles->len);

	if (success && !keep_original)
		group_remove_articles (my_group, articles);

	debug_exit ("articlelist_article_to_folder");
}
static void
articlelist_selected_copy_to_folder_nolock (void)
{
	GPtrArray * sel;
	debug_enter ("articlelist_selected_copy_to_folder");

       	sel = articlelist_get_selected_articles_nolock ();
	articlelist_article_to_folder (sel, TRUE);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("articlelist_selected_copy_to_folder");
}
static void
articlelist_selected_copy_to_folder_deep_nolock (void)
{
	GPtrArray * sel;
	GPtrArray * all;
	debug_enter ("articlelist_selected_copy_to_folder_deep_nolock");

	sel = articlelist_get_selected_articles_nolock ();
	all = article_get_unique_threads (sel, GET_SUBTHREAD);
	articlelist_article_to_folder (all, TRUE);
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (all, TRUE);

	debug_exit ("articlelist_selected_copy_to_folder_deep_nolock");
}
static void
articlelist_selected_move_to_folder_nolock (void)
{
	GPtrArray * sel;
	debug_enter ("articlelist_selected_move_to_folder_nolock");

       	sel = articlelist_get_selected_articles_nolock ();
	articlelist_article_to_folder (sel, FALSE);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("articlelist_selected_move_to_folder_nolock");
}
static void
articlelist_selected_move_to_folder_deep_nolock (void)
{
	GPtrArray * sel;
	GPtrArray * all;
	debug_enter ("articlelist_selected_move_to_folder_deep_nolock");

       	sel = articlelist_get_selected_articles_nolock ();
       	all = article_get_unique_threads (sel, GET_SUBTHREAD);
	articlelist_article_to_folder (all, FALSE);
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (all, TRUE);

	debug_exit ("articlelist_selected_move_to_folder_deep_nolock");
}

static void
articlelist_selected_cancel (void)
{
	Article * a;
	debug_enter ("articlelist_selected_cancel");

       	a = articlelist_get_selected_article_nolock ();
	if (a != NULL)
		article_cancel (a);

	debug_exit ("articlelist_selected_cancel");
}

static void
articlelist_selected_supersede (void)
{
	Article * a;
	debug_enter ("articlelist_selected_supersede");

       	a = articlelist_get_selected_article_nolock ();
	if (a != NULL)
		article_supersede (a);

	debug_exit ("articlelist_selected_supersede");
}

/****
*****
*****  DOWNLOADING ARTICLE BODIES
*****
****/


static void
articlelist_selected_download_nolock (void)
{
	GPtrArray * sel;
	Task * task;
	debug_enter ("articlelist_selected_download_nolock");

	g_return_if_fail (my_group != NULL);

	/* add the task */
       	sel = articlelist_get_selected_articles_nolock ();
	task = TASK(task_bodies_new(my_group, sel));
	if (task != NULL)
		queue_add (task);

	/* cleanup */
	g_ptr_array_free (sel, TRUE);
	debug_exit ("articlelist_selected_download_nolock");
}

static void
articlelist_selected_thread_download_nolock (void)
{
	GPtrArray * sel;
	GPtrArray * all;
	Task * task;
	debug_enter ("articlelist_selected_thread_download_nolock");

	/* add the task */
	sel = articlelist_get_selected_articles_nolock ();
	all = article_get_unique_threads (sel, GET_SUBTHREAD);
	task = TASK(task_bodies_new(my_group, all));
	if (task != NULL)
		queue_add (task);

	/* cleanup */
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (all, TRUE);

	debug_exit ("articlelist_selected_thread_download_nolock");
}


/***
****
****
****   IGNORING / WATCHING THREADS
****
****
***/

static void
rekill_thread_nolock (Article * article)
{
	guint i;
	GPtrArray * articles;
	gint state;
	debug_enter ("rekill_thread_nolock");

	g_return_if_fail (article != NULL);

	/* get the articles and their state */
	state = thread_get_state (article);
       	articles = g_ptr_array_new ();
	article_get_entire_thread (article, articles);

	/* update the tree */
	gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));
	for (i=0; i!=articles->len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(articles,i));
		articlelist_update_node_impl (a, FALSE);
	}
	gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	debug_exit ("rekill_thread_nolock");
}

static void
articlelist_spamize_nolock (void)
{
	guint i;
	GPtrArray * articles;
	debug_enter ("articlelist_spamize_nolock");

	articles = articlelist_get_selected_articles_nolock ();
	for (i=0; i<articles->len; ++i)
	{
		GtkWidget * w = article_filter_dialog_new (ARTICLE(g_ptr_array_index(articles,i)));
		gnome_dialog_set_parent (GNOME_DIALOG(w), GTK_WINDOW(Pan.window));
	        gtk_widget_show_all (w);
	}

	g_ptr_array_free (articles, TRUE);
	debug_exit ("articlelist_spamize_nolock");
}

static void
articlelist_selected_thread_watch_nolock (void)
{
	guint i;
	GPtrArray * sel;
	debug_enter ("articlelist_selected_thread_watch_nolock");

	sel = articlelist_get_selected_articles_nolock ();
	for (i=0; i<sel->len; ++i)
	{
		Article * a = ARTICLE(g_ptr_array_index(sel,i));
		gint o = thread_get_state (a);
		gint n = o==THREAD_WATCHED ? THREAD_NOTHING : THREAD_WATCHED;
		thread_set_state (a, n);
		rekill_thread_nolock (a);
	}

	g_ptr_array_free (sel, TRUE);
	debug_exit ("articlelist_selected_thread_watch_nolock");
}

static void
articlelist_selected_thread_ignore_nolock (void)
{
	guint i;
	GPtrArray * sel;
	debug_enter ("articlelist_selected_thread_ignore_nolock");

	sel = articlelist_get_selected_articles_nolock ();
	for (i=0; i<sel->len; ++i)
	{
		Article * a = ARTICLE(g_ptr_array_index(sel,i));
		gint o = thread_get_state (a);
		gint n = o==THREAD_IGNORED ? THREAD_NOTHING : THREAD_IGNORED;
		thread_set_state (a, n);
		rekill_thread_nolock (a);
	}

	g_ptr_array_free (sel, TRUE);
	debug_exit ("articlelist_selected_thread_ignore_nolock");
}

/***
****
****
****  POPULATING THE TREE:  UTILITY FUNCTIONS
****
****
***/

static GnomePixmap*
get_column_1_pixmap (const Article * a)
{
	GnomePixmap * retval = NULL;
	const gboolean is_new = article_is_new (a);
	const gboolean is_read = article_is_read (a);

	g_return_val_if_fail (a!=NULL, NULL);

	if (article_flag_on (a, STATE_ERROR))
		retval = error_pixmap;
	else if (article_flag_on (a, STATE_MULTIPART_ALL))
		retval = is_new ? mp_complete_pixmap : old_mp_complete_pixmap;
	else if (article_flag_on (a, STATE_MULTIPART_SOME))
		retval = is_new ? mp_incomplete_pixmap : old_mp_incomplete_pixmap;
	else if (is_new && is_read)
		retval = new_read_pixmap;
	else if (is_new && !is_read)
		retval = new_unread_pixmap;
	else if (!is_new && !is_read)
		retval = old_unread_pixmap;
	else
		retval = old_read_pixmap;

	return retval;
}

static GnomePixmap*
get_column_2_pixmap (const Article * a)
{
	GnomePixmap * retval = NULL;

	g_return_val_if_fail (article_is_valid(a), NULL);

	if (article_flag_on (a, STATE_DOWNLOAD_FLAGGED))
		retval = flagged_for_download_pixmap;
	else if (article_flag_on (a, STATE_DECODED))
		retval = binary_decoded_pixmap;
	else if (article_flag_on (a, STATE_SAVE_QUEUED))
		retval = decode_queue_pixmap;
	else if (article_has_body (a))
		retval = article_cached_pixmap;

	return retval;
}

static GdkFont * normalfont = NULL;
static GdkFont * newrepliesfont = NULL;

static GdkFont*
pan_load_font (const gchar * name)
{
	return use_gdk_fontset_load
		? gdk_fontset_load (name)
		: gdk_font_load (name);
}

static gboolean rebuild_style = TRUE;

static void
articlelist_set_node_style_nolock (GtkCTreeNode   * node,
			           const Article  * article)
{
	GtkCTree* tree = GTK_CTREE (Pan.article_ctree);

	/* initialize the styles/fonts the first time through */
	if (rebuild_style)
	{
		GtkStyle * s = gtk_widget_get_style (GTK_WIDGET(Pan.window));
		gint       i;

		rebuild_style = FALSE;

		for (i=0; i<2; i++)
		{
			GdkFont * f = i ? newrepliesfont : normalfont;

			normal_style[i] = gtk_style_copy (s);
			normal_style[i]->fg[0] = thread_normal_color;
			if (f != NULL)
				normal_style[i]->font = f;

			read_style[i] = gtk_style_copy (s);
			read_style[i]->fg[0] = thread_read_color;
			if (f != NULL)
				read_style[i]->font = f;

			killfile_style[i] = gtk_style_copy (s);
			killfile_style[i]->fg[0] = killfile_color;
			if (f != NULL)
				killfile_style[i]->font = f;

			watched_style[i] = gtk_style_copy (s);
			watched_style[i]->fg[0] = watched_color;
			if (f != NULL)
				watched_style[i]->font = f;
		}
	}

	if (1)
	{
		GtkStyle ** style = NULL;
		const gboolean is_read = article ? article_is_read (article) : FALSE;
		const gboolean is_new = article ? article_is_new (article) : FALSE;
		const gboolean new_children = article ? article->new_children!=0 : FALSE;
		const gboolean unread_children = article ? article->unread_children!=0 : FALSE;
		const gint thread_state = article ? thread_get_state(article) : THREAD_NOTHING;

		if (thread_state == THREAD_IGNORED)
			style = killfile_style;
		else if (thread_state == THREAD_WATCHED)
			style = watched_style;
		else if (!is_read || (is_threaded && unread_children))
			style = normal_style;
		else
			style = read_style;

		gtk_ctree_node_set_row_style (tree, node, style[0]);

		if (is_new || (is_threaded && new_children))
			gtk_ctree_node_set_cell_style (tree, node, 2, 
				style[1]);
	}
}

static void
apply_filter_tests (Filter * filter, GPtrArray * headers)
{
	register int i;
	register Article * a;
	GPtrArray * tmp;
	GPtrArray * family;

	g_return_if_fail (headers!=NULL);

	/* make a working copy of the headers */
	tmp = pan_g_ptr_array_dup (headers);

	/* clear the filter state */
	for (i=0; i!=tmp->len; ++i) {
		a = ARTICLE(g_ptr_array_index(tmp,i));
		a->self_passes_filter = FALSE;
		a->tree_passes_filter = FALSE;
	}

	/* remove the articles that don't pass */
	if (filter != NULL)
		filter_remove_failures (filter, tmp);

	/* mark up the successes */
	family = g_ptr_array_new ();
	for (i=0; i<tmp->len; ++i)
	{
		gint j;
		a = ARTICLE(g_ptr_array_index(tmp,i));

		/* this article passes the filter */
		a->self_passes_filter = TRUE;
		a->tree_passes_filter = TRUE;

		/* process the family */
		g_ptr_array_set_size (family, 0);
		switch (my_filter_show) {
			case FILTER_SHOW_MATCHES:
				break;
			case FILTER_SHOW_MATCHES_AND_REPLIES:
				article_get_subthread (a, family);
				break;
			case FILTER_SHOW_MATCHES_AND_REFERENCES:
				article_get_references (a, family);
				break;
			case FILTER_SHOW_THREAD:
				article_get_entire_thread (a, family);
				break;
		}
		for (j=0; j<family->len; ++j) {
			Article * relative = ARTICLE(g_ptr_array_index(family,j));
			relative->tree_passes_filter = TRUE;
		}
	}

	/* cleanup */
	g_ptr_array_free (tmp, TRUE);
	g_ptr_array_free (family, TRUE);
}

/***
****
****
****  POPULATING THE TREE:  THE DIRTY WORK
****
****
***/

static GtkCTreeNode*
add_article_to_ctree_nolock (GtkCTree      * tree,
                             GtkCTreeNode  * parent,
                             GtkCTreeNode  * sibling,
                             Article       * article,
                             gboolean        expanded)
{
	GnomePixmap * col_1;
	GnomePixmap * col_2;
	gchar line_count_buf[32];
	GtkCTreeNode * node;
	gulong lcountsum;
	gchar * text[COLUMNS] = { NULL, NULL, NULL, NULL, NULL, NULL };
	gchar author[128];
	gchar date[128];
	gint i;

	/* author */
	article_get_short_author_str (article, author, sizeof(author));

	/* date */
	get_date_display_string (article->date, thread_date_format, date, sizeof(date));

	/* line count */
	lcountsum = article->linecount;
	if (article_flag_on (article, STATE_MULTIPART_ALL)) {
		gint part = 2;
		GSList * l;
		for (l=article->threads; l!=NULL; l=l->next) {
			Article * child = ARTICLE(l->data);
			if (child->part == part) {
				lcountsum += child->linecount;
				++part;
			}
		}
	}
	commatize_ulong (lcountsum, line_count_buf);

	/* icons */
	col_1 = get_column_1_pixmap (article);
	col_2 = get_column_2_pixmap (article);
       
	/* populate the line */
	i = 0;
	i++; /* skip icon column 1 */
	i++; /* skip icon column 2 */
	text[i++] = (gchar*) article_get_subject (article);
	text[i++] = line_count_buf;
	text[i++] = (gchar*)author;
	text[i++] = date;
	g_assert (i==COLUMNS);

	/* insert the node */
	node = gtk_ctree_insert_node (tree, parent, sibling,
	                              (char**)text, GNOME_PAD_SMALL,
	                              NULL, NULL, NULL, NULL,
	                              FALSE, expanded);
	gtk_ctree_node_set_pixmap (tree, node, 0, col_1->pixmap, col_1->mask);
	if (col_2 != NULL)
		gtk_ctree_node_set_pixmap (tree, node, 1, col_2->pixmap, col_2->mask);
	gtk_ctree_node_set_row_data (tree, node, article);
	articlelist_set_node_style_nolock (node, article);

	/* update hash table */
	g_static_mutex_lock (&hash_mutex);
	g_hash_table_insert (messageid_to_node, (gpointer)article_get_message_id(article), node);
	g_static_mutex_unlock (&hash_mutex);

	return node;
}

static void
build_article_ctree_recursive (GtkCTree      * tree,
                               GtkCTreeNode  * parent,
                               GSList        * list,
                               gboolean        expanded)
{
	GSList * tmp;
	GSList * pos;
	GtkCTreeNode * sibling = NULL;

	/* sanity check */
	g_return_if_fail (parent!=NULL);

	/* this item gets inserted before the 'sibling' node,
	 * so we must add in reverse order */
	tmp = g_slist_reverse(g_slist_copy(list));
        for (pos=tmp; pos!=NULL; pos=pos->next)
        {
		Article* article = ARTICLE(pos->data);
		GtkCTreeNode* node = NULL;

		/* possibly don't show subtrees that don't measure up to filter */
		if (is_threaded && !article->tree_passes_filter)
			continue;

		/* possibly don't show 2...n of multipart nodes */
		if (hide_mpart_child_nodes
			&& (article->parent!=NULL)
			&& (article->parent->state&STATE_MULTIPART_ALL)
			&& (article->part>1))
			continue;

		node = add_article_to_ctree_nolock (tree, parent, sibling, article, expanded);
		if (article->threads!=NULL)
			build_article_ctree_recursive (tree, node, article->threads, TRUE);

		sibling = node;
        }

	/* cleanup */
	g_slist_free (tmp);
}


static void
articlelist_repopulate_nolock (const Group   * group,
                               Article      ** article_buf,
                               gint            article_qty,
                               GPtrArray     * sel_articles)
{
	GtkCTree * tree = GTK_CTREE (Pan.article_ctree);
	GtkCTreeNode * sibling = NULL;
	GtkCTreeNode * root = NULL;
	gchar * text[COLUMNS] = { NULL, NULL, NULL, NULL, NULL, NULL };
	gint i;
	debug_enter ("articlelist_repopulate_nolock");

	/* clear out old */
	gtk_clist_freeze (GTK_CLIST(tree));
	gtk_clist_clear (GTK_CLIST(tree));
	clear_hash_table ();
	
	/* add root node */
	text[2] = (char*) group_get_readable_name (group);
	root = gtk_ctree_insert_node (tree, NULL, NULL,
	                              text, GNOME_PAD_SMALL,
	                              articlelist_closed_pixmap->pixmap,
	                              articlelist_closed_pixmap->mask,
	                              articlelist_open_pixmap->pixmap,
	                              articlelist_open_pixmap->mask,
	                              FALSE, FALSE);
	gtk_ctree_node_set_row_data (tree, root, NULL);
	articlelist_set_node_style_nolock (root, NULL);

	/* add the articles */
	for (i=article_qty-1; i>=0; --i) /* slightly faster to add to the top of ctree */
	{
		Article * article = article_buf[i];
		GtkCTreeNode * node = NULL;
		gboolean has_parent;

		/* make sure we pass the state filter */
		if (!article)
			continue;
		if (is_threaded && !article->tree_passes_filter)
			continue;
		if (!is_threaded && !article->self_passes_filter)
			continue;

		has_parent = article->parent != NULL;

		if ((!is_threaded && article->part<2) ||
			(is_threaded && (!has_parent || !article->parent->tree_passes_filter)))
			sibling = node = add_article_to_ctree_nolock (tree, root, sibling, article, FALSE);

		/* if threads are turned on, and there are children of this thread... */
		if (is_threaded && node!=NULL && article->threads!=NULL)
			build_article_ctree_recursive (tree, node, article->threads, TRUE);
	}

	/* update selection */
	if (sel_articles!=NULL && sel_articles->len)
	{
		gint i;
		GPtrArray * tmp;

		tmp = g_ptr_array_new ();
		pan_g_ptr_array_reserve (tmp, sel_articles->len);
		for (i=0; i<sel_articles->len; ++i) {
			Article * a = ARTICLE(g_ptr_array_index(sel_articles,i));
			GtkCTreeNode * node = articlelist_get_node_from_message_id (article_get_message_id(a));
			if (node != NULL)
				g_ptr_array_add (tmp, node);
		}

		if (tmp->len)
			articlelist_set_selected_nodes_nolock ((GtkCTreeNode**)tmp->pdata, tmp->len);

		/* cleanup */
		g_ptr_array_free (tmp, TRUE);
	}

	/* finished; display */
	gtk_ctree_expand (tree, root);
	gtk_clist_thaw (GTK_CLIST(tree));
	gtk_widget_queue_resize (Pan.article_ctree);

	debug_exit ("articlelist_repopulate_nolock");
}

/***
****
****
****  UPDATING EXISTING NODES: THE DIRTY WORK
****
****
***/

static void 
articlelist_update_node_fast_nolock (GtkCTreeNode   * node,
                                     const Article  * article)
{
	GtkCTree * ctree = GTK_CTREE(Pan.article_ctree);
	GnomePixmap * col_1 = get_column_1_pixmap (article);
	GnomePixmap * col_2 = get_column_2_pixmap (article);

	gtk_ctree_set_node_info (ctree, node,
	                         article_get_subject(article),
	                         GNOME_PAD_SMALL,
	                         NULL, NULL, NULL, NULL,
	                         GTK_CTREE_ROW(node)->is_leaf,
	                         GTK_CTREE_ROW(node)->expanded );

	gtk_ctree_node_set_pixmap (ctree, node, 0, col_1->pixmap, col_1->mask);
	if (col_2 == NULL)
		gtk_ctree_node_set_text (ctree, node, 1, "");
	else
		gtk_ctree_node_set_pixmap (ctree, node, 1, col_2->pixmap, col_2->mask);

	articlelist_set_node_style_nolock (node, article);
}
static GtkCTreeNode*
articlelist_get_node_from_message_id (const char* message_id)
{
	GtkCTreeNode * retval = NULL;

	if (is_nonempty_string(message_id))
	{
		g_static_mutex_lock (&hash_mutex);
		retval = g_hash_table_lookup (messageid_to_node, message_id);
		g_static_mutex_unlock (&hash_mutex);
	}

	return retval;
}
static void
articlelist_update_node_impl (const Article* article, gboolean need_lock)
{
	GtkCTreeNode* node =
		articlelist_get_node_from_message_id (
			article_get_message_id (article));

        if (node != NULL)
	{
		/* update the article state */
		Article* real_article =
			articlelist_get_article_from_node (node);
		real_article->state = article->state;

		/* repaint */
		if (need_lock)
			pan_lock ();
                articlelist_update_node_fast_nolock (node, real_article);
		if (need_lock)
			pan_unlock ();
	}
}

static gint
articlelist_update_articles_idle (gpointer data)
{
	gint i;
	GPtrArray * a = (GPtrArray*) data;

	/* update the nodes */
	pan_lock_unconditional ();
	for (i=0; i<a->len; ++i)
	{
		const Article * article = ARTICLE(g_ptr_array_index(a,i));
		if (article->group == my_group)
			articlelist_update_node_impl (article, FALSE);
	}
	pan_unlock_unconditional ();

	/* cleanup */
	g_ptr_array_free (a, TRUE);
	return 0;
}

static void
articlelist_update_articles (const Article** a, gint qty)
{
	gint i;
	GPtrArray * tmp;

	/* make a list of all the articles that affect the ctree */
	tmp = g_ptr_array_new ();
	for (i=0; i<qty; ++i)
		if (articlelist_get_node_from_message_id (article_get_message_id(a[i])))
			g_ptr_array_add (tmp, (gpointer)a[i]);

	/* if the ctree needs updating, queue an update */
	if (!tmp->len) {
		g_ptr_array_free (tmp, TRUE);
	} else {
		pan_lock ();
		run_in_main_thread_nolock (articlelist_update_articles_idle, (gpointer)tmp);
		pan_unlock ();
	}
}

/***
****
****
****  MANAGING UNREAD CHILDREN COUNTS
****
****
***/

static void
articlelist_unread_inc_nolock (GtkCTreeNode* node, int inc)
{
	GtkCTreeNode * child = node;
	debug_enter ("articlelist_unread_inc_nolock");

	/* update the node & parents */
	while (node != NULL)
	{
		Article * a = articlelist_get_article_from_node (node);
		if (a == NULL)
			break;

		/* all the parents have one less/more unread child */
		if (child != node)
			a->unread_children = MAX(a->unread_children+inc, 0);

		articlelist_update_node_fast_nolock (node, a);

		node = GTK_CTREE_ROW(node)->parent;
	}

	debug_exit ("articlelist_unread_inc_nolock");
}

static gint
articlelist_articles_read_changed_idle (gpointer data)
{
	gint i;
	GPtrArray * articles;
	GtkCList * clist;
	debug_enter ("articlelist_articles_read_changed_idle");

 	articles = (GPtrArray *) data;
	clist = GTK_CLIST(Pan.article_ctree);

	pan_lock_unconditional ();
	gtk_clist_freeze (clist);
	for (i=0; i!=articles->len; ++i)
	{
		const Article * article = ARTICLE(g_ptr_array_index(articles,i));
		const Article * a = NULL;
		GtkCTreeNode* node = NULL;

		/* find the matching node & article */
		node = articlelist_get_node_from_message_id (article_get_message_id(article));
		if (node != NULL)
			a = articlelist_get_article_from_node (node);

		/* if we've got a match, update the node */
		if (a != NULL) {
			int inc = article_is_read (article) ? -1 : 1;
			articlelist_unread_inc_nolock (node, inc);
		}
	}
	gtk_clist_thaw (clist);
	pan_unlock_unconditional ();

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	debug_exit ("articlelist_articles_read_changed_idle");
	return 0;
}

static void
articlelist_articles_read_changed (const Article ** articles, gint len)
{
	debug_enter ("articlelist_articles_read_changed");

	g_return_if_fail (articles!=NULL);
	g_return_if_fail (len>=1);

	if (my_group == articles[0]->group)
	{
		GPtrArray * a = g_ptr_array_new ();
		pan_g_ptr_array_assign  (a, (gpointer*)articles, len);
		pan_lock ();
		run_in_main_thread_nolock (articlelist_articles_read_changed_idle, a);
		pan_unlock ();
	}

	debug_exit ("articlelist_articles_read_changed");
}



/***
****
****  LIVING WITH ACACHE
****
***/

static gint
articlelist_acache_changed_idle (gpointer data)
{
	gchar * message_id;
	GtkCTreeNode * node;
	debug_enter ("articlelist_acache_changed_idle");

 	message_id = (gchar*) data;
	node = articlelist_get_node_from_message_id (message_id);
        if (node != NULL)
	{
		Article * a = articlelist_get_article_from_node (node);
		pan_lock_unconditional ();
                articlelist_update_node_fast_nolock (node, a);
		pan_unlock_unconditional ();
	}

	g_free (message_id);
	debug_exit ("articlelist_acache_changed_idle");
	return 0;
}

static void
articlelist_acache_changed_cb (gpointer call_object, gpointer call_arg, gpointer user_data)
{
	const gchar * message_id;
	debug_enter ("articlelist_acache_changed_cb");

 	message_id = (const gchar*) call_object;
	if (articlelist_get_node_from_message_id (message_id) != NULL)
	{
		pan_lock ();
		run_in_main_thread_nolock (articlelist_acache_changed_idle, g_strdup(message_id));
		pan_unlock ();
	}

	debug_exit ("articlelist_acache_changed_cb");
}

/***
****
****
****  SELECTION / KEYPRESS / BUTTONPRESS HANDLING
****
****
***/

/**
 * The following buttonpress, keypress, and selection callbacks are all here
 * to interact with each other to do one thing: force this damn GtkCTree to
 * have the UI behavior that we want:
 * 
 * (1) Never activate an article during a group selection
 * 
 * (2) If only a single article is selected by single mouse click or
 *     by up/down arrow, and the 'single click loads' prefs is on,
 *     or if it's selected by a double click, then load the article.
 *
 * The problem with (1) is that the GtkCTree fires a selection event for each
 * row selected, rather than once for the whole range, so there's no way to
 * tell if the first selection event is a single or the beginning of a range.
 * The workaround is to use an idle func that gets invoked after all the
 * selections are done.  It asks the tree what's selected and uses that in
 * the "article selection changed" pan callback.  (2) is handled by the
 * button_click_count variable, which keeps click information from the
 * kepress/buttonpress callbacks and uses it in the selection callback.
 */
static gint button_click_count = -1;
static gint mb = -1;



static void
articlelist_selection_changed_cb (gpointer call_object,
                                  gpointer call_arg,
                                  gpointer user_data)
{
	GPtrArray * articles;
	Article * single_article = NULL;
	debug_enter ("articlelist_selection_changed_cb");

	/* load the specified article */
       	articles = pan_g_ptr_array_dup ((GPtrArray*)call_arg);
	if (articles!=NULL && articles->len==1)
		single_article = ARTICLE(g_ptr_array_index(articles,0));
        if ((single_article!=NULL)
		&& ((button_click_count==2) /* double click */
			|| (button_click_count==1
				&& (  (mb==2 && articlelist_mb2_action==CLICK_ACTION_LOAD)
				    ||(mb==1 && articlelist_mb1_action==CLICK_ACTION_LOAD)))))
	{
		pan_warn_if_fail (single_article->group != NULL);
		pan_warn_if_fail (single_article->group->server != NULL);
		gui_page_set (MESSAGE_PAGE, Pan.text);
		queue_add (TASK(task_body_new(single_article)));
	}

	button_click_count = -1;
	mb = -1;

	debug_exit ("articlelist_selection_changed_cb");
}

static void
articlelist_key_press (GtkWidget      * widget,
                       GdkEventKey    * event,
                       gpointer         data)
{
	switch (event->keyval)
	{
		case GDK_Up: case GDK_Down:
		case GDK_Left: case GDK_Right:
		case GDK_Page_Up: case GDK_Page_Down:
			button_click_count = 1;
			break;
		default:
			button_click_count = -1;
	}
}

static void
articlelist_button_press (GtkWidget* widget, GdkEventButton* bevent)
{
	switch (bevent->button)
	{
		case 1:
		case 2:
			mb = bevent->button;
			if (bevent->type == GDK_2BUTTON_PRESS)
				button_click_count = 2;
			else
				button_click_count = 1;
			break;
		case 3:
			articlelist_menu_popup_nolock (bevent);
			break;
	}
}

static gboolean select_callback_pending = FALSE;
 
static gint
tree_select_row_idle (gpointer data)
{
	GPtrArray * articles;
	debug_enter ("tree_select_row_idle");

	/* let everyone know that the selection has changed */
	pan_lock_unconditional ();
       	articles = articlelist_get_selected_articles_nolock ();
	pan_unlock_unconditional ();
	pan_callback_call (articlelist_selection_changed,
			   Pan.article_ctree,
			   articles);

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	select_callback_pending = FALSE;
	debug_exit ("tree_select_row_idle");
	return FALSE;
}
static void
tree_select_row_cb (GtkCTree     *tree,
		    GtkCTreeNode *node,
		    gint          column,
		    gpointer      data)
{
	if (!select_callback_pending)
	{
		select_callback_pending = TRUE;
		run_in_main_thread_nolock (tree_select_row_idle, NULL);
	}
}

/***
****
****
****
***/

static void
articlelist_click_column (GtkCList* clist, int n)
{
	int type;

	switch (n) {
		case 0: type = ARTICLE_SORT_READ_STATE; break;
		case 1: type = ARTICLE_SORT_ACTION_STATE; break;
		case 2: type = ARTICLE_SORT_SUBJECT; break;
		case 3: type = ARTICLE_SORT_LINES; break;
		case 4: type = ARTICLE_SORT_AUTHOR; break;
		case 5: type = ARTICLE_SORT_DATE; break;
		default: pan_warn_if_reached(); type = -ARTICLE_SORT_DATE; break;
	}

	if (type == sort_type)
		type = -type;

	articlelist_set_sort_type (type);
}

/***
****  Callbacks
***/

static void
articlelist_group_marked_read_cb (gpointer group, gpointer unused1, gpointer unused2)
{
	if (my_group!=NULL && my_group==GROUP(group))
	{
		articlelist_all_mark_read ();
		pan_lock ();
		articlelist_refresh_nolock ();
		pan_unlock ();
	}
}

static void
articlelist_group_emptied_cb (gpointer group_data, gpointer foo, gpointer bar)
{
	const Group * group = GROUP(group_data);

	if (my_group!=NULL && my_group==group)
	{
		pan_lock ();
		articlelist_set_group_nolock (NULL);
		pan_unlock ();
	}
}


static void
articlelist_groups_removed_cb (gpointer server_data,
                               gpointer groups_array,
                               gpointer unused)
{
	guint i;
	gboolean my_group_removed = FALSE;
	const GPtrArray * groups = (const GPtrArray*) groups_array;

	for (i=0; !my_group_removed && i<groups->len; ++i)
		if (GROUP(g_ptr_array_index(groups,i)) == my_group)
			my_group_removed = TRUE;

	if (my_group_removed)
		articlelist_set_group_nolock (NULL);
}

static void
articlelist_articles_changed_cb (gpointer event, gpointer foo, gpointer bar)
{
	ArticleChangeEvent * e = (ArticleChangeEvent*) event;
	switch (e->type)
	{
		case ARTICLE_CHANGED_READ:
			articlelist_articles_read_changed ((const Article**)e->articles, e->article_qty);
			break;
		case ARTICLE_CHANGED_DIRTY:
			articlelist_update_articles ((const Article**)e->articles, e->article_qty);
			break;
		default:
			pan_warn_if_reached ();
			break;
	}
}                                                                                                                               

/***
****
***/

static void
article_ctree_destroy_cb (void)
{
	/* stop listening to events */
	pan_callback_remove (article_get_articles_changed_callback(),
	                     articlelist_articles_changed_cb, NULL);
	pan_callback_remove (server_get_groups_removed_callback(),
	                     articlelist_groups_removed_cb, NULL);
	pan_callback_remove (acache_bodies_added_callback,
	                     articlelist_acache_changed_cb, NULL);
	pan_callback_remove (acache_bodies_removed_callback,
	                     articlelist_acache_changed_cb, NULL);
	pan_callback_remove (articlelist_selection_changed,
	                     articlelist_selection_changed_cb, NULL);
	pan_callback_remove (group_get_group_marked_read_callback(),
	                     articlelist_group_marked_read_cb, NULL);
	pan_callback_remove (group_get_group_emptied_callback(),
	                     articlelist_group_emptied_cb, NULL);

	gtk_widget_destroy (GTK_WIDGET(articlelist_closed_pixmap));
	gtk_widget_destroy (GTK_WIDGET(articlelist_open_pixmap));
	gtk_widget_destroy (GTK_WIDGET(binary_decoded_pixmap));
	gtk_widget_destroy (GTK_WIDGET(decode_queue_pixmap));
	gtk_widget_destroy (GTK_WIDGET(error_pixmap));
	gtk_widget_destroy (GTK_WIDGET(flagged_for_download_pixmap));
	gtk_widget_destroy (GTK_WIDGET(mp_complete_pixmap));
	gtk_widget_destroy (GTK_WIDGET(mp_incomplete_pixmap));
	gtk_widget_destroy (GTK_WIDGET(old_mp_complete_pixmap));
	gtk_widget_destroy (GTK_WIDGET(old_mp_incomplete_pixmap));

	pan_callback_free (articlelist_thread_changed);
	pan_callback_free (articlelist_selection_changed);
}

/*
 * Due to a bug in GtkCList (as of 1.2) we can't moveto a window that
 * isn't drawable.  A object data flag is set if we wanted to move that
 * when we map (right here) we can take care of the move
 */
static void
map_articlelist_cb (GtkWidget * w, gpointer user_data)
{
	if (gtk_object_get_data(GTK_OBJECT(w), "pan_moveto_hack") != NULL)
	{
		const GList * list = GTK_CLIST(w)->selection;
		if (list != NULL)
		{
			pan_lock ();
			gtk_ctree_node_moveto (GTK_CTREE(w),
			                       GTK_CTREE_NODE(list->data),
			                       0, (gfloat)0.5, (gfloat)0.0);
			pan_unlock ();
		}

		gtk_object_remove_data (GTK_OBJECT(w), "pan_moveto_hack");
	}
}


static void
tree_toggle_focus_row_cb (GtkCList   * clist,
                          gpointer     user_data)
{
	/* This is just a hack to allow space-reading when the article ctree
	   has focus.  The space key is used in clists to toggle the row
	   focus, so if we don't stop the signal here ctree calls
	   resync_selection, which reselects the focused node, and makes
	   space reading loop back on the same article each time. */
	gtk_signal_emit_stop_by_name (GTK_OBJECT(clist), "toggle_focus_row");
}


/*--------------------------------------------------------------------
 * generate the listing of articles, for the "Articles" tab
 *--------------------------------------------------------------------*/
GtkWidget*
create_articlelist_ctree (void)
{
	int i;
	GtkWidget * w;
	GtkCList * list;
	GtkCTree * tree;
	GtkWidget * s_window;
	GtkWidget * vbox;
	const char * titles[COLUMNS];

	my_articles = g_ptr_array_new ();

	/* callbacks */
	articlelist_thread_changed = pan_callback_new ();
	articlelist_selection_changed = pan_callback_new ();

	pan_callback_add (article_get_articles_changed_callback(),
	                  articlelist_articles_changed_cb, NULL);
	pan_callback_add (server_get_groups_removed_callback(),
	                  articlelist_groups_removed_cb, NULL);
	pan_callback_add (acache_bodies_added_callback,
	                  articlelist_acache_changed_cb, NULL);
	pan_callback_add (acache_bodies_removed_callback,
	                  articlelist_acache_changed_cb, NULL);
	pan_callback_add (articlelist_selection_changed,
	                  articlelist_selection_changed_cb, NULL);
	pan_callback_add (group_get_group_marked_read_callback(),
	                  articlelist_group_marked_read_cb, NULL);
	pan_callback_add (group_get_group_emptied_callback(),
	                  articlelist_group_emptied_cb, NULL);

	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GNOME_PAD_SMALL);

	/* filter */
	gtk_box_pack_start (GTK_BOX(vbox), article_toolbar_new(), FALSE, FALSE, 0);

	/* get the titles */
	for (i=0; i<COLUMNS; ++i)
		titles[i] = column_to_title (i);

	/* load the pixmaps */
	old_read_pixmap = GNOME_PIXMAP(gnome_pixmap_new_from_xpm_d(old_read_xpm));
	old_unread_pixmap = GNOME_PIXMAP(gnome_pixmap_new_from_xpm_d(old_unread_xpm));
	new_read_pixmap =  GNOME_PIXMAP(gnome_pixmap_new_from_xpm_d(new_read_xpm));
	new_unread_pixmap =  GNOME_PIXMAP(gnome_pixmap_new_from_xpm_d(new_unread_xpm));

	flagged_for_download_pixmap = GNOME_PIXMAP(gnome_pixmap_new_from_xpm_d (flagged_xpm));
	articlelist_closed_pixmap = GNOME_PIXMAP (gnome_stock_pixmap_widget_new (Pan.window, GNOME_STOCK_MENU_BOOK_RED));
	articlelist_open_pixmap = GNOME_PIXMAP (gnome_stock_pixmap_widget_new (Pan.window, GNOME_STOCK_MENU_BOOK_OPEN));
        article_cached_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (disk_xpm));
	decode_queue_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (bluecheck_xpm));
	error_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (x_xpm));
	binary_decoded_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (binary_xpm));
	mp_complete_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (binary_complete_xpm));
	mp_incomplete_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (binary_incomplete_xpm));
	old_mp_complete_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (old_binary_complete_xpm));
	old_mp_incomplete_pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_xpm_d (old_binary_incomplete_xpm));

	clear_hash_table ();

	/* create the widget */
	Pan.article_ctree = w = gtk_ctree_new_with_titles (COLUMNS, 2, (gchar**)titles);
	tree = GTK_CTREE(w);
	list = GTK_CLIST(w);
	list->button_actions[1] = list->button_actions[0];
	gtk_signal_connect (GTK_OBJECT(w), "tree-select-row", tree_select_row_cb, NULL);
	gtk_signal_connect (GTK_OBJECT(w), "tree-unselect-row", tree_select_row_cb, NULL);
	gtk_signal_connect (GTK_OBJECT(w), "toggle_focus_row", tree_toggle_focus_row_cb, NULL);

	/* fonts */
	normalfont = pan_load_font (thread_normal_font);
	newrepliesfont = pan_load_font (thread_new_replies_font);
	pan_widget_set_font (Pan.article_ctree, thread_new_replies_font);

	/* wrap it in a scrolled window */
	s_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (
		GTK_SCROLLED_WINDOW(s_window),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER(s_window), Pan.article_ctree);

	/* ui settings */
	gtk_ctree_set_line_style (tree, GTK_CTREE_LINES_DOTTED);
	gtk_clist_set_selection_mode (list, GTK_SELECTION_EXTENDED);
	gtk_clist_set_column_justification (list, 3, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_width (list, 0, 16);
	gtk_clist_set_column_width (list, 1, 16);
	gtk_clist_set_column_width (list, 2, 220);
	gtk_clist_set_column_width (list, 3, 30);
	gtk_clist_set_column_width (list, 4, 120);
	gtk_clist_set_column_width (list, 5, 40);

	/* create the right click popup menu */
	header_pane_popup_menu = gnome_popup_menu_new (header_pane_popup_uiinfo);

	/* connect signals */
	gtk_signal_connect (GTK_OBJECT(tree), "button_press_event",
	                    GTK_SIGNAL_FUNC(articlelist_button_press), NULL);
	gtk_signal_connect (GTK_OBJECT(tree), "key_press_event",
	                    GTK_SIGNAL_FUNC(articlelist_key_press), NULL);
	gtk_signal_connect (GTK_OBJECT(tree), "click_column",
	                    GTK_SIGNAL_FUNC(articlelist_click_column), NULL);
	gtk_signal_connect (GTK_OBJECT(tree), "destroy",
	                    GTK_SIGNAL_FUNC(article_ctree_destroy_cb), NULL);

	pan_callback_add (group_get_articles_removed_callback(),
			  group_articles_removed_cb,
			  NULL);

	pan_callback_add (group_get_articles_added_callback(),
			  group_articles_added_cb,
			  NULL);

	/* due to a bug in GtkCList (as of v1.2) we can't moveto an
	 * undrawable window -- so we remember to moveto upon mapping */
	gtk_signal_connect_after (GTK_OBJECT(tree), "map",
	                          GTK_SIGNAL_FUNC(map_articlelist_cb), NULL);

	gtk_box_pack_start (GTK_BOX(vbox), s_window, TRUE, TRUE, 0);

	return vbox;
}
void
articlelist_reset_style_nolock (void)
{
	pan_lock ();
	normalfont = pan_load_font (thread_normal_font);
	newrepliesfont = pan_load_font (thread_new_replies_font);
	pan_widget_set_font (Pan.article_ctree, thread_new_replies_font);
	pan_unlock ();

	rebuild_style = TRUE;
	articlelist_refresh_nolock ();
}


/**
***
**/

void
articlelist_expand_all_threads (void)
{
	GtkCTree * tree;
	GtkCTreeNode * node;
	GtkCTreeNode * sel;
	debug_enter ("articlelist_expand_all_threads");

 	tree = GTK_CTREE(Pan.article_ctree);
	pan_lock ();
	sel = articlelist_get_selected_node ();
	node = gtk_ctree_node_nth (tree, 0);
	if (node != NULL) {
		gtk_ctree_expand_recursive (tree, node);
		if (sel != NULL)
			articlelist_set_selected_nodes_nolock (&sel, 1);
	}
	pan_unlock ();
	debug_exit ("articlelist_expand_all_threads");
}

void
articlelist_collapse_all_threads (void)
{
	GtkCTree * tree;
	GtkCTreeNode * root;
	debug_enter ("articlelist_collapse_all_threads");

	tree = GTK_CTREE(Pan.article_ctree);

	/* get the root of the tree */
	pan_lock ();
	root = gtk_ctree_node_nth (tree, 0);
	pan_unlock ();

	/* if the tree's not empty, collapse the threads */
	if (root != NULL)
	{
		GtkCTreeNode * sel; 

		/* find the parent node that will be selected when we're done */
		sel = articlelist_get_selected_node ();
		while (sel!=NULL && GTK_CTREE_ROW(sel)->parent!=root)
			sel = GTK_CTREE_ROW(sel)->parent;

		/* collapse the threads */
		pan_lock ();
		gtk_ctree_collapse_recursive (tree, root);
		gtk_ctree_expand (tree, root);
		pan_unlock ();

		/* select the topmost parent of the old selection */
		if (sel != NULL) {
			pan_lock ();
			articlelist_set_selected_nodes_nolock (&sel, 1);
			pan_unlock ();
		}
	}

	debug_exit ("articlelist_collapse_all_threads");
}


/**
***
***  ARTICLE THREADING
***
**/


void
articlelist_set_threaded (gboolean threaded_on)
{
	if (threaded_on != is_threaded)
	{
		is_threaded = threaded_on;
		refresh_nolock_impl (0);
	}
}

/**
***
***  POPUP MENU 
***
**/

static void
articlelist_menu_popup_nolock (GdkEventButton* bevent)
{
	const Article * article = articlelist_get_selected_article_nolock ();
	const gboolean have_article = article!=NULL;
	gint i;

	i = -1;
	gtk_widget_set_sensitive (articlelist_menu[++i].widget, have_article); /* save */
	gtk_widget_set_sensitive (articlelist_menu[++i].widget, have_article); /* save */
	gtk_widget_set_sensitive (articlelist_menu[++i].widget, have_article); /* save */
	++i;
	gtk_widget_set_sensitive (articlelist_menu[++i].widget, TRUE); /* download flagged */
	gtk_widget_set_sensitive (articlelist_menu[++i].widget, have_article); /* flag */
	gtk_widget_set_sensitive (articlelist_menu[++i].widget, have_article); /* flag */
	gtk_widget_set_sensitive (articlelist_menu[++i].widget, have_article); /* unflag */
	gtk_widget_set_sensitive (articlelist_menu[++i].widget, have_article); /* unflag */
	++i;
	gtk_widget_set_sensitive (articlelist_menu[++i].widget, have_article); /* delete */
	gtk_widget_set_sensitive (articlelist_menu[++i].widget, have_article); /* delete */

	gtk_menu_popup (GTK_MENU (header_pane_popup_menu),
			NULL, NULL, NULL, NULL,
			bevent->button, bevent->time);
}

#define GNOMEUIINFO_ITEM_ACCEL(label, tooltip, callback, accel, modifier) \
	{ GNOME_APP_UI_ITEM, label, tooltip, (gpointer)callback, NULL, NULL, \
		GNOME_APP_PIXMAP_NONE, NULL, accel, (GdkModifierType)modifier,  NULL }
#define GNOMEUIINFO_ITEM_STOCK_ACCEL(label, tooltip, callback, stock_id, accel, modifier) \
	{ GNOME_APP_UI_ITEM, label, tooltip, (gpointer)callback, NULL, NULL, \
		GNOME_APP_PIXMAP_STOCK, stock_id, accel, (GdkModifierType)modifier,  NULL }
#define GNOMEUIINFO_ITEM_XPM_ACCEL(label, tooltip, callback, xpm_data, accel, modifier) \
	{ GNOME_APP_UI_ITEM, label, tooltip, (gpointer)callback, NULL, NULL, \
		GNOME_APP_PIXMAP_DATA, xpm_data, accel, (GdkModifierType)modifier,  NULL }

GnomeUIInfo
articlelist_menu [] =
{
	GNOMEUIINFO_ITEM_ACCEL (N_("_Mark as Read"),
	                        N_("Mark Selected Articles as Read"),
	                        articlelist_selected_mark_read_nolock,
	                        'M', 0),
	GNOMEUIINFO_ITEM_ACCEL (N_("Mark _Threads as Read"),
	                        N_("Mark Threads as Read"),
	                        articlelist_selected_mark_read_thread_nolock,
	                        'M', GDK_SHIFT_MASK),
	GNOMEUIINFO_ITEM_NONE (N_("Mark as _Unread"),
	                       N_("Mark as Unread"),
	                       articlelist_selected_mark_unread_nolock),
	GNOMEUIINFO_ITEM_NONE (N_("Mark Threads as U_nread"),
	                       N_("Mark Threads as Unread"),
	                       articlelist_selected_mark_unread_thread_nolock),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("D_ownload Flagged"),
	                              N_("Download Flagged Articles"),
	                              articlelist_download_flagged_bodies,
	                              GNOME_STOCK_MENU_MAIL_RCV,
	                              'J', GDK_SHIFT_MASK|GDK_CONTROL_MASK),
	GNOMEUIINFO_ITEM_XPM_ACCEL   (N_("_Flag"),
	                              N_("Flag Selected Articles"),
	                              articlelist_selected_flag_for_dl_nolock,
	                              flagged_xpm, 'J', 0),
	GNOMEUIINFO_ITEM_XPM_ACCEL   (N_("Flag T_hreads"),
	                              N_("Flag Selected Threads"),
	                              articlelist_selected_flag_for_dl_nolock,
	                              flagged_xpm, 'J', GDK_CONTROL_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Unflag"),
	                              N_("Unf_lag Selected Articles"),
	                              articlelist_selected_unflag_for_dl_nolock,
	                              GNOME_STOCK_MENU_BLANK, 'K', 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Unflag Thr_eads"),
	                              N_("Unflag Selected Threads"),
	                              articlelist_selected_unflag_for_dl_nolock,
	                              GNOME_STOCK_MENU_BLANK, 'K', GDK_CONTROL_MASK),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_NONE  (N_("Download"),
	                        N_("Download Selected Articles"),
	                        articlelist_selected_download_nolock),
	GNOMEUIINFO_ITEM_STOCK (N_("Download Threads"),
	                        N_("Download Selected Threads"),
	                        articlelist_selected_thread_download_nolock,
	                        GNOME_STOCK_MENU_MAIL_RCV),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_XPM_ACCEL   (N_("_Watch Threads"),
	                              N_("Watch Selected Threads"),
	                              articlelist_selected_thread_watch_nolock,
	                              watched_xpm, 'W', 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Ignore Threads"),
	                              N_("Ignore Selected Threads"),
	                              articlelist_selected_thread_ignore_nolock,
	                              GNOME_STOCK_MENU_TRASH_FULL, 'I', 0),
	GNOMEUIINFO_ITEM_XPM_ACCEL   (N_("Add to Bo_zo or Spam Filter"),
	                              N_("Add to Bozo or Spam Filter"),
	                              articlelist_spamize_nolock, 
	                              bozo_xpm, 'Z', GDK_CONTROL_MASK),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_DATA (N_("Cop_y to Folder"),
	                       N_("Copy to Folder"),
	                       articlelist_selected_copy_to_folder_nolock, NULL,
	                       copy_message_xpm),
	GNOMEUIINFO_ITEM_DATA (N_("Co_py Threads to Folder"),
	                       N_("Copy Threads to Folder"),
	                       articlelist_selected_copy_to_folder_deep_nolock, NULL,
	                       copy_message_xpm),
	GNOMEUIINFO_ITEM_DATA (N_("Mo_ve to Folder"),
	                       N_("Move to Folder"),
	                       articlelist_selected_move_to_folder_nolock, NULL,
	                       move_message_xpm),
	GNOMEUIINFO_ITEM_DATA (N_("Move Thread to Folder"),
	                       N_("Move Thread to Folder"),
	                       articlelist_selected_move_to_folder_deep_nolock, NULL,
	                       move_message_xpm),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_NONE (N_("_Cancel..."),
	                       N_("Cancel Selected Article which you Posted"),
	                       articlelist_selected_cancel),
	GNOMEUIINFO_ITEM_NONE (N_("_Supersede..."),
	                       N_("Supersede Selected Article which you Posted"),
	                       articlelist_selected_supersede),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Delete"),
	                              N_("Delete"),
	                              article_action_delete_selected_articles,
	                              GNOME_STOCK_MENU_TRASH, 'D', 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Delete Th_reads"),
	                              N_("Delete Threads"),
	                              article_action_delete_selected_threads,
	                              GNOME_STOCK_MENU_TRASH, 'D', GDK_SHIFT_MASK),
	GNOMEUIINFO_END
};

static GnomeUIInfo
header_pane_popup_uiinfo [] =
{
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Save _Articles"),
	                              N_("Save the selected articles and their attachments."),
	                              article_action_selected_save,
	                              GNOME_STOCK_MENU_SAVE, 'S', 0),
        GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Save Articles As..."),
	                              N_("Dialog to set how you want to save the selected articles"),
	                              article_action_selected_save_as,
	                              GNOME_STOCK_MENU_SAVE, 'S', GDK_CONTROL_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Save Articles (A_ttachments only)"),
	                              N_("Save the selected articles' attachments."),
	                              article_action_selected_save_attachments,
	                              GNOME_STOCK_MENU_SAVE, 'S', GDK_SHIFT_MASK),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("D_ownload Flagged"),
	                              N_("Download Flagged Articles"),
	                              articlelist_download_flagged_bodies,
	                              GNOME_STOCK_MENU_MAIL_RCV,
	                              'J', GDK_SHIFT_MASK|GDK_CONTROL_MASK),
	GNOMEUIINFO_ITEM_XPM_ACCEL   (N_("_Flag"),
	                              N_("Flag Selected Articles"),
	                              articlelist_selected_flag_for_dl_nolock,
	                              flagged_xpm, 'J', 0),
	GNOMEUIINFO_ITEM_XPM_ACCEL   (N_("Flag T_hreads"),
	                              N_("Flag Selected Threads"),
	                              articlelist_selected_flag_for_dl_nolock,
	                              flagged_xpm, 'J', GDK_CONTROL_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Unflag"),
	                              N_("Unf_lag Selected Articles"),
	                              articlelist_selected_unflag_for_dl_nolock,
	                              GNOME_STOCK_MENU_BLANK, 'K', 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Unflag Thr_eads"),
	                              N_("Unflag Selected Threads"),
	                              articlelist_selected_unflag_for_dl_nolock,
	                              GNOME_STOCK_MENU_BLANK, 'K', GDK_CONTROL_MASK),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Delete"),
	                              N_("Delete"),
	                              article_action_delete_selected_articles,
	                              GNOME_STOCK_MENU_TRASH, 'D', 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Delete Th_reads"),
	                              N_("Delete Threads"),
	                              article_action_delete_selected_threads,
	                              GNOME_STOCK_MENU_TRASH, 'D', GDK_SHIFT_MASK),
};


/***
****  Events
***/

PanCallback*
articlelist_get_group_changed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_group_changed (Group * g)
{
	pan_callback_call (articlelist_get_group_changed_callback(),
	                   Pan.article_ctree,
	                   g);
}

/***
****
****  REFRESH
****
***/

static gint
refresh_mainthread_end (gpointer p)
{
	ArgSet * argset;
	Group * group;
	RefreshActions actions;
	StatusItem * status;
	GPtrArray * articles;
	GPtrArray * tmp;
	debug_enter ("refresh_mainthread_end");

	/* pump out the arguments */
	argset = (ArgSet*) p;
	actions = GPOINTER_TO_INT (argset_get (argset, 0));
	group = (Group*) argset_get (argset, 1);
	articles = (GPtrArray*) argset_get (argset, 2);
	status = (StatusItem*) argset_get (argset, 3);

	/* update the UI */
	pan_lock_unconditional ();
	tmp = articlelist_get_selected_articles_nolock ();
	articlelist_repopulate_nolock (group, (Article**)articles->pdata, articles->len, tmp);
	g_ptr_array_free (tmp, TRUE);
	pan_g_ptr_array_assign (my_articles, articles->pdata, articles->len);
	pan_unlock_unconditional ();

	/* cleanup - unref the safe-keeping refs */
	group_unref_articles (group, NULL);

	/* cleanup - done with status-item */
	status_item_set_active (status, FALSE);
	pan_object_unref (PAN_OBJECT(status));
	
	/* cleanup - free the tmp stuff */
	g_ptr_array_free (articles, TRUE);
	argset_free (argset);

	debug_exit ("refresh_mainthread_end");
	return 0;
}

static void
refresh_worker (void * p)
{
	ArgSet * argset;
	Group * group;
	RefreshActions actions;
	StatusItem * status;
	GPtrArray * articles;
	debug_enter ("refresh_worker");

	/* pump out the arguments */
	argset = (ArgSet*) p;
	actions = GPOINTER_TO_INT (argset_get (argset, 0));
	group = (Group*) argset_get (argset, 1);
	articles = (GPtrArray*) argset_get (argset, 2);
	status = (StatusItem*) argset_get (argset, 3);

	/* refresh the articles */
	if (actions & REFRESH_RELOAD) {
		GPtrArray * tmp;
		status_item_emit_status (status, _("Reloading Group"));
		tmp = group_get_article_array (group);
		pan_g_ptr_array_assign (articles, tmp->pdata, tmp->len);
		g_ptr_array_free (tmp, TRUE);
	}
	if (actions & REFRESH_SORT) {
		status_item_emit_status (status, _("Resorting Articles"));
		sort_articles ((Article**)articles->pdata, articles->len, abs(sort_type), sort_type>0);
	}
	if (actions & REFRESH_THREAD) {
		status_item_emit_status (status, _("Rethreading Articles"));
		group_thread_if_needed (group, NULL);
	}
	if (actions & REFRESH_FILTER) {
		status_item_emit_status (status, _("Refiltering Articles"));
		apply_filter_tests (my_filter, articles);
	}
	status_item_emit_status (status, _("Rebuilding Header Pane..."));

	/* hard work done; back to the main thread */
	pan_lock ();
	run_in_main_thread_nolock (refresh_mainthread_end, argset);
	pan_unlock ();

	debug_exit ("refresh_worker");
}

static gint
refresh_mainthread_begin (gpointer p)
{
	Group * group;
	ArgSet * argset;
	debug_enter ("refresh_mainthread_begin");

	/* pump out the arguments */
	argset = (ArgSet*) p;

	group = my_group;
	if (group != NULL)
	{
		gchar * pch;
		GPtrArray * articles;
		StatusItem * status;

		argset_add (argset, group);

		/* ref the articles for safe-keeping */
		group_ref_articles (group, NULL);
		articles = pan_g_ptr_array_dup (my_articles);
		argset_add (argset, articles);

		/* create the status-item */
		pch = g_strdup_printf (_("Refreshing Group \"%s\""), group_get_readable_name(group));
		status = status_item_new_with_description (pch);
		g_free (pch);
		status_item_set_active (status, TRUE);
		argset_add (argset, status);

		/* pass the hard work to another thread */
		run_in_worker_thread (refresh_worker, argset);
	}
	else
	{
		argset_free (argset);
	}

	debug_exit ("refresh_mainthread_begin");
	return 0;
}

static void
refresh_nolock_impl (RefreshActions actions)
{
	run_in_main_thread_nolock (refresh_mainthread_begin, argset_new1(GINT_TO_POINTER(actions)));
}

void
articlelist_refresh_nolock (void)
{
	refresh_nolock_impl (~0);
}

/***
****
****  SET THE GROUP
****
***/

static void
clear_group_nolock (void)
{
	/* clear the UI */
	gtk_clist_clear (GTK_CLIST(Pan.article_ctree));
	clear_hash_table ();

	/* clear the model */
	if (my_group != NULL)
	{
		group_unref_articles (my_group, NULL);
		g_ptr_array_set_size (my_articles, 0);
		my_group = NULL;
		run_in_main_thread_nolock ((GSourceFunc)fire_group_changed, my_group);
	}
}

static void
headers_fetched (gpointer callobj, gpointer callarg, gpointer client)
{
	Group * group = GROUP(client);

	if (group->article_qty > 0)
	{
		pan_lock ();
		articlelist_set_group_nolock (group);
		pan_unlock ();
	}
}

static gint
set_group_mainthread_end (gpointer p)
{
	ArgSet * argset;
	Group * group;
	Group * old_group;
	StatusItem * status;
	GPtrArray * articles;
	GPtrArray * tmp;
	gint sort;
	debug_enter ("set_group_mainthread_end");

	/* pump out the arguments */
	argset = (ArgSet*) p;
	group = (Group*) argset_get (argset, 0);
	status = (StatusItem*) argset_get (argset, 1);
	articles = (GPtrArray*) argset_get (argset, 2);
	sort = GPOINTER_TO_INT (argset_get (argset, 3));

	/* update the UI */
	pan_lock_unconditional ();
	articlelist_set_sort_bits_nolock (sort);
	tmp = articlelist_get_selected_articles_nolock ();
	articlelist_repopulate_nolock (group, (Article**)articles->pdata, articles->len, tmp);
	g_ptr_array_free (tmp, TRUE);
	pan_unlock_unconditional ();

	/* update the model */
	if (group != NULL)
		group_ref_articles (group, NULL);
	old_group = my_group;
	tmp = pan_g_ptr_array_dup (my_articles);
	my_group = group;
	pan_g_ptr_array_assign (my_articles, articles->pdata, articles->len);
	if (old_group != NULL)
		group_unref_articles (old_group, NULL);
	fire_group_changed (my_group);

	/* no longer using the status */
	status_item_set_active (status, FALSE);
	pan_object_unref (PAN_OBJECT(status));

	/* unref the safeguard refs from set_group_worker */
	group_unref_articles (group, NULL);

	/* cleanup memory */
	g_ptr_array_free (tmp, TRUE);
	argset_free (argset);

	debug_exit ("set_group_mainthread_end");
	return 0;
}

static void
set_group_worker (void * p)
{
	ArgSet * argset;
	Group * group;
	StatusItem * status;
	GPtrArray * articles;
	gint sort;
	Filter * filter;
	debug_enter ("set_group_worker");

	/* pump out the arguments */
	argset = (ArgSet*) p;
	group = (Group*) argset_get (argset, 0);
	status = (StatusItem*) argset_get (argset, 1);

	/* get the articles */
	group_ref_articles (group, status);
	articles = group_get_article_array (group);
	argset_add (argset, articles);

	/* get sorting information */
	sort = group->sort_style;
	if (!sort || abs(sort)>ARTICLE_SORT_TYPES_QTY)
		sort = -ARTICLE_SORT_DATE;
	argset_add (argset, GINT_TO_POINTER(sort));

	/* massage them into shape */
	status_item_emit_status (status, _("Sorting Articles"));
	sort_articles ((Article**)articles->pdata, articles->len, abs(sort), sort>0);
	status_item_emit_status (status, _("Threading Articles"));
	group_thread_if_needed (group, NULL);
	status_item_emit_status (status, _("Filtering Articles"));
	filter = filter_create_from_bits (group->filter_name, group->filter_bits);
	articlelist_set_filter_bits_nolock (filter, group->filter_show);
	apply_filter_tests (filter, articles);
	pan_object_unref (PAN_OBJECT(filter));
	status_item_emit_status (status, _("Updating Header Pane..."));

	/* hard work over, pass back to main thread */
	pan_lock ();
	run_in_main_thread_nolock (set_group_mainthread_end, argset);
	pan_unlock ();

	debug_exit ("set_group_worker");
}

static gint
set_group_mainthread_begin (gpointer p)
{
	gchar * pch;
	ArgSet * argset;
	Group * group;
	StatusItem * status;
	debug_enter ("set_group_mainthread_begin");

	/* pump out the arguments */
	argset = (ArgSet*) p;
	group = (Group*) argset_get (argset, 0);

	/* create the status-item */
	pch = g_strdup_printf (_("Loading Group \"%s\""), group_get_readable_name(group));
	status = status_item_new_with_description (pch);
	g_free (pch);
	status_item_set_active (status, TRUE);
	argset_add (argset, status);

	/* pass the hard work to another thread */
	run_in_worker_thread (set_group_worker, argset);
	debug_exit ("set_group_mainthread_begin");
	return 0;
}

void
articlelist_set_group_nolock (Group * group)
{
	if (group == NULL)
	{
		clear_group_nolock ();
	}
	else if (group->article_qty == 0 && !group_is_folder(group))
	{
		Task * task;

		clear_group_nolock ();

		task = TASK (task_headers_new (group, HEADERS_NEW));
		pan_callback_add (task->task_ran_callback, headers_fetched, group);
		queue_add (task);
	}
	else
	{
		run_in_main_thread_nolock (set_group_mainthread_begin, argset_new1(group));

                if (fetch_new_on_group_load)
			queue_add (TASK(task_headers_new (group, HEADERS_NEW)));
	}
}
