/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "acache.h"
#include "article.h"
#include "articlelist.h"
#include "debug.h"
#include "gnksa.h"
#include "message-window.h"
#include "nntp.h"
#include "server.h"
#include "group.h"
#include "prefs.h"
#include "queue.h"
#include "task-func-ptr.h"
#include "util.h"
#include "util-mime.h"

/***
****
****  LIFE CYCLE
****
***/

void
article_destructor (PanObject* o)
{
	Article * a = ARTICLE(o);
        g_return_if_fail (a != NULL);

	/* clean up article parts */
	debug1 (DEBUG_PAN_OBJECT, "article_destructor: %p", a);
	g_slist_free (a->threads);
	if (a->headers != NULL)
		g_hash_table_destroy (a->headers);

	/* clean up parent */
	pan_object_destructor (o);
}

void
article_constructor (Article * article, Group * group)
{
	g_return_if_fail (article!=NULL);
	g_return_if_fail (group!=NULL);

	/* init the superclass */
	pan_object_constructor (PAN_OBJECT(article), article_destructor);

	/* init this class */
        debug1 (DEBUG_PAN_OBJECT, "article_constructor: %p", article);
	article->group = group;
	article->self_passes_filter = FALSE;
	article->tree_passes_filter = FALSE;
	article->part = (gint8)0;
	article->parts = (gint8)0;
	article->crosspost_qty = (gint8)0;
	article->linecount = (guint16)0;
	article->state = (guint16)0;
	article->unread_children = (guint16)0;
	article->date = (time_t)0;
	article->number = (gulong)0;
	article->subject = NULL;
	article->author_real = NULL;
	article->author_addr = NULL;
	article->message_id = NULL;
	article->references = NULL;
	article->parent = NULL;
	article->threads = NULL;
	article->headers = NULL;
}

Article*
article_new (Group * group)
{
	Article* article = g_new (Article, 1);
	article_constructor (article, group);
	return article;
}


static void
article_dup_ghfunc (gpointer key, gpointer val, gpointer user_data)
{
	Article * article = ARTICLE(user_data);
	article_set_header (article, key, val, DO_CHUNK);
}
Article*
article_dup (Group * group, const Article * article)
{
	Article * retval;

	g_return_val_if_fail (group!=NULL, NULL);
	g_return_val_if_fail (article!=NULL, NULL);

#define chunk_if_exists(A) \
	is_nonempty_string (A) ? group_chunk_string (group, A) : NULL

	retval                      = article_new (group);
	retval->self_passes_filter  = FALSE;
	retval->tree_passes_filter  = FALSE;
	retval->unread_children     = 0;
	retval->group               = group;
	retval->parts               = article->parts;
	retval->part                = article->part;
	retval->linecount           = article->linecount;
	retval->crosspost_qty       = article->crosspost_qty;
	retval->state               = article->state;
	retval->date                = article->date;
	retval->number              = article->number;
	retval->subject             = chunk_if_exists (article->subject);
	retval->author_real         = chunk_if_exists (article->author_real);
	retval->author_addr         = chunk_if_exists (article->author_addr);
	retval->message_id          = chunk_if_exists (article->message_id);
	retval->references          = chunk_if_exists (article->references);
	retval->parent              = NULL;
	retval->threads             = NULL;
	retval->headers             = NULL;

	if (article->headers != NULL)
		g_hash_table_foreach (article->headers, article_dup_ghfunc, retval);

	return retval;
}

/***
****
***/

static void
article_set_dirty (Article * article)
{
	g_return_if_fail (article != NULL);
	g_return_if_fail (article->group != NULL);

	article->group->articles_dirty = TRUE;
	articlelist_update_node (article);
}



/**
 * Remove an article from its parent/child threadings.  The children and
 * parent are linked together instead.  This is useful for not breaking
 * the threading of articles surrounding an article which is about to be
 * destroyed.
 */
void
article_isolate (Article* article)
{
	GSList *l = NULL;
	Article *parent = article->parent;

	if (parent)
		parent->threads = g_slist_remove (parent->threads, article);

	for (l=article->threads; l!=NULL; l=l->next) {
		Article *child = ARTICLE(l->data);
		child->parent = parent;
		if (parent)
			parent->threads = g_slist_append (parent->threads, child);
	}

	g_slist_free (article->threads);
	article->threads = NULL;
	article->parent = NULL;
}


gboolean
article_is_new (const Article * a)
{
	g_return_val_if_fail (a!=NULL, FALSE);
	g_return_val_if_fail (a->group!=NULL, FALSE);
	return a->number > a->group->article_high_old;
}

gboolean
article_thread_has_new (const Article * top)
{
	guint i;
	GPtrArray * thread = g_ptr_array_new ();
	gboolean has_new = FALSE;

	g_return_val_if_fail (top!=NULL, FALSE);

	article_get_subthread (ARTICLE(top), thread);
	for (i=0; i!=thread->len && !has_new; ++i)
		if (article_is_new(ARTICLE(g_ptr_array_index(thread,i))))
			has_new = TRUE;
	g_ptr_array_free (thread, TRUE);

	return has_new;
}


/***
****
****  READ / UNREAD
****
***/

gboolean
article_is_read (const Article * a)
{
	g_return_val_if_fail (a!=NULL, FALSE);
	g_return_val_if_fail (a->group!=NULL, FALSE);

	return newsrc_is_article_read (group_get_newsrc(a->group), a->number);
}

/**
 * Mark the article (and possibly multipart children) unread.
 * @param article the article to mark as unread.
 * @return the number of articles marked unread in this call to this function.
 */
static int
article_set_unread_impl (Article * article)
{
	int unread_qty = 0;
	const gboolean was_read = newsrc_mark_article (group_get_newsrc(article->group),
	                                               article->number,
	                                               FALSE);
	if (was_read) {
		++unread_qty;
		articlelist_article_read_changed (article);
	}

	/* FIXME: I don't think this special case is such a great idea.
	   Special case: if it's a multipart, mark 1..n as read too */
	if (article->part>0 && article->parts>1 && article->threads!=NULL) {
		GSList * l;
		for (l=article->threads; l!=NULL; l=l->next) {
			Article * child = ARTICLE(l->data);
			if (child->part > 0)
				unread_qty += article_set_unread_impl (child);
		}
	}

	return unread_qty;
}

/**
 * Mark the article (and possibly multipart children) read.
 * @param article the article to mark as read.
 * @return the number of articles marked read in this call to this function.
 */
static int
article_set_read_impl (Article * article)
{
	int read_qty = 0;
	const gboolean was_read = newsrc_mark_article (group_get_newsrc(article->group),
	                                               article->number,
	                                               TRUE);
	if (!was_read) {
		++read_qty;
		articlelist_article_read_changed (article);
	}

	/* FIXME: I don't think this special case is such a great idea.
	   Special case: if it's a multipart, mark 1..n as read too */
	if (article->part>0 && article->parts>1 && article->threads!=NULL) {
		GSList * l;
		for (l=article->threads; l!=NULL; l=l->next) {
			Article * child = ARTICLE(l->data);
			if (child->part > 0)
				read_qty += article_set_read_impl (child);
		}
	}

	return read_qty;
}

void
articles_set_read (Article ** articles, int article_qty, gboolean read)
{
	int i;
	int read_inc;

	g_return_if_fail (articles != NULL);

	for (read_inc=i=0; i<article_qty; ++i)
	{
		Article * article = articles[i];

		g_return_if_fail (article != NULL);
		g_return_if_fail (article->group != NULL);
		g_assert (articles[0]->group == article->group);

		if (read)
			read_inc += article_set_read_impl (article);
		else 
			read_inc -= article_set_unread_impl (article);
	}

	if (read_inc)
		group_inc_article_read_qty (articles[0]->group, read_inc);
}

void
article_set_read (Article * article, gboolean read)
{
	articles_set_read (&article, 1, read);
}
        
        


/**
*** PUBLIC MUTATORS
**/

void
article_add_flag (Article  * article,
                  gushort    flag)
{
	g_return_if_fail (article != NULL);
	article->state |= flag;
	article_set_dirty (article);
}


void
article_remove_flag (Article         * article,
                     gushort           flag)
{
	g_return_if_fail (article != NULL);
	article->state &= ~flag;
	article_set_dirty (article);
}

void
article_set_flag (Article * a, gushort flag, gboolean b)
{
	g_return_if_fail (a != NULL);
	if (b)
		a->state |= flag;
	else
		a->state &= ~flag;
	article_set_dirty (a);
}


/***
****
****  THREADS
****
***/

gchar*
article_get_thread_message_id (const Article* article)
{
	char * pch;
	const gchar * refs = article_get_header (article, HEADER_REFERENCES);
	const gchar * msg_id = article_get_header (article, HEADER_MESSAGE_ID);

	g_return_val_if_fail (article!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string(msg_id), NULL);

	if (is_nonempty_string(refs))
	{
		pch = get_next_token_str (refs, ' ', NULL);
	}
	else /* top of the thread */
	{
		pch = g_strdup (msg_id);
	}

	return pch;
}

static void
article_get_entire_thread_impl (Article* top, GPtrArray* setme)
{
	GSList *l;
	g_ptr_array_add (setme, top);
	for (l=top->threads; l!=NULL; l=l->next)
		article_get_entire_thread_impl (ARTICLE(l->data), setme);
}

static void
article_get_thread_impl (Article * article,
                         GPtrArray * setme,
                         gboolean children_only)
{
	Article* top = article;

	if (top != NULL)
	{
		if (!children_only)
			while (top->parent != NULL)
				top = top->parent;

		pan_g_ptr_array_reserve (setme, 128);
		article_get_entire_thread_impl (top, setme);
	}
}

void
article_get_subthread (Article* article, GPtrArray* setme)
{
	article_get_thread_impl (article, setme, TRUE);
}

void
article_get_entire_thread (Article* article, GPtrArray* setme)
{
	article_get_thread_impl (article, setme, FALSE);
}


/***
****
****  OTHER HEADERS
****
***/

void
article_remove_header (Article * a, const gchar * header_name)
{
	g_return_if_fail (a!=NULL);
	g_return_if_fail (header_name!=NULL);

	if (a->headers!=NULL)
	{
		g_hash_table_remove (a->headers, header_name);

		if (!g_hash_table_size(a->headers))
		{
			g_hash_table_destroy (a->headers);
			a->headers = NULL;
		}
	}
}

const gchar*
article_get_header (const Article * a, const gchar * key)
{
	const gchar * retval = NULL;

	g_return_val_if_fail (a!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string(key), NULL);

	if (key == HEADER_MESSAGE_ID) retval = a->message_id;
	else if (key == HEADER_SUBJECT) retval = a->subject;
	else if (key == HEADER_REFERENCES) retval = a->references;
	else if (key == HEADER_FROM) pan_warn_if_reached ();
	else if (!strcmp(key, HEADER_MESSAGE_ID)) retval = a->message_id;
	else if (!strcmp (key, HEADER_SUBJECT)) retval = a->subject;
	else if (!strcmp (key, HEADER_REFERENCES)) retval = a->references;
	else if (!strcmp (key, HEADER_FROM)) pan_warn_if_reached ();
	else if (a->headers!=NULL)
		retval = (const gchar*) g_hash_table_lookup (a->headers, key);

	return retval;
}

gboolean
article_header_is_internal (const gchar * key)
{
	return is_nonempty_string(key) && !g_strncasecmp(key,"Pan-",4);
}

gboolean
article_header_is_extra (const gchar * key)
{
	/* sanity check */
	if (!is_nonempty_string(key)) return FALSE;

	/* pan internals aren't user-specified headers */
	if (article_header_is_internal(key)) return FALSE;

	/* other headers that are handled explicitly elsewhere */
	if (!pan_strcmp(key,HEADER_FOLLOWUP_TO)) return FALSE;
	if (!pan_strcmp(key,HEADER_NEWSGROUPS)) return FALSE;
	if (!pan_strcmp(key,HEADER_ORGANIZATION)) return FALSE;
	if (!pan_strcmp(key,HEADER_REPLY_TO)) return FALSE;

	return TRUE;
}

static void
ensure_extra_headers_exists (Article * a)
{
	if (a->headers == NULL)
		a->headers = g_hash_table_new (g_str_hash, g_str_equal);
}

void
article_set_header (Article         * a,
                    const gchar     * key,
                    const gchar     * val,
                    HeaderAction      action)
{
	g_return_if_fail (is_nonempty_string(key));

	/* chunk if necessary */
	if (action & DO_CHUNK)
	{
		key = group_chunk_string (a->group, key);
		val = group_chunk_string (a->group, val);
	}

	/* if we're possibly erasing a header, remove the old value */
	if (!is_nonempty_string (val))
		article_remove_header (a, key);

	/* add the new header */
	if (is_nonempty_string (val))
	{
		if      (!pan_strcmp (key, HEADER_MESSAGE_ID)) a->message_id = val;
		else if (!pan_strcmp (key, HEADER_SUBJECT)) a->subject = val;
		else if (!pan_strcmp (key, HEADER_REFERENCES)) a->references = val;
		else if (!pan_strcmp (key, HEADER_FROM)) article_set_author_from_header (a, val);
		else {
			ensure_extra_headers_exists (a);
			g_hash_table_insert (a->headers,
			                     (gpointer)key,
			                     (gpointer)val);
			article_set_dirty (a);
		}
	}
}

static void
article_get_all_headers_ghfunc (gpointer key, gpointer value, gpointer user_data)
{
	GPtrArray * a = (GPtrArray*)user_data;
	g_ptr_array_add (a, key);
	g_ptr_array_add (a, value);
}

GPtrArray*
article_get_all_headers (const Article * a)
{
	GPtrArray * retval = g_ptr_array_new ();

	if (a->headers != NULL)
	{
		pan_g_ptr_array_reserve (retval, g_hash_table_size(a->headers)*2);
		g_hash_table_foreach (a->headers, article_get_all_headers_ghfunc, retval);
	}

	return retval;
}

gboolean
article_has_attachment (const Article *a)
{
	const gchar * filename = article_get_header (a, PAN_ATTACH_FILE);
	if (filename != NULL)
		return TRUE;
	else
		return FALSE;
}

/***
****
****  BODY
****
***/

gboolean
article_has_body (const Article * a)
{
	gboolean retval = FALSE;

	/* maybe the article's in a folder so the body
	 * is hidden as a header... */
	if (!retval) {
		const gchar * body = article_get_header (a, PAN_BODY);
		if (body != NULL)
			retval = TRUE;
	}

	/* try to get it from acache... */
	if (!retval && is_nonempty_string(a->message_id))
		retval = acache_has_message (a->message_id);

	return retval;
}

gchar*
article_get_author_str (const Article * a)
{
        gchar * retval;

	if (a==NULL)
	{
		retval = g_strdup ("");
	}
	else if (is_nonempty_string(a->author_addr))
	{
		if (is_nonempty_string(a->author_real))
		{
			retval = g_strdup_printf ("\"%s\" <%s>", a->author_real, a->author_addr);
		}
		else
		{
			retval = g_strdup (a->author_addr);
		}
	}
	else if (is_nonempty_string(a->author_real))
	{
		retval = g_strdup (a->author_real);
	}
	else
	{
		retval = g_strdup ("");
	}

	g_strstrip (retval);

	return retval;
}


gchar*
article_get_message (const Article * a)
{
	gchar * retval = NULL;

	if (retval==NULL && a!=NULL && acache_has_message (a->message_id))
	{
		retval = acache_get_message (a->message_id);
	}

	if (retval==NULL && a!=NULL)
	{
		GString * s = g_string_new (NULL);
		gchar * pch;

		pch = article_get_headers (a);
		g_string_append (s, pch);
		g_free (pch);

		g_string_append (s, "\n");

		pch = article_get_body (a);
		g_string_append (s, pch);
		g_free (pch);

		retval = s->str;
		g_string_free (s, FALSE);
	}

	return retval;
}

gchar*
article_get_headers (const Article * a)
{
	gchar * retval = NULL;

	/* do we have it in acache? */
	if (retval==NULL && a!=NULL && acache_has_message (a->message_id))
	{
		gchar * s = acache_get_message (a->message_id);
		gchar * pch = strstr (s, "\n\n");
		if (pch != NULL)
			retval = g_strndup (s, (gint)(pch-s));
		g_free (s);
	}

	/* can we build it from a? */
	if (retval==NULL && a!=NULL)
	{
		gchar * tmp;
		const gchar * c_tmp;
		GString * s = g_string_new (NULL);

		/* subject */
		c_tmp = a->subject;
		if (is_nonempty_string (c_tmp))
			g_string_sprintfa (s, _("Subject: %s\n"), c_tmp);

		/* author */
	       	tmp = article_get_author_str (a);
		if (is_nonempty_string (tmp))
			g_string_sprintfa (s, _("From: %s\n"), tmp);
		g_free (tmp);

		/* date */
		if (a->date != 0) {
			tmp = rfc822_date_generate (a->date);
			g_string_sprintfa (s, _("Date: %s\n"), tmp);
			g_free (tmp);
		}

		/* message-id */
		c_tmp = a->message_id;
		if (is_nonempty_string (c_tmp))
			g_string_sprintfa (s, _("Message-Id: %s\n"), c_tmp);

		/* references */
		c_tmp = a->references;
		if (is_nonempty_string (c_tmp))
			g_string_sprintfa (s, _("References: %s\n"), c_tmp);

		/* extra headers */
		if (1) {
			gint i;
			GPtrArray * z = article_get_all_headers (a);
			for (i=0; i<z->len; i+=2) {
				const char* key = (const char*) g_ptr_array_index (z, i);
				const char* val = (const char*) g_ptr_array_index (z, i+1);
				if (!article_header_is_internal (key))
					g_string_sprintfa (s, "%s: %s\n",  key, val);
			}
			g_ptr_array_free (z, TRUE);
		}

		retval = s->str;
		g_string_free (s, FALSE);
	}

	return retval;
}

gchar*
article_get_body (const Article * a)
{
	gchar * retval = NULL;

	/* see if we've got the body hidden in a header... */
	if (retval==NULL && a!=NULL)
	{
		const gchar * body = article_get_header (a, PAN_BODY);
		if (body != NULL)
			retval = g_strdup (body);
	}

	/* see if we've got it in acache... */
	if (retval==NULL && a!=NULL && acache_has_message(a->message_id))
	{
		gchar * s = acache_get_message (a->message_id);
		gboolean is_html = FALSE;
		GMimeMessage * m = pan_g_mime_parser_construct_message (s);
		retval = g_mime_message_get_body (m, TRUE, &is_html);
		g_mime_message_destroy (m);
		g_free (s);
	}

	return retval;
}

gchar*
article_get_attribution_string (const Article * a)
{
	gchar * pch = NULL;
	gchar * retval = NULL;
	const gchar * cpch = NULL;
	GString * gstr = NULL;

	pan_warn_if_fail (is_nonempty_string(attribution_line));

	/* get the unsubstituted string */
	gstr = g_string_new (attribution_line);

	/* substitutions: message-id */
	cpch = a==NULL ? "" : a->message_id;
	if (cpch != NULL)
		pan_g_string_replace (gstr, "%i", cpch);

	/* substitutions: date */
	pch = a==NULL ? g_strdup("") : rfc822_date_generate(a->date);
	pan_g_string_replace (gstr, "%d", pch);
	g_free (pch);

	/* substitutions: author */
	pch = a==NULL ? g_strdup("") : article_get_author_str (a);
	pan_g_string_replace (gstr, "%a", pch);
	g_free (pch);

	/* cleanup & return the result */
	retval = gstr->str;
	g_string_free (gstr, FALSE);
	return retval;
}

void
article_set_author_from_header (Article * a, const gchar * header_from)
{
	gchar * author_addr = NULL;
	gchar * author_real = NULL;

	g_return_if_fail (a!=NULL);
	g_return_if_fail (a->group!=NULL);
	g_return_if_fail (is_nonempty_string(header_from));

	/* note that we don't check gnksa's return value here,
	   because often posted messages on usenet are 
	   syntactically invalid */
	gnksa_do_check_from (header_from, &author_addr, &author_real);

	if (is_nonempty_string (author_addr))
		a->author_addr = group_chunk_string (a->group, author_addr);

	if (is_nonempty_string (author_real)) {
		gnksa_strip_realname (author_real);
		a->author_real = group_chunk_string (a->group, author_real);
	}

	article_set_dirty (a);

	g_free (author_addr);
	g_free (author_real);
}


/***
****
****   CANCEL / SUPERSEDE
****
***/

/**
 * Find a matching article without comparing by message-id.
 * This is because many news servers alter the message-id of
 * an article being posted, so we can't rely on searching by
 * the message-id that Pan generated.
 *
 * If a match is found, the refcount of Pan.sent is incremented
 * by one and must be decremented by the caller.
 */
static Article*
find_matching_sent_article (const Article * article)
{
	Group * sent = folder_get_by_name(PAN_SENT);
	Article * retval = NULL;
	GPtrArray * articles;
	guint i;

	g_return_val_if_fail (article!=NULL, NULL);
	g_return_val_if_fail (sent!=NULL, NULL);

	group_ref_articles (sent, NULL);
	articles = group_get_articles (sent);

	for (i=0; retval==NULL && i!=articles->len; ++i)
	{
		gchar * body_1;
		gchar * body_2;
		Article * compare = ARTICLE(g_ptr_array_index(articles,i));

		/* comparison #1: subject must match */
		if (pan_strcmp (compare->subject, article->subject))
			continue;

		/* comparison #2: author address must match */
		if (pan_strcmp (compare->author_addr, article->author_addr))
			continue;

		/* comparison #3: body address must match */
		body_1 = article_get_body (article);
		body_2 = article_get_body (compare);
		if (body_1!=NULL && body_2!=NULL)
		{
			g_strstrip (body_1);
			g_strstrip (body_2);
			if (!strcmp (body_1, body_2))
				retval = compare;
		}
		g_free (body_1);
		g_free (body_2);
	}

	if (retval == NULL)
		group_unref_articles (sent, NULL);

	return retval;
}

static gint
article_cancel_run (TaskFuncPtr* task, gpointer user_data)
{
        int status = nntp_cancel (STATUS_ITEM(task),
	                          ARTICLE(user_data),
	                          TASK(task)->sock);

	if (status == TASK_SUCCESS)
	{
		/* FIXME: add something to the article to denote that it's been cancelled */
	}

	return 0;
}
static gchar*
article_cancel_describe (const StatusItem * status)
{
	return g_strdup (_("Cancelling article"));
}
static void
article_cancel_dtor (gpointer data)
{
	group_unref_articles (folder_get_by_name(PAN_SENT), NULL);
}
void
article_cancel (Article * article)
{
	Server * server;
	Article * ours;
	const gchar * newsgroups;

	g_return_if_fail (article != NULL);
	server = article->group==NULL ? NULL : article->group->server;
	g_return_if_fail (server != NULL);
	newsgroups = article_get_header (article, HEADER_NEWSGROUPS);
	g_return_if_fail (is_nonempty_string (newsgroups));

	ours = find_matching_sent_article (article);
	if (ours == NULL)
	{
		pan_error_dialog (_("Couldn't find matching article in folder `pan.sent'!"));
	}
	else
	{
		queue_add (TASK(task_func_ptr_new (server,
			article_cancel_describe,
			article_cancel_run,
			ours,
			article_cancel_dtor)));
	}
}

void
article_supersede (const Article * article)
{
	const Article * ours;
	Group * sendlater = folder_get_by_name (PAN_SENDLATER);

	/* entry assertions */
	g_return_if_fail (article != NULL);
	g_return_if_fail (sendlater != NULL);

	/* find the original copy so that the headers will match */
	ours = find_matching_sent_article (article);
	if (ours == NULL)
	{
		pan_error_dialog (_("Couldn't find matching article in folder `pan.sent'!"));
	}
	else
	{
		gchar * message = article_get_message (article);
		gchar * message_id = gnksa_generate_message_id (ours->author_addr);
		Article * super = article_dup (sendlater, ours);
		article_set_header (super, HEADER_MESSAGE_ID, message_id, DO_CHUNK);
		article_set_header (super, HEADER_SUPERSEDES, article->message_id, DO_CHUNK);

		/* edit the copy */
		message_edit_window (NULL, super);

		/* cleanup */
		g_free (message_id);
		g_free (message);
		group_unref_articles (ours->group, NULL);
		/* FIXME: super dangles */
	}
}

/****
*****
****/

void
article_set_from_raw_message   (Article         * a,
                                const gchar     * text)
{
	guint i;
	guint16 lines;
	GMimeMessage * msg;
	GPtrArray * arr;
	gchar * body;
	const gchar * pch;
	gchar * tmp;
	gboolean foo;

	/* get the message */
	g_return_if_fail (a != NULL);
	g_return_if_fail (is_nonempty_string(text));
	msg = pan_g_mime_parser_construct_message (text);
	g_return_if_fail (msg != NULL);

	/* body */
	a->date = msg->header->date;
	body = g_mime_message_get_body (msg, TRUE, &foo);
	article_set_header (a, PAN_BODY, body, DO_CHUNK);
	for (lines=0, pch=body; pch!=NULL && *pch; ++pch)
		if (*pch=='\n')
			++lines;
	a->linecount = lines;
	g_free (body);

	/* headers */
	article_set_header (a, HEADER_FROM, msg->header->from, DO_CHUNK);
	article_set_header (a, HEADER_REPLY_TO, msg->header->reply_to, DO_CHUNK);
	article_set_header (a, HEADER_SUBJECT, msg->header->subject, DO_CHUNK);
	article_set_header (a, HEADER_MESSAGE_ID, msg->header->message_id, DO_CHUNK);
	arr = msg->header->arbitrary_headers;
	for (i=0; i!=arr->len; ++i) {
		GMimeHeader * h = (GMimeHeader*) g_ptr_array_index (arr, i);
		const gchar * name = h->name;
		if (!pan_strcmp (name, "Author")) /* back compatability with 0.9.2 */
			name = HEADER_FROM;
		article_set_header (a, h->name, h->value, DO_CHUNK);
	}

	/* some of these old archives didn't keep the author's name?!! */
	tmp = article_get_author_str (a);
	if (!is_nonempty_string(tmp)) {
		gchar * foo = get_default_author_from ();
		article_set_author_from_header (a, foo);
		g_free (foo);
	}
	g_free (tmp);

	/* state */
	pch = article_get_header (a, "Status");
	if (pch != NULL) {
		const gboolean read = strchr (pch, 'R') != NULL;
		article_set_read (a, read);
		/* const gboolean old = strchr (pch, 'O') != NULL; */
		article_remove_header (a, "Status");
	}

	/* references */
	pch = article_get_header (a, HEADER_REFERENCES);
	if (pch != NULL)
		a->references = pch;

	/* cleanup */
	g_mime_message_destroy (msg);
}
