/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <glib.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkctree.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkmain.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-url.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-pixmap.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-uidefs.h>

#include "article.h"
#include "articlelist.h"
#include "article-thread.h"
#include "article-toolbar.h"
#include "bozo.h"
#include "debug.h"
#include "decode.h"
#include "dialogs/dialogs.h"
#include "file-headers.h"
#include "gnksa.h"
#include "grouplist.h"
#include "fnmatch.h"
#include "globals.h"
#include "group.h"
#include "gui.h"
#include "gui-paned.h"
#include "log.h"
#include "message-window.h"
#include "prefs.h"
#include "queue.h"
#include "save.h"
#include "status-item.h"
#include "task-body.h"
#include "task-bodies.h"
#include "task-decode.h"
#include "task-headers.h"
#include "thread-watch.h"
#include "text.h"
#include "util.h"

#include "xpm/blue_flag.xpm"
#include "xpm/book_open.xpm"
#include "xpm/book_closed.xpm"
#include "xpm/mini_page.xpm"
#include "xpm/bluecheck.xpm"
#include "xpm/x.xpm"
#include "xpm/binary.xpm"
#include "xpm/binary_complete.xpm"
#include "xpm/binary_incomplete.xpm"

/***
****
****
****  Abandon hope, all ye who enter here
****
****
***/



/**********************************************************
 * Private Variables
 **********************************************************/

/* state */
static pthread_mutex_t hash_mutex;
static Group * my_group                          = NULL;
static GPtrArray * article_headers               = NULL;
static GHashTable * messageid_to_node            = NULL;
static gboolean is_threaded                      = TRUE;
static int sort_type                             = ARTICLE_SORT_SUBJECT;
static pthread_mutex_t article_ctree_lock        = PTHREAD_MUTEX_INITIALIZER;

/* article list */
static GnomePixmap * article_read_pixmap         = NULL;
static GnomePixmap * articlelist_closed_pixmap   = NULL;
static GnomePixmap * articlelist_open_pixmap     = NULL;
static GnomePixmap * binary_decoded_pixmap       = NULL;
static GnomePixmap * decode_queue_pixmap         = NULL;
static GnomePixmap * error_pixmap                = NULL;
static GnomePixmap * flagged_for_download_pixmap = NULL;
static GnomePixmap * mp_complete_pixmap          = NULL;
static GnomePixmap * mp_incomplete_pixmap        = NULL;

static void tree_select_row_cb (GtkCTree*, GtkCTreeNode*, gint, gpointer);

PanCallback * articlelist_state_filter_changed   = NULL;
PanCallback * articlelist_group_changed          = NULL;
PanCallback * articlelist_sort_changed           = NULL;
PanCallback * articlelist_thread_changed         = NULL;
PanCallback * articlelist_selection_changed      = NULL;

GtkStyle * killfile_style                        = NULL;
GtkStyle * watched_style                         = NULL;
GtkStyle * unread_style                          = NULL;
GtkStyle * read_style                            = NULL;
GtkStyle * new_style                             = NULL;

GdkColor new_color;
GdkColor read_color;
GdkColor unread_color;
GdkColor killfile_color;
GdkColor watched_color;

GtkWidget * article_ctree_menu;


/**********************************************************
 * Private Functions
 **********************************************************/

static void
articlelist_menu_popup (GdkEventButton* bevent);
static void
articlelist_unread_inc_nolock (GtkCTreeNode * child, int inc);

static void
articlelist_repopulate (const Group* group,
                        Article** article_buf,
                       	int article_qty,
                        StatusItem *item,
                        GPtrArray * sel_articles);

static GtkCTreeNode* articlelist_get_node_from_message_id (
	const char* message_id);

static GtkCTreeNode*
add_article_to_ctree (GtkCTree      * tree,
		      GtkCTreeNode  * parent,
		      GtkCTreeNode  * sibling,
		      Article       * article,
		      StatusItem    * item,
		      gboolean        expanded);

static void articlelist_update_node_impl (const Article* article,
                                          gboolean need_lock);

static GnomePixmap* get_article_pixmap (const Article*);

static void articlelist_click_column (GtkCList*, int col_clicked);

static void article_ctree_destroy_cb (void);

static void articlelist_update_node_fast_nolock (GtkCTreeNode*, const Article*);

static void articlelist_selected_delete_attachment (void);

/***
****
****
****  POPUP MENUS
****
****
***/


static GnomeUIInfo search_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Search for user on deja"),
		N_("Search for other postings by this use on deja.com"),
		search_deja,
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo filter_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Watch Thread"),
		N_("Watch Thread"),
		articlelist_selected_thread_watch, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'W', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Ignore Thread"),
		N_("Ignore Thread"),
		articlelist_selected_thread_ignore, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'I', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Bozoize Author"),
		N_("Bozoize Author"),
		articlelist_bozoize_author, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'\0', 0, NULL
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo save_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("_Save Article Text..."),
		N_("Save the article headers and body to a text file."),
		save_current_article, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
		'S', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Save Binary Attachment"),
		N_("Save Binary Attachment"),
		articlelist_selected_decode, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
		'S', GDK_SHIFT_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Save Binary Attachment As..."),
		N_("Save Binary Attachment As..."),
		articlelist_selected_decode_as, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
		'S', GDK_CONTROL_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Open Binary Attachment"),
		N_("Open Binary Attachment"),
		articlelist_selected_open, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN,
		'O', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Delete Decoded Binary Attachment"),
		N_("Delete Decoded Binary Attachment"),
		articlelist_selected_delete_attachment, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
		'\0', 0, NULL
	},
	GNOMEUIINFO_END
};


static GnomeUIInfo delete_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Selected Articles"),
		N_("Selected Articles"),
		articlelist_selected_delete, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
		0, 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Selected Articles and Replies"),
		N_("Selected Articles and Replies"),
		articlelist_selected_delete_thread, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
		0, 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Entire Group"),
		N_("Entire Group"),
		articlelist_all_delete, NULL,
		NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
		0, 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Delete Decoded Binary Attachment"),
		N_("Delete Decoded Binary Attachment"),
		articlelist_selected_delete_attachment, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
		0, 0, NULL
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo folder_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Edit Selected Article"),
		N_("Edit Selected Article"),
		articlelist_selected_edit
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Copy Selected Articles"),
		N_("Copy Selected Articles"),
		articlelist_selected_copy_to_folder
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Copy Selected Articles and Replies"),
		N_("Copy Selected Articles and Replies"),
		articlelist_selected_copy_to_folder_deep
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Move Selected Articles"),
		N_("Move Selected Articles"),
		articlelist_selected_move_to_folder
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Move Selected Articles and Replies"),
		N_("Move Selected Articles and Replies"),
		articlelist_selected_move_to_folder_deep
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo mark_read_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Selected Articles"),
		N_("Selected Articles"),
		articlelist_selected_mark_read, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'M', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Selected Articles and Replies"),
		N_("Selected Articles and Replies"),
		articlelist_selected_mark_read_deep, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'M', GDK_SHIFT_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Selected Threads"),
		N_("Selected Threads"),
		articlelist_selected_mark_read_thread, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'M', GDK_MOD1_MASK, NULL
	},
        {
		GNOME_APP_UI_ITEM,
		N_("Entire Group"),
		N_("Entire Group"),
		articlelist_all_mark_read, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'M', GDK_SHIFT_MASK|GDK_CONTROL_MASK, NULL,
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo mark_unread_menu[] =
{

	{
		GNOME_APP_UI_ITEM,
		N_("Selected Articles"),
		N_("Selected Articles"),
		articlelist_selected_mark_unread
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Selected Articles and Replies"),
		N_("Selected Articles and Replies"),
		articlelist_selected_mark_unread_deep
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Selected Threads"),
		N_("Selected Threads"),
		articlelist_selected_mark_unread_thread,
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Entire Group"),
		N_("Entire Group"),
		articlelist_all_mark_unread,
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo bodies_menu[] =
{
        {
		GNOME_APP_UI_ITEM,
		N_("Selected Articles"),
		N_("Selected Articles"),
		articlelist_selected_download
	},                                                                      
	{
		GNOME_APP_UI_ITEM,
		N_("Selected Articles and Replies"),
		N_("Selected Articles and Replies"),
		articlelist_selected_thread_download
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Entire Group"),
		N_("Entire Group"),
		articlelist_all_download,
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Flag Selected Articles"),
		N_("Flag Selected Articles"),
		articlelist_selected_flag_for_dl, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'J', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Flag Selected Articles and Replies"),
		N_("Flag Selected Articles and Replies"),
		articlelist_selected_flag_for_dl_deep, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'J', GDK_SHIFT_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Unflag Selected Articles"),
		N_("Unflag Selected Articles"),
		articlelist_selected_unflag_for_dl, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'K', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Unflag Selected Articles and Replies"),
		N_("Unflag Selected Articles and Replies"),
		articlelist_selected_unflag_for_dl_deep, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'K', GDK_SHIFT_MASK, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Download Flagged Articles"),
		N_("Download Flagged Articles"),
		articlelist_download_flagged_bodies, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'J', GDK_SHIFT_MASK|GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_END
};

GnomeUIInfo articlelist_menu [] =
{
        {
		GNOME_APP_UI_ITEM, 
		N_("Read Selected Message"),
		N_("Read Selected Message"),
		articlelist_selected_read_body, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'V', 0, NULL
	},
        {
		GNOME_APP_UI_ITEM, 
		N_("Read Selected Message in New Window"),
		N_("Read Selected Message in New Window"),
		articlelist_selected_read_body_in_new_window, NULL,
		NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'V', GDK_SHIFT_MASK, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_SUBTREE,
		N_("Save"),
		N_("Save"),
		&save_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_SUBTREE,
		N_("Search Offsite"),
		N_("Search Offsite"),
		&search_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_SUBTREE,
		N_("Filter"),
		N_("Filter"),
		&filter_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_SUBTREE,
		N_("Mark as Read"),
		N_("Mark as Read"),
		&mark_read_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_SUBTREE,
		N_("Mark as Unread"),
		N_("Mark as Unread"),
		&mark_unread_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_SUBTREE,
		N_("Download Bodies"),
		N_("Download Bodies"),
		&bodies_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_SUBTREE,
		N_("Folder Tools"),
		N_("Folder Tools"),
		&folder_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_SUBTREE,
		N_("Delete"),
		N_("Delete"),
		&delete_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Select All Articles"),
		N_("Select All Articles"),
		articlelist_select_all
	},
	GNOMEUIINFO_END
};

/***
****
****
****  INTERNAL UTILITIES:  SELECTIONS, ARTICLES, and NODES
****
****
***/

static Article*
articlelist_get_article_from_node (GtkCTreeNode* node)
{
	Article* article = NULL;

	if (node != NULL)
		article = ARTICLE(
			  gtk_ctree_node_get_row_data (
				  GTK_CTREE(Pan.article_ctree), node));

	return article;
}

GtkCTreeNode*
articlelist_get_selected_node (void)
{
	GtkCTreeNode* node = NULL;

	const GList* list =  GTK_CLIST(Pan.article_ctree)->selection;
	if (list != NULL)
		node = GTK_CTREE_NODE(list->data);

	return node;
}


GPtrArray*
articlelist_get_selected_articles (void)
{
	GPtrArray * retval;
	GList * list;

	retval = g_ptr_array_new ();

	list = GTK_CLIST(Pan.article_ctree)->selection;
	pan_g_ptr_array_reserve (retval, g_list_length(list));
	for (; list!=NULL; list=list->next)
	{
		GtkCTreeNode * n = GTK_CTREE_NODE(list->data);
		Article * a = articlelist_get_article_from_node(n);
		if (a != NULL)
			g_ptr_array_add (retval, a);
	}

	return retval;
}

Article*
articlelist_get_selected_article (void)
{
	return articlelist_get_article_from_node (
		articlelist_get_selected_node());
}

void
articlelist_set_selected_nodes (GtkCTreeNode **nodes, gint node_qty)
{
	if (nodes!=NULL && node_qty>0)
	{
		gint i;
		gfloat hadj;
		GtkCTree * tree = GTK_CTREE(Pan.article_ctree);
		GtkCList * list = GTK_CLIST(Pan.article_ctree);

		pan_lock();
		gtk_clist_freeze (list);
		gtk_ctree_unselect_recursive (tree, gtk_ctree_node_nth(tree, 0));

		for (i=0; i<node_qty; ++i)
		{
			GtkCTreeNode * node = nodes[i];
			GtkCTreeNode * n = NULL;

			gtk_ctree_select (tree, node);

			/* show all the children */
			for (n=GTK_CTREE_ROW(node)->parent; n; n=GTK_CTREE_ROW(n)->parent)
				gtk_ctree_expand(tree, n);
		}

		/* set adjustment */
		hadj = gtk_clist_get_hadjustment(list)->value;
		gtk_ctree_node_moveto (tree, nodes[0], 0, (gfloat)0.5, (gfloat)0.0);
		gtk_adjustment_set_value (gtk_clist_get_hadjustment(list), hadj);

		gtk_clist_thaw (list);
		pan_unlock();
	}
}

/***
****
****
****  INTERNAL UTILITIES:   GETTING THREADS
****
****
***/

typedef enum
{
	GET_WHOLE_THREAD,
	GET_SUBTHREAD
}
ThreadGet;

static int
selthread_comp (const void* va, const void* vb)
{
	return strcmp (((const Article*)va)->message_id,
	               (*(const Article**)vb)->message_id);
}

static GPtrArray*
articlelist_get_unique_threads (const GPtrArray * articles,
                                ThreadGet thread_get)
{
	int i;
	GPtrArray* all = g_ptr_array_new ();
	GPtrArray* thread = g_ptr_array_new ();

	/* get the thread for each article passed in */
	for (i=0; i<articles->len; ++i)
	{
		int j;
		gboolean exact_match = FALSE;
		Article * a = ARTICLE(g_ptr_array_index(articles,i));

		/* if article is already in all, then so is
		   its thread, so we can skip it. */
		lower_bound (a, all->pdata, all->len, sizeof(Article*),
		             selthread_comp, &exact_match);
		if (exact_match)
			continue;

		/* get the thread associated with sel... */
		g_ptr_array_set_size (thread, 0);
		if (thread_get == GET_SUBTHREAD)
			article_get_subthread (a, thread);
		else
			article_get_entire_thread (a, thread);

		/* for each article in the thread, make sure there's exactly
		   one instance of it in the "all" array */
		for (j=0; j<thread->len; ++j)
		{
			Article * b = ARTICLE(g_ptr_array_index(thread,j));
			int index;
			exact_match = FALSE;
			index = lower_bound (b, all->pdata, all->len,
					     sizeof(Article*),
					     selthread_comp, &exact_match);
			if (!exact_match)
				pan_g_ptr_array_insert (all, b, index);
		}
	}

	g_ptr_array_free (thread, TRUE);
	return all;
}

/***
****
****
****  OFFSITE SEARCH
****
****
***/

/**
 * Start an offline search, such as through deja or remarq.
 * @param Article to key off of
 * @param fmt the URL, including substitution formats:
 *        %a will be replaced with the article's author's email address
 */
static void
search_offsite (const Article * a, const gchar * fmt)
{
	GString * url = NULL;

	/* sanity checking */
	g_return_if_fail (a != NULL);
	g_return_if_fail (is_nonempty_string(fmt));

	/* set up the URL */
       	url = g_string_new (fmt);

	/* replace "%a" with the article's author's email address */
	if (strstr (url->str, "%a") != NULL)
		pan_g_string_replace (url, "%a", a->author_addr);

	/* go! */
	gnome_url_show (url->str);

	/* cleanup */
	g_string_free (url, TRUE);
}

/**
 * Search deja for articles by the same author
 */
void
search_deja (void)
{
	Article * a = articlelist_get_selected_article ();
	if (a != NULL)
		search_offsite (a, "http://www.deja.com/dnquery.xp?QRY=%a&ST=MS&svcclass=dnserver&DBS=1");
}

/***
****
****
****  NAVIGATION
****
****
***/

GtkCTreeNode*
articlelist_node_next (GtkCTreeNode * node, gboolean sibling_only)
{
	GtkCTreeNode* n;

	if (!node)
		return NULL;

	if (sibling_only)
		return GTK_CTREE_ROW(node)->sibling;

	if (GTK_CTREE_ROW(node)->children)
		return GTK_CTREE_ROW(node)->children;

	for (n=node; n!=NULL; n=GTK_CTREE_ROW(n)->parent)
		if (GTK_CTREE_ROW(n)->sibling)
			return GTK_CTREE_ROW(n)->sibling;

	return NULL;
}

GtkCTreeNode*
articlelist_node_prev (GtkCTreeNode * node, gboolean sibling_only)
{
	GtkCTreeNode * parent;
	GtkCTreeNode * sibling;
	
	if (!node)
		return NULL;

	/* get parent */	
	parent = GTK_CTREE_ROW(node)->parent;
	if (!parent)
		return NULL;

	/* parent's first child */
	sibling=GTK_CTREE_ROW(parent)->children;
	if (sibling==node) /* this is the first child */
		return parent;

	/* previous sibling of node */
	while (GTK_CTREE_ROW(sibling)->sibling != node) {
		sibling = GTK_CTREE_ROW(sibling)->sibling;
		pan_warn_if_fail (sibling != NULL);
	}

	if (sibling_only)
		return sibling;

	/* find the absolutely last child of the older sibling */
	for (;;) {
		GtkCTreeNode* tmp = GTK_CTREE_ROW(sibling)->children;
		if (!tmp)
			return sibling;

		/* last child of sibling */
		while (GTK_CTREE_ROW(tmp)->sibling != NULL)
			tmp = GTK_CTREE_ROW(tmp)->sibling;

		sibling = tmp;
	}

	pan_warn_if_reached ();
}

static GtkCTreeNode*
find_prev_thread (GtkCTreeNode * node)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode * top = gtk_ctree_node_nth (tree, 0);
	const int row_qty = GTK_CLIST(tree)->rows;
	gboolean wrap = FALSE;

	/* nothing to do */
	if (row_qty < 2)
		return NULL;

	/* get starting point */	
	if (node == NULL)
		node = articlelist_get_selected_node ();

	/* find previous */
	if (node != NULL)
		while ( node!=top && GTK_CTREE_ROW(node)->parent!=top )
			node = GTK_CTREE_ROW(node)->parent;

	/* either no node selected or we're at the first thread; wrap */
	if (!node || articlelist_node_prev(node,TRUE)==top) {
		wrap = TRUE;
		node = gtk_ctree_node_nth ( tree, row_qty-1 );
		while ( node!=top && GTK_CTREE_ROW(node)->parent!=top )
			node = GTK_CTREE_ROW(node)->parent;
	}

	/* if we didn't wrap around, we need to back up one. */
	if (!wrap)
		node = articlelist_node_prev (node, TRUE);

	return node;
}

static GtkCTreeNode*
find_next_thread (GtkCTreeNode * node)
{
	GtkCTree* tree;
	GtkCTreeNode * top;

	tree = GTK_CTREE(Pan.article_ctree);
	top = gtk_ctree_node_nth (tree, 0);

	/* if nothing to read... */
	if (GTK_CLIST(tree)->rows<2)
		return NULL;

	/* find the start node */
	if (node == NULL)
		node = articlelist_get_selected_node ();

	/* walk backwards */
	if (node != NULL)
	{
		while (node!=top && GTK_CTREE_ROW(node)->parent!=top)
			node = GTK_CTREE_ROW(node)->parent;
		if (node==top)
			return NULL;
		node = GTK_CTREE_ROW(node)->sibling;
	}
	else if (GTK_CLIST(tree)->rows > 1)
	{
		node = gtk_ctree_node_nth (tree, 1);
	}

	return node;
}

static GtkCTreeNode*
find_prev_article (GtkCTreeNode * node)
{
	GtkCTree * tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode * top = gtk_ctree_node_nth (tree, 0);
	const int row_qty = GTK_CLIST(tree)->rows;
	gboolean wrap = FALSE;

	/* if nothing to read... */
	if (row_qty<2)
		return NULL;

	/* get a starting node... */
	if (node==NULL) {
		node = articlelist_get_selected_node (); 
		if ( !node || articlelist_node_prev(node,TRUE)==top ) {
			wrap = TRUE;
			node = gtk_ctree_node_nth (tree, row_qty-1 );
		}
	}

	/* if we didn't wrap, we need to move backwards by hand */
	if (!wrap)
		node = articlelist_node_prev (node, FALSE);

	return node;
}

static GtkCTreeNode*
find_next_article (GtkCTreeNode * node)
{
	GtkCTree * tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode * retval = NULL;

	/* if nothing to read... */
	if (GTK_CLIST(tree)->rows <2)
		return NULL;

	/* get starting point */
	if (node == NULL)
		node = articlelist_get_selected_node ();
	if (!node)
		node = gtk_ctree_node_nth (tree, 0);

	/* move to next node */
	retval = articlelist_node_next (node, FALSE);

	return retval;
}

static GtkCTreeNode*
find_next_unread (GtkCTreeNode * node)
{
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);

	/* find the first node */
	if (node == NULL) {
		node = articlelist_get_selected_node ();
		if (!node)
			node = gtk_ctree_node_nth (tree, 0);
	}

	/* march through all the next nodes untili we find an unread one */
	for (;;) {
		const Article* article;
		node = articlelist_node_next (node, FALSE);
		if (node == NULL) /* end of articles */
			break;
		article = articlelist_get_article_from_node (node);
		if (article && !article_is_read (article))
			break;
	}

	return node;
}
static GtkCTreeNode*
find_next_unread_body (GtkCTreeNode * node)
{
	node = find_next_unread (node);
	while (node!=NULL && !article_has_body(articlelist_get_article_from_node(node)))
		node = find_next_unread (node);
	return node;
}
void
articlelist_skip_to_next_thread (void)
{
	GtkCTreeNode * node = find_next_thread (NULL);
	if (node != NULL)
		articlelist_set_selected_nodes (&node, 1);
}
void
articlelist_skip_to_prev_thread (void)
{
	GtkCTreeNode * node = find_prev_thread (NULL);
	if (node != NULL)
		articlelist_set_selected_nodes (&node, 1);
}
void
articlelist_skip_to_prev (void)
{
	GtkCTreeNode * node = find_prev_article (NULL);
	if (node != NULL)
		articlelist_set_selected_nodes (&node, 1);
}
void
articlelist_view_prev (void)
{
	articlelist_skip_to_prev ();
	articlelist_selected_read_body ();
}

void
articlelist_skip_to_message_id (const gchar * message_id)
{
	GtkCTreeNode * node = articlelist_get_node_from_message_id (message_id);
	if (node != NULL)
		articlelist_set_selected_nodes (&node, 1);
}
void
articlelist_view_message_id (const gchar * message_id)
{
	articlelist_skip_to_message_id (message_id);
	articlelist_selected_read_body ();
}
   

void
articlelist_skip_to_next (void)
{
	GtkCTreeNode * node = find_next_article (NULL);
	if (node != NULL)
		articlelist_set_selected_nodes (&node, 1);
}
void
articlelist_view_next (void)
{
	articlelist_skip_to_next ();
	articlelist_selected_read_body ();
}
void
articlelist_skip_to_next_unread (void)
{
	GtkCTreeNode * node = find_next_unread (NULL);
	if (node != NULL)
		articlelist_set_selected_nodes (&node, 1);
}
void
articlelist_view_next_unread (void)
{
	articlelist_skip_to_next_unread ();
	articlelist_selected_read_body ();
}
void
articlelist_skip_to_next_unread_body (void)
{
	GtkCTreeNode * node = find_next_unread_body (NULL);
	if (node != NULL)
		articlelist_set_selected_nodes (&node, 1);
}
void
articlelist_view_next_unread_body (void)
{
	articlelist_skip_to_next_unread_body ();
	articlelist_selected_read_body ();
}

void
articlelist_selected_read_body (void)
{
	Article * article = articlelist_get_selected_article ();
	if (article != NULL)
		queue_add(TASK(task_body_new_defaults(article)));
}

void
articlelist_selected_read_body_in_new_window (void)
{
	Article * article = articlelist_get_selected_article ();
	if (article != NULL)
		message_read_window (NULL, article);
}

/***
****
****
****  CLEARING OUT THE ARTICLELIST
****
****
***/

static void
clear_hash_table (void)
{
	pthread_mutex_lock (&hash_mutex);
	if (messageid_to_node != NULL)
		g_hash_table_destroy (messageid_to_node);
	messageid_to_node = g_hash_table_new (g_str_hash, g_str_equal);
	pthread_mutex_unlock (&hash_mutex);
}

static void
clear_clist (void)
{
	GtkCList *list = GTK_CLIST(Pan.article_ctree);
	if (list->rows)
	{
		pan_lock();
		gtk_clist_clear (GTK_CLIST(Pan.article_ctree));
		pan_unlock();
	}
}

static void
clear_article_buffer (StatusItem * status)
{
	clear_hash_table ();
	clear_clist ();

	g_ptr_array_set_size (article_headers, 0);
	if (my_group != NULL)
		group_unref_articles (my_group, status);
}

/***
****
****
****   SORTING
****
****
***/

static const gchar*
column_to_title (int col)
{
	switch (col) {
		case 0: return _("Subject");
		case 1: return _("New");
		case 2: return _("Lines");
		case 3: return _("Author");
		case 4: return _("Date (Time)");
		default: break;
	}
	
	pan_warn_if_reached();
	return _("BUG!!");
}

static int
sort_type_to_column (int type)
{
	int col = 0;
	switch (abs(type)) {
		case ARTICLE_SORT_SUBJECT: col=0; break;
		case ARTICLE_SORT_UNREAD_CHILDREN: col=1; break;
		case ARTICLE_SORT_LINES: col=2; break;
		case ARTICLE_SORT_AUTHOR: col=3; break;
		case ARTICLE_SORT_DATE: col=4; break;
		default: pan_warn_if_reached(); break;
	}

	return col;
}

static void
articlelist_set_sort_type_nosort (int type, gboolean force)
{
	char * pch;
	int col = 0;

	if (!force && (type==sort_type))
		return;

	/* update old column */
	if (abs(sort_type) != abs(type)) {
		col = sort_type_to_column (sort_type);
		pan_lock();
		gtk_clist_set_column_title (GTK_CLIST(Pan.article_ctree), col, column_to_title(col));
		pan_unlock();
	}

	/* update sort type, fire callbacks */
	sort_type = type;
	pan_callback_call (articlelist_sort_changed, Pan.article_ctree, GINT_TO_POINTER(type));

	/* update new column */
	col = sort_type_to_column (sort_type);
	pch = g_strdup_printf ("%c%s", (type>0?'+':'-'), column_to_title(col));
	pan_lock();
	gtk_clist_set_column_title (GTK_CLIST(Pan.article_ctree), col, pch);
	pan_unlock();
	g_free (pch);
}	

static gchar*
sort_selected_describe (const StatusItem* item)
{
	return g_strdup (_("Sorting Articles"));
}

static void
sort_in_thread (void* data)
{
	const int sort = GPOINTER_TO_INT(data);
        Group * group = articlelist_get_group ();
        StatusItem * item = NULL;

        if (!group) /* articlelist is empty */
                return;

	pthread_mutex_lock (&article_ctree_lock);

	if (sort != sort_type)
	{
		GPtrArray * selected;

		/* create a new status item to get sort/thread messages */
		item = STATUS_ITEM(status_item_new(sort_selected_describe));
		pan_object_ref (PAN_OBJECT(item));
		pan_object_sink (PAN_OBJECT(item));
		status_item_set_active (item, TRUE);

		/* tell the world what we're doing */
		status_item_emit_init_steps (item, article_headers->len*2);
		status_item_emit_status (item, _("Sorting Articles"));

		/* update the sort type */
		articlelist_set_sort_type_nosort (sort, FALSE);
		group_set_sort_style (my_group, sort);
		sort_articles ((Article**)article_headers->pdata,
		               article_headers->len,
		               abs(sort),
		               sort>0);

		/* update articles */
		selected = articlelist_get_selected_articles ();
		articlelist_repopulate (my_group,
		                        (Article**)article_headers->pdata,
		                        article_headers->len,
		                        item,
		                        selected);
		g_ptr_array_free (selected, TRUE);

		/* clean out the status item */
		status_item_set_active (item, FALSE);
		pan_object_unref(PAN_OBJECT(item));
	}

	pthread_mutex_unlock (&article_ctree_lock);
}

void
articlelist_set_sort_type (int sort)
{
       	pthread_t thread;
       	pthread_create (&thread, NULL, (void*)sort_in_thread, GINT_TO_POINTER(sort));
       	pthread_detach (thread);
}

/***
****
****
****   ccc
****
****
***/

Group*
articlelist_get_group (void)
{
	return my_group;
}


/**
***
**/

typedef struct
{
	Article * a;
	gint depth;
}
SortDepthStruct;

static int
compare_pSDS_to_pSDS_by_depth (const void * a, const void * b)
{
	return ((SortDepthStruct*)a)->depth -
	       ((SortDepthStruct*)b)->depth;
}

static void
sort_articles_by_depth (GPtrArray * articles, gboolean parents_first)
{
	guint i;
	SortDepthStruct * buf;

	/* make an array of article,depth pairs */
	buf = g_new0 (SortDepthStruct, articles->len);
	for (i=0; i!=articles->len; ++i)
	{
		SortDepthStruct * item = buf + i;
		Article * tmp;
		item->a = ARTICLE(g_ptr_array_index(articles,i));
		item->depth = 0;
		for (tmp=item->a; tmp->parent!=NULL; tmp=tmp->parent)
			++item->depth;
	}

	/* sort the array */
	qsort (buf,
	       articles->len,
	       sizeof(SortDepthStruct),
	       compare_pSDS_to_pSDS_by_depth);


	/* put back into the original GPtrArray */
	if (parents_first)
		for (i=0; i!=articles->len; ++i)
			g_ptr_array_index(articles,i)= buf[i].a;
	else
		for (i=0; i!=articles->len; ++i)
			g_ptr_array_index(articles,i)= buf[articles->len-1-i].a;

	/* cleanup */
	g_free (buf);
}

static int
group_articles_added_cb (gpointer call_obj,
                         gpointer call_arg,
                         gpointer client_arg)
{
	Group * group = GROUP(call_obj);
	GPtrArray * added = (GPtrArray*) call_arg;

	/* sanity checks */
	g_return_val_if_fail (group!=NULL, 0);
	g_return_val_if_fail (added!=NULL, 0);
	g_return_val_if_fail (added->len>0, 0);

	/* if it's not important to us, don't bother */
	if (group != my_group)
		return 0;

	articlelist_reload();
	return 0;
}


static void
remove_article_nolock (Article * article)
{
	int i;
	GtkCTreeNode * node = NULL;

	/* sanity checks */
	g_return_if_fail (article!=NULL);
	g_return_if_fail (is_nonempty_string(article->message_id));

	/* if this article currently has a node, remove the node */
	node = articlelist_get_node_from_message_id (article->message_id);
	if (node!=NULL)
	{
		g_assert (articlelist_get_article_from_node(node) == article);

		/* reparent kids */
		if (is_threaded && article->threads!=NULL)
		{
			GSList* l;

			for (l=article->threads; l!=NULL; l=l->next)
			{
				const char * id = ARTICLE(l->data)->message_id;
				GtkCTreeNode * n;
			        n = articlelist_get_node_from_message_id (id);
				if (n!=NULL)
					gtk_ctree_move (
						GTK_CTREE(Pan.article_ctree), n,
						GTK_CTREE_ROW(node)->parent,
						GTK_CTREE_ROW(node)->sibling);
			}
		}

		/* remove from hashtable */
		pthread_mutex_lock (&hash_mutex);
		g_hash_table_remove (messageid_to_node, article->message_id);
		pthread_mutex_unlock (&hash_mutex);

		/* don't remove the node yet; we need it
		   for article_unread_inc () below */
	}

	/* If this article is in our article headers array, remove it... */
	for (i=0; i!=article_headers->len; ++i) {
		if (article == ARTICLE(g_ptr_array_index(article_headers,i))) {
			g_ptr_array_remove_index (article_headers, i);
			break;
		}
	}


	/* if the article's unread, update the parents' unread children count */
	if (!article_is_read (article))
	{
		GtkCTreeNode *n = node;
		if (n == NULL)
		{
			Article* a;
			for (a=article->parent; a!=NULL && !n; a=a->parent)
				n = articlelist_get_node_from_message_id (
					article->message_id);
		}
		if (n != NULL) 
		{
			articlelist_unread_inc_nolock (n, -1);
		}
	}

	/* remove this node from the tree */
	if (node != NULL)
		gtk_ctree_remove_node (GTK_CTREE(Pan.article_ctree), node);
}

static int
group_articles_removed_cb (gpointer call_obj,
                           gpointer call_arg,
                           gpointer client_data)
{
	Group * group = GROUP(call_obj);
	GPtrArray * removed = (GPtrArray*) call_arg;
	GPtrArray * tmp;

	/* sanity checks */
	g_return_val_if_fail (group!=NULL, 0);
	g_return_val_if_fail (removed!=NULL, 0);
	g_return_val_if_fail (removed->len>0, 0);

	/* if it's not important to us, don't bother */
	if (group != my_group)
		return 0;

	tmp = pan_g_ptr_array_dup (removed);

	/* make sure we do the children first--it reduces gtkclist confusion
	   to go from bottom to top, and this reduces the number of cnode
	   rearranging that we have to do */
	sort_articles_by_depth (tmp, FALSE);

	/* we now know that articles have been removed
	   from the group the articlelist is currently
	   looking at.  Try to remove all traces of them
	   from this list. */
	pan_lock ();
	gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));
	pan_g_ptr_array_foreach (tmp, (GFunc)remove_article_nolock, NULL);
	gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));
	pan_unlock ();

	/* cleanup */
	g_ptr_array_free (tmp, TRUE);

	return 0;
}


/****
*****
*****  MARKING ARTICLES READ/UNREAD
*****
****/

static void
articlelist_mark_articles (const GPtrArray * articles, gboolean read)
{
	g_return_if_fail (articles!=NULL);
	g_return_if_fail (articles->len!=0);

	/* freeze the tree */
	pan_lock ();
	gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));
	pan_unlock();

	/* set the articles as read or unread */
	articles_set_read ((Article**)articles->pdata, articles->len, read);

	/* thaw the tree */
	pan_lock ();
	gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));
	pan_unlock();
}

void
articlelist_all_mark_read (void)
{
	/* update the unread count... */
	if (article_headers != NULL) {
		guint i;
		for (i=0; i!=article_headers->len; ++i) {
			Article * a = g_ptr_array_index(article_headers,i);
			a->unread_children = 0;
		}
	}

	group_mark_all_read (my_group, TRUE);
}
void
articlelist_all_mark_unread (void)
{
	if (article_headers != NULL)
	{
		/* FIXME: update unread_children */
	}

	group_mark_all_read (my_group, FALSE);
}

void
articlelist_selected_mark_unread (void)
{
	GPtrArray * sel = articlelist_get_selected_articles ();
	if (sel->len != 0)
		articlelist_mark_articles (sel, FALSE);
	g_ptr_array_free (sel, TRUE);
}
void
articlelist_selected_mark_read (void)
{
	GPtrArray * sel = articlelist_get_selected_articles ();
	if (sel->len != 0)
		articlelist_mark_articles (sel, TRUE);
	g_ptr_array_free (sel, TRUE);
}

static void
articlelist_selected_mark_threads_impl (int threads, gboolean read)
{
	GPtrArray * sel = articlelist_get_selected_articles ();
	GPtrArray * all = articlelist_get_unique_threads (sel, threads);
	if (all->len != 0)
		articlelist_mark_articles (all, read);
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (all, TRUE);
}
void
articlelist_selected_mark_unread_deep (void)
{
	articlelist_selected_mark_threads_impl (GET_SUBTHREAD, FALSE);
}
void
articlelist_selected_mark_unread_thread (void)
{
	articlelist_selected_mark_threads_impl (GET_WHOLE_THREAD, FALSE);
}
void
articlelist_selected_mark_read_deep (void)
{
	articlelist_selected_mark_threads_impl (GET_SUBTHREAD, TRUE);
}
void
articlelist_selected_mark_read_thread (void)
{
	articlelist_selected_mark_threads_impl (GET_WHOLE_THREAD, TRUE);
}


void
articlelist_selected_unflag_for_dl (void)
{
	guint i;
	GPtrArray * sel = articlelist_get_selected_articles ();
	for (i=0; i!=sel->len; ++i)
		article_remove_flag (ARTICLE(g_ptr_array_index(sel,i)), STATE_DOWNLOAD_FLAGGED);
	g_ptr_array_free (sel, TRUE);
}

void
articlelist_selected_unflag_for_dl_deep (void)
{
	guint i;
	GPtrArray * sel = articlelist_get_selected_articles ();
	GPtrArray * all = articlelist_get_unique_threads (sel, GET_SUBTHREAD);
	for (i=0; i!=all->len; ++i)
		article_remove_flag (ARTICLE(g_ptr_array_index(all,i)), STATE_DOWNLOAD_FLAGGED);
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (all, TRUE);
}

void
articlelist_selected_flag_for_dl (void)
{
	guint i;
	GPtrArray * sel = articlelist_get_selected_articles ();
	for (i=0; i!=sel->len; ++i)
		article_add_flag (ARTICLE(g_ptr_array_index(sel,i)), STATE_DOWNLOAD_FLAGGED);
	g_ptr_array_free (sel, TRUE);
}

void
articlelist_selected_flag_for_dl_deep (void)
{
	guint i;
	GPtrArray * sel = articlelist_get_selected_articles ();
	GPtrArray * all = articlelist_get_unique_threads (sel, GET_SUBTHREAD);
	for (i=0; i!=all->len; ++i)
		article_add_flag (ARTICLE(g_ptr_array_index(all,i)), STATE_DOWNLOAD_FLAGGED);
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (all, TRUE);
}

void
articlelist_download_flagged_bodies (void)
{
	guint i;
	GPtrArray * flagged;
	Task * task;

	/* build the array of flagged articles */	
	flagged = g_ptr_array_new ();
	for (i=0; i!=article_headers->len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(article_headers,i));
		if (article_flag_on (a, STATE_DOWNLOAD_FLAGGED))
			g_ptr_array_add (flagged, a);
	}

	/* queue a task */
	if (flagged->len)
	{
		task = TASK(task_bodies_new (my_group, flagged));
		if (task != NULL)
			queue_add (task);
		else
			g_ptr_array_free (flagged, TRUE);
	}
}

/****
*****
*****  COPYING TO FOLDERS
*****
****/

static void
articlelist_article_to_folder (GPtrArray * articles, gboolean keep_original)
{
	int i;
	Group * folder;
	GPtrArray * addme;

	/* make sure the user's selected a folder. */
       	folder = grouplist_get_selected_group ();
	if (folder==NULL || !group_is_folder(folder)) {
		pan_error_dialog (_("Please Select a Folder from the Group list."));
		return;
	}

	/* find the articles which we're adding to the folder -- either by making
	   copies of the original articles (for copying) or the original articles
	   themselves (for moving) */
	addme = g_ptr_array_new ();
	for (i=0; i!=articles->len; ++i)
	{
		gchar * body;
		Article * original = ARTICLE(g_ptr_array_index(articles,i));
		Article * article = keep_original ? article_dup(folder,original) : original;

		/* move the body out of acache */
		body = article_get_body (original);
		article_set_header (article, PAN_BODY, body, DO_CHUNK);
		g_free (body);

		/* add the article to the folder */
		g_ptr_array_add (addme, article);
	}

	/* add the articles to the folder */
	if (addme->len != 0)
		group_add_articles_remove_unused (folder, addme, NULL);

	/* clean up */
	g_ptr_array_free (addme, TRUE);
}
void
articlelist_selected_copy_to_folder (void)
{
	GPtrArray * sel = articlelist_get_selected_articles ();
	articlelist_article_to_folder (sel, TRUE);
	g_ptr_array_free (sel, TRUE);
}
void
articlelist_selected_copy_to_folder_deep (void)
{
	GPtrArray * sel = articlelist_get_selected_articles ();
	GPtrArray * all = articlelist_get_unique_threads (sel, GET_SUBTHREAD);
	articlelist_article_to_folder (all, TRUE);
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (all, TRUE);
}
void
articlelist_selected_move_to_folder (void)
{
	GPtrArray * sel = articlelist_get_selected_articles ();
	articlelist_article_to_folder (sel, FALSE);
	g_ptr_array_free (sel, TRUE);
}
void
articlelist_selected_move_to_folder_deep (void)
{
	GPtrArray * sel = articlelist_get_selected_articles ();
	GPtrArray * all = articlelist_get_unique_threads (sel, GET_SUBTHREAD);
	articlelist_article_to_folder (all, FALSE);
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (all, TRUE);
}
void
articlelist_selected_edit (void)
{
	Article * a = articlelist_get_selected_article ();
	if (a != NULL)
		message_edit_window (NULL, a);
}

void
articlelist_selected_cancel (void)
{
	Article * a = articlelist_get_selected_article ();
	if (a != NULL)
		article_cancel (a);
}

void
articlelist_selected_supersede (void)
{
	Article * a = articlelist_get_selected_article ();
	if (a != NULL)
		article_supersede (a);
}

/****
*****
*****  DOWNLOADING ARTICLE BODIES
*****
****/

void
articlelist_all_download (void)
{
	GPtrArray * articles;
	Task * task;

	/* sanity check */
	g_return_if_fail (my_group!=NULL);
	g_return_if_fail (article_headers!=NULL);
	g_return_if_fail (article_headers->len>0);

	/* build the list */
	articles = g_ptr_array_new ();
	pan_g_ptr_array_assign (articles,
	                        article_headers->pdata,
				article_headers->len);


	/* queue the task */
	task = TASK(task_bodies_new (my_group, articles));
	if (task != NULL)
		queue_add (task);

	/* cleanup */
	if (task == NULL)
		g_ptr_array_free (articles, TRUE);
}

void
articlelist_selected_download (void)
{
	GPtrArray * sel = articlelist_get_selected_articles ();
	Task * task;

	g_return_if_fail (my_group != NULL);

	/* add the task */
	task = TASK(task_bodies_new(my_group, sel));
	if (task != NULL)
		queue_add (task);

	/* cleanup */
	if (task == NULL)
		g_ptr_array_free (sel, TRUE);
}



void
articlelist_selected_thread_download (void)
{
	GPtrArray * sel = articlelist_get_selected_articles ();
	GPtrArray * all = articlelist_get_unique_threads (sel, GET_SUBTHREAD);
	Task * task;

	/* add the task */
	task = TASK(task_bodies_new(my_group, all));
	if (task != NULL)
		queue_add (task);

	/* cleanup */
	g_ptr_array_free (sel, TRUE);
	if (task == NULL)
		g_ptr_array_free (all, TRUE);
}

/***
****
****
****   IGNORING / WATCHING THREADS
****
****
***/

static void
rekill_thread (Article * article)
{
	guint i;
	GPtrArray * articles;
	gint state;

	g_return_if_fail (article != NULL);

	/* get the articles and their state */
	state = thread_get_state (article);
       	articles = g_ptr_array_new ();
	article_get_entire_thread (article, articles);

	/* update the tree */
	pan_lock ();
	gtk_clist_freeze (GTK_CLIST(Pan.article_ctree));
	for (i=0; i!=articles->len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(articles,i));
		articlelist_update_node_impl (a, FALSE);
	}
	gtk_clist_thaw (GTK_CLIST(Pan.article_ctree));
	pan_unlock();

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
}

void
articlelist_bozoize_author (void)
{
	Article* a = articlelist_get_selected_article ();
	if (a != NULL)
	{
		const gchar * addr = a->author_addr;
		const gchar * real = a->author_real;
		if (is_bozo (addr, real))
			bozo_remove (addr, real);
		else
			bozo_add (addr, real);
	}
}

void
articlelist_selected_thread_watch (void)
{
	Article* article = articlelist_get_selected_article ();
	if (article != NULL)
	{
		gint o = thread_get_state (article);
		gint n = o==THREAD_WATCHED ? THREAD_NOTHING : THREAD_WATCHED;
		thread_set_state (article, n);
		rekill_thread (article);
	}
}

void
articlelist_selected_thread_ignore (void)
{
	Article* article = articlelist_get_selected_article ();
	if (article != NULL)
	{
		gint o = thread_get_state (article);
		gint n = o==THREAD_IGNORED ? THREAD_NOTHING : THREAD_IGNORED;
		thread_set_state (article, n);
		rekill_thread (article);
	}
}

/**
***
***
**/

static void
articlelist_selected_decode_impl (gboolean open)
{
	GSList * sel;
	GtkWidget * w = Pan.article_ctree;

	/* sanity checks... */
	g_return_if_fail (my_group!=NULL);

	/* loop through all the selected articles */
	gtk_clist_freeze (GTK_CLIST(w));
	for (sel=(GSList*)GTK_CLIST(w)->selection; sel!=NULL; sel=sel->next)
	{
		GSList * articles = NULL;
		GtkCTreeNode * node = GTK_CTREE_NODE(sel->data);
		Article * a = articlelist_get_article_from_node (node);
		const gchar * filename;

		/* make sure this is an article... */
		if (a == NULL)
			continue;

		filename = article_get_header (a, PAN_HEADER_FILENAME);
		if (is_nonempty_string(filename) && file_exists(filename)) {
			if (open)
				open_outside_file (filename);
			continue;
		}

		/* build an array of this node and all its children. */
		articles = g_slist_copy (a->threads);
		articles = g_slist_prepend (articles, a);

		/* queue for decode */
		task_decode_validate_and_queue_or_self_destruct (
			TASK_DECODE(
				task_decode_new(
					(Group*)my_group,
					a->subject,
					NULL, NULL,
					articles, open)), -1);

		/* cleanup this loop */
		g_slist_free (articles);
	}

	/* cleanup */
	gtk_ctree_unselect_recursive (GTK_CTREE(w), NULL);
	gtk_clist_thaw (GTK_CLIST(w));
}

void
articlelist_selected_decode_as (void)
{
	guint i;
	GPtrArray * sel;
	GtkWidget * w = Pan.article_ctree;
	GSList * gslists_of_articles = NULL;

	/* sanity checks... */
	g_return_if_fail (my_group!=NULL);

	/* loop through all the selected articles */
	gtk_clist_freeze (GTK_CLIST(w));
	sel = articlelist_get_selected_articles ();
	for (i=0; i!=sel->len; ++i)
	{
		GSList * l;
		Article * article = ARTICLE(g_ptr_array_index(sel,i));

		/* build an array of this node and all its children. */
		l = g_slist_copy (article->threads);
		l = g_slist_prepend (l, article);

		/* add this to the gslist of articles */
		gslists_of_articles = g_slist_prepend (gslists_of_articles, l);
	}

	/* save_attach_as owns gslists_of_articles now */
	gslists_of_articles = g_slist_reverse (gslists_of_articles);
	save_attachment_as (my_group, gslists_of_articles);

	/* cleanup */
	gtk_ctree_unselect_recursive (GTK_CTREE(w), NULL);
	gtk_clist_thaw (GTK_CLIST(w));
	g_ptr_array_free (sel, TRUE);
}

void
articlelist_selected_decode (void)
{		
	articlelist_selected_decode_impl (FALSE);
}

void
articlelist_selected_open (void)
{
	articlelist_selected_decode_impl (TRUE);
}


static void
articlelist_selected_delete_attachment (void)
{
	guint i;
	GPtrArray * sel;
	GtkWidget * w = Pan.article_ctree;

	/* sanity checks... */
	g_return_if_fail (my_group!=NULL);

	/* loop through all the selected articles */
	gtk_clist_freeze (GTK_CLIST(w));
	sel = articlelist_get_selected_articles ();
	for (i=0; i!=sel->len; ++i)
	{
		Article * article = ARTICLE(g_ptr_array_index(sel,i));
		const gchar * filename = article_get_header (article, PAN_HEADER_FILENAME);
		if (filename == NULL)
			continue;

		if (!file_exists (filename) || (remove (filename) == -1))
		{
			/* the user deleted or moved the file,
			   or unlink() fails for some other reason */
			pan_error_dialog (_("Unable to delete attachment file: \"%s\n\n"
					    "Errno says: %s"),
					  filename,
					  g_strerror (errno));
		}

		article_remove_flag (article, STATE_DECODED);
		article_set_read (article, FALSE);
		article_remove_header (article, PAN_HEADER_FILENAME);
	}

	/* cleanup */
	gtk_ctree_unselect_recursive (GTK_CTREE(w), NULL);
	gtk_clist_thaw (GTK_CLIST(w));
	g_ptr_array_free (sel, TRUE);
}

void
articlelist_select_all (void)
{
	pan_lock ();
	gtk_ctree_select_recursive (GTK_CTREE(Pan.article_ctree), NULL);
	pan_unlock ();
}


/***
****
****
****  POPULATING THE TREE:  UTILITY FUNCTIONS
****
****
***/

static GnomePixmap*
get_article_pixmap (const Article* article)
{
	GnomePixmap * retval = NULL;

	g_return_val_if_fail (article!=NULL, NULL);

	if (article_flag_on (article, STATE_ERROR))
		retval = error_pixmap;

	else if (article_flag_on (article, STATE_DOWNLOAD_FLAGGED))
		retval = flagged_for_download_pixmap;

	else if (article_flag_on (article, STATE_DECODE_QUEUED))
		retval = decode_queue_pixmap;

	else if (article_flag_on (article, STATE_DECODED))
		retval = binary_decoded_pixmap;

	else if (article_is_read (article))
		retval = article_read_pixmap;

	else if (article_flag_on (article, STATE_MULTIPART_ALL))
		retval = mp_complete_pixmap;

	else if (article_flag_on (article, STATE_MULTIPART_SOME))
		retval = mp_incomplete_pixmap;

	return retval;
}


static void
articlelist_set_node_style_nolock (GtkCTreeNode  * node,
			           const Article * article)
{
	GtkCTree* tree = GTK_CTREE (Pan.article_ctree);

	/* initialize the styles/fonts the first time through */
	if (unread_style == NULL)
	{
		GtkStyle * s = gtk_widget_get_style (GTK_WIDGET(Pan.window));
		killfile_style = gtk_style_copy (s);
		killfile_style->fg[0] = killfile_color;
		watched_style = gtk_style_copy (s);
		watched_style->fg[0] = watched_color;
		unread_style = gtk_style_copy (s);
		unread_style->fg[0] = unread_color;
		read_style = gtk_style_copy (s);
		read_style->fg[0] = read_color;
		new_style = gtk_style_copy (s);
		new_style->fg[0] = new_color;

		if (articlelist_font != NULL)
		{
			GdkFont * font = use_gdk_fontset_load
				? gdk_fontset_load (articlelist_font)
				: gdk_font_load (articlelist_font);

			if (font != NULL)
			{
				gdk_font_ref (font);
				killfile_style->font = font;

				gdk_font_ref (font);
				unread_style->font = font;

				gdk_font_ref (font);
				read_style->font = font;

				gdk_font_ref (font);
				new_style->font = font;
			}
		}
	}

	/* if user has set a font, then we must set the articlelist style */
	if (articlelist_font != NULL)
	{
		gint thread_state;
		GtkStyle * style = NULL;

		if (!article)
			style = read_style;

		else if ((thread_state = thread_get_state (article)) != THREAD_NOTHING) {
			style = thread_state==THREAD_IGNORED
				? killfile_style
				: watched_style;
		}

		else if (article_is_read (article)
			|| article_flag_on (article, STATE_DECODED))
			style = read_style;

		else if (article_is_new (article))
			style = new_style;

		else
			style = unread_style;

		gtk_ctree_node_set_row_style (tree, node, style);
	}
	else /* otherwise just set the color; it's a little cheaper */
	{
		GdkColor * color = NULL;
		gint thread_state;

		if (!article)
			color = &read_color;

		else if ((thread_state = thread_get_state (article)) != THREAD_NOTHING) {
			color = thread_state==THREAD_IGNORED
				? &killfile_color
				: &watched_color;
		}

		else if (article_is_read (article) || article_flag_on (article, STATE_DECODED))
			color = &read_color;

		else if (article_is_new(article))
			color = &new_color;

		else
			color = &unread_color;


		gtk_ctree_node_set_foreground (tree, node, color);
	}
}

static gboolean
article_passes_filter (const Article* article)
{
	gboolean step_passed;
	const guint state_filter = my_group->state_filter;

	const gboolean is_complete_bin =
		article_flag_on (article, STATE_MULTIPART_ALL)
		|| (article->part>1 && article_flag_on(article->parent, STATE_MULTIPART_ALL));
	const gboolean is_incomplete_bin =
		article_flag_on (article, STATE_MULTIPART_SOME)
		|| (article->part>1 && article_flag_on(article->parent, STATE_MULTIPART_SOME));
	const gboolean complete_bin_on =
		state_filter & STATE_FILTER_COMPLETE_BINARIES;
	const gboolean incomplete_bin_on =
		state_filter & STATE_FILTER_INCOMPLETE_BINARIES;
	const gboolean nonbin_on =
		state_filter & STATE_FILTER_NONBINARIES;
	const gboolean unread_on =
		state_filter & STATE_FILTER_UNREAD;
	const gboolean read_on =
		state_filter & STATE_FILTER_READ;
	const gboolean new_on =
		state_filter & STATE_FILTER_NEW;
	const gboolean is_read =
		article_is_read (article);
	const gboolean is_new =
		article_is_new (article);

	const gint thread_state =
		thread_get_state (article);
	const gboolean ignored_on =
		state_filter & STATE_FILTER_IGNORED;
	const gboolean is_ignored =
		thread_state == THREAD_IGNORED;
	const gboolean watch_on =
		state_filter & STATE_FILTER_WATCHED;
	const gboolean is_watched =
		thread_state == THREAD_WATCHED;
	const gboolean normal_rank_on =
		state_filter & STATE_FILTER_NORMAL_RANK;

	/* filter on complete/incomplete/nonbinary */
	if (!(complete_bin_on && is_complete_bin)
		&& !(incomplete_bin_on && is_incomplete_bin)
		&& !(nonbin_on && !is_complete_bin && !is_incomplete_bin))
		return FALSE;

	/* filter on read */
	if (!(unread_on && !is_read) &&
		!(read_on && is_read) &&
		!(new_on && is_new && !is_read))
		return FALSE;

	/* filter on killfile, watch, normal rank */
	if (!(ignored_on && is_ignored) &&
		!(watch_on && is_watched) &&
		!(normal_rank_on && !is_ignored && !is_watched))
		return FALSE;

	/* filter on decoded/queued/idle */
	step_passed = FALSE;
	if ((state_filter&STATE_FILTER_SAVED)
		&& article_flag_on (article, STATE_DECODED))
		step_passed = TRUE;
	else if ((state_filter&STATE_FILTER_QUEUED)
		&& ((article->state&STATE_DECODE_QUEUED)
	       		|| (article->state&STATE_DOWNLOAD_FLAGGED)))
		step_passed = TRUE;
	else if ((state_filter&STATE_FILTER_IDLE)
		&& !article_flag_on(article, STATE_DECODED)
		&& !article_flag_on(article, STATE_DECODE_QUEUED)
		&& !article_flag_on(article, STATE_DOWNLOAD_FLAGGED))
		step_passed = TRUE;
	if (!step_passed)
		return FALSE;

	return TRUE;
}

static void
apply_filter_tests (void)
{
	register int i;
	register Article *article;

	/* clear the filter state */
	for (i=0; i!=article_headers->len; ++i)
	{
		article = ARTICLE(g_ptr_array_index(article_headers,i));
		article->self_passes_filter = FALSE;
		article->tree_passes_filter = FALSE;
	}

	/* apply the filter tests */
	for (i=0; i!=article_headers->len; ++i)
	{
		article = ARTICLE(g_ptr_array_index(article_headers,i));
		if (article_passes_filter (article))
		{
			article->self_passes_filter = TRUE;

			while (article && !article->tree_passes_filter) {
				article->tree_passes_filter = TRUE;
				article = article->parent;
			}
		}
	}
}

/***
****
****
****  POPULATING THE TREE:  THE DIRTY WORK
****
****
***/

static GtkCTreeNode*
add_article_to_ctree (GtkCTree      * tree,
                      GtkCTreeNode  * parent,
                      GtkCTreeNode  * sibling,
                      Article       * article,
                      StatusItem    * status,
                      gboolean        expanded)
{
        gchar childbuf[32];
	gchar line_count_buf[32];
	GnomePixmap * icon;
	GtkCTreeNode * node;
	gulong lcountsum;
	gchar * text[5];
	gchar * author;
	gchar * date;

	/* author */
	if (is_nonempty_string (article->author_real))
		author = g_strdup (article->author_real);
	else if (is_nonempty_string (article->author_addr)) {
		author = g_strdup (article->author_addr);
		if (author != NULL) {
			gchar * pch = strchr (author, '@');
			if (pch != NULL)
				*pch = '\0';
		}
	}
	else author = g_strdup (_("?"));

	/* date */
	date = get_date_display_string (article->date);

	/* number of children */
	*childbuf = '\0';
	if (article->unread_children)
		sprintf (childbuf, "%d", (int)article->unread_children);

	/* line count */
	lcountsum = article->linecount;
	if (article_flag_on (article, STATE_MULTIPART_ALL)) {
		GSList *l;
		for (l=article->threads; l; l=l->next)
			lcountsum += ARTICLE(l->data)->linecount;
	}
	commatize_ulong (lcountsum, line_count_buf);

	/* icon */
	icon = get_article_pixmap (article);
       
	if (status != NULL)
		status_item_emit_next_step (status);

	/* insert the node */
	text[0] = (gchar*)article->subject;
	text[1] = childbuf;
	text[2] = line_count_buf;
	text[3] = (gchar*)author;
	text[4] = date;
	pan_lock ();
	node = gtk_ctree_insert_node (
		tree, parent, sibling,
		(char**)text, 5,
		(icon ? icon->pixmap : NULL),
		(icon ? icon->mask : NULL),
		(icon ? icon->pixmap : NULL),
		(icon ? icon->mask : NULL),
		FALSE, expanded);
	gtk_ctree_node_set_row_data (tree, node, article);
	articlelist_set_node_style_nolock (node, article);
	pan_unlock ();

	/* update hash table */
	pthread_mutex_lock (&hash_mutex);
	g_hash_table_insert (messageid_to_node, (gpointer)article->message_id, node);
	pthread_mutex_unlock (&hash_mutex);

	g_free (author);
	g_free (date);
	return node;
}

static void
build_article_ctree_recursive (GtkCTree      * tree,
                               GtkCTreeNode  * parent,
                               GSList        * list,
                               StatusItem    * status,
                               gboolean        expanded)
{
	GSList * tmp;
	GSList * pos;
	GtkCTreeNode * sibling = NULL;

	/* sanity check */
	g_return_if_fail (parent!=NULL);

	/* this item gets inserted before the 'sibling' node,
	 * so we must add in reverse order */
	tmp = g_slist_reverse(g_slist_copy(list));
        for (pos=tmp; pos!=NULL; pos=pos->next)
        {
		Article* article = ARTICLE(pos->data);
		GtkCTreeNode* node = NULL;

		/* possibly don't show subtrees that don't measure up to filter */
		if (is_threaded && !article->tree_passes_filter)
			continue;

		/* possibly don't show 2...n of multipart nodes */
		if (hide_mpart_child_nodes
			&& (article->parent!=NULL)
			&& (article->parent->state&STATE_MULTIPART_ALL)
			&& (article->part!=0))
			continue;

		node = add_article_to_ctree (tree,
		                             parent,
		                             sibling,
		                             article,
		                             status,
		                             expanded);
		if (article->threads!=NULL)
			build_article_ctree_recursive (tree,
			                               node,
			                               article->threads,
			                               status,
			                               TRUE);

		sibling = node;
        }

	/* cleanup */
	g_slist_free (tmp);
}



typedef struct
{
	const Group *group;
	Article **article_buf;
	int article_qty;
	StatusItem *item;
	int index;
	GtkCTreeNode *root;
}
BuildArticleListStruct;

static void
add_nodes (gpointer data)
{
	BuildArticleListStruct *build = (BuildArticleListStruct*) data;
	GtkCTree * tree = GTK_CTREE (Pan.article_ctree);
	GtkCTreeNode * sibling = NULL;
	const gchar* subject_filter_str = article_toolbar_get_filter();
	gint i;

	for (i=build->article_qty-1; i>=0; --i)
	{
		Article* article = build->article_buf[i];
		GtkCTreeNode * node = NULL;
		gboolean has_parent;
		gboolean filter_match;

		/* make sure we pass the state filter */
		if (is_threaded && !article->tree_passes_filter)
			continue;
		if (!is_threaded && !article->self_passes_filter)
			continue;

		/* make sure we pass the subject filter */
		filter_match =
			subject_filter_str==NULL ||
			!fnmatch(subject_filter_str,article->subject,PAN_CASEFOLD);
		if (!filter_match)
			continue;

		has_parent = article->parent != NULL;

		if ((!is_threaded && article->part<2) || (is_threaded && !has_parent))
		{
			node = add_article_to_ctree (tree,
			                             build->root,
			                             sibling,
			                             article,
			                             build->item,
			                             FALSE);
			sibling = node;
		}

		/* if threads are turned on, and there are children of this thread... */
		if (is_threaded && node!=NULL && article->threads!=NULL)
		{
			build_article_ctree_recursive (tree,
			                               node,
			                               article->threads,
			                               build->item,
			                               TRUE);
		}
	}
}

/* fires next_step article_qty times */
static void
articlelist_repopulate (const Group* group,
                        Article** article_buf,
                       	int article_qty,
                        StatusItem *item,
                        GPtrArray * sel_articles)
{
	gint i;
	GtkCTree * tree = GTK_CTREE (Pan.article_ctree);
	char * text[5];
	BuildArticleListStruct build;

	build.group = group;
	build.article_buf = article_buf;
	build.article_qty = article_qty;
	build.item = item;
	build.index = 0;
	build.root = NULL;

	status_item_emit_status (item, _("Updating article display"));

	/* clear out old */
	clear_hash_table ();
	pan_lock();
	gtk_clist_freeze (GTK_CLIST(tree));
	gtk_clist_clear (GTK_CLIST(tree));
	
	/* add root node */
	text[0] = (char*) group_get_readable_name (group);
	text[1] = text[2] = text[3] = text[4] = NULL;
	build.root = gtk_ctree_insert_node (
		tree, NULL, NULL,
		text, 5, articlelist_closed_pixmap->pixmap,
		articlelist_closed_pixmap->mask,
		articlelist_open_pixmap->pixmap,
		articlelist_open_pixmap->mask, FALSE, FALSE );
	gtk_ctree_node_set_row_data (tree, build.root, NULL);
	articlelist_set_node_style_nolock (build.root, NULL);
	pan_unlock ();

	g_hash_table_freeze (messageid_to_node);
	add_nodes (&build);
	g_hash_table_thaw (messageid_to_node);

	pan_lock ();
	gtk_ctree_expand (tree, build.root);
	gtk_clist_thaw (GTK_CLIST(tree));
	gtk_widget_queue_resize (Pan.article_ctree);
	pan_unlock ();

	/* set the selection */
	if (1)
	{
		GPtrArray * tmp = g_ptr_array_new ();

		/* get an array of nodes... */
		for (i=0; sel_articles!=NULL && i<sel_articles->len; ++i) {
			Article * a = ARTICLE(g_ptr_array_index(sel_articles,i));
			GtkCTreeNode * node = articlelist_get_node_from_message_id (a->message_id);
			if (node != NULL)
				g_ptr_array_add (tmp, node);
		}

		/* select the nodes */
		articlelist_set_selected_nodes ((GtkCTreeNode**)tmp->pdata, tmp->len);

		/* cleanup */
		g_ptr_array_free (tmp, TRUE);
	}
}

/***
****
****
****  POPULATING THE TREE:  THE CONTROLLERS
****
****
***/

void
articlelist_refresh (StatusItem* item)
{
	GPtrArray * sel;

	pthread_mutex_lock (&article_ctree_lock);

        /* tell the StatusItem what we're doing */
	status_item_emit_init_steps (item, article_headers->len*2);

	/* update articles */
	sel = articlelist_get_selected_articles ();
	articlelist_repopulate (my_group,
				(Article**)article_headers->pdata,
				article_headers->len,
				item,
				sel);

	pthread_mutex_unlock (&article_ctree_lock);
	
	g_ptr_array_free (sel, TRUE);
}

static void
articlelist_set_contents (Group        * group,
			  StatusItem   * status)
{
	int i;
	GPtrArray * a;

	g_return_if_fail (group != NULL);

	pthread_mutex_lock (&article_ctree_lock);

	/* get sorting information */
	i = group->sort_style;
	if (!i || abs(i)>ARTICLE_SORT_TYPES_QTY)
		i = -ARTICLE_SORT_DATE;
	articlelist_set_sort_type_nosort (i, TRUE);

	/* let the user know what the heck we're doing */
	if (status != NULL) {
		int steps = 0;
		if (my_group!=NULL)
			steps += my_group->article_qty; /* for saving */
		if (group!=NULL)
			steps += group->article_qty * 2; /* thread, node */
		if (group!=NULL && !group_articles_reffed(group))
			steps += group->article_qty; /* load */
		status_item_emit_init_steps (status, steps);
	}

	/* we ref the new articles before unreffing the old, in case this
	   is a forced reload where new==old.  In this instance putting the
	   ref first keeps us from having to free, then reload the articles */
	group_ref_articles (group, status);
	clear_article_buffer (status);
	my_group = group;
	group_thread_if_needed (my_group, status);
	a = group_get_articles (my_group);
	pan_g_ptr_array_assign (article_headers, a->pdata, a->len);
	pan_callback_call (articlelist_group_changed,
	                   Pan.article_ctree,
	                   my_group);

	/* filter */
	apply_filter_tests ();

	/* sort */
	sort_articles ((Article**)article_headers->pdata,
	               article_headers->len,
		       abs(sort_type),
		       sort_type>0);

	/* add to tree */
	status_item_emit_status (status, _("Displaying article list"));
	articlelist_repopulate (group,
	                        (Article**)article_headers->pdata,
	                        article_headers->len,
	                        status,
	                        NULL);

	pthread_mutex_unlock (&article_ctree_lock);
}

/***
****
****
****  SETTING THE CURRENT GROUP
****
****
***/

static gchar*
set_current_group_describe (const StatusItem* item)
{
	return g_strdup (_("Loading Group Headers"));
}

static void
articlelist_set_group_maybe (Group * g, gboolean force_reload)
{
	StatusItem * status;

	/* if nothing to do, then quit */
	if (g==my_group && !force_reload)
		return;

	/* set up a status item so that the users will see what's happening.. */
	status = STATUS_ITEM(status_item_new(set_current_group_describe));
	status_item_set_active (status, TRUE);

	if (g!=NULL && g->article_qty!=0) /* group & cached headers exist */
	{
		Group * prev_g = my_group;
		articlelist_set_contents (g, status);
		if (prev_g!=my_group && fetch_new_on_group_load && !group_is_folder(g))
			queue_add (TASK(task_headers_new (g, HEADERS_NEW, 1)));
	}
	else /* probably have to hit the news server for articles */
	{
		clear_article_buffer (NULL);
		my_group = g;
		pan_callback_call (articlelist_group_changed, Pan.article_ctree, g);

		if (my_group!=NULL)
		{
			group_ref_articles (my_group, status);

		       	if (g->article_qty==0 && !group_is_folder(g))
				dialog_download_headers (g);
		}
	}

	/* clean up status item */
	status_item_set_active (status, FALSE);
	pan_object_unref(PAN_OBJECT(status));
}
void
articlelist_set_group_thread (Group * group)
{
	pthread_t thread;
	pthread_create (&thread, NULL, (void*)articlelist_set_group, group);
	pthread_detach (thread);
}
void
articlelist_set_group (Group * group)
{
	articlelist_set_group_maybe (group, FALSE);
}
void
articlelist_reload (void)
{
	articlelist_set_group_maybe (my_group, TRUE);
}

/***
****
****
****  UPDATING EXISTING NODES: THE DIRTY WORK
****
****
***/

static void 
articlelist_update_node_fast_nolock (GtkCTreeNode   * node,
                                     const Article  * article)
{
        char childbuf[16];
	GnomePixmap * icon = get_article_pixmap (article);

	*childbuf = '\0';
	if (article->unread_children)
		sprintf (childbuf, "%d", article->unread_children);

	gtk_ctree_set_node_info (
		GTK_CTREE(Pan.article_ctree), node,
		article->subject, 5,
		(icon ? icon->pixmap : NULL),
		(icon ? icon->mask : NULL),
		(icon ? icon->pixmap : NULL),
		(icon ? icon->mask : NULL),
		GTK_CTREE_ROW(node)->is_leaf,
		GTK_CTREE_ROW(node)->expanded );
	gtk_ctree_node_set_text (GTK_CTREE(Pan.article_ctree), node,
				 1, childbuf);

	articlelist_set_node_style_nolock (node, article);
}
static GtkCTreeNode*
articlelist_get_node_from_message_id (const char* message_id)
{
	GtkCTreeNode * retval = NULL;

	if (is_nonempty_string(message_id))
	{
		pthread_mutex_lock (&hash_mutex);
		retval = g_hash_table_lookup (messageid_to_node, message_id);
		pthread_mutex_unlock (&hash_mutex);
	}

	return retval;
}
static void
articlelist_update_node_impl (const Article* article, gboolean need_lock)
{
	GtkCTreeNode* node =
		articlelist_get_node_from_message_id (article->message_id);

        if (node != NULL)
	{
		/* update the article state */
		Article* real_article =
			articlelist_get_article_from_node (node);
		real_article->state = article->state;

		/* repaint */
		if (need_lock)
			pan_lock ();
                articlelist_update_node_fast_nolock (node, real_article);
		if (need_lock)
			pan_unlock ();
	}
}
void
articlelist_update_node (const Article* article)
{
	if (article->group == my_group)
		articlelist_update_node_impl (article, TRUE);
}

/***
****
****
****  MANAGING UNREAD CHILDREN COUNTS
****
****
***/


static void
articlelist_unread_inc_nolock (GtkCTreeNode* node, int inc)
{
	GtkCTreeNode * child = node;

	/* update the node & parents */
	while (node != NULL)
	{
		Article * a = articlelist_get_article_from_node (node);
		if (a == NULL)
			break;

		/* all the parents have one less/more unread child */
		if (child != node)
			a->unread_children = MAX(a->unread_children+inc, 0);

		articlelist_update_node_fast_nolock (node, a);

		node = GTK_CTREE_ROW(node)->parent;
	}
}
void
articlelist_article_read_changed (const Article * article)
{
	GtkCTreeNode* node = NULL;

	/* sanity checks */
	g_return_if_fail (article!=NULL);

	/* find the matching node */
	node = articlelist_get_node_from_message_id (article->message_id);
	if (node != NULL)
	{
		int inc = 0;
		Article* a = NULL;

		/* update the read flag of our article */
		a = articlelist_get_article_from_node (node);
		g_return_if_fail (a != NULL);
		if (a != article)
			article_set_read (a, article_is_read (article));

		/* update unread child count of our article and its ancestors */
		inc = article_is_read (article) ? -1 : 1;
		pan_lock ();
		articlelist_unread_inc_nolock (node,inc);
		pan_unlock ();
	}
}

/***
****
****
****  DELETING ARTICLES
****
****
***/

static void
articlelist_delete_articles (GPtrArray * articles)
{
	GtkCList * clist = GTK_CLIST(Pan.article_ctree);

	if (articles->len)
	{
		/* freeze the ui so that we can batch up refreshes. */
		pan_lock();
		gtk_ctree_unselect_recursive (
			GTK_CTREE(clist),
			gtk_ctree_node_nth(GTK_CTREE(clist), 0));
		pan_unlock();

		/* delete the thread */
		group_remove_articles (my_group, articles);
	}
}

void
articlelist_all_delete (void)
{
	articlelist_delete_articles (article_headers);
}
void
articlelist_selected_delete_thread (void)
{
	GPtrArray * sel = articlelist_get_selected_articles ();
	GPtrArray * all = articlelist_get_unique_threads (sel, GET_SUBTHREAD);

	/* delete the articles */
	articlelist_delete_articles (all);

	/* cleanup */
	g_ptr_array_free (all, TRUE);
	g_ptr_array_free (sel, TRUE);
}

void
articlelist_selected_delete (void)
{
	GPtrArray * articles = articlelist_get_selected_articles ();
	int i, len;

	/* if it's a multipart, delete the children too */
	for (i=0, len=articles->len; i!=len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(articles,i));
		if (a->part==1 && a->parts>1) {
			GSList * l;
			for (l=a->threads; l!=NULL; l=l->next)
				if (ARTICLE(l->data)->part > 1)
					g_ptr_array_add (articles, l->data);
		}
	}

	/* delete the articles */
	articlelist_delete_articles (articles);

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
}

/***
****
****
****  SELECTION / KEYPRESS / BUTTONPRESS HANDLING
****
****
***/

/**
 * The following buttonpress, keypress, and selection callbacks are all here
 * to interact with each other to do one thing: force this damn GtkCTree to
 * have the UI behavior that we want:
 * 
 * (1) Never activate an article during a group selection
 * 
 * (2) If only a single article is selected by single mouse click or
 *     by up/down arrow, and the 'single click loads' prefs is on,
 *     or if it's selected by a double click, then load the article.
 *
 * The problem with (1) is that the GtkCTree fires a selection event for each
 * row selected, rather than once for the whole range, so there's no way to
 * tell if the first selection event is a single or the beginning of a range.
 * The workaround is to use an idle func that gets invoked after all the
 * selections are done.  It asks the tree what's selected and uses that in
 * the "article selection changed" pan callback.  (2) is handled by the
 * button_click_count variable, which keeps click information from the
 * kepress/buttonpress callbacks and uses it in the selection callback.
 */
static gint button_click_count = -1;
static gint mb = -1;

static gint
articlelist_selection_changed_cb (gpointer call_object,
                                  gpointer call_arg,
                                  gpointer user_data)
{
	GPtrArray * articles = articlelist_get_selected_articles ();
	Article * single_article = NULL;

	/* load the specified article */
	if (articles!=NULL && articles->len==1)
		single_article = ARTICLE(g_ptr_array_index(articles,0));
        if ((single_article!=NULL)
		&& ((button_click_count==2) /* double click */
			|| (button_click_count==1 /*&& Pan.viewmode==GUI_PANED*/
				&& (  (mb==2 && articlelist_mb2_action==CLICK_ACTION_LOAD)
				    ||(mb==1 && articlelist_mb1_action==CLICK_ACTION_LOAD)))))
	{
		gui_page_set (MESSAGE_PAGE, Pan.text);
		queue_add (TASK(task_body_new_defaults(single_article)));
	}

	button_click_count = -1;
	mb = -1;

	return 0;
}

static void
articlelist_key_press (GtkWidget      * widget,
                       GdkEventKey    * event,
                       gpointer         data)
{
	switch (event->keyval)
	{
		case GDK_Up: case GDK_Down:
		case GDK_Left: case GDK_Right:
		case GDK_Page_Up: case GDK_Page_Down:
			button_click_count = 1;
			break;
		default:
			button_click_count = -1;
	}
}

static void
articlelist_button_press (GtkWidget* widget, GdkEventButton* bevent)
{
	switch (bevent->button)
	{
		case 1:
		case 2:
			mb = bevent->button;
			if (bevent->type == GDK_2BUTTON_PRESS)
				button_click_count = 2;
			else
				button_click_count = 1;
			break;
		case 3:
			articlelist_menu_popup (bevent);
			break;
	}
}

static gboolean select_callback_pending = FALSE;
 
static gint
tree_select_row_idle (gpointer data)
{
	GPtrArray * articles = articlelist_get_selected_articles ();

	/* let everyone know that the selection has changed */
	pan_callback_call (articlelist_selection_changed,
			   Pan.article_ctree,
			   articles);

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	select_callback_pending = FALSE;
	return FALSE;
}
static void
tree_select_row_cb (GtkCTree     *tree,
		    GtkCTreeNode *node,
		    gint          column,
		    gpointer      data)
{
	if (!select_callback_pending)
	{
		select_callback_pending = TRUE;
		gtk_idle_add (tree_select_row_idle, NULL);
	}
}

/***
****
****
****  ccc
****
****
***/

static void
articlelist_click_column (GtkCList* clist, int n)
{
	int type;
	switch (n)
	{
		case 0:
			type = ARTICLE_SORT_SUBJECT;
			break;
		case 1:
			type = ARTICLE_SORT_UNREAD_CHILDREN;
			break;
		case 2:
			type = ARTICLE_SORT_LINES;
			break;
		case 3:
			type = ARTICLE_SORT_AUTHOR;
			break;
		case 4:
			type = ARTICLE_SORT_DATE;
			break;
		default:
			pan_warn_if_reached();
			type = -ARTICLE_SORT_DATE;
			break;
	}

	if (type == sort_type)
		type = -type;

	articlelist_set_sort_type (type);
}

static void
article_ctree_destroy_cb (void)
{
	gtk_widget_destroy (GTK_WIDGET(articlelist_closed_pixmap));
	gtk_widget_destroy (GTK_WIDGET(articlelist_open_pixmap));
	gtk_widget_destroy (GTK_WIDGET(article_read_pixmap));
	gtk_widget_destroy (GTK_WIDGET(binary_decoded_pixmap));
	gtk_widget_destroy (GTK_WIDGET(decode_queue_pixmap));
	gtk_widget_destroy (GTK_WIDGET(error_pixmap));
	gtk_widget_destroy (GTK_WIDGET(flagged_for_download_pixmap));
	gtk_widget_destroy (GTK_WIDGET(mp_complete_pixmap));
	gtk_widget_destroy (GTK_WIDGET(mp_incomplete_pixmap));

	pan_callback_free (articlelist_state_filter_changed);
	pan_callback_free (articlelist_group_changed);
	pan_callback_free (articlelist_sort_changed);
	pan_callback_free (articlelist_thread_changed);
	pan_callback_free (articlelist_selection_changed);
}

static int
articlelist_group_changed_cb (gpointer call_obj,
			      gpointer call_arg,
			      gpointer user_data)
{
	const Group* group = (const Group*) call_arg;

	/* the group has changes, so the group state filter has too */
	pan_callback_call (
		articlelist_state_filter_changed,
		Pan.article_ctree,
		GUINT_TO_POINTER(group ? group->state_filter : 0));

	return 0;
}

static void
tree_toggle_focus_row_cb (GtkCList   * clist,
                          gpointer     user_data)
{
	/* This is just a hack to allow space-reading when the article ctree
	   has focus.  The space key is used in clists to toggle the row
	   focus, so if we don't stop the signal here ctree calls
	   resync_selection, which reselects the focused node, and makes
	   space reading loop back on the same article each time. */
	gtk_signal_emit_stop_by_name (GTK_OBJECT(clist), "toggle_focus_row");
}


/*--------------------------------------------------------------------
 * generate the listing of articles, for the "Articles" tab
 *--------------------------------------------------------------------*/
GtkWidget*
create_articlelist_ctree (void)
{
	int i;
	GtkCList* list;
	GtkCTree* tree;
	GtkWidget *s_window;
	GtkWidget *vbox;
	const char* titles[5];

	article_headers = g_ptr_array_new ();

	/* callbacks */
	articlelist_group_changed = pan_callback_new ();
	articlelist_state_filter_changed = pan_callback_new ();
	articlelist_sort_changed = pan_callback_new ();
	articlelist_thread_changed = pan_callback_new ();
	articlelist_selection_changed = pan_callback_new ();

	pan_callback_add (articlelist_selection_changed,
	                  articlelist_selection_changed_cb,
	                  NULL);

	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GNOME_PAD_SMALL);

	/* filter */
	gtk_box_pack_start (GTK_BOX(vbox), article_toolbar_new(), FALSE, FALSE, 0);

	/* get the titles */
	for (i=0; i<5; ++i)
		titles[i] = column_to_title (i);

	/* load the pixmaps */
	flagged_for_download_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d (blue_flag_xpm);
	articlelist_closed_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d (book_closed_xpm);
	articlelist_open_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d (book_open_xpm);
	article_read_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d (mini_page_xpm);
	decode_queue_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d (bluecheck_xpm);
	error_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d (x_xpm);
	binary_decoded_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d (binary_xpm);
	mp_complete_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d (binary_complete_xpm);
	mp_incomplete_pixmap = (GnomePixmap*)
		gnome_pixmap_new_from_xpm_d (binary_incomplete_xpm);

	if (!gdk_color_alloc (cmap, &unread_color))
		g_error ("couldn't allocate unread color");
	if (!gdk_color_alloc (cmap, &new_color))
		g_error ("couldn't allocate new color");
	if (!gdk_color_alloc (cmap, &read_color))
		g_error ("couldn't allocate read color");
	if (!gdk_color_alloc (cmap, &killfile_color))
		g_error ("couldn't allocate ignored-thread color");
	if (!gdk_color_alloc (cmap, &watched_color))
		g_error ("couldn't allocate watched-thread color");

	clear_hash_table ();

	/* create the widget */
	Pan.article_ctree = gtk_ctree_new_with_titles (5, 0, (gchar**)titles);
	widget_set_font (Pan.article_ctree, articlelist_font);
	tree = GTK_CTREE(Pan.article_ctree);
	list = GTK_CLIST(Pan.article_ctree);
	list->button_actions[1] = list->button_actions[0];
	gtk_signal_connect (GTK_OBJECT(tree), "tree-select-row", tree_select_row_cb, NULL);
	gtk_signal_connect (GTK_OBJECT(tree), "tree-unselect-row", tree_select_row_cb, NULL);
	gtk_signal_connect (GTK_OBJECT(tree), "toggle_focus_row", tree_toggle_focus_row_cb, NULL);

	pthread_mutex_init (&hash_mutex, NULL);


	/* wrap it in a scrolled window */
	s_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (
		GTK_SCROLLED_WINDOW(s_window),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER(s_window), Pan.article_ctree);

	/* ui settings */
	gtk_ctree_set_line_style (tree, GTK_CTREE_LINES_DOTTED);
	gtk_clist_set_selection_mode (list, GTK_SELECTION_EXTENDED);
	gtk_clist_set_column_width (list, 0, 220);
	gtk_clist_set_column_width (list, 1, 40);
	gtk_clist_set_column_width (list, 2, 40);
	gtk_clist_set_column_width (list, 3, 120);
	gtk_clist_set_column_width (list, 4, 40);
	gtk_clist_set_column_justification (list, 1, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification (list, 2, GTK_JUSTIFY_RIGHT);

	/* create the right click popup menu */
	article_ctree_menu = gnome_popup_menu_new (articlelist_menu);

	/* connect signals */
	gtk_signal_connect (
		GTK_OBJECT(tree), "button_press_event",
		GTK_SIGNAL_FUNC(articlelist_button_press), NULL);
	gtk_signal_connect (
		GTK_OBJECT(tree), "key_press_event",
		GTK_SIGNAL_FUNC(articlelist_key_press), NULL);
	gtk_signal_connect (
		GTK_OBJECT(tree), "click_column",
		GTK_SIGNAL_FUNC(articlelist_click_column), NULL);
	gtk_signal_connect (
		GTK_OBJECT(tree), "destroy",
		GTK_SIGNAL_FUNC(article_ctree_destroy_cb), NULL);

	pan_callback_add (articlelist_group_changed,
			  articlelist_group_changed_cb,
			  NULL);

	pan_callback_add (group_get_articles_removed_callback(),
			  group_articles_removed_cb,
			  NULL);

	pan_callback_add (group_get_articles_added_callback(),
			  group_articles_added_cb,
			  NULL);

	gtk_box_pack_start (GTK_BOX(vbox), s_window, TRUE, TRUE, 0);

	return vbox;
}
void
articlelist_reset_colors (void)
{
	if (unread_style != NULL)
	{
		gtk_style_unref (killfile_style);
		gtk_style_unref (watched_style);
		gtk_style_unref (unread_style);
		gtk_style_unref (read_style);
		gtk_style_unref (new_style);

		killfile_style = NULL;
		watched_style = NULL;
		unread_style = NULL;
		read_style = NULL;
		new_style = NULL;
	}

	articlelist_reload ();
}


/**
***
**/

void
articlelist_expand_all_threads (void)
{
	GtkCTree * tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode * node;
	GtkCTreeNode * sel;

	pan_lock ();
	sel = articlelist_get_selected_node ();
	node = gtk_ctree_node_nth (tree, 0);
	if (node != NULL) {
		gtk_ctree_expand_recursive (tree, node);
		if (sel != NULL)
			articlelist_set_selected_nodes (&sel, 1);
	}
	pan_unlock ();
}

void
articlelist_collapse_all_threads (void)
{
	GtkCTree * tree = GTK_CTREE(Pan.article_ctree);
	GtkCTreeNode * root;

	/* get the root of the tree */
	pan_lock ();
	root = gtk_ctree_node_nth (tree, 0);
	pan_unlock ();

	/* if the tree's not empty, collapse the threads */
	if (root != NULL)
	{
		GtkCTreeNode * sel; 

		/* find the parent node that will be selected when we're done */
		sel = articlelist_get_selected_node ();
		while (sel!=NULL && GTK_CTREE_ROW(sel)->parent!=root)
			sel = GTK_CTREE_ROW(sel)->parent;

		/* collapse the threads */
		pan_lock ();
		gtk_ctree_collapse_recursive (tree, root);
		gtk_ctree_expand (tree, root);
		pan_unlock ();

		/* select the topmost parent of the old selection */
		if (sel != NULL)
			articlelist_set_selected_nodes (&sel, 1);
	}
}

/**
***
***  ARTICLE FILTERING
***
**/

static gchar*
filter_selected_describe (const StatusItem* item)
{
	return g_strdup (_("Filtering Articles"));
}
static void
filter_changed_thread (void *data)
{
	StatusItem *item = NULL;

	if (!my_group) /* articlelist is idle */
		return;

	/* create a new status item to get sort/thread messages */
	item = STATUS_ITEM(status_item_new(filter_selected_describe));
	pan_object_ref (PAN_OBJECT(item));
	pan_object_sink (PAN_OBJECT(item));
	status_item_set_active (item, TRUE);

	/* filter */
	apply_filter_tests ();

	/* repopulate */
	articlelist_refresh (item);

	/* clean out the status item */
	status_item_set_active (item, FALSE);
	pan_object_unref(PAN_OBJECT(item));
}

void
articlelist_poke_state_filter (guint flag, gboolean on)
{
	pthread_t thread;

	/* someone toggled a button, or something,
	   even though there's no group */
	if (!my_group)
		return;

	/* update the state if necessary,
	   return otherwise */
	if (on) {
		if (my_group->state_filter & flag)
			return;
		group_set_state_filter (my_group, my_group->state_filter|flag);
	} else {
		if (!(my_group->state_filter & flag))
			return;
		group_set_state_filter (my_group, my_group->state_filter&=~flag);
	}

	/* notify listeners of the change */
	pan_callback_call (
		articlelist_state_filter_changed,
		Pan.article_ctree,
		GUINT_TO_POINTER((guint)my_group->state_filter));

	pthread_create (&thread, NULL, (void*)filter_changed_thread, NULL);
	pthread_detach (thread);
}



/**
***
***  ARTICLE THREADING
***
**/

static gchar*
thread_selected_describe (const StatusItem* item)
{
	return g_strdup (_("Threading Articles"));
}

static void
articlelist_set_threaded_thread (void* data)
{
	const gboolean threaded_on = GPOINTER_TO_INT(data);
	StatusItem *item = NULL;

	/* create a new status item to get thread messages */
	item = STATUS_ITEM(status_item_new(thread_selected_describe));
	pan_object_ref (PAN_OBJECT(item));
	pan_object_sink (PAN_OBJECT(item));
	status_item_set_active (item, TRUE);

	/* update the articlelist */
	is_threaded = threaded_on;
	pan_callback_call (articlelist_thread_changed,
			   Pan.article_ctree, GINT_TO_POINTER(threaded_on));
	if (my_group!=NULL)
		articlelist_refresh (item);

        /* clean out the status item */
	status_item_set_active (item, FALSE);
	pan_object_unref(PAN_OBJECT(item));
}

void
articlelist_set_threaded (gboolean threaded_on)
{
	if (threaded_on != is_threaded)
	{
		pthread_t thread;
		pthread_create (&thread,
				NULL, (void*)articlelist_set_threaded_thread,
				GINT_TO_POINTER(threaded_on));
		pthread_detach (thread);
	}
}

/**
***
***  POPUP MENU 
***
**/

static void
articlelist_menu_popup (GdkEventButton* bevent)
{
	const Article *article = articlelist_get_selected_article();
	const gboolean have_article = article!=NULL;
	const gboolean group_has_articles = article_headers->len!=0;
	const gboolean is_folder = my_group!=NULL && group_is_folder (my_group);

	int i = 0;
	GList* l = NULL;
	pan_lock();

	l=GTK_MENU_SHELL(article_ctree_menu)->children; /* open */
	gtk_widget_set_sensitive (GTK_WIDGET(l->data), have_article); 

	l=l->next; /* separator */
	l=l->next; /* respond */
	gtk_widget_set_sensitive (GTK_WIDGET(l->data), have_article);
	l=l->next; /* binary attachments */
	gtk_widget_set_sensitive (GTK_WIDGET(l->data), have_article);
	l=l->next; /* search offsite */
	gtk_widget_set_sensitive (GTK_WIDGET(l->data), have_article);
	l=l->next; /* watch/ignore */
	gtk_widget_set_sensitive (GTK_WIDGET(l->data), have_article);

	l=l->next; /* separator */
	l=l->next; /* mark as read */
	gtk_widget_set_sensitive (GTK_WIDGET(l->data), TRUE);
	l=l->next; /* mark as unread */
	gtk_widget_set_sensitive (GTK_WIDGET(l->data), TRUE);
	l=l->next; /* download bodies */
	gtk_widget_set_sensitive (GTK_WIDGET(l->data), TRUE);
	l=l->next; /* delete */
	gtk_widget_set_sensitive (GTK_WIDGET(l->data), TRUE);

	l=l->next; /* separator */
	l=l->next; /* select all articles */
	gtk_widget_set_sensitive (GTK_WIDGET(l->data), TRUE);

	/* TRACK SUBMENU */
	i = -1;
	++i; /* i is ignore */
	gtk_widget_set_sensitive (filter_menu[i].widget, have_article);
	++i; /* i is watch */
	gtk_widget_set_sensitive (filter_menu[i].widget, have_article);
	++i; /* i is bozoize */
	gtk_widget_set_sensitive (filter_menu[i].widget, have_article);

	/* DELETE SUBMENU */
	i = -1;
	++i; /* i is delete selected article */
	gtk_widget_set_sensitive (delete_menu[i].widget, have_article);
	++i; /* i is delete selected articles + replies */
	gtk_widget_set_sensitive (delete_menu[i].widget, have_article);
	++i; /* delete all */
	gtk_widget_set_sensitive (delete_menu[i].widget, group_has_articles);
	++i; /* separator */
	++i; /* i is delete decoded binary attachment */
	gtk_widget_set_sensitive (delete_menu[i].widget, have_article);

	/* MARK READ SUBMENU */
	i = -1;
	++i; /* i is mark selected read */
	gtk_widget_set_sensitive (mark_read_menu[i].widget, have_article);
	++i; /* i is mark selected + replies read */
	gtk_widget_set_sensitive (mark_read_menu[i].widget, have_article);
	++i; /* i is mark selected threads */
	gtk_widget_set_sensitive (mark_read_menu[i].widget, have_article);
	++i; /* i is mark group */
	gtk_widget_set_sensitive (mark_read_menu[i].widget, group_has_articles);

	/* MARK UNREAD SUBMENU */
	i = -1;
	++i; /* i is mark selected */
	gtk_widget_set_sensitive (mark_unread_menu[i].widget, have_article);
	++i; /* i is mark selected + replies */
	gtk_widget_set_sensitive (mark_unread_menu[i].widget, have_article);
	++i; /* i is mark selected threads */
	gtk_widget_set_sensitive (mark_unread_menu[i].widget, have_article);
	++i; /* i is mark group */
	gtk_widget_set_sensitive (mark_unread_menu[i].widget, group_has_articles);

	/* DOWNLOAD BODIES SUBMENU */
	i = -1;
	++i; /* i is mark selected */
	gtk_widget_set_sensitive (bodies_menu[i].widget, have_article);
	++i; /* i is mark selected + replies */
	gtk_widget_set_sensitive (bodies_menu[i].widget, have_article);
	++i; /* i is mark group */
	gtk_widget_set_sensitive (bodies_menu[i].widget, group_has_articles);

	/* FOLDER SUBMENU */
	i = -1;
	++i; /* i is edit */
	gtk_widget_set_sensitive (folder_menu[i].widget, is_folder);

	/* show the popup */
	gtk_menu_popup (GTK_MENU (article_ctree_menu),
			NULL, NULL, NULL, NULL,
			bevent->button, bevent->time);
	pan_unlock();
}
