/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "bozo.h"
#include "gnksa.h"
#include "prefs.h"
#include "util.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

static GPtrArray * _bozos_unique = NULL;

static GPtrArray * _bozos_sorted_by_name = NULL;

static GPtrArray * _bozos_sorted_by_address = NULL;

static GMemChunk * _bozo_chunk = NULL;

static GStringChunk * _bozo_string_chunk = NULL;

static gboolean dirty = FALSE;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE
************/

/*****
******
*****/

/*****
******
*****/

static gchar*
get_bozo_filename (void)
{
	return g_strdup_printf ("%s/bozo.txt", data_dir);
}

/*****
******
*****/

static int
compare_pBozo_to_ppBozo_by_name (const void * a, const void * b)
{
	return pan_strcmp (((const Bozo*)a)->real,
	                   (*(const Bozo**)b)->real);
}

static int
compare_pBozo_to_ppBozo_by_address (const void * a, const void * b)
{
	return pan_strcmp (((const Bozo*)a)->address,
	                   (*(const Bozo**)b)->address);
}

/*****
******
*****/

static void
bozo_add_impl (Bozo * bozo)
{
	int index;
	gboolean exact_match;
	gboolean added = FALSE;

	g_return_if_fail (bozo != NULL);
	g_return_if_fail (is_nonempty_string(bozo->real) || is_nonempty_string(bozo->address));

	/* add by name */
	if (is_nonempty_string(bozo->real))
	{
		exact_match = FALSE;
		index = lower_bound (bozo,
				     _bozos_sorted_by_name->pdata,
				     _bozos_sorted_by_name->len,
				     sizeof(gpointer),
				     compare_pBozo_to_ppBozo_by_name,
				     &exact_match);
		if (!exact_match) {
			pan_g_ptr_array_insert (_bozos_sorted_by_name, bozo, index);
			added = dirty = TRUE;
		}
	}

	/* add by address */
	if (is_nonempty_string(bozo->address))
	{
		exact_match = FALSE;
		index = lower_bound (bozo,
				     _bozos_sorted_by_address->pdata,
				     _bozos_sorted_by_address->len,
				     sizeof(gpointer),
				     compare_pBozo_to_ppBozo_by_address,
				     &exact_match);
		if (!exact_match) {
			pan_g_ptr_array_insert (_bozos_sorted_by_address, bozo, index);
			added = dirty = TRUE;
		}
	}

	/* keep track of our bozos */
	if (added)
		g_ptr_array_add (_bozos_unique, bozo);
}

void
bozo_add (const gchar * address, const gchar * realname)
{
	Bozo * bozo;
	gchar * real_tmp = g_strdup (realname);

	g_return_if_fail (is_nonempty_string(address) || is_nonempty_string(realname));

	/* massage the real name */
	if (is_nonempty_string(real_tmp))
		gnksa_strip_realname (real_tmp);

	/* add the bozo */
	bozo = g_chunk_new (Bozo, _bozo_chunk);
	bozo->real = is_nonempty_string (real_tmp)
		? g_string_chunk_insert (_bozo_string_chunk, real_tmp)
		: "";
	bozo->address = is_nonempty_string (address)
		? g_string_chunk_insert (_bozo_string_chunk, address)
		: "";
	bozo_add_impl (bozo);

	/* cleanup */
	g_free (real_tmp);
}

/*****
******
*****/

static void
bozo_free (void)
{
	if (_bozos_unique != NULL) {
		g_ptr_array_free (_bozos_unique, TRUE);
		_bozos_unique = NULL;
	}

	if (_bozos_sorted_by_name != NULL) {
		g_ptr_array_free (_bozos_sorted_by_name, TRUE);
		_bozos_sorted_by_name = NULL;
	}

	if (_bozos_sorted_by_address != NULL) {
		g_ptr_array_free (_bozos_sorted_by_address, TRUE);
		_bozos_sorted_by_address = NULL;
	}

	if (_bozo_string_chunk != NULL) {
		g_string_chunk_free (_bozo_string_chunk);
		_bozo_string_chunk = NULL;
	}

	if (_bozo_chunk != NULL) {
		g_mem_chunk_destroy (_bozo_chunk);
		_bozo_chunk = NULL;
	}
}

static void
bozo_init_prep (void)
{
	g_return_if_fail (_bozo_chunk==NULL);
	g_return_if_fail (_bozo_string_chunk==NULL);
	g_return_if_fail (_bozos_sorted_by_name==NULL);
	g_return_if_fail (_bozos_sorted_by_address==NULL);
	g_return_if_fail (_bozos_unique==NULL);

	_bozo_chunk = g_mem_chunk_create (Bozo, 64, G_ALLOC_ONLY);
	_bozo_string_chunk = g_string_chunk_new (1024);
	_bozos_sorted_by_name = g_ptr_array_new ();
	_bozos_sorted_by_address = g_ptr_array_new ();
	_bozos_unique = g_ptr_array_new ();
}

void
bozo_clear (void)
{
	bozo_free ();
	bozo_init_prep ();
	dirty = TRUE;
}

/*****
******
*****/

static void
bozo_init_module (void)
{
	if (_bozo_chunk == NULL)
	{
		gchar * filename;
		GArray * file;
		gchar ** v;
		gint i;

		/* since we're loading from disk, clear the dirty flag */
		dirty = FALSE;

		/* initialize the internal variables */
		bozo_init_prep ();

		/* load the file */
		filename = get_bozo_filename ();
		file = read_file (filename);
		v = file==NULL ? NULL : g_strsplit (file->data, "\n", -1);
		for (i=0; v!=NULL && v[i]!=NULL; ++i)
		{
			gchar * addr = NULL;
			gchar * real = NULL;
			const gchar * line = v[i];

			/* skip comments */
			if (*line=='#')
				continue;

			/* must have at least one of (real, addr) */
			gnksa_do_check_from (line, &addr, &real);
			if (real==NULL && addr==NULL)
				continue;

			/* create a bozo */
			bozo_add (addr, real);

			/* cleanup */
			g_free (real);
			g_free (addr);
		}

		if (_bozos_unique->len != 0)
			g_message (_("Found %d bozos in %s"), (int)_bozos_unique->len, filename);

		/* cleanup */
		g_free (filename);
		if (file != NULL)
			g_array_free (file, TRUE);
		g_strfreev (v);
	}
}

/*****
******
*****/

/*****
******
*****/


/*****
******
*****/


/************
*************  PUBLIC
************/

gboolean
is_bozo (const gchar * address, const gchar * real)
{
	Bozo bozo;
	gboolean retval = FALSE;

	bozo.real = (gchar*) real;
	bozo.address = (gchar*) address;

	/* make sure the bozos are loaded */
	if (_bozo_chunk == NULL)
		bozo_init_module ();

	if (is_nonempty_string(address))
	{
		lower_bound (&bozo,
		             _bozos_sorted_by_address->pdata,
		             _bozos_sorted_by_address->len,
		             sizeof(gpointer),
		             compare_pBozo_to_ppBozo_by_address,
		             &retval);
	}
	else if (is_nonempty_string(real))
	{
		lower_bound (&bozo,
		             _bozos_sorted_by_name->pdata,
		             _bozos_sorted_by_name->len,
		             sizeof(gpointer),
		             compare_pBozo_to_ppBozo_by_name,
		             &retval);
	}


	return retval;
}

void
bozo_remove (const gchar * address, const gchar * real)
{
	gint i;
	gboolean done = FALSE;

	if (_bozo_chunk == NULL)
		bozo_init_module ();

	for (i=0; !done && i<_bozos_unique->len; ++i)
	{
		gint j;
		Bozo * bozo = (Bozo*) g_ptr_array_index (_bozos_unique, i);
		GPtrArray * a;

		/* if it's not a match, continue */
		if (pan_strcmp (address, bozo->address))
			continue;
		if (pan_strcmp (real, bozo->real))
			continue;


		/**
		***  Got a match, remove from our tables
		**/

		done = TRUE;

		/* remove from bozos_unique... */
		g_ptr_array_remove_index_fast (_bozos_unique, i);

		/* remove from sorted-by-address... */
		a = _bozos_sorted_by_address;
		for (j=0; j<a->len; ++j) {
			if (bozo == g_ptr_array_index (a, j)) {
				g_ptr_array_remove_index (a, j);
				break;
			}
		}

		/* remove from sorted-by-name... */
		a = _bozos_sorted_by_name;
		for (j=0; j<a->len; ++j) {
			if (bozo == g_ptr_array_index (a, j)) {
				g_ptr_array_remove_index (a, j);
				break;
			}
		}
	}
}


void
bozo_shutdown_module (void)
{
	if (dirty)
	{
		GString * str = g_string_new (NULL);
		guint i;

		/* write the bozo file to disk */
		gchar * filename = get_bozo_filename ();
		FILE * fp = fopen (filename, "w+");
		fprintf (fp, "# Bozo file used by the Pan newsreader\n");
		fprintf (fp, "# 1 bozo per line and must have realname or address or both\n");
		fprintf (fp, "# realname must be in quotes, address must be in <>\n");
		for (i=0; i!=_bozos_unique->len; ++i)
		{
			const Bozo * bozo = (const Bozo*) g_ptr_array_index (_bozos_unique, i);

			/* build the string... */
			g_string_truncate (str, 0);
			if (is_nonempty_string(bozo->real)) {
				g_string_sprintfa (str, "\"%s\"", bozo->real);
				if (is_nonempty_string (bozo->address))
					g_string_append_c (str, ' ');
			}
			if (is_nonempty_string(bozo->address))
				g_string_sprintfa (str, "<%s>", bozo->address);

			/* write the string... */
			if (str->len != 0) {
				g_string_append_c (str, '\n');
				fputs (str->str, fp);
			}
		}

		/* since we just saved, we're not dirty anymore */
		dirty = FALSE;

		/* cleanup */
		fclose (fp);
		g_free (filename);
		g_string_free (str, TRUE);
	}

	bozo_free ();
}


/*****
******
*****/

GPtrArray*
bozo_get_bozos (void)
{
	GPtrArray * a;

	if (_bozo_chunk == NULL)
		bozo_init_module ();

       	a = g_ptr_array_new ();
	pan_g_ptr_array_assign (a,  _bozos_unique->pdata, _bozos_unique->len);
	return a;
}
