/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include <gmime/gmime-utils.h>

#include "acache.h"
#include "gnksa.h"
#include "log.h"
#include "nntp.h"
#include "prefs.h"
#include "task.h"
#include "util.h"

gboolean online_prompt = TRUE;

/** deprecated, use nntp_command */
extern const char * sockwrite_err_msg;
extern const char * sockread_err_msg;

enum
{
	OK_GROUP				= 211,
	OK_AUTH					= 281,
	NEED_AUTHDATA				= 381,
	ERR_NOAUTH				= 480,
	ERR_AUTHREJ				= 482,
	ERR_ACCESS				= 502
};


/**
***
**/

static gint
nntp_get_response (PanSocket     * socket,
                   const gchar  ** setme_result)
{
	gint val;
	gint retval;
	const gchar * msg = NULL;

	val = pan_socket_getline (socket, &msg);
	if (val)
	{
		if (msg == NULL)
			msg = sockread_err_msg;
		if (setme_result != NULL)
			*setme_result = msg;
		retval = val;
	}
	else
	{
		if (setme_result != NULL)
			*setme_result = msg;
		retval = atoi (msg);
	}

	return retval;
}

gint
nntp_command (StatusItem           * status,
              PanSocket            * sock,
              const gchar         ** setme_response,
              const gchar          * command)
{
	int val;
	size_t len;

	g_return_val_if_fail (sock!=NULL, -1);
	g_return_val_if_fail (is_nonempty_string(command), -1);

	/**
	***  Send a command...
	**/

	len = strlen (command);
	if (len>=2 && command[len-2]=='\r' && command[len-1]=='\n')
	{
		val = pan_socket_putline (sock, command);
	}
	else /* not terminated in \r\n, so add */
	{
		gchar * tmp = g_strdup_printf ("%s\r\n", command);
		val = pan_socket_putline (sock, tmp);
		g_free (tmp);
	}
	if (val)
	{
		if (setme_response != NULL)
			*setme_response = sockwrite_err_msg;
		return val;
	}

	/**
	***  Get a response...
	**/

	val = nntp_get_response (sock, setme_response);

	if (val == ERR_NOAUTH)
	{
		val = nntp_authenticate (status, sock);
		if (val != TASK_SUCCESS)
			return -1;

		return nntp_command (status, sock, setme_response, command);
	}
	else return val;
}
 
gint
nntp_command_va (StatusItem           * status,
                 PanSocket            * sock,
                 const gchar         ** setme_result,
                 const gchar          * command_va,
                 ...)
{
	va_list args;
	char* line;
	int retval;

	g_return_val_if_fail (sock!=NULL, -1);
	g_return_val_if_fail (command_va!=NULL, -1);

	va_start (args, command_va);
	line = g_strdup_vprintf (command_va, args);
	va_end (args);

	retval = nntp_command (status, sock, setme_result, line);

	g_free (line);
	return retval;
}

/**
***
**/

gint
nntp_disconnect (StatusItem   * status,
                 PanSocket    * socket)
{
	const gchar * reply;
	gint val;

	/* entry assertions */
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (socket!=NULL, TASK_FAIL_HOPELESS);

	/* try to disconnect */
	val = nntp_command (status, socket, &reply, "QUIT");
	if (val != 205) {
		status_item_emit_error_va (status, _("QUIT failed: %s"), reply);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
	}

	/* disconnected succcessfully */
	return TASK_SUCCESS;
}

/**
***
**/

gint
nntp_handshake (StatusItem  * status,
                PanSocket   * socket,
                gboolean    * setme_posting_ok)
{
	gint val;
	const gchar * response = NULL;

	/* sanity checks */
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (socket!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (setme_posting_ok!=NULL, TASK_FAIL_HOPELESS);

	/* get the server's handshake message */
	val = nntp_get_response (socket, &response);
	if (val!=200 && val!=201) {
		status_item_emit_error_va (status, _("NNTP handshake failed: %s"), response);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
	}

	/* can we post? */
	*setme_posting_ok = val==200;
	return TASK_SUCCESS;
}

gint
nntp_authenticate (StatusItem     * status,
                   PanSocket      * sock)
{
	gint val;
	const gchar * response = NULL;

	/* entry assertions */
	g_return_val_if_fail (status!=NULL, TASK_FAIL);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (is_nonempty_string(sock->nntp_username), TASK_FAIL_HOPELESS);

	/* set the username */
	val = nntp_command_va (status, sock, &response, "AUTHINFO user %s", sock->nntp_username);
	if (val!=OK_AUTH && val!=NEED_AUTHDATA) {
		if (status != NULL)
			status_item_emit_error_va (status, _("Authentication failed: %s"), response);
		return TASK_FAIL;
	}

	/* set the password, if required */
	if (val==NEED_AUTHDATA) {
		if (!is_nonempty_string(sock->nntp_password)) {
			if (status != NULL)
				status_item_emit_error_va (status, _("Authentication failed: need a password"));
			return TASK_FAIL;
		}
		val = nntp_command_va (status, sock, &response, "AUTHINFO pass %s", sock->nntp_password);
	}

	/* if we failed, emit an error */	
	if (val!=OK_AUTH) {
		if (status != NULL)
			status_item_emit_error_va (status, _("Authentication failed: %s"), response);
		return TASK_FAIL;
	}

	return TASK_SUCCESS;
}

gint
nntp_can_post (StatusItem   * status,
               PanSocket    * sock,
               gboolean     * setme_can_post)
{
	gint val;
	const gchar * response = NULL;

	/* entry assertions */
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (setme_can_post!=NULL, TASK_FAIL_HOPELESS);

	/* ask the server if we can post or not */
	val = nntp_command (status, sock, &response, "MODE READER");
	if (val==200)
	{
		*setme_can_post = TRUE;
		return 0;
	}
	if (val==201)
	{
		*setme_can_post = FALSE;
		return 0;
	}
	else
	{
		*setme_can_post = FALSE;
		if (status != NULL)
			status_item_emit_error_va (status, _("MODE READER check failed: %s"),  response);
		return val;
	}
}

/**
***
**/

gint
nntp_set_group (StatusItem   * status,
                PanSocket    * sock,
                const char   * group_name)
{
	gint val;
	const gchar * response = NULL;

	/* sanity checks */
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(group_name), TASK_FAIL_HOPELESS);

	/* we're already there */
	if (!pan_strcmp (sock->nntp_group_name, group_name))
		return TASK_SUCCESS;

	/* change to that group */
	val = nntp_command_va (status, sock, &response, "GROUP %s", group_name);
	if (val != OK_GROUP) {
		status_item_emit_error_va (status, _("Unable to set group: %s"), response);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
	}

	/* update this socket's current group */
	replace_gstr (&sock->nntp_group_name, g_strdup(group_name));
	return TASK_SUCCESS;
}

/**
***
**/

gint
nntp_post (StatusItem    * status,
           PanSocket     * sock,
           const gchar   * msg)
{
	gint val;
	const gchar * response;

	/* entry assertions */
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(msg), TASK_FAIL_HOPELESS);

	/* if we're in mute mode, don't post */
	if (pan_mute)
	{
		g_message ("Mute: Your Message will not actually be posted.");
		printf ("\n\n\nYour message\n%s<end of message>\n\n\n", msg);
		fflush (NULL);
	}
	else
	{
		/* tell the server we want to post */
		val = nntp_command (status, sock, &response, "POST");
		if (val != 340) {
			status_item_emit_error_va (status, "Posting failed: %s", response);
			return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
		}

		/* post the article */
		val = nntp_command_va (status, sock, &response, "%s\r\n.\r\n", msg);
		if (val != 240) {
			status_item_emit_error_va (status, "Posting failed: %s", response);
			return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
		}
	}

	/* if we made it this far, we succeeded! */
	return TASK_SUCCESS;
}

/**
***
**/

gint
nntp_noop (StatusItem    * status,
           PanSocket     * socket)
{
	gboolean can_post = FALSE;

	/* entry assertions */
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (socket!=NULL, TASK_FAIL);

	/* do work */
	status_item_emit_status_va (status, _("\"stay connected\" sent to %s"), socket->host); 
	return nntp_can_post (status, socket, &can_post);
}

/**
***
**/

static gboolean
set_header_if_match (Article * a,
                     const gchar * header,
                     const gchar * header_key,
                     const gchar * article_key)
{
	gboolean match = FALSE;
	size_t len;

	/* entry assertions */
	g_return_val_if_fail (a!=NULL, FALSE);
	g_return_val_if_fail (is_nonempty_string(header_key), FALSE);
	g_return_val_if_fail (is_nonempty_string(article_key), FALSE);
	g_return_val_if_fail (header!=NULL, FALSE);

	/* if this is the header we're looking for, keep it. */
	len = strlen (header_key);
	if (!strncmp (header, header_key, len))
	{
		gchar * b = g_mime_utils_8bit_header_decode (header+len);
		article_set_header (a, article_key, b, DO_CHUNK);
		g_free (b);
		match = TRUE;
	}

	return match;
}

gint
nntp_article_download (StatusItem      * status,
                       PanSocket       * sock,
                       Group           * group,
                       Article         * article,
                       const gboolean  * abort,
                       gint              verbose)
{
	const gchar * response;
	gboolean getting_headers;
	gint retval;
	gint val;
	GString * buf;

	/* sanity checks */
	g_return_val_if_fail (status!=NULL, TASK_FAIL);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (group!=NULL, TASK_FAIL);
	g_return_val_if_fail (article!=NULL, TASK_FAIL);
	g_return_val_if_fail (abort!=NULL, TASK_FAIL);

	/* let the user know what we're doing */
	if (verbose & NNTP_VERBOSE_TEXT)
		status_item_emit_status (status, article->subject);

	/* request the article from the server */
	val = nntp_command_va (status, sock, &response, "ARTICLE %lu", article->number);
	if (val != 220) {
		status_item_emit_error_va (status, _("Getting article from server failed: %s"), response);
		if (val==-1)
			return TASK_FAIL;
		article_add_flag (article, STATE_ERROR);
		return TASK_FAIL_HOPELESS;
	}

	/* try to read the article... */
	getting_headers = TRUE;
	buf = g_string_new (NULL);
	retval = TASK_SUCCESS;
	if (verbose & NNTP_VERBOSE_INIT_STEPS)
		status_item_emit_init_steps (status, article->linecount);
	for (;;)
	{
		gchar * line;

		/* check for end cases: user abort, socket error, and success */
		if (*abort || pan_socket_getline (sock, &response)) {
			retval = TASK_FAIL;
			break;
		}
		if (!strncmp(response, ".\r\n", 3)) {
			retval = TASK_SUCCESS;
			break;
		}

		/* strip out the \r */
		line = pan_substitute (response, "\r\n", "\n");

		/* save the line */
		g_string_append (buf, line);
		if (verbose & NNTP_VERBOSE_NEXT_STEP)
			status_item_emit_next_step (status);

		/* save the headers that we want to save */
		if (getting_headers)
		{
			if (!strcmp(line, "\n")) /* header/body separator */
				getting_headers = FALSE;
			else 
			{
				Article * a = article;
				gboolean matched = FALSE;

				/* trim linefeed */
				g_strstrip (line);

				/* check for headers to store in Article */
				if (!matched)
					matched = set_header_if_match (a, line,
						"Newsgroups: ",
						HEADER_NEWSGROUPS);
				if (!matched)
					matched = (set_header_if_match (a, line,
						"Reply-To: ",
						HEADER_REPLY_TO));
				if (!matched)
					matched = (set_header_if_match (a, line,
						"Followup-To: ",
						HEADER_FOLLOWUP_TO));
			}
		}

		g_free (line);
	}

	/* save the message */
	acache_set_message (article->message_id, buf->str);
	g_string_free (buf, TRUE);

	if (retval == TASK_SUCCESS)
		article_remove_flag (article, STATE_DOWNLOAD_FLAGGED);

	return retval;
}


/**
***
**/

gint
nntp_get_group_info (StatusItem       * status,
                     PanSocket        * sock,
                     const char       * group_name,
                     gulong           * article_qty,
                     gulong           * low_num,
                     gulong           * high_num,
                     const gboolean   * abort)
{
	gint val;
	const gchar * response = NULL;

	/* entry assertions */
	g_return_val_if_fail (status!=NULL, TASK_FAIL);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (is_nonempty_string(group_name), TASK_FAIL);
	g_return_val_if_fail (article_qty!=NULL, TASK_FAIL);
	g_return_val_if_fail (low_num!=NULL, TASK_FAIL);
	g_return_val_if_fail (high_num!=NULL, TASK_FAIL);
	g_return_val_if_fail (abort!=NULL, TASK_FAIL);

	/* change to that group */
	val = nntp_command_va (status, sock, &response, "GROUP %s", group_name);
	if (val != OK_GROUP) {
		status_item_emit_error_va (status, _("Unable to set group: %s"), response);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
	}

	/* success; parse the results */
	sscanf (response, "%*d %lu %lu %lu", article_qty, low_num, high_num);
	replace_gstr (&sock->nntp_group_name, g_strdup(group_name));
	return TASK_SUCCESS;
}

/**
***
**/

gint
nntp_articlelist_download (StatusItem       * status,
                           PanSocket        * sock,
                           Group            * group,
                           gulong             low,
                           gulong             high,
                           const gboolean   * abort,
                           const char       * progress_fmt,
                           GPtrArray        * setme)
{
	const char * buffer;
	const char * response;
	gulong total;
	gulong first;
	gulong last;
	gboolean read_status = 0;
	gint result;
	gint val;

	/* sanity checks */
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (group!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (abort!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(progress_fmt), TASK_FAIL_HOPELESS);
	g_return_val_if_fail (setme!=NULL, TASK_FAIL_HOPELESS);

	/**
	***  Get up-to-date information about this group
	**/

	total = first = last = 0;
	result = nntp_get_group_info (status,
	                              sock,
	                              group->name,
	                              &total,
	                              &first,
	                              &last,
	                              abort);
	if (result != TASK_SUCCESS)
		return result;


	/**
	***  If no articles to get, then we're already done
	**/

	if (total == 0) {
		const gchar * n = group_get_readable_name (group);
		status_item_emit_status_va (status, _("No articles found for group \"%s\""), n);
		return TASK_SUCCESS;
	}

	/**
	***  Tell the server that we want a series of article headers...
	**/

	val = *abort ? -1 : nntp_command_va (status, sock, &response, "XOVER %lu-%lu", low, high);
	if (val != 224) {
		status_item_emit_status_va (status, _("Getting header list failed: %s"), response);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
	}

	/**
	***  Walk through all the articles headers that the server spits back
	**/

	pan_g_ptr_array_reserve (setme, setme->len + (high-low));
	while (!*abort && !((read_status = pan_socket_getline (sock, &buffer))))
	{
		gint part = 0;
	        gint parts = 0;
		const gchar * pch;
		const gchar * s;
		gchar * tmp;
		Article * a;

		/* handle end-of-list */
		if (!strncmp(buffer, ".\r\n", 3))
			break;

		/* create the article data */
		pch = buffer;
		a = article_new (group);
		group->articles_dirty = TRUE;

		/* article number */
		a->number = get_next_token_ulong (pch, '\t', &pch);

		/* subject */
		tmp = get_next_token_str (pch, '\t', &pch);
		replace_gstr (&tmp, g_mime_utils_8bit_header_decode(tmp));
		if (is_nonempty_string(tmp))
			article_set_header(a, HEADER_SUBJECT, tmp, DO_CHUNK);
		g_free (tmp);

		/* author */
		tmp = get_next_token_str (pch, '\t', &pch);
		replace_gstr (&tmp, g_mime_utils_8bit_header_decode(tmp));
		if (is_nonempty_string(tmp))
			article_set_author_from_header (a, tmp);
		g_free (tmp);

		/* date */
		tmp = get_next_token_str (pch, '\t', &pch);
		a->date = is_nonempty_string(tmp)
			? g_mime_utils_header_decode_date (tmp, NULL)
			: (time_t)0;
		g_free (tmp);

		/* message id */
		tmp = get_next_token_str (pch, '\t', &pch);
		if (is_nonempty_string(tmp))
			article_set_header(a, HEADER_MESSAGE_ID, tmp, DO_CHUNK);
		g_free (tmp);

		/* references */
		tmp = get_next_token_str (pch, '\t', &pch);
		if (is_nonempty_string(tmp))
			article_set_header(a, HEADER_REFERENCES, tmp, DO_CHUNK);
		g_free (tmp);

		/* unused field: size */
		skip_next_token (pch, '\t', &pch);

		/* linecount */
		a->linecount = get_next_token_int(pch, '\t', &pch);

		/* crosspost */
		tmp = get_next_token_str (pch, '\t', &pch);
		if (tmp==NULL || strncmp (tmp, "Xref:", 5))
			a->crosspost_qty = 1;
		else {
			gchar * foo;
			a->crosspost_qty = 0;
			for (foo=tmp+5; *foo; ++foo)
				if (*foo==':')
					++a->crosspost_qty;
		}
		g_free (tmp);

		/**
		***  Validity Checking
		**/

		pch = article_get_header (a, HEADER_SUBJECT);
		if (!is_nonempty_string(pch)) {
			status_item_emit_error_va (status, _("Corrupt header skipped - no subject: %s"), buffer);
			pan_object_unref (PAN_OBJECT(a));
			continue;
		}

		pch = article_get_header (a, HEADER_MESSAGE_ID);
		if (!is_nonempty_string(pch)) {
			status_item_emit_error_va (status, _("Corrupt header skipped - no msg id: %s"), buffer);
			pan_object_unref (PAN_OBJECT(a));
			continue;
		}

		if (a->date == 0) {
			status_item_emit_error_va (status, _("Corrupt header skipped - no date: %s"), buffer);
			pan_object_unref (PAN_OBJECT(a));
			continue;
		}

		tmp = article_get_author_str (a);
		if (!is_nonempty_string(tmp)) {
			status_item_emit_error_va (status, _("Corrupt header skipped - no author: %s"), buffer);
			pan_object_unref (PAN_OBJECT(a));
			g_free (tmp);
			continue;
		}
		g_free (tmp);


		/* Look for the (n/N) or [n/N] construct in subject lines,
		 * starting at the end of the string and working backwards */
		part = 0;
		parts = 0;
		s = a->subject;
		pch = s + strlen(s) - 1;
		while (pch != s)
		{
			/* find the ']' of [n/N] */
			--pch;
			if ((pch[1]!=')' && pch[1]!=']') || !isdigit((int)*pch))
				continue;

			/* find the '/' of [n/N] */
			while (s!=pch && isdigit((int)*pch))
				--pch;
			if (s==pch || (*pch!='/' && *pch!='|'))
				continue;

			/* N -> parts */
			parts = atoi (pch+1);
			--pch;

			/* find the '[' of [n/N] */
			while (s!=pch && isdigit((int)*pch))
				--pch;
			if (s==pch || (*pch!='(' && *pch!='[')) {
				parts = 0;
				continue;
			}

			/* n -> part */
			part = atoi (pch+1);
			break;
		}

		/* if not a multipart yet, AND if it's a big message, AND
		   it's either in one of the pictures/fan/sex groups or it
		   has commonly-used image names in the subject, guess it's
		   a single-part binary */
		if (!parts
			&& a->linecount>400
			&& (((strstr(group->name,"binaries")
					|| strstr(group->name,"fan")
				        || strstr(group->name,"mag")
					|| strstr(group->name,"sex")))
				|| ((strstr(s,".jpg") || strstr(s,".JPG")
					|| strstr(s,".jpeg") || strstr(s,".JPEG")
					|| strstr(s,".gif") || strstr(s,".GIF")
					|| strstr(s,".tiff") || strstr(s,".TIFF")))))
			part = parts = 1;

		/* Verify Multipart info */
		if ((parts>=1) && (part<=parts)) {
			/* yay, we found one! */
			a->parts = parts;
			a->part = part;
		}
		else {
			a->parts = 0;
			a->part = 0;
		}

		/* Add the article to the article list */
		g_ptr_array_add (setme, a);

		/* update the count & progress ui */
		status_item_emit_next_step (status);
		if (!(setme->len % 29))
			status_item_emit_status_va (status, progress_fmt, setme->len, (high-low));
	}

	if (read_status!=0)
		status_item_emit_error (status, sockread_err_msg);
	if (read_status!=0 || *abort)
		return TASK_FAIL;

	/* if we've made it this far, we succeeded! */
	return TASK_SUCCESS;
}

/**
***
**/

gint
nntp_download_bodies (StatusItem       * status,
                      Group            * group,
                      PanSocket        * sock,
                      const gboolean   * abort,
                      const GPtrArray  * const articles,
                      gint             * index)
{
	gint result;
	gint i;
	gint lines_so_far;
	gint lines_total;

	/* sanity checks */
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (group!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (abort!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (articles!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (index!=NULL, TASK_FAIL_HOPELESS);

	/* change to the group we want */
	if (*abort)
		return TASK_FAIL;
	result = nntp_set_group (status, sock, group->name);
	if (result != TASK_SUCCESS) {
		const gchar * g = group_get_readable_name (group);
		status_item_emit_error_va (status, _("Couldn't select group \"%s\""), g);
		return result;
	}

	/* status item */
	lines_so_far = lines_total = 0;
	for (i=0; i<articles->len; ++i) {
		Article * article = ARTICLE(g_ptr_array_index(articles,i));
		lines_total += article->linecount;
		if (i<*index)
			lines_so_far += article->linecount;
	}
	status_item_emit_init_steps (status, lines_total);
	status_item_emit_set_step (status, lines_so_far);

	/* download the articles */
	for (; *index<articles->len && !*abort; ++(*index))
	{
		const gchar * pch;
		Article * article = ARTICLE(g_ptr_array_index(articles,*index));

		status_item_emit_status_va (status,
			_("Getting article %d of %d"),
			1+*index, articles->len);

		/* download if we have to.  Abort if the download failed but
		 * might succeed later.  Hopeless downloads silently fall
		 * through; it's pointless to postpone the task for them. */
		pch = article_get_header (article, HEADER_MESSAGE_ID);
		if (!acache_has_message (pch))
		{
			result = nntp_article_download (status,
			                                sock,
			                                group,
			                                article,
			                                abort,
			                                NNTP_VERBOSE_NEXT_STEP);
		}
		else
		{
			result = TASK_SUCCESS;

			/* since it's cached locally, we can unflag */
			article_remove_flag (article, STATE_DOWNLOAD_FLAGGED);
		}

		switch (result)
		{
			case TASK_FAIL:
				status_item_emit_status_va (status, _("Downloading article \"%s\" failed."), article->subject);
				return result;
			case TASK_SUCCESS:
			case TASK_FAIL_HOPELESS:
				break;
			default:
				pan_warn_if_reached ();
		}
	}

	return TASK_SUCCESS;
}


gint
nntp_cancel (StatusItem     * status,
             const Article  * article,
             PanSocket      * sock)
{
	GString * msg;
	gchar * author;
	const gchar * newsgroups;
	gint retval;
	const gchar * msg_id = article_get_header (article, HEADER_MESSAGE_ID);

	/* sanity checks */
	g_return_val_if_fail (sock != NULL, TASK_FAIL);
	g_return_val_if_fail (status != NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (article != NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(msg_id), TASK_FAIL_HOPELESS);
	newsgroups = article_get_header (article, HEADER_NEWSGROUPS);
	g_return_val_if_fail (is_nonempty_string(newsgroups), TASK_FAIL_HOPELESS);

	/* let the user know what we're doing */
	status_item_emit_status_va (status, _("Cancelling message \"%s\""), msg_id);

	/* build the message to post */
	msg = g_string_new (NULL);
	author = article_get_author_str(article);
        replace_gstr (&author, g_mime_utils_8bit_header_encode(author));
	g_string_sprintfa (msg, "From: %s\r\n", author);
	g_string_sprintfa (msg, "Newsgroups: %s\r\n", newsgroups);
	g_string_sprintfa (msg, "Subject: cancel %s\r\n", msg_id);
	g_string_sprintfa (msg, "Control: cancel %s\r\n", msg_id);
	g_string_sprintfa (msg, "\r\n");
	g_string_sprintfa (msg, "Ignore\r\nArticle canceled by Pan %s\r\n", VERSION);

	/* post the cancel message */
	retval = nntp_post (status, sock, msg->str);

	/* cleanup */
	g_free (author);
	g_string_free (msg, TRUE);

	return retval;
}
