/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <gnome.h>

#include <stdlib.h>
#include <string.h>

#include "acache.h"
#include "articlelist.h"
#include "globals.h"
#include "gui.h"
#include "gui-headers.h"
#include "gui-paned.h"
#include "nntp.h"
#include "prefs.h"
#include "queue.h"
#include "task-headers.h"
#include "text.h"
#include "util.h"

#include "xpm/layout_1.xpm"
#include "xpm/layout_2.xpm"
#include "xpm/layout_3.xpm"
#include "xpm/layout_4.xpm"
#include "xpm/layout_5.xpm"
#include "xpm/traffic_green.xpm"
#include "xpm/traffic_red.xpm"


#define DISPLAY_PREFS_PREVIEW_FONT_SIZE 12


/* server profile */
typedef struct
{
	GtkWidget * profile_name_entry;
	GtkWidget * server_address;
	GtkWidget * server_port;
	GtkWidget * auth_cbutton;
	GtkWidget * server_username_label;
	GtkWidget * server_username;
	GtkWidget * server_password_label;
	GtkWidget * server_password;
	GtkWidget * msgid_cbutton;
	GtkWidget * max_connections;
	GtkWidget * idle_secs_before_timeout;
	Server * server;
}
server_profile;

typedef struct
{
	const Server * server;
	GtkWidget * max_connections_sb;
	GtkWidget * online_button;
	GtkWidget * reserve_tb;
	gboolean online_value;
}
server_connection;

typedef struct
{
	GnomePropertyBox * box;
	GtkWidget * server_clist;

	/* smtp server*/
	GtkWidget * smtp_address;
	GtkWidget * smtp_port;

	GtkWidget * grouplist_mb1_action_load_tb;
	GtkWidget * grouplist_mb1_action_select_tb;
	GtkWidget * grouplist_mb2_action_load_tb;
	GtkWidget * grouplist_mb2_action_select_tb;

	GtkWidget * articlelist_mb1_action_load_tb;
	GtkWidget * articlelist_mb1_action_select_tb;
	GtkWidget * articlelist_mb2_action_load_tb;
	GtkWidget * articlelist_mb2_action_select_tb;

	/* cache */
	GtkWidget * cache_megs_sb;
	GtkWidget * flush_cache_on_exit_check;

	/* general */
	GtkWidget * display_article_dates_in_local_time_cbutton;
	GtkWidget * fetch_new_on_group_load_cbutton;
	/* GtkWidget * break_thread_when_subject_changes_cbutton; */
	GtkWidget * online_prompt_first_cbutton;
	GtkWidget * online_without_asking_cbutton;
	GtkWidget * online_never_cbutton;
	GtkWidget * hide_mpart_child_nodes_cbutton;
	GtkWidget * attribution_entry;
	GtkWidget * external_mailer_entry;
	GtkWidget * external_editor_entry;
	GtkWidget * auto_part_selection;

	/* user */
	GtkWidget * user_email;
	GtkWidget * user_reply_to;
	GtkWidget * user_organization;
	GtkWidget * user_fullname;
	GtkWidget * user_signature_file;

	/* wrapping */
	GtkWidget * wrap_column_tb;
	GtkWidget * wrap_column_sb;

	/* directories */
	GtkWidget * dir_download;
	GtkWidget * dir_temp;
	GtkWidget * dir_data;

	/* display */
	GtkWidget * use_gdk_fontset_load_checkbutton;
	GtkWidget * grouplist_gfp;
	GtkWidget * articlelist_gfp;
	GtkWidget * message_gfp;
	GtkWidget * read_gcp;
	GtkWidget * unread_gcp;
	GtkWidget * new_gcp;
	GtkWidget * ignored_gcp;
	GtkWidget * watched_gcp;
	GtkWidget * text_fg_gcp;
	GtkWidget * text_bg_gcp;
	GtkWidget * text_quoted_gcp;

	/* layout */
	GtkWidget * layout_page;

	/* article headers */
	GtkWidget * article_headers_clist;

	/* connections */
	GtkWidget  * max_connections;
	GtkWidget  * max_tries;
	GSList * server_connections;

}
PrefsWindow;

typedef struct
{
	Server *old;
	Server *new;
}
prefs_server;


static gchar* layout_get_new_string (GtkWidget * layout_page);
static void prefs_apply_cb (GtkWidget *box, gint page_num);
static void prefs_cancel_cb (void);
static void new_server_cb (void);
static void edit_server_cb (void);
static void delete_server_cb (void);
static void prefs_create_clist (void);
static void edit_profile_dialog_clicked (GnomeDialog*, int button, prefs_server*);
static void edit_profile_dialog (prefs_server *data);
static void server_clist_button_press (GtkWidget*, GdkEventButton*);
static void prefs_servers_changed (void);
static GtkWidget * prefs_general_page ( void );
static GtkWidget * prefs_smtp_page (void);
static GtkWidget * prefs_cache_page (void);
static GtkWidget * prefs_server_page (void);
static GtkWidget * prefs_general_page (void);
static GtkWidget * prefs_directories_page (void);
static GtkWidget * prefs_connections_page (void);
static GtkWidget * prefs_userinfo_page (void);
static GtkWidget * prefs_grouplist_page (void);
static GtkWidget * prefs_layout_page (void);
static gulong get_header_flags (void);

static PrefsWindow * win;
static gboolean servers_changed = FALSE;
static GSList * server_deletes = NULL;
static GtkWidget *sections_clist;
static int nntp_clist_row = -1;
static int nntp_connections_row = -1;
static GSList * sprof_list = NULL; /* holds the profile editor GnomeDialogs */


extern GtkTooltips * ttips;

gchar * layout_str = NULL;
gint wrap_column                              = 74;
gint watched_thread_span_days                 = 30;
gint mail_server_port                         = 0;
/* gboolean break_thread_when_subject_changes    = FALSE; */
gboolean display_article_dates_in_local_time  = FALSE;
gboolean hide_mpart_child_nodes               = TRUE;
gboolean pan_mute                             = FALSE;
gboolean auto_part_selection                  = TRUE;
gboolean use_gdk_fontset_load                 = FALSE;
gboolean fetch_new_on_group_load              = TRUE;

ListClickAction grouplist_mb1_action          = CLICK_ACTION_LOAD;
ListClickAction grouplist_mb2_action          = CLICK_ACTION_SELECT;
ListClickAction articlelist_mb1_action        = CLICK_ACTION_LOAD;
ListClickAction articlelist_mb2_action        = CLICK_ACTION_SELECT;

GoingOnlinePreference going_online_preference = ONLINE_PROMPT;
gchar * temp_dir = NULL;
gchar * data_dir = NULL;
gchar * download_dir = NULL;
gchar * grouplist_font = NULL;
gchar * articlelist_font = NULL;
gchar * message_body_font = NULL;
gchar * attribution_line = NULL;
gchar * external_mailer = NULL;
gchar * external_editor = NULL;
gchar * extra_posting_headers = NULL;
gchar * mail_server_address = NULL;

#define connect_signal_to_prefs_changed(object,signal_name) \
	gtk_signal_connect_object (GTK_OBJECT(object), signal_name, \
	                           GTK_SIGNAL_FUNC(gnome_property_box_changed), \
	                           GTK_OBJECT(win->box))

#define connect_signal_to_profile_changed(object,signal_name,dialog) \
	gtk_signal_connect_object (GTK_OBJECT(object), signal_name, \
	                           GTK_SIGNAL_FUNC(edit_profile_changed), \
	                           GTK_OBJECT(dialog))

/**
***  UPDATE UTILS
**/

static void
update_entry_and_bool_from_toggle_button (gboolean * setme,
                                          const char * key,
                                          GtkWidget * toggle)
{
	const gboolean b = gtk_toggle_button_get_active (
		GTK_TOGGLE_BUTTON(toggle));

	if (b != *setme) {
		*setme = b;
		gnome_config_set_bool (key, b);
	}
}

static void
set_config_from_editable (const char   * key,
                          GtkWidget    * entry)
{
	gchar * text = gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1);
	g_strstrip (text);
	if (is_nonempty_string(text))
		gnome_config_set_string (key, text);
	else
		gnome_config_clean_key (key);
	g_free (text);
}

static void
replace_string_and_config_from_editable (gchar       ** setme,
                                         const char   * key,
                                         GtkWidget    * e)
{
	gchar * s = gtk_editable_get_chars (GTK_EDITABLE(e), 0, -1);
	g_strstrip (s);
	if (!pan_strcmp(*setme, s))
		g_free (s);
	else {
		gnome_config_set_string (key, s);
		replace_gstr (setme, s);
	}
}

static void
set_color (GdkColor      * color,
           const gchar   * key,
           guint           r,
           guint           g,
           guint           b)
{
	GString * s = g_string_new (NULL);

	g_string_sprintf (s, "/Pan/Display/%s_r=%u", key, r);
	color->red = gnome_config_get_int (s->str);

	g_string_sprintf (s, "/Pan/Display/%s_g=%u", key, g);
	color->green = gnome_config_get_int (s->str);

	g_string_sprintf (s, "/Pan/Display/%s_b=%u", key, b);
	color->blue = gnome_config_get_int (s->str);

	g_string_free (s, TRUE);
}

static gboolean
handle_font_picker (GtkWidget * gfp, gchar ** p_old_font, const gchar * config_key)
{
	const gchar * new_font;
	gboolean changed = FALSE;

	/* get new... */
	new_font = gnome_font_picker_get_font_name (GNOME_FONT_PICKER(gfp));

	/* compare old with new... */
	if (pan_strcmp (*p_old_font, new_font))
	{
		replace_gstr (p_old_font, g_strdup(new_font));
		gnome_config_set_string (config_key, new_font);
		changed = TRUE;
	}

	return changed;
}

static gboolean
handle_color_picker (const gchar    * key,
                     GdkColor       * color,
                     GtkWidget      * w)
{
	gboolean color_changed = FALSE;
	gushort r=0, g=0, b=0, a=0;
	gnome_color_picker_get_i16 (GNOME_COLOR_PICKER(w), &r, &g, &b, &a);
	if ((color->red != r)
		|| (color->green != g)
		|| (color->blue != b))
	{
		GString * s = g_string_new (NULL);

		color_changed = TRUE;

		/* update the color */
		gdk_colormap_free_colors (cmap, color, 1);
		color->red = r;
		color->green = g;
		color->blue = b;
		if (!gdk_color_alloc (cmap, color))
			g_error ("couldn't allocate `%s' color", key);

		/* update the gnome config */
		g_string_sprintf (s, "/Pan/Display/%s_r", key);
		gnome_config_set_int (s->str, r);
		g_string_sprintf (s, "/Pan/Display/%s_g", key);
		gnome_config_set_int (s->str, g);
		g_string_sprintf (s, "/Pan/Display/%s_b", key);
		gnome_config_set_int (s->str, b);
		g_string_sprintf (s, "/Pan/Display/%s_a", key);
		gnome_config_set_int (s->str, a);

		/* cleanup */
		g_string_free (s, TRUE);
	}

	return color_changed;
}

/**
***  UPDATE EVERYTHING
**/



/* Ok or Apply pressed in the Preferences dialog, save all information */
static void
prefs_apply_cb (GtkWidget   * box,
                gint          page_num)
{
	int cache_old;
	int cache_new;
	gboolean tmp_bool = FALSE;
	gboolean articlelist_changed = FALSE;
	gboolean server_connections_changed = FALSE;
	gboolean color_changed;

	/* -1 means all pages, ignore anything else */
	if (page_num != -1)
		return;

	/* Commit all of the data to the config file */

	/* No entry for these yet
	 * gnome_config_set_int("/Pan/Articles/Expiration", 30);
	 */

	set_config_from_editable (
		"/Pan/User/Full_Name",
		win->user_fullname);

	set_config_from_editable (
		"/Pan/User/Email",
		win->user_email);

	set_config_from_editable (
		"/Pan/User/Reply_To",
		win->user_reply_to);

	set_config_from_editable (
		"/Pan/User/Organization",
		win->user_organization);

	set_config_from_editable (
		"/Pan/User/Signature_File",
		gnome_file_entry_gtk_entry(
			GNOME_FILE_ENTRY(win->user_signature_file)));

	set_config_from_editable (
		"/Pan/Paths/download_dir",
		gnome_file_entry_gtk_entry(
			GNOME_FILE_ENTRY(win->dir_download)));

	set_config_from_editable (
		"/Pan/Paths/temp_dir",
		gnome_file_entry_gtk_entry(
			GNOME_FILE_ENTRY(win->dir_temp)));

	set_config_from_editable (
		"/Pan/Paths/data_dir",
		gnome_file_entry_gtk_entry(
			GNOME_FILE_ENTRY(win->dir_data)));

	/**
	***  Mail Server
	**/

	replace_string_and_config_from_editable (&mail_server_address,
	                                         "/Pan/Mail/smtp_address",
	                                         win->smtp_address);

	if (1) {
		int i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(win->smtp_port));
		if (i != mail_server_port) {
			gnome_config_set_int ("/Pan/Mail/smtp_port", i);
			mail_server_port = i;
		}
	}

	/**
	***  Windows
	**/

	/* update the articlelist colors */
	articlelist_changed |= handle_color_picker ("unread_color", &unread_color, win->unread_gcp);
	articlelist_changed |= handle_color_picker ("read_color", &read_color, win->read_gcp);
	articlelist_changed |= handle_color_picker ("new_color", &new_color, win->new_gcp);
	articlelist_changed |= handle_color_picker ("killfile_color", &killfile_color, win->ignored_gcp);
	articlelist_changed |= handle_color_picker ("watched_color", &watched_color, win->watched_gcp);

	/* update the text window colors */
	color_changed = FALSE;
	color_changed |= handle_color_picker ("text_fg_color", &text_fg_color, win->text_fg_gcp);
	color_changed |= handle_color_picker ("text_bg_color", &text_bg_color, win->text_bg_gcp);
	color_changed |= handle_color_picker ("text_quoted_color", &text_quoted_color, win->text_quoted_gcp);
	if (color_changed)
	{
		/* must use a new style* in order to update bg color */
		GtkStyle *style=gtk_style_copy(gtk_widget_get_style(Pan.text));
		style->text[0] = text_fg_color;
		style->base[0] = text_bg_color;
		gtk_widget_set_style (Pan.text, style);
		text_refresh ();
	}

	update_entry_and_bool_from_toggle_button (
		&fetch_new_on_group_load,
		"/Pan/General/fetch_new_on_group_load",
		win->fetch_new_on_group_load_cbutton);

	update_entry_and_bool_from_toggle_button (
		&use_gdk_fontset_load,
		"/Pan/General/use_gdk_fontset_load",
		win->use_gdk_fontset_load_checkbutton);

	/* Handle font pickers */
	if (handle_font_picker (win->articlelist_gfp,
	                        &articlelist_font,
	                        "/Pan/Display/articlelist_font"))
	{
		GdkFont * font = use_gdk_fontset_load
			? gdk_fontset_load (articlelist_font)
			: gdk_font_load (articlelist_font);

		if (font!=NULL) {
			if (killfile_style!=NULL)
				killfile_style->font = font;
			if (unread_style!=NULL)
				unread_style->font = font;
			if (read_style!=NULL)
				read_style->font = font;
			if (new_style!=NULL)
				new_style->font = font;
		}

		articlelist_changed = TRUE;
	}

	if (handle_font_picker (win->grouplist_gfp,
	                        &grouplist_font,
	                        "/Pan/Display/grouplist_font"))
		widget_set_font (GTK_WIDGET(Pan.group_clist), grouplist_font);

	if (handle_font_picker (win->message_gfp,
	                        &message_body_font,
	                        "/Pan/Display/message_body_font")) {
		widget_set_font (GTK_WIDGET(Pan.text), message_body_font);
		text_refresh ();
	}

	/* cache */
	cache_old = gnome_config_get_int_with_default(
		"/Pan/Cache/MaxSize=1242880", NULL);
	cache_new = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(win->cache_megs_sb));
	cache_new *= (1024*1024); /* convert to bytes */
	if (cache_old != cache_new) {
		gnome_config_set_int ("/Pan/Cache/MaxSize", cache_new);
		acache_expire ();
	}
	gnome_config_set_bool ("/Pan/Cache/FlushOnExit",
			gtk_toggle_button_get_active (
				GTK_TOGGLE_BUTTON( win->flush_cache_on_exit_check)));

	/* article headers */
	if (1) {
		gulong flags = get_header_flags ( );
		if (header_flags != flags) {
			gnome_config_set_int ("/Pan/State/Headers", flags);
			header_flags = flags;
			text_refresh ();
		}
	}
	
	if (servers_changed) {
		/* Handle the server mess */
		prefs_servers_changed ();
	}

	/**
	***  Filling
	**/

	update_entry_and_bool_from_toggle_button (
		&auto_part_selection,
		"/Pan/General/auto_part_selection",
		win->auto_part_selection);

	if (1)
	{
		gboolean tmp = text_get_wrap ();

		update_entry_and_bool_from_toggle_button (
			&tmp,
			"/Pan/General/do_wrap",
			win->wrap_column_tb);

		text_set_wrap (tmp);
	}

	/**
        ***  Group list mouse clicks
        **/

	if (1) {
		ListClickAction old;

		old = grouplist_mb1_action;
		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(win->grouplist_mb1_action_load_tb)))
			grouplist_mb1_action = CLICK_ACTION_LOAD;
		else
			grouplist_mb1_action = CLICK_ACTION_SELECT;
		if (grouplist_mb1_action != old)
			gnome_config_set_int ("/Pan/General/grouplist_mb1_action", grouplist_mb1_action);

		old = grouplist_mb2_action;
		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(win->grouplist_mb2_action_load_tb)))
			grouplist_mb2_action = CLICK_ACTION_LOAD;
		else
			grouplist_mb2_action = CLICK_ACTION_SELECT;
		if (grouplist_mb2_action != old)
			gnome_config_set_int ("/Pan/General/grouplist_mb2_action", grouplist_mb2_action);

		old = articlelist_mb1_action;
		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(win->articlelist_mb1_action_load_tb)))
			articlelist_mb1_action = CLICK_ACTION_LOAD;
		else
			articlelist_mb1_action = CLICK_ACTION_SELECT;
		if (articlelist_mb1_action != old)
			gnome_config_set_int ("/Pan/General/articlelist_mb1_action", articlelist_mb1_action);

		old = articlelist_mb2_action;
		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(win->articlelist_mb2_action_load_tb)))
			articlelist_mb2_action = CLICK_ACTION_LOAD;
		else
			articlelist_mb2_action = CLICK_ACTION_SELECT;
		if (articlelist_mb2_action != old)
			gnome_config_set_int ("/Pan/General/articlelist_mb2_action", articlelist_mb2_action);
	}


#if 0	
	tmp_bool = break_thread_when_subject_changes;
	update_entry_and_bool_from_toggle_button (
		&break_thread_when_subject_changes,
		"/Pan/General/break_thread_when_subject_changes",
		win->break_thread_when_subject_changes_cbutton);
	if (tmp_bool != break_thread_when_subject_changes)
		articlelist_changed = TRUE;
#endif


	tmp_bool = display_article_dates_in_local_time;
	update_entry_and_bool_from_toggle_button (
		&display_article_dates_in_local_time,
		"/Pan/General/display_article_dates_in_local_time",
		win->display_article_dates_in_local_time_cbutton);
	if (tmp_bool != display_article_dates_in_local_time)
		articlelist_changed = TRUE;


	tmp_bool = hide_mpart_child_nodes;
	update_entry_and_bool_from_toggle_button (
		&hide_mpart_child_nodes,
		"/Pan/General/hide_mpart_child_nodes",
		win->hide_mpart_child_nodes_cbutton);
	if (tmp_bool != hide_mpart_child_nodes)
		articlelist_changed = TRUE;


	if (1) {
		GtkSpinButton* w = GTK_SPIN_BUTTON(win->wrap_column_sb);
		int i = gtk_spin_button_get_value_as_int (w);
		if (i != wrap_column) {
			wrap_column = i;
			gnome_config_set_int ("/Pan/General/wrap_column", i);
			text_refresh ();

		}
	}

	/*FIXME: there needs to be a way to edit this in the UI */
	gnome_config_set_string ("/Pan/Posting/Extra_Headers", extra_posting_headers);

	replace_string_and_config_from_editable (
		&attribution_line,
		"/Pan/User/attribution_line",
		win->attribution_entry);

	replace_string_and_config_from_editable (
		&external_mailer,
		"/Pan/User/external_mailer",
		win->external_mailer_entry);

	replace_string_and_config_from_editable (
		&external_editor,
		"/Pan/User/external_editor",
		win->external_editor_entry);


	/* server connections ui */
	if (1)
	{
		GSList * l;
		for (l=win->server_connections; l!=NULL; l=l->next)
		{
			server_connection * sc = (server_connection*) l->data;
			Server * server = SERVER(sc->server);
			int i;

			/* max connections for this server */
			i = gtk_spin_button_get_value_as_int (
				GTK_SPIN_BUTTON(sc->max_connections_sb));
			if (i != server->max_connections)
			{
				server->max_connections = i;
				server_connections_changed = TRUE;
			}

			/* online radiobox */
			i = sc->online_value;
			if ((i!=0) != (server->online!=0))
			{
				server_set_online (server, i);
				server_connections_changed = TRUE;
			}

			/* reserve nonleech */
			i = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(sc->reserve_tb));
			if ((i!=0) != (server->reserve_nonleech!=0))
			{
				server->reserve_nonleech = i;
				server_connections_changed = TRUE;
			}
		}
	}

	/* how to go online */
	if (1)
	{
		GoingOnlinePreference pref;
		GtkToggleButton * never = GTK_TOGGLE_BUTTON (win->online_never_cbutton);
		GtkToggleButton * always = GTK_TOGGLE_BUTTON (win->online_without_asking_cbutton);

		if (gtk_toggle_button_get_active (never))
			pref = ONLINE_NO;
		else if (gtk_toggle_button_get_active (always))
			pref = ONLINE_YES;
		else
			pref = ONLINE_PROMPT;

		if (pref != going_online_preference)
		{
			going_online_preference = pref;
			gnome_config_set_int ("/Pan/General/Going_Online", (int)pref);
		}
	}

	/* maximum number of connections */
	if (1) {
		GtkSpinButton * sb = GTK_SPIN_BUTTON (win->max_connections);
		const int i = gtk_spin_button_get_value_as_int (sb);
		if (i != queue_get_max_sockets()) {
			queue_set_max_sockets (i);
			server_connections_changed = TRUE;
			gnome_config_set_int ("/Pan/General/Max_Connections_Total", i);
		}
	}

	/* maximum number of tries */
	if (1) {
		GtkSpinButton * sb = GTK_SPIN_BUTTON (win->max_tries);
		const int i = gtk_spin_button_get_value_as_int (sb);
		if (i != queue_get_max_tries()) {
			queue_set_max_tries (i);
			server_connections_changed = TRUE;
			gnome_config_set_int ("/Pan/General/Max_Task_Tries_Total", i);
		}
	}

	/* layout */
	if (1) {
		gchar * str = layout_get_new_string (win->layout_page);
		if (strcmp (str, layout_str)) {
			replace_gstr (&layout_str, g_strdup(str));
			gnome_config_set_string ("/Pan/Display/Layout", layout_str);
			gui_set_layout (GUI_PANED);
		}
		g_free (str);
	}

	/**
	***
	***
	**/

	if (articlelist_changed)
		articlelist_reset_colors ();

	if (server_connections_changed) {
		queue_wakeup ();
		server_menu_update ();
	}

	gnome_config_sync ();

	prefs_init ();
}

static void
prefs_servers_changed (void)
{
	gint row = 0;
	prefs_server * p_server;
	GString * path = g_string_new (NULL);

	while ((p_server = gtk_clist_get_row_data (GTK_CLIST(win->server_clist), row)))
	{
		if (p_server->new)
		{
			if (p_server->old)
			{
				/* Modified */
				Server * new = p_server->new;
				Server * old = p_server->old;

				replace_gstr (&old->name, g_strdup(new->name));
				replace_gstr (&old->address, g_strdup(new->address));
				replace_gstr (&old->username, g_strdup(new->username));
				replace_gstr (&old->password, g_strdup(new->password));
				old->port = new->port;
				old->need_auth = new->need_auth;
				old->max_connections = new->max_connections;
				old->idle_secs_before_timeout = new->idle_secs_before_timeout;
				old->gen_msgid = new->gen_msgid;
			}
			else
			{
				/* New addition */
				Server * new = p_server->new;
				g_string_sprintf (path, "/%s/%s/", data_dir, new->name);
				directory_check (path->str);
				Pan.serverlist = g_slist_append (Pan.serverlist, new);
				p_server->new = NULL;
			}
		}

		++row;
	}

	if (server_deletes != NULL)
	{
		GSList * l;

		/* remove from server list */
		for (l=server_deletes; l!=NULL; l=l->next)
		{
			prefs_server * p_server = l->data;
			Server * server= p_server->old;
			GSList * m;
			GSList ** pm;

			/* remove it from the pan server list */
			Pan.serverlist = g_slist_remove (Pan.serverlist, server);

			/* remove from the prefs connections tab */
			pm = &win->server_connections;
			for (m=*pm; m!=NULL; m=m->next)
			{
				server_connection * sc = (server_connection*) m->data;
				if (sc->server == server)
				{
					*pm = g_slist_remove (*pm, sc);
					g_free (sc);
					break;
				}
			}

			/* destroy the server */
			pan_object_unref (PAN_OBJECT(server));
		}

		/* cleanup */
		g_slist_foreach (server_deletes, (GFunc)g_free, NULL);
		g_slist_free (server_deletes);
		server_deletes = NULL;
	}

	server_menu_update ();
	g_string_free (path, TRUE);
}

static void
ensure_trailing_slash (gchar ** pch)
{
	gchar * m = strrchr (*pch, G_DIR_SEPARATOR);
	if (!m || m[1]!='\0')
		replace_gstr (pch, g_strdup_printf ("%s%c", *pch, G_DIR_SEPARATOR));
}


/**
 * prefs_init:
 *
 * Called from pan_init() before gtk_main(), this is where all the
 * user preferences get loaded in.
 **/
void
prefs_init (void)
{
	static gchar * static_tmp_dir = NULL;
	gchar *pch = NULL;

	/* get the layout string */
	replace_gstr (&layout_str, gnome_config_get_string ("/Pan/Display/Layout"));
	if (!is_nonempty_string (layout_str))
		replace_gstr (&layout_str, g_strdup ("4gta"));

	/* get temp directory */
	replace_gstr (&temp_dir, gnome_config_get_string ("/Pan/Paths/temp_dir"));
	if (is_nonempty_string (temp_dir))
		ensure_trailing_slash (&temp_dir);
	else
		temp_dir = gnome_util_prepend_user_home (".pan/tmp/");
	directory_check (temp_dir);

	/* get data directory */
	replace_gstr (&data_dir, gnome_config_get_string ("/Pan/Paths/data_dir"));
	if (is_nonempty_string (data_dir))
		ensure_trailing_slash (&data_dir);
	else
		data_dir = gnome_util_prepend_user_home (".pan/data/");
	directory_check (data_dir);

	/* get download directory */
	replace_gstr (&download_dir, gnome_config_get_string ("/Pan/Paths/download_dir"));
	if (is_nonempty_string (download_dir))
		ensure_trailing_slash (&download_dir);
	else
		download_dir = gnome_util_prepend_user_home ("/News/Pan/");
	directory_check (download_dir);


	/* (Set TMPDIR env var, so that uulib uses the right tmpdir) */
	pch = g_strdup_printf ("TMPDIR=%s", temp_dir);
        putenv (pch);
	replace_gstr (&static_tmp_dir, pch); /* purify says not to free pch */
	pch = NULL;

	/* general settings */

	text_set_wrap (gnome_config_get_bool("/Pan/General/do_wrap=TRUE"));
	wrap_column = gnome_config_get_int ("/Pan/General/wrap_column=74");
	online_prompt = gnome_config_get_bool ("/Pan/General/online_prompt");
	use_gdk_fontset_load = gnome_config_get_bool ("/Pan/General/use_gdk_fontset_load=FALSE");
	fetch_new_on_group_load = gnome_config_get_bool ("/Pan/General/fetch_new_on_group_load=TRUE");

	grouplist_mb1_action = gnome_config_get_int ("/Pan/General/grouplist_mb1_action=1");
	grouplist_mb2_action = gnome_config_get_int ("/Pan/General/grouplist_mb2_action=0");
	articlelist_mb1_action = gnome_config_get_int ("/Pan/General/articlelist_mb1_action=1");
	articlelist_mb2_action = gnome_config_get_int ("/Pan/General/articlelist_mb2_action=0");

	auto_part_selection = gnome_config_get_bool ("/Pan/General/auto_part_selection=TRUE");
#if 0
	break_thread_when_subject_changes = gnome_config_get_bool (
		"/Pan/General/break_thread_when_subject_changes=FALSE");
#endif
	display_article_dates_in_local_time = gnome_config_get_bool ("/Pan/General/display_article_dates_in_local_time=FALSE");
	hide_mpart_child_nodes = gnome_config_get_bool ("/Pan/General/hide_mpart_child_nodes=TRUE");
	going_online_preference = gnome_config_get_int ("/Pan/General/Going_Online=0");
	header_flags = (gulong) gnome_config_get_int ("/Pan/State/Headers=243");

	/* user settings */
	replace_gstr (&extra_posting_headers, gnome_config_get_string ("/Pan/Posting/Extra_Headers"));
	if (extra_posting_headers == NULL)
		extra_posting_headers = g_strdup ("X-No-Productlinks: Yes");

	replace_gstr (&attribution_line, gnome_config_get_string ("/Pan/User/attribution_line"));
	if (!attribution_line)
		attribution_line = g_strdup (_("In article %i, %a wrote:"));

	replace_gstr (&external_mailer, gnome_config_get_string ("/Pan/User/external_mailer"));
	if (!external_mailer)
		external_mailer = g_strdup ("xterm -e mutt -s '%s' -i '%t' '%r'");

	replace_gstr (&external_editor, gnome_config_get_string ("/Pan/User/external_editor"));
	if (!external_editor)
		external_editor = g_strdup ("xterm -e vi '%t'");

	/* mail server settings */
	replace_gstr (&mail_server_address, gnome_config_get_string ("/Pan/Mail/smtp_address"));
	mail_server_port = gnome_config_get_int ("/Pan/Mail/smtp_port=25");

	/* display settings */
	replace_gstr (&grouplist_font, gnome_config_get_string ("/Pan/Display/grouplist_font"));
	replace_gstr (&articlelist_font, gnome_config_get_string ("/Pan/Display/articlelist_font"));
	replace_gstr (&message_body_font, gnome_config_get_string ("/Pan/Display/message_body_font"));
	set_color (&unread_color,     "unread_color",         0,     0, 47165);
	set_color (&read_color,       "read_color",        8995,  8995,  8995);
	set_color (&new_color,        "new_color",        65535,  8995,  8995);
	set_color (&killfile_color,   "killfile_color",   35723, 17733,  4883);
	set_color (&watched_color,    "watched_color",        0, 52428,     0);
	set_color (&text_fg_color,    "text_fg_color",        0,     0,     0);
	set_color (&text_bg_color,    "text_bg_color",    65335, 65335, 65335);
	set_color (&text_quoted_color,"text_quoted_color",    0,     0, 65335);
}

static void
prefs_select_row_cb (GtkCList *clist,
		     gint row,
		     gint column,
		     GdkEventButton *event,
		     gpointer user_data)
{
	gtk_notebook_set_page (GTK_NOTEBOOK (win->box->notebook), row);
}

static int
prefs_add_section_page (gchar     *name,
			GtkWidget *widget)
{
	gchar * text[2];

	text[0] = name;
	text[1] = NULL;

	gtk_notebook_append_page (GTK_NOTEBOOK(win->box->notebook),
	                          widget,
				  gtk_label_new (name));
	return gtk_clist_append (GTK_CLIST (sections_clist), text);

}

static void
prefs_create_clist (void)
{
	GtkCList * list = GTK_CLIST(win->server_clist);
	GSList * l;

	/* build the clist */
	gtk_clist_freeze (list);
	gtk_clist_clear (list);
	for (l=Pan.serverlist; l!=NULL; l=l->next)
	{
		Server * server = SERVER(l->data);
		prefs_server * p_server;
		gchar * text[2];
		int row;

		/* skip the internal folder... */
		if (!pan_strcmp (server->name, INTERNAL_SERVER_NAME))
			continue;

		p_server = g_new0 (prefs_server, 1);
		p_server->old = server;
		text[0] = server->name ? server->name : _("Not Named");
		text[1] = server->address ? server->address : _("No Address");
		row = gtk_clist_prepend (list, text);
		gtk_clist_set_row_data (list, row, p_server);
	}
	gtk_clist_thaw (list);
}


static void
edit_profile_dialog_clicked (GnomeDialog     * dialog,
                             int               button,
                             prefs_server    * p_server)
{
	server_profile *sprof = gtk_object_get_data(GTK_OBJECT(dialog),"sprof");

	g_assert (sprof);

	if (button == 0)
	{
		gint row;
		Server *server;

		/* OK button */
		server = server_new ();
		server->name = gtk_editable_get_chars (GTK_EDITABLE (sprof->profile_name_entry), 0, -1);
		server->address = gtk_editable_get_chars (GTK_EDITABLE (sprof->server_address), 0, -1);
		server->port = atoi((gtk_entry_get_text (GTK_ENTRY(sprof->server_port))));
		server->need_auth = (!GTK_TOGGLE_BUTTON (sprof->auth_cbutton)->active) ? 0 : 1;
		server->gen_msgid = (!GTK_TOGGLE_BUTTON (sprof->msgid_cbutton)->active) ? 0 : 1;
		server->username = gtk_editable_get_chars (GTK_EDITABLE (sprof->server_username), 0, -1);
		server->password = gtk_editable_get_chars (GTK_EDITABLE (sprof->server_password), 0, -1);
		server->max_connections = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(sprof->max_connections));
		server->idle_secs_before_timeout = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (sprof->idle_secs_before_timeout));

		if (p_server == NULL)
		{
			gchar *text[2];
			GtkCList * list = GTK_CLIST(win->server_clist);

			p_server = g_new0 (prefs_server, 1);
			p_server->new = server;
			text[0] = server->name ? server->name : _("Not Named");
			text[1] = server->address ? server->address : _("No Address");
			row = gtk_clist_append (list, text);
			gtk_clist_set_row_data (list, row, p_server);
		}
		else
		{
			GtkCList * list = GTK_CLIST(win->server_clist);
			if (p_server->new)
				pan_object_unref (PAN_OBJECT(p_server->new));
			p_server->new = server;
			row = gtk_clist_find_row_from_data (list, p_server);
			gtk_clist_set_text (list, row, 1, server->address);
			gtk_clist_set_text (list, row, 0, server->name);
		}
	}

	gtk_widget_destroy (GTK_WIDGET (dialog));
	sprof_list = g_slist_remove (sprof_list, dialog);
}



static void
auth_cbutton_set (GtkWidget* widget, server_profile *sprof)
{
	const gboolean b = GTK_TOGGLE_BUTTON(widget)->active;
	servers_changed = TRUE;
	gtk_widget_set_sensitive (sprof->server_username_label, b);
	gtk_widget_set_sensitive (sprof->server_username, b);
	gtk_widget_set_sensitive (sprof->server_password_label, b);
	gtk_widget_set_sensitive (sprof->server_password, b);
}

static void
edit_profile_changed (GnomeDialog *d)
{
        servers_changed = TRUE;
        gnome_dialog_set_sensitive (d, 0, TRUE);
        gnome_property_box_changed (win->box);
}

/*---[ edit_profile_dialog ]------------------------------------------
 * The dialog for editing a server profile.
 *--------------------------------------------------------------------*/
static void
edit_profile_dialog (prefs_server *p_server)
{
	GtkWidget * w;
	GtkWidget * dialog;
	GtkWidget * table;
	GtkWidget * frame;
	GtkWidget * vbox;
	GtkWidget * hbox;
	GtkAdjustment * adj;
	Server * server;
	server_profile * sprof;

	if (p_server)
		if (p_server->new)
			server = p_server->new;
		else
			server = p_server->old;
	else 
		server = NULL;

	sprof = g_new0 (server_profile, 1);
	
	dialog = gnome_dialog_new (_("New/Edit Server"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL, NULL);
	
	gtk_object_set_data_full (GTK_OBJECT(dialog), "sprof", sprof, g_free);

	gnome_dialog_set_sensitive (GNOME_DIALOG(dialog), 0, FALSE);

	gtk_signal_connect (GTK_OBJECT(dialog), "clicked",
			    GTK_SIGNAL_FUNC (edit_profile_dialog_clicked),
			    p_server);

	table = gtk_table_new (4, 3, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);

	gtk_container_add (GTK_CONTAINER(GNOME_DIALOG(dialog)->vbox), table);

	/**
	***  FRAME:  server information
	**/

	frame = gtk_frame_new (_("Server Information"));
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	/* profile name label */
	w = gtk_label_new (_("Profile Name"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_box_pack_start (GTK_BOX(hbox), w, TRUE, TRUE, 0);

	/* profile name entry field */
	w = gtk_entry_new ();
	if (server && server->name)
		gtk_entry_set_text (GTK_ENTRY(w), server->name);
	connect_signal_to_prefs_changed (w, "changed");
	connect_signal_to_profile_changed (w, "changed", dialog);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	sprof->profile_name_entry = w;

	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	/* server address label */
	w = gtk_label_new (_("Server Address"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox), w, TRUE, TRUE, 0);

	/* server address */
	w = gtk_entry_new ();
	if (server && server->address)
		gtk_entry_set_text (GTK_ENTRY(w), server->address);
	connect_signal_to_prefs_changed (w, "changed");
	connect_signal_to_profile_changed (w, "changed", dialog);
	gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
	sprof->server_address = w;
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	/* server port label */
	w = gtk_label_new (_("Server Port"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox), w, TRUE, TRUE, 0);

	/* server port */
	w = gtk_entry_new ();
	if (1) {
		int num = server && server->port ? server->port : 119;
		char buf[32];
		sprintf (buf, "%d", num);
		gtk_entry_set_text (GTK_ENTRY(w), buf);
	}
	connect_signal_to_prefs_changed (w, "changed");
	connect_signal_to_profile_changed (w, "changed", dialog);
	gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
	sprof->server_port = w;

	gtk_table_attach (GTK_TABLE(table), frame, 
			  0, 2, 0, 3,
			  GTK_EXPAND | GTK_FILL,
			  GTK_EXPAND | GTK_FILL,
			  0, 0);

/* FRAME: Authorization */
	frame = gtk_frame_new (_("Authorization"));
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
	
  /* needs autorization information checkbox */


	w = gtk_check_button_new_with_label (_("My server requires my username and password"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), server && server->need_auth);
	gtk_signal_connect (GTK_OBJECT(w), "toggled", GTK_SIGNAL_FUNC (auth_cbutton_set), sprof);
	connect_signal_to_prefs_changed (w, "toggled");
	connect_signal_to_profile_changed (w, "toggled", dialog);
	gtk_box_pack_start (GTK_BOX (vbox), w, FALSE, FALSE, 0);
	sprof->auth_cbutton = w;

	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	/* server username */
	w = gtk_label_new (_("Username"));
	gtk_box_pack_start (GTK_BOX(hbox), w, TRUE, TRUE, 0);
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_widget_set_sensitive (w, server && server->need_auth);
	sprof->server_username_label = w;

	w = gtk_entry_new ();
	gtk_widget_set_sensitive (w, server && server->need_auth);
	if (server && server->username)
		gtk_entry_set_text (GTK_ENTRY(w), server->username);
	connect_signal_to_prefs_changed (w, "changed");
	connect_signal_to_profile_changed (w, "changed", dialog);
	gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
	sprof->server_username = w;

	/* server password */
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	w = gtk_label_new (_("Password"));
	gtk_widget_set_sensitive (w, server && server->need_auth);
	gtk_box_pack_start (GTK_BOX (hbox), w, TRUE, TRUE, 0);
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	sprof->server_password_label = w;

	w = gtk_entry_new ();
	gtk_widget_set_sensitive (w, server && server->need_auth);
	gtk_entry_set_visibility (GTK_ENTRY(w), FALSE);
	if (server && server->password)
		gtk_entry_set_text (GTK_ENTRY(w), server->password);
	connect_signal_to_prefs_changed (w, "changed");
	connect_signal_to_profile_changed (w, "changed", dialog);
	gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
	sprof->server_password = w;

	gtk_table_attach (GTK_TABLE(table), frame, 
			  0, 2, 4, 6,
			  GTK_EXPAND | GTK_FILL,
			  GTK_EXPAND | GTK_FILL,
			  0, 0);
	
	/* FRAME: Misc */
        frame = gtk_frame_new (_("Misc"));
        vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
        gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD_SMALL);
        gtk_container_add (GTK_CONTAINER (frame), vbox);

	/* SPIN: Max Connections */
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	w = gtk_label_new (_("Maximum Connections:"));
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	adj = (GtkAdjustment *) gtk_adjustment_new (4.0, 1.0, 4.0, 1.0, 1.0, 0.0);
	sprof->max_connections = gtk_spin_button_new (adj, 0, 0);
	if (server && server->max_connections)
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (sprof->max_connections),
					   server->max_connections);

	connect_signal_to_prefs_changed (adj, "value_changed");
	connect_signal_to_profile_changed (adj, "value_changed", dialog);
	gtk_box_pack_start (GTK_BOX(hbox), sprof->max_connections, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);


	/* SPIN: Idle Time */
        hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	w = gtk_label_new (_("Idle Seconds Before Timeout:"));
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	adj = (GtkAdjustment *) gtk_adjustment_new (180.0, 1.0, 360.0, 10.0, 10.0, 0.0);
	sprof->idle_secs_before_timeout = gtk_spin_button_new (adj, 0, 0);

	if (server)  {
		gtk_spin_button_set_value (
			GTK_SPIN_BUTTON (sprof->idle_secs_before_timeout),
			server->idle_secs_before_timeout);
        }

	connect_signal_to_prefs_changed (adj, "value_changed");
	connect_signal_to_profile_changed (adj, "value_changed", dialog);
	gtk_box_pack_start (GTK_BOX (hbox), sprof->idle_secs_before_timeout, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

        /* TOGGLE: Generate Message-ID */
        w = gtk_check_button_new_with_label (_("Generate Message-ID"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), server && server->gen_msgid);
	connect_signal_to_prefs_changed (w, "toggled");
	connect_signal_to_profile_changed (w, "toggled", dialog);
        gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, 0);
        sprof->msgid_cbutton = w;

        gtk_table_attach (GTK_TABLE(table), frame,
                          0, 2, 7, 9,
                          GTK_EXPAND | GTK_FILL,
                          GTK_EXPAND | GTK_FILL,
                          0, 0);

	sprof_list = g_slist_prepend (sprof_list, dialog);

	gtk_widget_show_all (dialog);
}


static void
server_clist_button_press (GtkWidget *widget, GdkEventButton *bevent)
{
	int row;
	int column;
	prefs_server *p_server = NULL;

	if (bevent->button != 1)
		gtk_signal_emit_stop_by_name (GTK_OBJECT(widget),
					      "button_press_event");

	if ((bevent->button) == 1 && (bevent->type == GDK_2BUTTON_PRESS))
	{
		gtk_signal_emit_stop_by_name (GTK_OBJECT(widget),
					      "button_press_event");
		gtk_clist_get_selection_info (GTK_CLIST(widget), bevent->x,
					      bevent->y, &row, &column);
		p_server = gtk_clist_get_row_data (GTK_CLIST(widget), row);
		edit_profile_dialog (p_server);
	}
}


static void
new_server_cb (void)
{
	edit_profile_dialog (NULL);
}


static void
edit_server_cb (void)
{
	GList * list = GTK_CLIST(win->server_clist)->selection;
	if (list != NULL)
	{
		prefs_server * p_server = gtk_clist_get_row_data (
			GTK_CLIST (win->server_clist),
			GPOINTER_TO_INT (list->data));
		edit_profile_dialog (p_server);
	}
}


static void
delete_server_cb (void)
{
	GtkCList * clist = GTK_CLIST(win->server_clist);
	GList * l = clist->selection;
	if (l != NULL)
	{
		const int row = GPOINTER_TO_INT(l->data);
		prefs_server * p_server = gtk_clist_get_row_data (clist, row);

		server_deletes = g_slist_prepend (server_deletes, p_server);
		gtk_clist_remove (clist, row);
		gnome_property_box_changed (win->box);
		servers_changed = TRUE;
	}
}


/* Cancel button pressed callback */
static void
prefs_cancel_cb (void)
{
	gint r = 0;
	prefs_server * p_server;

	while ((p_server = gtk_clist_get_row_data (GTK_CLIST(win->server_clist), r))) {
		if (p_server->new)
			pan_object_unref (PAN_OBJECT(p_server->new));
		g_free (p_server);
		++r;
	}

	/* cleanup widgets */
	g_slist_foreach (sprof_list, (GFunc)gtk_widget_destroy, NULL);
	g_slist_free (sprof_list);
	sprof_list = NULL;

	/* cleanup main window */
	gtk_widget_destroy (GTK_WIDGET(win->box));
	g_free (win);
	win = NULL;
}


/*--------------------------------------------------------------------
 * 'general' properties: toggles and such that relate to general
 * functionality and operation of Pan
 *--------------------------------------------------------------------*/
static GtkWidget*
prefs_general_page (void)
{
	int row = 0;
	GtkWidget *w;
	GtkWidget *table;
	table = gtk_table_new (18, 1, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);

	/* gdk_font_load */
	w = gtk_check_button_new_with_label (_("Use gdk_fontset_load instead of gdk_font_load"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), use_gdk_fontset_load);
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w,
		_("Toggling this and restarting Pan is a good idea if your fonts aren't being displayed properly."), NULL);
	win->use_gdk_fontset_load_checkbutton = w;

	/* auto part selection */
	w = win->auto_part_selection =
		gtk_check_button_new_with_label (
			_("Automatically guess a part when duplicate parts "
			  "appear in a multipart post"));
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(w), auto_part_selection);
	connect_signal_to_prefs_changed (w, "toggled");

	row = 0;

	gtk_table_attach (GTK_TABLE(table),
			  win->use_gdk_fontset_load_checkbutton,
			  0, 1, row, row+1,
			  GTK_FILL|GTK_EXPAND, GTK_FILL,
			  4, 0);

	++row;

	gtk_table_attach (GTK_TABLE(table),
			  win->auto_part_selection,
			  0, 1, row, row+1,
			  GTK_FILL|GTK_EXPAND, GTK_FILL,
			  4, 0);

	++row;

	w = gtk_hseparator_new();
	gtk_table_attach (
		GTK_TABLE(table), w,
		0, 1, row, row+1,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		0, GNOME_PAD_BIG);

	++row;

	w = gtk_label_new (
		_("Attribution Line (%i for message-id, %a for author, %d for date)"));
	gtk_misc_set_alignment (
		GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (
		GTK_TABLE(table), w,
		0, 1, row, row+1,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		0, 0);

	++row;

	win->attribution_entry = gtk_entry_new();
	gtk_entry_set_text (GTK_ENTRY(win->attribution_entry), attribution_line);
	connect_signal_to_prefs_changed (win->attribution_entry, "changed");
	gtk_table_attach (
		GTK_TABLE(table),
		win->attribution_entry,
		0, 1, row, row+1,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		0, 0);

	++row;

	w = gtk_hseparator_new();
	gtk_table_attach (
		GTK_TABLE(table), w,
		0, 1, row, row+1,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		0, GNOME_PAD_BIG);

	++row;

	gtk_table_attach (GTK_TABLE(table),
			  gtk_label_new(_("External Mail Program (%r for recipient, %s for subject, %t for body filename)")),
			  0, 1, row, row+1,
			  GTK_FILL|GTK_EXPAND, GTK_FILL,
			  0, 0);

	++row;

	win->external_mailer_entry = gtk_entry_new();
	gtk_entry_set_text (GTK_ENTRY(win->external_mailer_entry), external_mailer);
	connect_signal_to_prefs_changed (win->external_mailer_entry, "changed");
	gtk_table_attach (GTK_TABLE(table),
			  win->external_mailer_entry,
			  0, 1, row, row+1,
			  GTK_FILL|GTK_EXPAND, GTK_FILL,
			  0, 0);

	++row;

	gtk_table_attach (
		GTK_TABLE(table),
		gtk_label_new(_("External Editor (%t for temporary filename)")),
		0, 1, row, row+1,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		0, 0);

	++row;

	win->external_editor_entry = gtk_entry_new();
	gtk_entry_set_text (
		GTK_ENTRY(win->external_editor_entry), external_editor);
	connect_signal_to_prefs_changed (win->external_editor_entry, "changed");
	gtk_table_attach (
		GTK_TABLE(table),
		win->external_editor_entry,
		0, 1, row, row+1,
		GTK_FILL|GTK_EXPAND, GTK_FILL,
		0, 0);

	++row;

	return table;
}




static GtkWidget*
create_font_picker (const gchar * name, const char * config_key, GtkWidget ** setme_picker)
{
	GtkWidget * picker;
	GtkWidget * label;
	GtkWidget * hbox;
	gchar * pch;

	/* create the label */
	label = gtk_label_new (name);

	/* create the picker */
	*setme_picker = picker = gnome_font_picker_new ();
	gnome_font_picker_set_mode (GNOME_FONT_PICKER(picker),
	                            GNOME_FONT_PICKER_MODE_FONT_INFO);
	gnome_font_picker_fi_set_use_font_in_label (GNOME_FONT_PICKER(picker),
	                                            TRUE,
	                                            DISPLAY_PREFS_PREVIEW_FONT_SIZE);
	pch = gnome_config_get_string (config_key);
	if (is_nonempty_string (pch))
		gnome_font_picker_set_font_name (GNOME_FONT_PICKER (picker), pch);
	g_free (pch);
	connect_signal_to_prefs_changed (picker, "font_set");

	/* create the container */
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(hbox), picker, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);

	return hbox;
}


static GtkWidget*
create_color_picker (const gchar * name, const GdkColor * c, GtkWidget ** setme_picker)
{
	GtkWidget * picker;
	GtkWidget * label;
	GtkWidget * hbox;

	/* create the label */
	label = gtk_label_new (name);

	/* create the picker */
	*setme_picker = picker = gnome_color_picker_new ();
	gnome_color_picker_set_i16 (GNOME_COLOR_PICKER(picker), c->red, c->green, c->blue, 0);
	gtk_button_set_relief (GTK_BUTTON(picker), GTK_RELIEF_NONE);
	connect_signal_to_prefs_changed (picker, "color_set");

	/* create the container */
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(hbox), picker, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);

	return hbox;
}

/**
***  prefs_text_page
**/

static gulong
get_header_flags (void)
{
	GtkCList *clist = GTK_CLIST(win->article_headers_clist);
	GList* sel = clist->selection;
	gulong flags = 0;

	while ( sel != NULL ) {
		int row_index = GPOINTER_TO_INT ( sel->data );
		gulong* pulong = gtk_clist_get_row_data ( clist, row_index );
		flags |= *pulong;
		sel = sel->next;
	}

	return flags;
}

typedef struct
{
	gchar* text;
	gulong flag;
}
MessageListThrowawayStruct;

static GtkWidget*
prefs_text_page (void)
{
	GtkAdjustment * adj;
	GtkWidget * frame;
	GtkWidget * vbox;
	GtkWidget * w;
	GtkWidget * v;

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GNOME_PAD_SMALL);

	/**
	***  Fonts
	**/

	v = gtk_vbox_new (FALSE, 0);
	w = create_font_picker (_("Message Body"), "/Pan/Display/message_body_font", &win->message_gfp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, GNOME_PAD_SMALL);
	frame = gtk_frame_new (_("Fonts"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GNOME_PAD_SMALL);

	/**
	*** Colors
	**/

	v = gtk_vbox_new (FALSE, 0);
	w = create_color_picker (_("Body Background"), &text_bg_color, &win->text_bg_gcp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	w = create_color_picker (_("Body Foreground"), &text_fg_color, &win->text_fg_gcp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	w = create_color_picker (_("Quoted Text"), &text_quoted_color, &win->text_quoted_gcp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	frame = gtk_frame_new (_("Colors"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GNOME_PAD_SMALL);

	/**
	***  Fill
	**/

	v = gtk_hbox_new (FALSE, 0);

	w = gtk_check_button_new_with_label (_("Fill articles for reading at column"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), text_get_wrap());
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	win->wrap_column_tb = w;

	adj = (GtkAdjustment *) gtk_adjustment_new (256.0, 1.0, 256.0, 1.0, 1.0, 0.0);
	w = gtk_spin_button_new (adj, 0, 0);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON(w), wrap_column);
	connect_signal_to_prefs_changed (adj, "value_changed");
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	win->wrap_column_sb = w;

	frame = gtk_frame_new (_("Text Wrapping"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GNOME_PAD_SMALL);

	/**
	***  Headers
	**/

	if (1) {
		GtkCList* list = GTK_CLIST (gtk_clist_new(1));
		gint i;
		gulong flags;
		MessageListThrowawayStruct rows [] = {
			{NULL, UI_HEADER_AUTHOR},
			{NULL, UI_HEADER_DATE},
			{NULL, UI_HEADER_FOLLOWUP_TO},
			{NULL, UI_HEADER_MESSAGE_ID},
			{NULL, UI_HEADER_NEWSGROUPS},
			{NULL, UI_HEADER_REFERENCES},
			{NULL, UI_HEADER_REPLY_TO},
			{NULL, UI_HEADER_SUBJECT}
		};
		const int row_qty = sizeof(rows) / sizeof(rows[0]);
		flags = (gulong) gnome_config_get_int_with_default (
			"/Pan/State/Headers=243", NULL);
		rows[0].text = _("Author");
		rows[1].text = _("Date");
		rows[2].text = _("Followup-To");
		rows[3].text = _("Message-Id");
		rows[4].text = _("Newsgroups (if more than one group)");
		rows[5].text = _("References (if article is a reply)");
		rows[6].text = _("Reply-To (if different from Author)");
		rows[7].text = _("Subject");
		win->article_headers_clist = GTK_WIDGET(list);
		gtk_clist_set_selection_mode (list, GTK_SELECTION_MULTIPLE);

		for (i=0; i!=row_qty; ++i)
		{
			gint row = gtk_clist_insert (list, -1, &rows[i].text);

			gulong* plong = g_new0 (gulong, 1);
			*plong = rows[i].flag;
			gtk_clist_set_row_data_full (list, row, plong, g_free);

			if (rows[i].flag & flags)
				gtk_clist_select_row ( list, row, -1 );
		}

		connect_signal_to_prefs_changed (list, "select-row");
		connect_signal_to_prefs_changed (list, "unselect-row");
		v = gtk_vbox_new (FALSE, 0);
		gtk_container_set_border_width (GTK_CONTAINER(v), GNOME_PAD);
		gtk_box_pack_start (GTK_BOX(v), GTK_WIDGET(list), FALSE, FALSE, 0);
		frame = gtk_frame_new (_("Headers to Show"));
		gtk_container_add (GTK_CONTAINER(frame), v);
		gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GNOME_PAD_SMALL);
	}

	return vbox;
}


static GtkWidget*
prefs_articlelist_page (void)
{
	GtkWidget * frame;
	GtkWidget * vbox;
	GtkWidget * w;
	GtkWidget * v;
	GtkWidget * b;

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GNOME_PAD_SMALL);

	/**
	***  Mouse buttons
	**/

	v = gtk_vbox_new (FALSE, 0);

	b = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	w = gtk_label_new (_("Mouse Button 1:"));
	gtk_box_pack_start (GTK_BOX(b), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label (NULL, _("Selects Article"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), articlelist_mb1_action==CLICK_ACTION_SELECT);
	win->articlelist_mb1_action_select_tb = w;
	gtk_box_pack_start (GTK_BOX(b), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(w), _("Loads Article"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), articlelist_mb1_action==CLICK_ACTION_LOAD);
	win->articlelist_mb1_action_load_tb = w;

	gtk_box_pack_start (GTK_BOX(b), w, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(v), b, FALSE, FALSE, 0);

	b = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	w = gtk_label_new (_("Mouse Button 2:"));
	gtk_box_pack_start (GTK_BOX(b), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label (NULL, _("Selects Article"));
	win->articlelist_mb2_action_select_tb = w;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), articlelist_mb2_action==CLICK_ACTION_SELECT);
	gtk_box_pack_start (GTK_BOX(b), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(w), _("Loads Article"));
	win->articlelist_mb2_action_load_tb = w;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), articlelist_mb2_action==CLICK_ACTION_LOAD);
	gtk_box_pack_start (GTK_BOX(b), w, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(v), b, FALSE, FALSE, 0);

	connect_signal_to_prefs_changed (win->articlelist_mb1_action_select_tb, "toggled");
	connect_signal_to_prefs_changed (win->articlelist_mb1_action_load_tb,   "toggled");
	connect_signal_to_prefs_changed (win->articlelist_mb2_action_select_tb, "toggled");
	connect_signal_to_prefs_changed (win->articlelist_mb2_action_load_tb,   "toggled");

	frame = gtk_frame_new (_("Selection"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_container_set_border_width (GTK_CONTAINER(v), GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GNOME_PAD_SMALL);

	/**
	***  Fonts
	**/

	v = gtk_vbox_new (FALSE, 0);
	w = create_font_picker (_("Article List Font"), "/Pan/Display/articlelist_font", &win->articlelist_gfp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, GNOME_PAD_SMALL);
	frame = gtk_frame_new (_("Fonts"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GNOME_PAD_SMALL);

	/**
	***  Colors
	**/

	v = gtk_vbox_new (FALSE, 0);
	w = create_color_picker (_("Unread Messages"), &unread_color, &win->unread_gcp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	w = create_color_picker (_("Read Messages"), &read_color, &win->read_gcp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	w = create_color_picker (_("New Messages"), &new_color, &win->new_gcp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	w = create_color_picker (_("Ignored Threads"), &killfile_color, &win->ignored_gcp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	w = create_color_picker (_("Watched Threads"), &watched_color, &win->watched_gcp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	frame = gtk_frame_new (_("Colors"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_container_set_border_width (GTK_CONTAINER(v), GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GNOME_PAD_SMALL);

	/**
	***  General
	**/

	v = gtk_vbox_new (FALSE, 0);

#if 0
	/* break threads when subject changes? */
	w = gtk_check_button_new_with_label (_("When a followup changes the subject, show as new thread"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), break_thread_when_subject_changes);
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	win->break_thread_when_subject_changes_cbutton = w;
#endif

	/* utc or local time */
	w = gtk_check_button_new_with_label (_("Display article dates in local time instead of UTC"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), display_article_dates_in_local_time);
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	win->display_article_dates_in_local_time_cbutton = w;

	/* multipart: one node or all */
	w = gtk_check_button_new_with_label (_("Show only the first part of complete multipart binary posts"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), hide_mpart_child_nodes);
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	win->hide_mpart_child_nodes_cbutton = w;

	frame = gtk_frame_new (_("Display"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_container_set_border_width (GTK_CONTAINER(v), GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GNOME_PAD_SMALL);

	return vbox;
}



/**
***
***  LAYOUT
***
**/

static gchar*
layout_get_new_string (GtkWidget * page)
{
	int i;
	int layout = -1;
	char str[8];
	GtkCList * clist = GTK_CLIST(gtk_object_get_data (GTK_OBJECT(page), "order_clist"));
	GtkWidget ** layout_buttons = (GtkWidget**) gtk_object_get_data (GTK_OBJECT(page), "layout_buttons");

	for (i=0; i<5; ++i) {
		GtkToggleButton * tb = GTK_TOGGLE_BUTTON(layout_buttons[i]);
		if (gtk_toggle_button_get_active(tb))
			layout = i + 1;
	}

	sprintf (str, "%d%c%c%c", layout,
		(gchar) GPOINTER_TO_INT (gtk_clist_get_row_data (clist, 0)),
		(gchar) GPOINTER_TO_INT (gtk_clist_get_row_data (clist, 1)),
		(gchar) GPOINTER_TO_INT (gtk_clist_get_row_data (clist, 2)));

	return g_strdup (str);
}

static void
layout_changed_cb (GtkToggleButton * togglebutton, gpointer user_data)
{
	GtkWidget ** layout_buttons = (GtkWidget**) user_data;
	static gboolean dampen_feedback_loop = FALSE;

	if (!dampen_feedback_loop) {
		int i;
		dampen_feedback_loop = TRUE;
		for (i=0; i<5; ++i) {
			GtkToggleButton * tb = GTK_TOGGLE_BUTTON(layout_buttons[i]);
			gboolean active = togglebutton==tb;
			if (gtk_toggle_button_get_active(tb) != active)
				gtk_toggle_button_set_active (tb, active);
		}
		dampen_feedback_loop = FALSE;
	}
}

static const gchar*
layout_get_pane_name (char ch)
{
	switch (ch) {
		case 'g': case 'G': return _("Grouplist Pane");
		case 'a': case 'A': return _("Article Pane");
		case 't': case 'T': return _("Thread Pane");
		default: return _("Error");
	}
}

static void
layout_update_clist_row (GtkCList * clist, int row)
{
	char ch = (char)GPOINTER_TO_INT(gtk_clist_get_row_data(clist,row));
	const char * pane_name = layout_get_pane_name (ch);
	char buf[512];
	sprintf (buf, "%d. %s", (row+1), pane_name);
	gtk_clist_set_text (clist, row, 0, buf);
}

static void
layout_arrow_up_cb (GtkButton * button, gpointer user_data)
{
	GtkCList * clist = GTK_CLIST (user_data);
	GList * sel = clist->selection;
	int row = -1;
	if (sel != NULL)
		row = GPOINTER_TO_INT (sel->data);
	if (row > 0) {
		gtk_clist_row_move (clist, row, row-1);
		layout_update_clist_row (clist, row-1);
		layout_update_clist_row (clist, row);
	}
}

static void
layout_arrow_down_cb (GtkButton * button, gpointer user_data)
{
	GtkCList * clist = GTK_CLIST (user_data);
	GList * sel = clist->selection;
	int row = -1;
	if (sel != NULL)
		row = GPOINTER_TO_INT (sel->data);
	if (row>-1 && row<clist->rows) {
		gtk_clist_row_move (clist, row, row+1);
		layout_update_clist_row (clist, row);
		layout_update_clist_row (clist, row+1);
	}
}

static GtkWidget*
prefs_layout_page (void)
{
	GtkWidget * hbox;
	GtkWidget * bbox;
	GtkWidget * vbox;
	GtkWidget * w;
	GtkWidget * icon;
	GtkWidget * clist;
	const char lch = *layout_str;
	const char * pch;
	int row;
	gchar buf [64];
	gchar * text [] = { buf };
	GtkWidget ** layout_buttons = g_malloc (sizeof(GtkWidget*) * 5);

	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);

	w = gtk_toggle_button_new ();
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), lch=='1');
	icon = gnome_pixmap_new_from_xpm_d (layout_1_xpm);
	gtk_container_add (GTK_CONTAINER(w), icon);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT(w), "toggled", GTK_SIGNAL_FUNC(layout_changed_cb), layout_buttons);
	connect_signal_to_prefs_changed (w, "toggled");
	layout_buttons[0] = w;

	w = gtk_toggle_button_new ();
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), lch=='2');
	icon = gnome_pixmap_new_from_xpm_d (layout_2_xpm);
	gtk_container_add (GTK_CONTAINER(w), icon);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT(w), "toggled", GTK_SIGNAL_FUNC(layout_changed_cb), layout_buttons);
	connect_signal_to_prefs_changed (w, "toggled");
	layout_buttons[1] = w;

	w = gtk_toggle_button_new ();
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), lch=='3');
	icon = gnome_pixmap_new_from_xpm_d (layout_3_xpm);
	gtk_container_add (GTK_CONTAINER(w), icon);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT(w), "toggled", GTK_SIGNAL_FUNC(layout_changed_cb), layout_buttons);
	connect_signal_to_prefs_changed (w, "toggled");
	layout_buttons[2] = w;

	w = gtk_toggle_button_new ();
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), lch=='4');
	icon = gnome_pixmap_new_from_xpm_d (layout_4_xpm);
	gtk_container_add (GTK_CONTAINER(w), icon);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT(w), "toggled", GTK_SIGNAL_FUNC(layout_changed_cb), layout_buttons);
	connect_signal_to_prefs_changed (w, "toggled");
	layout_buttons[3] = w;

	w = gtk_toggle_button_new ();
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), lch=='5');
	icon = gnome_pixmap_new_from_xpm_d (layout_5_xpm);
	gtk_container_add (GTK_CONTAINER(w), icon);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT(w), "toggled", GTK_SIGNAL_FUNC(layout_changed_cb), layout_buttons);
	connect_signal_to_prefs_changed (w, "toggled");
	layout_buttons[4] = w;

	/* panes clist */
	clist = gtk_clist_new (1);
	gtk_clist_set_column_width (GTK_CLIST(clist), 0, 120);

	pch = layout_str + 1;
	sprintf (buf, "1. %s", layout_get_pane_name(*pch));
	row = gtk_clist_insert (GTK_CLIST(clist), -1, text);
	gtk_clist_set_row_data (GTK_CLIST(clist), row, GINT_TO_POINTER((int)*pch));

	++pch;
	sprintf (buf, "2. %s", layout_get_pane_name(*pch));
	row = gtk_clist_insert (GTK_CLIST(clist), -1, text);
	gtk_clist_set_row_data (GTK_CLIST(clist), row, GINT_TO_POINTER((int)*pch));

	++pch;
	sprintf (buf, "3. %s", layout_get_pane_name(*pch));
	row = gtk_clist_insert (GTK_CLIST(clist), -1, text);
	gtk_clist_set_row_data (GTK_CLIST(clist), row, GINT_TO_POINTER((int)*pch));

	/* panes arrowbuttons */
	bbox = gtk_vbox_new (FALSE, 0);
	w = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER(w), gtk_arrow_new (GTK_ARROW_UP, GTK_SHADOW_OUT));
	gtk_box_pack_start (GTK_BOX(bbox), w, FALSE, TRUE, 0);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(layout_arrow_up_cb), clist);
	connect_signal_to_prefs_changed (w, "clicked");
	w = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER(w), gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT));
	gtk_box_pack_end (GTK_BOX(bbox), w, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(layout_arrow_down_cb), clist);
	connect_signal_to_prefs_changed (w, "clicked");
	gtk_box_pack_start (GTK_BOX(hbox), clist, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox), bbox, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	gtk_object_set_data (GTK_OBJECT(vbox), "order_clist", clist);
	gtk_object_set_data_full (GTK_OBJECT(vbox), "layout_buttons", layout_buttons, g_free);
	return vbox;
}

/**
***
***  GROUPLIST
***
**/

static GtkWidget*
prefs_grouplist_page (void)
{
	GtkWidget * frame;
	GtkWidget * vbox;
	GtkWidget * w;
	GtkWidget * v;
	GtkWidget * b;

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GNOME_PAD_SMALL);

	/**
	***  Mouse buttons
	**/

	v = gtk_vbox_new (FALSE, 0);

	b = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	w = gtk_label_new (_("Mouse Button 1:"));
	gtk_box_pack_start (GTK_BOX(b), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label (NULL, _("Selects Group"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), grouplist_mb1_action==CLICK_ACTION_SELECT);
	win->grouplist_mb1_action_select_tb = w;
	gtk_box_pack_start (GTK_BOX(b), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(w), _("Loads Group"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), grouplist_mb1_action==CLICK_ACTION_LOAD);
	win->grouplist_mb1_action_load_tb = w;
	gtk_box_pack_start (GTK_BOX(b), w, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(v), b, FALSE, FALSE, 0);

	b = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	w = gtk_label_new (_("Mouse Button 2:"));
	gtk_box_pack_start (GTK_BOX(b), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label (NULL, _("Selects Group"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), grouplist_mb2_action==CLICK_ACTION_SELECT);
	win->grouplist_mb2_action_select_tb = w;
	gtk_box_pack_start (GTK_BOX(b), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(w), _("Loads Group"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), grouplist_mb2_action==CLICK_ACTION_LOAD);
	win->grouplist_mb2_action_load_tb = w;
	gtk_box_pack_start (GTK_BOX(b), w, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(v), b, FALSE, FALSE, 0);
	connect_signal_to_prefs_changed (win->grouplist_mb1_action_select_tb, "toggled");
	connect_signal_to_prefs_changed (win->grouplist_mb1_action_load_tb,   "toggled");
	connect_signal_to_prefs_changed (win->grouplist_mb2_action_select_tb, "toggled");
	connect_signal_to_prefs_changed (win->grouplist_mb2_action_load_tb,   "toggled");

	frame = gtk_frame_new (_("Selection"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_container_set_border_width (GTK_CONTAINER(v), GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GNOME_PAD_SMALL);

	/**
	***  Fonts
	**/

	v = gtk_vbox_new (FALSE, 0);
	w = create_font_picker (_("Group list font"), "/Pan/Display/grouplist_font", &win->grouplist_gfp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, GNOME_PAD_SMALL);
	frame = gtk_frame_new (_("Fonts"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GNOME_PAD_SMALL);

	/**
	***  Options
	**/

	v = gtk_vbox_new (FALSE, 0);

	/* download new headers */
	w = gtk_check_button_new_with_label (_("Download New Headers when the Group is Loaded"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), fetch_new_on_group_load);
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	win->fetch_new_on_group_load_cbutton = w;

	frame = gtk_frame_new (_("Options"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_container_set_border_width (GTK_CONTAINER(v), GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GNOME_PAD_SMALL);

	/**
	***
	**/

	return vbox;
}


/*---[ prefs_connections_page ]---------------------------------------
 * Connections preferences page for setting the data, temp, and download
 * dirs in Pan.
 *--------------------------------------------------------------------*/

static GtkWidget*
inset (GtkWidget* w)
{
	GtkWidget* outer;
	GtkWidget* inner;

	inner = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(inner), GTK_SHADOW_NONE);
	gtk_container_set_border_width (GTK_CONTAINER(inner), GNOME_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER(inner), w);

	outer = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(outer), GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER(outer), inner);
	gtk_container_set_border_width (GTK_CONTAINER(inner), 0);

	return outer;
}

static void
online_clicked_cb (GtkButton *b, gpointer user_data)
{
	char * pch;
	server_connection * sc = (server_connection*) user_data;
	GtkWidget * w;

	sc->online_value = !sc->online_value;
	pch = sc->online_value
		? g_strdup_printf (_("Server \"%s\" is Online"), sc->server->name)
		: g_strdup_printf (_("Server \"%s\" is Offline; no new tasks will be processed"), sc->server->name);

	pan_lock ();
       	w = sc->online_button;
	gtk_container_remove (GTK_CONTAINER(w), GTK_BIN(w)->child);
	gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (sc->online_value?traffic_green_xpm:traffic_red_xpm));
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, pch, "");
	gtk_widget_show_all (w);
	pan_unlock ();

	gnome_property_box_changed (win->box);

	g_free (pch);
}

static GtkWidget*
prefs_connections_page (void)
{
	GtkAdjustment *adj;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *radiobox;
	GtkWidget *table;
	GtkWidget *frame;
	GtkWidget *w;
	GSList *l;
	int row;
	const int max_per_server =
		gnome_config_get_int ("/Pan/General/Max_Connections_Per_Server=4");

	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	/* max connections total */
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	w = gtk_label_new (_("Maximum Total Number of Connections"));
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	adj = (GtkAdjustment *) gtk_adjustment_new (128.0, 1.0, 128.0, 1.0, 1.0, 0.0);
	win->max_connections = gtk_spin_button_new (adj, 0, 0);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (win->max_connections), queue_get_max_sockets());
	connect_signal_to_prefs_changed (adj, "value_changed");
	gtk_box_pack_start (GTK_BOX(hbox), win->max_connections, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	/* task retries */
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	w = gtk_label_new (_("Maximum Number of Task Retries before Giving Up"));
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	adj = (GtkAdjustment *) gtk_adjustment_new (4.0, 1.0, 999.0, 1.0, 1.0, 0.0);
	win->max_tries = gtk_spin_button_new (adj, 0, 0);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (win->max_tries), queue_get_max_tries());
	connect_signal_to_prefs_changed (adj, "value_changed");
	gtk_box_pack_start (GTK_BOX(hbox), win->max_tries, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	/* ask before going online */
	radiobox = gtk_vbox_new (FALSE, 0);
	w = gtk_radio_button_new_with_label (NULL, _("Pan should ask before going online"));
	gtk_box_pack_start (GTK_BOX(radiobox), w, FALSE, FALSE, 0);
	win->online_prompt_first_cbutton = w;
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(w), _("Pan should go online as needed without asking"));
	gtk_box_pack_start (GTK_BOX(radiobox), w, FALSE, FALSE, 0);
	win->online_without_asking_cbutton = w;
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(w), _("Pan should quietly stay offline until told otherwise"));
	gtk_box_pack_start (GTK_BOX(radiobox), w, FALSE, FALSE, 0);
	win->online_never_cbutton = w;
	switch (going_online_preference) {
		case ONLINE_YES: w = win->online_without_asking_cbutton; break;
		case ONLINE_NO: w = win->online_never_cbutton; break;
		case ONLINE_PROMPT:
		default: w = win->online_prompt_first_cbutton; break;
	}
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), TRUE);
	gtk_box_pack_start (GTK_BOX(vbox), radiobox, FALSE, FALSE, 0);
	connect_signal_to_prefs_changed (win->online_without_asking_cbutton, "toggled");
	connect_signal_to_prefs_changed (win->online_prompt_first_cbutton, "toggled");
	connect_signal_to_prefs_changed (win->online_never_cbutton, "toggled");


	/* pack the vbox into a frame, which is itself the first element in a new vbox */
	frame = gtk_frame_new (_("Global Settings"));
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GNOME_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER(frame), vbox);
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_BIG);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	/* per-server settings */
	table = gtk_table_new (g_slist_length(Pan.serverlist)+1, 3, FALSE);
	w = gtk_label_new (_("News Server"));
	gtk_misc_set_padding (GTK_MISC(w), GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, 0, 1,  GTK_FILL, GTK_FILL, 0, 0);
	w = gtk_label_new (_("Maximum\nConnections"));
	gtk_misc_set_padding (GTK_MISC(w), GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, 0, 1,  GTK_FILL, GTK_FILL, 0, 0);
	w = gtk_label_new (_("Online Status"));
	gtk_misc_set_padding (GTK_MISC(w), GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_table_attach (GTK_TABLE (table), w, 2, 3, 0, 1,  GTK_FILL, GTK_FILL, 0, 0);
	w = gtk_label_new (_("Reserve one connection\nfor non-decode tasks"));
	gtk_misc_set_padding (GTK_MISC(w), GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_table_attach (GTK_TABLE (table), w, 3, 4, 0, 1,  GTK_FILL, GTK_FILL, 0, 0);
	for (row=1, l=Pan.serverlist; l!=NULL; l=l->next, ++row)
	{
		const Server* server = SERVER(l->data);
		server_connection * sc;
		char * pch;

		/* skip the internal server */
		if (!pan_strcmp(server->name, INTERNAL_SERVER_NAME))
			continue;

		sc = g_new0 (server_connection, 1);
		sc->server = server;

		/* server name */
		w = gtk_label_new (server->name);
		gtk_table_attach (GTK_TABLE(table), inset(w), 0, 1, row, row+1, GTK_FILL, GTK_FILL, 0, 0);

		/* max connections per server */
		adj = (GtkAdjustment *) gtk_adjustment_new (max_per_server, 1.0, max_per_server, 1.0, 1.0, 0.0);
		w = gtk_spin_button_new (adj, 0, 0);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON(w), server->max_connections);
		connect_signal_to_prefs_changed (adj, "value_changed");
		gtk_table_attach (GTK_TABLE(table), inset(w), 1, 2, row, row+1, GTK_FILL, GTK_FILL, 0, 0);
		sc->max_connections_sb = w;

		/* online/online */
		w = sc->online_button = gtk_button_new ();
		gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
		sc->online_value = server->online;
		pch = sc->online_value
			? g_strdup_printf (_("Server \"%s\" is Online"), server->name)
			: g_strdup_printf (_("Server \"%s\" is Offline; no new tasks will be processed"), server->name);
		gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, pch, "");
		g_free (pch);
		gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (server->online?traffic_green_xpm:traffic_red_xpm));
		gtk_signal_connect (GTK_OBJECT(w), "clicked", online_clicked_cb, sc);
		gtk_table_attach (GTK_TABLE(table), inset(w), 2, 3, row, row+1, GTK_FILL, GTK_FILL, 0, 0);

		/* reserve */
		w = gtk_check_button_new_with_label (_("Reserve"));
		sc->reserve_tb = w;
		gtk_table_attach (GTK_TABLE(table), inset(w), 3, 4, row, row+1, GTK_FILL, GTK_FILL, 0, 0);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), server->reserve_nonleech);
		connect_signal_to_prefs_changed (w, "toggled");

		/* remember this server */
		win->server_connections = g_slist_append (win->server_connections, sc);
	}

	frame = gtk_frame_new (_("Per-Server Settings"));
	gtk_container_set_border_width (GTK_CONTAINER(table), GNOME_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER(frame), table);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	return vbox;
}


/*---[ prefs_directories_page ]---------------------------------------
 * Directory preferences page for setting the data, temp, and download
 * dirs in Pan.
 *--------------------------------------------------------------------*/
static GtkWidget *
prefs_directories_page (void)
{
	GtkWidget *table;
	GtkWidget *label;
	gchar *p;

	table = gtk_table_new (3, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);

	label = gtk_label_new (_("Download Directory:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 0, 0);

	win->dir_download = gnome_file_entry_new ("Prefs", _("Download Directory"));
	if ((p = gnome_config_get_string ("/Pan/Paths/download_dir")) == NULL)
		p = gnome_util_prepend_user_home ("News/Pan/");
	gtk_entry_set_text (GTK_ENTRY(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (win->dir_download))), p);
	gnome_file_entry_set_directory (GNOME_FILE_ENTRY (win->dir_download), TRUE);

	g_free (p);
	connect_signal_to_prefs_changed (gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (win->dir_download)), "changed");
	gtk_table_attach (GTK_TABLE(table), win->dir_download, 1, 2, 0, 1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	label = gtk_label_new (_("Temporary Directory:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 0, 0);

	win->dir_temp = gnome_file_entry_new ("Prefs", _("Temporary Directory"));
	if ((p = gnome_config_get_string("/Pan/Paths/temp_dir")) == NULL) {
		p = gnome_util_prepend_user_home(".pan/tmp/");
	}
	gtk_entry_set_text (GTK_ENTRY(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (win->dir_temp))), p);
	gnome_file_entry_set_directory (GNOME_FILE_ENTRY (win->dir_temp), TRUE);
	g_free (p);
	connect_signal_to_prefs_changed (gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (win->dir_temp)), "changed");
	gtk_table_attach (GTK_TABLE(table), win->dir_temp, 1, 2, 1,
			  2, GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

/* Data Directory: */
	label = gtk_label_new (_("Data Directory:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 0, 0);

	win->dir_data = gnome_file_entry_new ("Prefs", _("Data Directory"));
	if ((p = gnome_config_get_string("/Pan/Paths/data_dir")) == NULL) {
		p = gnome_util_prepend_user_home(".pan/data/");
	}
	gtk_entry_set_text (GTK_ENTRY(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (win->dir_data))), p);
	gnome_file_entry_set_directory (GNOME_FILE_ENTRY (win->dir_data), TRUE);
	g_free (p);
	connect_signal_to_prefs_changed (gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (win->dir_data)), "changed");
	gtk_table_attach (GTK_TABLE(table), win->dir_data, 1, 2, 2,
			  3, GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	return table;
}

/**
***
***
**/

static GtkWidget *
prefs_headers_page (void)
{
	GtkWidget * label;

	label = gtk_label_new (_("Placeholder"));

	return label;
}

/*---[ prefs_userinfo_page ]------------------------------------------
 * page that holds the user information such as full name,
 * e-mail, organization, and signature file (this page is optional).
 *--------------------------------------------------------------------*/
static GtkWidget *
prefs_userinfo_page (void)
{
	GtkWidget * frame;
	GtkWidget * table;
	GtkWidget * label;
	char *p;

	table = gtk_table_new (5, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);

	label = gtk_label_new (_("Full Name:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 0, 0);

	win->user_fullname = gtk_entry_new ();
	if ((p = gnome_config_get_string("/Pan/User/Full_Name"))) {
		gtk_entry_set_text (GTK_ENTRY(win->user_fullname), p);
		g_free(p);
	}
	connect_signal_to_prefs_changed (win->user_fullname, "changed");
	gtk_table_attach (GTK_TABLE(table), win->user_fullname, 1, 2, 0, 1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	label = gtk_label_new (_("Email Address:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 0, 0);

	win->user_email = gtk_entry_new ();
	if ((p = gnome_config_get_string("/Pan/User/Email"))) {
		gtk_entry_set_text (GTK_ENTRY(win->user_email), p);
		g_free(p);
	}
	connect_signal_to_prefs_changed (win->user_email, "changed");
	gtk_table_attach (GTK_TABLE(table), win->user_email, 1, 2, 1, 2,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	label = gtk_label_new (_("Reply To:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
    gtk_table_attach (GTK_TABLE(table), label, 0, 1, 2, 3,
				GTK_FILL, GTK_FILL, 0, 0);

	win->user_reply_to = gtk_entry_new ();
    if ((p = gnome_config_get_string("/Pan/User/Reply_To"))) {
		gtk_entry_set_text (GTK_ENTRY(win->user_reply_to), p);
		g_free(p);
	}
	connect_signal_to_prefs_changed (win->user_reply_to, "changed");
	gtk_table_attach (GTK_TABLE(table), win->user_reply_to, 1,
				2, 2, 3, GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);
				
	label = gtk_label_new (_("Organization:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 3, 4,
			  GTK_FILL, GTK_FILL, 0, 0);

	win->user_organization = gtk_entry_new ();
	if ((p = gnome_config_get_string("/Pan/User/Organization"))) {
		gtk_entry_set_text (GTK_ENTRY(win->user_organization), p);
		g_free(p);
	}
	connect_signal_to_prefs_changed (win->user_organization, "changed");
	gtk_table_attach (GTK_TABLE(table), win->user_organization, 1,
			  2, 3, 4, GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	label = gtk_label_new (_("Signature File:"));
	gtk_misc_set_alignment (GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, 4, 5,
			  GTK_FILL, GTK_FILL, 0, 0);

	win->user_signature_file = gnome_file_entry_new ("Prefs", _("Signature File:"));
	if ((p = gnome_config_get_string("/Pan/User/Signature_File"))) {
		gtk_entry_set_text (GTK_ENTRY(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (win->user_signature_file))), p);
		g_free(p);
	}
	connect_signal_to_prefs_changed (gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(win->user_signature_file)), "changed");
	gtk_table_attach (GTK_TABLE(table), GTK_WIDGET (win->user_signature_file), 1, 2, 4, 5,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, 4, 0);

	/* frame */
	frame = gtk_frame_new (_("User Identity"));
	gtk_container_add (GTK_CONTAINER(frame), table);

	return frame;
}



static void
clear_acache_clicked_cb (GtkButton *button, gpointer data)
{
	const int i = acache_expire_all ();
	if (i != 0)
	{
		gchar* buf = g_strdup_printf ("%d files deleted.", i);
		gnome_ok_dialog (buf);
		g_free (buf);
	}
}

/* Build and return the 'Cache' page */
static GtkWidget*
prefs_cache_page (void)
{
	const int cache_max = gnome_config_get_int_with_default("/Pan/Cache/MaxSize=1242880", NULL);
	const gboolean flush_on_exit = gnome_config_get_bool("/Pan/Cache/FlushOnExit=true");
	GtkObject * adj;
	GtkWidget * frame;
	GtkWidget * label;
	GtkWidget * vbox = gtk_vbox_new (FALSE, GNOME_PAD);
	GtkWidget * table;
	GtkWidget * hbox;
	GtkWidget * button;
	GtkWidget * tb;
	GtkWidget * w;

	label = gtk_label_new (_("The cache is used to keep local copies of articles\nto reduce the time connected to the network."));
	gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD_SMALL);

	table = gtk_table_new (3, 3, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);
	label = gtk_label_new (_("Disk Cache:"));
	gtk_table_attach_defaults (GTK_TABLE(table), label, 0, 1, 0, 1);

	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	adj = gtk_adjustment_new (cache_max/(1024*1024), 0.1, 1000, 0.1, 0.1, 0.1);
	win->cache_megs_sb = w = gtk_spin_button_new (GTK_ADJUSTMENT(adj), 0.1, 1);
	connect_signal_to_prefs_changed (w, "changed");
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, GNOME_PAD_SMALL);
	label = gtk_label_new (_("Megs"));
	gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, GNOME_PAD_SMALL);
	gtk_table_attach_defaults (GTK_TABLE(table), hbox, 1, 2, 0, 1);

	button = gtk_button_new_with_label (_("Clear Disk Cache"));
	gtk_signal_connect (GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC (clear_acache_clicked_cb),
			    NULL);
	gtk_table_attach_defaults (GTK_TABLE(table), button, 2, 3, 0, 1);
	gtk_box_pack_start (GTK_BOX(vbox), table, FALSE, FALSE, GNOME_PAD_SMALL);

	tb = win->flush_cache_on_exit_check =
		gtk_check_button_new_with_label (_("Delete Cache when exiting Pan"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(tb), flush_on_exit);
	connect_signal_to_prefs_changed (tb, "toggled");
	gtk_box_pack_start (GTK_BOX(vbox), tb, FALSE, FALSE, GNOME_PAD_SMALL);

	/* frame */
	frame = gtk_frame_new (_("Article Cache"));
	gtk_container_add (GTK_CONTAINER(frame), vbox);

	return frame;
}

/* Build and return the 'SMTP Server' page */
static GtkWidget *
prefs_smtp_page (void)
{
	GtkWidget * frame;
	GtkWidget * table;
	GtkWidget * w;

	table = gtk_table_new (2, 2, FALSE);
        gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD);
        gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);

	/* address */
	w = gtk_label_new (_("Address:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 4, 0);
	win->smtp_address = w = gtk_entry_new ();
	if (is_nonempty_string(mail_server_address))
		gtk_entry_set_text (GTK_ENTRY(w), mail_server_address);
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, 0, 1, GTK_FILL|GTK_EXPAND, GTK_FILL, 4, 0);
	connect_signal_to_prefs_changed (w, "changed");

	/* port */
	w = gtk_label_new (_("Port:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 4, 0);
	win->smtp_port = w = gtk_spin_button_new (
		GTK_ADJUSTMENT(gtk_adjustment_new (mail_server_port, 0, 65536, 1, 1, 1)), 1, 0);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 1, 2,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, 4, 0);
	connect_signal_to_prefs_changed (w, "changed");

	/* frame */
	frame = gtk_frame_new (_("Mail Server"));
	gtk_container_add (GTK_CONTAINER(frame), table);

	return frame;
}

/*---[ prefs_server_page ]--------------------------------------------
 * This is the Server tab of the preferences.  It has a list of the
 * server's and buttons that allow users to make a new server entry,
 * edit a previous one, or delete one.  This doesn't actually let
 * them switch servers, that's done from the main menu of Pan.
 *--------------------------------------------------------------------*/
static GtkWidget *
prefs_server_page (void)
{
	GtkWidget *vbox, *hbox, *button;

	char *titles[2];
	titles[0] = _("Name");
	titles[1] = _("Address");

	hbox = gtk_hbox_new (FALSE, 0);

	win->server_clist = gtk_clist_new_with_titles (2, titles);
	gtk_clist_column_titles_passive (GTK_CLIST(win->server_clist));
	gtk_clist_set_selection_mode (GTK_CLIST(win->server_clist),
				      GTK_SELECTION_SINGLE);

	gtk_clist_set_column_width (GTK_CLIST(win->server_clist), 0, 100);
	gtk_clist_set_column_auto_resize (GTK_CLIST(win->server_clist), 1, TRUE);

	gtk_widget_set_usize (win->server_clist, 280, 200);
	gtk_box_pack_start (GTK_BOX(hbox), win->server_clist, TRUE,
			    TRUE, 5);
	gtk_signal_connect (GTK_OBJECT(win->server_clist),
			    "button_press_event",
			    GTK_SIGNAL_FUNC(server_clist_button_press), NULL);

	prefs_create_clist ();

	/* one button vbox */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox), vbox, FALSE, FALSE, 5);

	/* new server button */
	button = gtk_button_new_with_label (_("New..."));
	gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, GNOME_PAD);
	gtk_signal_connect (GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC (new_server_cb), NULL);

	/* edit server button */
	button = gtk_button_new_with_label (_("Edit..."));
	gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, GNOME_PAD);
	gtk_signal_connect (GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC (edit_server_cb), NULL);

	/* delete server button */
	button = gtk_button_new_with_label(_("Delete"));
	gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, GNOME_PAD);
	gtk_signal_connect (GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC (delete_server_cb), NULL);

	return hbox;
}

void
prefs_spawn_to_news_connections (void)
{
	prefs_spawn ();

	pan_lock ();
	gtk_clist_select_row (GTK_CLIST(sections_clist), nntp_connections_row, 0);
	pan_unlock ();
}

void
prefs_spawn_to_news_servers (void)
{
	prefs_spawn ();

	pan_lock ();
	gtk_clist_select_row (GTK_CLIST(sections_clist), nntp_clist_row, 0);
	pan_unlock ();
}

/**
 * prefs_spawn:
 *
 * Build and draw the main Preferences dialog.  Its a GnomePropertyBox
 * with a notebook in it.
 **/
void
prefs_spawn (void)
{
	GtkWidget * dialog;
	GtkWidget * hbox;

	/* There can be only one! (properties window) */
	if (win) {
		gdk_window_raise (GTK_WIDGET(win->box)->window);
		return;
	}

	win = g_new0 (PrefsWindow, 1);
	dialog = gnome_property_box_new ();
	win->box = GNOME_PROPERTY_BOX (dialog);

	gtk_window_set_title (GTK_WINDOW (win->box), _("Pan: Preferences"));

	hbox = gtk_hbox_new (FALSE, GNOME_PAD);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);

	sections_clist = gtk_clist_new (1);
	gtk_clist_set_selection_mode (GTK_CLIST (sections_clist),
				      GTK_SELECTION_BROWSE);
	gtk_signal_connect (GTK_OBJECT (sections_clist), "select_row",
			    GTK_SIGNAL_FUNC (prefs_select_row_cb), NULL);


	/* add all the pages... */
	prefs_add_section_page (_("Cache"), prefs_cache_page());
	prefs_add_section_page (_("Directories"), prefs_directories_page());
	prefs_add_section_page (_("Display - Groups"), prefs_grouplist_page());
	prefs_add_section_page (_("Display - Threads"), prefs_articlelist_page());
	prefs_add_section_page (_("Display - Article"), prefs_text_page());
	prefs_add_section_page (_("General"), prefs_general_page());
	prefs_add_section_page (_("Layout"), win->layout_page=prefs_layout_page());
	prefs_add_section_page (_("Mail Server"), prefs_smtp_page());
	nntp_clist_row = prefs_add_section_page (_("News Servers"), prefs_server_page());
	nntp_connections_row =prefs_add_section_page (_("Online Settings"), prefs_connections_page());
	prefs_add_section_page (_("Posting - Headers"), prefs_headers_page());
	prefs_add_section_page (_("Posting - Identity"), prefs_userinfo_page());

	gtk_notebook_set_show_tabs (GTK_NOTEBOOK(win->box->notebook),
				    FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK(win->box->notebook),
				      FALSE);

	gtk_signal_connect (GTK_OBJECT (win->box), "apply",
			    GTK_SIGNAL_FUNC (prefs_apply_cb), NULL);
	gtk_signal_connect (GTK_OBJECT (win->box), "destroy",
			    GTK_SIGNAL_FUNC (prefs_cancel_cb), NULL);


	gtk_widget_ref (GNOME_PROPERTY_BOX (win->box)->notebook);
	
	gtk_container_remove (GTK_CONTAINER (GNOME_DIALOG (win->box)->vbox),
			      GNOME_PROPERTY_BOX (win->box)->notebook);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG(win->box)->vbox),
			    hbox, TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (hbox),
			    sections_clist,
			    TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (hbox),
			    GNOME_PROPERTY_BOX (win->box)->notebook,
			    TRUE, TRUE, 0);

	gtk_widget_unref (GNOME_PROPERTY_BOX (win->box)->notebook);

	gtk_widget_set_usize (GTK_WIDGET (sections_clist), 150, 0);
        gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(Pan.window));
	gtk_widget_show_all (dialog);
}
