/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <stdlib.h>

#include "debug.h"
#include "fnmatch.h"
#include "group.h"
#include "rule.h"
#include "util.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE
************/

/*****
******
*****/

/*****
******
*****/

/*****
******
*****/

/************
*************  PROTECTED
************/

void
rule_constructor (Rule * r)
{
	g_return_if_fail (r!=NULL);

	pan_object_constructor (PAN_OBJECT(r), rule_destructor);
	debug1 (DEBUG_PAN_OBJECT, "rule_ctor: %p", r);

	r->name = NULL;
	r->hits = 0;
	r->tries = 0;
	r->apply_to_incoming = TRUE;
	r->dirty = FALSE;

	r->group_type = RULE_GROUP_ALL;
	r->group_wildcard = NULL;
	r->group_list = NULL;

	r->criteria = NULL;

	r->action = rule_action_new ();
}

void
rule_destructor (PanObject * obj)
{
	Rule * r = RULE(obj);

	debug1 (DEBUG_PAN_OBJECT, "rule_dtor: %p", r);

	/* free misc */
	g_free (r->name);

	/* free the group wildcard string */
	g_free (r->group_wildcard);
	r->group_wildcard = NULL;

	/* free the group list */
	if (r->group_list != NULL) {
		pan_g_ptr_array_foreach (r->group_list, (GFunc)g_free, NULL);
		g_ptr_array_free (r->group_list, TRUE);
		r->group_list = NULL;
	}

	/* free the criteria */
	if (r->criteria != NULL) {
		pan_object_unref (PAN_OBJECT(r->criteria));
		r->criteria = NULL;
	}

	/* free the action */
	pan_object_unref (PAN_OBJECT(r->action));
	r->action = NULL;

	/* free the superclass */
	pan_object_destructor (obj);
}

/************
*************  PUBLIC
************/

Rule*
rule_new (void)
{
	Rule * r = g_new0 (Rule, 1);
	rule_constructor (r);
	return r;
}

/*****
******
*****/

Rule*
rule_dup (const Rule * rule)
{
	Rule * r;

	g_return_val_if_fail (rule!=NULL, NULL);

	r = rule_new ();
	/* FIXME: duplicate "rule" here */

	return r;
}

/*****
******
*****/

gboolean
rule_test_group (const Rule * rule,
                 const gchar * groupname)
{
	gboolean retval = FALSE;

	g_return_val_if_fail (rule!=NULL, FALSE);
	g_return_val_if_fail (is_nonempty_string(groupname), FALSE);

	switch (rule->group_type)
	{
		case RULE_GROUP_ALL:
			retval = TRUE;
			break;

		case RULE_GROUP_WILDCARD:
			retval = !fnmatch(rule->group_wildcard,groupname,PAN_CASEFOLD);
			break;

		case RULE_GROUP_LIST:
		{
			guint i;
			for (i=0; rule->group_list!=NULL && i!=rule->group_list->len; ++i) {
				const gchar * name = (const gchar*) g_ptr_array_index (rule->group_list,i);
				if (!g_strcasecmp(groupname,name)) {
					retval = TRUE;
					break;
				}
			}
			break;
		}
		default:
			pan_warn_if_reached ();
	}

	return retval;
}

static void
rule_remove_group_failures (const Rule * rule,
                            GPtrArray * articles)
{
	guint i;

	g_return_if_fail (rule!=NULL);
	g_return_if_fail (articles!=NULL);

	if (rule->group_type == RULE_GROUP_ALL) /* no failures */
		return;

	for (i=0; i!=articles->len; )
	{
		Group * group = ARTICLE(g_ptr_array_index(articles,i))->group;
		gboolean pass = rule_test_group (rule, group->name);

		if (pass)
			++i;
		else
			g_ptr_array_remove_index_fast (articles, i);
	}
}

void
rule_apply (Rule * rule, GPtrArray * articles)
{
	gboolean deleted;
	GPtrArray * tmp;

	/* sanity checks */
	g_return_if_fail (rule!=NULL);
	g_return_if_fail (articles!=NULL);

	/* create a scratch copy of the articles */
	tmp = g_ptr_array_new ();
	pan_g_ptr_array_assign (tmp, articles->pdata, articles->len);

	/* weed out non-matches */
	rule_remove_group_failures (rule, tmp);
	rule_criteria_remove_failures (rule->criteria, tmp);
	if (tmp->len != 0)
		g_message ("%u articles match rule %s", tmp->len, rule->name);

	/* update the stats */
	rule->tries += articles->len;
	rule->hits += tmp->len;
	rule->dirty = TRUE;

	/* apply the action to the remaining articles */
	deleted = FALSE;
	rule_action_apply (rule->action, tmp, &deleted, rule->name);
	if (deleted) {
		guint i;
		for (i=0; i!=tmp->len; ++i) {
			gpointer p = g_ptr_array_index(tmp,i);
			g_ptr_array_remove_fast (articles, p);
		}
	}

	/* cleanup */
	g_ptr_array_free (tmp, TRUE);
}


/*****
******
*****/
