/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/libgnomeui.h>

#include "articlelist.h"
#include "debug.h"
#include "file-grouplist.h"
#include "file-headers.h"
#include "globals.h"
#include "grouplist.h"
#include "gui.h"
#include "prefs.h"
#include "queue.h"
#include "server.h"
#include "util.h"

static void server_list_save (void);


/**
***  Private Utilities
**/

static gint
server_name_compare (gconstpointer a, gconstpointer b)
{
	const Server* sa = (const Server*) a;
	const Server* sb = (const Server*) b;
	return g_strcasecmp (sa->name, sb->name);
}

/**
***  Server Module
**/

static void
server_folders_load (void)
{
	Server * server = server_get_by_name (INTERNAL_SERVER_NAME);

	if (server == NULL) /* new user; create the folder server */
	{
		/* add the Folder Server */
		server = server_new ();
		server->address = g_strdup ("Mock Server for Folders");
		server->name = g_strdup (INTERNAL_SERVER_NAME);
		Pan.serverlist = g_slist_insert_sorted (Pan.serverlist,
		                                        server,
		                                        server_name_compare);
	}
	else /* server already exists; load its grouplist */
	{
		file_grouplist_load (server, NULL);
	}

	if (folder_get_by_name(PAN_SENDLATER) == NULL)
	{
		Group * group = group_new (server, PAN_SENDLATER);
		group->readable_name = server_chunk_string (server, _(PAN_SENDLATER));
		group_set_string_field (group, &group->description,
			_("Articles/Mail queued for posting via the \"Send Later\" button"));
		group_set_is_folder (group, TRUE);
		server_add_groups (server, &group, 1, NULL, NULL);
		g_message (_("Created folder pan.sendlater"));
	}

	if (folder_get_by_name(PAN_SENT) == NULL)
	{
		Group * group = group_new (server, PAN_SENT);
		group->readable_name = server_chunk_string (server, _(PAN_SENT));
		group_set_string_field (group, &group->description,
			_("Articles/Mail posted before"));
		group_set_is_folder (group, TRUE);
		server_add_groups (server, &group, 1, NULL, NULL);
		g_message (_("Created folder pan.sent"));
	}
}

static Server*
server_fetch (const char *name)
{
	Server * server = NULL;
	gboolean has_section;
	gchar * buf;

	/* find out if we've got a gnome_config section for this server */
	buf = g_strdup_printf ("/Pan/%s/", name);
	gnome_config_push_prefix (buf);
	has_section = gnome_config_has_section (buf);
	replace_gstr (&buf, NULL);

	if (has_section)
	{
		server = server_new ();
		server->address = gnome_config_get_string ("Address");
		if (server->address == NULL)
		{
			pan_object_unref (PAN_OBJECT(server));
			server = NULL;
		}
		else
		{
			gchar * path;
			gchar * pch;
			
			/* make sure the server's config directory exists */
		       	path = g_strdup_printf ("/%s/%s/", data_dir, name);
			directory_check (path);

			/* load the server from gnome_config */
			server->port =
				gnome_config_get_int ("Port=119");
			server->name =
				g_strdup(name);
			server->gen_msgid =
				gnome_config_get_int ("Generate_Msgid=0");
			server->need_auth =
				gnome_config_get_int ("Auth_Required=0");
			server->username =
				gnome_config_get_string ("Username");
			server->password =
				gnome_config_get_string ("Password");
			server->max_connections =
				gnome_config_get_int ("Max_Connections=2");
			server->idle_secs_before_timeout =
				gnome_config_get_int ("Idle_Timeout_Secs=180");
			server->online =
				gnome_config_get_bool ("Online=TRUE");
			server->reserve_nonleech =
				gnome_config_get_bool ("Reserve_Nonleech_Connection=TRUE");
			/* no good way to get a ulong from gnome_config? */
			server->last_newgroup_list_time = 1;
			pch = gnome_config_get_string ("Last_Grouplist_Update");
			if (is_nonempty_string(pch))
				server->last_newgroup_list_time = strtoul (pch, NULL, 10);

			/* cleanup */
			g_free (path);
			g_free (pch);
		}
	}
	gnome_config_pop_prefix ();

	return server;
}

/**
 * Load a list of servers from the user's ~/.gnome/Pan file
 * @return a GSList of Server's for each server found
 */
static GSList*
server_list_load (void)
{
	GSList * l = NULL;
	gchar * p = gnome_config_get_string ("/Pan/Servers/Names"); /*commas*/

	/* build all the servers in the list */
	if (p != NULL)
	{
		const char * walk = p;
		GString * str = g_string_new (NULL);

		/* create a server object for each server in the list */
		while (get_next_token_g_str (walk, ',', &walk, str))
		{
			Server * server = server_fetch (str->str);
			if (server != NULL)
				l = g_slist_insert_sorted (l,
				                           server,
				                           server_name_compare);
		}

		/* cleanup */
		g_string_free (str, TRUE);
		g_free (p);
	}

	return l;

}

static void
server_init_servers (void)
{
	static gboolean inited = FALSE;

	if (!inited)
	{
		inited = TRUE;
		Pan.serverlist = server_list_load ();
		server_folders_load ();
	}
}

void
server_shutdown (void)
{
	GSList * l;
	Server * last;

	/* remember which server we're on so that
	   we can default to it next time... */
	last = grouplist_get_server ();
	if (last && is_nonempty_string(last->name))
	{
		gnome_config_set_string ("/Pan/State/Server", last->name);
		gnome_config_sync ();
	}

	/* save the updated server info */
	server_list_save ();

	/* close down the servers */
	l = Pan.serverlist;
	Pan.serverlist = NULL;
	g_slist_foreach (l, (GFunc)pan_object_unref, NULL);
	g_slist_free (l);
}

Server*
server_get_by_name (const char* servername)
{
	GSList * l;
	Server * retval = NULL;

	g_return_val_if_fail (is_nonempty_string(servername), NULL);

	server_init_servers ();

	for (l=Pan.serverlist; retval==NULL && l!=NULL; l=l->next) {
		Server * s = SERVER(l->data);
		if (!pan_strcmp (s->name, servername))
			retval = s;
	}

	return retval;
}

/**
***  Server Object Life Cycle
**/

void
server_constructor (Server              * server,
                    PanObjectDestructor   destructor)
{
	PanObject *obj = PAN_OBJECT(server);

	/* init the superclass bits */
	pan_object_constructor (obj, destructor);

	/* init the status-item bits */
	debug1 (DEBUG_PAN_OBJECT, "server_constructor: %p", server);
	server->online_status_changed = pan_callback_new ();
	server->posting = FALSE;
	server->groups_dirty = FALSE;
	server->gen_msgid = FALSE;
	server->need_auth = FALSE;
	server->online = FALSE;
	server->online_prompted = FALSE;
	server->reserve_nonleech = TRUE;
	server->address = NULL;
	server->name = NULL;
	server->username = NULL;
	server->password = NULL;
	server->port = 25;
	server->max_connections = 1;
	server->idle_secs_before_timeout = 180;
	server->groups = g_ptr_array_new ();
	server->last_newgroup_list_time = 1; /* Typhoon can't handle 0 */
	server->chunk = NULL;
}

const gchar*
server_chunk_string (Server * server,
                     const gchar * chunkme)
{
	g_return_val_if_fail (server!=NULL, NULL);

	if (server->chunk == NULL)
		server->chunk = g_string_chunk_new (4096);

	/* group names and descriptions have almost zero chance for overlap,
	   so looking for duplicates is wasted cycles -- don't insert_const. */
	return g_string_chunk_insert (server->chunk, chunkme);
}

void
server_destructor (PanObject *obj)
{
	Server* server = SERVER(obj);

	/* clear out the server bits */
	debug1 (DEBUG_PAN_OBJECT, "server_destructor: %p", obj);
	file_grouplist_save (server, NULL);
	pan_g_ptr_array_foreach (server->groups, (GFunc)pan_object_unref, NULL);
	g_ptr_array_free (server->groups, TRUE);
	g_free (server->name);
	g_free (server->address);
	g_free (server->username);
	g_free (server->password);
	if (server->chunk != NULL)
		g_string_chunk_free (server->chunk);
	pan_callback_free (server->online_status_changed);

	/* clear out the parent */
	pan_object_destructor (obj);
}

Server*
server_new (void)
{
	Server* server = g_new0 (Server, 1);
	debug1 (DEBUG_PAN_OBJECT, "server_new: %p", server);
	server_constructor (server, server_destructor);
	return server;
}

/**
***  Server Object
**/

void
server_set_online (Server    * server,
                   gboolean    online)
{
	g_return_if_fail (server!=NULL);
	server->online = online ? 1 : 0;
	if (online)
		queue_wakeup ();
}


/**
***  GUI Stuff
**/

static void
server_selected_cb (GtkWidget  * w,
                    Server     * server)
{
	if (GTK_CHECK_MENU_ITEM(w)->active)
		grouplist_set_server (server);
}

static int server_qty = 0;
static gchar * first_server_name = NULL;
static GnomeUIInfo * server_menu = NULL;
static GnomeUIInfo * server_menu_radiobox = NULL;
static GnomeUIInfo end = GNOMEUIINFO_END;

void
server_menu_update (void)
{
	GSList* l;
	GnomeUIInfo * buf;
	GnomeUIInfo * radio_box;
	GString * str = g_string_new (NULL);
	gchar * path;
	gchar * new_first_server_name = NULL;
	int i;

	/* build the new menu */
	l = Pan.serverlist;
	buf = g_new0 (GnomeUIInfo, g_slist_length(l)+2);
	i = 0;
	if (!l)
	{
		GnomeUIInfo * info = &buf[i];
		info->type = GNOME_APP_UI_ITEM;
		info->label = _("No Servers Defined");
		info->hint = _("No Servers Defined");
		info->moreinfo = NULL;
		info->user_data = NULL;
		++i;
	}
	for (; l!=NULL; l=l->next)
	{
		Server *s = SERVER(l->data);
		GnomeUIInfo * info = buf + i;

		/* skip 'invisible' servers used for folders, etc. */
		if (!strncmp (s->name, INTERNAL_SERVER_NAME, strlen(INTERNAL_SERVER_NAME)))
			continue;

		info->type = GNOME_APP_UI_ITEM;
		info->label = s->name;
		info->hint = s->name;
		info->moreinfo = server_selected_cb;
		info->user_data = s;
		++i;
		if (new_first_server_name == NULL)
			new_first_server_name = g_strdup (s->name);
	}
	buf[i] = end;

	radio_box = g_new0 (GnomeUIInfo, 2);
	radio_box[0] = radio_box[1] = end;
	radio_box[0].type = GNOME_APP_UI_RADIOITEMS;
	radio_box[0].label = radio_box[0].hint = _("Online");
	radio_box[0].moreinfo = buf;


	/* update the main menu */
	if (is_nonempty_string(first_server_name))
	{
		gchar* pch = g_strdup_printf ("Online/%s", first_server_name);
		gnome_app_remove_menus (GNOME_APP(Pan.window), pch, server_qty);
		g_free (pch);
	}

	path = g_strdup_printf ("%s/%s", _("_Online"), "<Separator>");
	gnome_app_insert_menus (GNOME_APP(Pan.window), path, radio_box);
	g_free (path);

	/* update the local varialbes */
	g_free (first_server_name);
	g_free (server_menu);
	g_free (server_menu_radiobox);
	first_server_name = new_first_server_name;
	server_menu = buf;
	server_menu_radiobox = radio_box;
	server_qty = i;

	/* save groups to disk */
	server_list_save ();

	/* cleanup */
	g_string_free (str, TRUE);
}

/****
*****
*****  SERVER LOADING ON STARTUP
*****
****/

void
server_init (void)
{
	gchar *last = NULL;
	GSList *l = NULL;
	int i;

	/* This gets called before gtk_main and
	 * at the end of guru_finish */
	server_init_servers ();
	server_menu_update ();

	/* try hard to select a server.. if there's one in the ini file, use
	 * it; otherwise, go with the first one in the list (if there is one)
	 */
	last = gnome_config_get_string ("/Pan/State/Server");
	if (!last && Pan.serverlist)
		last = g_strdup ((SERVER(Pan.serverlist->data)->name));
	if (last!=NULL)
	{
		for (i=0, l=Pan.serverlist; l; l=l->next)
		{
			Server *server = SERVER(l->data);

			/* skip 'invisible' servers used for folders, etc. */
			if (!strncmp (server->name, INTERNAL_SERVER_NAME, strlen(INTERNAL_SERVER_NAME)))
				continue;

			if (!strcmp(server->name, last)) {
				gtk_check_menu_item_set_active (
		                	GTK_CHECK_MENU_ITEM(
						server_menu[i].widget), TRUE);
				grouplist_set_server (server);
				break;
			}

			++i;
		}
		g_free (last);
	}
}

void
server_add_new_server (Server * server)
{
	g_return_if_fail (server!=NULL);

	server_init_servers ();

	Pan.serverlist = g_slist_insert_sorted (Pan.serverlist,
	                                        server,
	                                        server_name_compare);
}


static void
server_list_save (void)
{
	GString * buf = g_string_new (NULL);
	GString * servers = g_string_new (NULL);
	GSList *l = NULL;
	gchar * p;

	/* Out with the old */
	p = gnome_config_get_string ("/Pan/Servers/Names");
	if (p != NULL) {
		const char * pch = p;
		while (get_next_token_g_str(pch, ',', &pch, buf)) {
			g_string_prepend (buf, "/Pan/");
			gnome_config_clean_section (buf->str);
		}
		g_free (p);
	}
	gnome_config_sync ();

	/* In with the new */
	for (l=Pan.serverlist; l!=NULL; l=l->next)
	{
		char * pch;
		Server *s = SERVER(l->data);

		g_string_sprintf (buf, "/Pan/%s/Address", s->name);
		gnome_config_set_string (buf->str, s->address);

		g_string_sprintf (buf, "/Pan/%s/Port", s->name );
		gnome_config_set_int (buf->str, s->port);

		g_string_sprintf (buf, "/Pan/%s/Auth_Required", s->name);
		gnome_config_set_int (buf->str, s->need_auth);

		g_string_sprintf (buf, "/Pan/%s/Generate_Msgid", s->name);
		gnome_config_set_int (buf->str, s->gen_msgid);

		g_string_sprintf (buf, "/Pan/%s/Username", s->name);
		gnome_config_set_string (buf->str, s->username);

		g_string_sprintf (buf, "/Pan/%s/Password", s->name);
		gnome_config_set_string (buf->str, s->password);

		g_string_sprintf (buf, "/Pan/%s/Max_Connections", s->name);
		gnome_config_set_int (buf->str, s->max_connections);

		g_string_sprintf (buf, "/Pan/%s/Idle_Timeout_Secs", s->name);
		gnome_config_set_int (buf->str, s->idle_secs_before_timeout);

		g_string_sprintf (buf, "/Pan/%s/Online", s->name);
		gnome_config_set_bool (buf->str, s->online);

		/* no good way to store a ulong in gnome_config? */
		pch = g_strdup_printf ("%lu", s->last_newgroup_list_time);
		g_string_sprintf (buf,"/Pan/%s/Last_Grouplist_Update", s->name);
		gnome_config_set_string (buf->str, pch);
		g_free (pch);

		g_string_sprintf (buf, "/Pan/%s/Reserve_Nonleech_Connection", s->name);
		gnome_config_set_bool (buf->str, s->reserve_nonleech);

		if (servers->len)
			g_string_append_c (servers, ',');
		g_string_append (servers, s->name);
	}

	gnome_config_set_string ("/Pan/Servers/Names", servers->str);
	g_string_free (servers, TRUE);
	g_string_free (buf, TRUE);

	gnome_config_sync ();
}



/**
***
**/

static void
online_prompt_cb (gint reply,
                  gpointer user_data)
{
	const gboolean online = !reply;
	Server * server = SERVER(user_data);
	if (server->online != online)
	{
		server_set_online (SERVER(user_data), online);
		server->online_prompted = FALSE;
	}
}

void
server_prompt_to_go_online (Server* server)
{
	g_return_if_fail (server!=NULL);

	if (!server->online && !server->online_prompted)
	{
		gchar* question;

		server->online_prompted = TRUE;

		question = g_strdup_printf (_("Go Online for Server \"%s\"?"),
		                            server->name);

		pan_lock ();
		gnome_question_dialog_parented (question,
		                                online_prompt_cb,
		                                server,
		                                GTK_WINDOW(Pan.window));
		pan_unlock ();

		g_free (question);
	}
}


/***
****
****  NEWSRC
****
***/

static void
import_newsrc (Server * server, const gchar * filename)
{
	GString * line;
	const gchar * foo;
	GPtrArray * addme;
	GArray * body = read_file (filename);

	if (!body)
		return;
       
	line = g_string_new (NULL);
	addme = g_ptr_array_new ();
	foo = body->data;

	/* march through each line of newsrc... */
	while (get_next_token_g_str (foo, '\n', &foo, line))
	{
		gchar * pch;

		/* get the group name */
		pch = strpbrk (line->str, "!:");
		if (pch != NULL)
		{
			Newsrc * n;
		       	gboolean subscribed = FALSE;
			gchar * name = g_strndup (line->str, pch-line->str);
			Group * g = group_get_by_name (server, name);

			if (g == NULL) /* new group */
			{
				g = group_new (server, name);
				pan_object_ref (PAN_OBJECT(g));
				pan_object_sink (PAN_OBJECT(g));
				g_ptr_array_add (addme, g);
			}

			/* find out whether or not the group's subscribed,
			   get its articles read */
			n = group_get_newsrc (g);
			newsrc_import (n, &subscribed, line->str);
			group_set_article_range (g, n->group_low, n->group_high);
			group_set_subscribed (g, subscribed);
			g_free (name);
		}
	}

	/* add the new groups */
	if (addme->len != 0)
		server_add_groups (server, (Group**)addme->pdata, addme->len, NULL, NULL);

	/* cleanup */
	g_array_free (body, TRUE);
	g_string_free (line, TRUE);
	g_ptr_array_free (addme, TRUE);
}

static void
newsrc_import_dialog_clicked_cb (GnomeDialog * dialog, gint num, gpointer data)
{
	Server * server = SERVER(data);

	g_return_if_fail (server != NULL);

	if (num == 0)
	{
		GnomeFileEntry * gfe = gtk_object_get_data (
			GTK_OBJECT(dialog), "file_entry");
		const char * filename = gnome_file_entry_get_full_path (gfe, 0);
		import_newsrc (server, filename);
	}

        gtk_widget_destroy (GTK_WIDGET(dialog));
}

void
server_import_newsrc_dialog (Server * server)
{
	GtkWidget * hbox;
	GtkWidget * dialog;
	GtkWidget * w;
	GtkWidget * file_entry;
	gchar * s;

	pan_lock ();

	dialog = gnome_dialog_new (_("Import .newsrc File"),
	                           GNOME_STOCK_BUTTON_OK,
	                           GNOME_STOCK_BUTTON_CANCEL,
	                           NULL);

	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);

	s = _("Select .newsrc File to Read:");
	w = gtk_label_new (s);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	file_entry = gnome_file_entry_new ("newsrc", s);
	gnome_file_entry_set_default_path (GNOME_FILE_ENTRY(file_entry),
	                                   g_get_home_dir());
	gtk_signal_connect (GTK_OBJECT(dialog), "clicked",
	                    GTK_SIGNAL_FUNC (newsrc_import_dialog_clicked_cb),
	                    server);
	gtk_box_pack_start (GTK_BOX(hbox), file_entry, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(GNOME_DIALOG(dialog)->vbox), hbox, 0, 0, 0);
	gtk_object_set_data (GTK_OBJECT(dialog), "file_entry", file_entry);
	gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(Pan.window));
	gtk_widget_show_all (dialog);
	pan_unlock();
}

/**
***
**/

static void
export_newsrc (Server * server, const gchar * filename)
{
	FILE * fp = fopen (filename, "w");
	if (fp != NULL)
	{
		int i;
		GString * s = g_string_sized_new (4096);
		for (i=0; i<server->groups->len; ++i)
		{
			Group * g = GROUP(g_ptr_array_index(server->groups, i));
			Newsrc * n = group_has_newsrc(g) ? group_get_newsrc(g) : NULL;
			g_string_truncate (s, 0);
			newsrc_export (n, g->name, group_is_subscribed(g), s);
			fprintf (fp, "%s\n", s->str);
		}

		fclose (fp);
		g_message (_("%d groups written to `%s'"), i, filename);
	}
}

static void
newsrc_export_dialog_clicked_cb (GnomeDialog * dialog, int num, gpointer data)
{
	Server * server = SERVER(data);

	g_return_if_fail (server != NULL);

	if (num == 0)
	{
		GnomeFileEntry * gfe = gtk_object_get_data (
			GTK_OBJECT(dialog), "file_entry");
		const char * filename = gnome_file_entry_get_full_path (gfe, 0);
		export_newsrc (server, filename);
	}

        gtk_widget_destroy (GTK_WIDGET(dialog));
}
void
server_export_newsrc_dialog (Server * server)
{
	GtkWidget * hbox;
	GtkWidget * dialog;
	GtkWidget * w;
	GtkWidget * file_entry;
	gchar * s;

	pan_lock ();

	dialog = gnome_dialog_new (_("Export .newsrc File"),
	                           GNOME_STOCK_BUTTON_OK,
	                           GNOME_STOCK_BUTTON_CANCEL,
	                           NULL);

	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);

	s = _("Select .newsrc File to Write:");
	w = gtk_label_new (s);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	file_entry = gnome_file_entry_new ("newsrc", s);
	gnome_file_entry_set_default_path (GNOME_FILE_ENTRY(file_entry),
	                                   g_get_home_dir());
	gtk_signal_connect (GTK_OBJECT(dialog), "clicked",
	                    GTK_SIGNAL_FUNC (newsrc_export_dialog_clicked_cb),
	                    server);
	gtk_box_pack_start (GTK_BOX(hbox), file_entry, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(GNOME_DIALOG(dialog)->vbox), hbox, 0, 0, 0);
	gtk_object_set_data (GTK_OBJECT(dialog), "file_entry", file_entry);
	gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(Pan.window));
	gtk_widget_show_all (dialog);
	pan_unlock();
}


/***
****
****  GROUPLIST
****
***/

static int
compare_name_ppgroup_name (const void* a, const void* b)
{
	return strcmp ((const gchar*)a, (*(const Group**)b)->name);
}

static int
compare_pgroup_ppgroup_name (const void* a, const void* b)
{
	return strcmp (((const Group*)a)->name, (*(const Group**)b)->name);
}

static int
compare_ppgroup_ppgroup_name (const void * a, const void * b)
{
	return strcmp ((*(const Group**)a)->name, (*(const Group**)b)->name);
}


void
server_add_groups (Server       * server,
                   Group       ** groups,
                   gint           group_qty,
                   GPtrArray    * fillme_used,
                   GPtrArray    * fillme_not_used)
{
	guint i, j;
	GPtrArray * old;
	GPtrArray * tmp;
	Group** merged = NULL;
	int merged_qty = 0;
	gboolean dirty = FALSE;

	/* entry assertions */
	g_return_if_fail (server != NULL);
	g_return_if_fail (groups != NULL);
	g_return_if_fail (group_qty >= 1);

	/* make a temporary sorted array of the groups being added */
	tmp = g_ptr_array_new ();
	pan_g_ptr_array_assign (tmp, (gpointer*)groups, group_qty);
	pan_g_ptr_array_sort (tmp, compare_ppgroup_ppgroup_name);

	/* so now we have two sorted lists; merge them. */
	old = server->groups;
	merged = g_new (Group*, old->len + tmp->len);
	for (i=j=0; i!=old->len || j!=tmp->len; )
	{
		Group * addme = NULL;

		if (i==old->len)
		{
			addme = g_ptr_array_index (tmp, j++);
			dirty = TRUE;
			if (fillme_used)
				g_ptr_array_add (fillme_used, addme);
		}
		else if (j==tmp->len)
		{
			addme = g_ptr_array_index (old, i++);
		}
		else
		{
			Group * g1 = GROUP(g_ptr_array_index(tmp,j));
			Group * g2 = GROUP(g_ptr_array_index(old,i));
			const int comp = pan_strcmp (g1->name, g2->name);

			if (comp<0)
			{
				addme = g1;
				dirty = TRUE;
				if (fillme_used)
					g_ptr_array_add (fillme_used, addme);
				++j;
			}
			else
			{
				if (g2->permission != g1->permission) {
					g2->permission = g1->permission;
					dirty = TRUE;
				}
				addme = g2;
				++i;
				if (!comp && g1!=g2 && fillme_not_used)
				{
					g_ptr_array_add (fillme_not_used, g1);
					++j;
				}
			}
		}

		if (addme != NULL)
		{
			if (addme->server != server)
			{
				server_remove_group (addme->server, addme);
				addme->server = server;
			}

			merged[merged_qty++] = addme;
		}
	}

	/* if any change was made, update everything */
	if (dirty)
	{
		pan_g_ptr_array_assign (old, (gpointer*)merged, merged_qty);
		server->groups_dirty = TRUE;
		grouplist_update_ui (server);
	}

	/* cleanup */
	g_free (merged);
	g_ptr_array_free (tmp, TRUE);
}


void
server_remove_group (Server * server, Group * group)
{
	guint index;
	gboolean exact_match = FALSE;

	g_return_if_fail (server != NULL);
	g_return_if_fail (group != NULL);
	g_return_if_fail (group->server == server);

	/* does this group exist in the server's list? */
	index = lower_bound (group,
	                     server->groups->pdata,
	                     server->groups->len,
	                     sizeof(gpointer),
	                     compare_pgroup_ppgroup_name,
			     &exact_match);

	/* if so, act... */
	if (exact_match)
	{
		Group * g = GROUP(g_ptr_array_remove_index (server->groups, index));

		/* update the grouplist if necessary */
		if (grouplist_get_server() == server)
			grouplist_remove_row (group);

		/* remove it */
		g->server = NULL;
		pan_object_unref (PAN_OBJECT(g));
		server->groups_dirty = TRUE;
	}
}

Group*
group_get_by_name (Server        * server,
	           const char    * group_name)
{
	Group * retval = NULL;

	g_return_val_if_fail (server!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string(group_name), NULL);

	if (server != NULL)
	{
		int index;
		gboolean exact_match = FALSE;

		/* find where to add this group */
		index = lower_bound (group_name,
				     server->groups->pdata,
				     server->groups->len,
				     sizeof(gpointer),
				     compare_name_ppgroup_name,
				     &exact_match);

		if (exact_match)
			retval = GROUP(g_ptr_array_index(server->groups,index));
	}

	return retval;
}

gboolean
server_has_group (const Server  * server,
                  const char    * group_name)
{
	return group_get_by_name((Server*)server,group_name) != NULL;
}


void
server_destroy_group (Server * server, Group * group)
{
	g_return_if_fail (server != NULL);
	g_return_if_fail (group != NULL);
	g_return_if_fail (group->server == server);

	/* unload group (if it's loaded) */
	if (articlelist_get_group() == group)
		articlelist_set_group (NULL);

	/* destroy the data files */
	file_headers_destroy (group);

	/* remove the group from the server list */
	server_remove_group (server, group);
}
