/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>  

#include "util.h"
#include "util-wrap.h"

extern int wrap_column;

/****
*****  LINE WRAPPING
****/

static void
get_leader (const gchar * line,
            GString     * setme)
{
	g_string_truncate (setme, 0);
	for (; is_nonempty_string(line) && (*line=='>' || isspace((int)*line)); ++line)
		g_string_append_c (setme, *line);
}

typedef struct
{
	gchar * leader;
	gchar * content;
}
Paragraph;

static void
paragraph_free (Paragraph * p)
{
	g_free (p->leader);
	g_free (p->content);
	g_free (p);
}

/**
 * FIXME: doesn't handle paragraphs whose first lines are indented
 */
static GPtrArray*
get_paragraphs (const gchar * body)
{
	gint i = 0;
	gchar ** lines = g_strsplit (body, "\n", -1);
	GString * leader = g_string_new (NULL);
	GString * content = g_string_new (NULL);
	GString * last_leader = g_string_new (NULL);
	GPtrArray * paragraphs = g_ptr_array_new ();

	for (i=0; ; ++i)
	{
		gchar * line = NULL;
		get_leader (lines[i], leader);

		if (lines[i])
			line = lines[i] + leader->len;

		if (content->len /* something to keep... */
		    && (!line /* eof forces paragraph break */
		        || !*line /* empty line forces paragraph break */
			|| (line && !isalpha((int)*line)) /* source code? */
			|| pan_strcmp(leader->str,last_leader->str) /*break*/
		       )
		    )
		{
			Paragraph * p = g_new (Paragraph, 1);
			p->content = g_strdup (content->str);
			p->leader = g_strdup (last_leader->str);
			g_ptr_array_add (paragraphs, p);
			g_string_truncate (content, 0);
		}

		if (line) /* don't process a non-line */
		{
			if (!*line /* empty line forces paragraph break */
				|| !isalpha((int)*line)) /* source code */
			{
				Paragraph * p = g_new (Paragraph, 1);
				p->content = g_strdup (line);
				p->leader = g_strdup (leader->str);
				g_ptr_array_add (paragraphs, p);
			}
			else /* augment paragraph */
			{
				if (content->len)
					g_string_append_c (content, ' ');
				g_string_append (content, line);
			}
		}

		/* prepare for next iteration */
		g_string_assign (last_leader, leader->str);

		/* loop past the last line, to get even the last paragraph */
		if (!line)
			break;
	}

	/* make the paragraphs into one continuous line each */
	for (i=0; i<paragraphs->len; ++i)
	{
		Paragraph *p = (Paragraph*) g_ptr_array_index (paragraphs, i);
		char* pch;
		for (pch=p->content; *pch; ++pch)
		{
			/* if end of line, remove those characters */
			if (*pch=='\n')
			{
				/* also remove any trailing spaces */
				char * it = pch;
				while (it!=p->content && isspace((int)it[-1]))
					--it;
				g_memmove (it, pch+1, strlen(pch+1)+1); /* +1 for '\0' */
			}
		}
	}

	/* cleanup */
	g_string_free (content, TRUE);
	g_string_free (leader, TRUE);
	g_string_free (last_leader, TRUE);
	g_strfreev (lines);

	return paragraphs;
}

static gchar*
wrap_long_lines (gchar   * string,
                 gint      maxlen)
{
	gchar * linefeed_here;
	gchar * pch;
	gchar * line_start;


	/* walk through the entire string */
	linefeed_here = NULL;
	for (line_start=pch=string; *pch; )
	{
		/* a linefeed could go here; remember this space */
		if (isspace((int)*pch) || *pch=='\n')
			linefeed_here = pch;

		/* line's too long; add a linefeed if we can */
		if (pch-line_start>=maxlen && linefeed_here!=NULL)
		{
			*linefeed_here = '\n';
			pch = line_start = linefeed_here + 1;
			linefeed_here = NULL;
		}
		else ++pch;
	}

	return string;
}

static void
fill_paragraph (const Paragraph   * p,
                GPtrArray         * setme)
{
	GString * str = g_string_new (NULL);
	gchar * pch = g_strconcat (p->content, "\n", NULL);
	const gchar * march = pch;
	const gint wrap_len = wrap_column - strlen(p->leader);

	wrap_long_lines (pch, wrap_len);
	while ((get_next_token_g_str (march, '\n', &march, str)))
		g_ptr_array_add (setme, g_strconcat(p->leader, str->str, NULL));

	g_free (pch);
	g_string_free (str, TRUE);
}

gchar*
fill_body (const gchar * body)
{
	GPtrArray* array = g_ptr_array_new ();
	GPtrArray* paragraphs;
	gchar * new_body;
	gchar * tmp_body;
	gchar * sig;

	/* sanity checks */
	g_return_val_if_fail (is_nonempty_string(body), NULL);

	/* get a temp copy of the body -- we don't wrap the signature. */
	tmp_body = pan_substitute (body, "\r", "");
	sig = strstr (tmp_body, "\n-- \n");
	if (sig != NULL)
		*sig = '\0';

	/* fill the paragraphs */
       	paragraphs = get_paragraphs (tmp_body);
	pan_g_ptr_array_foreach (paragraphs, (GFunc)fill_paragraph, array);

	/* make a single string of all filled lines */
	g_ptr_array_add (array, NULL);
	if (1) {
		gint i;
		GString * s = g_string_new (NULL);
		for (i=0; g_ptr_array_index(array,i)!=NULL; ++i) {
			g_string_append (s, (const char*)g_ptr_array_index(array,i));
			g_string_append_c (s, '\n');
		}
		new_body = s->str;
		g_string_free (s, FALSE);
	}

	/* if we had a sig, put it back in */
	if (sig != NULL) {
		*sig = '\n';
		replace_gstr (&new_body, g_strjoin("\n",new_body, sig, NULL));
	}

	/* cleanup */
	pan_g_ptr_array_foreach (paragraphs, (GFunc)paragraph_free, NULL);
	g_ptr_array_free (paragraphs, TRUE);
	pan_g_ptr_array_foreach (array, (GFunc)g_free, NULL);
	g_ptr_array_free (array, TRUE);
	g_free (tmp_body);

	return new_body;
}


/***
****
***/


gchar*
mute_quoted_text_in_body (const gchar * text)
{
	gchar * retval;
	const gchar * march;
	gboolean last_line_was_quote;
	GString * buf;
	GString * line;

	g_return_val_if_fail (text!=NULL, NULL);

	march = text;
	buf = g_string_new (NULL);
	line = g_string_new (NULL);
	last_line_was_quote = FALSE;
	while (get_next_token_g_str (march, '\n', &march, line))
	{
		const gboolean is_quote = line->len!=0 && line->str[0]=='>';

		/* if we're starting a quote muting, say so */
		if (!is_quote)
		{
			g_string_append (buf, line->str);
			g_string_append_c (buf, '\n');
		}
		else if (!last_line_was_quote)
		{
			g_string_append (buf, _("> [quoted text muted]"));
			g_string_append_c (buf, '\n');
		}

		last_line_was_quote = is_quote;
	}

	retval = buf->str;
	g_string_free (buf, FALSE);
	g_string_free (line, TRUE);
	return retval;
}


gchar*
rot13 (const gchar * text)
{
	gchar * retval = NULL;
	const char * a = "abcdefghijklmnopqrstuvwxyz";

	if (text != NULL)
	{
		gchar * pch;

		retval = g_strdup (text);

		for (pch=retval; *pch!='\0'; ++pch)
		{
			if (isalpha((int)*pch))
			{
				const gboolean uc = isupper((int)*pch);
				*pch = tolower(*pch);
				*pch = a[((*pch-'a')+13)%26];
				if (uc)
					*pch = toupper(*pch);
			}
		}
	}

	return retval;
}
