/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>

#include <pthread.h>

#include <glib.h>
#include <gtk/gtkeditable.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkframe.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkmain.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-pixmap.h>
#include <libgnomeui/gnome-uidefs.h>

#include <string.h>

#include "articlelist.h"
#include "article-thread.h"
#include "article-toolbar.h"
#include "globals.h"
#include "gui.h"
#include "status-item.h"
#include "util.h"

#include "xpm/article_read.xpm"
#include "xpm/article_unread.xpm"
#include "xpm/article_new.xpm"
#include "xpm/disk.xpm"
#include "xpm/favorite.xpm"
#include "xpm/bomb.xpm"
#include "xpm/binary_complete.xpm"
#include "xpm/binary_incomplete.xpm"
#include "xpm/binary.xpm"
#include "xpm/bluecheck.xpm"
#include "xpm/mini_page.xpm"

static GtkWidget* toolbar = NULL;

static GtkWidget* show_unread_button = NULL;
static GtkWidget* show_read_button = NULL;
static GtkWidget* show_new_button = NULL;
static GtkWidget* show_complete_binary_button = NULL;
static GtkWidget* show_incomplete_binary_button = NULL;
static GtkWidget* show_nonbinary_button = NULL;
static GtkWidget* show_watched_button = NULL;
static GtkWidget* show_killfile_button = NULL;
static GtkWidget* show_normal_rank_button = NULL;
static GtkWidget* show_saved_button = NULL;
static GtkWidget* show_queued_button = NULL;
static GtkWidget* show_idle_button = NULL;
static GtkWidget* show_cached_button = NULL;
static GtkWidget* show_non_cached_button = NULL;

/*****
******  SUBJECT FILTER
*****/

static gchar *subject_filter_str = NULL;

gchar*
article_toolbar_get_filter (void)
{
	return g_strdup (subject_filter_str);
}

static gchar*
filter_selected_describe (const StatusItem* item)
{
	return g_strdup (_("Filtering Articles"));
}

static void
filter_selected_thread (void* data)
{
	Group *group = articlelist_get_group ();
	StatusItem *item = NULL;

	if (!group) /* articlelist is idle */
		return;

	/* create a new status item to get thread messages */
	item = STATUS_ITEM(status_item_new(filter_selected_describe));
	status_item_set_active (item, TRUE);

	/* update the articlelist */
	articlelist_refresh (item);

	/* clean out the status item */
	status_item_set_active (item, FALSE);
	pan_object_unref(PAN_OBJECT(item));
}
static void
article_filter_cb (GtkWidget* w, gpointer data)
{
	gchar * s;
	gchar * pch;
	pthread_t thread;

	/* wrap the key in wildcards to make it a substring search...
	   unless there are already wildcards present, indicating the
	   user already knows what he wants */
	s = gtk_editable_get_chars (GTK_EDITABLE(w), 0, -1);
	s = g_strstrip (s);
	if (strchr(s,'*'))
		pch = g_strdup (s);
	else if (*s)
		pch = g_strdup_printf("*%s*",s);
	else
		pch = NULL;
	g_free (s);

	/* if the new string differs from the old one, then update. */
	if (!pan_strcmp (subject_filter_str, pch))
		g_free (pch);
	else {
		replace_gstr (&subject_filter_str, pch);
		pthread_create (&thread, NULL, (void*)filter_selected_thread, NULL);
		pthread_detach (thread);
	}
}

/*****
******  STATE FILTER
*****/

static gboolean dampen_feedback_loop = FALSE;

static gint
state_filter_changed_idle (gpointer data)
{
	guint state_filter = GPOINTER_TO_UINT(data);

	pan_lock_unconditional ();
	dampen_feedback_loop = TRUE;

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_new_button),
	                              state_filter & STATE_FILTER_NEW);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_unread_button),
	                              state_filter & STATE_FILTER_UNREAD);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_read_button),
	                              state_filter & STATE_FILTER_READ);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_complete_binary_button),
	                              state_filter & STATE_FILTER_COMPLETE_BINARIES);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_incomplete_binary_button),
	                              state_filter & STATE_FILTER_INCOMPLETE_BINARIES);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_nonbinary_button),
	                              state_filter & STATE_FILTER_NONBINARIES);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_watched_button),
	                              state_filter & STATE_FILTER_WATCHED);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_killfile_button),
	                              state_filter & STATE_FILTER_IGNORED);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_normal_rank_button),
	                              state_filter & STATE_FILTER_NORMAL_RANK);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_saved_button),
	                              state_filter & STATE_FILTER_SAVED);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_queued_button),
	                              state_filter & STATE_FILTER_QUEUED);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_idle_button),
	                              state_filter & STATE_FILTER_IDLE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_cached_button),
	                              state_filter & STATE_FILTER_CACHED);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_non_cached_button),
	                              state_filter & STATE_FILTER_NOT_CACHED);


	dampen_feedback_loop = FALSE;
	pan_unlock_unconditional ();

	return 0;
}
static gint
state_filter_changed_cb (gpointer call_object,
                         gpointer call_arg,
                         gpointer user_data)
{
	pan_lock ();
	gtk_idle_add (state_filter_changed_idle, call_arg);
	pan_unlock ();
	return 0;
}


/*****
******  CREATE TOOLBAR
*****/

extern GtkTooltips *ttips;

static void
state_filter_button_toggled_cb (GtkToggleButton* tb, gpointer user_data)
{
        const guint flag = GPOINTER_TO_UINT(user_data);
	const gboolean on = gtk_toggle_button_get_active (tb);

	if (!dampen_feedback_loop)
		articlelist_poke_state_filter (flag, on);
}

GtkWidget*
article_toolbar_new (void)
{
	GtkWidget *w = NULL;
	GtkWidget *w2 = NULL;
	GtkWidget *frame = NULL;
	GtkWidget *hbox;

	toolbar = gtk_hbox_new (FALSE, GNOME_PAD);

	/* filter label */
	w = gtk_label_new ( _("Filters:") );

	/* filter one: new/read/unread */
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	hbox = gtk_hbox_new (FALSE, 0);
	show_new_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show New Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_NEW));
	w2 = gnome_pixmap_new_from_xpm_d (article_new_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_unread_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Unread Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_UNREAD));
	w2 = gnome_pixmap_new_from_xpm_d (article_unread_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_read_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Read Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_READ));
	w2 = gnome_pixmap_new_from_xpm_d (article_read_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame), hbox);
	gtk_box_pack_start (GTK_BOX(toolbar), frame, FALSE, FALSE, 0);

	/* filter two: watched/killfile/other */
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	hbox = gtk_hbox_new (FALSE, 0);
	show_watched_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Watched Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_WATCHED));
	w2 = gnome_pixmap_new_from_xpm_d (favorite_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_killfile_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Ignored Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_IGNORED));
	w2 = gnome_pixmap_new_from_xpm_d (bomb_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_normal_rank_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Articles neither Watched nor Ignored"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_NORMAL_RANK));
	w2 = gnome_pixmap_new_from_xpm_d (mini_page_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame), hbox);
	gtk_box_pack_start (GTK_BOX(toolbar), frame, FALSE, FALSE, 0);

	/* filter three: complete/uncomplete/text */
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	hbox = gtk_hbox_new (FALSE, 0);
	show_complete_binary_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Complete Binary Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_COMPLETE_BINARIES));
	w2 = gnome_pixmap_new_from_xpm_d (binary_complete_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_incomplete_binary_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Incomplete Binary Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_INCOMPLETE_BINARIES));
	w2 = gnome_pixmap_new_from_xpm_d (binary_incomplete_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_nonbinary_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Non-Binary Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_NONBINARIES));
	w2 = gnome_pixmap_new_from_xpm_d (mini_page_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame), hbox);
	gtk_box_pack_start (GTK_BOX(toolbar), frame, FALSE, FALSE, 0);

	/* filter four: saved/queued/neither */
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	hbox = gtk_hbox_new (FALSE, 0);
	show_saved_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Saved Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_SAVED));
	w2 = gnome_pixmap_new_from_xpm_d (binary_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_queued_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Queued Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_QUEUED));
	w2 = gnome_pixmap_new_from_xpm_d (bluecheck_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_idle_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Articles not Saved nor in Queue"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_IDLE));
	w2 = gnome_pixmap_new_from_xpm_d (mini_page_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame), hbox);
	gtk_box_pack_start (GTK_BOX(toolbar), frame, FALSE, FALSE, 0);

	/* filter five: cached/not cached */
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	hbox = gtk_hbox_new (FALSE, 0);
	show_cached_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Articles which are Cached Locally"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_CACHED));
	w2 = gnome_pixmap_new_from_xpm_d (disk_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_non_cached_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Articles which are not Cached Locally"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_NOT_CACHED));
	w2 = gnome_pixmap_new_from_xpm_d (mini_page_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame), hbox);
	gtk_box_pack_start (GTK_BOX(toolbar), frame, FALSE, FALSE, 0);

	/* Filter label */
        w = gtk_label_new (_("Find:"));
	gtk_box_pack_start (GTK_BOX(toolbar), w, FALSE, FALSE, 0);

	/* Filter */
	w = gtk_entry_new();
	gtk_signal_connect (GTK_OBJECT (w), "activate", GTK_SIGNAL_FUNC(article_filter_cb), w);
	gtk_signal_connect (GTK_OBJECT (w), "focus-out-event", GTK_SIGNAL_FUNC(article_filter_cb), w);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w,
		_("Type in a subject search string and press ENTER. "
	          "Wildcards are allowed."), "");
	gtk_box_pack_start (GTK_BOX(toolbar), w, TRUE, TRUE, 0);

	/* show the toolbar */
	gtk_widget_show_all (toolbar);

	/* listen for changes to the articlelist "state filter"
	   so that we can update the menus accordingly */
	pan_callback_add (articlelist_state_filter_changed,
	                  state_filter_changed_cb,
	                  NULL);

	return toolbar;
}
