/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#ifndef __ARTICLE_H__
#define __ARTICLE_H__

#include <time.h>
#include <glib.h>
#include "group.h"
#include "pan-callback.h"
#include "pan-object.h"
#include "status-item.h"

#define ARTICLE(a) ((Article *)a)

enum {
	STATE_NONE		= 0,
	STATE_MULTIPART_ALL	= 1<<0,
	STATE_MULTIPART_SOME	= 1<<1,
	STATE_DOWNLOAD_FLAGGED	= 1<<2,
	STATE_DECODE_QUEUED	= 1<<3,
	STATE_DECODED		= 1<<4,
	STATE_ERROR		= 1<<5,
	STATE_CACHED		= 1<<7,
	STATE_DECODE_FAILED	= 1<<8,
};

typedef struct _Article
{
        /* parent class */
	PanObject pan_object;

	guint self_passes_filter : 1; /* articlelist */
	guint tree_passes_filter : 1; /* articlelist */
	gint8 crosspost_qty;	      /* # of groups posted to */
	gint16 part;		      /* part #; 0 if no attachments */
	gint16 parts;		      /* # of parts in multipart message */
	guint16 linecount;	      /* # of lines in the body */
	guint16 state;                /* see the state enum in this file*/
	guint16 unread_children;      /* articlelist: unread kids in thread */
	time_t date;                  /* date posted, as parsed into time_t */
	GSList * threads;             /* articlelist: list of child threads */
	struct _Article * parent;     /* articlelist: parent in threading */
	const gchar * subject;        /* article's subject */
	const gchar * author_addr;    /* article's author -- address*/
	const gchar * message_id;     /* article's message-id */
	const gchar * author_real;    /* (maybe NULL) article's author -- real name */
	const gchar * references;     /* (maybe NULL) references string for this article */

	/* At some point this may be an array for 'article-centric' */
	gulong number;               /* (PROTECTED) number for article */
	Group * group;               /* (PROTECTED) group this article is in */

	/* holds infrequently-existing headers.
	   It should be used sparingly to avoid the overhead of hashtable.
	   But, since it lets us remove char*'s from Article for lesser-used
	   headers, it actually decreases sizeof(Article) and reduces memory
	   use most of the time.  See the header #defines below...
	*/
	GHashTable* headers;
}
Article;

/* Headers commonly stored in article->headers. */
#define HEADER_FOLLOWUP_TO        "Followup-To"
#define HEADER_NEWSGROUPS         "Newsgroups"
#define HEADER_ORGANIZATION       "Organization"
#define HEADER_REFERENCES         "References"
#define HEADER_REPLY_TO           "Reply-To"
#define HEADER_SUPERSEDES         "Supersedes"
/* Headers stored in their own fields but here for completeness */
#define HEADER_SUBJECT            "Subject"
#define HEADER_FROM               "From"
#define HEADER_MESSAGE_ID         "Message-Id"
/* Headers beginning with Pan- are for internal use. */
#define PAN_ATTRIBUTION           "Pan-Attribution"
#define PAN_ATTACH_FILE           "Pan-Attach-File"
#define PAN_HEADER_FILENAME       "Pan-Attached-Filename"
#define PAN_BODY                  "Pan-Body"
#define PAN_MAIL_TO               "Pan-Mail-To"
#define PAN_SERVER                "Pan-Server"
#define PAN_REVERSE_PATH          "Pan-Reverse-Path"
#define PAN_LINES_PER_PART        "Pan-Lines-Per-Part"
#define PAN_REPLY_PORTION         "Pan-Reply-Portion"

/**
***  PROTECTED
**/

void         article_constructor            (Article         * article,
                                             Group           * group);

void         article_destructor             (PanObject       * article);

/**
***  PUBLIC LIFE CYCLE
**/

Article*     article_new                    (Group           * group);

Article*     article_dup                    (Group           * group,
                                             const Article   * article);

/**
***  PUBLIC MUTATORS
**/

void         article_set_author_from_header (Article          * article,
                                             const gchar      * header_from);

void         articles_add_flag               (Article        ** articles,
                                              gint              qty,
                                              gushort           flag_to_set);

void         articles_remove_flag            (Article        ** articles,
                                              gint              qty,
                                              gushort           flag_to_remove);

/**
***  Headers
**/

typedef enum { DO_CHUNK=1 } HeaderAction;

void         article_set_header             (Article         * article,
                                             const gchar     * header_name,
                                             const gchar     * header_value,
                                             HeaderAction      action);

void         article_remove_header          (Article         * article,
                                             const gchar     * header_name);

const gchar* article_get_header             (const Article   * article,
                                             const gchar     * header_name);

GPtrArray*   article_get_all_headers        (const Article   * article);

gboolean     article_header_is_extra        (const gchar     * key);

gboolean     article_header_is_internal     (const gchar     * key);

/**
***  PUBLIC ACCESSORS
**/

gboolean     article_has_attachment         (const Article   * article);

gboolean     article_has_body               (const Article   * article);

gchar*       article_get_body               (const Article   * article);

gchar*       article_get_headers            (const Article   * article);

gchar*       article_get_message            (const Article   * article);

gchar*       article_get_attribution_string (const Article   * article);

gchar*       article_get_author_str         (const Article   * article);

gchar*       article_get_short_author_str    (const Article  * article);

void         article_set_from_raw_message   (Article         * article,
                                             const gchar     * text);

/**
***  READ
**/

gboolean     article_is_read                (const Article   * article);

gboolean     article_is_new                 (const Article   * article);

void         article_set_read               (Article         * article,
                                             gboolean          read);

void         articles_set_read              (Article        ** articles,
                                             int               article_qty,
                                             gboolean          read);


#define article_flag_on(a, flag) (a && (a->state&flag))


/**
 * @param article the article whose thread is to be returned
 * @param GPtrArray a container to hold all the articles found in
 *                  this article's thread.
 */
void           article_get_entire_thread    (Article         * article,
                                             GPtrArray       * setme);

void           article_get_subthread        (Article         * article,
                                             GPtrArray       * setme);

void           article_copy_to_folder       (Article         * article,
                                             Group           * from,
                                             Group           * to);

void           article_cancel               (Article         * article);

void           article_supersede            (const Article   * article);

gchar*         article_decode_header        (const gchar     * encoded);

/**
 * This reparents all of article's children to be children of article's parent,
 * removes article from its parent's child list, and
 * clears out all of article's parent/child references.
 */
void           article_isolate             (Article         * article);

gchar*         article_get_thread_message_id (const Article*);

extern const gchar * default_incoming_name_real;

extern const gchar * default_incoming_name_addr;

#endif /* __ARTICLE_H__ */
