/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "acache.h"
#include "debug.h"
#include "decode.h"
#include "log.h"
#include "util.h"
#include "util-mime.h"

/**
*** Private Routines
**/

static gchar*
create_filename (const gchar       * path,
                 const gchar       * subject,
                 const gchar       * default_filename,
                 const GMimePart   * part)
{
	gchar * retval;
	GString * filename;
	debug_enter ("create_filename");

	g_return_val_if_fail (is_nonempty_string(path), NULL);
	g_return_val_if_fail (part!=NULL, NULL);

       	filename = g_string_new (NULL);

	/* first try the filename specified by the user */
	if (!filename->len && is_nonempty_string(default_filename))
		g_string_assign (filename, default_filename);

	/* otherwise try the filename specified by the article */
	if (!filename->len) {
		const gchar * pch = g_mime_part_get_filename (part);
		if (is_nonempty_string(pch))
			g_string_assign (filename, pch);
	}

	/* otherwise try the article's subject */
	if (!filename->len && is_nonempty_string(subject))
		g_string_assign (filename, subject);

	/* otherwise punt */
	if (!filename->len)
		g_string_assign (filename, _("UNKNOWN"));

	/* filter out directory characters */
	if (1) {
		const gchar * pch;
		GString * tmp = g_string_sized_new (filename->len);
		for (pch=filename->str; *pch; ++pch) {
			if (*pch==G_DIR_SEPARATOR)
				g_string_append_c (tmp, '_');
			else if (*pch=='\\')
				g_string_append (tmp, "\\\\");
			else
				g_string_append_c (tmp, *pch);
		}
		g_string_assign (filename, tmp->str);
		g_string_free (tmp, TRUE);
	}

	/* add path */
	if (is_nonempty_string(path)) {
		g_string_prepend_c (filename, G_DIR_SEPARATOR);
		g_string_prepend (filename, path);
	}

	/* make sure we've got a unique file */
	if (file_exists (filename->str)) {
		const gint len = filename->len;
		int i = 1;
		for (;;) {
			g_string_truncate (filename, len);
			g_string_sprintfa (filename, ".%d", i++);
			if (!file_exists (filename->str))
				break;
		}
	}

	retval = filename->str;
	g_string_free (filename, FALSE);
	pan_normalize_filename_inplace (retval);
	g_message (_("Using filename [%s]"), retval);
	debug_exit ("create_filename");
	return retval;
}

static gchar*
create_big_article (GPtrArray * articles)
{
	gchar ** bodies;
	gchar * full = NULL;
	gulong full_len = 0;
	gint i;
	debug_enter ("create_big_article");

	/* sanity clause */
	g_return_val_if_fail (articles!=NULL, NULL);
	g_return_val_if_fail (articles->len, NULL);

	/* get the articles & find out how big the buffer must be */
	full_len = 0;
	bodies = g_new (gchar*, articles->len);
	for (i=0; i!=articles->len; ++i)
	{
		const Article * a;
		gchar * body;

		a = ARTICLE(g_ptr_array_index(articles,i));
		body = bodies[i] = acache_get_message (a->message_id);
		if (body == NULL)
			log_add_va (LOG_ERROR, _("Couldn't find body for message `%s' - decoding may be incomplete or incorrect"), a->subject);
		else
			full_len += strlen(body);
	}

	/* make the buffer */
	if (full_len != 0)
	{
		size_t offset = 0;

		/* allocate the buffer */
		full = g_new (gchar, full_len+1);

		/* move the articles into it */
		for (i=0; i!=articles->len; ++i)
		{
			const gchar * begin = bodies[i];
			const gchar * end;

			/* article 1: keep headers.  all others: drop. */
			if (begin != NULL) {
				const gchar * eoh = strstr (begin, "\n\n");
				if (eoh != NULL) {
					eoh += 2;
					if (begin!=NULL && i==0) {
						memcpy (full+offset, begin, (eoh-begin));
						offset += (eoh-begin);
					}
				}
				begin = eoh;
			}

			/* all articles strip out leading space */
			while (isspace((int)*begin))
				++begin;

			/* if there's a 'cut here', skip everything leading up to it */
			if (1) {
				const gchar * cut = pan_stristr (begin, "cut here");
				if (cut != NULL)
					skip_next_token (cut, '\n', &begin);
			}

			/* if there's a second 'cut here', cut everything after it */
			end = begin + strlen(begin);
			if (1) {
				const gchar * cut = pan_stristr (begin, "cut here");
				if (cut != NULL) {
					while (cut!=begin && *cut!='\n')
						--cut;
					end = cut+1;
				}
			}

			/* if there's a body... */
			if (end != begin)
			{
				/* all articles strip out trailing space */
				--end;
				while (end!=begin && isspace((int)*end))
					--end;

				/* copy the body */
				if (begin!=NULL)
				{
					const size_t len = 1 + (end-begin);
					memcpy (full+offset, begin, len);
					offset += len;
					memcpy (full+offset, "\n", 1);
					offset += 1;
				}
			}
		}

		/* make sure the big item is zero-terminated */
		full[offset] = '\0';
	}

	/* cleanup */
	for (i=0; i!=articles->len; ++i)
		g_free (bodies[i]);
	g_free (bodies);

	debug_exit ("create_big_article");
	return full;
}

static void
get_array_of_decodable_parts_func (GMimePart * part, gpointer data)
{
	GPtrArray * a = (GPtrArray*) data;
	const GMimeContentType * type = g_mime_part_get_content_type (part);
	if ((is_nonempty_string(g_mime_part_get_filename (part))) ||
		(!g_mime_content_type_is_type (type, "text", "*") && !g_mime_content_type_is_type (type, "multipart", "*")))
		g_ptr_array_add (a, part);
}

static void
get_decodable_parts (GPtrArray       * articles,
                     GMimeMessage   ** setme_msg,
                     GPtrArray      * fillme_array)
{
	gchar * text;
	debug_enter ("get_decodable_parts");

	/* sanity check */
	g_return_if_fail (articles!=NULL);

	/* get the message */
	text = create_big_article (articles);
	*setme_msg = pan_g_mime_parser_construct_message (text);
	g_free (text);

	/* build the array of decodable parts */
	g_mime_message_foreach_part (*setme_msg, get_array_of_decodable_parts_func, fillme_array);

	debug_exit ("get_decodable_parts");
}

/**
*** Public Routines
**/

void
decode_article (const decode_data * dd)
{
	static GStaticMutex mutex = G_STATIC_MUTEX_INIT;
	GPtrArray * articles;
	GMimeMessage * mm;
	GPtrArray * attachments;
	GSList * p;
	gint i = 0;
	gboolean use_mutex;
	gboolean success = TRUE;
	Article * first;
	debug_enter ("decode_article");

	/* sanity clause */
	g_return_if_fail (dd!=NULL);
	g_return_if_fail (dd->server!=NULL);
	g_return_if_fail (dd->articles!=NULL);
	for (p=dd->articles; p!=NULL; p=p->next) {
		Article * a = ARTICLE(p->data);	
		g_return_if_fail (a!=NULL);
		g_return_if_fail (is_nonempty_string(a->message_id));
		g_return_if_fail (acache_has_message(a->message_id));
	}

	first = ARTICLE(dd->articles->data);

	/* This is sad, really. individual GMimeMessages are safe enough;
	   the reason we use mutexes here is that decoding a big mp3 takes
	   up so _much_ memory that running two at once makes my system
	   swap to its knees. */
	use_mutex = g_slist_length (dd->articles) > 3;
	if (use_mutex) {
		status_item_emit_status_va (dd->item, _("Waiting to decode \"%s\""), first->subject);
		g_static_mutex_lock (&mutex);
	}

	/* let the client know what we're doing */
	status_item_emit_status_va (dd->item, _("Processing \"%s\""), first->subject);

	/* make a GPtrArray of the articles */
	articles = g_ptr_array_new ();
	for (p=dd->articles; p!=NULL; p=p->next)
		g_ptr_array_add (articles, p->data);

	/* get the decodable parts */
	status_item_emit_status_va (dd->item, _("Decoding \"%s\""), first->subject);
	mm = NULL;
	attachments = g_ptr_array_new ();
	get_decodable_parts (articles, &mm, attachments);

	/* decode & save the parts */
	success = attachments->len != 0;
	for (i=0; success && i!=attachments->len; ++i)
	{
		guint len;
		const gchar * content;
		FILE * fp = NULL;
		const GMimePart * part = (const GMimePart*) g_ptr_array_index (attachments, i);
		gchar * filename = create_filename (dd->path, first->subject, dd->filename, part);

		/* make sure there's content to write */
		content = g_mime_part_get_content (part, &len);
		if (content==NULL || len==0)
			success = FALSE;

		/* try to open a file for writing */
		if (success)
		{
			gchar * path = g_dirname (filename);
			directory_check (path);
			g_free (path);
			fp = fopen (filename, "w+");
			if (fp == NULL)
				success = FALSE;
		}

		/* write the content */
		if (success) {
			fwrite (content, sizeof(char), len, fp);
			fclose (fp);
			fp = NULL;
			status_item_emit_status_va (dd->item, _("Saved \"%s\""), filename);
		}

		/* log the success */
		if (success) {
			gchar * msg = g_strdup_printf (
				_("Decoded \"%s\" from group \"%s\", \"%s\" part #%d"),
				filename,
				(dd->group ? dd->group->name : "Unknown"),
				first->subject, i+1);
			log_add (LOG_IMPORTANT, msg);
			debug0 (DEBUG_DECODE, msg);
			g_free (msg);
		}

		/* remember the filename */
		if (success && !article_get_header (first, PAN_HEADER_FILENAME))
			article_set_header (first, PAN_HEADER_FILENAME, filename, DO_CHUNK);

                /* maybe open the file */
                if (success && dd->open)
                        open_outside_file (filename);

		/* cleanup this iteration */
		g_free (filename);
	}

	/* update the node */
	articles_remove_flag (&first, 1, STATE_DECODE_QUEUED);
	articles_add_flag (&first, 1, (success?STATE_DECODED:STATE_DECODE_FAILED));

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	g_mime_message_destroy (mm);
	g_ptr_array_free (attachments, TRUE);
	if (use_mutex)
		g_static_mutex_unlock (&mutex);
	debug_exit ("decode_article");
}
