/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <string.h>
#include <stdlib.h>

#include <glib.h>
#include <gtk/gtkcheckbutton.h>
#include <gtk/gtkcontainer.h>
#include <gtk/gtklabel.h>
#include <gtk/gtktogglebutton.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-stock.h>

#include "acache.h"
#include "article-thread.h"
#include "articlelist.h"
#include "debug.h"
#include "file-headers.h"
#include "globals.h"
#include "group.h"
#include "grouplist.h"
#include "log.h"
#include "newsrc.h"
#include "nntp.h"
#include "pan-callback.h"
#include "prefs.h"
#include "gui.h"
#include "queue.h"
#include "server.h"
#include "task-bodies.h"
#include "task-headers.h"
#include "task-grouplist.h"
#include "util.h"

/**
***
**/

static PanCallback * group_articles_removed = NULL;

static PanCallback * group_articles_added = NULL;

PanCallback*
group_get_articles_added_callback (void)
{
	if (group_articles_added == NULL)
		group_articles_added = pan_callback_new ();
	return group_articles_added;
}

PanCallback*
group_get_articles_removed_callback (void)
{
	if (group_articles_removed == NULL)
		group_articles_removed = pan_callback_new ();
	return group_articles_removed;
}

/**
***
**/

Group*
group_new (Server * server, const char * name)
{
	Group * g = g_new (Group, 1);
	debug1 (DEBUG_PAN_OBJECT, "group_new: %p", g);
	pan_warn_if_fail (is_nonempty_string(name));
        group_constructor (g, group_destructor, server, name);
	return g;
}

void
group_constructor (Group                * g,
                   PanObjectDestructor    dtor,
		   Server               * server,
		   const char           * name)
{
	g_return_if_fail (server != NULL);
	pan_warn_if_fail (is_nonempty_string(name));

	pan_object_constructor (PAN_OBJECT(g), dtor);

	debug1 (DEBUG_PAN_OBJECT, "group_constructor: %p", g);
	g->articles_dirty = FALSE;
	g->articles_threading_dirty = FALSE;
	g->flags = (gint8)0;
	g->sort_style = (gint8)0;
	g->state_filter = ~((guint16)STATE_FILTER_IGNORED);
	g->article_qty = (gint32)0;
	g->article_read_qty = (gint32)0;
	g->article_low = (gulong)0;
	g->article_high_old = (gulong)0;
	g->permission = '?';
	g->article_high = (gulong)0;
	g->server = server;
	g->name = server_chunk_string (server, name);
	g->description = NULL;
	g->readable_name = NULL;
	g->download_dir = NULL;
	g->chunk = NULL;

	g->_articles_refcount = (gint)0;
	g->_newsrc = NULL;
	g->_articles = NULL;
	g->_one_big_chunk = NULL;
}

void
group_destructor (PanObject * o)
{
	Group * g = GROUP(o);

	g_return_if_fail (o!=NULL);

	debug1 (DEBUG_PAN_OBJECT, "group_destructor: %p", g);

	/* newsrc */
	if (g->_newsrc != NULL)
		pan_object_unref (PAN_OBJECT(g->_newsrc));

	/* articles */
	/* FIXME: make sure the refcount is down to 0 */
	g_free (g->_one_big_chunk);
	if (g->chunk != NULL)
		g_string_chunk_free (g->chunk);
	if (g->_articles != NULL) {
		g_hash_table_destroy (g->_articles);
		g->_articles = NULL;
	}

	pan_object_destructor (o);
}

/***
****
****  PUBLIC ACCESSORS / MUTATORS
****
***/

void
group_set_new (Group * group, gboolean is_new)
{
	g_return_if_fail (group != NULL);

	if (is_new)
		group_set_flags (group, group->flags|GROUP_NEW);
	else
		group_set_flags (group, group->flags&~GROUP_NEW);
}

gboolean 
group_is_new (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return (group->flags & GROUP_NEW) ? 1 : 0;
}

void
group_set_subscribed (Group * group, gboolean subscribed)
{
	g_return_if_fail (group != NULL);

	if (subscribed)
		group_set_flags (group, group->flags|GROUP_SUBSCRIBED);
	else
		group_set_flags (group, group->flags&~GROUP_SUBSCRIBED);
}

gboolean 
group_is_subscribed (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);

	return (group->flags & GROUP_SUBSCRIBED) ? 1 : 0;
}

void
group_set_dirty (Group * group)
{
	g_return_if_fail (group != NULL);
	group->server->groups_dirty = TRUE;
}

void
group_set_download_dir (Group * group, const gchar* download_dir)
{
	group_set_string_field (group, &group->download_dir, download_dir);
}

void
group_set_flags (Group * group, guint flags)
{
	group->flags = flags;
	group_set_dirty (group);
}

void
group_set_state_filter (Group * group, guint state_filter)
{
	g_return_if_fail (group != NULL);

	group->state_filter = state_filter;
	group_set_dirty (group);
}

void
group_set_sort_style (Group * group, int sort_style)
{
	g_return_if_fail (group != NULL);

	group->sort_style = sort_style;
	group_set_dirty (group);
}

void
group_set_article_range (Group * group,
                         gulong article_low,
			 gulong article_high)
{
	group->article_low = article_low;
	group->article_high = article_high;
	group_set_dirty (group);
}

void
group_mark_new_article_number (Group         * group,
                               gulong          everything_above_this_is_new)
{
	group->article_high_old = everything_above_this_is_new;
}

void
group_set_article_qty (Group * group, gint article_qty)
{
	g_return_if_fail (group != NULL);

	if (article_qty != group->article_qty)
	{
		const Group * const_group = (const Group*) group;
		group->article_qty = article_qty;
		group_set_dirty (group);
		grouplist_update_groups (&const_group, 1);
	}
}

void
group_set_article_read_qty (Group * group, gint article_read_qty)
{
	const Group * const_group = (const Group*) group;

	g_return_if_fail (group != NULL);

	article_read_qty = MAX (article_read_qty, 0);
	group->article_read_qty = article_read_qty;
	group_set_dirty (group);
	grouplist_update_groups (&const_group, 1);
}
void
group_inc_article_read_qty (Group * group, gint inc)
{
	if (inc != 0)
		group_set_article_read_qty (group, group->article_read_qty + inc);
}

void
group_mark_all_read (Group * g, gboolean read)
{
	Newsrc * newsrc;

	g_return_if_fail (g != NULL);

	newsrc = group_get_newsrc (g);
	newsrc_set_group_range (newsrc, g->article_low, g->article_high);
	newsrc_mark_all (newsrc, read);
	group_set_article_read_qty (g, (read ? g->article_qty : 0));

	/* FIXME: this should be a fired event to decouple group/articlelist */
	if (g == articlelist_get_group())
		articlelist_reload ();
}

gboolean
group_is_read_only (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return group->permission == 'n';
}

gboolean
group_is_moderated (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return group->permission == 'm';
}

gboolean
group_is_folder (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return group->flags & GROUP_FOLDERS ? 1 : 0;
}

void
group_set_is_folder (Group * group, gboolean folder)
{
	if (!folder != !group_is_folder(group))
	{
		if (folder)
			group->flags |= GROUP_FOLDERS;
		else
			group->flags &= ~GROUP_FOLDERS;
		group_set_dirty (group);
	}
}

/***
****
****  DELETE GROUP
****
***/

static gboolean
real_group_empty_ghrfunc (gpointer key, gpointer value, gpointer data)
{
	pan_object_unref (PAN_OBJECT(value));
	return TRUE;
}
static void
real_group_empty (Group * group, gboolean clear_counts)
{
	g_return_if_fail (group != NULL);

	/* unload group (if it's loaded) */
	if (articlelist_get_group() == group)
		articlelist_set_group (NULL);

	/* clear out the article headers */
	file_headers_destroy (group);

	/* wipe out all the articles... */
	if (group->_articles != NULL)
		g_hash_table_foreach_remove (group->_articles, real_group_empty_ghrfunc, NULL);
	group_set_article_qty (group, 0);
	group_set_article_read_qty (group, 0);

	/* if user wants to clear the range, do that too */
	if (clear_counts)
		group_set_article_range (group, 0, 0);

	/* flush the flags */
	if (group->flags & GROUP_SUBSCRIBED)
		group_set_flags (group, GROUP_SUBSCRIBED);
	else
		group_set_flags (group, 0);
}

/* callback for button clicks in group_empty_dialog */
static void
group_empty_cb (GnomeDialog *dialog,
		int button_number,
		gpointer group)
{
	switch (button_number)
	{
		case 0: /* 'yes' */
		{
			GtkWidget * w = gtk_object_get_data (GTK_OBJECT(dialog), "reset");
			gboolean clear_counts = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(w));
			real_group_empty (group, clear_counts);
			break;
		}
		case 1: /* 'no' */
			break;

		default: /* cancel */
			break;
	};

	gtk_widget_destroy (GTK_WIDGET(dialog));
}

void
group_empty_dialog (Group *group)
{
	GString * str = g_string_new (NULL);
	GtkWidget * dialog = NULL;
	GtkWidget * w = NULL;

	/* sanity check */
	g_return_if_fail (group!=NULL);

	pan_lock();

	/* create dialog */
	g_string_sprintf (str,
	                  _("Clean group \"%s\""),
	                  group_get_readable_name(group));
	dialog = gnome_dialog_new (str->str,
				   GNOME_STOCK_BUTTON_YES,
				   GNOME_STOCK_BUTTON_NO, NULL);

	/* body */
	g_string_sprintf (str,
	                  _("Remove all messages from group: \n%s ?"),
	                  group_get_readable_name(group));
	w = gtk_label_new (str->str);
	gtk_container_add (GTK_CONTAINER(GNOME_DIALOG(dialog)->vbox), w);

	/* "clear count?" check button */
	w = gtk_check_button_new_with_label (_("Reset new message counter"));
	gtk_container_add (GTK_CONTAINER(GNOME_DIALOG(dialog)->vbox), w);
	gtk_object_set_data (GTK_OBJECT(dialog), "reset", w);

	/* listen for clicks */
	gtk_signal_connect (GTK_OBJECT(dialog), "clicked",
			    GTK_SIGNAL_FUNC (group_empty_cb), group);

        gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(Pan.window));
	gtk_widget_show_all (dialog);
	pan_unlock();

	g_string_free (str, TRUE);
}


/**
***  ARTICLE LIST
**/

static void
group_add_articles_ghfunc (gpointer key, gpointer val, gpointer data)
{
	guint i;
	Group * group = GROUP(key);
	GPtrArray * articles = (GPtrArray*)val;

	/* this is so that unreffing from the old group doesn't kill
	   the article -- this ref, plus group_remove_article's unref,
	   has a net change of zero */
	for (i=0; i!=articles->len; ++i)
		pan_object_ref (PAN_OBJECT(g_ptr_array_index(articles,i)));

	/* remove the articles from the other group */
	group_remove_articles (group, articles);

	/* clean up */
	g_ptr_array_free (articles, TRUE);
}

void
group_add_article (Group * group, Article * article)
{
	GPtrArray * a;
	debug_enter ("group_add_article");

	a = g_ptr_array_new ();
	g_ptr_array_add (a, article);
	group_add_articles (group, a, NULL, NULL, NULL);
	g_ptr_array_free (a, TRUE);

	debug_exit ("group_add_article");
}

 
void
group_add_articles_remove_unused (Group            * group,
                                  GPtrArray        * articles,
                                  StatusItem       * status)
{
	GPtrArray * ignored;
	GPtrArray * used;

	/* entry assertions */
	g_return_if_fail (group != NULL);
	g_return_if_fail (articles != NULL);
	g_return_if_fail (articles->len != 0);

	/* add the articles */
	ignored  = g_ptr_array_new ();
	used = g_ptr_array_new ();
	group_add_articles (group, articles, status, used, ignored);

	/* destroy unused articles & remove from the incoming array */
	pan_g_ptr_array_assign (articles, used->pdata, used->len);
	pan_g_ptr_array_foreach (ignored, (GFunc)pan_object_unref, NULL);

	/* cleanup */
	g_ptr_array_free (ignored, TRUE);
	g_ptr_array_free (used, TRUE);
}

void
group_add_articles (Group             * group,
                    const GPtrArray   * articles,
                    StatusItem        * status,
                    GPtrArray         * used,
                    GPtrArray         * ignored)
{
	gint old_qty;
	gint new_read_qty;
	guint i;
	gulong low;
	gulong high;
	gboolean tmp_ref;
	GPtrArray * tmp_used;
	GHashTable * ours;
	debug_enter ("group_add_articles");

	/* sanity checks */
	g_return_if_fail (group!=NULL);
	g_return_if_fail (group->_articles_refcount>=0);
	g_return_if_fail (articles!=NULL);
	if (!articles->len) return;
	g_return_if_fail (articles->len>0);

	/* warn the caller if they're doing something inefficient by adding
	   articles to a group whose refcount is zero.  The old articles in
	   this group have to be loaded too; otherwise they'll be overwritten
	   when we save this group to disk without them. */
	if ((tmp_ref = !group->_articles_refcount))
		group_ref_articles (group, status);

	/**
	***  Remove the articles from their old groups.
	***  Batching the removal makes the code uglier but
	***  it runs faster.
	**/

	if (1)
	{
		GHashTable * remove_from_old = NULL;

		for (i=0; i!=articles->len; ++i)
		{
			Article * article = ARTICLE(g_ptr_array_index(articles,i));
			Group * oldgroup = article->group;
			gboolean array_existed;
			GPtrArray * to_remove;

			if (oldgroup==NULL || oldgroup==group)
				continue;

			/* ensure the hash table exists */
			if (remove_from_old == NULL)
				remove_from_old = g_hash_table_new (g_direct_hash, g_direct_equal);

			/* add this array to the array of articles to remove from its group */
			to_remove = g_hash_table_lookup (remove_from_old, oldgroup);
			array_existed = to_remove != NULL;
			if (!array_existed)
				to_remove = g_ptr_array_new ();
			g_ptr_array_add (to_remove, article);

			/* we hash the 'remove these' array by group */
			if (!array_existed)
				g_hash_table_insert (remove_from_old, oldgroup, to_remove);
		}

		if (remove_from_old != NULL)
		{
			g_hash_table_foreach (remove_from_old, group_add_articles_ghfunc, NULL);
			g_hash_table_destroy (remove_from_old);
		}
	}

	/**
	***  Get a handle on our current stats...
	**/

	low = group->article_low;
	high = group->article_high;
	ours = group_get_articles (group);
	old_qty = (int) g_hash_table_size (ours);

	/**
	***  Now try to add the articles that the client passed in
	**/

	new_read_qty = 0;
	tmp_used = g_ptr_array_new ();
	pan_g_ptr_array_reserve (tmp_used, articles->len);
	for (i=0; i!=articles->len; ++i)
	{
		Article * a = ARTICLE(g_ptr_array_index(articles,i));

		if (g_hash_table_lookup (ours, a->message_id) == NULL)
		{
			a->group = group;
			g_hash_table_insert (ours, (gpointer)a->message_id, a);
			g_ptr_array_add (tmp_used, a);

			if (a->number<low)
				low  = a->number;
			if (a->number>high)
				high = a->number;
			if (newsrc_is_article_read (group_get_newsrc(group), a->number))
				++new_read_qty;
		}
		else if (ignored != NULL)
		{
			g_ptr_array_add (ignored, a);
		}
	}

	/**
	***  Now update our stats if articles were added
	**/

	if (tmp_used->len != 0)
	{
		const Group * const_group = group;

		/* maybe update the article range */
		if (high!=group->article_high || low!=group->article_low)
			group_set_article_range (group, low, high);

		/* update the group article stats */
		group->articles_threading_dirty = TRUE;
		group_set_article_qty (group, g_hash_table_size(ours));
		if (old_qty != 0)
			new_read_qty += group->article_read_qty;
		group_set_article_read_qty (group, new_read_qty);
		group_set_articles_dirty (group, TRUE);
		grouplist_update_groups (&const_group, 1);

		pan_callback_call (group_get_articles_added_callback(), group, tmp_used);
	}

	if (used != NULL)
		pan_g_ptr_array_assign (used, tmp_used->pdata, tmp_used->len);

	/* cleanup */
	g_ptr_array_free (tmp_used, TRUE);
	if (tmp_ref)
		group_unref_articles (group, status);
	debug_exit ("group_add_articles");
}

void
group_set_articles_dirty (Group * group, gboolean dirty)
{
	g_return_if_fail (group!=NULL);

	group->articles_dirty = dirty;
}

void
group_thread_if_needed (Group * group, StatusItem * status)
{
	debug_enter ("group_thread_if_needed");

	g_return_if_fail (group != NULL);

	if (group->articles_threading_dirty && group->_articles!=NULL && g_hash_table_size(group->_articles)!=0)
	{
		GPtrArray * a;

		a = group_get_article_array (group);
		thread_articles (a, status);
		check_multipart_articles (a);
		group->articles_threading_dirty = FALSE;

		g_ptr_array_free (a, TRUE);
	}

	debug_exit ("group_thread_if_needed");
}

 
gpointer
group_get_article_by_message_id (Group * group, const gchar * message_id)
{
	g_return_val_if_fail (group!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string(message_id), NULL);
	return g_hash_table_lookup (group_get_articles(group), message_id);
}

void
group_remove_article (Group * group, Article * article)
{
	GPtrArray * a = g_ptr_array_new ();
	g_ptr_array_add (a, article);
	group_remove_articles (group, a);
	g_ptr_array_free (a, TRUE);
}

void
group_remove_articles (Group * group, const GPtrArray * articles)
{
	guint i;
	int removed_read;
	int removed_total;
	GPtrArray * tmp_remove;
	GHashTable * ours;
	debug_enter ("group_remove_articles");

	/* sanity checks */
	g_return_if_fail (group != NULL);
	g_return_if_fail (articles != NULL);
	if (!articles->len) return;
	for (i=0; i!=articles->len; ++i)
		g_assert (ARTICLE(g_ptr_array_index(articles,i))->group==group);

	/* let everyone know that articles are going to be removed */
	tmp_remove = pan_g_ptr_array_dup ((GPtrArray*)articles);
	sort_articles ((Article**)tmp_remove->pdata, tmp_remove->len, ARTICLE_SORT_MSG_ID, TRUE);
	pan_callback_call (group_get_articles_removed_callback(), group, tmp_remove);

	/* is this the last gasp for these articles? */
	for (i=0; i!=tmp_remove->len; ++i) {
		Article * article = ARTICLE(g_ptr_array_index(tmp_remove,i));
		if (PAN_OBJECT(article)->ref_count == 1) /* about to die */
			article_isolate (article);
	}

	/**
	***  Try to remove the articles
	**/

	ours = group_get_articles (group);
	removed_total = 0;
	removed_read = 0;
	for (i=0; i!=tmp_remove->len; ++i)
	{
		Article * a = ARTICLE(g_ptr_array_index(tmp_remove,i));
		Article * our_a = ARTICLE(g_hash_table_lookup (ours, a->message_id));

		if (our_a != NULL)
		{
			g_hash_table_remove (ours, a->message_id);

			++removed_total;
			if (article_is_read (our_a))
				++removed_read;

			our_a->group = NULL;
			pan_object_unref (PAN_OBJECT(our_a));
		}
	}

	/**
	***  Update the stats
	**/

	if (removed_total != 0)
	{
		const Group * const_group = group;

		/* update the articles stats */
		group->articles_threading_dirty = TRUE;
		group->article_qty -= removed_total;
		group->article_read_qty -= removed_read;
		group_set_articles_dirty (group, TRUE);
		grouplist_update_groups (&const_group, 1);
	}

	/* cleanup */
	g_ptr_array_free (tmp_remove, TRUE);
	debug_exit ("group_remove_articles");
}

void
group_expire_old_articles (Group * g)
{
	guint i;
	GPtrArray * tmp;
	GPtrArray * removeme;
	debug_enter ("group_expire_old_articles");

	g_return_if_fail (g != NULL);

	/* find old articles to remove */
	tmp = group_get_article_array (g);
	removeme = g_ptr_array_new ();
	pan_g_ptr_array_reserve (removeme, tmp->len);
	for (i=0; i!=tmp->len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(tmp,i));
		if ((a->number<g->article_low) || (a->number>g->article_high))
			g_ptr_array_add (removeme, a);
	}

	/* remove the old articles */
	if (removeme->len != 0)
	{
		log_add_va (LOG_INFO, _("Expired %u articles from `%s'"),
			removeme->len,
			group_get_readable_name(g));

		group_remove_articles (g, removeme);
	}

	/* cleanup */
	g_ptr_array_free (tmp, TRUE);
	g_ptr_array_free (removeme, TRUE);
	debug_exit ("group_expire_old_articles");
}


/***
****
***/

void
group_ref_articles (Group * group, StatusItem * status)
{
	debug_enter ("group_ref_articles");

	/* sanity clause */
	g_return_if_fail (group != NULL);
	g_return_if_fail (group->_articles_refcount >= 0);

	/* if we're up from zero, load the headers */
	if (++group->_articles_refcount == 1)
		file_headers_load (group, status);

	/*g_message ("group %s refcount inc to %d", group->name, group->_articles_refcount);*/
	group_set_article_qty (group, g_hash_table_size(group->_articles));
	debug_exit ("group_ref_articles");
}

static void
group_unref_articles_ghfunc (gpointer key, gpointer val, gpointer data)
{
	pan_object_unref (PAN_OBJECT(val));
}
void
group_unref_articles (Group * group, StatusItem * status)
{
	debug_enter ("group_unref_articles");

	/* sanity clause */
	g_return_if_fail (group != NULL);
	g_return_if_fail (group->_articles_refcount > 0);

	/* if down to zero, save & flush the articles */
	if (!--group->_articles_refcount)
	{
		file_headers_save (group, status);

		/* free the articles */
		g_hash_table_foreach (group->_articles, group_unref_articles_ghfunc, NULL);
		g_hash_table_destroy (group->_articles);
		group->_articles = NULL;

		/* free the memory used by those articles */
		replace_gstr (&group->_one_big_chunk, NULL);
		if (group->chunk != NULL) {
			g_string_chunk_free (group->chunk);
			group->chunk = NULL;
		}
	}

	/*g_message ("group %s refcount dec to %d", group->name, group->_articles_refcount);*/
	debug_exit ("group_unref_articles");
}

/**
***
**/

Group*
folder_get_by_name (const gchar * foldername)
{
	Server * server;

	/* sanity clause */
	g_return_val_if_fail (is_nonempty_string(foldername), NULL);

	/* get the folder server... */
	server = server_get_by_name (INTERNAL_SERVER_NAME);
	g_return_val_if_fail (server!=NULL, NULL);

	/* get the folder... */
	return group_get_by_name (server, foldername);
}


const gchar*
group_get_readable_name (const Group * group)
{
	const gchar * retval;

	if (group == NULL)
	{
		pan_warn_if_reached ();
		retval = _("(No Group)");
	}
	else
	{
		retval = is_nonempty_string (group->readable_name)
			? group->readable_name
			: group->name;
	}

	return retval;
}

/**
***
**/

const gchar*
group_chunk_string (Group         * group,
                    const gchar   * chunkme)
{
	gchar * retval = NULL;

	g_return_val_if_fail (group!=NULL, NULL);

	if (group->chunk == NULL)
		group->chunk = g_string_chunk_new (8192);

	if (chunkme != NULL)
		retval = g_string_chunk_insert_const (group->chunk, chunkme);

	return retval;
}

void
group_set_string_field (Group         * group,
	                const gchar  ** field_ptr,
			const gchar   * chunkme_replacement)
{
	g_return_if_fail (group!=NULL);
	g_return_if_fail (group->server!=NULL);
	g_return_if_fail (field_ptr!=NULL);

	*field_ptr = chunkme_replacement==NULL
		? NULL
		: server_chunk_string (group->server, chunkme_replacement);
}


/***
****
****  NEW FOLDER DIALOG
****
***/

static void
new_folder_cb (gchar* string, gpointer data)
{
	Group * group = NULL;
	Server * server = server_get_by_name (INTERNAL_SERVER_NAME);
	gchar * tmp;
	debug_enter ("new_folder_cb");

	/* user hit cancel */
	if (string == NULL)
		return;

	/* make sure the user entered something */
	tmp = g_strdup (string);
	g_strstrip (tmp);
	if (!is_nonempty_string (tmp)) {
		pan_error_dialog (_("Invalid folder name: \"%s\""), string);
		g_free (tmp);
		return;
	}

	/* make sure there's not already a folder with that name */
	g_strdown (tmp);
	group = folder_get_by_name (tmp);
	if (group != NULL) {
		pan_error_dialog (_("Folder \"%s\" already exists."), string);
		g_free (tmp);
		return;
	}

	/* make sure it's not in the reserved namespace */
	if (!strncmp(tmp, "pan.", 4)) {
		pan_error_dialog (_("Please don't begin folders with \"pan.\"; it confuses me terribly."));
		g_free (tmp);
		return;
	}

	/* create new folder */
	group = group_new (server, tmp);
	group_set_is_folder (group, TRUE);
	group_set_dirty (group);
	server_add_groups (server, &group, 1, NULL, NULL);

	/* cleanup */
	g_free (tmp);
	debug_exit ("new_folder_cb");
}

void
group_new_folder_dialog (void)
{
	gchar * prompt = g_strdup (_("New Folder Name:"));
	gnome_request_dialog (FALSE, prompt, NULL, 128,
			      new_folder_cb,
			      NULL, GTK_WINDOW(Pan.window));
	g_free (prompt);
}

void
group_download_all_articles (Group * group)
{
	Task * task = NULL;
	GPtrArray * a = NULL;

	g_return_if_fail (group != NULL);
	g_return_if_fail (group->_articles != NULL);

	group_ref_articles (group, NULL);
	a = group_get_article_array (group);
	if (a->len)
		task = TASK(task_bodies_new (group, a));
	if (task != NULL)
		queue_add (task);
	group_unref_articles (group, NULL);

	/* cleanup */
	g_ptr_array_free (a, TRUE);
}

Newsrc*
group_get_newsrc (Group * group)
{
	g_return_val_if_fail (group!=NULL, NULL);

	if (group->_newsrc == NULL)
		group->_newsrc = newsrc_new (NULL, group->article_low, group->article_high);

	return group->_newsrc;
}

gboolean
group_has_newsrc (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return group->_newsrc != NULL;
}

GHashTable*
group_get_articles (Group * group)
{
	g_return_val_if_fail (group!=NULL, NULL);

	if (group->_articles == NULL)
		group->_articles = g_hash_table_new (g_str_hash, g_str_equal);

	return group->_articles;
}

GPtrArray*
group_get_article_array (Group * group)
{
	GHashTable * hash;
	GPtrArray * retval;

	g_return_val_if_fail (group!=NULL, NULL);

	hash = group_get_articles (group);
	retval = g_ptr_array_new ();
	pan_hash_to_ptr_array (hash, retval);

	return retval;
}
