/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#ifndef __QUEUE_H__
#define __QUEUE_H__

#include <glib.h>

#include "task.h"



extern void queue_init (int max_sockets, int max_tries);

extern void queue_add (Task*);

extern void queue_shutdown (gboolean save_tasks);


/**
***  This rest of this file is for the queue manager.
**/



/**
***  CALLBACKS
***
***  These callbacks are called when the queue changes.
**/

/**
 * @param call_object the Task* added.
 * @param call_arg the GINT_TO_POINTER() of the position in the queue
 */
extern PanCallback *queue_task_added;

/**
 * @param call_object the Task* being removed.
 * @param call_arg the GINT_TO_POINTER() of the position in the queue
 */
extern PanCallback *queue_task_removed;

/**
 * @param call_object the Task* being repositioned.
 * @param call_arg int* to two ints: the the old and new position in the queue
 */
extern PanCallback *queue_task_moved;

extern PanCallback *queue_task_status_changed;

extern PanCallback *queue_max_tries_changed;

/**
***  QUEUE MANIPULATION
**/

/**
 * @return a GSList of Task* currently in the queue.
 */
extern const GSList* queue_get_tasks (void);

/**
 * Add an item to the queue.  This increases the task's refcount by 1.
 * @param task the task to insert.
 * @param index into the queue, or -1 to add at the end.
 */
extern void queue_task_insert (Task*, int index);

/**
 * Remove an item from the queue.  This decreases the task's refcount by 1.
 * @param task the task to remove
 */
extern void queue_task_remove (Task*);

/**
 * Reposition an item in the queue.
 * @param task the task to move
 * @param index the new position in the queue, or -1 for the end.
 */
extern void queue_task_move (Task*, int index);

/**
 * Requeue a failed task.
 * @param task
 * @index the new position in the queue, or -1 for the end.
 */
extern void queue_task_requeue_failed (Task*, int index);


extern void queue_task_abort (Task*);

/**
***  CONNECTION
**/

extern int queue_get_max_sockets (void);

extern void queue_set_max_sockets (int);

extern int queue_get_max_tries (void);

extern void queue_set_max_tries (int);

extern gboolean queue_has_dedicated_article_thread (void);

extern void queue_set_dedicated_article_thread (gboolean b);

extern void queue_wakeup (void);

/**
***  TASK STATUS
**/

typedef enum
{
	QUEUE_TASK_STATUS_NOT_QUEUED	= 0,
	QUEUE_TASK_STATUS_QUEUED	= 1,
	QUEUE_TASK_STATUS_RUNNING	= 2,
	QUEUE_TASK_STATUS_FAILED	= 4,
	QUEUE_TASK_STATUS_ABORTING	= 5,
	QUEUE_TASK_STATUS_CONNECTING	= 6
}
QueueTaskStatus;

/**
 * @return what the queue thinks the specified Task is doing,
 *         or 0 if the specified item is not in the queue.
 */
extern QueueTaskStatus queue_get_task_status (const Task*);

guint queue_get_running_task_count (void);


#endif /* __QUEUE_H__ */
