/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <glib.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>

#include "article.h"
#include "debug.h"
#include "log.h"
#include "nntp.h"
#include "task-grouplist.h"
#include "util.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static gint task_grouplist_run (Task*);

static int nntp_grouplist_download_all (TaskGroupList*);

static int nntp_grouplist_download_new (TaskGroupList*);

static gchar* task_grouplist_describe (const StatusItem*);

static void task_grouplist_destructor (PanObject*);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

PanObject*
task_grouplist_new (Server *server,
                    gboolean high_priority,
                    GrouplistDownloadType dl_type)
{
	TaskGroupList *item = g_new0 (TaskGroupList, 1);

	debug1 (DEBUG_PAN_OBJECT, "task_grouplist_new: %p", item);
	
	/* construct superclass... */
	task_constructor (TASK(item),
		task_grouplist_destructor,
		task_grouplist_describe,
		task_grouplist_run, server, high_priority, TRUE);

	/* construct this class... */
	item->server = server;
	item->download_type = dl_type;
	item->got_list = FALSE;
	item->groups = g_ptr_array_new ();

	return PAN_OBJECT(item);
}

/*****
******
*****/

static void
task_grouplist_destructor (PanObject * o)
{
	TaskGroupList * task = TASK_GROUPLIST(o);

	/* destruct this class */
	debug1 (DEBUG_PAN_OBJECT, "task_grouplist_destructor: %p", o);
	pan_g_ptr_array_foreach (task->groups, (GFunc)pan_object_unref, NULL);
	g_ptr_array_free (task->groups, TRUE);

	/* destruct the superclass */
	task_destructor (o);
}

static gchar*
task_grouplist_describe (const StatusItem* item)
{
	const gchar* action = NULL;

	/* sanity checks */
	g_return_val_if_fail (item!=NULL, NULL);

	switch (TASK_GROUPLIST(item)->download_type) {
		case GROUPLIST_ALL: action=_("all groups"); break;
		case GROUPLIST_NEW: action=_("new groups"); break;
		default: pan_warn_if_reached(); action=_("BUG IN CODE"); break;
	}

	return g_strdup_printf ( _("Getting %s from %s"),
		action,
		TASK_GROUPLIST(item)->server->name );
}

static gint
task_grouplist_run (Task *item)
{
	TaskGroupList *item_g = TASK_GROUPLIST(item);
	Server *server = item->server;
	gint result;
	guint i;

	status_item_emit_status_va (STATUS_ITEM(item),
		_("Got %u groups for \"%s\""),
		item_g->groups->len,
		server->name);

	/**
	***  Try to get the groups from the server
	**/

	switch (item_g->download_type)
	{
		case GROUPLIST_ALL:
			result = nntp_grouplist_download_all (item_g);
			break;
		case GROUPLIST_NEW:
			result = nntp_grouplist_download_new (item_g);
			break;
		default:
			pan_warn_if_reached ();
			result = TASK_FAIL_HOPELESS;
			break;
	}
	if (result != TASK_SUCCESS)
		return result;

	/**
	***  The downloading of groups is done;
	***  update Pan with the list that we got
	**/

	status_item_emit_status_va (STATUS_ITEM(item),
		_("Merging groups for \"%s\""), server->name);

	/* for the next time we call GROUPLIST_NEW... */
	server->last_newgroup_list_time = time(0);

	/* add the groups we don't already have */
	if (1) {
		GPtrArray * unused = g_ptr_array_new ();
		server_add_groups (server, (Group**)item_g->groups->pdata, item_g->groups->len, NULL, unused);
		for (i=0; i!=unused->len; ++i)
			pan_object_unref (PAN_OBJECT(g_ptr_array_index(unused,i)));
		g_ptr_array_free (unused, TRUE);
	}
	g_ptr_array_set_size (item_g->groups, 0);

	return TASK_SUCCESS;
}


/*---[ nntp_grouplist_download ]--------------------------------------
 *
 *--------------------------------------------------------------------*/

static int
compare_name_ppgroup (const void * a, const void * b)
{
	return strcmp ((const gchar*)a, (*(const Group**)b)->name);
}

static int
compare_ppgroup_ppgroup (const void * a, const void * b)
{
	return strcmp ((*(const Group**)a)->name,
	               (*(const Group**)b)->name);
}


static int
nntp_grouplist_download_all (TaskGroupList * item)
{
	GPtrArray * groups = item->groups;
	const gboolean * const abort = &TASK(item)->hint_abort;
	PanSocket * sock = TASK(item)->sock;
	Server * server = TASK(item)->server;
	int read_err = 0;
	int description_count = 0;
	gint val = 0;
	const gchar * response = NULL;
	StatusItem * status = STATUS_ITEM(item);

	status_item_emit_status_va (status,
		_("Got %u groups for \"%s\""),
		groups->len,
		server->name);

	/**
	***  Get the group list, if we haven't already
	**/

	if (!item->got_list)
	{
		gint val;
		const gchar * response = NULL;

		val = *abort ? -1 : nntp_command (status, sock, &response, "LIST");
		if (val != 215) {
			status_item_emit_error_va (status, ("Group LIST command failed: %s"), response);
			return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
		}

		/* read the group list from the server. */
		while (!*abort && !(read_err=pan_socket_getline(sock,&response)))
		{
			const gchar * march = response;
			Group * group;
			gchar * name;
			gchar * permission;

			/* check for end of list */
			if (!strcmp(response,".\r\n"))
				break;

			/* parse the data line */
			name = get_next_token_str (march, ' ', &march);
			skip_next_token (march, ' ', &march); /* skip low number */
			skip_next_token (march, ' ', &march); /* skip high number */
			permission = get_next_token_str (march, ' ', &march);

			/* create a new group object */
			group = group_new (TASK(item)->server, name);
			group->permission = *permission;
			group_set_dirty (group); /* make sure it gets saved */
			g_ptr_array_add (groups, group);

			/* periodic status feedback */
			if (groups->len % 256 == 0)
			{
				char buf[32];
				commatize_ulong ((gulong)groups->len, buf);
				status_item_emit_status_va (status,
					_("Got %s groups for \"%s\""),
					buf, server->name);
			}

			/* cleanup */
			g_free (name);
			g_free (permission);
		}

		if (!read_err)
			item->got_list = TRUE;

		if (read_err)
			return TASK_FAIL;
	}

	/* sort so that we can bsearch them... */
	qsort (groups->pdata,
	       groups->len,
	       sizeof(Group*),
	       compare_ppgroup_ppgroup);


	/**
	***  DESCRIPTIONS
	**/

	/* now try to get the descriptions...
	   this is implemented as a second pass because the overlap between
	   LIST and LIST NEWSGROUPS isn't well defined:  The latter may only
	   show the groups with a description, and may not be available
	   on older servers.  To make sure we get all the groups, and
	   descriptions for as many as we can, we use both commands.
	   -- csk */

	val = *abort ? -1 : nntp_command (status, sock, &response, "LIST NEWSGROUPS");
	if (val != 215) {
		/* no big deal; possibly server doesn't support this command */
		status_item_emit_error_va (status, _("List Newsgroups failed: %s"), response);
		return val==-1 ? TASK_FAIL : TASK_SUCCESS;
	}

	/* march through the descriptions */
	description_count = 0;
	while (!*abort && !(read_err=pan_socket_getline(sock,&response)))
	{
		const gchar * pch = response;
		gchar * name = NULL;
		gchar * description = NULL;

		/* check for end of line */
		if (!strcmp(response,".\r\n"))
			break;	

		/* get the name */
		while (*pch && !isspace((int)*pch)) ++pch;
		name = g_strndup (response, pch-response);
		response = pch;

		/* get the description */
		if ((pch = strstr (response, "\r\n"))) {
			description = g_strndup (response, pch-response);
			g_strstrip (description);
		}

		/* let the user know we're still on the job */
		++description_count;
		if (description_count % 43 == 0)
		{
			char buf[32];
			commatize_ulong ((gulong)description_count, buf);
			status_item_emit_status_va (
				STATUS_ITEM(item),
				_("Got %s descriptions for \"%s\""),
				buf, server->name);
		}

		/* try to update the group's description field */
		if (is_nonempty_string(name) &&
		    is_nonempty_string(description))
		{
			int i;
			gboolean exact_match = FALSE;

			i = lower_bound (name,
			                 groups->pdata, groups->len,
					 sizeof(gpointer),
					 compare_name_ppgroup,
					 &exact_match);
			if (exact_match)
			{
				Group * g = GROUP(g_ptr_array_index(groups,i));
				group_set_string_field (g,
				                        &g->description,
				                        description);
			}
		}

		g_free (name);
		g_free (description);
	}
	if (read_err)
		return TASK_FAIL;

	log_add_va (LOG_INFO, _("Got %d groups for server `%s'"),
		groups->len,
		server->name);

	return TASK_SUCCESS;
}


/*--------------------------------------------------------------------
 * get "new" groups
 * 
 * based on RFC977 - JEL
 *--------------------------------------------------------------------*/
static int
nntp_grouplist_download_new (TaskGroupList * item)
{
	GPtrArray * groups = item->groups;
	const gboolean * const abort = &TASK(item)->hint_abort;
	const Server *server = TASK(item)->server;
	const gchar * response = NULL;
	gint count = 0;
	gint val = 0;
	char datestr[64];
	struct tm gmt;
	PanSocket * sock = TASK(item)->sock;
	StatusItem * status = STATUS_ITEM(item);
	Group * group = NULL;

	status_item_emit_status_va (STATUS_ITEM(item),
		_("Retrieving 0 new groups for \"%s\""), server->name);

	/* get the date of the last grouplist update */
	gmtime_r (&server->last_newgroup_list_time, &gmt);
	strftime (datestr, sizeof(datestr), "%Y%m%d %H%M%S GMT", &gmt);

	/* ask the server which groups have been added since then */
	val = *abort ? -1 : nntp_command_va (status, sock, &response, "NEWGROUPS %s", datestr);
	if (val==501) /* 501 Usage: NEWGROUPS yymmdd hhmmss ["GMT"] */
	{
		/* Paul McGarrry <mcgarray@tig.com.au> and his Newsmaster
		 * Simon Lyall <simon.lyall@ihug.co.nz> report that nntpcache
		 * isn't handling four-digit years.  So if the four-digit year
		 * fails, let's fall back to a two-digit year.
		 */
		val = *abort ? -1 : nntp_command_va (status, sock, &response, "NEWGROUPS %s", datestr+2);
	}
	if (val != 231) /* 231 = list of new groups follows */
	{
		status_item_emit_error_va (status, _("New groups retrieval failed: %s"), response);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
	}

	/**
	***  Get the list of new groups
	**/

	if (*abort || pan_socket_getline (sock, &response))
		return TASK_FAIL;

	while (strncmp(response, ".\r\n", 3))
	{
		gchar * name;

		/* create a Group */
		name = get_next_token_str (response, ' ', NULL);
		group = group_new (TASK(item)->server, name);
		group_set_dirty (group); /* make sure it gets saved */
		g_free (name);
		group->flags |= GROUP_NEW;
		g_ptr_array_add (groups, group);

		if (++count % 43 == 0)
		{
			status_item_emit_progress (STATUS_ITEM(item), count);

			status_item_emit_status_va (STATUS_ITEM(item),
				_("Retrieving %d new groups for \"%s\""), 
				count, server->name);
		}

		if (*abort || pan_socket_getline (sock, &response))
		{
			pan_g_ptr_array_foreach (groups, (GFunc)pan_object_unref, NULL);
			return TASK_FAIL;
		}
	}

	log_add_va (LOG_INFO, _("Got %d new groups"), count);

	status_item_emit_status_va (STATUS_ITEM(item),
		_("Got %d news groups for \"%s\""), count, server->name);

	return TASK_SUCCESS;
}
