/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <gnome.h>

#include <pan/base-prefs.h>

#include <pan/dialogs/dialogs.h>
#include <pan/globals.h>
#include <pan/grouplist.h>
#include <pan/prefs.h>
#include <pan/util.h>


/* Private structure for group properties */
typedef struct
{
	GnomePropertyBox * dialog;
	GtkWidget * nickname_entry;
	GtkWidget * download_dir; /* GnomeFileEntry */
	Group * group;
	Server * server;
}
GroupProps;


static GtkWidget* group_props_directories_page (GroupProps *gp);
static void group_props_apply (GtkWidget *widget, gint page_num, GroupProps *gp);
static void group_props_cancel (GtkWidget *widget, GroupProps *gp);


/* Apply button callback */
static void
group_props_apply (GtkWidget *widget, gint page_num, GroupProps *gp)
{
	Group * group = gp->group;
	const gchar * cpch;

	g_return_if_fail (gp != NULL);

	/* nickname */
	cpch = gtk_entry_get_text (GTK_ENTRY (gp->nickname_entry));
	if (cpch==NULL || !*cpch || !strcmp (cpch, group->name))
		cpch = NULL;
	replace_gstr (&group->readable_name, g_strdup(cpch));

	/* download directory */
	cpch = gtk_entry_get_text (GTK_ENTRY (gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY(gp->download_dir))));
	if (cpch==NULL || !*cpch || !strcmp (cpch, download_dir))
		cpch = NULL;
	group_set_download_dir (group, cpch);

	/* update UI */
	group_set_dirty (group);
	grouplist_update_groups ((const Group **)&group, 1);
}


/* Cancel/Close button callback */
static void
group_props_cancel (GtkWidget *widget, GroupProps *gp)
{
	gtk_widget_destroy (GTK_WIDGET (gp->dialog));

	g_free (gp);
}


/* Build and return directory-specific options page */
static GtkWidget *
group_props_directories_page (GroupProps *gp)
{
	GtkWidget * table;
	GtkWidget * w;
	const gchar * cpch;

	table = gtk_table_new (2, 2, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);
        gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
        gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);

	/* nickname label */
	w = gtk_label_new (_("Group Name"));
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w,
			  0, 1,
			  0, 1,
			  GTK_FILL | GTK_EXPAND,
			  GTK_FILL,
			  4, 0);

	/* nickname entry */
	w = gp->nickname_entry = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY(w), group_get_readable_name (gp->group));
	gtk_signal_connect_object (GTK_OBJECT(w),
				   "changed",
				   GTK_SIGNAL_FUNC (gnome_property_box_changed),
				   GTK_OBJECT (gp->dialog));
	gtk_table_attach (GTK_TABLE (table), w,
			  1, 2,
			  0, 1,
			  GTK_FILL | GTK_EXPAND,
			  GTK_FILL,
			  4, 0);

	/* download dir label */
	w = gtk_label_new (_("Directory for saving attached files"));
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w,
			  0, 1,
			  1, 2,
			  GTK_FILL | GTK_EXPAND,
			  GTK_FILL,
			  4, 0);

	/* download dir file entry */
	w = gp->download_dir = gnome_file_entry_new ("Group Properties", ("Directory for saving attached files"));
	cpch = gp->group->download_dir;
	if (!is_nonempty_string(cpch))
		cpch = download_dir;
	gtk_entry_set_text (GTK_ENTRY (gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (w))), cpch);
	gnome_file_entry_set_directory (GNOME_FILE_ENTRY(w), TRUE);
	gtk_signal_connect_object (GTK_OBJECT(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY(w))),
				   "changed",
				   GTK_SIGNAL_FUNC (gnome_property_box_changed),
				   GTK_OBJECT (gp->dialog));
	gtk_table_attach (GTK_TABLE (table), w,
			  1, 2,
			  1, 2,
			  GTK_FILL,
			  GTK_FILL,
			  4, 0);


	return table;
}


/**
 * dialog_group_props:
 * @group: The group to set the properties for.
 * 
 * Open the group properties dialog.
 **/
void
dialog_group_props (Group *group)
{
	gchar *buf;
	GroupProps *gp;

	g_return_if_fail (group != NULL);

	gp = g_new0 (GroupProps, 1);
	gp->dialog = GNOME_PROPERTY_BOX (gnome_property_box_new());
	gp->group = group;
	gp->server = grouplist_get_server ();

	gtk_notebook_append_page (GTK_NOTEBOOK (gp->dialog->notebook),
				  group_props_directories_page (gp),
				  gtk_label_new (_("Group Properties")));

	/* pretty up the title */
	buf = g_strdup_printf (
		"Properties for: %s",
		group_get_readable_name(group));
	gtk_window_set_title (GTK_WINDOW (gp->dialog), buf);
	g_free (buf);

	/* connect signals */
	gtk_signal_connect (GTK_OBJECT (gp->dialog),
			    "apply",
			    GTK_SIGNAL_FUNC (group_props_apply),
			    gp);
	gtk_signal_connect (GTK_OBJECT (gp->dialog),
			    "destroy",
			    GTK_SIGNAL_FUNC (group_props_cancel),
			    gp);

	gnome_dialog_set_parent (GNOME_DIALOG(gp->dialog),
	                         GTK_WINDOW(Pan.window));
	gtk_widget_show_all (GTK_WIDGET(gp->dialog));
}
