///
// Copyright (C) 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "pageviewent.h"
#include "document/document.h"

namespace { 			// local utilities

  typedef std::vector<Gdk::Point> ScrPolygon;
  
  ScrPolygon polygonToScreen(const View& view, const Polygon& poly) {
    // Note; this should be done with the generic copy algorithm
    ScrPolygon result;
    for(Polygon::const_iterator i = poly.begin(); i != poly.end(); ++i)
      result.push_back(view.pt2scr(*i));
    return result;
  }
  
}

PageViewent::PageViewent(View& view, const FrameRef pageframe)
  : GroupViewent(view, pageframe), page(pageframe)
{}

PageViewent::~PageViewent() {}

namespace {
  void draw_guides(const Page &page, View &view) {
    Glib::RefPtr<Gdk::Drawable> win = view.get_win();
    Glib::RefPtr<Gdk::GC> gc = view.get_gc();

    gc->set_line_attributes(1, Gdk::LINE_ON_OFF_DASH, Gdk::CAP_BUTT, 
			    Gdk::JOIN_MITER);
    gc->set_foreground(view.get_color(Color::guide));
    
    for(Page::Guides::const_iterator i = page.guides.begin(); 
	i != page.guides.end(); ++i) {
      Gdk::Point start, end;
      if(i->is_horizontal) {
	start = view.pt2scr(Vector(0, i->position));
	end = view.pt2scr(Vector(page.get_width(), i->position));
      } else {
	start = view.pt2scr(Vector(i->position, 0));
	end = view.pt2scr(Vector(i->position, page.get_height()));
      }
      win->draw_line(gc, 
		     start.get_x(), start.get_y(),
		     end.get_x(), end.get_y());
    }
  }
}

#ifdef PNUM_HACK
// this is duplicated from page.cc
namespace {
  struct PNumHack { //page numbering hack
    static const int xpos=40; //points
    static const int ypos=40; //points
    static const int font_size=10;
  } pnum_hack;
}
#endif

bool PageViewent::draw_content() {
  Glib::RefPtr<Gdk::Drawable> win = view.get_win();
  Glib::RefPtr<Gdk::GC> gc = view.get_gc();

  // draw background color
  gc->set_foreground(view.get_color(Color::bg));
  Gdk::Point origin = view.pt2scr(Vector(0, page->get_height()));
  win->draw_rectangle(gc, true, origin.get_x(), origin.get_y(), 
		      int(view.pt2scr(page->get_width()) + 0.5),
		      int(view.pt2scr(page->get_height()) + 0.5));

  // draw the guides
  draw_guides(*page, view);

#ifdef PNUM_HACK
  // Draw page numbering hack:
  {
    int page_num = page->get_page_num();
    bool odd = page_num % 2;
    int xpos = odd ? int(page->get_width() - pnum_hack.xpos) : pnum_hack.xpos;
    int ypos = pnum_hack.ypos;
    Gdk::Point pos = view.pt2scr(Vector(xpos, ypos));
    int fs = int(view.pt2scr(pnum_hack.font_size));
    int fs30 = fs * 3 / 10;
    int fs70 = fs * 7 / 10;
    gc->set_line_attributes(1, Gdk::LINE_SOLID, Gdk::CAP_BUTT, 
			    Gdk::JOIN_MITER);
    gc->set_foreground(view.get_color(Color::locked));
    win->draw_line(gc, pos.get_x(), pos.get_y()+fs30, 
		   pos.get_x()+fs, pos.get_y()+fs30);
    win->draw_line(gc, pos.get_x(), pos.get_y()+fs70, 
		   pos.get_x()+fs, pos.get_y()+fs70);
    win->draw_line(gc, pos.get_x()+fs30, pos.get_y(), 
		   pos.get_x()+fs30, pos.get_y()+fs);
    win->draw_line(gc, pos.get_x()+fs70, pos.get_y(), 
		   pos.get_x()+fs70, pos.get_y()+fs);
  }
#endif

  // Draw the actual content
  bool done = true;

  const Document::Selection &selection
    = Document::containing(*page).selected();
  
  for(ChildVec::reverse_iterator i = childvec.rbegin(); 
      i != childvec.rend(); 
      i++) {
    // draw content
    done = (*i)->draw_content() && done;
    
    // draw border
    bool selected = 
      find(selection.begin(), selection.end(), (*i)->get_frame())
      != selection.end();
    gc->set_line_attributes
      (1, selected ? Gdk::LINE_SOLID : Gdk::LINE_ON_OFF_DASH, 
       Gdk::CAP_BUTT, 
       Gdk::JOIN_MITER);
    gc->set_foreground(view.get_color((*i)->get_frame()->get_lock() 
				      ? Color::locked
 				      : Color::frame));
    win->draw_polygon(gc, false,
		      polygonToScreen(view, 
				      (*i)->get_frame()->get_box()->get_polygon()));
  }
  
  return done;
}

void PageViewent::clear() {}
