# This file is part of pybliographer
# 
# Copyright (C) 1998 Frederic GOBRY
# Email : gobry@idiap.ch
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# $Id: BibTeX.py,v 1.32 1999/09/28 07:32:23 gobry Exp $

# Extension module for BibTeX files

import Pyblio.Base, _bibtex
import os,sys, tempfile, pwd, time, traceback, re, string

from types import *
from Pyblio.Fields import *
from Pyblio import Types, Base, Config, Autoload

def _nativify (field, type):
	obj = _bibtex.reverse (type, field)
	return _bibtex.get_native (obj)


def entry_write (entry, output):
	""" Print an entry as BiBTeX """

	native = (entry.id == 'BibTeX')


	# get the entry description
	tp = entry.type
	
	output.write ('@%s{%s,\n' % (tp.name, entry.name))

	# start with the crossref if it exists
	if entry.crossref:
		output.write ('  %-14s = ' % crossref)
		output.write (Base.format ('{' +
					   entry.crossref.key.key +
					   '},',
					   75, 19, 19) [19:] + ',\n')
	
	# write down the personal keys only
	dico = {}
	def fill_dict (entry, dico = dico):
		dico [entry] = 1
		return 1
	
	map (fill_dict, entry.personal_keys ())
		
	for f in tp.mandatory:
		field = string.lower (f.name)
		
		if dico.has_key (field):
			if native:
				text = entry.get_native (field)
			else:
				type = Types.gettype (tp, field)
				text = _nativify (entry [field], type)
				
			output.write ('  %-14s = ' % f.name)
			output.write (Base.format (text,
						   75, 19, 19) [19:] + ',\n')
			del dico [field]

	for f in tp.optional:
		field = string.lower (f.name)
		
		if dico.has_key (field):
			if native:
				text = entry.get_native (field)
			else:
				type = Types.gettype (tp, field)
				text = _nativify (entry [field], type)
				
			output.write ('  %-14s = ' % f.name)
			output.write (Base.format (text,
						   75, 19, 19) [19:] + ',\n')
			del dico [field]

	for f in dico.keys ():
		if native:
			text = entry.get_native (f)
		else:
			type = Types.gettype (tp, f)
			text = _nativify (entry [f], type)
			
		output.write ('  %-14s = ' % f)
		output.write (Base.format (text,
					   75, 19, 19) [19:] + ',\n')
		
        output.write ('}\n\n')
	return

	
def my_write (database, output):
	# write header

	output.write ("% This file has been generated by Pybliographer\n\n")
	
	need_header = []

	if database.id == 'BibTeX':
		need_header = [database]
	else:
		# ugly hack to get the BibTeX databases involved, and the
		# necessary crossrefs
		db = {}
		def check_entry (entry, need, database = database):
			# guess what, we are in a RefDB !
			if entry.id == 'BibTeX':
				base = database.base (entry)
				# ensure unicity
				need [base.key] = base
			return


		# get the needed databases
		database.foreach (check_entry, db)
			
		need_header = db.values ()

	# write the string definitions
	if len (need_header) > 0:
		for db in need_header:
			parser = db.get_parser ()
		
			# write the list of strings
			dict = _bibtex.get_dict (parser)
			
			if len (dict.keys ()) > 0:
				for k in dict.keys ():
					output.write ('@string{ ')
					value = _bibtex.get_native (dict [k])
					output.write ("%s \t= %s" % \
						      (k, value))
					output.write ('}\n')
				output.write ('\n')

	cr = {}
	
	def expand_entry (entry, database, cr = cr):
		# if we carry a crossref
		if entry.crossref:
			cr [entry.crossref.key] = entry.crossref
		return

	# get the crossrefs
	database.foreach (expand_entry, database)
			
	# write the crossrefs first
	for e in cr.values ():
		entry_write (e, output)
		
	# write all the other entries
	for e in database.keys ():
		# do not write crossrefs twice
		if not cr.has_key (e):
			entry_write (database [e], output)
	
	return

# --------------------------------------------------
# Register a method to open BibTeX files
# --------------------------------------------------

def my_open (entity, check):
	
	method, address, file, p, q, f = entity
	base = None

	if (not check) or (method == 'file' and file [-4:] == '.bib'):
		base = DataBase (file)
		
	return base


Autoload.register ('format', 'BibTeX', {'open'  : my_open,
					'write' : my_write })

# --------------------------------------------------
# --------------------------------------------------

class BibTextField (Text):
	def __init__ (self, text, native):
		Text.__init__ (self, text)
		self.native = native
		return
	
	def format (self, fmt):

		if string.lower (fmt) == 'latex':
			return self.native

		return Text.format (self, fmt)
	
# --------------------------------------------------
#  Entry inherits from Base
# --------------------------------------------------

class Entry (Base.Entry):

	id = 'BibTeX'

	def __init__ (self, key, name, type, content, parser):
		Base.Entry.__init__ (self, key, name, type, content)
		self.__text = {}

		self.parser = parser
		return
	
	
	def __delitem__ (self, key):
		# First, eventually remove from cache
		if self.__text.has_key (key):
			del self.__text [key]

		del self.__dict [key]
		return

	def __setitem__ (self, key, value):
		# First, eventually remove from cache
		if self.__text.has_key (key):
			del self.__text [key]

		type = Types.gettype (self.type, key)
		self.__dict [key] = _bibtex.reverse (type, value)
		return

	
	def get_native (self, key):
		""" Return object in its native format """

		if self.__text.has_key (key):
			obj = self.__text [key] [0]
			if obj.modified ():
				type = Types.gettype (self.type, key)
				self.__dict [key] = \
					    _bibtex.reverse (type, obj)
				# update cache
				del self.__text [key]
			
		obj = self.__dict [key]
		return _bibtex.get_native (obj)
	
	def get_latex (self, key):
		""" Returns latex part """
		type = Types.gettype (self.type, key)

		if self.__text.has_key (key):
			obj = self.__text [key] [0]
			if obj.modified ():
				type = Types.gettype (self.type, key)
				self.__dict [key] = \
					    _bibtex.reverse (type, obj)
				# update cache
				del self.__text [key]
			
		obj = self.__dict [key]
		return _bibtex.get_latex (self.parser, obj, type)

	def set_native (self, key, value):
		""" set in native format """
		
		if self.__text.has_key (key):
			del self.__text [key]

		type = Types.gettype (self.type, key)
		self.__dict [key] =  _bibtex.set_native (value, type)
		return
	
		
	def text (self, key):
		# look in the cache first
		if self.__text.has_key (key):
			return self.__text [key]

		if not self.__dict.has_key (key):
			# try in the crossrefs
			if self.crossref:
				if self.crossref.has_key (key):
					return self.crossref.text (key)

		obj = self.__dict [key]
			
			
		# search its declared type

		type = Types.gettype (self.type, key)
		ret = _bibtex.expand (self.parser, obj, type)
		
		if ret [0] == Types.TypeAuthor:
			# Author
			val = AuthorGroup ()
			for aut in ret [3]:
				val.append (Author (aut))
			
		elif ret [0] == Types.TypeDate:
			# Date
			val = Date ((ret [3], None, None))
			
		else:
			# Any other text
			val = BibTextField (ret [2], self.get_latex (key))

		val.clean ()
		self.__text [key] = (val, ret [1])
		
		return (val, ret [1])



# --------------------------------------------------

class DataBase (Base.DataBase):

	id = 'BibTeX'

	__keyid = 0
	
	properties = {
		'edit'        : 1,
		'change_id'   : 1,
		'change_type' : 1,
		'add'         : 1,
		'remove'      : 1,
		'has_extra'   : 1,
		'crossref'    : 1,
		'native'      : 1,
		}
	
	def __parsefile__ (self):
		self.__parser = None
		self.__recursive = 0
		self.__current = 0
		
		self.__dict = {}
		
		# Ouvrir le fichier associe
		self.__parser = _bibtex.open (self.name,
					      Config.get ("bibtex/strict").data)
		
		finished = 0
		errors = []
		
		# Creer la base de cles
		while 1:
			try:
				retval = _bibtex.next (self.__parser)
			except IOError, err:
				errors.append (str (err))
				continue
			
			if retval == None: break

			name, type, offset, line, object = retval
			
			key = Base.Key (self, name)

			if not self.__dict.has_key (key):
				type = Types.getentry (type)
				entry = Entry (key, name, type, object,
					       self.__parser)
				self.__dict [key] = entry
				continue
				
			errors.append ("%s:%d: key `%s' already defined" % (
				self.name, line, name))

		if len (errors) > 0:
			raise IOError, string.join (errors, "\n")

		# solve the crossreferences
		for e in self.__dict.values ():
			if e.has_key ('crossref'):
				
				key = Base.Key (self, e ['crossref'].text)
				if self.has_key (key):
					cr = self [key]
					# we don't allow nested crossrefs
					if cr.has_key ('crossref'):
						errors.append ("in entry %s: nested crossrefs" %
							       e.key.key)
						continue
					
					# crossref is no real field
					del e ['crossref']
					
					e.crossref = cr
				else:
					errors.append ("in entry %s: unknown crossref" %
						       e.key.key)
		return
	
		
	def __init__ (self, basename):
		"Initialisation"
		Base.DataBase.__init__ (self, basename)
		self.__parsefile__ ()
		return

	def new_entry (self):
		""" returns a newly created entry """

		name = None
		key  = None
		
		# generate a new key
		while (not name) or self.has_key (key):
			name = "entry-%d" % self.__keyid
			key = Base.Key (self, name)
			
			self.__keyid = self.__keyid + 1

		# get the first type
		type = Config.get ("base/defaulttype").data

		# return the new entry
		return Entry (key, name, type, {}, self.__parser)

	
	def __del__ (self):
		pass

	def get_parser (self):
		return self.__parser
	

	def __repr__ (self):
		""
		return "<BibTeX database `%s' (%d entries)>" % \
		       (self.name, len (self))

	# ==================================================

	def update (self):
		""" save the database """
		
		# backup file
		os.rename (self.name, self.name + '.bak')
		
		tmpfile = open (self.name, 'w')
		
		my_write (self, tmpfile)
		tmpfile.close ()

		self.__parsefile__ ()
		return

	
