#!/usr/bin/env python
'''This is a client to the test/everything/server test program
distributed with ORBit2.'''

import sys
import unittest
import ORBit

ORBit.load_typelib('Everything')

import CORBA
# this one is loaded from the typelib ...
from orbit import test

# from test/everything/constants.h
class constants:
    '''constants from ORBit2/test/everything/constants.h.  placed in a
    class to keep them organised.
    Once I get a working server implementation, this will be moved out
    into a separate module, shared with the server tests.'''

    STRING_IN        = 'In string'
    STRING_INOUT_IN  = 'Inout in string'
    STRING_INOUT_OUT = 'Inout out string'
    STRING_OUT       = 'Out string'
    STRING_RETN      = 'Retn String'

    LONG_IN          = 0x12345678
    LONG_INOUT_IN    = 0x34567812
    LONG_INOUT_OUT   = 0x56781234
    LONG_OUT         = 0x78123456
    LONG_RETN        = -1430532899 # 0xAABBCCDD

    LONG_LONG_IN          = 0x12345678L
    LONG_LONG_INOUT_IN    = 0x34567812L
    LONG_LONG_INOUT_OUT   = 0x56781234L
    LONG_LONG_OUT         = 0x78123456L
    LONG_LONG_RETN        = 0xAABBCCDDL

    FLOAT_IN         = 127.13534
    FLOAT_INOUT_IN   = 124.89432
    FLOAT_INOUT_OUT  = 975.12694
    FLOAT_OUT        = 112.54575
    FLOAT_RETN       = 354.23535

    SHORT_IN         = 0x1234
    SHORT_INOUT_IN   = 0x3456
    SHORT_INOUT_OUT  = 0x5678
    SHORT_OUT        = 0x7812
    SHORT_RETN       = -21829 # 0xAABB

    SEQ_STRING_IN        = ( 'in1', 'in2', 'in3', 'in4' )
    SEQ_STRING_INOUT_IN  = ('inout1', 'inout2', 'inout3', 'inout4' )
    SEQ_STRING_INOUT_OUT = ( 'inout21', 'inout22', 'inout23', 'inout24' )
    SEQ_STRING_OUT       = ( 'out1', 'out2', 'out3', 'out4' )
    SEQ_STRING_RETN      = ( 'retn1', 'retn2', 'retn3', 'retn4' )

    SEQ_LONG_IN          = ( LONG_IN, LONG_INOUT_IN, 15, 7 )
    SEQ_LONG_INOUT_IN    = ( LONG_INOUT_OUT, LONG_OUT, 7, 15 )
    SEQ_LONG_INOUT_OUT   = ( LONG_OUT, LONG_RETN, 8, 9 )
    SEQ_LONG_OUT         = ( LONG_INOUT_IN, LONG_INOUT_OUT, 15, 7 )
    SEQ_LONG_RETN        = ( LONG_RETN, LONG_IN, 2, 3 )

class EverythingTestCase(unittest.TestCase):

    def setUp(self):
        self.orb = CORBA.ORB_init(sys.argv, 'orbit-local-orb')

        ior = file('../../../ORBit2/test/everything/iorfile', 'r').read()
        self.factory = self.orb.string_to_object(ior)
    def tearDown(self):
        del self.factory
        del self.orb

    def test01_attribute(self):
        '''attributes'''
        objref = self.factory.getBasicServer()
        assert objref.is_a('IDL:orbit/test/BasicServer:1.0')

        val = objref.foo
        assert val == constants.STRING_RETN

        objref.foo = constants.STRING_IN

        lval = objref.bah
        assert lval == constants.LONG_RETN

    def test02_string(self):
        '''strings'''
        objref = self.factory.getBasicServer()

        retn, inout, out = objref.opString(constants.STRING_IN,
                                           constants.STRING_INOUT_IN)
        assert retn == constants.STRING_RETN
        assert inout == constants.STRING_INOUT_OUT
        assert out == constants.STRING_OUT

        objref.opOneWay(constants.STRING_IN)

    def test03_long(self):
        '''longs'''
        objref = self.factory.getBasicServer()

        retn, inout, out = objref.opLong(constants.LONG_IN,
                                         constants.LONG_INOUT_IN)
        assert retn == constants.LONG_RETN
        assert inout == constants.LONG_INOUT_OUT
        assert out == constants.LONG_OUT

    def test04_longlong(self):
        '''long longs'''
        objref = self.factory.getBasicServer()

        retn, inout, out = objref.opLongLong(constants.LONG_LONG_IN,
                                             constants.LONG_LONG_INOUT_IN)
        assert retn == constants.LONG_LONG_RETN
        assert inout == constants.LONG_LONG_INOUT_OUT
        assert out == constants.LONG_LONG_OUT

    def test05_float(self):
        '''floats'''
        objref = self.factory.getBasicServer()

        retn, inout, out = objref.opFloat(constants.FLOAT_IN,
                                          constants.FLOAT_INOUT_IN)
        assert abs(retn - constants.FLOAT_RETN) < 0.00005
        assert abs(inout - constants.FLOAT_INOUT_OUT) < 0.00005
        assert abs(out - constants.FLOAT_OUT) < 0.00005

    def test06_double(self):
        '''doubles'''
        objref = self.factory.getBasicServer()

        retn, inout, out = objref.opDouble(constants.FLOAT_IN,
                                           constants.FLOAT_INOUT_IN)
        assert retn == constants.FLOAT_RETN
        assert inout == constants.FLOAT_INOUT_OUT
        assert out == constants.FLOAT_OUT

    def test07_enum(self):
        '''enums'''
        objref = self.factory.getBasicServer()

        retn, inout, out = objref.opEnum(test.ENUM_IN, test.ENUM_INOUT_IN)
        assert retn == test.ENUM_RETN
        assert inout == test.ENUM_INOUT_OUT
        assert out == test.ENUM_OUT

    def test08_exception(self):
        '''exceptions'''
        objref = self.factory.getBasicServer()

        got_exception = 0
        try:
            objref.opException()
        except test.TestException, e:
            assert e.reason == constants.STRING_IN
            assert e.number == constants.LONG_IN
            assert len(e.aseq) == 1
            assert e.aseq[0] == constants.LONG_IN
            got_exception = 1
        assert got_exception

    def test09_is_a(self):
        '''is_a'''
        assert self.factory.is_a('IDL:CORBA/Object:1.0')
        assert self.factory.is_a('IDL:omg.org/CORBA/Object:1.0')

    def test10_fixed_length_struct(self):
        '''fixed length structs'''
        objref = self.factory.getStructServer()

        inarg = test.FixedLengthStruct()
        inarg.a = constants.SHORT_IN
        inoutarg = test.FixedLengthStruct()
        inoutarg.a = constants.SHORT_INOUT_IN

        retn, inout, out = objref.opFixed(inarg, inoutarg)
        assert retn.a == constants.SHORT_RETN
        assert inout.a == constants.SHORT_INOUT_OUT
        assert out.a == constants.SHORT_OUT
    
    def test11_variable_length_struct(self):
        '''variable length structs'''
        objref = self.factory.getStructServer()

        inarg = test.VariableLengthStruct(constants.STRING_IN)
        inoutarg = test.VariableLengthStruct(constants.STRING_INOUT_IN)

        retn, inout, out = objref.opVariable(inarg, inoutarg)
        assert retn.a == constants.STRING_RETN
        assert inout.a == constants.STRING_INOUT_OUT
        assert out.a == constants.STRING_OUT
    
    def test12_compound_struct(self):
        '''compound structs'''
        objref = self.factory.getStructServer()

        inarg = test.CompoundStruct(test.VariableLengthStruct(constants.STRING_IN))
        inoutarg = test.CompoundStruct(test.VariableLengthStruct(constants.STRING_INOUT_IN))

        retn, inout, out = objref.opCompound(inarg, inoutarg)
        assert retn.a.a == constants.STRING_RETN
        assert inout.a.a == constants.STRING_INOUT_OUT
        assert out.a.a == constants.STRING_OUT

    def test13_object_struct(self):
        '''object structs'''
        objref = self.factory.getStructServer()

        inarg = test.ObjectStruct()
        inarg.serv = objref

        objref.opObjectStruct(inarg)

    def test14_struct_any(self):
        '''any structs'''
        objref = self.factory.getStructServer()

        a = objref.opStructAny()
        assert a.a == constants.STRING_IN
        assert a.b.value() == constants.LONG_IN

    def test15_unbounded_sequence(self):
        '''unbounded sequences'''
        objref = self.factory.getSequenceServer()

        retn, inout, out = objref.opStrSeq(constants.SEQ_STRING_IN,
                                           constants.SEQ_STRING_INOUT_IN)
        assert retn == constants.SEQ_STRING_RETN[:len(retn)]
        assert inout == constants.SEQ_STRING_INOUT_OUT[:len(inout)]
        assert out == constants.SEQ_STRING_OUT[:len(out)]

    def test16_bounded_sequence(self):
        '''bounded sequences'''
        objref = self.factory.getSequenceServer()

        inarg = [
            test.CompoundStruct(test.VariableLengthStruct(constants.SEQ_STRING_IN[0])),
            test.CompoundStruct(test.VariableLengthStruct(constants.SEQ_STRING_IN[1]))
            ]
        inoutarg = [
            test.CompoundStruct(test.VariableLengthStruct(constants.SEQ_STRING_INOUT_IN[0])),
            test.CompoundStruct(test.VariableLengthStruct(constants.SEQ_STRING_INOUT_IN[1]))
            ]

        retn, inout, out = objref.opBoundedStructSeq(inarg, inoutarg)
        for i in range(len(retn)):
            assert retn[i].a.a == constants.SEQ_STRING_RETN[i]
        for i in range(len(inout)):
            assert inout[i].a.a == constants.SEQ_STRING_INOUT_OUT[i]
        for i in range(len(out)):
            assert out[i].a.a == constants.SEQ_STRING_OUT[i]

    def test17_fixed_length_union(self):
        '''fixed length unions'''
        objref = self.factory.getUnionServer()

        inarg = test.FixedLengthUnion('a', constants.LONG_IN)
        inoutarg = test.FixedLengthUnion('b', 't')

        retn, inout, out = objref.opFixed(inarg, inoutarg)
        assert retn._d == 'd'
        assert retn._v == CORBA.FALSE
        assert inout._d == 'c'
        assert inout._v == CORBA.TRUE
        assert out._d == 'a'
        assert out._v == constants.LONG_OUT

    def test18_variable_length_union(self):
        '''variable length unions'''
        objref = self.factory.getUnionServer()

        inarg = test.VariableLengthUnion(1, constants.LONG_IN)
        inoutarg = test.VariableLengthUnion(2, constants.STRING_INOUT_IN)

        retn, inout, out = objref.opVariable(inarg, inoutarg)
        assert retn._d == 4
        assert retn._v == CORBA.FALSE
        assert inout._d == 3
        assert inout._v == CORBA.TRUE
        assert out._d == 1
        assert out._v == constants.LONG_OUT
    
    def test19_misc_union(self):
        '''misc type unions'''
        objref = self.factory.getUnionServer()

        inseq = [
            test.VariableLengthUnion(4, CORBA.TRUE),
            test.VariableLengthUnion(2, "blah"),
            test.VariableLengthUnion(55, constants.LONG_IN)
            ]
        inarg = test.VariableLengthUnion(1, "blah de blah")

        retn, out = objref.opMisc(inseq, inarg)

        assert retn._d == test.EnumUnion.red
        assert retn._v == constants.LONG_IN
        assert out._d == 22
        for i in range(20):
            assert out._v[i] == 'Numero %d' % i
    
    def test20_fixed_length_array(self):
        '''fixed length arrays'''
        objref = self.factory.getArrayServer()

        retn, inout, out = objref.opLongArray(constants.SEQ_LONG_IN,
                                              constants.SEQ_LONG_INOUT_IN)
        assert retn == constants.SEQ_LONG_RETN
        assert inout == constants.SEQ_LONG_INOUT_OUT
        assert out == constants.SEQ_LONG_OUT

    def test21_variable_length_array(self):
        '''variable length arrays'''
        objref = self.factory.getArrayServer()

        retn, inout, out = objref.opStrArray(constants.SEQ_STRING_IN,
                                             constants.SEQ_STRING_INOUT_IN)
        assert retn == constants.SEQ_STRING_RETN
        assert inout == constants.SEQ_STRING_INOUT_OUT
        assert out == constants.SEQ_STRING_OUT

    def test22_any_long(self):
        '''any with longs'''
        objref = self.factory.getAnyServer()

        tc = CORBA.TypeCode('IDL:omg.org/CORBA/long:1.0')
        retn, inout, out = objref.opAnyLong(CORBA.Any(tc, constants.LONG_IN),
                                            CORBA.Any(tc, constants.LONG_INOUT_IN))

        assert retn.typecode() == tc
        assert retn.value() == constants.LONG_RETN
        assert inout.typecode() == tc
        assert inout.value() == constants.LONG_INOUT_OUT
        assert out.typecode() == tc
        assert out.value() == constants.LONG_OUT

    def test23_any_string(self):
        '''any with strings'''
        objref = self.factory.getAnyServer()

        tc = CORBA.TypeCode('IDL:omg.org/CORBA/string:1.0')
        retn, inout, out = objref.opAnyString(CORBA.Any(tc, constants.STRING_IN),
                                              CORBA.Any(tc, constants.STRING_INOUT_IN))

        assert retn.typecode() == tc
        assert retn.value() == constants.STRING_RETN
        assert inout.typecode() == tc
        assert inout.value() == constants.STRING_INOUT_OUT
        assert out.typecode() == tc
        assert out.value() == constants.STRING_OUT

    def test24_any_str_seq(self):
        '''any with string sequences'''
        objref = self.factory.getAnyServer()

        retn = objref.opAnyStrSeq()

    def test25_any_struct(self):
        '''any with structs'''
        objref = self.factory.getAnyServer()

        tc = CORBA.TypeCode('IDL:orbit/test/VariableLengthStruct:1.0')
        inarg = CORBA.Any(tc, test.VariableLengthStruct(constants.STRING_IN))
        inoutarg = CORBA.Any(tc, test.VariableLengthStruct(
            constants.STRING_INOUT_IN))

        retn, inout, out = objref.opAnyStruct(inarg, inoutarg)

        assert retn.typecode() == tc
        assert retn.value().a == constants.STRING_RETN
        assert inout.typecode() == tc
        assert inout.value().a == constants.STRING_INOUT_OUT
        assert out.typecode() == tc
        assert out.value().a == constants.STRING_OUT

    def test26_any_exception(self):
        '''any with exception'''

        tc = CORBA.TypeCode('IDL:orbit/test/TestException:1.0')
        any = CORBA.Any(tc, test.TestException('reason', 42, [], self.factory))
        del any

    def test27_any_equivalence(self):
        '''anys equivalence'''

        tc = CORBA.TypeCode('IDL:orbit/test/unionSeq:1.0')
        seq = [
            test.VariableLengthUnion(4, CORBA.TRUE),
            test.VariableLengthUnion(2, 'blah'),
            test.VariableLengthUnion(55, constants.LONG_IN)
            ]
        a = CORBA.Any(tc, seq)
        b = CORBA.Any(tc, seq)

        assert a == b

    def test28_typecode(self):
        '''TypeCodes'''
        objref = self.factory.getAnyServer()

        inarg = CORBA.TypeCode('IDL:orbit/test/ArrayUnion:1.0')
        inoutarg = CORBA.TypeCode('IDL:orbit/test/AnyServer:1.0')
        retn, inout, out = objref.opTypeCode(inarg, inoutarg)

        assert retn == CORBA.TypeCode('IDL:orbit/test/VariableLengthStruct:1.0')
        assert inout == CORBA.TypeCode('IDL:orbit/test/TestException:1.0')
        assert out == CORBA.TypeCode('IDL:orbit/test/AnEnum:1.0')

if __name__ == '__main__':
    unittest.main()
