/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* 
 *  arch-tag: Implementation of main playback logic object
 *
 *  Copyright (C) 2002, 2003 Jorn Baayen <jorn@nl.linux.org>
 *  Copyright (C) 2002,2003 Colin Walters <walters@debian.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <gtk/gtk.h>
#include <glade/glade.h>
#include <config.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnome/gnome-i18n.h>

#ifdef HAVE_MMKEYS
#include <X11/Xlib.h>
#include <X11/XF86keysym.h>
#include <gdk/gdkx.h>
#endif /* HAVE_MMKEYS */

#include "rb-property-view.h"
#include "rb-shell-player.h"
#include "rb-stock-icons.h"
#include "rb-glade-helpers.h"
#include "rb-file-helpers.h"
#include "rb-cut-and-paste-code.h"
#include "rb-dialog.h"
#include "rb-preferences.h"
#include "rb-debug.h"
#include "rb-player.h"
#include "rb-header.h"
#include "totem-pl-parser.h"
#include "rb-metadata.h"
#include "bacon-volume.h"
#include "rb-remote.h"
#include "rb-iradio-source.h"
#include "rb-library-source.h"
#include "eel-gconf-extensions.h"
#include "rb-util.h"
#include "rb-play-order.h"
#include "rb-statusbar.h"

#ifdef HAVE_XIDLE_EXTENSION
#include <X11/extensions/xidle.h>
#endif /* HAVE_XIDLE_EXTENSION */

typedef enum
{
	PLAY_BUTTON_PLAY,
	PLAY_BUTTON_PAUSE,
	PLAY_BUTTON_STOP
} PlayButtonState;

static const char* const state_to_play_order[2][2] =
	{{"linear",	"linear-loop"},
	 {"shuffle",	"random-by-age-and-rating"}};

static void rb_shell_player_class_init (RBShellPlayerClass *klass);
static void rb_shell_player_init (RBShellPlayer *shell_player);
static GObject *rb_shell_player_constructor (GType type, guint n_construct_properties,
					     GObjectConstructParam *construct_properties);
static void rb_shell_player_finalize (GObject *object);
static void rb_shell_player_set_property (GObject *object,
					  guint prop_id,
					  const GValue *value,
					  GParamSpec *pspec);
static void rb_shell_player_get_property (GObject *object,
					  guint prop_id,
					  GValue *value,
					  GParamSpec *pspec);

static void rb_shell_player_cmd_previous (GtkAction *action,
			                  RBShellPlayer *player);
static void rb_shell_player_cmd_play (GtkAction *action,
			              RBShellPlayer *player);
static void rb_shell_player_cmd_pause (GtkAction *action,
			               RBShellPlayer *player);
static void rb_shell_player_cmd_stop (GtkAction *action,
			              RBShellPlayer *player);
static void rb_shell_player_playpause_button_cb (GtkButton *button,
						  RBShellPlayer *player);
static void rb_shell_player_cmd_next (GtkAction *action,
			              RBShellPlayer *player);
static void rb_shell_player_shuffle_changed_cb (GtkAction *action,
						RBShellPlayer *player);
static void rb_shell_player_repeat_changed_cb (GtkAction *action,
					       RBShellPlayer *player);
static void rb_shell_player_previous_button_cb (GtkButton *button,
						RBShellPlayer *player);
static void rb_shell_player_next_button_cb (GtkButton *button,
					    RBShellPlayer *player);
static void rb_shell_player_cmd_song_info (GtkAction *action,
					   RBShellPlayer *player);
static void rb_shell_player_set_playing_source_internal (RBShellPlayer *player,
							 RBSource *source,
							 gboolean sync_entry_view);
static void rb_shell_player_set_play_button (RBShellPlayer *player,
			                     PlayButtonState state);
static void rb_shell_player_sync_with_source (RBShellPlayer *player);
static void rb_shell_player_sync_with_selected_source (RBShellPlayer *player);
static void rb_shell_player_entry_changed_cb (RhythmDB *db,
							RhythmDBEntry *entry,
				       		GSList *changes,
				       		RBShellPlayer *player);
				       
static void rb_shell_player_playing_entry_deleted_cb (RBEntryView *view,
						      RhythmDBEntry *entry,
						      RBShellPlayer *playa);
static void rb_shell_player_entry_view_changed_cb (RBEntryView *view,
						   RBShellPlayer *playa);
static void rb_shell_player_entry_activated_cb (RBEntryView *view,
						RhythmDBEntry *entry,
						RBShellPlayer *playa);
static void rb_shell_player_property_row_activated_cb (RBPropertyView *view,
						       const char *name,
						       RBShellPlayer *playa);
static void rb_shell_player_volume_changed_cb (GConfClient *client,
					       guint cnxn_id,
					       GConfEntry *entry,
					       RBShellPlayer *playa);
static void rb_shell_player_volume_widget_changed_cb (BaconVolumeButton *vol,
						      RBShellPlayer     *playa);
static void rb_shell_player_sync_volume (RBShellPlayer *player); 
static void rb_shell_player_sync_replaygain (RBShellPlayer *player,
                                             RhythmDBEntry *entry);
static void tick_cb (RBPlayer *player, long elapsed, gpointer data);
static void eos_cb (RBPlayer *player, gpointer data);
static void error_cb (RBPlayer *player, const GError *err, gpointer data);
static void buffering_cb (RBPlayer *player, guint progress, gpointer data);
static void rb_shell_player_error (RBShellPlayer *player, gboolean async, const GError *err);

static void info_available_cb (RBPlayer *player,
                               RBMetaDataField field,
                               GValue *value,
                               gpointer data);
static void rb_shell_player_set_play_order (RBShellPlayer *player,
					    const gchar *new_val);

static void rb_shell_player_sync_play_order (RBShellPlayer *player);
static void rb_shell_player_sync_control_state (RBShellPlayer *player);

static void gconf_play_order_changed (GConfClient *client,guint cnxn_id,
				      GConfEntry *entry, RBShellPlayer *player);

#ifdef HAVE_MMKEYS
static void grab_mmkey (int key_code, GdkWindow *root);
static GdkFilterReturn filter_mmkeys (GdkXEvent *xevent,
				      GdkEvent *event,
				      gpointer data);
static void rb_shell_player_init_mmkeys (RBShellPlayer *shell_player);
#endif /* HAVE_MMKEYS */

#define CONF_STATE		CONF_PREFIX "/state"

struct RBShellPlayerPrivate
{
	RhythmDB *db;

	gboolean syncing_state;
	
	RBSource *selected_source;
	RBSource *source;
	RhythmDBEntry *playing_attempt_entry;
	gboolean did_retry;
	GTimeVal last_retry;

	GtkUIManager *ui_manager;
	GtkActionGroup *actiongroup;
	GtkActionGroup *play_action_group;
	GtkActionGroup *pause_action_group;
	GtkActionGroup *stop_action_group;

	gboolean handling_error;

	RBPlayer *mmplayer;

	GList *active_uris;

	char *song;
	gboolean have_url;
	char *url;

	gboolean have_previous_entry;

	RBPlayOrder *play_order;

	GError *playlist_parse_error;

	gboolean last_jumped;
	gboolean last_skipped;

	GtkTooltips *tooltips;
	GtkWidget *prev_button;
	PlayButtonState playbutton_state;
	GtkWidget *play_pause_stop_button;
	GtkWidget *play_image;
	GtkWidget *pause_image;
	GtkWidget *stop_image;
	GtkWidget *next_button;

	RBHeader *header_widget;
	RBStatusbar *statusbar_widget;

	GtkWidget *shuffle_button;
	GtkWidget *volume_button;
	GtkWidget *magic_button;

	RBRemote *remote;

	guint gconf_play_order_id;
	guint gconf_state_id;

	gboolean mute;
	float pre_mute_volume;
};

enum
{
	PROP_0,
	PROP_SOURCE,
	PROP_DB,
	PROP_UI_MANAGER,
	PROP_ACTION_GROUP,
	PROP_PLAY_ORDER,
	PROP_PLAYING,
	PROP_VOLUME,
	PROP_STATUSBAR
};

enum
{
	WINDOW_TITLE_CHANGED,
	DURATION_CHANGED,
	PLAYING_SOURCE_CHANGED,
	PLAYING_CHANGED,
	PLAYING_SONG_CHANGED,
	PLAYING_URI_CHANGED,
	LAST_SIGNAL
};

static GtkActionEntry rb_shell_player_actions [] =
{
	{ "ControlPrevious", GTK_STOCK_MEDIA_PREVIOUS, N_("P_revious"), "<control>Left",
	  N_("Start playing the previous song"),
	  G_CALLBACK (rb_shell_player_cmd_previous) },
	{ "ControlNext", GTK_STOCK_MEDIA_NEXT, N_("_Next"), "<control>Right",
	  N_("Start playing the next song"),
	  G_CALLBACK (rb_shell_player_cmd_next) },
	{ "MusicProperties", GTK_STOCK_PROPERTIES, N_("_Properties"), "<control>P",
	  N_("Show information on the selected song"),
	  G_CALLBACK (rb_shell_player_cmd_song_info) },
};
static guint rb_shell_player_n_actions = G_N_ELEMENTS (rb_shell_player_actions);

static GtkActionEntry rb_shell_player_play_action [] =
{
	{ "ControlPlay", GTK_STOCK_MEDIA_PLAY, N_("_Play"), "<control>space",
	  N_("Start playback"),
	  G_CALLBACK (rb_shell_player_cmd_play) },
};

static GtkActionEntry rb_shell_player_pause_action [] =
{
	{ "ControlPause", GTK_STOCK_MEDIA_PAUSE, N_("_Pause"), "<control>space",
	  N_("Pause playback"),
	  G_CALLBACK (rb_shell_player_cmd_pause) },
};

static GtkActionEntry rb_shell_player_stop_action [] =
{
	{ "ControlStop", GTK_STOCK_MEDIA_STOP, N_("_Stop"), "<control>space",
	  N_("Stop playback"),
	  G_CALLBACK (rb_shell_player_cmd_stop) },
};

static GtkToggleActionEntry rb_shell_player_toggle_entries [] =
{
	{ "ControlShuffle", NULL, N_("Sh_uffle"), "<control>U",
	  N_("Play songs in a random order"),
	  G_CALLBACK (rb_shell_player_shuffle_changed_cb) },
	{ "ControlRepeat", NULL, N_("_Repeat"), "<control>R",
	  N_("Play first song again after all songs are played"),
	  G_CALLBACK (rb_shell_player_repeat_changed_cb) }
};
static guint rb_shell_player_n_toggle_entries = G_N_ELEMENTS (rb_shell_player_toggle_entries);

static GObjectClass *parent_class = NULL;

static guint rb_shell_player_signals[LAST_SIGNAL] = { 0 };

GType
rb_shell_player_get_type (void)
{
	static GType rb_shell_player_type = 0;

	if (rb_shell_player_type == 0)
	{
		static const GTypeInfo our_info =
		{
			sizeof (RBShellPlayerClass),
			NULL,
			NULL,
			(GClassInitFunc) rb_shell_player_class_init,
			NULL,
			NULL,
			sizeof (RBShellPlayer),
			0,
			(GInstanceInitFunc) rb_shell_player_init
		};

		rb_shell_player_type = g_type_register_static (GTK_TYPE_HBOX,
							       "RBShellPlayer",
							       &our_info, 0);
	}

	return rb_shell_player_type;
}

static void
rb_shell_player_class_init (RBShellPlayerClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = rb_shell_player_finalize;
	object_class->constructor = rb_shell_player_constructor;

	object_class->set_property = rb_shell_player_set_property;
	object_class->get_property = rb_shell_player_get_property;

	g_object_class_install_property (object_class,
					 PROP_SOURCE,
					 g_param_spec_object ("source",
							      "RBSource",
							      "RBSource object",
							      RB_TYPE_SOURCE,
							      G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
					 PROP_UI_MANAGER,
					 g_param_spec_object ("ui-manager",
							      "GtkUIManager",
							      "GtkUIManager object",
							      GTK_TYPE_UI_MANAGER,
							      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property (object_class,
					 PROP_DB,
					 g_param_spec_object ("db",
							      "RhythmDB",
							      "RhythmDB object",
							      RHYTHMDB_TYPE,
							      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property (object_class,
					 PROP_ACTION_GROUP,
					 g_param_spec_object ("action-group",
							      "GtkActionGroup",
							      "GtkActionGroup object",
							      GTK_TYPE_ACTION_GROUP,
							      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
	/* If you change these, be sure to update the CORBA interface
	 * in rb-remote-bonobo.c! */
	g_object_class_install_property (object_class,
					 PROP_PLAY_ORDER,
					 g_param_spec_string ("play-order", 
							      "play-order", 
							      "What play order to use",
							      "linear",
							      G_PARAM_READABLE));
	g_object_class_install_property (object_class,
					 PROP_PLAYING,
					 g_param_spec_boolean ("playing", 
							       "playing", 
							      "Whether Rhythmbox is currently playing", 
							       FALSE,
							       G_PARAM_READABLE));
	
	g_object_class_install_property (object_class,
					 PROP_VOLUME,
					 g_param_spec_float ("volume", 
							     "volume", 
							     "Current playback volume",
							     0.0f, 1.0f, 1.0f,
							     G_PARAM_READWRITE));
	
	g_object_class_install_property (object_class,
					 PROP_STATUSBAR,
					 g_param_spec_object ("statusbar", 
							      "RBStatusbar", 
							      "RBStatusbar object", 
							      RB_TYPE_STATUSBAR,
							      G_PARAM_READWRITE));


	rb_shell_player_signals[WINDOW_TITLE_CHANGED] =
		g_signal_new ("window_title_changed",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (RBShellPlayerClass, window_title_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__STRING,
			      G_TYPE_NONE,
			      1,
			      G_TYPE_STRING);

	rb_shell_player_signals[DURATION_CHANGED] =
		g_signal_new ("duration_changed",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (RBShellPlayerClass, duration_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__STRING,
			      G_TYPE_NONE,
			      1,
			      G_TYPE_STRING);

	rb_shell_player_signals[PLAYING_SOURCE_CHANGED] =
		g_signal_new ("playing-source-changed",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (RBShellPlayerClass, playing_source_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__OBJECT,
			      G_TYPE_NONE,
			      1,
			      RB_TYPE_SOURCE);

	rb_shell_player_signals[PLAYING_CHANGED] =
		g_signal_new ("playing-changed",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (RBShellPlayerClass, playing_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__BOOLEAN,
			      G_TYPE_NONE,
			      1,
			      G_TYPE_BOOLEAN);

	rb_shell_player_signals[PLAYING_SONG_CHANGED] =
		g_signal_new ("playing-song-changed",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (RBShellPlayerClass, playing_song_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__POINTER,
			      G_TYPE_NONE,
			      1,
			      G_TYPE_POINTER);

	rb_shell_player_signals[PLAYING_URI_CHANGED] =
		g_signal_new ("playing-uri-changed",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (RBShellPlayerClass, playing_uri_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__STRING,
			      G_TYPE_NONE,
			      1,
			      G_TYPE_STRING);
}

static GObject *
rb_shell_player_constructor (GType type, guint n_construct_properties,
			     GObjectConstructParam *construct_properties)
{
	RBShellPlayer *player;
	RBShellPlayerClass *klass;
	GObjectClass *parent_class;  

	klass = RB_SHELL_PLAYER_CLASS (g_type_class_peek (RB_TYPE_SHELL_PLAYER));

	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	player = RB_SHELL_PLAYER (parent_class->constructor (type, n_construct_properties,
							     construct_properties));

	gtk_action_group_add_actions (player->priv->actiongroup,
				      rb_shell_player_actions,
				      rb_shell_player_n_actions,
				      player);
	player->priv->play_action_group = gtk_action_group_new ("PlayActions");
	gtk_action_group_set_translation_domain (player->priv->play_action_group,
						 GETTEXT_PACKAGE);

	gtk_action_group_add_actions (player->priv->play_action_group,
				      rb_shell_player_play_action,
				      1, player);
	gtk_ui_manager_insert_action_group (player->priv->ui_manager,
					    player->priv->play_action_group, 0);
	player->priv->pause_action_group = gtk_action_group_new ("PauseActions");
	gtk_action_group_set_translation_domain (player->priv->pause_action_group,
						 GETTEXT_PACKAGE);
	gtk_action_group_add_actions (player->priv->pause_action_group,
				      rb_shell_player_pause_action,
				      1, player);
	gtk_ui_manager_insert_action_group (player->priv->ui_manager,
					    player->priv->pause_action_group, 0);
	player->priv->stop_action_group = gtk_action_group_new ("StopActions");
	gtk_action_group_set_translation_domain (player->priv->stop_action_group,
						 GETTEXT_PACKAGE);

	gtk_action_group_add_actions (player->priv->stop_action_group,
				      rb_shell_player_stop_action,
				      1, player);
	gtk_ui_manager_insert_action_group (player->priv->ui_manager,
					    player->priv->stop_action_group, 0);
	gtk_action_group_add_toggle_actions (player->priv->actiongroup,
					     rb_shell_player_toggle_entries,
					     rb_shell_player_n_toggle_entries,
					     player);
	rb_shell_player_set_playing_source (player, NULL);
	rb_shell_player_sync_play_order (player);

	return G_OBJECT (player);
}

static void
volume_pre_unmount_cb (GnomeVFSVolumeMonitor *monitor, 
		       GnomeVFSVolume *volume, 
		       RBShellPlayer *player)
{
	gchar *uri_mount_point;
	gchar *volume_mount_point;
	RhythmDBEntry *entry;
	const char *uri;

	if (rb_shell_player_get_playing (player)) {
		return;
	}

	entry = rb_shell_player_get_playing_entry (player);
	if (entry == NULL) {
		/* At startup for example, playing path can be NULL */
		return;
	}

	uri = rhythmdb_entry_get_string (entry, RHYTHMDB_PROP_LOCATION);
	uri_mount_point = rb_uri_get_mount_point (uri);
	volume_mount_point = gnome_vfs_volume_get_activation_uri (volume);

	if (!strcmp (uri_mount_point, volume_mount_point)) {
		rb_shell_player_stop (player);
	}
	g_free (uri_mount_point);
	g_free (volume_mount_point);
}

static void
reemit_playing_signal (RBShellPlayer *player, GParamSpec *pspec, gpointer data)
{
	g_signal_emit (player, rb_shell_player_signals[PLAYING_CHANGED], 0,
		       rb_player_playing (player->priv->mmplayer));
}

static void
rb_shell_player_init (RBShellPlayer *player)
{
	GError *error = NULL;
	GtkWidget *hbox, *image;
	GtkWidget *alignment;

	player->priv = g_new0 (RBShellPlayerPrivate, 1);

	player->priv->mmplayer = rb_player_new (&error);
	if (error != NULL) {
		GtkWidget *dialog;
		dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_CLOSE,
						 _("Failed to create the player: %s"),
						 error->message);
		gtk_dialog_run (GTK_DIALOG (dialog));
		exit (1);
	}

	player->priv->last_jumped = FALSE;
	player->priv->last_skipped = FALSE;

	gtk_box_set_spacing (GTK_BOX (player), 12);

	g_signal_connect_object (G_OBJECT (player->priv->mmplayer),
				 "info",
				 G_CALLBACK (info_available_cb),
				 player, 0);

	g_signal_connect_object (G_OBJECT (player->priv->mmplayer),
				 "eos",
				 G_CALLBACK (eos_cb),
				 player, 0);

	g_signal_connect_object (G_OBJECT (player->priv->mmplayer),
				 "tick",
				 G_CALLBACK (tick_cb),
				 player, 0);

	g_signal_connect_object (G_OBJECT (player->priv->mmplayer),
				 "error",
				 G_CALLBACK (error_cb),
				 player, 0);

	g_signal_connect_object (G_OBJECT (player->priv->mmplayer),
				 "buffering",
				 G_CALLBACK (buffering_cb),
				 player, 0);

	g_signal_connect (G_OBJECT (gnome_vfs_get_volume_monitor ()), 
			  "volume-pre-unmount",
			  G_CALLBACK (volume_pre_unmount_cb),
			  player);

	player->priv->gconf_play_order_id =
		eel_gconf_notification_add (CONF_STATE_PLAY_ORDER,
					    (GConfClientNotifyFunc)gconf_play_order_changed,
					    player);

	hbox = gtk_hbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);

	player->priv->tooltips = gtk_tooltips_new ();
	gtk_tooltips_enable (player->priv->tooltips);

	/* Previous button */
	image = gtk_image_new_from_stock (GTK_STOCK_MEDIA_PREVIOUS,
					 GTK_ICON_SIZE_LARGE_TOOLBAR);

	player->priv->prev_button = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER (player->priv->prev_button), image);
	g_signal_connect (G_OBJECT (player->priv->prev_button),
			  "clicked", G_CALLBACK (rb_shell_player_previous_button_cb), player);
	gtk_tooltips_set_tip (GTK_TOOLTIPS (player->priv->tooltips), 
			      GTK_WIDGET (player->priv->prev_button), 
			      _("Play previous song"), NULL);

	/* Button images */
	player->priv->play_image = gtk_image_new_from_stock (GTK_STOCK_MEDIA_PLAY,
							    GTK_ICON_SIZE_LARGE_TOOLBAR);
	g_object_ref (player->priv->play_image);
	player->priv->pause_image = gtk_image_new_from_stock (GTK_STOCK_MEDIA_PAUSE,
							     GTK_ICON_SIZE_LARGE_TOOLBAR);
	g_object_ref (player->priv->pause_image);
	player->priv->stop_image = gtk_image_new_from_stock (GTK_STOCK_MEDIA_STOP,
							    GTK_ICON_SIZE_LARGE_TOOLBAR);
	g_object_ref (player->priv->stop_image);

	player->priv->play_pause_stop_button = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER (player->priv->play_pause_stop_button), player->priv->play_image);
	player->priv->playbutton_state = PLAY_BUTTON_PLAY;

	g_signal_connect (G_OBJECT (player->priv->play_pause_stop_button),
			  "clicked", G_CALLBACK (rb_shell_player_playpause_button_cb), player);

	/* Next button */
	image = gtk_image_new_from_stock (GTK_STOCK_MEDIA_NEXT,
					 GTK_ICON_SIZE_LARGE_TOOLBAR);
	player->priv->next_button = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER (player->priv->next_button), image);
	g_signal_connect (G_OBJECT (player->priv->next_button),
			  "clicked", G_CALLBACK (rb_shell_player_next_button_cb), player);
	gtk_tooltips_set_tip (GTK_TOOLTIPS (player->priv->tooltips), 
			      GTK_WIDGET (player->priv->next_button), 
			      _("Play next song"), NULL);

	gtk_box_pack_start (GTK_BOX (hbox), player->priv->prev_button, FALSE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), player->priv->play_pause_stop_button, FALSE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), player->priv->next_button, FALSE, TRUE, 0);

	alignment = gtk_alignment_new (0.0, 0.5, 1.0, 0.0);
	gtk_container_add (GTK_CONTAINER (alignment), hbox);
	gtk_box_pack_start (GTK_BOX (player), alignment, FALSE, TRUE, 0);

	alignment = gtk_alignment_new (0.0, 0.5, 1.0, 0.0);
	player->priv->header_widget = rb_header_new (player->priv->mmplayer);
	gtk_container_add (GTK_CONTAINER (alignment), GTK_WIDGET (player->priv->header_widget));
	gtk_box_pack_start (GTK_BOX (player), alignment, TRUE, TRUE, 0);

	player->priv->volume_button = bacon_volume_button_new (GTK_ICON_SIZE_LARGE_TOOLBAR,
							       0.0, 1.0, 0.02);
	rb_shell_player_sync_volume (player);
	g_signal_connect (player->priv->volume_button, "value-changed",
			  G_CALLBACK (rb_shell_player_volume_widget_changed_cb),
			  player);

	gtk_tooltips_set_tip (GTK_TOOLTIPS (player->priv->tooltips), 
			      GTK_WIDGET (player->priv->volume_button), 
			      _("Change the music volume"), NULL);

	alignment = gtk_alignment_new (0.0, 0.5, 1.0, 0.0);
	gtk_container_add (GTK_CONTAINER (alignment), player->priv->volume_button);
	gtk_box_pack_end (GTK_BOX (player), alignment, FALSE, TRUE, 5);

	player->priv->gconf_state_id = 
		eel_gconf_notification_add (CONF_STATE_VOLUME,
					    (GConfClientNotifyFunc) rb_shell_player_volume_changed_cb,
					    player);

	g_signal_connect (player, "notify::playing",
			  G_CALLBACK (reemit_playing_signal), NULL);
				 
#ifdef HAVE_MMKEYS
	/* Enable Multimedia Keys */
	rb_shell_player_init_mmkeys (player);
#endif /* HAVE_MMKEYS */
}

static void
rb_shell_player_finalize (GObject *object)
{
	RBShellPlayer *player;

	g_return_if_fail (object != NULL);
	g_return_if_fail (RB_IS_SHELL_PLAYER (object));

	player = RB_SHELL_PLAYER (object);

	g_return_if_fail (player->priv != NULL);

	if (player->priv->playing_attempt_entry)
		rhythmdb_entry_unref (player->priv->db, player->priv->playing_attempt_entry);

	eel_gconf_notification_remove(player->priv->gconf_play_order_id);
	eel_gconf_notification_remove(player->priv->gconf_state_id);

	eel_gconf_set_float (CONF_STATE_VOLUME,
			     rb_player_get_volume (player->priv->mmplayer));

	g_object_unref (G_OBJECT (player->priv->mmplayer));

	g_object_unref (G_OBJECT (player->priv->play_order));

	gtk_object_destroy (GTK_OBJECT (player->priv->tooltips));

	if (player->priv->remote != NULL)
		g_object_unref (G_OBJECT (player->priv->remote));
	
	g_free (player->priv);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
rb_shell_player_set_property (GObject *object,
			      guint prop_id,
			      const GValue *value,
			      GParamSpec *pspec)
{
	RBShellPlayer *player = RB_SHELL_PLAYER (object);

	switch (prop_id)
	{
	case PROP_SOURCE:
		if (player->priv->selected_source != NULL) {
			RBEntryView *songs = rb_source_get_entry_view (player->priv->selected_source);
			GList *extra_views = rb_source_get_extra_views (player->priv->selected_source);

			g_signal_handlers_disconnect_by_func (G_OBJECT (songs),
							      G_CALLBACK (rb_shell_player_entry_view_changed_cb),
							      player);
			g_signal_handlers_disconnect_by_func (G_OBJECT (songs),
							      G_CALLBACK (rb_shell_player_entry_activated_cb),
							      player);
			for (; extra_views; extra_views = extra_views->next)
				g_signal_handlers_disconnect_by_func (G_OBJECT (extra_views->data),
								      G_CALLBACK (rb_shell_player_property_row_activated_cb),
								      player);
			g_list_free (extra_views);
			
		}

		
		player->priv->selected_source = g_value_get_object (value);
		rb_debug ("selected source %p", g_value_get_object (value));

		rb_shell_player_sync_with_selected_source (player);
		rb_shell_player_sync_buttons (player);

		if (player->priv->selected_source != NULL) {
			RBEntryView *songs = rb_source_get_entry_view (player->priv->selected_source);
			GList *extra_views = rb_source_get_extra_views (player->priv->selected_source);

			g_signal_connect_object (G_OBJECT (songs),
						 "changed",
						 G_CALLBACK (rb_shell_player_entry_view_changed_cb),
						 player, 0);
			g_signal_connect_object (G_OBJECT (songs),
						 "entry-activated",
						 G_CALLBACK (rb_shell_player_entry_activated_cb),
						 player, 0);
			for (; extra_views; extra_views = extra_views->next)
				g_signal_connect_object (G_OBJECT (extra_views->data),
							 "property-activated",
							 G_CALLBACK (rb_shell_player_property_row_activated_cb),
							 player, 0);

			g_list_free (extra_views);
			
			/* Set database object */
			g_object_set (G_OBJECT (player->priv->header_widget),
				      "db", player->priv->db, NULL);
		}
		
		break;
	case PROP_UI_MANAGER:
		player->priv->ui_manager = g_value_get_object (value);
		break;
	case PROP_DB:
		player->priv->db = g_value_get_object (value);
		
		/* Listen for changed entries to update metadata display */
		g_signal_connect_object (G_OBJECT (player->priv->db),
			 "entry_changed",
			 G_CALLBACK (rb_shell_player_entry_changed_cb),
			 player, 0);
		break;
	case PROP_ACTION_GROUP:
		player->priv->actiongroup = g_value_get_object (value);
		break;
	case PROP_PLAY_ORDER:
		eel_gconf_set_string (CONF_STATE_PLAY_ORDER, 
				      g_value_get_string (value));
		break;
	case PROP_VOLUME:
		eel_gconf_set_float (CONF_STATE_VOLUME, 
				     g_value_get_float (value));
		break;
	case PROP_STATUSBAR:
		player->priv->statusbar_widget = g_value_get_object (value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void 
rb_shell_player_get_property (GObject *object,
			      guint prop_id,
			      GValue *value,
			      GParamSpec *pspec)
{
	RBShellPlayer *player = RB_SHELL_PLAYER (object);

	switch (prop_id)
	{
	case PROP_SOURCE:
		g_value_set_object (value, player->priv->selected_source);
		break;
	case PROP_UI_MANAGER:
		g_value_set_object (value, player->priv->ui_manager);
		break;
	case PROP_DB:
		g_value_set_object (value, player->priv->db);
		break;
	case PROP_ACTION_GROUP:
		g_value_set_object (value, player->priv->actiongroup);
		break;
	case PROP_PLAY_ORDER:
	{
		char *play_order = eel_gconf_get_string (CONF_STATE_PLAY_ORDER);
		if (!play_order)
			play_order = g_strdup ("linear");
		g_value_set_string_take_ownership (value, play_order);
		break;
	}
	case PROP_PLAYING:
		g_value_set_boolean (value, rb_player_playing (player->priv->mmplayer));
		break;
	case PROP_VOLUME:
		g_value_set_float (value, rb_player_get_volume (player->priv->mmplayer));
		break;
	case PROP_STATUSBAR:
		g_value_set_object (value, player->priv->statusbar_widget);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

GQuark
rb_shell_player_error_quark (void)
{
	static GQuark quark = 0;
	if (!quark)
		quark = g_quark_from_static_string ("rb_shell_player_error");

	return quark;
}

void
rb_shell_player_set_selected_source (RBShellPlayer *player,
				     RBSource *source)
{
	g_return_if_fail (RB_IS_SHELL_PLAYER (player));
	g_return_if_fail (RB_IS_SOURCE (source));

	g_object_set (G_OBJECT (player),
		      "source", source,
		      NULL);
}

RBSource *
rb_shell_player_get_playing_source (RBShellPlayer *player)
{
	return player->priv->source;
}


RBShellPlayer *
rb_shell_player_new (RhythmDB *db, GtkUIManager *mgr, 
		     GtkActionGroup *actiongroup)
{
	return g_object_new (RB_TYPE_SHELL_PLAYER,			       
			     "ui-manager", mgr,
			     "action-group", actiongroup,
			     "db", db,
			     NULL);
}

RhythmDBEntry *
rb_shell_player_get_playing_entry (RBShellPlayer *player)
{
	
	if (player->priv->source) {
		RBEntryView *songs;
		songs = rb_source_get_entry_view (player->priv->source);
		return rb_entry_view_get_playing_entry (songs);
	}
	return NULL;
}

static gboolean
rb_shell_player_have_first (RBShellPlayer *player, RBSource *source)
{
	RBEntryView *songs;
	if (source) {
		songs = rb_source_get_entry_view (source);
		return rb_entry_view_get_first_entry (songs) != NULL;
	}
	return FALSE;
}

static void
rb_shell_player_open_playlist_location (TotemPlParser *playlist, const char *uri,
					const char *title, const char *genre,
					RBShellPlayer *player)
{
	GError *error = NULL;

	if (rb_player_playing (player->priv->mmplayer))
		return;

	if (!rb_player_open (player->priv->mmplayer, uri, &error)) {
		if (player->priv->playlist_parse_error != NULL) {
			g_error_free (player->priv->playlist_parse_error);
			player->priv->playlist_parse_error = NULL;
		}
		player->priv->playlist_parse_error = g_error_copy (error);
		return;
	}

	rb_player_play (player->priv->mmplayer, &error);
	if (error)
		player->priv->playlist_parse_error = g_error_copy (error);

	g_object_notify (G_OBJECT (player), "playing");
}

static gboolean
rb_shell_player_open_location (RBShellPlayer *player,
			       const char *location,
			       GError **error)
{
	char *unescaped;
	TotemPlParser *playlist;
	gboolean was_playing;
	gboolean playlist_parsed;
	TotemPlParserResult playlist_result = TOTEM_PL_PARSER_RESULT_UNHANDLED;

	unescaped = gnome_vfs_unescape_string_for_display (location);
	rb_debug ("Opening %s...", unescaped);
	g_free (unescaped);

	was_playing = rb_player_playing (player->priv->mmplayer);

	if (!rb_player_close (player->priv->mmplayer, error))
		return FALSE;

	g_free (player->priv->song);
	player->priv->song = NULL;

	playlist = totem_pl_parser_new ();
	g_signal_connect_object (G_OBJECT (playlist), "entry",
				 G_CALLBACK (rb_shell_player_open_playlist_location),
				 player, 0);
	totem_pl_parser_add_ignored_mimetype (playlist, "x-directory/normal");

	playlist_parsed = FALSE;
	if (rb_source_try_playlist (player->priv->source)) {
		playlist_result = totem_pl_parser_parse (playlist, location, FALSE);
		playlist_parsed = (playlist_result != TOTEM_PL_PARSER_RESULT_UNHANDLED);
	}
	g_object_unref (playlist);
	if (!playlist_parsed) {
		/* We get here if we failed to parse as a playlist */
		if (!rb_player_open (player->priv->mmplayer, location, error))
			return FALSE;
	} else if (playlist_result == TOTEM_PL_PARSER_RESULT_ERROR
		   && error
		   && player->priv->playlist_parse_error) {
		g_propagate_error (error, player->priv->playlist_parse_error);
	} else if (playlist_result == TOTEM_PL_PARSER_RESULT_SUCCESS) {
		/* We already started playback in the open_playlist_location
		 * callback
		 */
		;
	}

	rb_statusbar_set_progress (player->priv->statusbar_widget, -1.0, NULL);
	
	if (!rb_player_play (player->priv->mmplayer, error))
		return FALSE;

	if (!was_playing) {
		g_object_notify (G_OBJECT (player), "playing");
	}
	return TRUE;
}

static gboolean
rb_shell_player_open_entry (RBShellPlayer *player, RhythmDBEntry *entry, GError **error)
{
	return rb_shell_player_open_location (player, entry->location, error);
}

static gboolean
rb_shell_player_play (RBShellPlayer *player, GError **error)
{
	RBEntryView *songs = rb_source_get_entry_view (player->priv->selected_source);

	if (!rb_player_play (player->priv->mmplayer, error))
		return FALSE;

	rb_entry_view_set_playing (songs, TRUE);

	rb_shell_player_sync_with_source (player);
	rb_shell_player_sync_buttons (player);

	return TRUE;
}

static void
rb_shell_player_set_entry_playback_error (RBShellPlayer *player,
					  RhythmDBEntry *entry,
					  char *message)
{
	GValue value = { 0, };

	g_return_if_fail (RB_IS_SHELL_PLAYER (player));

	g_value_init (&value, G_TYPE_STRING);
	g_value_set_string (&value, message);
	rhythmdb_entry_set (player->priv->db,
			    entry,
			    RHYTHMDB_PROP_PLAYBACK_ERROR,
			    &value);
	g_value_unset (&value);
	rhythmdb_commit (player->priv->db);
}

static gboolean
do_next_idle (RBShellPlayer *player)
{
	GError *error = NULL;
	
	if (!rb_shell_player_do_next (player, &error)) {
		if (error->domain == RB_SHELL_PLAYER_ERROR
		    && error->code == RB_SHELL_PLAYER_ERROR_END_OF_PLAYLIST) {
			rb_debug ("No next entry, stopping playback");
			rb_shell_player_set_playing_source (player, NULL);
			g_object_notify (G_OBJECT (player), "playing");
		}
		g_warning ("do_next_idle: Unhandled error: %s", error->message);
	}

	return FALSE;
}

static gboolean
rb_shell_player_set_playing_entry (RBShellPlayer *player, RhythmDBEntry *entry, GError **error)
{
	RBEntryView *songs;
	GError *tmp_error = NULL;
	
	g_return_val_if_fail (player->priv->source != NULL, TRUE);
	g_return_val_if_fail (entry != NULL, TRUE);
	
	songs = rb_source_get_entry_view (player->priv->source);

	if (player->priv->playing_attempt_entry)
		rhythmdb_entry_unref (player->priv->db, player->priv->playing_attempt_entry);
	rhythmdb_entry_ref (player->priv->db, entry);
	player->priv->playing_attempt_entry = entry;
	if (!rb_shell_player_open_entry (player, entry, &tmp_error))
		goto lose;
	rb_shell_player_sync_replaygain (player, entry);
	if (!rb_shell_player_play (player, &tmp_error))
		goto lose;

	rb_entry_view_set_playing_entry (songs, entry);

	rb_debug ("Success!");
	/* clear error on successful playback */
	g_free (entry->playback_error);
	entry->playback_error = NULL;

	g_signal_emit (G_OBJECT (player),
		       rb_shell_player_signals[PLAYING_SONG_CHANGED], 0,
		       entry);
	g_signal_emit (G_OBJECT (player),
		       rb_shell_player_signals[PLAYING_URI_CHANGED], 0,
		       entry->location);

	rb_shell_player_sync_with_source (player);
	rb_shell_player_sync_buttons (player);

	return TRUE;
 lose:
	/* Ignore errors, shutdown the player */
	rb_player_close (player->priv->mmplayer, NULL);
	/* Mark this song as failed */
	rb_shell_player_set_entry_playback_error (player, entry, tmp_error->message);
	g_propagate_error (error, tmp_error);
	rhythmdb_entry_unref (player->priv->db, player->priv->playing_attempt_entry);
	player->priv->playing_attempt_entry = NULL;
	return FALSE;
}

static void
gconf_play_order_changed (GConfClient *client,guint cnxn_id,
			  GConfEntry *entry, RBShellPlayer *player)
{
	rb_debug ("gconf play order changed");
	player->priv->syncing_state = TRUE;
	rb_shell_player_sync_play_order (player);
	rb_shell_player_sync_buttons (player);
	rb_shell_player_sync_control_state (player);
	g_object_notify (G_OBJECT (player), "play-order");
	player->priv->syncing_state = FALSE;
}

gboolean
rb_shell_player_get_playback_state (RBShellPlayer *player,
				    gboolean *shuffle,
				    gboolean *repeat)
{
	int i, j;
	char *play_order;

	play_order = eel_gconf_get_string (CONF_STATE_PLAY_ORDER);
	if (!play_order) {
		g_warning (CONF_STATE_PLAY_ORDER " gconf key not found!");
		return FALSE;
	}

	for (i = 0; i < G_N_ELEMENTS(state_to_play_order); i++)
		for (j = 0; j < G_N_ELEMENTS(state_to_play_order[0]); j++)
			if (!strcmp (play_order, state_to_play_order[i][j]))
				goto found;

	g_free (play_order);
	return FALSE;

found:
	*shuffle = i > 0;
	*repeat = j > 0;
	g_free (play_order);
	return TRUE;
}

static void 
rb_shell_player_set_play_order (RBShellPlayer *player, const gchar *new_val)
{
	char *old_val;
	g_object_get (G_OBJECT (player), "play-order", &old_val, NULL);
	if (strcmp (old_val, new_val) != 0) {
		/* The notify signal will be emitted by the gconf notifier */
		eel_gconf_set_string (CONF_STATE_PLAY_ORDER, new_val);
	}
	g_free (old_val);
}

void
rb_shell_player_set_playback_state (RBShellPlayer *player, gboolean shuffle, gboolean repeat)
{
	const char *neworder = state_to_play_order[shuffle ? 1 : 0][repeat ? 1 : 0];
	rb_shell_player_set_play_order (player, neworder);
}

static void
rb_shell_player_sync_play_order (RBShellPlayer *player)
{
	char *new_play_order = eel_gconf_get_string (CONF_STATE_PLAY_ORDER);

	if (!new_play_order) {
		g_critical (CONF_STATE_PLAY_ORDER " gconf key not found!");
		new_play_order = g_strdup ("linear");
	}

	if (player->priv->play_order != NULL)
		g_object_unref (player->priv->play_order);
	
	player->priv->play_order = rb_play_order_new (new_play_order, player);
	g_free (new_play_order);
}

void
rb_shell_player_jump_to_current (RBShellPlayer *player)
{
	RBSource *source;
	RhythmDBEntry *entry;
	RBEntryView *songs;

	source = player->priv->source ? player->priv->source :
		player->priv->selected_source;

	songs = rb_source_get_entry_view (source);

	entry = rb_shell_player_get_playing_entry (player);	

	if (entry == NULL)
		return;
	
	rb_entry_view_scroll_to_entry (songs, entry);
	rb_entry_view_select_entry (songs, rb_entry_view_get_playing_entry (songs));
}

gboolean
rb_shell_player_do_previous (RBShellPlayer *player, GError **error)
{
	RhythmDBEntry* entry;

	if (player->priv->source != NULL) {
		rb_debug ("going to previous");

		entry = rb_play_order_get_previous (player->priv->play_order);
			
		if (entry) {
			rb_debug ("previous song found, doing previous");
			rb_play_order_go_previous (player->priv->play_order);
			if (!rb_shell_player_set_playing_entry (player, entry, error))
				return FALSE;
		} else {
			rb_debug ("no previous song found, signaling error");
			g_set_error (error,
				     RB_SHELL_PLAYER_ERROR,
				     RB_SHELL_PLAYER_ERROR_END_OF_PLAYLIST,
				     _("No previous song"));
			return FALSE;
		}
		
		rb_shell_player_jump_to_current (player);
		return TRUE;
	} else {
		g_set_error (error,
			     RB_SHELL_PLAYER_ERROR,
			     RB_SHELL_PLAYER_ERROR_NOT_PLAYING,
			     _("Not currently playing"));
		return FALSE;
	}
}

gboolean
rb_shell_player_do_next (RBShellPlayer *player, GError **error)
{
	if (player->priv->source != NULL) {
		RhythmDBEntry *entry = rb_play_order_get_next (player->priv->play_order);
		
		if (entry) {
			rb_play_order_go_next (player->priv->play_order);
			if (!rb_shell_player_set_playing_entry (player, entry, error))
				return FALSE;
		} else {
			g_set_error (error,
				     RB_SHELL_PLAYER_ERROR,
				     RB_SHELL_PLAYER_ERROR_END_OF_PLAYLIST,
				     _("No next song"));
			return FALSE;
		}
		rb_shell_player_jump_to_current (player);
	}
	return TRUE;
}

static void
rb_shell_player_do_previous_or_seek (RBShellPlayer *player, GError **error)
{
	rb_debug ("previous");
	/* If we're in the first 3 seconds go to the previous song,
	 * else restart the current one.
	 */
	if (player->priv->source != NULL
	    && rb_source_can_pause (player->priv->source)
	    && rb_player_get_time (player->priv->mmplayer) > 3) {
		rb_debug ("after 3 second previous, restarting song");
		rb_player_set_time (player->priv->mmplayer, 0);
		rb_header_sync_time (player->priv->header_widget);
	} else {
		rb_shell_player_do_previous (player, NULL);
	}
}

static void
rb_shell_player_previous_button_cb (GtkButton *button,
				    RBShellPlayer *player)
{
	rb_shell_player_do_previous_or_seek (player, NULL);
}

static void
rb_shell_player_next_button_cb (GtkButton *button,
				RBShellPlayer *player)
{
	rb_shell_player_do_next (player, NULL);
}

static void
rb_shell_player_cmd_previous (GtkAction *action,
			      RBShellPlayer *player)
{
	rb_shell_player_do_previous_or_seek (player, NULL);
}

static void
rb_shell_player_cmd_next (GtkAction *action,
			  RBShellPlayer *player)
{
	rb_debug ("next");
	rb_shell_player_do_next (player, NULL);
}

void
rb_shell_player_play_entry (RBShellPlayer *player,
			    RhythmDBEntry *entry)
{
	GError *error = NULL;
	rb_shell_player_set_playing_source (player, player->priv->selected_source);
	if (!rb_shell_player_set_playing_entry (player, entry, &error)) {
		rb_shell_player_error (player, FALSE, error);
		g_clear_error (&error);
	}
}

static void
rb_shell_player_cmd_play (GtkAction *action,
			  RBShellPlayer *player)
{
	GError *error = NULL;
	rb_debug ("play!");
	if (!rb_shell_player_playpause (player, TRUE, &error))
		rb_error_dialog (NULL, _("Couldn't start playback: %s"), error->message);
	g_clear_error (&error);
}

static void
rb_shell_player_playpause_button_cb (GtkButton *button,
				     RBShellPlayer *player)
{
	/* FIXME - do something with error... */
	rb_shell_player_playpause (player, FALSE, NULL);
}

gboolean
rb_shell_player_playpause (RBShellPlayer *player, gboolean ignore_stop, GError **error)
{
	gboolean ret;

	rb_debug ("doing playpause");

	g_return_val_if_fail (RB_IS_SHELL_PLAYER (player), TRUE);

	ret = TRUE;

	switch (player->priv->playbutton_state) {
	case PLAY_BUTTON_STOP:
		if (!ignore_stop || !rb_source_can_pause (player->priv->source)) {
			rb_debug ("setting playing source to NULL");
			rb_shell_player_set_playing_source (player, NULL);
			break;
		}
		/* fall through */
	case PLAY_BUTTON_PAUSE:
		rb_debug ("pausing mm player");
		rb_player_pause (player->priv->mmplayer);
		break;
	case PLAY_BUTTON_PLAY:
	{
		RhythmDBEntry *entry;
		if (player->priv->source == NULL) {
			/* no current stream, pull one in from the currently
			 * selected source */
			rb_debug ("no playing source, using selected source");
			rb_shell_player_set_playing_source (player, player->priv->selected_source);
		}

		entry = rb_shell_player_get_playing_entry (player);
		if (entry == NULL) {
			RBEntryView *songs = rb_source_get_entry_view (player->priv->source);

			GList* selection = rb_entry_view_get_selected_entries (songs);
			if (selection != NULL) {
				rb_debug ("choosing first selected entry");
				entry = (RhythmDBEntry*) selection->data;
			} else {
				entry = rb_play_order_get_next (player->priv->play_order);
			}
			if (entry != NULL) {
				rb_play_order_go_next (player->priv->play_order);
				if (!rb_shell_player_set_playing_entry (player, entry, error))
					ret = FALSE;
			}
		} else {
			if (!rb_shell_player_play (player, error)) {
				rb_shell_player_set_playing_source (player, NULL);
				ret = FALSE;
			}
		}
	}
	break;
	default:
		g_assert_not_reached ();
	}
	rb_shell_player_sync_with_source (player);
	rb_shell_player_sync_buttons (player);
	g_object_notify (G_OBJECT (player), "playing");
	return ret;
}

static void
rb_shell_player_cmd_pause (GtkAction *action,
			   RBShellPlayer *player)
{
	rb_debug ("pausing");
	/* FIXME handle error */
	rb_shell_player_playpause (player, TRUE, NULL);

}

static void
rb_shell_player_cmd_stop (GtkAction *action,
			  RBShellPlayer *player)
{
	rb_debug ("STOP FACTION WINS AGAIN!!");
	rb_shell_player_set_playing_source (player, NULL);
}

static void
rb_shell_player_sync_control_state (RBShellPlayer *player)
{
	gboolean shuffle, repeat;
	GtkAction *action;
	rb_debug ("syncing control state");

	if (!rb_shell_player_get_playback_state (player, &shuffle,
						 &repeat))
		return;

	action = gtk_action_group_get_action (player->priv->actiongroup,
					      "ControlShuffle");
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), shuffle);
	action = gtk_action_group_get_action (player->priv->actiongroup,
					      "ControlRepeat");
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), repeat);
}

static void
rb_shell_player_sync_volume (RBShellPlayer *player)
{
	float volume = eel_gconf_get_float (CONF_STATE_VOLUME);
	if (volume < 0.0)
		volume = 0.0;
	else if (volume > 1.0)
		volume = 1.0;
	bacon_volume_button_set_value (BACON_VOLUME_BUTTON (player->priv->volume_button),
				       volume);
	rb_player_set_volume (player->priv->mmplayer,
					volume);
					
	rb_shell_player_sync_replaygain (player, 
					 rb_shell_player_get_playing_entry (player));					
}

void
rb_shell_player_toggle_mute (RBShellPlayer *player)
{
	if (player->priv->mute) {
		rb_player_set_volume (player->priv->mmplayer, player->priv->pre_mute_volume);
		player->priv->mute = FALSE;
	} else {
		player->priv->pre_mute_volume = rb_player_get_volume (player->priv->mmplayer);
		rb_player_set_volume (player->priv->mmplayer, 0.0);
		player->priv->mute = TRUE;
	}
}

static void
rb_shell_player_sync_replaygain (RBShellPlayer *player, RhythmDBEntry *entry)
{
	double entry_track_gain = 0;
	double entry_track_peak = 0;
	double entry_album_gain = 0;
	double entry_album_peak = 0;
	
	if (entry != NULL) {
             	entry_track_gain = entry->track_gain;
             	entry_track_peak = entry->track_peak;
             	entry_album_gain = entry->album_gain;
             	entry_album_peak = entry->album_peak;
	}

	rb_player_set_replaygain (player->priv->mmplayer, entry_track_gain, 
				  entry_track_peak, entry_album_gain, entry_album_peak);
}

static void
rb_shell_player_volume_changed_cb (GConfClient *client,
				   guint cnxn_id,
				   GConfEntry *entry,
				   RBShellPlayer *playa)
{
	rb_debug ("volume changed");
	rb_shell_player_sync_volume (playa);
}

static void
rb_shell_player_volume_widget_changed_cb (BaconVolumeButton *vol,
					  RBShellPlayer     *playa)
{
	eel_gconf_set_float (CONF_STATE_VOLUME,
			     bacon_volume_button_get_value (vol));
}

static void
rb_shell_player_shuffle_changed_cb (GtkAction *action,
				    RBShellPlayer *player)
{
	const char *neworder;
	gboolean shuffle, repeat;

	if (player->priv->syncing_state)
		return;

	rb_debug ("shuffle changed");

	if (!rb_shell_player_get_playback_state (player, &shuffle, &repeat))
		return;
	shuffle = !shuffle;
	neworder = state_to_play_order[shuffle ? 1 : 0][repeat ? 1 : 0];
	rb_shell_player_set_play_order (player, neworder);
}
	
static void
rb_shell_player_repeat_changed_cb (GtkAction *action,
				   RBShellPlayer *player)
{
	const char *neworder;
	gboolean shuffle, repeat;
	rb_debug ("repeat changed");

	if (player->priv->syncing_state)
		return;

	if (!rb_shell_player_get_playback_state (player, &shuffle, &repeat))
		return;
	repeat = !repeat;
	neworder = state_to_play_order[shuffle ? 1 : 0][repeat ? 1 : 0];
	rb_shell_player_set_play_order (player, neworder);
}

static void
rb_shell_player_cmd_song_info (GtkAction *action,
			       RBShellPlayer *player)
{
	rb_debug ("song info");

	rb_source_song_properties (player->priv->selected_source);
}

static void
rb_shell_player_playing_entry_deleted_cb (RBEntryView *view,
					  RhythmDBEntry *entry,
					  RBShellPlayer *playa)
{
	rb_debug ("playing entry removed!");
	/* Here we are called via a signal from the entry view.
	 * Thus, we ensure we don't call back into the entry view
	 * to change things again.  When the playing entry is removed,
	 * the entry view takes care of setting itself to stop playing.
	 */
	rb_shell_player_set_playing_source_internal (playa, NULL, FALSE);
}

static void
rb_shell_player_entry_view_changed_cb (RBEntryView *view,
				       RBShellPlayer *playa)
{
	rb_debug ("entry view changed");
	rb_shell_player_sync_buttons (playa);
}

static void
rb_shell_player_entry_activated_cb (RBEntryView *view,
				   RhythmDBEntry *entry,
				   RBShellPlayer *playa)
{
	GError *error = NULL;

	g_return_if_fail (entry != NULL);

	rb_debug  ("got entry %p activated", entry);
	
	rb_shell_player_set_playing_source (playa, playa->priv->selected_source);

	if (!rb_shell_player_set_playing_entry (playa, entry, &error)) {
		rb_shell_player_error (playa, FALSE, error);
		g_clear_error (&error);
	}
}

static void
rb_shell_player_property_row_activated_cb (RBPropertyView *view,
					   const char *name,
					   RBShellPlayer *playa)
{
	RhythmDBEntry *entry;
	RBEntryView *songs;
	GError *error = NULL;

	rb_debug  ("got property activated");
	
	rb_shell_player_set_playing_source (playa, playa->priv->selected_source);

	/* RHYTHMDBFIXME - do we need to wait here until the query is finished?
	 */
	songs = rb_source_get_entry_view (playa->priv->source);
	entry = rb_entry_view_get_first_entry (songs);

	if (!entry)
		return;

	if (!rb_shell_player_set_playing_entry (playa, entry, &error)) {
		rb_shell_player_error (playa, FALSE, error);
		g_clear_error (&error);
	}
}

static void
rb_shell_player_set_play_button (RBShellPlayer *player,
			         PlayButtonState state)
{
	gtk_container_remove (GTK_CONTAINER (player->priv->play_pause_stop_button),
			      gtk_bin_get_child (GTK_BIN (player->priv->play_pause_stop_button)));

	gtk_action_group_set_visible (player->priv->play_action_group, FALSE);
	gtk_action_group_set_visible (player->priv->pause_action_group, FALSE);
	gtk_action_group_set_visible (player->priv->stop_action_group, FALSE);

	switch (state)
	{
	case PLAY_BUTTON_PLAY:
		rb_debug ("setting play button");
		gtk_action_group_set_visible (player->priv->play_action_group, TRUE);
		gtk_container_add (GTK_CONTAINER (player->priv->play_pause_stop_button),
				   player->priv->play_image);
		break;
	case PLAY_BUTTON_PAUSE:
		rb_debug ("setting pause button");
		gtk_action_group_set_visible (player->priv->pause_action_group, TRUE);
		gtk_container_add (GTK_CONTAINER (player->priv->play_pause_stop_button),
				   player->priv->pause_image);
		break;
	case PLAY_BUTTON_STOP:
		rb_debug ("setting STOP button");
		gtk_action_group_set_visible (player->priv->stop_action_group, TRUE);
		gtk_container_add (GTK_CONTAINER (player->priv->play_pause_stop_button),
				   player->priv->stop_image);
		break;
	default:
		g_error ("Should not get here!");
		break;
	}
	
	gtk_widget_show_all (GTK_WIDGET (player->priv->play_pause_stop_button));

	player->priv->playbutton_state = state;
}

static void
rb_shell_player_entry_changed_cb (RhythmDB *db, RhythmDBEntry *entry,
				       GSList *changes, RBShellPlayer *player)
{
	GSList *t;
	RhythmDBEntry *playing_entry = rb_shell_player_get_playing_entry (player);
	
	/* We try to update only if the changed entry is currently playing */
	if (entry != playing_entry) {
		return;
	}
	
	/* We update only if the artist, title or album has changed */
	for (t = changes; t; t = t->next)
	{
		RhythmDBEntryChange *change = t->data;
		switch (change->prop)
		{
			case RHYTHMDB_PROP_TITLE:
			case RHYTHMDB_PROP_ARTIST:
			case RHYTHMDB_PROP_ALBUM:
				rb_shell_player_sync_with_source (player);
				return;
			default:
				break;
		}
	}
}

static void
rb_shell_player_sync_with_source (RBShellPlayer *player)
{
	const char *entry_title = NULL;
	const char *artist = NULL;	
	char *title;
	RhythmDBEntry *entry;
	char *duration;

	entry = rb_shell_player_get_playing_entry (player);
	rb_debug ("playing source: %p, active entry: %p", player->priv->source, entry);

	if (entry != NULL) {
		entry_title = rb_refstring_get (entry->title);
		artist = rb_refstring_get (entry->artist);
	}

	if (player->priv->have_url)
		rb_header_set_urldata (player->priv->header_widget,
				       entry_title,
				       player->priv->url);
	else
		rb_header_set_urldata (player->priv->header_widget,
				       NULL, NULL);

	if (player->priv->song && entry_title)
		title = g_strdup_printf ("%s (%s)", player->priv->song,
					 entry_title);
	else if (entry_title && artist)
		title = g_strdup_printf ("%s - %s", artist, entry_title);
	else if (entry_title)
		title = g_strdup (entry_title);
	else
		title = NULL;

	duration = rb_header_get_elapsed_string (player->priv->header_widget);

	g_signal_emit (G_OBJECT (player), rb_shell_player_signals[WINDOW_TITLE_CHANGED], 0,
		       title);
	g_signal_emit (G_OBJECT (player), rb_shell_player_signals[DURATION_CHANGED], 0,
		       duration);
	g_free (duration);

	/* Sync the player */
	if (player->priv->song)
		rb_header_set_title (player->priv->header_widget, title);
	else
		rb_header_set_title (player->priv->header_widget, entry_title);
	g_free (title);
	
	rb_header_set_playing_entry (player->priv->header_widget, entry);
	rb_header_sync (player->priv->header_widget);
}

void
rb_shell_player_sync_buttons (RBShellPlayer *player)
{
	GtkAction *action;
	RBSource *source;
	gboolean not_empty = FALSE;
	gboolean have_previous = FALSE;
	gboolean have_next = FALSE;
	PlayButtonState pstate = PLAY_BUTTON_PLAY;
        gboolean not_small;
	RBEntryView *view;

	source = rb_shell_player_get_playing_entry (player) == NULL ?
		 player->priv->selected_source : player->priv->source;

	rb_debug ("syncing with source %p", source);

	/* If we have a source and it's not empty, next and prev depend
	 * on the availability of the next/prev entry. However if we are 
	 * shuffling only next make sense and if we are repeating next
	 * is always ok (restart)
	 */
	if (source && rb_shell_player_have_first (player, source)) {
		RBEntryView *songs;
		songs = rb_source_get_entry_view (source);

		not_empty = TRUE;

		/* Should these be up to the play order? */
		have_previous = rb_player_opened (player->priv->mmplayer);
		player->priv->have_previous_entry = (rb_entry_view_get_previous_entry (songs) != NULL);

		have_next = rb_play_order_has_next (player->priv->play_order);
	}

	gtk_widget_set_sensitive (GTK_WIDGET (player->priv->play_pause_stop_button), not_empty);
	action = gtk_action_group_get_action (player->priv->actiongroup,
					      "ControlPrevious");
	g_object_set (G_OBJECT (action), "sensitive", have_previous, NULL);
	gtk_widget_set_sensitive (GTK_WIDGET (player->priv->prev_button), have_previous);
	action = gtk_action_group_get_action (player->priv->actiongroup,
					      "ControlNext");
	g_object_set (G_OBJECT (action), "sensitive", have_next, NULL);

        not_small = !eel_gconf_get_boolean (CONF_UI_SMALL_DISPLAY);
	action = gtk_action_group_get_action (player->priv->actiongroup,
					      "ViewJumpToPlaying");
	g_object_set (G_OBJECT (action),
		      "sensitive",
		      rb_shell_player_get_playing_entry (player) != NULL
		      && not_small, NULL);

	view = rb_source_get_entry_view (player->priv->selected_source);
	action = gtk_action_group_get_action (player->priv->actiongroup,
					      "MusicProperties");
	g_object_set (G_OBJECT (action), "sensitive", rb_entry_view_have_selection (view),
		      NULL);

	if (rb_player_playing (player->priv->mmplayer)) {
		if (player->priv->source == player->priv->selected_source
		    && rb_source_can_pause (RB_SOURCE (player->priv->selected_source)))
			pstate = PLAY_BUTTON_PAUSE;
		else
			pstate = PLAY_BUTTON_STOP;

		rb_entry_view_set_playing (view, TRUE);

		action = gtk_action_group_get_action (player->priv->play_action_group,
						      "ControlPlay");
		g_object_set (G_OBJECT (action), "sensitive", TRUE, NULL);

	} else  {
		if (rb_player_opened (player->priv->mmplayer)
		    || player->priv->source == NULL
		    || player->priv->source == player->priv->selected_source)
			pstate = PLAY_BUTTON_PLAY;
		else
			pstate = PLAY_BUTTON_STOP;

		rb_entry_view_set_playing (view, FALSE);

		action = gtk_action_group_get_action (player->priv->play_action_group,
						      "ControlPlay");
		g_object_set (G_OBJECT (action),
			      "sensitive",
			      rb_shell_player_have_first (player, source), NULL);

	}

	rb_shell_player_set_play_button (player, pstate);
}

void
rb_shell_player_set_playing_source (RBShellPlayer *player,
				    RBSource *source)
{
	rb_shell_player_set_playing_source_internal (player, source, TRUE);
}

static void
rb_shell_player_set_playing_source_internal (RBShellPlayer *player,
					     RBSource *source,
					     gboolean sync_entry_view)

{
	if (player->priv->source == source && source != NULL)
		return;

	rb_debug ("setting playing source to %p", source);

	/* Stop the already playing source. */
	if (player->priv->source != NULL) {
		RBEntryView *songs = rb_source_get_entry_view (player->priv->source);
		if (sync_entry_view) {
			rb_debug ("source is already playing, stopping it");
			rb_entry_view_set_playing_entry (songs, NULL);
			rb_entry_view_set_playing (songs, FALSE);
		}
		g_signal_handlers_disconnect_by_func (G_OBJECT (songs),
						      G_CALLBACK (rb_shell_player_playing_entry_deleted_cb),
						      player);
	}
	
	player->priv->source = source;

	if (player->priv->playing_attempt_entry)
		rhythmdb_entry_unref (player->priv->db, player->priv->playing_attempt_entry);
	player->priv->playing_attempt_entry = NULL;

	if (source != NULL) {
		RBEntryView *songs = rb_source_get_entry_view (player->priv->source);
		g_signal_connect_object (G_OBJECT (songs),
					 "playing_entry_deleted",
					 G_CALLBACK (rb_shell_player_playing_entry_deleted_cb),
					 player, 0);
	}

	if (player->priv->play_order)
		rb_play_order_playing_source_changed (player->priv->play_order);


	g_free (player->priv->url);
	g_free (player->priv->song);
	player->priv->song = NULL;
	player->priv->url = NULL;
	player->priv->have_url = FALSE;

	if (source == NULL)
		rb_shell_player_stop (player);

	rb_shell_player_sync_with_source (player);
	if (player->priv->selected_source)
		rb_shell_player_sync_buttons (player);
	g_signal_emit (G_OBJECT (player), rb_shell_player_signals[PLAYING_SOURCE_CHANGED],
		       0, source);
}

void
rb_shell_player_stop (RBShellPlayer *player)
{
	GError *error = NULL;
	rb_debug ("stopping");

	g_return_if_fail (RB_IS_SHELL_PLAYER (player));

	if (rb_player_playing (player->priv->mmplayer))
		rb_player_pause (player->priv->mmplayer);
	rb_player_close (player->priv->mmplayer, &error);
	if (error) {
		rb_error_dialog (NULL,
				 _("Couldn't stop playback"),
				 "%s", error->message);
		g_error_free (error);
	}
}

gboolean
rb_shell_player_get_playing (RBShellPlayer *player)
{
	g_return_val_if_fail (RB_IS_SHELL_PLAYER (player), -1);

	return rb_player_playing (player->priv->mmplayer);
}

RBPlayer *
rb_shell_player_get_mm_player (RBShellPlayer *player)
{
	g_return_val_if_fail (RB_IS_SHELL_PLAYER (player), NULL);

	return player->priv->mmplayer;
}

long
rb_shell_player_get_playing_time (RBShellPlayer *player)
{
	g_return_val_if_fail (RB_IS_SHELL_PLAYER (player), 0);
	
	return rb_player_get_time (player->priv->mmplayer);
}

void
rb_shell_player_set_playing_time (RBShellPlayer *player, long time)
{
	g_return_if_fail (RB_IS_SHELL_PLAYER (player));
	
	if (rb_player_seekable (player->priv->mmplayer))
		rb_player_set_time (player->priv->mmplayer, time);
}

void
rb_shell_player_seek (RBShellPlayer *player, long offset)
{
	g_return_if_fail (RB_IS_SHELL_PLAYER (player));

	if (rb_player_seekable (player->priv->mmplayer)) {
		long t = rb_player_get_time (player->priv->mmplayer);
		rb_player_set_time (player->priv->mmplayer, t + offset);
	}
}

long
rb_shell_player_get_playing_song_duration (RBShellPlayer *player)
{
	RhythmDBEntry *current_entry;
	
	g_return_val_if_fail (RB_IS_SHELL_PLAYER (player), -1);
	
	current_entry = rb_shell_player_get_playing_entry (player);

	if (current_entry == NULL) {
		rb_debug ("Did not get playing entry : return -1 as length");
		return -1;
	}
	
	return current_entry->duration;
}

static void
rb_shell_player_sync_with_selected_source (RBShellPlayer *player)
{
	rb_debug ("syncing with selected source: %p", player->priv->selected_source);
	if (player->priv->source == NULL)
	{
		rb_debug ("no playing source, new source is %p", player->priv->selected_source);

		player->priv->have_url = rb_source_have_url (player->priv->selected_source);

		rb_shell_player_sync_with_source (player);
	}
}


static void
eos_cb (RBPlayer *mmplayer, gpointer data)
{
 	RBShellPlayer *player = RB_SHELL_PLAYER (data);
	rb_debug ("eos!");

	GDK_THREADS_ENTER ();

	if (player->priv->source != NULL) {
		RhythmDBEntry *entry = rb_shell_player_get_playing_entry (player);
		RBSource *source = player->priv->source;

		rb_debug ("updating play statistics");

		switch (rb_source_handle_eos (player->priv->source))
		{
		case RB_SOURCE_EOF_ERROR:
			rb_error_dialog (NULL, _("Stream error"),
					 _("Unexpected end of stream!"));
			rb_shell_player_set_playing_source (player, NULL);
			break;
		case RB_SOURCE_EOF_RETRY: {
			GTimeVal current;
			g_get_current_time (&current);
			if (player->priv->did_retry
			    && (current.tv_sec - player->priv->last_retry.tv_sec) < 4) {
				rb_debug ("Last retry was less than 4 seconds ago...aborting retry playback");
				player->priv->did_retry = FALSE;
				break;
			} else {
				player->priv->did_retry = TRUE;
				g_get_current_time (&(player->priv->last_retry));
				rb_shell_player_play_entry (player, entry);
			}
		}
			break;
		case RB_SOURCE_EOF_NEXT:
			rb_shell_player_do_next (player, NULL);
			break;
		}

		rb_source_update_play_statistics (source,
						  player->priv->db,
						  entry);
	}

	GDK_THREADS_LEAVE ();
}

static void
rb_shell_player_error (RBShellPlayer *player, gboolean async, const GError *err)
{
	RBEntryView *songs;
	RhythmDBEntry *entry;
	RhythmDBEntry *displayed_entry;

	g_assert (player->priv->handling_error == FALSE);

	player->priv->handling_error = TRUE;

	entry = player->priv->playing_attempt_entry;

	songs = rb_source_get_entry_view (player->priv->source);

	displayed_entry = rb_entry_view_get_playing_entry (songs);
	if (displayed_entry != entry) {
		/* This can happen if there was an error - the currently
		 * displayed entry won't match what we were trying to play.
		 * In order to get the play order to actually go next, we
		 * set the currently playing entry as a hack here.  This
		 * should really be fixed so RBPlayOrder also tracks
		 * the attempted entry, and just uses the view to decide
		 * where the next/previous one is.
		 */
		rb_entry_view_set_playing_entry (songs, entry);
	}

	g_printerr ("playback error: %s", err->message);
	/* For synchronous errors the entry playback error has already been set */
	if (entry && async)
		rb_shell_player_set_entry_playback_error (player, entry, err->message);

	if (err->code == RB_PLAYER_ERROR_NO_AUDIO)
		rb_shell_player_set_playing_source (player, NULL);
	else
		g_idle_add ((GSourceFunc)do_next_idle, player);

	player->priv->handling_error = FALSE;
}

static void
error_cb (RBPlayer *mmplayer, const GError *err, gpointer data)
{
	RBShellPlayer *player = RB_SHELL_PLAYER (data);

	if (player->priv->handling_error)
		return;

	if (player->priv->source == NULL) {
		rb_debug ("ignoring error (no source): %s", err->message);
		return;
	}

	GDK_THREADS_ENTER ();

	rb_shell_player_error (player, TRUE, err);
	
	rb_debug ("exiting error hander");
	GDK_THREADS_LEAVE ();
}

static void
tick_cb (RBPlayer *mmplayer, long elapsed, gpointer data)
{
 	RBShellPlayer *player = RB_SHELL_PLAYER (data);

	GDK_THREADS_ENTER ();

	rb_header_sync_time (player->priv->header_widget);

	if (rb_player_playing (mmplayer)) {
		static int callback_runs = 0;
		callback_runs++;
		if (callback_runs >= RB_PLAYER_TICK_HZ) {
			gchar *duration;

			duration = rb_header_get_elapsed_string (player->priv->header_widget);
			g_signal_emit (G_OBJECT (player), rb_shell_player_signals[DURATION_CHANGED],
				       0, duration);
			g_free (duration);
			callback_runs = 0;
		}
	}


	GDK_THREADS_LEAVE ();
}

static void
info_available_cb (RBPlayer *mmplayer,
                   RBMetaDataField field,
                   GValue *value,
                   gpointer data)
{
        RBShellPlayer *player = RB_SHELL_PLAYER (data);
        RBEntryView *songs;
        RhythmDBEntry *entry;
        RhythmDBPropType entry_field = 0;
        gboolean changed = FALSE;
        gboolean set_field = FALSE;
        rb_debug ("info: %d", field);

        /* Sanity check, this signal may come in after we stopped the
         * player */
        if (player->priv->source == NULL
            || !rb_player_opened (player->priv->mmplayer)) {
                rb_debug ("Got info_available but no playing source!");
                return;
        }

        GDK_THREADS_ENTER ();

        songs = rb_source_get_entry_view (player->priv->source);
        entry = rb_entry_view_get_playing_entry (songs);

        if (entry == NULL) {
                rb_debug ("Got info_available but no playing entry!");
                goto out_unlock;
        }

	if (entry->type != RHYTHMDB_ENTRY_TYPE_IRADIO_STATION) {
		rb_debug ("Got info_available but entry isn't an iradio station");
		goto out_unlock;
	}

	switch (field)	{
	case RB_METADATA_FIELD_TITLE:
	{
		char *song = g_value_dup_string (value);
		if (!g_utf8_validate (song, -1, NULL)) {
			g_warning ("Invalid UTF-8 from internet radio: %s", song);
			goto out_unlock;
		}

		if ((!song && player->priv->song)
		    || !player->priv->song
		    || strcmp (song, player->priv->song)) {
			changed = TRUE;
			g_free (player->priv->song);
			player->priv->song = song;
		}
		else
			g_free (song);
		break;
	}
	case RB_METADATA_FIELD_GENRE:
	{
		const char *genre = g_value_get_string (value);
		const char *existing;
		if (!g_utf8_validate (genre, -1, NULL)) {
			g_warning ("Invalid UTF-8 from internet radio: %s", genre);
			goto out_unlock;
		}

		/* check if the db entry already has a genre; if so, don't change it */
		existing = rb_refstring_get (entry->genre);
		if ((existing == NULL) || 
		    (strcmp (existing, "") == 0) ||
		    (strcmp (existing, _("Unknown")) == 0)) {
			entry_field = RHYTHMDB_PROP_GENRE;
			rb_debug ("setting genre of iradio station to %s", genre);
			set_field = TRUE;
		} else {
			rb_debug ("iradio station already has genre: %s; ignoring %s", existing, genre);
		}
		break;
	}
	case RB_METADATA_FIELD_COMMENT:
	{
		const char *name = g_value_get_string (value);
		const char *existing;
		if (!g_utf8_validate (name, -1, NULL)) {
			g_warning ("Invalid UTF-8 from internet radio: %s", name);
			goto out_unlock;
		}

		/* check if the db entry already has a title; if so, don't change it.
		 * consider title==URI to be the same as no title, since that's what
		 * happens for stations imported by DnD or commandline args.
		 * if the station title really is the same as the URI, then surely
		 * the station title in the stream metadata will say that too..
		 */
		existing = rb_refstring_get (entry->title);
		if ((existing == NULL) || 
		    (strcmp(existing, "") == 0) || 
		    (strcmp(existing, entry->location) == 0)) {
			entry_field = RHYTHMDB_PROP_TITLE;
			rb_debug ("setting title of iradio station to %s", name);
			set_field = TRUE;
		} else {
			rb_debug ("iradio station already has title: %s; ignoring %s", existing, name);
		}
		break;
	}
	default:
		break;
	}

	if (changed)
		rb_shell_player_sync_with_source (player);

	if (set_field && entry_field != 0) {
		rhythmdb_entry_set (player->priv->db, entry, entry_field, value);
		rhythmdb_commit (player->priv->db);
	}

 out_unlock:
	GDK_THREADS_LEAVE ();
}

static void
buffering_cb (RBPlayer *mmplayer, guint progress, gpointer data)
{
 	RBShellPlayer *player = RB_SHELL_PLAYER (data);

	GDK_THREADS_ENTER ();
	rb_statusbar_set_progress (player->priv->statusbar_widget, ((double)progress)/100, _("Buffering"));
	GDK_THREADS_LEAVE ();
}


const char *
rb_shell_player_get_playing_path (RBShellPlayer *shell_player)
{
	return shell_player->priv->url;
}

#ifdef HAVE_MMKEYS
static void
grab_mmkey (int key_code, GdkWindow *root)
{
	gdk_error_trap_push ();

	XGrabKey (GDK_DISPLAY (), key_code,
		  0,
		  GDK_WINDOW_XID (root), True,
		  GrabModeAsync, GrabModeAsync);
	XGrabKey (GDK_DISPLAY (), key_code,
		  Mod2Mask,
		  GDK_WINDOW_XID (root), True,
		  GrabModeAsync, GrabModeAsync);
	XGrabKey (GDK_DISPLAY (), key_code,
		  Mod5Mask,
		  GDK_WINDOW_XID (root), True,
		  GrabModeAsync, GrabModeAsync);
	XGrabKey (GDK_DISPLAY (), key_code,
		  LockMask,
		  GDK_WINDOW_XID (root), True,
		  GrabModeAsync, GrabModeAsync);
	XGrabKey (GDK_DISPLAY (), key_code,
		  Mod2Mask | Mod5Mask,
		  GDK_WINDOW_XID (root), True,
		  GrabModeAsync, GrabModeAsync);
	XGrabKey (GDK_DISPLAY (), key_code,
		  Mod2Mask | LockMask,
		  GDK_WINDOW_XID (root), True,
		  GrabModeAsync, GrabModeAsync);
	XGrabKey (GDK_DISPLAY (), key_code,
		  Mod5Mask | LockMask,
		  GDK_WINDOW_XID (root), True,
		  GrabModeAsync, GrabModeAsync);
	XGrabKey (GDK_DISPLAY (), key_code,
		  Mod2Mask | Mod5Mask | LockMask,
		  GDK_WINDOW_XID (root), True,
		  GrabModeAsync, GrabModeAsync);
	
	gdk_flush ();
        if (gdk_error_trap_pop ()) {
		rb_debug ("Error grabbing key");
	}
}

static GdkFilterReturn
filter_mmkeys (GdkXEvent *xevent, GdkEvent *event, gpointer data)
{
	XEvent *xev;
	XKeyEvent *key;
	RBShellPlayer *player;
	xev = (XEvent *) xevent;
	if (xev->type != KeyPress) {
		return GDK_FILTER_CONTINUE;
	}

	key = (XKeyEvent *) xevent;

	player = (RBShellPlayer *)data;

	if (XKeysymToKeycode (GDK_DISPLAY (), XF86XK_AudioPlay) == key->keycode) {	
		rb_shell_player_playpause (player, TRUE, NULL);
		return GDK_FILTER_REMOVE;
	} else if (XKeysymToKeycode (GDK_DISPLAY (), XF86XK_AudioPause) == key->keycode) {	
		if (rb_shell_player_get_playing	(player)) {
			rb_shell_player_playpause (player, TRUE, NULL);
		}
		return GDK_FILTER_REMOVE;
	} else if (XKeysymToKeycode (GDK_DISPLAY (), XF86XK_AudioStop) == key->keycode) {
		rb_shell_player_set_playing_source (player, NULL);
		return GDK_FILTER_REMOVE;		
	} else if (XKeysymToKeycode (GDK_DISPLAY (), XF86XK_AudioPrev) == key->keycode) {
		rb_shell_player_cmd_previous (NULL, player);
		return GDK_FILTER_REMOVE;		
	} else if (XKeysymToKeycode (GDK_DISPLAY (), XF86XK_AudioNext) == key->keycode) {
		rb_shell_player_cmd_next (NULL, player);
		return GDK_FILTER_REMOVE;
	} else {
		return GDK_FILTER_CONTINUE;
	}
}

static void
rb_shell_player_init_mmkeys (RBShellPlayer *shell_player)
{
	gint keycodes[] = {0, 0, 0, 0, 0};
	GdkDisplay *display;
	GdkScreen *screen;
	GdkWindow *root;
	guint i, j;

	keycodes[0] = XKeysymToKeycode (GDK_DISPLAY (), XF86XK_AudioPlay);
	keycodes[1] = XKeysymToKeycode (GDK_DISPLAY (), XF86XK_AudioStop);
	keycodes[2] = XKeysymToKeycode (GDK_DISPLAY (), XF86XK_AudioPrev);
	keycodes[3] = XKeysymToKeycode (GDK_DISPLAY (), XF86XK_AudioNext);
	keycodes[4] = XKeysymToKeycode (GDK_DISPLAY (), XF86XK_AudioPause);

	display = gdk_display_get_default ();

	for (i = 0; i < gdk_display_get_n_screens (display); i++) {
		screen = gdk_display_get_screen (display, i);

		if (screen != NULL) {
			root = gdk_screen_get_root_window (screen);

			for (j = 0; j < G_N_ELEMENTS (keycodes) ; j++) {
				if (keycodes[j] != 0)
					grab_mmkey (keycodes[j], root);
			}

			gdk_window_add_filter (root, filter_mmkeys,
					       (gpointer) shell_player);
		}
	}
}
#endif /* HAVE_MMKEYS */
