// standard imports
const Gettext = imports.gettext.domain("the-board");
const Lang = imports.lang;
const Mainloop = imports.mainloop;
const Tweener = imports.tweener.tweener;

// gi imports
const Tb = imports.gi.Tb;
const Clutter = imports.gi.Clutter;
const GLib = imports.gi.GLib;
const Gtk = imports.gi.Gtk;
const Mx = imports.gi.Mx;
const Pango = imports.gi.Pango;

// ui imports
const Thing = imports.ui.thing;
const ToolBox = imports.ui.toolBox;
const Toolbar = imports.ui.toolbar;

// util imports
const Path = imports.util.path;

const NAME = Gettext.gettext("Photo");
const STYLE = Path.THINGS_DATA_DIR + "photo/style.css";

const _TAPE_IMAGE = Path.THINGS_DATA_DIR + "photo/tape.png"

const _SHOW_BUTTON_BOX_TIME = 0.5;
const _UPDATE_SIZE_TIME = 0.3;
const _UPDATE_SIZE_TRANSITION = 'easeOutCubic';

function PhotoThing(args) {
    this._init(args);
}

PhotoThing.prototype = {
    __proto__: Thing.Thing.prototype,

    _init : function(args) {
        args = args || {};

        args.content = this;

        if ('initialWidth' in args) {
            this._initialWidth = args.initialWidth;
        } else {
            this._initialWidth = 320;
        }

        if ('initialHeight' in args) {
            this._initialHeight = args.initialHeight;
        } else {
            this._initialHeight = 213;
        }

        if ('minWidth' in args) {
            this._minWidth = args.minWidth;
        } else {
            this._minWidth = 320;
        }

        if ('minHeight' in args) {
            this._minHeight = args.minHeight;
        } else {
            this._minHeight = 213;
        }

        this._style = new Mx.Style();
        this._style.load_from_file(STYLE);

        this._createPhotoBox();
        this._createPhoto();
        this._createCaptionText();
        this._createTapes();

        Thing.Thing.prototype._init.apply(this, [args]);
    },

    _createPhotoBox : function() {
        this._photoBox =
            new Tb.Box({ orientation: Tb.BoxOrientation.VERTICAL,
                         xAlign: Tb.BoxAlignment.FILL,
                         yAlign: Tb.BoxAlignment.FILL,
                         name: "photo-thing-photo-box" });

        this._photoBox.set_style(this._style);
    },

    _createPhoto : function() {
        this._photo =
            new Clutter.Texture({ keepAspectRatio: true,
                                  loadAsync: true });

        this._photoBox.append(this._photo,
                              Tb.BoxPackFlags.EXPAND);
    },

    _createCaptionText : function() {
        this._captionLabel =
            new Mx.Label({ xAlign: Mx.Align.MIDDLE,
                           name: "photo-thing-caption-label" });

        this._captionLabel.set_style(this._style);

        this._captionLabel.clutterText.lineAlignment = Pango.Alignment.CENTER;

        this._captionLabel.connect("key-press-event",
                                   Lang.bind(this, this._onCaptionTextKeyPressEvent));

        this._captionLabel.clutterText.connect("text-changed",
                                               Lang.bind(this,
                                                         this._onCaptionTextChanged));

        this._photoBox.append(this._captionLabel,
                              Tb.BoxPackFlags.NONE);
    },

    _createTapes : function() {
        this._topTape =
            new Clutter.Texture({ keepAspectRatio: true,
                                  loadAsync: true,
                                  filename: _TAPE_IMAGE });

        this._photoBox.append(this._topTape,
                              Tb.BoxPackFlags.FIXED);

        this._topTape.anchorX = 25;
        this._topTape.anchorY = 15;

        this._bottomTape =
            new Clutter.Texture({ keepAspectRatio: true,
                                  loadAsync: true,
                                  filename: _TAPE_IMAGE });

        this._photoBox.append(this._bottomTape,
                              Tb.BoxPackFlags.FIXED);

        this._photoBox.set_fixed_child_align(this._bottomTape,
                                             Tb.BoxAlignment.END,
                                             Tb.BoxAlignment.END);

        this._bottomTape.anchorX = -35;
        this._bottomTape.anchorY = -25;
    },

    _connectPhotoSignals : function(fromState) {
        this._disconnectPhotoSignals();

        this._photoSizeChangeId =
            this._photo.connect("size-change",
                                Lang.bind(this,
                                          this._onPhotoSizeChange));

        this._photoLoadFinishedId =
            this._photo.connect("load-finished",
                                Lang.bind(this,
                                          this._onPhotoLoadFinished,
                                          fromState));
    },

    _disconnectPhotoSignals : function() {
        if (this._photoSizeChangeId) {
            this._photo.disconnect(this._photoSizeChangeId);
            delete this._photoSizeChangeId
        }

        if (this._photoLoadFinishedId) {
            this._photo.disconnect(this._photoLoadFinishedId);
            delete this._photoLoadFinishedId;
        }
    },

    _updateSpinner : function() {
        // FIXME: show/hide spinner depending on the
        // loading state of the photo
    },

    _updateImageFilename : function(imageFilename, fromState) {
        if (this._imageFilename == imageFilename) {
            return;
        }

        this._imageFilename = imageFilename;

        if (this._imageFilename) {
            this._connectPhotoSignals(fromState);
            this._updateSpinner();

            // hide photo while loading the new image file
            this._photo.opacity = 0;

            // start loading the new image file
            this._photo.filename = this._imageFilename;
        }

        if (!fromState) {
            this.emit('save');
        }
    },

    _updateInitialSize : function(width, height) {
        let aspectRatio = width / height;

        let newWidth;
        let newHeight;

        // FIXME: the image might be actually smaller
        // than the space available. What do we do in
        // that case?

        if (width >= height) {
            newWidth = this._initialWidth;
            newHeight = newWidth / aspectRatio;
        } else {
            newHeight = this._initialHeight;
            newWidth = newHeight * aspectRatio;
        }

        this._photoWidth = newWidth;
        this._photoHeight = newHeight;
    },

    _updatePhotoWithFileChooser : function() {
        let chooser = new Gtk.FileChooserDialog();

        chooser.add_button(Gtk.STOCK_CANCEL,
                           Gtk.ResponseType.REJECT);
        chooser.add_button(Gtk.STOCK_OK,
                           Gtk.ResponseType.ACCEPT);

        let imageFilter = new Gtk.FileFilter();

        imageFilter.set_name('Images');
        imageFilter.add_pixbuf_formats()

        chooser.add_filter(imageFilter);

        let picsDir = GLib.get_user_special_dir(GLib.UserDirectory.DIRECTORY_PICTURES);
        chooser.set_current_folder(picsDir);

        chooser.set_transient_for(this.context.gtkWindow);

        let response = chooser.run();

        if (response != Gtk.ResponseType.ACCEPT) {
            chooser.destroy();
            return;
        }

        let filename = chooser.get_filename();

        // Destroy dialog first, then set image
        chooser.destroy();

        // we have to restore key focus on stage
        // because the chooser grabs key focus
        this._captionLabel.clutterText.grab_key_focus();

        this._updateImageFilename(filename,
                                  false /* not from state*/);
    },

    _onCaptionTextKeyPressEvent : function(o, event) {
        let key = event.get_key_symbol();

        switch (key) {
        case Clutter.Return:
            this._updatePhotoWithFileChooser();
            return true;
        case Clutter.Escape:
            this.emit("deactivate");
            return true;
        }

        return false;
    },

    _onCaptionTextChanged : function() {
        this.emit('save');
    },

    _onPhotoSizeChange : function(photo, width, height) {
        if (this._photoWidth > 0 && this._photoHeight > 0) {
            return;
        }

        this._updateInitialSize(width, height);
    },

    _onPhotoLoadFinished : function(texture, error, fromState) {
        this._disconnectPhotoSignals();
        this._updateSpinner();

        [minTextHeight, naturalTextHeight] =
            this._captionLabel.get_preferred_height(-1);

        let thingWidth = this._photoWidth;
        let thingHeight = this._photoHeight + naturalTextHeight;

        this._minWidth = thingWidth;
        this._minHeight = thingHeight;

        if (!fromState) {
            Tweener.addTween(this.actor,
                             { width: thingWidth,
                               height: thingHeight,
                               time: fromState ? 0 : _UPDATE_SIZE_TIME,
                               transition: _UPDATE_SIZE_TRANSITION });
        }

        Tweener.addTween(this._photo,
                         { opacity: 255,
                           delay: fromState ? 0 : _UPDATE_SIZE_TIME,
                           time: fromState ? 0 : _UPDATE_SIZE_TIME });

        delete this._photoWidth;
        delete this._photoHeight;
    },

    activate : function() {
        this._captionLabel.clutterText.editable = true;
        this._captionLabel.clutterText.grab_key_focus();
    },

    deactivate : function() {
        this._captionLabel.clutterText.editable = false;
    },

    loadState : function(state) {
        if ('imageFilename' in state) {
            let fromState = 'width' in state &&
                            'height' in state;

            this._updateImageFilename(state.imageFilename,
                                      fromState);
        }

        if ('text' in state) {
            this._captionLabel.text = state.text;
        }
    },

    getState : function() {
        return { imageFilename: this._imageFilename,
                 text: this._captionLabel.text };
    },

    doAction : function(actionName, actionArgs) {
        if (actionName == "chooseFile") {
            this._updatePhotoWithFileChooser();
        }
    },

    validateSize : function(width, height) {
        // minWidth and minHeight always have a valid aspect
        // ratio once the image is loaded (see _onPhotoLoadFinished)
        let aspectRatio = this._minWidth / this._minHeight;

        // the point here is to keep aspect ratio while
        // resize the photo thing
        if (this._minWidth > this._minHeight) {
            return [width, width / aspectRatio];
        } else {
            return [height * aspectRatio, height];
        }
    },

    destroy : function() {
        this._disconnectPhotoSignals();

        if (this._photoBox) {
            this._photoBox.destroy();
            delete this._photoBox;
        }
    },

    get initialWidth() {
        return this._initialWidth;
    },

    get initialHeight() {
        return this._initialHeight;
    },

    get minWidth() {
        return this._minWidth;
    },

    get minHeight() {
        return this._minHeight;
    },

    get contentActor() {
        return this._photoBox;
    }
}

function create(args) {
    return new PhotoThing(args);
}

function createToolbar(args) {
    let toolbar =
        new Toolbar.Toolbar({ title: NAME,
                              visible: false });

    let toolBox =
        new ToolBox.ToolBox({ title: Gettext.gettext("Load from") });

    toolBox.addButton({ label: Gettext.gettext("File"),
                        actionName: "chooseFile" });

    toolbar.addToolBox(toolBox);

    return toolbar;
}
