/* valaccodecompiler.vala
 *
 * Copyright (C) 2007-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <gobject/valaccodecompiler.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/list.h>
#include <vala/valasourcefile.h>
#include <glib/gstdio.h>
#include <gobject/gvaluecollector.h>




enum  {
	VALA_CCODE_COMPILER_DUMMY_PROPERTY
};
static gboolean vala_ccode_compiler_package_exists (const char* package_name);
static gpointer vala_ccode_compiler_parent_class = NULL;
static void vala_ccode_compiler_finalize (ValaCCodeCompiler* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



ValaCCodeCompiler* vala_ccode_compiler_construct (GType object_type) {
	ValaCCodeCompiler* self;
	self = (ValaCCodeCompiler*) g_type_create_instance (object_type);
	return self;
}


ValaCCodeCompiler* vala_ccode_compiler_new (void) {
	return vala_ccode_compiler_construct (VALA_TYPE_CCODE_COMPILER);
}


static gboolean vala_ccode_compiler_package_exists (const char* package_name) {
	GError * inner_error;
	char* pc;
	gint exit_status;
	g_return_val_if_fail (package_name != NULL, FALSE);
	inner_error = NULL;
	pc = g_strconcat ("pkg-config --exists ", package_name, NULL);
	exit_status = 0;
	{
		gboolean _tmp0;
		g_spawn_command_line_sync (pc, NULL, NULL, &exit_status, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_SPAWN_ERROR) {
				goto __catch0_g_spawn_error;
			}
			goto __finally0;
		}
		return (_tmp0 = 0 == exit_status, pc = (g_free (pc), NULL), _tmp0);
	}
	goto __finally0;
	__catch0_g_spawn_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			gboolean _tmp1;
			vala_report_error (NULL, e->message);
			return (_tmp1 = FALSE, (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), pc = (g_free (pc), NULL), _tmp1);
		}
	}
	__finally0:
	if (inner_error != NULL) {
		pc = (g_free (pc), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return FALSE;
	}
	pc = (g_free (pc), NULL);
}


/**
 * Compile generated C code to object code and optionally link object
 * files.
 *
 * @param context a code context
 */
void vala_ccode_compiler_compile (ValaCCodeCompiler* self, ValaCodeContext* context, const char* cc_command, char** cc_options, int cc_options_length1) {
	GError * inner_error;
	char* pc;
	char* _tmp1;
	char* pkgflags;
	gint exit_status;
	const char* _tmp11;
	char* cmdline;
	gboolean _tmp21;
	gboolean _tmp22;
	GeeList* source_files;
	GeeList* c_source_files;
	char* _tmp34;
	char* _tmp33;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	inner_error = NULL;
	pc = g_strdup ("pkg-config --cflags");
	if (!vala_code_context_get_compile_only (context)) {
		char* _tmp0;
		_tmp0 = NULL;
		pc = (_tmp0 = g_strconcat (pc, " --libs", NULL), pc = (g_free (pc), NULL), _tmp0);
	}
	_tmp1 = NULL;
	pc = (_tmp1 = g_strconcat (pc, " gobject-2.0", NULL), pc = (g_free (pc), NULL), _tmp1);
	if (vala_code_context_get_thread (context)) {
		char* _tmp2;
		_tmp2 = NULL;
		pc = (_tmp2 = g_strconcat (pc, " gthread-2.0", NULL), pc = (g_free (pc), NULL), _tmp2);
	}
	{
		GeeList* _tmp3;
		GeeIterator* _tmp4;
		GeeIterator* _pkg_it;
		_tmp3 = NULL;
		_tmp4 = NULL;
		_pkg_it = (_tmp4 = gee_iterable_iterator ((GeeIterable*) (_tmp3 = vala_code_context_get_packages (context))), (_tmp3 == NULL) ? NULL : (_tmp3 = (gee_collection_object_unref (_tmp3), NULL)), _tmp4);
		while (gee_iterator_next (_pkg_it)) {
			char* pkg;
			pkg = (char*) gee_iterator_get (_pkg_it);
			if (vala_ccode_compiler_package_exists (pkg)) {
				char* _tmp6;
				char* _tmp5;
				_tmp6 = NULL;
				_tmp5 = NULL;
				pc = (_tmp6 = g_strconcat (pc, _tmp5 = g_strconcat (" ", pkg, NULL), NULL), pc = (g_free (pc), NULL), _tmp6);
				_tmp5 = (g_free (_tmp5), NULL);
			}
			pkg = (g_free (pkg), NULL);
		}
		(_pkg_it == NULL) ? NULL : (_pkg_it = (gee_collection_object_unref (_pkg_it), NULL));
	}
	pkgflags = NULL;
	exit_status = 0;
	{
		char* _tmp9;
		gboolean _tmp8;
		char* _tmp7;
		_tmp9 = NULL;
		_tmp7 = NULL;
		_tmp8 = g_spawn_command_line_sync (pc, &_tmp7, NULL, &exit_status, &inner_error);
		pkgflags = (_tmp9 = _tmp7, pkgflags = (g_free (pkgflags), NULL), _tmp9);
		_tmp8;
		if (inner_error != NULL) {
			if (inner_error->domain == G_SPAWN_ERROR) {
				goto __catch1_g_spawn_error;
			}
			goto __finally1;
		}
		if (exit_status != 0) {
			char* _tmp10;
			_tmp10 = NULL;
			vala_report_error (NULL, _tmp10 = g_strdup_printf ("pkg-config exited with status %d", exit_status));
			_tmp10 = (g_free (_tmp10), NULL);
			pc = (g_free (pc), NULL);
			pkgflags = (g_free (pkgflags), NULL);
			return;
		}
	}
	goto __finally1;
	__catch1_g_spawn_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			vala_report_error (NULL, e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			pc = (g_free (pc), NULL);
			pkgflags = (g_free (pkgflags), NULL);
			return;
		}
	}
	__finally1:
	if (inner_error != NULL) {
		pc = (g_free (pc), NULL);
		pkgflags = (g_free (pkgflags), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	/* TODO compile the C code files in parallel*/
	if (cc_command == NULL) {
		cc_command = "cc";
	}
	_tmp11 = NULL;
	cmdline = (_tmp11 = cc_command, (_tmp11 == NULL) ? NULL : g_strdup (_tmp11));
	if (vala_code_context_get_debug (context)) {
		char* _tmp12;
		_tmp12 = NULL;
		cmdline = (_tmp12 = g_strconcat (cmdline, " -g", NULL), cmdline = (g_free (cmdline), NULL), _tmp12);
	}
	if (vala_code_context_get_compile_only (context)) {
		char* _tmp13;
		_tmp13 = NULL;
		cmdline = (_tmp13 = g_strconcat (cmdline, " -c", NULL), cmdline = (g_free (cmdline), NULL), _tmp13);
	} else {
		if (vala_code_context_get_output (context) != NULL) {
			const char* _tmp14;
			char* output;
			gboolean _tmp15;
			gboolean _tmp16;
			char* _tmp20;
			char* _tmp19;
			char* _tmp18;
			_tmp14 = NULL;
			output = (_tmp14 = vala_code_context_get_output (context), (_tmp14 == NULL) ? NULL : g_strdup (_tmp14));
			_tmp15 = FALSE;
			_tmp16 = FALSE;
			if (vala_code_context_get_directory (context) != NULL) {
				_tmp16 = _vala_strcmp0 (vala_code_context_get_directory (context), "") != 0;
			} else {
				_tmp16 = FALSE;
			}
			if (_tmp16) {
				_tmp15 = !g_path_is_absolute (vala_code_context_get_output (context));
			} else {
				_tmp15 = FALSE;
			}
			if (_tmp15) {
				char* _tmp17;
				_tmp17 = NULL;
				output = (_tmp17 = g_strdup_printf ("%s%c%s", vala_code_context_get_directory (context), (gint) G_DIR_SEPARATOR, vala_code_context_get_output (context)), output = (g_free (output), NULL), _tmp17);
			}
			_tmp20 = NULL;
			_tmp19 = NULL;
			_tmp18 = NULL;
			cmdline = (_tmp20 = g_strconcat (cmdline, _tmp19 = g_strconcat (" -o ", _tmp18 = g_shell_quote (output), NULL), NULL), cmdline = (g_free (cmdline), NULL), _tmp20);
			_tmp19 = (g_free (_tmp19), NULL);
			_tmp18 = (g_free (_tmp18), NULL);
			output = (g_free (output), NULL);
		}
	}
	_tmp21 = FALSE;
	_tmp22 = FALSE;
	if (vala_code_context_get_legacy_headers (context)) {
		_tmp22 = vala_code_context_get_directory (context) != NULL;
	} else {
		_tmp22 = FALSE;
	}
	if (_tmp22) {
		_tmp21 = _vala_strcmp0 (vala_code_context_get_directory (context), "") != 0;
	} else {
		_tmp21 = FALSE;
	}
	/* make sure include files can be found if -d is used */
	if (_tmp21) {
		char* _tmp25;
		char* _tmp24;
		char* _tmp23;
		_tmp25 = NULL;
		_tmp24 = NULL;
		_tmp23 = NULL;
		cmdline = (_tmp25 = g_strconcat (cmdline, _tmp24 = g_strconcat (" -I", _tmp23 = g_shell_quote (vala_code_context_get_directory (context)), NULL), NULL), cmdline = (g_free (cmdline), NULL), _tmp25);
		_tmp24 = (g_free (_tmp24), NULL);
		_tmp23 = (g_free (_tmp23), NULL);
	}
	/* we're only interested in non-pkg source files */
	source_files = vala_code_context_get_source_files (context);
	{
		GeeIterator* _file_it;
		_file_it = gee_iterable_iterator ((GeeIterable*) source_files);
		while (gee_iterator_next (_file_it)) {
			ValaSourceFile* file;
			file = (ValaSourceFile*) gee_iterator_get (_file_it);
			if (!vala_source_file_get_external_package (file)) {
				char* _tmp29;
				char* _tmp28;
				char* _tmp27;
				char* _tmp26;
				_tmp29 = NULL;
				_tmp28 = NULL;
				_tmp27 = NULL;
				_tmp26 = NULL;
				cmdline = (_tmp29 = g_strconcat (cmdline, _tmp28 = g_strconcat (" ", _tmp27 = g_shell_quote (_tmp26 = vala_source_file_get_csource_filename (file)), NULL), NULL), cmdline = (g_free (cmdline), NULL), _tmp29);
				_tmp28 = (g_free (_tmp28), NULL);
				_tmp27 = (g_free (_tmp27), NULL);
				_tmp26 = (g_free (_tmp26), NULL);
			}
			(file == NULL) ? NULL : (file = (vala_source_file_unref (file), NULL));
		}
		(_file_it == NULL) ? NULL : (_file_it = (gee_collection_object_unref (_file_it), NULL));
	}
	c_source_files = vala_code_context_get_c_source_files (context);
	{
		GeeIterator* _file_it;
		_file_it = gee_iterable_iterator ((GeeIterable*) c_source_files);
		while (gee_iterator_next (_file_it)) {
			char* file;
			char* _tmp32;
			char* _tmp31;
			char* _tmp30;
			file = (char*) gee_iterator_get (_file_it);
			_tmp32 = NULL;
			_tmp31 = NULL;
			_tmp30 = NULL;
			cmdline = (_tmp32 = g_strconcat (cmdline, _tmp31 = g_strconcat (" ", _tmp30 = g_shell_quote (file), NULL), NULL), cmdline = (g_free (cmdline), NULL), _tmp32);
			_tmp31 = (g_free (_tmp31), NULL);
			_tmp30 = (g_free (_tmp30), NULL);
			file = (g_free (file), NULL);
		}
		(_file_it == NULL) ? NULL : (_file_it = (gee_collection_object_unref (_file_it), NULL));
	}
	/* add libraries after source files to fix linking
	 with --as-needed and on Windows*/
	_tmp34 = NULL;
	_tmp33 = NULL;
	cmdline = (_tmp34 = g_strconcat (cmdline, _tmp33 = g_strconcat (" ", g_strstrip (pkgflags), NULL), NULL), cmdline = (g_free (cmdline), NULL), _tmp34);
	_tmp33 = (g_free (_tmp33), NULL);
	{
		char** cc_option_collection;
		int cc_option_collection_length1;
		int cc_option_it;
		cc_option_collection = cc_options;
		cc_option_collection_length1 = cc_options_length1;
		for (cc_option_it = 0; cc_option_it < cc_options_length1; cc_option_it = cc_option_it + 1) {
			const char* _tmp38;
			char* cc_option;
			_tmp38 = NULL;
			cc_option = (_tmp38 = cc_option_collection[cc_option_it], (_tmp38 == NULL) ? NULL : g_strdup (_tmp38));
			{
				char* _tmp37;
				char* _tmp36;
				char* _tmp35;
				_tmp37 = NULL;
				_tmp36 = NULL;
				_tmp35 = NULL;
				cmdline = (_tmp37 = g_strconcat (cmdline, _tmp36 = g_strconcat (" ", _tmp35 = g_shell_quote (cc_option), NULL), NULL), cmdline = (g_free (cmdline), NULL), _tmp37);
				_tmp36 = (g_free (_tmp36), NULL);
				_tmp35 = (g_free (_tmp35), NULL);
				cc_option = (g_free (cc_option), NULL);
			}
		}
	}
	{
		g_spawn_command_line_sync (cmdline, NULL, NULL, &exit_status, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_SPAWN_ERROR) {
				goto __catch2_g_spawn_error;
			}
			goto __finally2;
		}
		if (exit_status != 0) {
			char* _tmp39;
			_tmp39 = NULL;
			vala_report_error (NULL, _tmp39 = g_strdup_printf ("cc exited with status %d", exit_status));
			_tmp39 = (g_free (_tmp39), NULL);
		}
	}
	goto __finally2;
	__catch2_g_spawn_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			vala_report_error (NULL, e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally2:
	if (inner_error != NULL) {
		pc = (g_free (pc), NULL);
		pkgflags = (g_free (pkgflags), NULL);
		cmdline = (g_free (cmdline), NULL);
		(source_files == NULL) ? NULL : (source_files = (gee_collection_object_unref (source_files), NULL));
		(c_source_files == NULL) ? NULL : (c_source_files = (gee_collection_object_unref (c_source_files), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	/* remove generated C source and header files */
	{
		GeeIterator* _file_it;
		/* remove generated C source and header files */
		_file_it = gee_iterable_iterator ((GeeIterable*) source_files);
		/* remove generated C source and header files */
		while (gee_iterator_next (_file_it)) {
			ValaSourceFile* file;
			/* remove generated C source and header files */
			file = (ValaSourceFile*) gee_iterator_get (_file_it);
			if (!vala_source_file_get_external_package (file)) {
				gboolean _tmp41;
				if (!vala_code_context_get_save_csources (context)) {
					char* _tmp40;
					_tmp40 = NULL;
					g_unlink (_tmp40 = vala_source_file_get_csource_filename (file));
					_tmp40 = (g_free (_tmp40), NULL);
				}
				_tmp41 = FALSE;
				if (vala_code_context_get_legacy_headers (context)) {
					_tmp41 = !vala_code_context_get_save_cheaders (context);
				} else {
					_tmp41 = FALSE;
				}
				if (_tmp41) {
					char* _tmp42;
					_tmp42 = NULL;
					g_unlink (_tmp42 = vala_source_file_get_cheader_filename (file));
					_tmp42 = (g_free (_tmp42), NULL);
				}
			}
			(file == NULL) ? NULL : (file = (vala_source_file_unref (file), NULL));
		}
		(_file_it == NULL) ? NULL : (_file_it = (gee_collection_object_unref (_file_it), NULL));
	}
	pc = (g_free (pc), NULL);
	pkgflags = (g_free (pkgflags), NULL);
	cmdline = (g_free (cmdline), NULL);
	(source_files == NULL) ? NULL : (source_files = (gee_collection_object_unref (source_files), NULL));
	(c_source_files == NULL) ? NULL : (c_source_files = (gee_collection_object_unref (c_source_files), NULL));
}


static void vala_value_ccode_compiler_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_value_ccode_compiler_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_ccode_compiler_unref (value->data[0].v_pointer);
	}
}


static void vala_value_ccode_compiler_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_ccode_compiler_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_value_ccode_compiler_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_value_ccode_compiler_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaCCodeCompiler* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = vala_ccode_compiler_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_value_ccode_compiler_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaCCodeCompiler** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_ccode_compiler_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* vala_param_spec_ccode_compiler (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaParamSpecCCodeCompiler* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_TYPE_CCODE_COMPILER), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer vala_value_get_ccode_compiler (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_CCODE_COMPILER), NULL);
	return value->data[0].v_pointer;
}


void vala_value_set_ccode_compiler (GValue* value, gpointer v_object) {
	ValaCCodeCompiler* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_CCODE_COMPILER));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_TYPE_CCODE_COMPILER));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_ccode_compiler_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_ccode_compiler_unref (old);
	}
}


static void vala_ccode_compiler_class_init (ValaCCodeCompilerClass * klass) {
	vala_ccode_compiler_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_COMPILER_CLASS (klass)->finalize = vala_ccode_compiler_finalize;
}


static void vala_ccode_compiler_instance_init (ValaCCodeCompiler * self) {
	self->ref_count = 1;
}


static void vala_ccode_compiler_finalize (ValaCCodeCompiler* obj) {
	ValaCCodeCompiler * self;
	self = VALA_CCODE_COMPILER (obj);
}


GType vala_ccode_compiler_get_type (void) {
	static GType vala_ccode_compiler_type_id = 0;
	if (vala_ccode_compiler_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_value_ccode_compiler_init, vala_value_ccode_compiler_free_value, vala_value_ccode_compiler_copy_value, vala_value_ccode_compiler_peek_pointer, "p", vala_value_ccode_compiler_collect_value, "p", vala_value_ccode_compiler_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeCompilerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_compiler_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeCompiler), 0, (GInstanceInitFunc) vala_ccode_compiler_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_ccode_compiler_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaCCodeCompiler", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_ccode_compiler_type_id;
}


gpointer vala_ccode_compiler_ref (gpointer instance) {
	ValaCCodeCompiler* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void vala_ccode_compiler_unref (gpointer instance) {
	ValaCCodeCompiler* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_CCODE_COMPILER_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




