/* valaerrortype.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valaerrortype.h>
#include <vala/valacodenode.h>
#include <vala/valasourcefile.h>
#include <vala/valacodecontext.h>
#include <vala/valanamespace.h>




struct _ValaErrorTypePrivate {
	ValaErrorDomain* _error_domain;
	ValaErrorCode* _error_code;
};

#define VALA_ERROR_TYPE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ERROR_TYPE, ValaErrorTypePrivate))
enum  {
	VALA_ERROR_TYPE_DUMMY_PROPERTY
};
static gboolean vala_error_type_real_compatible (ValaDataType* base, ValaDataType* target_type);
static char* vala_error_type_real_to_qualified_string (ValaDataType* base, ValaScope* scope);
static ValaDataType* vala_error_type_real_copy (ValaDataType* base);
static char* vala_error_type_real_get_cname (ValaDataType* base);
static char* vala_error_type_real_get_lower_case_cname (ValaDataType* base, const char* infix);
static gboolean vala_error_type_real_equals (ValaDataType* base, ValaDataType* type2);
static ValaSymbol* vala_error_type_real_get_member (ValaDataType* base, const char* member_name);
static char* vala_error_type_real_get_type_id (ValaDataType* base);
static gboolean vala_error_type_real_is_reference_type_or_type_parameter (ValaDataType* base);
static gboolean vala_error_type_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_error_type_parent_class = NULL;
static void vala_error_type_finalize (ValaCodeNode* obj);



ValaErrorType* vala_error_type_construct (GType object_type, ValaErrorDomain* error_domain, ValaErrorCode* error_code, ValaSourceReference* source_reference) {
	ValaErrorType* self;
	self = (ValaErrorType*) g_type_create_instance (object_type);
	vala_error_type_set_error_domain (self, error_domain);
	vala_data_type_set_data_type ((ValaDataType*) self, (ValaTypeSymbol*) error_domain);
	vala_error_type_set_error_code (self, error_code);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	return self;
}


ValaErrorType* vala_error_type_new (ValaErrorDomain* error_domain, ValaErrorCode* error_code, ValaSourceReference* source_reference) {
	return vala_error_type_construct (VALA_TYPE_ERROR_TYPE, error_domain, error_code, source_reference);
}


static gboolean vala_error_type_real_compatible (ValaDataType* base, ValaDataType* target_type) {
	ValaErrorType * self;
	ValaErrorType* _tmp2;
	ValaDataType* _tmp1;
	ValaErrorType* et;
	gboolean _tmp7;
	self = (ValaErrorType*) base;
	g_return_val_if_fail (target_type != NULL, FALSE);
	/* temporarily ignore type parameters */
	if (vala_data_type_get_type_parameter (target_type) != NULL) {
		return TRUE;
	}
	_tmp2 = NULL;
	_tmp1 = NULL;
	et = (_tmp2 = (_tmp1 = target_type, VALA_IS_ERROR_TYPE (_tmp1) ? ((ValaErrorType*) _tmp1) : NULL), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	/* error types are only compatible to error types */
	if (et == NULL) {
		gboolean _tmp3;
		return (_tmp3 = FALSE, (et == NULL) ? NULL : (et = (vala_code_node_unref (et), NULL)), _tmp3);
	}
	/* every error type is compatible to the base error type */
	if (et->priv->_error_domain == NULL) {
		gboolean _tmp4;
		return (_tmp4 = TRUE, (et == NULL) ? NULL : (et = (vala_code_node_unref (et), NULL)), _tmp4);
	}
	/* otherwhise the error_domain has to be equal */
	if (et->priv->_error_domain != self->priv->_error_domain) {
		gboolean _tmp5;
		return (_tmp5 = FALSE, (et == NULL) ? NULL : (et = (vala_code_node_unref (et), NULL)), _tmp5);
	}
	if (et->priv->_error_code == NULL) {
		gboolean _tmp6;
		return (_tmp6 = TRUE, (et == NULL) ? NULL : (et = (vala_code_node_unref (et), NULL)), _tmp6);
	}
	return (_tmp7 = et->priv->_error_code == self->priv->_error_code, (et == NULL) ? NULL : (et = (vala_code_node_unref (et), NULL)), _tmp7);
}


static char* vala_error_type_real_to_qualified_string (ValaDataType* base, ValaScope* scope) {
	ValaErrorType * self;
	self = (ValaErrorType*) base;
	if (self->priv->_error_domain == NULL) {
		return g_strdup ("GLib.Error");
	} else {
		return vala_symbol_get_full_name ((ValaSymbol*) self->priv->_error_domain);
	}
}


static ValaDataType* vala_error_type_real_copy (ValaDataType* base) {
	ValaErrorType * self;
	ValaErrorType* result;
	self = (ValaErrorType*) base;
	result = vala_error_type_new (self->priv->_error_domain, self->priv->_error_code, vala_code_node_get_source_reference ((ValaCodeNode*) self));
	vala_data_type_set_value_owned ((ValaDataType*) result, vala_data_type_get_value_owned ((ValaDataType*) self));
	vala_data_type_set_nullable ((ValaDataType*) result, vala_data_type_get_nullable ((ValaDataType*) self));
	return (ValaDataType*) result;
}


static char* vala_error_type_real_get_cname (ValaDataType* base) {
	ValaErrorType * self;
	self = (ValaErrorType*) base;
	return g_strdup ("GError*");
}


static char* vala_error_type_real_get_lower_case_cname (ValaDataType* base, const char* infix) {
	ValaErrorType * self;
	self = (ValaErrorType*) base;
	if (self->priv->_error_domain == NULL) {
		if (infix == NULL) {
			return g_strdup ("g_error");
		} else {
			return g_strdup_printf ("g_%s_error", infix);
		}
	} else {
		if (self->priv->_error_code == NULL) {
			return vala_symbol_get_lower_case_cname ((ValaSymbol*) self->priv->_error_domain, infix);
		} else {
			return vala_symbol_get_lower_case_cname ((ValaSymbol*) self->priv->_error_code, infix);
		}
	}
}


static gboolean vala_error_type_real_equals (ValaDataType* base, ValaDataType* type2) {
	ValaErrorType * self;
	ValaErrorType* _tmp1;
	ValaDataType* _tmp0;
	ValaErrorType* et;
	gboolean _tmp3;
	self = (ValaErrorType*) base;
	g_return_val_if_fail (type2 != NULL, FALSE);
	_tmp1 = NULL;
	_tmp0 = NULL;
	et = (_tmp1 = (_tmp0 = type2, VALA_IS_ERROR_TYPE (_tmp0) ? ((ValaErrorType*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	if (et == NULL) {
		gboolean _tmp2;
		return (_tmp2 = FALSE, (et == NULL) ? NULL : (et = (vala_code_node_unref (et), NULL)), _tmp2);
	}
	return (_tmp3 = self->priv->_error_domain == et->priv->_error_domain, (et == NULL) ? NULL : (et = (vala_code_node_unref (et), NULL)), _tmp3);
}


static ValaSymbol* vala_error_type_real_get_member (ValaDataType* base, const char* member_name) {
	ValaErrorType * self;
	ValaNamespace* _tmp0;
	ValaNamespace* root_symbol;
	ValaSymbol* _tmp1;
	ValaSymbol* _tmp2;
	ValaSymbol* gerror_symbol;
	ValaSymbol* _tmp3;
	self = (ValaErrorType*) base;
	g_return_val_if_fail (member_name != NULL, NULL);
	_tmp0 = NULL;
	root_symbol = (_tmp0 = vala_code_context_get_root (vala_source_file_get_context (vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)))), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
	_tmp1 = NULL;
	_tmp2 = NULL;
	gerror_symbol = (_tmp2 = vala_scope_lookup (vala_symbol_get_scope (_tmp1 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) root_symbol), "GLib")), "Error"), (_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL)), _tmp2);
	_tmp3 = NULL;
	return (_tmp3 = vala_scope_lookup (vala_symbol_get_scope (gerror_symbol), member_name), (root_symbol == NULL) ? NULL : (root_symbol = (vala_code_node_unref (root_symbol), NULL)), (gerror_symbol == NULL) ? NULL : (gerror_symbol = (vala_code_node_unref (gerror_symbol), NULL)), _tmp3);
}


static char* vala_error_type_real_get_type_id (ValaDataType* base) {
	ValaErrorType * self;
	self = (ValaErrorType*) base;
	return g_strdup ("G_TYPE_POINTER");
}


static gboolean vala_error_type_real_is_reference_type_or_type_parameter (ValaDataType* base) {
	ValaErrorType * self;
	self = (ValaErrorType*) base;
	return TRUE;
}


static gboolean vala_error_type_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaErrorType * self;
	self = (ValaErrorType*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (self->priv->_error_domain != NULL) {
		return vala_code_node_check ((ValaCodeNode*) self->priv->_error_domain, analyzer);
	}
	return TRUE;
}


ValaErrorDomain* vala_error_type_get_error_domain (ValaErrorType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_error_domain;
}


void vala_error_type_set_error_domain (ValaErrorType* self, ValaErrorDomain* value) {
	g_return_if_fail (self != NULL);
	self->priv->_error_domain = value;
}


ValaErrorCode* vala_error_type_get_error_code (ValaErrorType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_error_code;
}


void vala_error_type_set_error_code (ValaErrorType* self, ValaErrorCode* value) {
	g_return_if_fail (self != NULL);
	self->priv->_error_code = value;
}


static void vala_error_type_class_init (ValaErrorTypeClass * klass) {
	vala_error_type_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_error_type_finalize;
	g_type_class_add_private (klass, sizeof (ValaErrorTypePrivate));
	VALA_DATA_TYPE_CLASS (klass)->compatible = vala_error_type_real_compatible;
	VALA_DATA_TYPE_CLASS (klass)->to_qualified_string = vala_error_type_real_to_qualified_string;
	VALA_DATA_TYPE_CLASS (klass)->copy = vala_error_type_real_copy;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_error_type_real_get_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_lower_case_cname = vala_error_type_real_get_lower_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->equals = vala_error_type_real_equals;
	VALA_DATA_TYPE_CLASS (klass)->get_member = vala_error_type_real_get_member;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_error_type_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type_or_type_parameter = vala_error_type_real_is_reference_type_or_type_parameter;
	VALA_CODE_NODE_CLASS (klass)->check = vala_error_type_real_check;
}


static void vala_error_type_instance_init (ValaErrorType * self) {
	self->priv = VALA_ERROR_TYPE_GET_PRIVATE (self);
}


static void vala_error_type_finalize (ValaCodeNode* obj) {
	ValaErrorType * self;
	self = VALA_ERROR_TYPE (obj);
	VALA_CODE_NODE_CLASS (vala_error_type_parent_class)->finalize (obj);
}


GType vala_error_type_get_type (void) {
	static GType vala_error_type_type_id = 0;
	if (vala_error_type_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaErrorTypeClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_error_type_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaErrorType), 0, (GInstanceInitFunc) vala_error_type_instance_init, NULL };
		vala_error_type_type_id = g_type_register_static (VALA_TYPE_REFERENCE_TYPE, "ValaErrorType", &g_define_type_info, 0);
	}
	return vala_error_type_type_id;
}




