/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "xchat.h"
#include <ctype.h>
#ifdef USE_GNOME
#include <zvt/zvtterm.h>
#endif
#include <time.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/stat.h>
#include "cfgfiles.h"
#include "gtkutil.h"
#include "plugin.h"

extern GSList *sess_list;
extern GdkColor colors[];
extern GdkFont *font_normal;
extern GdkFont *font_bold;
extern GdkFont *dialog_font_normal;
extern GdkFont *dialog_font_bold;
extern GtkStyle *normaltab_style;
extern GtkStyle *redtab_style;
extern GtkStyle *channelwin_style;
extern struct xchatprefs prefs;
extern struct session *current_tab;
extern void check_special_chars(char *);
extern int buf_get_line(char *, char **, int *, int len);
extern void show_and_unfocus(GtkWidget *);

GtkWidget *pevent_dialog = NULL, *pevent_dialog_twid, *pevent_dialog_entry, *pevent_dialog_list, *pevent_dialog_hlist;

int pevt_build_string(char *input, char **output, int *max_arg);

long timecat(char *buf)
{
   time_t timval = time(0);
   char *tim = ctime(&timval) + 11;
   tim[9] = 0;
   strcat(buf, tim);
   return strlen(buf);
}

unsigned char *strip_color(unsigned char *text)
{
   int comma, done;
   int j = 0, i = 0, len = strlen(text);
   char *buf = malloc(len + 2);

   while(i < len)
   {
      switch(text[i])
      {
       case 2:
	 break;
       case 3:
	 comma = FALSE;
	 done = FALSE;
	 while(i < len && !done)
	 {
	    i++;
	    if(!isdigit(text[i]))
	    {
	       switch(text[i])
	       {
		case ',':
		  if(comma) done = TRUE; else comma = TRUE;
		  break;
		case ' ':
		  done = TRUE;
		  i++;
		  break;
		default:
		  done = TRUE;
	       }
	    }
	 }
	 i--;
	 break;
       default:
	 buf[j] = text[i];
	 j++;
      }
      i++;
   }
   buf[j] = 0;

   return buf;
}


#ifdef USE_GNOME

/*                0  1  2  3  4  5  6  7  8     9     10 11    12    13    14 */
int colconv[] = { 7, 0, 4, 2, 1, 3, 5, 3, 3+10, 2+10, 6, 6+10, 4+10, 5+10, 10+0, 0 };

void PrintTextRawZvt(GtkWidget *textwidget, unsigned char *text)
{
   int dotime = FALSE;
   char num[8];
   int comma, k, i = 0, j = 0, len = strlen(text);
   unsigned char *newtext = malloc(len + 1024);

   if(prefs.timestamp)
   {
      newtext[0] = 0;
      j = timecat(newtext);
   }

   while(i < len)
   {
      if(dotime && text[i] != 0)
      {
	 dotime = FALSE;
	 newtext[j] = 0;
	 j = timecat(newtext);
      }
      switch(text[i])
      {
       case 3:
	 i++;
	 if(text[i] == ' ')
	 {
	    newtext[j] = 27; j++;
	    newtext[j] = '['; j++;
	    newtext[j] = '0'; j++;
	    newtext[j] = 'm'; j++;
	    goto jump2;
	 }
	 k = 0;
	 comma = FALSE;
	 while(i < len)
	 {
	    if(text[i] >= '0' && text[i] <= '9' && k < 2)
	    {
	       num[k] = text[i];
	       k++;
	    } else {
	       int col, mirc;
	       num[k] = 0;
	       newtext[j] = 27; j++;
	       newtext[j] = '['; j++;
	       if(k == 0)
	       {
		  newtext[j] = '0'; j++;
		  newtext[j] = 'm'; j++;
	       } else {
		  if(comma) col = 40; else col = 30;
		  mirc = atoi(num);
		  mirc = colconv[mirc];
		  if(mirc > 9)
	          {
		     mirc -= 10;
		     sprintf((char *)&newtext[j], "1;%dm", mirc + col);
		  } else {
		     sprintf((char *)&newtext[j], "0;%dm", mirc + col);
		  }
		  j = strlen(newtext);
	       }
	       switch(text[i])
	       {
		case ',': comma = TRUE; break;
		default: goto jump;
	       }
	       k = 0;
	    }
	    i++;
	 }
	 break;
       case '\007':
	 if(!prefs.filterbeep)
	 {
	    newtext[j] = text[i];
	    j++;
	 }
	 break;
       case '\n':
	 newtext[j] = '\r';
	 j++;
	 if(prefs.timestamp) dotime = TRUE;
       default:
	 newtext[j] = text[i];
	 j++;
      }
      jump2:
      i++;
      jump:
   }
   newtext[j] = 0;
   zvt_term_feed((ZvtTerm*)textwidget, newtext, j);
   free(newtext);
}

#else

void PrintTextRawZvt(GtkWidget *textwidget, unsigned char *text)
{
}

#endif

void cut_down_text(GtkWidget *textwidget)
{
   if(prefs.bufsize > 0)
   {
      long n = gtk_text_get_length((GtkText*)textwidget);			 
      if(n > prefs.bufsize)
      {
	 gtk_text_set_point((GtkText*)textwidget, n - prefs.bufsize);
	 gtk_text_backward_delete((GtkText*)textwidget, n - prefs.bufsize);
	 gtk_text_set_point((GtkText*)textwidget, gtk_text_get_length((GtkText*)textwidget));
      }
   }
}

void PrintTextRaw(GtkWidget *textwidget, unsigned char *text,
		  char bg, char fg,
		  GdkFont *fontnorm,
		  GdkFont *fontbold)
{
   int dotime = FALSE;
   int esc = FALSE;
   int comma = FALSE;
   unsigned char buf[4096];
   unsigned char num[4];
   int bcol = bg, col = fg, j = 0, i = 0, k = 0;
   int scroll = FALSE;
   GtkAdjustment *adj;
   GdkFont *font = fontnorm;
   int bold = FALSE;

   adj = (GTK_TEXT(textwidget))->vadj;
   if(adj->value == adj->upper - adj->lower - adj->page_size) scroll = TRUE;

   if(prefs.timestamp)
   {
      buf[0] = 0;
      j = timecat(buf);
   }

   gtk_text_freeze (GTK_TEXT(textwidget));
   while(1)
   {
      if(dotime && text[i] != 0)
      {
	 dotime = FALSE;
	 buf[j] = 0;
	 j = timecat(buf);
      }
      if(esc)
      {
	 if(text[i] == ' ')
	 {
	    esc = FALSE; bcol = bg; col = fg;
	 } else {
	    while(text[i] >= '0' && text[i] <= '9' && k < 2)
	    {
	       num[k] = text[i];
	       k++; i++;
	    }
	    num[k] = 0; k = 0;
	    switch(text[i])
	    {
	     case ',':
	       comma = TRUE;
	       col = atoi(num);
	       if(col < 0 || col > 15) col = 1;
	       break;
	     default:
	       if(comma)
	       {
		  comma = FALSE;
		  bcol = atoi(num);
		  if(bcol < 0 || bcol > 15) bcol = bg;
	       } else {
		  col = atoi(num);
		  if(col < 0 || col > 15) col = fg;
	       }
	       if(bcol == 1 && col == 1) col = 0;
	       if(bcol == 0 && col == 0) col = 1;
	       goto norm;
	    }
	 }
      } else {
norm:
	 esc = FALSE;
	 switch(text[i])
	 {
	  case 0:
	    goto jump;
	  case 3: /* CTRL-C */
	    buf[j] = 0;
	    if(j>0)
	    {
	       gtk_text_insert(GTK_TEXT(textwidget), font, &colors[col], &colors[bcol],  buf, j);
	       j = 0;
	    }
	    esc = TRUE; k = 0;
	    break;
	  case '\007': /* beep */
	    if(!prefs.filterbeep)
	    {
	       buf[j] = 7; j++;
	       buf[j] = 0;
	       gtk_text_insert(GTK_TEXT(textwidget), font, &colors[col], &colors[bcol],  buf, j);
	       j = 0;
	       gdk_beep();
	    }
	    break;
	  case '\026': /* REVERSE */
	  case '\037': /* UNDERLINE */
	  case '\002': /* BOLD */
	    buf[j] = 0;
	    if(j>0)
	    {
	       gtk_text_insert(GTK_TEXT(textwidget), font, &colors[col], &colors[bcol],  buf, j);
	       j = 0;
	    }
	    if(bold)
	    {
	       bold = FALSE;
	       font = fontnorm;
	    } else {
	       bold = TRUE;
	       font = fontbold;
	    }
	    break;
	  case '\017': /* ALL OFF */
	    buf[j] = 0;
	    if(j>0)
	    {
	       gtk_text_insert(GTK_TEXT(textwidget), font, &colors[col], &colors[bcol],  buf, j);
	       j = 0;
	    }
	    font = fontnorm;
	    break;
	  default:
	    esc = FALSE;
	    comma = FALSE;
	    buf[j] = text[i];
	    j++;
	    if(j == 4095) j = 4094;
	    if(text[i] == '\n' && prefs.timestamp) dotime = TRUE;
	 }
      }
      i++;
   }
   jump:
   if(j)
   {
      gtk_text_insert(GTK_TEXT(textwidget), font, &colors[col], &colors[bcol], buf, j);
      cut_down_text(textwidget);
      gtk_text_thaw(GTK_TEXT(textwidget));
      if(scroll)
	gtk_adjustment_set_value(adj, adj->upper - adj->lower - adj->page_size);
   } else {
      cut_down_text(textwidget);
      gtk_text_thaw(GTK_TEXT(textwidget));
   }
}

void PrintText(struct session *sess, unsigned char *text)
{
   if(!sess) sess = (struct session *)sess_list->data;

    if(sess->logfd != -1 && prefs.logging)
   {
      time_t timval = time(0);
      char *tim = ctime(&timval) + 11;
      char *temp = strip_color(text);
      tim[9] = 0;
      if(prefs.timestamp)
        write(sess->logfd, tim, strlen(tim));
      write(sess->logfd, temp, strlen(temp));
      free(temp);
   }

   if(sess->zvt)
     PrintTextRawZvt(sess->textgad, text);
   else {
      if(sess->is_dialog)
	PrintTextRaw(sess->textgad, text, prefs.dialog_bg_color, prefs.dialog_fg_color,
		     dialog_font_normal, dialog_font_bold);
      else
	PrintTextRaw(sess->textgad, text, prefs.bg_color, prefs.fg_color,
		     font_normal, font_bold);
   }

#ifdef USE_PANEL
   if(!sess->new_data && !sess->nick_said && sess != current_tab)
   {
      if(sess->panel_button)
	 gtk_widget_set_style(GTK_BIN(sess->panel_button)->child, redtab_style);
   }
#endif

   if(!sess->new_data && sess != current_tab &&
      sess->is_tab && !sess->nick_said)
   {
      sess->new_data = TRUE;
      gtk_widget_set_style(sess->changad, redtab_style);
   }
}

void	end_logging (int fd)
{
	char	obuf[512];
	time_t	currenttime;
	
	currenttime = time(NULL);
	write(fd, obuf, snprintf(obuf, 510, "**** ENDING LOGGING AT %s\n", ctime(&currenttime)));
	close(fd);
}

int	open_log_file (char *servname, char *channame)
{
	char	buf[512];
	int	fd;
	time_t	currenttime;
        struct  stat st;

        snprintf(buf, 510, "%s/xchatlogs", get_xdir());
        if(stat(buf, &st) < 0)
                mkdir(buf, S_IRUSR | S_IWUSR | S_IXUSR);

        if(prefs.no_server_logs)
          snprintf(buf, 510, "%s/xchatlogs/%s.xchatlog", get_xdir(), channame);
        else
          snprintf(buf, 510, "%s/xchatlogs/%s,%s.xchatlog", get_xdir(), servname, channame);
	fd = open(buf, O_CREAT | O_APPEND | O_WRONLY, 0x180);
	if (fd < 0)
		return -1;
	currenttime = time(NULL);
	write(fd, buf, snprintf(buf, 510, "**** BEGIN LOGGING AT %s\n", ctime(&currenttime)));

	return fd;
}

void	setup_logging (struct session *sess)
{
	if (sess->logfd != -1)
		end_logging(sess->logfd);
	sess->logfd = open_log_file(sess->server->servername, sess->channel);
}
/* Print Events stuff here --AGL */

/* Consider the following a NOTES file:

   This is a test patch, it is not debugged in any way and also most certainly wont work without USE_PLUGIN configured. Basically it uses the signal system (which has been detached from the main plugin system) to send text messages. The main upshot of this is:
   * Plugins and Perl scripts (when I get round to signaling perl.c) can intercept text events and do what they like
   * The default text engine can be config'ed
   
   By default it should appear *exactly* the same (I'm working hard not to change the default style) but if you go into Settings->Edit Event Texts you can change the text's. The format is thus:
   
   The normal %Cx (color) and %B (bold) etc work
   
   $x is replaced with the data in var x (e.g. $1 is often the nick)

   $axxx is replace with a single byte of value xxx (in base 10)
   
   AGL (990507)
*/

/* These lists are thus:
   pntevts_text[] are the strings the user sees (WITH %x etc)
   pntevts[] are the data strings with \000 etc
   evtnames[] are the names of the events
*/

/* To add a new event:

   Choose a name for the event (like JOIN)
   Add it's desc to the end of evtnames (just below)
   Make up a pevt_name_help struct (with a NULL at the end)
   Add pevt_name_help to the end of evthelp
   Add the default value to the end of pevt_defaults

   Open up plugin.h

   Add one to TE_MAX_VAL
   Add a #define XP_TE_NAME with a value of +1 on the last XP_TE
   Add one to XP_NUM

   Go back to text.c

   Append the number with TE_MAX_VAL is now to signal_2_te
   Append the num of args to te_num_args
   Append the XP_TE_NAME number to te_2_signal
*/

/* As of 990528 this code is in BETA --AGL */

/* Internals:

   On startup ~/.xchat/printevents.conf is loaded if it doesn't exist the
   defaults are loaded. Any missing events are filled from defaults.
   Each event is parsed by pevt_build_string and a binary output is produced
   which looks like:

   (byte) value: 0 = {
      (int) numbers of bytes
      (char []) that number of byte to be memcpy'ed into the buffer
   }
   1 =
      (byte) number of varable to insert
   2 = end of buffer

   Each XP_TE_* signal is hard coded to call textsignal_handler which calls
   display_event which decodes the data

   This means that this system *should be faster* than snprintf because
   it always 'knows' that format of the string (basically is preparses much
   of the work)

   --AGL
*/
      

char ** pntevts_text;
char ** pntevts;
char * evtnames [] = {
   "Join",
   "Channel Action",
   "Channel Message",
   "Private Message",
   "Change Nick",
   "New Topic",
   "Topic",
   "Kick",
   "Part",
   "Channel Creation",
   "Topic Creation",
   "Quit",
   "Ping Reply",
   "Notice",
   "You Joining",
   "Your Message",
   "Private Message to Dialog",
   "Your Nick Changing",
   "You're Kicked",
   "You Parting",
   "CTCP Sound",
   "CTCP Generic",
   "CTCP Generic to Channel",
   "Channel Set Key",
   "Channel Set Limit",
   "Channel Operator",
   "Channel Voice",
   "Channel Ban",
   "Channel Remove Keyword",
   "Channel Remove Limit",
   "Channel DeOp",
   "Channel DeVoice",
   "Channel UnBan",
   "Channel Mode Generic",
   "WhoIs Name Line",
   "WhoIs Channel/Oper Line",
   "WhoIs Server Line",
   "WhoIs Idle Line",
   "WhoIs Idle Line with Signon",
   "WhoIs Away Line",
   "WhoIs End",
   "User Limit",
   "Banned",
   "Invite",
   "Keyword",
   "MOTD Skipped",
   "Server Text",
   "Invited",
   "Users On Channel",
   "Nick Clash",
   "Nick Failed",
   "Unknown Host",
   "Connection Failed",
   "Connecting",
   "Connected",
   "Stop Connection",
   "Disconnected",
   "No DCC",
   "Delete Notify",
   "Add Notify",
   "Window Type Error",
   "Channel Modes",
   "Raw Modes",
   "Killed",
   "DCC Stall",
   "DCC Timeout",
   "DCC CHAT Failed",
   "DCC RECV File Open Error",
   "DCC RECV Failed",
   "DCC RECV Complete",
   "DCC Conection Failed",
   "DCC Connected",
   "DCC SEND Failed",
   "DCC SEND Complete",
   "DCC Offer",
   "DCC Abort",
   "DCC Offer Not Valid",
   "DCC CHAT Reoffer",
   "DCC CHAT Offering",
   "DCC DRAW Offer",
   "DCC CHAT Offer",
   "DCC RESUME Request",
   "DCC SEND Offer",
   "DCC Generic Offer",
   "Notify Online",
   "Notify Number",
   "Notify Empty",
   "No Running Process",
   "Process Already Running",
   "Server Lookup",
   "Server Connected",
   "Server Error",
   "Server Generic Message",
   "Found IP",
   "DCC Rename"
};

char *pevt_join_help[] = {
   "The nick of the joining person",
   "The channel being joined",
   "The host of the person",
   NULL
};

char *pevt_chanaction_help[] = {
   "Nickname (which may contain color)",
   "The action",
   NULL
};

char *pevt_chanmsg_help[] = {
   "Nickname (which may contain color)",
   "The text",
   NULL
};

char *pevt_privmsg_help[] = {
   "Nickname",
   "The message",
   NULL
};

char *pevt_changenick_help[] = {
   "Old nickname",
   "New nickname",
   NULL
};

char *pevt_newtopic_help[] = {
   "Nick of person who changed the topic",
   "Topic",
   "Channel",
   NULL
};

char *pevt_topic_help[] = {
   "Channel",
   "Topic",
   NULL
};

char *pevt_kick_help[] = {
   "The nickname of the kicker",
   "The person being kicked",
   "The channel",
   "The reason",
   NULL
};

char *pevt_part_help[] = {
   "The nick of the person leaving",
   "The host of the person",
   "The channel",
   NULL
};

char *pevt_chandate_help[] = {
   "The channel",
   "The time",
   NULL
};

char *pevt_topicdate_help[] = {
   "The channel",
   "The creator",
   "The time",
   NULL
};

char *pevt_quit_help[] = {
   "Nick",
   "Reason",
   NULL
};

char *pevt_pingrep_help[] = {
   "Who it's from",
   "The time in x.x format (see below)",
   NULL
};

char *pevt_notice_help[] = {
   "Who it's from",
   "The message",
   NULL
};

char *pevt_ujoin_help[] = {
   "The nick of the joining person",
   "The channel being joined",
   "The host of the person",
   NULL
};

char *pevt_uchanmsg_help[] = {
   "Nickname (which may contain color)",
   "The text",
   NULL
};

char *pevt_dprivmsg_help[] = {
   "Nickname",
   "The message",
   NULL
};

char *pevt_uchangenick_help[] = {
   "Old nickname",
   "New nickname",
   NULL
};

char *pevt_ukick_help[] = {
   "The nickname of the kicker",
   "The person being kicked",
   "The channel",
   "The reason",
   NULL
};

char *pevt_upart_help[] = {
   "The nick of the person leaving",
   "The host of the person",
   "The channel",
   NULL
};

char *pevt_ctcpsnd_help[] = {
   "The sound",
   "The nick of the person",
   NULL
};

char *pevt_ctcpgen_help[] = {
   "The CTCP event",
   "The nick of the person",
   NULL
};

char *pevt_ctcpgenc_help[] = {
   "The CTCP event",
   "The nick of the person",
   "The Channel it's going to",
   NULL
};

char *pevt_chansetkey_help[] = {
   "The nick of the person who set the key",
   "The key",
   NULL
};

char *pevt_chansetlimit_help[] = {
   "The nick of the person who set the limit",
   "The limit",
   NULL
};

char *pevt_chanop_help[] = {
   "The nick of the person who has been op'ed",
   "The nick of the person of did the op'ing",
   NULL
};

char *pevt_chanvoice_help[] = {
   "The nick of the person who has been voice'ed",
   "The nick of the person of did the voice'ing",
   NULL
};

char *pevt_chanban_help[] = {
   "The nick of the person who did the banning",
   "The ban mask",
   NULL
};

char *pevt_chanrmkey_help[] = {
   "The nick who removed the key",
   NULL
};

char *pevt_chanrmlimit_help[] = {
   "The nick who removed the limit",
   NULL
};

char *pevt_chandeop_help[] = {
   "The nick of the person of did the deop'ing",
   "The nick of the person who has been deop'ed",
   NULL
};

char *pevt_chandevoice_help[] = {
   "The nick of the person of did the devoice'ing",
   "The nick of the person who has been devoice'ed",
   NULL
};

char *pevt_chanunban_help[] = {
   "The nick of the person of did the unban'ing",
   "The ban mask",
   NULL
};

char *pevt_chanmodegen_help[] = {
   "The nick of the person setting the mode",
   "The mode's sign (+/-)",
   "The mode letter",
   "The channel it's being set on",
   NULL
};

char *pevt_whois1_help[] = {
   "Nickname",
   "Username",
   "Host",
   "Full name",
   NULL
};

char *pevt_whois2_help[] = {
   "Nickname",
   "Channel Membership/\"is an IRC operator\"",
   NULL
};

char *pevt_whois3_help[] = {
   "Nickname",
   "Server Information",
   NULL
};

char *pevt_whois4_help[] = {
   "Nickname",
   "Idle time",
   NULL
};

char *pevt_whois4t_help[] = {
   "Nickname",
   "Idle time",
   "Signon time",
   NULL
};

char *pevt_whois5_help[] = {
   "Nickname",
   "Away reason",
   NULL
};

char *pevt_whois6_help[] = {
   "Nickname",
   NULL
};

char *pevt_generic_channel_help[] = {
   "Channel Name",
   NULL
};

char *pevt_generic_none_help[] = {
   NULL
};

char *pevt_servertext_help[] = {
   "Text",
   NULL
};

char *pevt_invited_help[] = {
   "Channel Name",
   "Nick of person who invited you",
   NULL
};

char *pevt_usersonchan_help[] = {
   "Channel Name",
   "Users",
   NULL
};

char *pevt_nickclash_help[] = {
   "Nickname in use",
   "Nick being tried",
   NULL
};

char *pevt_connfail_help[] = {
   "Error String",
   NULL
};

char *pevt_connect_help[] = {
   "Host",
   "IP",
   "Port",
   NULL
};

char *pevt_sconnect_help[] = {
   "PID",
   NULL
};

char *pevt_generic_nick_help[] = {
   "Nickname",
   NULL
};

char *pevt_wintype_help[] = {
   "Type of window",
   NULL
};

char *pevt_chanmodes_help[] = {
   "Channel name",
   "Modes string",
   NULL
};

char *pevt_rawmodes_help[] = {
   "Nickname",
   "Modes string",
   NULL
};

char *pevt_kill_help[] = {
   "Nickname",
   "Reason",
   NULL
};

char *pevt_dccstall_help[] = {
   "DCC Type",
   "Filename",
   "Nickname",
   NULL
};

char *pevt_generic_file_help[] = {
   "Filename",
   NULL
};

char *pevt_dccrecverr_help[] = {
   "Filename",
   "Destination filename",
   "Nickname",
   NULL
};

char *pevt_dccrecvcomp_help[] = {
   "Filename",
   "Destination filename",
   "Nickname",
   "CPS",
   NULL
};

char *pevt_dccconfail_help[] = {
   "DCC Type",
   "Nickname",
   "Error string",
   NULL
};

char *pevt_dcccon_help[] = {
   "DCC Type",
   "Nickname",
   "IP address",
   "{to|from}",
   NULL
};

char *pevt_dccsendfail_help[] = {
   "Filename",
   "Nickname",
   NULL
};

char *pevt_dccsendcomp_help[] = {
   "Filename",
   "Nickname",
   "CPS",
   NULL
};

char *pevt_dccoffer_help[] = {
   "Filename",
   "Nickname",
   NULL
};

char *pevt_dccabort_help[] = {
   "DCC type",
   "Filename",
   "Nickname",
   NULL
};

char *pevt_dccresumeoffer_help[] = {
   "Nickname",
   "Filename",
   "Position",
   NULL
};

char *pevt_dccsendoffer_help[] = {
   "Nickname",
   "Filename",
   "Size",
   NULL
};

char *pevt_dccgenericoffer_help[] = {
   "DCC String",
   "Nickname",
   NULL
};

char *pevt_notifynumber_help[] = {
   "Number of notify items",
   NULL
};

char *pevt_serverlookup_help[] = {
   "Servername",
   NULL
};

char *pevt_servererror_help[] = {
   "Text",
   NULL
};

char *pevt_servergenmessage_help[] = {
   "Text",
   NULL
};

char *pevt_foundip_help[] = {
   "IP",
   NULL
};

char *pevt_dccrename_help[] = {
   "Old Filename",
   "New Filename",
   NULL
};

/* This is an indexed list of char ***
   you access the char * by evthelp[event][arg]
*/

char **evthelp[] = {
   pevt_join_help,
   pevt_chanaction_help,
   pevt_chanmsg_help,
   pevt_privmsg_help,
   pevt_changenick_help,
   pevt_newtopic_help,
   pevt_topic_help,
   pevt_kick_help,
   pevt_part_help,
   pevt_chandate_help,
   pevt_topicdate_help,
   pevt_quit_help,
   pevt_pingrep_help,
   pevt_notice_help,
   pevt_ujoin_help,
   pevt_uchanmsg_help,
   pevt_dprivmsg_help,
   pevt_uchangenick_help,
   pevt_ukick_help,
   pevt_upart_help,
   pevt_ctcpsnd_help,
   pevt_ctcpgen_help,
   pevt_ctcpgenc_help,
   pevt_chansetkey_help,
   pevt_chansetlimit_help,
   pevt_chanop_help,
   pevt_chanvoice_help,
   pevt_chanban_help,
   pevt_chanrmkey_help,
   pevt_chanrmlimit_help,
   pevt_chandeop_help,
   pevt_chandevoice_help,
   pevt_chanunban_help,
   pevt_chanmodegen_help,
   pevt_whois1_help,
   pevt_whois2_help,
   pevt_whois3_help,
   pevt_whois4_help,
   pevt_whois4t_help,
   pevt_whois5_help,
   pevt_whois6_help,
   pevt_generic_channel_help,
   pevt_generic_channel_help,
   pevt_generic_channel_help,
   pevt_generic_channel_help,
   pevt_generic_none_help,
   pevt_servertext_help,
   pevt_invited_help,
   pevt_usersonchan_help,
   pevt_nickclash_help,
   pevt_generic_none_help,
   pevt_generic_none_help,
   pevt_connfail_help,
   pevt_connect_help,
   pevt_generic_none_help,
   pevt_sconnect_help,
   pevt_generic_none_help,
   pevt_generic_none_help,
   pevt_generic_nick_help,
   pevt_generic_nick_help,
   pevt_wintype_help,
   pevt_chanmodes_help,
   pevt_rawmodes_help,
   pevt_kill_help,
   pevt_dccstall_help,
   pevt_dccstall_help,
   pevt_generic_nick_help,
   pevt_generic_file_help,
   pevt_dccrecverr_help,
   pevt_dccrecvcomp_help,
   pevt_dccconfail_help,
   pevt_dcccon_help,
   pevt_dccsendfail_help,
   pevt_dccsendcomp_help,
   pevt_dccoffer_help,
   pevt_dccabort_help,
   pevt_generic_none_help,
   pevt_generic_nick_help,
   pevt_generic_nick_help,
   pevt_generic_nick_help,
   pevt_generic_nick_help,
   pevt_dccresumeoffer_help,
   pevt_dccsendoffer_help,
   pevt_dccgenericoffer_help,
   pevt_generic_nick_help,
   pevt_notifynumber_help,
   pevt_generic_none_help,
   pevt_generic_none_help,
   pevt_generic_none_help,
   pevt_serverlookup_help,
   pevt_generic_none_help,
   pevt_servererror_help,
   pevt_servergenmessage_help,
   pevt_foundip_help,
   pevt_dccrename_help
};

char *pevt_defaults[] = {
   "%C4*%C *%C4*%C %C11 %B$1%B %C14(%C10$3%C14)%C  has joined $2",
   "%C13*%C  $1 $2",
   "%C2<%C $1%C2> %C $2",
   "%C12*%C13$1%C12* %C $2 %C ",
   "%C4*%C *%C4*%C  $1 is now known as $2",
   "%C4*%C *%C4*%C  $1 has changed the topic to: $2%C ",
   "%C4*%C *%C4*%C  Topic for %C11$1%C  is %C11$2%C ",
   "%C4*%C *%C4*%C  $1 has kicked $2 from $3 ($4)%C ",
   "%C4*%C *%C4*%C  $1 %C14(%C $2%C14)%C  has left $3",
   "%C4*%C *%C4*%C  Channel $1 created on $2",
   "%C4*%C *%C4*%C  Topic for %C11$1%C  set by %C11$2%C  %C11$3%C ",
   "%C4*%C *%C4*%C  $1 has quit %C14(%C $2%C14)%C ",
   "%C4*%C *%C4*%C  Ping reply from $1 : $2 second(s)",
   "%C12-%C13$1%C12- %C $2",
   "%C4*%C *%C4*%C %C11 %B$1%B %C14(%C10$3%C14)%C  has joined $2",
   "%C6<%C $1%C6> %C $2",
   "%C2<%C $1%C2> %C $2 %C ",
   "%C4*%C *%C4*%C  You are now known as $2",
   "%C4*%C *%C4*%C  You have been kicked from $3 by $2 ($4)%C ",
   "%C4*%C *%C4*%C  You have left channel $3",
   "%C4*%C *%C4*%C  Received a CTCP Sound $1 from $2",
   "%C4*%C *%C4*%C  Received a CTCP $1 from $2",
   "%C4*%C *%C4*%C  Received a CTCP $1 from $2 (to $3)", 
   "%C4*%C *%C4*%C  $1 sets channel keyword to $2",
   "%C4*%C *%C4*%C  $1 sets channel limit to $2",
   "%C4*%C *%C4*%C  $1 gives channel operator status to $2",
   "%C4*%C *%C4*%C  $1 gives voice to $2",
   "%C4*%C *%C4*%C  $1 sets ban on $2",
   "%C4*%C *%C4*%C  $1 removes channel keyword",
   "%C4*%C *%C4*%C  $1 removes user limit",
   "%C4*%C *%C4*%C  $1 removes channel operator status from $2",
   "%C4*%C *%C4*%C  $1 removes voice from $2",
   "%C4*%C *%C4*%C  $1 removes ban on $2",
   "%C4*%C *%C4*%C  $1 sets mode $2$3 $4",
   "%C4*%C *%C4*%C  %C12[%C $1%C12] %C14(%C $2@$3%C14) %C : $4",
   "%C4*%C *%C4*%C  %C12[%C $1%C12] %C $2",
   "%C4*%C *%C4*%C  %C12[%C $1%C12] %C $2",
   "%C4*%C *%C4*%C  %C12[%C $1%C12] %C idle %C11$2%C ",
   "%C4*%C *%C4*%C  %C12[%C $1%C12] %C idle %C11$2%C , signon: %C11$3%C ",
   "%C4*%C *%C4*%C  %C12[%C $1%C12] %C is away %C14(%C $2%C14)",
   "%C4*%C *%C4*%C  %C12[%C $1%C12] %C End of WHOIS list.",
   "%C4*%C *%C4*%C  Cannot join%C11 %B$1 %C (User limit reached).",
   "%C4*%C *%C4*%C  Cannot join%C11 %B$1 %C (You are banned).",
   "%C4*%C *%C4*%C  Cannot join%C11 %B$1 %C (Channel is invite only).",
   "%C4*%C *%C4*%C  Cannot join%C11 %B$1 %C (Requires keyword).\n",
   "%C4*%C *%C4*%C  MOTD Skipped.",
   "%C4*%C *%C4*%C  $1",
   "%C4*%C *%C4*%C  You have been invited to %C11$1 %C by %C11$2",
   "%C4*%C *%C4*%C  %C11Users on $1: %C $2",
   "%C4*%C *%C4*%C  $1 already in use. Retrying with $2..",
   "%C4*%C *%C4*%C  Nickname already in use. Use /NICK to try another.",
   "%C3*%C *%C3*%C  Unknown host. Maybe you misspelled it?",
   "%C3*%C *%C3*%C  Connection failed. Error: $1",
   "%C3*%C *%C3*%C  Connecting to $1 ($2) port $3..",
   "%C3*%C *%C3*%C  Connected. Now logging in..",
   "%C3*%C *%C3*%C  Stopped previous connection attempt (pid=$1)",
   "%C3*%C *%C3*%C  Disconnected.",
   "%C3*%C *%C3*%C  No such DCC.",
   "%C3*%C *%C3*%C  $1 deleted from notify list.",
   "%C3*%C *%C3*%C  $1 added to notify list.",
   "%C3*%C *%C3*%C  You can't do that in a $1 window.",
   "%C3*%C *%C3*%C  Channel $1 modes: $2",
   "%C3*%C *%C3*%C  $1 sets modes%B %C14[ %C $2%B%C14]",
   "%C3*%C *%C3*%C  You have been killed by $1 $2",
   "%C4*%C *%C4*%C  DCC $1 %C11$2 %C to %C11$3 %C stalled - aborting.",
   "%C4*%C *%C4*%C  DCC $1 %C11$2 %C to %C11$3 %C timeed out - aborting.",
   "%C4*%C *%C4*%C  DCC CHAT failed. Connection to $1 lost.",
   "%C4*%C *%C4*%C  DCC RECV: Cannot open $1 for writing - aborting.",
   "%C4*%C *%C4*%C  DCC RECV $1 ($2) failed. Connection to $3 lost.",
   "%C4*%C *%C4*%C  DCC RECV $1 ($2) from $3 complete ($4 cps).",
   "%C4*%C *%C4*%C  DCC $1 connect attempt to $2 failed (err=$3).",
   "%C4*%C *%C4*%C  DCC $1 connection established $4 $2 ($3)",
   "%C4*%C *%C4*%C  DCC SEND $1 failed. Connection to $2 lost.",
   "%C4*%C *%C4*%C  DCC SEND $1 to $2 complete ($3 cps).",
   "%C4*%C *%C4*%C  Offering %C11$1 %C to %C11$2%C ",
   "%C4*%C *%C4*%C  DCC $1 %C11$2 %C to %C11$3 %C aborted.",
   "%C4*%C *%C4*%C  No such DCC offer.",
   "%C4*%C *%C4*%C  Already offering CHAT to $1",
   "%C4*%C *%C4*%C  Offering DCC CHAT to $1",
   "%C4*%C *%C4*%C  Received a DCC DRAW offer from $1. Type %C11DCC DRAW $1 %C to accept",
   "%C4*%C *%C4*%C  Received a DCC CHAT offer from $1",
   "%C4*%C *%C4*%C  %C11$1 %C has requested to resume %C11$2 %C from %C11$3%C .",
   "%C4*%C *%C4*%C  %C11$1 %C has offered %C11$2 %C (%C11$3 %C bytes)",
   "%C4*%C *%C4*%C  Received '$1' from $2",
   "%C4*%C *%C4*%C  Notify: $1 is online.",
   "%C4*%C *%C4*%C  $1 users in notify list.",
   "%C4*%C *%C4*%C  Notify list is empty.",
   "%C4*%C *%C4*%C  No process is currently running",
   "%C4*%C *%C4*%C  A process is already running",
   "%C3*%C *%C3*%C  Looking up $1..",
   "%C3*%C *%C3*%C  Connected.",
   "%C4*%C *%C4*%C  $1",
   "%C4*%C *%C4*%C  $1",
   "%C4*%C *%C4*%C  Found your IP: [$1]",
   "%C4*%C *%C4*%C  The file %C11$1%C  already exists, it has been renamed to %C11$2%C ."
};

char signal_2_te[] = {
   -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
   21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38,
   39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56,
   57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74,
   75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92,
   93, 94
};

char te_num_args[] = {
   3, 2, 2, 2, 2, 3, 2, 4, 3, 2, 3, 2, 2, 2, 3, 2, 2, 2, 4, 3, 2, 2, 3, 2, 2,
   2, 2, 2, 1, 1, 2, 2, 2, 4, 4, 2, 2, 2, 3, 2, 1, 1, 1, 1, 1, 0, 1, 2, 2, 2,
   0, 0, 1, 3, 0, 1, 0, 0, 1, 1, 1, 2, 2, 2, 3, 3, 1, 1, 3, 4, 3, 4, 2, 3, 2,
   3, 0, 1, 1, 1, 1, 3, 3, 2, 1, 1, 0, 0, 0, 1, 0, 1, 1, 1, 2
};

char te_2_signal[] = {
   17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34,
   35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52,
   53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70,
   71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88,
   89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103, 104,
   105, 106, 107, 108, 109, 110, 111
};

void
pevent_load_defaults()
{
   int i, len;

   for (i = 0; i < TE_MAX_VAL + 1; i++) {
      len = strlen(pevt_defaults[i]);
      len++;
      if (pntevts_text[i])
	 free(pntevts_text[i]);
      pntevts_text[i] = malloc(len);
      memcpy(pntevts_text[i], pevt_defaults[i], len);
   }
}

void
pevent_make_pntevts()
{
   int i, m, len;
   char out[1024];

   for (i = 0; i < TE_MAX_VAL + 1; i++) {
      if (pntevts[i] != NULL)
	 free(pntevts[i]);
      if (pevt_build_string(pntevts_text[i], &(pntevts[i]), &m) != 0) {
	 snprintf(out, sizeof(out), "Error parsing event %s.\nLoading default", evtnames[i]);
	 gtkutil_simpledialog(out);
	 free(pntevts_text[i]);
	 len = strlen(pevt_defaults[i]) + 1;
	 pntevts_text[i] = malloc (len);
	 memcpy(pntevts_text[i], pevt_defaults[i], len);
	 if (pevt_build_string(pntevts_text[i], &(pntevts[i]), &m) != 0) {
	    fprintf(stderr, "XChat CRITICAL *** default event text failed to build!\n");
	    abort();
	 }
      }
      check_special_chars(pntevts[i]);
   }
}

int
pevent_load (char *filename)
{
   char *buf, *ibuf;
   int fd, i = 0, n, len, err = 0, pnt = 0;
   struct stat st;

   buf = malloc(1000);
   if (filename == NULL)
      snprintf(buf, 1000, "%s/printevents.conf", get_xdir());
   else
      strcpy(buf, filename);
   
   fd = open(buf, O_RDONLY);
   free(buf);
   if (fd < 0)
      return 1;
   if (fstat(fd, &st) != 0)
      return 1;
   ibuf = malloc(st.st_size);
   read(fd, ibuf, st.st_size);
   close(fd);
   
   while(buf_get_line(ibuf, &buf, &pnt, st.st_size)) {
      if (buf[0] == '#')
	 continue;
      if (strlen(buf) == 0)
	 continue;
      if (strcmp(buf, evtnames[i]) != 0) {
	 /* Sigh - it's out of order */
	 for (n = 0; n < TE_MAX_VAL; n++) {
	    if (strcmp(buf, evtnames[n]) == 0) {
	       i = n;
	       break;
	    }
	 }
	 if (n == TE_MAX_VAL) {
	    err++;
	    gtkutil_simpledialog("Config file contained data for unknown event");
	    if (strstr(buf, "$1")) {
	       gtkutil_simpledialog("The config file seems to be off by one,\nattempting to correct\n");
	       continue;
	    }
	    if (err == 3) {
	       gtkutil_simpledialog("Too many errors, aborting load");
	       free(ibuf);
	       return 1;
	    }
	    buf_get_line(ibuf, &buf, &pnt, st.st_size);
	    continue;
	 }
      }
      if (!buf_get_line(ibuf, &buf, &pnt, st.st_size)) {
	 gtkutil_simpledialog("The Print Events config file is corrupted, aborting.\nThe Print Events may be corrupted");
	 free(ibuf);
	 return 1;
      }
      err = 0;
      len = strlen(buf);
      len++;
      if (pntevts_text[i])
	 free(pntevts_text[i]);
      pntevts_text[i] = malloc(len);
      memcpy(pntevts_text[i], buf, len);
      i++;
      if (i > TE_MAX_VAL) {
	 break;
      }
   }
   
   free(ibuf);
   return 0;
}

void
pevent_check_all_loaded ()
{
   int i, len;
   char out[1024];

   for (i = 0; i < TE_MAX_VAL + 1; i++) {
      if (pntevts_text[i] == NULL) {
	 snprintf(out, sizeof(out), "The data for event %s failed to load. Reverting to defaults.\nThis may be because a new version of XChat is loading an old config file.\n\nCheck all print event texts are correct", evtnames[i]);
	 gtkutil_simpledialog(out);
	 len = strlen(pevt_defaults[i]) + 1;
	 pntevts_text[i] = malloc (len);
	 memcpy(pntevts_text[i], pevt_defaults[i], len);
      }
   }
}

void
load_text_events ()
{
   /* I don't free these as the only time when we don't need them
      is once XChat has exit(2)'ed, so why bother ?? --AGL */
   
   pntevts_text = malloc (sizeof(char *) * (TE_MAX_VAL + 1));
   memset(pntevts_text, 0, sizeof(char *) * (TE_MAX_VAL + 1));
   pntevts = malloc (sizeof(char *) * (TE_MAX_VAL + 1));
   memset(pntevts, 0, sizeof(char *) * (TE_MAX_VAL + 1));

   if (pevent_load(NULL))
      pevent_load_defaults();
   pevent_check_all_loaded();
   pevent_make_pntevts();
}

void
display_event(char *i, struct session *sess, int numargs, char **args)
{
   int len, oi, ii, *ar;
   char o[4096], d, a, done_all = FALSE;

   oi = ii = len = d = a = 0;

   if (i == NULL)
      abort();
   
   while (done_all == FALSE) {
      d = i[ii++];
      switch (d) {
      case 0:
	 *(&len) = (int) (*(i + ii));
	 ii += sizeof(int);
	 if (oi + len > sizeof(o)) {
	    printf("Overflow in display_event (%s)\n", i);
	    return;
	 }
	 memcpy(&(o[oi]), &(i[ii]), len);
	 oi += len;
	 ii += len;
	 break;
      case 1:
	 a = i[ii++];
	 if (a > numargs) {
	    printf("XChat DEBUG: display_event: arg > numargs (%d %d %s), ABORTING\n", a, numargs, i);
	    abort();
	 }
	 ar = (int *) args[(int)a];
	 if (ar == NULL) {
	    printf("Error args[a] == NULL in display_event\n");
	    abort();
	 }
	 len = strlen((char *) ar);
	 memcpy(&o[oi], ar, len);
	 oi += len;
	 break;
      case 2:
	 o[oi++] = '\n';
	 o[oi++] = 0;
	 done_all = TRUE;
	 continue;
      }
   }
   o[oi] = 0;
   PrintText(sess, o);
}

int
pevt_build_string(char *input, char **output, int *max_arg)
{
   struct pevt_stage1 *s = NULL, *base = NULL, *last = NULL, *next;
   int clen;
   char o[4096], d, *obuf, *i;
   int oi, ii, ti, max = -1, len = strlen(i), x;

   len = strlen(input);
   i = malloc (len + 1);
   memcpy(i, input, len + 1);
   check_special_chars(i);

   len = strlen(i);
   
   clen = oi = ii = ti = 0;

   for (;;) {
      if (ii == len)
	 break;
      d = i[ii++];
      if (d != '$') {
	 o[oi++] = d;
	 continue;
      }
      if (i[ii] == '$') {
	 o[oi++] = '$';
	 continue;
      }
      if (oi > 0) {
	 s = (struct pevt_stage1 *) malloc (sizeof(struct pevt_stage1));
	 if (base == NULL)
	    base = s;
	 if (last != NULL)
	    last->next = s;
	 last = s;
	 s->next = NULL;
	 s->data = malloc (oi + sizeof(int) + 1);
	 s->len = oi + sizeof(int) + 1;
	 clen += oi + sizeof(int) + 1;
	 s->data[0] = 0;
	 memcpy(&(s->data[1]), &oi, sizeof(int));
	 memcpy(&(s->data[1 + sizeof(int)]), o, oi);
	 oi = 0;
      }
      if (ii == len) {
	 gtkutil_simpledialog("String ends with a $");
	 return 1;
      }
      d = i[ii++];
      if (d == 'a') { /* Hex value */
	 x = 0;
	 if (ii == len)
	    goto a_len_error;
	 d = i[ii++];
	 d -= '0';
	 x = d * 100;
	 if (ii == len)
	    goto a_len_error;
	 d = i[ii++];
	 d -= '0';
	 x += d * 10;
	 if (ii == len)
	    goto a_len_error;
	 d = i[ii++];
	 d -= '0';
	 x += d;
	 if (x > 255)
	    goto a_range_error;
	 o[oi++] = x;
	 continue;
	 
      a_len_error:
	 gtkutil_simpledialog("String ends in $a");
	 return 1;
      a_range_error:
	 gtkutil_simpledialog("$a value is greater then 255");
	 return 1;
      }
      if (d < '0' || d > '9') {
	 snprintf(o, sizeof(o), "Error, invalid argument $%c\n", d);
	 gtkutil_simpledialog(o);
	 return 1;
      }
      d -= '0';
      if (max < d)
	 max = d;
      s = (struct pevt_stage1 *) malloc (sizeof(struct pevt_stage1));
      if (base == NULL)
	 base = s;
      if (last != NULL)
	 last->next = s;
      last = s;
      s->next = NULL;
      s->data = malloc (2);
      s->len = 2;
      clen += 2;
      s->data[0] = 1;
      s->data[1] = d - 1;
   }
   if (oi > 0) {
      s = (struct pevt_stage1 *) malloc (sizeof(struct pevt_stage1));
      if (base == NULL)
	 base = s;
      if (last != NULL)
	 last->next = s;
      last = s;
      s->next = NULL;
      s->data = malloc (oi + sizeof(int) + 1);
      s->len = oi + sizeof(int) + 1;
      clen += oi + sizeof(int) + 1;
      s->data[0] = 0;
      memcpy(&(s->data[1]), &oi, sizeof(int));
      memcpy(&(s->data[1 + sizeof(int)]), o, oi);
      oi = 0;
   }
   s = (struct pevt_stage1 *) malloc (sizeof(struct pevt_stage1));
   if (base == NULL)
      base = s;
   if (last != NULL)
      last->next = s;
   last = s;
   s->next = NULL;
   s->data = malloc(1);
   s->len = 1;
   clen += 1;
   s->data[0] = 2;
   
   oi = 0;
   s = base;
   obuf = malloc (clen);
   while (s) {
      next = s->next;
      memcpy(&obuf[oi], s->data, s->len);
      oi += s->len;
      free(s->data);
      free(s);
      s = next;
   }

   free(i);
   
   if (max_arg)
      *max_arg = max;
   if (output)
      *output = obuf;
   return 0;
}

int
textsignal_handler (struct session *sess, void *b, void *c,
			 void *d, void *e, char f)
{
   /* This handler *MUST* always be the last in the chain
      because it doesn't call the next handler
   */

   char te;
   char *args[8];
   int numargs, i;

   te = signal_2_te[current_signal];
   if (te == -1) {
      printf("error, textsignal_handler passed non TE signal (%d)\n", current_signal);
      abort();
   }
   numargs = te_num_args[ (int) te];
   i = 0;

   args[0] = b;
   args[1] = c;
   args[2] = d;
   args[3] = e;

   display_event(pntevts[ (int) te], sess, te_num_args[ (int) te], args);
   return 0;
}

void
printevent_setup ()
{
   int evt = 0, sign = 0;
   struct xp_signal *sig;

   sig = (struct xp_signal *) malloc(sizeof(struct xp_signal));

   sig->signal = -1;
   sig->naddr = NULL;
   sig->callback = XP_CALLBACK(textsignal_handler);
   sig->last = sig->next = NULL;
#ifdef USE_PLUGIN
   sig->mod = NULL;
#endif

   for (evt = 0; evt < TE_MAX_VAL + 1; evt++) {
      sign = te_2_signal[evt];
      sigroots[sign] = sig;
      sighandler[sign] = XP_CALLBACK(textsignal_handler);
   }
}

void
pevent_dialog_save (char *fn)
{
   int fd, i;
   char buf[512];

   if (!fn)
      snprintf(buf, 510, "%s/printevents.conf", get_xdir());
   else
      snprintf(buf, 510, "%s", fn);
   fd = open(buf, O_CREAT | O_TRUNC | O_WRONLY, 0x180);
   if (fd < 0) {
      gtkutil_simpledialog("Error opening config file\n");
      return;
   }

   write(fd, buf, snprintf(buf, 510, "# XChat text events config file\n\n"));
   
   for (i = 0; i < TE_MAX_VAL + 1; i++) {
      write(fd, buf, snprintf(buf, 510, "%s\n", evtnames[i]));
      write(fd, buf, snprintf(buf, 510, "%s\n", pntevts_text[i]));
   }
   close(fd);
}

static void
pevent_dialog_close (GtkWidget *wid, gpointer *arg)
{
   pevent_dialog = NULL;
   pevent_dialog_save(NULL);
}

static void
pevent_dialog_update (GtkWidget *wid, GtkWidget *twid)
{
   int  row, len, m;
   char *text, *out;

   row = gtkutil_clist_selection(pevent_dialog_list);
   if (row == -1)
      return;

   text = gtk_entry_get_text(GTK_ENTRY(wid));
   len = strlen(text);

   if (pevt_build_string(text, &out, &m) != 0) {
      gtkutil_simpledialog("There was an error parsing the string");
      return;
   }
   if (m > te_num_args[row]) {
      free(out);
      out = malloc(4096);
      snprintf(out, 4096, "This signal is only passed %d args, $%d is invalid", te_num_args[row], m);
      gtkutil_simpledialog(out);
      free(out);
      return;
   }
   
   gtk_clist_set_text(GTK_CLIST(pevent_dialog_list), row, 1, text);

   if (pntevts_text[row])
      free(pntevts_text[row]);
   if (pntevts[row])
      free(pntevts[row]);

   pntevts_text[row] = malloc(len + 1);
   memcpy(pntevts_text[row], text, len + 1);
   pntevts[row] = out;

   out = malloc (len + 2);
   memcpy(out, text, len + 1);
   out[len] = '\n';
   out[len + 1] = 0;
   check_special_chars(out);
   
   PrintTextRaw(twid, out, prefs.bg_color, prefs.fg_color,
		font_normal, font_bold);
   free(out);
}

static void
pevent_dialog_unselect(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even, gpointer none)
{
   gtk_entry_set_text(GTK_ENTRY(pevent_dialog_entry), "");
   gtk_clist_clear(GTK_CLIST(pevent_dialog_hlist));
}

static void
pevent_dialog_hfill(GtkWidget *list, int e)
{
   gchar *new[2];
   int i = 0;
   char *text, buf[64];

   text = evthelp[e][i];

   gtk_clist_clear(GTK_CLIST(list));
   while(text) {
      snprintf(buf, sizeof(buf), "%d", i + 1);
      if (text[0] == '\001')
	 text++;
      new[0] = buf;
      new[1] = text;
      gtk_clist_append(GTK_CLIST(list), new);

      i++;
      text = evthelp[e][i];
   }
}

static void
pevent_dialog_select(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even, gpointer none)
{
   char *cmd;   

   row = gtkutil_clist_selection(pevent_dialog_list);
   if(row != -1)
   {
      gtk_clist_get_text(GTK_CLIST(clist), row, 1, &cmd);
      gtk_entry_set_text(GTK_ENTRY(pevent_dialog_entry), cmd);
      pevent_dialog_hfill(pevent_dialog_hlist, row);
   } else {
      pevent_dialog_unselect(0, 0, 0, 0, 0);
   }
}

static void
pevent_dialog_fill (GtkWidget *list)
{
   int i;
   gchar *new[2];

   gtk_clist_clear(GTK_CLIST(list));
   
   for (i = 0; i < TE_MAX_VAL + 1; i++) {
      new[0] = evtnames[i];
      new[1] = pntevts_text[i];
      gtk_clist_append (GTK_CLIST(list), new);
   }
}

static void
pevent_save_req_cb (void *arg1, void *arg2, char *file)
{
   pevent_dialog_save(file);
   free(file);
}

static void
pevent_save_cb (GtkWidget *wid, void *data)
{
   if (data) {
      gtkutil_file_req("Print Texts File", pevent_save_req_cb, NULL, NULL,
		       TRUE);
      return;
   }
   pevent_dialog_save(NULL);
}

static void
pevent_load_req_cb (void *arg1, void *arg2, char *file)
{
   pevent_load(file);
   pevent_dialog_fill(pevent_dialog_list);
   pevent_dialog_select(pevent_dialog_list, -1, -1, NULL, NULL);
   free(file); /* Opps, it was my code anyhow ;( --AGL */
}

static void
pevent_load_cb (GtkWidget *wid, void *data)
{
   gtkutil_file_req("Print Texts File", pevent_load_req_cb, NULL, NULL, FALSE);
}

static void
pevent_ok_cb (GtkWidget *wid, void *data)
{
   gtk_widget_destroy(pevent_dialog);
}

static void
pevent_test_cb (GtkWidget *wid, GtkWidget *twid)
{
   int len, n;
   char *out, *text;

   for (n = 0; n < TE_MAX_VAL + 1; n++) {
      text = pntevts_text[n];
      len = strlen(text);
   
      out = malloc (len + 2);
      memcpy(out, text, len + 1);
      out[len] = '\n';
      out[len + 1] = 0;
      check_special_chars(out);
   
      PrintTextRaw(twid, out, prefs.bg_color, prefs.fg_color,
		   font_normal, font_bold);
      free(out);
   }
}

void
pevent_dialog_show ()
{
   GtkWidget *vbox, *vbox2, *hbox, *tbox, *wid, *bh, *th;
   gchar *titles[] = {"Event", "Text"};
   gchar *help_titles[] = {"$ Number", "Description"};

   if (pevent_dialog) return;
   
   pevent_dialog = gtkutil_window_new ("Edit Events", "Edit Events", 600, 500,
				      pevent_dialog_close, 0);
   gtk_window_set_policy(GTK_WINDOW(pevent_dialog), TRUE, TRUE, FALSE);

   vbox2 = gtk_vbox_new(0, 2);
   vbox = gtk_vbox_new(0, 2);
   gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
   gtk_container_add(GTK_CONTAINER(pevent_dialog), vbox);
   gtk_widget_show(vbox);

   wid = gtk_vpaned_new();
   th = gtk_vbox_new(0, 2);
   bh = gtk_vbox_new(0, 2);
   gtk_widget_show(th);
   gtk_widget_show(bh);
   gtk_paned_pack1(GTK_PANED(wid), th, 1, 1);
   gtk_paned_pack2(GTK_PANED(wid), bh, 0, 1);
   gtk_box_pack_start(GTK_BOX(vbox), wid, 1, 1, 0);
   gtk_widget_show(wid);
   pevent_dialog_list = gtkutil_clist_new(2, titles, th, GTK_POLICY_ALWAYS,
					 pevent_dialog_select, 0,
					 pevent_dialog_unselect, 0,
					 GTK_SELECTION_BROWSE);
   gtk_clist_set_column_width(GTK_CLIST(pevent_dialog_list), 0, 200);

   pevent_dialog_twid = gtk_text_new(0, 0);

   pevent_dialog_entry = gtk_entry_new_with_max_length(255);
   gtk_widget_set_usize(pevent_dialog_entry, 96, 0);


   gtk_signal_connect(GTK_OBJECT(pevent_dialog_entry), "activate",
		      GTK_SIGNAL_FUNC(pevent_dialog_update), pevent_dialog_twid);

   gtk_box_pack_start(GTK_BOX(bh), pevent_dialog_entry, 0, 0, 0);
   gtk_widget_show(pevent_dialog_entry);

   tbox = gtk_hbox_new(0, 0);
   gtk_container_add(GTK_CONTAINER(bh), tbox);
   gtk_widget_show(tbox);

   gtk_widget_set_style(pevent_dialog_twid, channelwin_style);
   gtk_widget_set_usize(pevent_dialog_twid,
			150,
			20);
   gtk_text_set_word_wrap (GTK_TEXT(pevent_dialog_twid), TRUE);
   gtk_container_add(GTK_CONTAINER(tbox), pevent_dialog_twid);
   gtk_widget_show(pevent_dialog_twid);

   wid = gtk_vscrollbar_new (GTK_TEXT(pevent_dialog_twid)->vadj);
   gtk_box_pack_start(GTK_BOX(tbox), wid, FALSE, FALSE, 0);
   show_and_unfocus(wid);

   pevent_dialog_hlist = gtkutil_clist_new(2, help_titles, bh,
					 GTK_POLICY_ALWAYS,
					 NULL, 0, NULL, 0,
					 GTK_SELECTION_BROWSE);
   gtk_clist_set_column_width(GTK_CLIST(pevent_dialog_list), 0, 65);
   gtk_widget_show(pevent_dialog_hlist);
   
   pevent_dialog_fill(pevent_dialog_list);

   hbox = gtk_hbox_new(0, 2);
   gtk_box_pack_end(GTK_BOX(vbox), hbox, 0, 0, 0);
   wid = gtk_button_new_with_label("Save");
   gtk_box_pack_end(GTK_BOX(hbox), wid, 0, 0, 0);
   gtk_signal_connect(GTK_OBJECT(wid), "clicked",
		      GTK_SIGNAL_FUNC(pevent_save_cb), NULL);
   gtk_widget_show(wid);
   wid = gtk_button_new_with_label("Save As");
   gtk_box_pack_end(GTK_BOX(hbox), wid, 0, 0, 0);
   gtk_signal_connect(GTK_OBJECT(wid), "clicked",
		      GTK_SIGNAL_FUNC(pevent_save_cb), (void *) 1);
   gtk_widget_show(wid);
   wid = gtk_button_new_with_label("Load From");
   gtk_box_pack_end(GTK_BOX(hbox), wid, 0, 0, 0);
   gtk_signal_connect(GTK_OBJECT(wid), "clicked",
		      GTK_SIGNAL_FUNC(pevent_load_cb), (void *) 0);
   gtk_widget_show(wid);
   wid = gtk_button_new_with_label("Test All");
   gtk_box_pack_end(GTK_BOX(hbox), wid, 0, 0, 0);
   gtk_signal_connect(GTK_OBJECT(wid), "clicked",
		      GTK_SIGNAL_FUNC(pevent_test_cb), pevent_dialog_twid);
   gtk_widget_show(wid);
#ifdef USE_GNOME
   wid = gnome_stock_button(GNOME_STOCK_BUTTON_OK);
#else
   wid = gtk_button_new_with_label("Ok");
#endif
   gtk_box_pack_start(GTK_BOX(hbox), wid, 0, 0, 0);
   gtk_signal_connect(GTK_OBJECT(wid), "clicked",
		      GTK_SIGNAL_FUNC(pevent_ok_cb), NULL);
   gtk_widget_show(wid);
   
   gtk_widget_show(hbox);
   
   gtk_widget_show(pevent_dialog);
}




