/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "voice.xpm"
#include "op.xpm"
#include "xchat.h"
#include "util.h"
#include "userlist.h"
#include "gtkutil.h"
#ifdef USE_IMLIB
#include <gdk_imlib.h>
#endif

extern GdkColor colors[];
extern GSList *sess_list;
extern struct xchatprefs prefs;
extern int notify_isnotify(struct session *sess, char *name);
extern void dcc_send(struct session *sess, char *tbuf, char *to, char *file);

GdkPixmap *op_pixmap, *voice_pixmap;
GdkBitmap *op_mask_bmp, *voice_mask_bmp;

int userlist_insertname_sorted(struct session *sess, struct user *newuser, int first);
int add_name(struct session *sess, char *name);
int sub_name(struct session *sess, char *name);


void init_userlist_xpm(struct session *sess)
{
   char **opxpm = op_xpm;
   char **voicexpm = voice_xpm;

#ifdef USE_IMLIB
   gdk_imlib_data_to_pixmap(opxpm, &op_pixmap, &op_mask_bmp);
   gdk_imlib_data_to_pixmap(voicexpm, &voice_pixmap, &voice_mask_bmp);
#else
   GtkStyle *style = gtk_widget_get_style(sess->window);
   op_pixmap = gdk_pixmap_create_from_xpm_d(sess->window->window,
					 &op_mask_bmp,
					 &style->bg[GTK_STATE_NORMAL],
					 opxpm);
   voice_pixmap = gdk_pixmap_create_from_xpm_d(sess->window->window,
					 &voice_mask_bmp,
					 &style->bg[GTK_STATE_NORMAL],
					 voicexpm);
#endif
}

void voice_myself(struct session *sess)
{
   if(sess->op_xpm) gtk_widget_destroy(sess->op_xpm);
   sess->op_xpm = gtk_pixmap_new(voice_pixmap, voice_mask_bmp);
   gtk_box_pack_start(GTK_BOX(sess->op_box), sess->op_xpm, 0, 0, 0);
   gtk_widget_show(sess->op_xpm);
}

void op_myself(struct session *sess)
{
   if(sess->op_xpm) gtk_widget_destroy(sess->op_xpm);
   sess->op_xpm = gtk_pixmap_new(op_pixmap, op_mask_bmp);
   gtk_box_pack_start(GTK_BOX(sess->op_box), sess->op_xpm, 0, 0, 0);
   gtk_widget_show(sess->op_xpm);
}

void userlist_shownumbers(struct session *sess)
{
   char tbuf[42];
   sprintf(tbuf, "%d ops, %d total", sess->ops, sess->total);
   gtk_label_set_text(GTK_LABEL(sess->namelistinfo), tbuf);
}

int nick_cmp(struct user *user1, struct user *user2)
{
   if(user1->op && !user2->op) return -1;
   if(!user1->op && user2->op) return +1;

   if(user1->voice && !user2->voice) return -1;
   if(!user1->voice && user2->voice) return +1;

   return strcasecmp(user1->user, user2->user);
}

void clear_user_list(struct session *sess)
{
   struct user *next;
   struct user *user = sess->userlist;
   gtk_clist_clear(GTK_CLIST(sess->namelistgad));
   while(user)
   {
      next = user->next;
      free(user);
      user = next;
   }
   sess->userlist = 0;
   sess->ops = 0;
   sess->total = 0;
}

struct user *find_name(struct session *sess, char *name)
{
   if(sess->namelistgad)
   {
      struct user *user = sess->userlist;
      while(user)
      {
	 if(!strcasecmp(user->user, name)) return user;
	 user = user->next;
      }
   }
   return FALSE;
}

void update_entry(struct session *sess, struct user *updateuser)
{
   gint row;
   int sel = FALSE;
   char nick[64];

   strcpy(nick, updateuser->user);
   if(updateuser->voice) sprintf(nick, "+%s", updateuser->user);
   if(updateuser->op) sprintf(nick, "@%s", updateuser->user);

   row = gtkutil_clist_selection(sess->namelistgad);
   if(row != -1)
   {
      if(gtk_clist_find_row_from_data(GTK_CLIST(sess->namelistgad),
				      (gpointer)updateuser) == row)
	sel = TRUE;
   }
   sub_name(sess, updateuser->user);
   row = add_name(sess, nick);

   if(sel)
     gtk_clist_select_row((GtkCList*)sess->namelistgad, row, 0);

   sess->ops = 0;
   sess->total = 0;
   updateuser = sess->userlist;
   while(updateuser)
   {
      sess->total++;
      if(updateuser->op) sess->ops++;
      updateuser = updateuser -> next;
   }
   userlist_shownumbers(sess);
}

void op_name(struct session *sess, char *name)
{
   struct user *user = find_name(sess, name);
   if(user)
   {
      user->op = TRUE;
      update_entry(sess, user);
   }
}

void deop_name(struct session *sess, char *name)
{
   struct user *user = find_name(sess, name);
   if(user)
   {
      user->op = FALSE;
      update_entry(sess, user);
   }
}

void voice_name(struct session *sess, char *name)
{
   struct user *user = find_name(sess, name);
   if(user)
   {
      user->voice = TRUE;
      update_entry(sess, user);
   }
}

void devoice_name(struct session *sess, char *name)
{
   struct user *user = find_name(sess, name);
   if(user)
   {
      user->voice = FALSE;
      update_entry(sess, user);
   }
}

void change_nick(struct session *sess, char *oldname, char *newname)
{
   struct user *user = find_name(sess, oldname);
   if(user)
   {
      strcpy(user->user, newname);
      update_entry(sess, user);
   }
}

int sub_name(struct session *sess, char *name)
{
   struct user *prev = sess->userlist;
   struct user *user = sess->userlist;

   while(user)
   {
      if(!strcmp(user->user, name))
      {
	 gint row = gtk_clist_find_row_from_data(GTK_CLIST(sess->namelistgad),
						 (gpointer)user);
	 if(prev != user) /* if not first entry */
	   prev->next = user->next;
	 else
	   sess->userlist = user->next;
	 if(user->op) sess->ops--;
	 sess->total--;
	 userlist_shownumbers(sess);
	 free(user);
	 gtk_clist_remove(GTK_CLIST(sess->namelistgad), row);
	 return TRUE;
      }
      prev = user;
      user = user->next;
   }
   return FALSE;
}

int userlist_insert(struct session *sess, struct user *newuser, gint row)
{   
   char *name = (char *)&(newuser->user);
   
   newuser->weight = 0;

   if(row == -1)
     row = gtk_clist_append(GTK_CLIST(sess->namelistgad), &name);
   else
     gtk_clist_insert(GTK_CLIST(sess->namelistgad), row, &name);
   gtk_clist_set_row_data(GTK_CLIST(sess->namelistgad), row, (gpointer)newuser);

   if(!strcmp(newuser->user, sess->server->nick))
   {
      if(newuser->voice) voice_myself(sess);
      if(newuser->op) op_myself(sess);
      if(!newuser->voice && !newuser->op)
      {
	 if(sess->op_xpm)
         {
	    gtk_widget_destroy(sess->op_xpm);
	    sess->op_xpm = 0;
	 }
      }
   }

   if(newuser->voice)
     gtk_clist_set_pixtext((GtkCList*)sess->namelistgad, row, 0,
			   newuser->user, 3, voice_pixmap, voice_mask_bmp);

   if(newuser->op)
   {
      gtk_clist_set_pixtext((GtkCList*)sess->namelistgad, row, 0,
			 newuser->user, 3, op_pixmap, op_mask_bmp);
      sess->ops++;
   }

   if (prefs.hilitenotify && notify_isnotify(sess, name))
   {
      gtk_clist_set_foreground((GtkCList*)sess->namelistgad, row,
		               &colors[prefs.nu_color]);
   }

   sess->total++;
   userlist_shownumbers(sess);
   return row;
}

int userlist_insertname_sorted(struct session *sess, struct user *newuser, int first)
{
   gint row = 0;
   struct user *user = sess->userlist;
   struct user *prev = sess->userlist;

   if(first) return(userlist_insert(sess, newuser, 0));

   do
   {
      if(nick_cmp(newuser, user) < 0)
      {
	 if(row == 0)
	   sess->userlist = newuser;
	 else
	   prev->next = newuser;
	 newuser->next = user;
	 return(userlist_insert(sess, newuser, row));
      }
      if(!user->next)
      {
	 user->next = newuser;
	 return(userlist_insert(sess, newuser, -1));
      }
      row++;
      prev = user;
      user = user->next;
   } while(1);
}

int add_name(struct session *sess, char *name)
{
   int first;
   struct user *user;
   int row;

   if(!sess->userlist)
   {
      sess->userlist = malloc(sizeof(struct user));
      user = sess->userlist;
      first = TRUE;
   } else {
      user = malloc(sizeof(struct user));
      first = FALSE;
   }

   if(user)
   {
      memset(user, 0, sizeof(struct user));
      if(*name == '@') { name++; user->op = TRUE; }
      if(*name == '+') { name++; user->voice = TRUE; }
      strcpy(user->user, name);
      row = userlist_insertname_sorted(sess, user, first);

      return row;
   }
   return -1;
}

void update_all_of(char *name)
{
   struct user *user;
   struct session *sess;
   GSList *list = sess_list;
   while(list)
   {
      sess = (struct session *)list->data;
      user = find_name(sess, name);
      if (user)
      {
	 update_entry(sess, user);
      }
      list = list->next;
   }
}

#ifdef USE_GNOME

void userlist_dnd_drop(GtkWidget *widget, GdkDragContext *context,
		       gint x, gint y,
		       GtkSelectionData *selection_data,
		       guint info, guint32 time,
		       struct session *sess)
{
   struct user *user;
   char tbuf[256];
   char *file;
   int row, col;
   GList *list;

   if (gtk_clist_get_selection_info(GTK_CLIST(widget), x, y, &row, &col) < 0)
     return;

   user = gtk_clist_get_row_data(GTK_CLIST(widget), row);
   if(!user) return;
   list = gnome_uri_list_extract_filenames(selection_data->data);
   while(list)
   {
      file = (char *)(list -> data);
      dcc_send(sess, tbuf, user->user, file);
      list = list -> next;
   }
   gnome_uri_list_free_strings(list);
}

int userlist_dnd_motion(GtkWidget *widget, GdkDragContext *context, gint x, gint y, guint ttime)
{
   int row, col;

   if(gtk_clist_get_selection_info(GTK_CLIST(widget), x, y, &row, &col) != -1)
   {
      gtk_clist_select_row(GTK_CLIST(widget), row, col);
   }
   return 1;
}

int userlist_dnd_leave (GtkWidget *widget, GdkDragContext *context, guint ttime)
{
   gtk_clist_unselect_all(GTK_CLIST(widget));
   return 1;
}

#endif
