/***************************************************************************
                          startupdialog  -  description
                             -------------------
    begin                : Thu Aug 26 2004
    copyright            : (C) 2004, 2011, 2014 by Thomas Friedrichsmeier
    email                : thomas.friedrichsmeier@kdemail.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "startupdialog.h"

#include <qradiobutton.h>
#include <qcheckbox.h>
#include <qbuttongroup.h>
#include <QGroupBox>
#include <qlabel.h>
#include <QListWidget>
#include <qstringlist.h>
#include <qpixmap.h>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QFileInfo>

#include <KLocalizedString>
#include <krecentfilesaction.h>
#include <KConfigGroup>
#include <QPushButton>

#include "../settings/rksettingsmodulegeneral.h"
#include "../misc/rkcommonfunctions.h"
#include "../misc/rkdialogbuttonbox.h"

#include "../debug.h"

StartupDialog::StartupDialog (QWidget *parent, StartupDialogResult *result, KRecentFilesAction *recent_files) : QDialog (parent) {
	RK_TRACE (DIALOGS);

	setModal (true);
	StartupDialog::result = result;

	setWindowTitle (i18n ("What would you like to do?"));

	QVBoxLayout *vbox = new QVBoxLayout ();
	setLayout (vbox);
	
	QLabel *pic = new QLabel (this);
	vbox->addWidget (pic);
	pic->setPixmap (QPixmap (RKCommonFunctions::getRKWardDataDir () + "icons/rkward_logo.png"));

	choser = new QButtonGroup (this);
	QGroupBox* choser_box = new QGroupBox (this);
	vbox->addWidget (choser_box);
	QVBoxLayout* choser_layout = new QVBoxLayout(choser_box);

	choser_layout->addWidget (empty_workspace_button = new QRadioButton (i18n ("Start with an empty workspace"), choser_box));
	choser->addButton (empty_workspace_button);
	choser_layout->addWidget (empty_table_button = new QRadioButton (i18n ("Start with an empty table"), choser_box));
	choser->addButton (empty_table_button);
	QUrl rdata_file = getRestoreFile ();
	choser_layout->addWidget (restore_workspace_button = new QRadioButton (choser_box));
	choser->addButton (restore_workspace_button);
	if (rdata_file.isEmpty ()) {
		restore_workspace_button->setEnabled (false);
		restore_workspace_button->setText (i18n ("Load workspace from current directory"));
	} else restore_workspace_button->setText (i18n ("Load workspace from current directory:\n%1", rdata_file.toLocalFile ()));
	choser_layout->addWidget (open_button = new QRadioButton (i18n ("Load an existing workspace:"), choser_box));
	choser->addButton (open_button);
	connect (open_button, &QRadioButton::toggled, this, &StartupDialog::openButtonSelected);
	empty_table_button->setChecked (true);

	file_list = new QListWidget (choser_box);
	file_list->setSelectionMode (QAbstractItemView::SingleSelection);
	file_list->setSortingEnabled (false);
	chose_file_item = new QListWidgetItem (i18n ("<<Open another file>>"), file_list);
	if (recent_files) {
		QList<QUrl> urls = recent_files->urls ();
		for (int i = 0; i < urls.length (); ++i) {
			file_list->addItem (urls[i].url (QUrl::PreferLocalFile));
		}
	}
	connect (file_list, &QListWidget::itemClicked, this, &StartupDialog::listClicked);
	connect (file_list, &QListWidget::itemDoubleClicked, this, &StartupDialog::listDoubleClicked);
	choser_layout->addWidget (file_list);
	choser_layout->addWidget (remember_box = new QCheckBox (i18n ("Always do this on startup"), choser_box));

	RKDialogButtonBox *buttonBox = new RKDialogButtonBox (QDialogButtonBox::Ok | QDialogButtonBox::Cancel, this);
	buttonBox->button (QDialogButtonBox::Ok)->setDefault (true);
	vbox->addWidget (buttonBox);
}

StartupDialog::~StartupDialog() {
	RK_TRACE (DIALOGS);
}

void StartupDialog::accept () {
	RK_TRACE (DIALOGS);

	if (empty_workspace_button->isChecked ()) {
		result->result = EmptyWorkspace;
	} else if (empty_table_button->isChecked ()) {
		result->result = EmptyTable;
	} else if (restore_workspace_button->isChecked ()) {
		result->result = RestoreFromWD;
		result->open_url = getRestoreFile ();
	} else if (open_button->isChecked ()) {
		QListWidgetItem *item = file_list->currentItem ();
		if (item == chose_file_item) {
			result->result = ChoseFile;
		} else {
			result->result = OpenFile;
			result->open_url = QUrl (item->text ());
		}
	} else {
		RK_ASSERT (false);
	}
	if (remember_box->isChecked ()) RKSettingsModuleGeneral::setStartupAction (result->result);
	QDialog::accept ();
}

void StartupDialog::reject () {
	RK_TRACE (DIALOGS);
	
	result->result = EmptyWorkspace;
	
	QDialog::reject ();
}

void StartupDialog::listDoubleClicked (QListWidgetItem *item) {
	RK_TRACE (DIALOGS);
	
	if (item) {
		open_button->setChecked (true);
		file_list->setCurrentItem (item);
		item->setSelected (true);
		accept ();
	}
}

void StartupDialog::listClicked (QListWidgetItem *item) {
	RK_TRACE (DIALOGS);
	
	if (item) {
		open_button->setChecked (true);
		openButtonSelected (true);		// always do processing
	}
}

void StartupDialog::openButtonSelected (bool checked) {
	RK_TRACE (DIALOGS);

	if (checked) {
		if (!file_list->currentItem ()) {
			file_list->setCurrentRow (0);
		}
		if (file_list->currentItem () != chose_file_item) {
			remember_box->setChecked (false);
			remember_box->setEnabled (false);
		} else {
			remember_box->setEnabled (true);
		}
	} else {
		remember_box->setEnabled (true);
	}
}

void StartupDialog::showEvent (QShowEvent *event) {
	RK_TRACE (DIALOGS);

	// somehow, trying to achieve this in the ctor leads to the dialog never actually being shown (KDE4.0 beta)
	setFixedWidth (width ());
	QDialog::showEvent (event);
}

// static
QUrl StartupDialog::getRestoreFile () {
	RK_TRACE (DIALOGS);

	QFileInfo rdata_file (".RData");
	if (rdata_file.exists ()) return QUrl::fromLocalFile (rdata_file.absoluteFilePath ());

	return QUrl ();
}

//static
StartupDialog::StartupDialogResult StartupDialog::getStartupAction (QWidget *parent, KRecentFilesAction *recent_files) {
	RK_TRACE (DIALOGS);

	StartupDialogResult result;
	result.result = RKSettingsModuleGeneral::startupAction ();

	if (result.result == RestoreFromWD) {
		result.open_url = getRestoreFile ();
		if (result.open_url.isEmpty ()) result.result = NoSavedSetting;
	}
	if (result.result != NoSavedSetting) {
		return result;
	}

	StartupDialog *dialog = new StartupDialog (parent, &result, recent_files);
	dialog->exec ();
	delete dialog;
	
	RK_DEBUG (DIALOGS, DL_DEBUG, "startup-dialog result: %d, url: %s", result.result, qPrintable (result.open_url.url ()));
	
	return result;
}

